;;; -*- Mode:Lisp; Package:ZWEI; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B); Vsp:0; Syntax:Common-Lisp -*-

;1;; File "CHANGELOG"*
;1;; Automatic changelog stuff.*
;1;; Written and maintained by Jamie Zawinski.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   25 Jul 88*	1Jamie Zawinski*  1Created.*
;1;;   22 Sep 88*	1Jamie Zawinski*  1Made the spacing be nicer.  Made it use *USER-ID1 if *FS:USER-PERSONAL-NAME-FIRST-NAME-FIRST
;1;;*			  1       is empty.*
;1;;   29 Sep 88*	1Jamie Zawinski*  1Made it not grind the name when the previous entry was by the same name on the same day.*
;1;;*			  1      Made it line the date-start and name-start up under the previous line.*
;1;;    7 Nov 88*	1Jamie Zawinski*  1Made it properly line up single-digits under other single-digits.*
;1;;   28 Dec 88*	1Jamie Zawinski*  1Added error trapping for buffers without changelogs.*
;1;;    5 Jan 89*	1Jamie Zawinski*  1Added support for variable width fonts, and for Electric Font Lock Mode.*
;1;;    6 Jan 89*	1Jamie Zawinski   Made it correctly indent the text of the log entry after the user name.*
;1;;*			1          Added **EDIT-CHANGELOG-COMTAB*,1 and *COM-CHANGELOG-INDENT1.*
;1;;   20 Jan 89*	1Jamie Zawinski   Oops, fixed indentation of date when previous changelog was multi-line - was looking at *LINE1 (which*
;1;;*			1          is the previous line) instead of *DATE-LINE1 (which is the first line of the previous changelog).*
;1;;     9 Feb 89*	1Jamie Zawinski   Made *COM-CHANGELOG-INDENT1 return the proper second value along with *DIS-TEXT1.*
;1;;    28 Mar 89*	1Jamie Zawinski   Defined 4COM-JUSTIFY-CURRENT-CHANGELOG*.*
;1;;*

(defun 4log-entry-candidate* (string)
  "2T if it is conceivable (from a quick look) that STRING is a log entry.
  This is true if the string has digit-characters and semicolons within the first ten characters.*"
  (let* ((max 10)
	 (len (length string)))
    (and (> len max)
	 (position-if #'digit-char-p string :end max)
	 (position #\; string :end max)
	 t)))

(defun 4line-empty-p* (string)
  "2T if there is only whitespace and semicolons in the string.*"
  (dotimes (i (length string) t)
    (let* ((c (char string i)))
      (unless (member c '(#\Space #\Tab #\;) :test #'char-equal) ;1 char-equal to ignore font.*
	(return nil)))))

(defun 4find-last-log-entry* (interval)
  "2Returns two values: a LINE, and a list of LINEs.  These are from the INTERVAL supplied.  The LINE will be the last line
  on which a changelog exists, and the list will be all of the lines before it on which log entries began.*"
  (declare (values end-of-log-line log-start-lines))
  (let* ((line-count 0)
	 (log-count 0)
	 (got-a-log nil)
	 (line (bp-line (interval-first-bp interval)))
	 (last-line nil)
	 (logs nil))
    (loop
      (unless line (barf "3This buffer has no changelog.*"))
      (setq last-line line)
      (setq line (line-next line))
      (incf line-count)
      (cond ((log-entry-candidate line)
	     (setq got-a-log t)
	     (push line logs)
	     (incf log-count))
	    ((line-empty-p line)
	     (when got-a-log
	       (let* ((next-line (line-next line)))
		 (unless (log-entry-candidate next-line)
		   (return (values last-line
				   logs))))))
	    ))))


(defun 4parse-log* (string)
  "2Returns four values.
    DOTW-P: whether a day of the week was present.
    DATE, MONTH, YEAR: these are each a CONS of a number or symbol, and the position at which it ended in the string.*"
  (declare (values dotw-p date.end month-symbol.end year.end))
  (let* (dotw
	 date
	 month
	 year)
    (let* ((start (position-if-not #'(lambda (x)
				       (member x '(#\Space #\Tab #\;) :test #'char-equal))
				   string))
	   thing)
      (macrolet ((grab (var)
		   `(multiple-value-bind (new-thing end-pos)
			                 (read-from-string string t nil :start start)
		      (setq ,var (cons new-thing start))
		      (setq start end-pos))))
	(grab thing)
	(cond ((numberp (car thing))
	       (setq dotw nil
		     date thing))
	      (t (setq dotw thing)
		 (grab date)))
	(grab month)
	(grab year)
	(values dotw date month year)))))


(defun 4substring-no-fonts* (string start &optional end)
  "2Like SUBSEQ, but if STRING is an ART-FAT-STRING, an ART-STRING will be returned, with the font bits discarded.*"
  (unless end (setq end (length string)))
  (let* ((length (- end start))
	 (new-string (make-string length)))
    (dotimes (i length)
      (setf (schar new-string i) (char string (+ start i))))
    new-string))


(defun 4tokenize* (string &optional (start 0) end)
  "2Break up the string into ``words,'' discarding the whitespace between them.  Returns a list of strings.*"
  (unless end (setq end (length string)))
  (let* ((tokens '())
	 (last-nonwhite-pos nil))
    (do* ((pos start (1+ pos))
	  )
	 ((>= pos end)
	  (when last-nonwhite-pos
	    (push (substring-no-fonts string last-nonwhite-pos) tokens)))
      (let* ((c (char string pos)))
	
	;1; Nuke the font bits of the character if it has any.*
	(when (plusp (char-font c)) (setq c (make-char c (char-bits c) 0)))
	
	(cond ((and (graphic-char-p c) (char/= c #\Space) (char/= c #\;))  ;1 it's a constituent character.*
	       (unless last-nonwhite-pos (setq last-nonwhite-pos pos))
	       )
	      (last-nonwhite-pos                            ;1 we've encountered whitespace after nonwhite.*
	       (push (substring-no-fonts string last-nonwhite-pos pos) tokens)
	       (setq last-nonwhite-pos nil)
	       (setq start pos))
	      )))
    (nreverse tokens)))


(defun 4date-of-entry* (string)
  "2Returns seven values:
     Whether the day of the week was specified in the string;
     The numeric day of the month which was specified;
     The string month which was specified;
     The numeric year which was specified;
     The position at which the date began;
     The position at which the first text after the date began;
     The probable position at which the text of the log (after the name) ended.*"
  (declare (values dotw-p day month year date-start-pos user-name-start-pos log-text-start-pos))
  (block NIL
    (let* ((strings (tokenize string 0 (min 50 (length string))))
	   dotw day month year year-string)
      (if (setq day (parse-integer (car strings) :junk-allowed t))
	  (pop strings)
	  (setq dotw (pop strings)
		day (or (parse-integer (pop strings) :junk-allowed t)
			(return nil))))
      (setq month (pop strings))
      (setq year-string (pop strings))
      (setq year (parse-integer year-string))
      ;1; We must use CHAR-EQUAL instead of CHAR= because we want to ignore the font bits.*
      (flet ((white-p (x) (or (char-equal x #\Space) (char-equal x #\Tab))))
	(let* ((after-semicolons (position #\; string :test-not #'char-equal))
	       (start-of-date (position-if-not #'white-p string :start (1+ after-semicolons)))
	       (start-of-year (lisp:search year-string string :test #'char-equal :start2 start-of-date))
	       (end-of-date (position-if #'white-p string :start (1+ start-of-year)))
	       (start-of-name (position-if-not #'white-p string :start (1+ end-of-date)))
	       (end-of-first-name (position-if #'white-p string :start (1+ start-of-name)))
	       (start-of-second-name (position-if-not #'white-p string :start (1+ end-of-first-name)))
	       (only-one-name-p (or (> (- start-of-second-name end-of-first-name) 2)
				    (char-equal #\Tab (char string end-of-first-name))))
	       (end-of-second-name (unless only-one-name-p
				     (position-if #'white-p string :start start-of-second-name)))
	       (start-of-log-text (if only-one-name-p
				      start-of-second-name
				      (position-if-not #'white-p string :start end-of-second-name)))
	       )
	  (values dotw day month year
		  start-of-date
		  start-of-name
		  start-of-log-text
		  ))))))


(defun 4find-last-date* (logs)
  "2Returns seven values:
    The time that was specified, as a universal time;
    Whether the day of the week was specified in the string;
    The string from which the last date was parsed;
     The position at which the date began;
     The position at which the first text after the date began;
     The probable position at which the text of the log (after the name) ended.*"
  (declare (values universal-time dotw-p line
		   date-start-pos user-name-start-pos log-text-start-pos))
  (dolist (line logs)
    (multiple-value-bind (dotw day month year date-start-pos user-name-start-pos log-text-start-pos) (date-of-entry line)
      (when day
	(return (values (time:parse-universal-time (format nil "3~a ~a ~a*" day month year))
			dotw
			line
			date-start-pos user-name-start-pos log-text-start-pos))))))


(defun 4same-day-p* (ut1 ut2)
  "2T if both universal times fall on the same day, month, and year.*"
  (multiple-value-bind (sec min hour day month year) (decode-universal-time ut1)
    (declare (ignore sec min hour))
    (multiple-value-bind (sec2 min2 hour2 day2 month2 year2) (decode-universal-time ut2)
      (declare (ignore sec2 min2 hour2))
      (and (= day day2)
	   (= month month2)
	   (= year year2)))))


(defun 4string-pos-to-pixel *(bp-line char-pos)
  "2Given a BP-LINE and an index into it, return that distance in pixels, considering *FONT* and *WINDOW*.*"
  (let* ((temp-bp (make-bp :bp-line bp-line :bp-index char-pos))
	 (pixels (bp-virtual-indentation temp-bp)))
    (flush-bp temp-bp)
    pixels))


(defun interval-char-width4 *(char font-number)
  "2Returns the width of the given character in the given font number.*"
  (let* ((fonts (send *interval* :get-attribute :fonts))
	 (font (nth font-number fonts)))
    (when (or (stringp font) (symbolp font))
      (setq font (symbol-value (intern (string font) "3FONTS*"))))
    (tv:sheet-string-length (window-sheet *window*) (string (make-char char (char-bits char) 0)) 0 nil nil font)))



(defvar 4*edit-changelog-comtab**
	(let* ((c (set-comtab 'EDIT-CHANGELOG-COMTAB '(#\Control-Tab    com-changelog-indent
						       #\Meta-Control-Q com-justify-current-changelog
						       ))))
	  (set-comtab-indirection c *recursive-edit-comtab*)
	  c)
  "2The comtab used while in a changelog recursive edit.*")


(defun 4grind-log-entry* (line logs &optional fonts-p)
  "2Grind a new changelog entry into the buffer after the log beginning on LINE.
  Returns a BP to the end of the new changelog line.*"
  (let* ((first-bp (make-bp :bp-line line :bp-index (length line)))
	 (bp (copy-bp first-bp :moves))
	 (*font* (if fonts-p
		     (if (numberp fonts-p) fonts-p 1)
		     *font*)))
    
    (insert-moving bp #\Newline)
    (insert-moving bp "3;*")
    (insert-moving bp (in-current-font "3;;*"))
    (multiple-value-bind (last-date dotw-p date-line date-start-pos user-name-start-pos log-text-start-pos)
			 (find-last-date logs)
      (let* ((now (get-universal-time))
	     (same-day-p (same-day-p now last-date))
	     (last-date-single-digit-p nil))
	(unless dotw-p
	  (multiple-value-bind (sec min hour day) (decode-universal-time last-date)
	    (declare (ignore sec min hour))
	    (when (< day 10) (setq last-date-single-digit-p t))))
	(unless same-day-p
	  (multiple-value-bind (sec min hour day month year dotw) (decode-universal-time now)
	    (declare (ignore sec min hour))
	    ;1;;*
	    ;1;; Indent to the position at which the date begins.*
	    (let* ((pos-to-indent (string-pos-to-pixel date-line date-start-pos)))
	      ;1;;*
	      ;1;; Deal with indention of days that are represented with different numbers of digits - like *" 5"1 and *"21"1.*
	      (cond ((and (< day 10) (not last-date-single-digit-p))
		     ;1; This is a single day and the last wasn't - increment X by the length of the first digit in the last date.*
		     (incf pos-to-indent (interval-char-width (char date-line date-start-pos) *font*)))
		    ((and (>= day 10) last-date-single-digit-p)
		     ;1; This is not a single day and the last was - decrement X by the length of the first digit in today's date.*
		     (let* ((tens-char (char (princ-to-string (floor day 10)) 0)))
		       (decf pos-to-indent (interval-char-width tens-char *font*))))
		    (t ;1 otherwise today and the last day have the same number of digits, and we don't need to do anything.*
		     nil))
	      (indent-to bp pos-to-indent))
	    ;1;; Turn MONTH and DOTW into strings.*
	    (setq month (svref #("" "3Jan*" "3Feb*" "3Mar*" "3Apr*" "3May*" "3Jun*" "3Jul*" "3Aug*" "3Sep*" "3Oct*" "3Nov*" "3Dec*") month)
		  dotw  (svref #("3Mon*" "3Tue*" "3Wed*" "3Thu*" "3Fri*" "3Sat*" "3Sun*") dotw))
	    (when dotw-p
	      (insert-moving bp (in-current-font dotw)) ;1; ## Do punctuation - if last date said "Mon, 3 jan" then put in the comma.*
	      (insert-moving bp (in-current-font #\Space)))
	    (insert-moving bp (in-current-font (format nil "3~D ~A ~D*" day month (- year 1900))))))
	;1;;*
	;1;; Indent to the position at which the user name begins, and write it.*
	;1;;*
	(indent-to bp (string-pos-to-pixel date-line user-name-start-pos))
	(let* ((name fs:user-personal-name-first-name-first))
	  (when (or (null name) (string= name "")) (setq name user-id))
	  (when (string= name "") (setq name "3Not Logged In*"))
	  ;1;*
	  ;1; Insert the user's name unless their name was in the previous changelog as well.*
	  ;1;*
	  (unless (and same-day-p (lisp:search name date-line :test #'char-equal))
	    (insert-moving bp (in-current-font name))
	    (insert-moving bp (in-current-font #\Space)) ;1 In case the name is very long, we want at least one space after it.*
	    )))
      ;1;*
      ;1; Indent to the position at which the text of the previous changelog began.*
      (indent-to bp (string-pos-to-pixel date-line log-text-start-pos)))
    (flush-bp first-bp)
    (end-line bp 0)))


(defcom 4com-insert-log-entry* 2"Add a new changelog entry in a recursive edit."* ()
  (multiple-value-bind (last-line log-list) (find-last-log-entry *interval*)
    (let* ((fonts-p (and (member 'ELECTRIC-FONT-LOCK-MODE *mode-name-list*)        ; 1Use fonts only if in Font-Lock mode, and*
			 (> (length (send *interval* :get-attribute :fonts)) 1)))  ; 1if there are more than one font specified.*
	   (log-bp (grind-log-entry last-line log-list fonts-p))
	   (*changelog-text-position* (bp-virtual-indentation log-bp)))
      (declare (special *changelog-text-position*)) ;1 used by *COM-CHANGELOG-INDENT1.*
      (unwind-protect
	  (with-bp (saved-bp (point) :normal)
	    (let* ((*comtab* *edit-changelog-comtab*))
	      (move-bp (point) log-bp)
	      (must-redisplay *window* DIS-TEXT)
	      (redisplay *window*)
	      (format *query-io* "3~&Type END when done editing changelog.*")
	      (catch 'ZWEI:EXIT-CONTROL-R
		(send *window* :edit nil *comtab* `("3[*" "3Edit-Changelog *" ,@*mode-line-list* "3]*")))
	      )
	    (move-bp (point) saved-bp))
	(flush-bp log-bp))))
  DIS-TEXT)


(defcom com-changelog-indent
	"2Indent to the beginning of the log text of the current changelog.  This is for use only within Insert Log Entry.*" ()
  (declare (special *changelog-text-position*)) ;1 Specially bound by *COM-INSERT-LOG-ENTRY1.*
  (indent-to (point) *changelog-text-position*)
  (values DIS-TEXT
	  (bp-line (point))))


(defcom 4com-justify-current-changelog*
	"2Make the indentation of the changelog which surrounds the point be the same as the previous one.
  Leaves the point at the end of this changelog.*" ()
  (multiple-value-bind (last-line log-list) (find-last-log-entry *interval*)
    (let* ((current-log nil)
	   (target-log nil)
	   (following-log nil)
	   (point (point)))
      (do* ((logs (reverse log-list) (cdr logs)))
	   (())
	(let* ((prev-log (first logs))
	       (this-log (second logs))
	       (next-log (third logs))
	       (this-bp (and this-log (create-bp this-log 0)))
	       (next-bp (and this-log (if next-log
					  (create-bp next-log 0)
					  (create-bp (line-next last-line) 0)))))
	  (unwind-protect
	      (cond ((null this-log) (return))
		    
		    ((or (bp-= this-bp point)
			 (and (bp-< this-bp point)
			      (bp-< point next-bp)))
		     (setq target-log prev-log
			   current-log this-log
			   following-log next-log)
		     (return)))
	    (when this-bp
	      (flush-bp this-bp)
	      (flush-bp next-bp)))))

      (unless current-log
	(let* ((topmost-bp (create-bp (nth (- (length log-list) 2) log-list) 0)))
	  (unwind-protect
	      (if (bp-< point topmost-bp)
		  (barf "3The point is on or above the first changelog entry.*")
		  (barf "3The point is not on a changelog entry.*"))
	    (flush-bp topmost-bp))))

      (with-undo-save ("3Changelog Justification*" (create-bp current-log 0)
						    (create-bp (or following-log (line-next last-line)) 0))
	
	(let* (pixel-text-target)  ;1 used later*
	1   *;1;*
	1   *;1; First, justify the current line.*
	1   *;1;*
	  (multiple-value-bind (dw od m y target-date-start target-name-start target-text-start) (date-of-entry target-log)
	    (declare (ignore dw m y))
	    (multiple-value-bind (dw nd m y this-date-start this-name-start this-text-start) (date-of-entry current-log)
	      (declare (ignore dw m y))

	      (let* ((last-date-single-digit-p (and (< od 10)
						    (not (char-equal #\0 (char target-log (1- target-date-start))))))
		     (date-pixel-increment 0))
		;1;; Deal with indention of days that are represented with different numbers of digits - like *" 5"1 and *"21"1.*
		;1;;*
		(cond ((and (< nd 10) (not last-date-single-digit-p))
		       ;1; This is a single day and the last wasn't - increment X by the length of the first digit in the last date.*
		       (setq date-pixel-increment (interval-char-width (char target-log target-date-start) *font*)))
		      ((and (>= nd 10) last-date-single-digit-p)
		1          *;1; This is not a single day and the last was - decrement X by the length of the first digit in today's date.*
		       (let* ((tens-char (char (princ-to-string (floor nd 10)) 0)))
			 (setq date-pixel-increment (- (interval-char-width tens-char *font*)))))
		      (t ;1 otherwise today and the last day have the same number of digits, and we don't need to do anything.*
		       nil))
		
		(let* ((bp1 (create-bp current-log this-date-start :moves))
		       (bp2 (create-bp current-log this-name-start :moves))
		       (bp3 (create-bp current-log this-text-start :moves)))
		  (delete-interval (backward-over *blanks* bp1) bp1)
		  (delete-interval (backward-over *blanks* bp2) bp2)
		  (delete-interval (backward-over *blanks* bp3) bp3)
		  (indent-to bp1 (+ (string-pos-to-pixel target-log target-date-start) date-pixel-increment))
		  (indent-to bp2 (string-pos-to-pixel target-log target-name-start))
		  (indent-to bp3 (string-pos-to-pixel target-log target-text-start))
		  (setq pixel-text-target (string-pos-to-pixel target-log target-text-start))
		  (flush-bp bp1)
		  (flush-bp bp2)
		  (flush-bp bp3)
		  ))))
	1   *;1;*
	1   *;1; Then, justify the continuation lines, if any.*
	1   *;1;*
	  (do* ((line (line-next current-log)
		      (line-next line)))
	       ((or (null line)
		    (eq line (or following-log (line-next last-line)))))
	    (let* ((bp (create-bp line 0)))
	      (move-bp bp (forward-over '(#\;) bp))
	      (delete-interval bp (forward-over *blanks* bp))
	      (indent-to bp pixel-text-target)
	      (flush-bp bp)))
	1   *;1;*
	1   *;1; Move the point to the end of this changelog entry.*
	1   *;1;*
	  (move-bp point (or following-log (line-next last-line)) 0)
	  ))))
  DIS-TEXT)
