;;; -*- Mode:Lisp; Package:ZWEI; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B TR12BI); Syntax:Common-Lisp -*-

;1;; File "*BOLD-LOCK1"*
;1;; Code to activate the 5Bold Lock* and 5Italic Lock* keys on the Explorer keyboard, and to define a Zmacs Minor Mode that uses them.*
;1;; Written and maintained by Jamie Zawinski.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;  27 Jan 89   Jamie Zawinski*	1Created.*
;1;;   1 Feb 89   Jamie Zawinski*	1Moved the Bold-Lock and Italic-Lock characters to a different place in the character set, since they*
;1;;*	1 *                       1  were shadowing some graphic characters.*
;1;;   2 Feb 89   Jamie Zawinski*	1Fixed a bug that happened when the font list in a buffer's Properties line was out of synch with the*
;1;;*				1  actual fonts in use.*
;1;;   1 Apr 89   Jamie Zawinski*	1Defined *BOLD-LOCK-STATE1, *ITALIC-LOCK-STATE1, and their SETF methods.  Removed the variables*
;1;;*				1  **BOLD-LOCK-P*1 and **ITALIC-LOCK-P*1, since the user could get into trouble by changing these.*
;1;;   2 Apr 89   Jamie Zawinski *	1Broke 4ENSURE-FONT-PRESENT* out of *BOLD-LOCK-HOOK1.*
;1;;   6 Apr 89   Jamie Zawinski *	1Made the command hook special-case coexistence with 5Electric Scribe Mode* as well as *
;1;;*				1 5Electric Font Lock Mode*.  I wish I could think of a better way...*
;1;;  28 Jul 89    Jamie Zawinski *	1Sometimes *:GET-ATTRIBUTE :FONTS1 returns a symbol, not a list (if the buffer has only one font specified.*
;1;;*				1 Made the mode hook deal with this... grrr.*
;1;;  17 Aug 89   Jamie Zawinski *	1I thought of a better way.  Mode-hooks which hack fonts should have a property, 5font-hacking-hook*.*
;1;;*				1 If there are any hooks in effect which have that property (other than self) then bold-lock mode maximizes*
;1;;*				1 fonts instead of being absolute.*
;1;;  24 Aug 89   Jamie Zawinski *	1Made Bold Lock Hook use 5:send-if-handles :saved-font-alist*, since 5node*s don't have them.*
;1;;*


;1;; Make the 5Bold-Lock* and 5Italic-Lock* keys be active, and have named-characters associated with them.*
;1;; The functions 4TICL:BOLD-LOCK-STATE* and 4TICL:**ITALIC-4LOCK-STATE1 may be used to interrogate the font keys.**
;1;; These functions may be 5setf*'ed, so that programs can turn these keys on or off; for example, a user could make an *
;1;; editor keybinding which made 5Super-B* be exactly the same as the 5Bold-Lock *key.*
;1;;*

(export '(ticl:bold-lock-state     ticl:italic-lock-state
	  4ticl:set-bold-lock-state* 4ticl:set-*italic4-lock-state*)
	"TICL")


(defun 4ticl:bold-lock-state *()
  "2Returns T or NIL, the state of the 6Bold Lock* key.  This may be changed with SETF.*"
  (logbitp 2 SYS:LOCK-BITS))

(defun 4ticl:italic-lock-state *()
  "2Returns T or NIL, the state of the 6Italic Lock* key.  This may be changed with SETF.*"
  (logbitp 1 SYS:LOCK-BITS))


(defun 4ticl:set-bold-lock-state *(newval)
  "2If you want to turn the 6Bold Lock* key on or off programatically, call this function with the desired state.*"
  (tv:write-keyboard (if newval 'TV:BOLD-KEY-LED-ON 'TV:BOLD-KEY-LED-OFF)))

(defun 4ticl:set-italic-lock-state *(newval)
  "2If you want to turn the 6Italic Lock* key on or off programatically, call this function with the desired state.*"
  (tv:write-keyboard (if newval 'TV:ITALIC-KEY-LED-ON 'TV:ITALIC-KEY-LED-OFF)))


(defsetf 4ticl:bold-lock-state * 4 *ticl:set-bold-lock-state)
(defsetf 4ticl:italic-lock-state *ticl:set-italic-lock-state)



;1;; Here's a dilemma.*
;1;;*
;1;; In the TI character mapping, there are no holes.  Every character code is assigned either to a graphic character or a named character.*
;1;; If we want to enable the Bold and Italic keys, we need to have a character code to give them. *
;1;;*
;1;;    o*	1This code must be less than 256, because the ninth bit is in the font (and higher bits also have meanings like that).*
;1;;    o*	1The code must not be a code occupied by a graphic character, or use of that character would be confused.*
;1;;    o*	1In the standard fonts, all characters are graphic except those between 127 and 160 inclusive.  So clearly, the new keys must go there.*
;1;;    o*	1But, all of the characters in that range are named - that is where5 System*, 5Network*, 5F1*, etc. are located.*
;1;;    o*	1However, there are characters which have names which are not present on the explorer keyboard - things like 5Stop-Output*, 5Quote*,*
;1;;*	1and 5Macro*.*
;1;;*
;1;; So, we use two of those - 5Hold-Output* and 5Stop-Output* - as 5Bold-Lock* and 5Italic-Lock*.*
;1;;*
;1;; If this code were run on a Lisp Machine which had 5Hold-Output* and 5Stop-Output* keys, then those would work like 5Bold-Lock* and 5Italic-Lock*.*
;1;; For this reason, this code is extremely TI dependant.*
;1;;*


(defconstant 4BOLD-CHAR-CODE *(char-code #\Hold-Output) "2The character code to assign to the the Bold Lock key.*")
(defconstant 4ITAL-CHAR-CODE *(char-code #\Stop-Output) "2The character code to assign to the the Italic Lock key.*")


(defun setup-4bold-and-italic-chars* ()
  "2Create named characters for the Bold Lock and Italic Lock keys, and insert them in the hardware-to-software mapping table.*"
  (flet ((add-char (name char-code scan-code)
	   ;1;*
	1    *;1; Add the named character.*
	   ;1;*
	   (let* ((cons (assoc name SYS:XR-SPECIAL-CHARACTER-NAMES)))
	     ;1; I don't understand why, but additions to this list must be made to the* CDDR1 or later, or they disappear.*
	     ;1; I guess it's because this list is in some funny memory area...*
	     (if cons
		 (setf (cdr cons) char-code)
		 (push (cons name char-code) (cddr SYS:XR-SPECIAL-CHARACTER-NAMES))))
	   ;1;*
	   ;1; Add the scan-code to the lookup table.*
	   ;1;*
	   (dotimes (i 5) (setf (aref SYS:KBD-TI-TABLE i scan-code) (logior #x8000 char-code))) ;1 set bit 15.*
	   (setf (aref SYS:KBD-TI-TABLE 5 scan-code) 0)
	   ;1;*
	   ;1; Mark the character as not being a graphic-character.  We must do this because the graphic-char-p vector was created long*
	   ;1; before we created this character, and might be incorrect.*
	   ;1;*
	   (setf (aref SYS:GRAPHIC-CHAR-P-VECTOR char-code) 0)
	   ))
    (let* ((bold-scan-code SYS:SCAN-CODE-BOLD-LOCK)
	   (ital-scan-code SYS:SCAN-CODE-ITAL-LOCK))
      (add-char :BOLD-LOCK   bold-char-code bold-scan-code)
      (add-char :ITALIC-LOCK ital-char-code ital-scan-code)
      
      (make-char bold-char-code))))


(setup-4bold-and-italic-chars*) ;1 Install this at load-time.*



;1;; Define a ZMACS minor-mode that looks at the Bold and Italic lock variables and sets the current font appropriately.*
;1;;*
;1;; If Bold-Lock Mode is on, the following behavior is exhibited:*
;1;;*
;1;; Make the current font be the font which corresponds to the current font modified by the Bold and Italic lock keys - *
;1;; for example, if the current font was5 TR12, TR12b, TR12i*, or5 TR12bi*, and the italic lock key was lit but the bold lock key was not,*
;1;; then the resultant font would be 5TR12i*.*
;1;;*
;1;; If Bold-Lock Mode attempts to make the current font be one which is not in the font list of the current buffer, the user is asked*
;1;; whether to add this font to the font list.  If the user replies 5no*, then Bold-Lock Mode will not ask again about that particular font - *
;1;; subsequent attempts to change to that font will be no-ops.*
;1;;*
;1;; If both Bold-Lock Mode and Electric Font-Lock Mode are on at the same time, the behavior is slightly different.*
;1;; When Bold-Lock Mode is determining the current font, it "maximizes" the facecode of the font - that is, if the current font is italic,*
;1;; but the Bold Lock key is lit and the Italic Lock key is not, then the resultant font will be Bold-Italic.*
;1;;*
;1;; Suppose the user has told Electric Font-Lock Mode that all docstrings are to be printed in the font 5TR12I*.  If the interaction between*
;1;; Bold-Lock Mode and Electric Font-Lock Mode was not what it is, then unless the user always hit Italic Lock before typing docstrings,*
;1;; docstrings would come out in 5TR12* instead of the requested 5TR12i*.*
;1;;*
;1;; Also, if the user is typing in a docstring, and the docstring font is 5TR12i*, the user may produce 5TR12bi* merely by hitting the Bold Lock key.*
;1;; *
;1;; *

(defminor 4com-bold-lock-mode* bold-lock-mode "1Bold-lock*" 6
	  "2Minor mode to fontify characters based on the Bold Lock and Italic Lock keys.
A positive argument turns the mode on, zero turns it off; no argument toggles.
This works alone, or in conjunction with Electric Font Lock Mode.*" ()
  (command-hook 'BOLD-LOCK-HOOK *command-hook*))


(defmacro 4bold-lock-interval-ignored-fonts *(interval) `(get ,interval :BL-IGNORED-FONTS))


(defvar 4*bold-lock-mode-tracks-keys-exactly* *nil
  "2If T, then Bold Lock Mode will set the current font to have exactly the face indicated by the Bold Lock and Italic Lock keys.
  If NIL, then it will maximize the face - if the current font is italic, and the Bold Lock key is lit, then the resultant font will be Bold Italic.
  This is the correct behavior for interaction with Electric Font Lock Mode, and the other is correct when Electric Font Lock Mode is not on.
  This is bound by BOLD-LOCK-HOOK.*")

(defprop 4bold*-lock-hook 20 command-hook-priority)	;1 Make this of a lower priority (higher number) than *FONT-LOCK-HOOK1.*

(defprop 4bold-lock-hook *t font-hacking-hook)

(defprop 4font-lock-hook *t font-hacking-hook)		;1 Define these properties here since they aren't defined elsewhere.*
(defprop 4scribe-mode-hook *t font-hacking-hook)


(defun 4bold*-lock-hook (char)
  (declare (ignore char))
  (unless (eq *interval* (window-interval *mini-buffer-window*))
    (let* ((font-number *font*)
	   ;1; It's faster to do 5(ignore-errors (send ... ))* than 5(send ... :send-if-handles ... )*.*
	   (font-prop (ignore-errors (send *interval* :get-attribute :fonts)))
	   (font-list (and (consp font-prop) (ignore-errors (send *interval* :saved-font-alist)))))
      ;1;*
      ;1; Maybe don't do anything if this buffer doesn't have declared fonts.*
      (when (or (and font-prop font-list) *electric-font-lock-if-no-fonts-declared-p*)
	
	(let* ((font-name (and font-prop (nth font-number font-prop)))
	       (old-font  (if font-prop
			      (cdr (assoc font-name font-list :test #'string-equal))
			      ucl:*default-font*))
	       ;1; Track keys exactly only if there are not other font-hacking hooks in effect.*
	       (4*bold-lock-mode-tracks-keys-exactly** (not (find-if #'(lambda (symbol)
								       (and (neq symbol 'bold-lock-hook)
									    (get symbol 'font-hacking-hook)))
								   *command-hook*)))
	       (new-font (and font-prop font-list font-name old-font
			      (bold-lock-find-font old-font (bold-lock-state) (italic-lock-state)))))
	  (unless (or (null new-font)
		      (eq old-font new-font))
	    (let* ((ignored-p (member new-font (bold-lock-interval-ignored-fonts *interval*)))
		   (new-number nil))
	      (unless ignored-p
		(setq new-number (ensure-font-present new-font))
		(unless new-number
		  (push new-font (bold-lock-interval-ignored-fonts *interval*))))
	      (when new-number
		(setq *font* new-number)
		(update-font-name)))))))))



(defun 4ensure-font-present *(new-font)
  "2If the font is not in the font map of **INTERVAL*2 then ask the user if they want to add it.
  If the user says yes, then add it, and update the font list.  Returns the font-number of this newly-added font.
  If the user says no, then return NIL.
  (If the font is already there, return its number.)*"
  (let* ((font-prop (send *interval* :get-attribute :fonts))
	 (new-number (position (tv:font-name new-font) font-prop :test #'string-equal)))
    (cond ((null new-number)
	   (let* ((add-p (fquery `(:fresh-line t :beep t :clear-input t
					       :type :readline :choices ,sys::yes-or-no-p-choices)
				 "3The font ~A is not in the font map of this buffer.  Add it? *"
				 (tv:font-name new-font))))
	     (when add-p
	       (send *interval* :set-attribute :fonts
		     (if font-prop
			 (append font-prop
				 (list (intern (string (tv:font-name new-font)) "3KEYWORD*")))
			 (list (intern (string (tv:font-name ucl:*default-font*)) "3KEYWORD*")
			       (intern (string (tv:font-name new-font)) "3KEYWORD*")))
		     :query)
	       (let* ((alist (mapcar #'(lambda (name)
					 (let* ((symbol (intern (string-upcase (string name)) "3FONTS*"))
						(value (tv:font-evaluate symbol)))
					   (cons symbol value)))
				     (send *interval* :get-attribute :fonts))))
		 (redefine-fonts *window* alist (send *interval* :get-attribute :VSP) nil))
	       (setq new-number (1- (length (send *interval* :get-attribute :fonts))))
	       new-number)))
	  (t new-number))))




(defvar 4*bold-lock-hook-font-states* *'()
  "2A cache used for quickly converting between the faces of fonts.
  Elements of this list are of the form:  *(2<plain-font-name> <bold-font-name> <italic-font-name> <bold-italic-font-name>*)")


(defun 4bold-lock-cache-font-faces *(font-name)
  "2Update the cache to contain cross-referencing for FONT-NAME.*"
  (setq font-name (string font-name))
  (let* ((basic-font-name  (intern (string-right-trim "3BI*" font-name) "3FONTS*"))
	 (bold-font-name   (intern (string-append basic-font-name "3B*") "3FONTS*"))
	 (italic-font-name (intern (string-append basic-font-name "3I*") "3FONTS*"))
	 (both-font-name   (intern (string-append basic-font-name "3BI*") "3FONTS*"))
	 (list (list basic-font-name bold-font-name italic-font-name both-font-name)))
    (push list 4*bold-lock-hook-font-states**)
    list))


(defun 4lookup-font-cached-faces *(font-name)
2  *"2Returns a list of the form:  *(2<plain-font-name> <bold-font-name> <italic-font-name> <bold-italic-font-name>*)2.
  FONT-NAME will be some element of the returned list.*"
  (or (dolist (x 4*bold-lock-hook-font-states**)
	(when (member font-name x :test #'eq) (return x)))
      (bold-lock-cache-font-faces font-name)))


(defsubst 4font-is-plain-p *      (font-name) (eq font-name (first  (4lookup-font-cached-faces* font-name))))
(defsubst 4font-is-bold-p *       (font-name) (eq font-name (second (4lookup-font-cached-faces* font-name))))
(defsubst 4font-is-italic-p *     (font-name) (eq font-name (third  (4lookup-font-cached-faces* font-name))))
(defsubst 4font-is-bold-italic-p *(font-name) (eq font-name (fourth (4lookup-font-cached-faces* font-name))))

(defun 4font-with-face *(font-name bold-p italic-p)
  (let* ((list (4lookup-font-cached-faces* font-name))
	 (already-bold-p   (or (eq font-name (fourth list)) (eq font-name (second list))))
	 (already-italic-p (or (eq font-name (fourth list)) (eq font-name (third list))))
	 )
    (unless 4*bold-lock-mode-tracks-keys-exactly**
      (setq bold-p (or bold-p already-bold-p)
	    italic-p (or italic-p already-italic-p)))
    (cond ((and bold-p italic-p) (fourth list))
	  (bold-p (second list))
	  (italic-p (third list))
	  (t (first list)))))


(defun 4bold-lock-find-font *(font bold-p italic-p)
  "2Given a font or font-name, returns the font object which is a bold or italic version of it.*"
  (let* ((font-name (tv:font-name font))
	 (new-font-name (font-with-face font-name bold-p italic-p)))
    (cond ((eq font-name new-font-name)
	   font)
	  (t
	   (if (boundp new-font-name)
	       (symbol-value new-font-name)
	       font)))))


;1;; Make it be a valid Extended Command.*
;1;;*
(set-comtab *zmacs-comtab* () '(("3Bold Lock Mode*" . com-bold-lock-mode)))
