;;; -*- Mode: LISP; Syntax: Common-Lisp; Base: 10; Package: ZWEI -*-

;;; ***************************************************************************
;;; *                                                                         *
;;; *                               BIBTEX-MODE                               *
;;; *                                                                         *
;;; *       ZMACS major mode for editing BibTeX bibliography databases.       *
;;; *                                                                         *
;;; *                         (C) 1988 Kenneth R. Traub                       *
;;; *                                                                         *
;;; *     This program is provided with absolutely no warranty whatsoever.    *
;;; *                                                                         *
;;; * The author grants permission for anyone to copy, modify, and distribute *
;;; * this program as long as the author's name appears on any such copy.     *
;;; *                                                                         *
;;; ***************************************************************************

(setf (get 'bibtex-mode :copyright-notice)
      "Copyright (C) 1988 Kenneth R. Traub")

;;; Written November, 1988, by Kenneth R. Traub.
;;; Bug reports to KT@XX.LCS.MIT.EDU.

;;; Don't mess with this!  This has been coded according to the rules BibTeX uses to say
;;; what an abbreviation is (one exception: \ is given LIST-SLASH syntax, even though the
;;; LaTeX manual implies that \ is alphabetic as far as abbreviations are concerned.  This is
;;; not a problem as long as nobody is foolish enough to actually use a \ as part of an
;;; abbreviation name).  It is not necessarily the best table for editing the contents
;;; of fields, for which you'd like the TeX syntax table.  Tough luck.
(defvar *bibtex-list-syntax-table*
	(make-syntax-table
	  '((32. LIST-ALPHABETIC)
	    
	    LIST-DELIMITER			;040 space
	    LIST-ALPHABETIC			;041 !
	    LIST-DOUBLE-QUOTE			;042 "
	    LIST-DELIMITER			;043 #
	    LIST-ALPHABETIC			;044 $
	    LIST-COMMENT			;045 %
	    LIST-ALPHABETIC			;046 &
	    LIST-DELIMITER			;047 '
	    LIST-OPEN				;050 (
	    LIST-CLOSE				;051 )
	    LIST-ALPHABETIC			;052 *
	    LIST-ALPHABETIC			;053 +
	    LIST-DELIMITER			;054 ,
	    LIST-ALPHABETIC			;055 -
	    LIST-ALPHABETIC			;056 .
	    LIST-ALPHABETIC			;057 /
	    (10. LIST-ALPHABETIC)		;DIGITS
	    LIST-ALPHABETIC			;072 :
	    LIST-ALPHABETIC			;073 ;
	    LIST-ALPHABETIC			;074 <
	    LIST-DELIMITER			;075 =
	    LIST-ALPHABETIC			;076 >
	    LIST-ALPHABETIC			;077 ?
	    LIST-ALPHABETIC			;100 @
	    (26. LIST-ALPHABETIC)		;LETTERS
	    LIST-OPEN				;133 [
	    LIST-SLASH				;134 \
	    LIST-CLOSE				;135 ]
	    LIST-ALPHABETIC			;136 ^
	    LIST-ALPHABETIC			;137 _
	    LIST-ALPHABETIC			;140 `
	    (26. LIST-ALPHABETIC)		;MORE LETTERS
	    LIST-OPEN				;173 {
	    LIST-ALPHABETIC			;174 |
	    LIST-CLOSE				;175 }
	    LIST-ALPHABETIC			;176 ~
	    LIST-ALPHABETIC			;177 integral
	    
	    LIST-ALPHABETIC			;200 null character
	    LIST-DELIMITER			;201 break
	    LIST-DELIMITER			;202 clear
	    LIST-DELIMITER			;203 call
	    LIST-DELIMITER			;204 escape (NOT altmode!)
	    LIST-DELIMITER			;205 backnext
	    LIST-DELIMITER			;206 help
	    LIST-DELIMITER			;207 rubout
	    LIST-DELIMITER			;210 bs
	    LIST-DELIMITER			;211 tab
	    LIST-DELIMITER			;212 line
	    LIST-DELIMITER			;213 vt
	    LIST-DELIMITER			;214 form = newpage
	    LIST-DELIMITER			;215 return = newline
	    (#o162 LIST-ALPHABETIC))))

(defvar *bibtex-word-syntax-table*
	(make-syntax-table
	  '((32. WORD-ALPHABETIC)
	    WORD-DELIMITER			;040 space
	    WORD-ALPHABETIC			;041 !
	    WORD-DELIMITER			;042 "
	    WORD-DELIMITER			;043 #
	    WORD-DELIMITER			;044 $
	    WORD-DELIMITER			;045 %
	    WORD-DELIMITER			;046 &
	    WORD-DELIMITER			;047 '
	    WORD-DELIMITER			;050 (
	    WORD-DELIMITER			;051 )
	    WORD-DELIMITER			;052 *
	    WORD-DELIMITER			;053 +
	    WORD-DELIMITER			;054 ,
	    WORD-DELIMITER			;055 -
	    WORD-DELIMITER			;056 .
	    WORD-DELIMITER			;057 /
	    (10. WORD-ALPHABETIC)		;Digits
	    WORD-DELIMITER			;072 :
	    WORD-DELIMITER			;073 ;
	    WORD-DELIMITER			;074 <
	    WORD-DELIMITER			;075 =
	    WORD-DELIMITER			;076 >
	    WORD-ALPHABETIC			;077 ?
	    WORD-DELIMITER			;100 @
	    (26. WORD-ALPHABETIC)		;Uppercase letters
	    WORD-DELIMITER			;133 [
	    WORD-DELIMITER			;134 \
	    WORD-DELIMITER			;135 ]
	    WORD-DELIMITER			;136 ^
	    WORD-ALPHABETIC			;137 _
	    WORD-DELIMITER			;140 `
	    (26. WORD-ALPHABETIC)		;Lowercase letters
	    WORD-DELIMITER			;173 {
	    WORD-DELIMITER			;174 |
	    WORD-DELIMITER			;175 }
	    WORD-ALPHABETIC			;176 ~
	    (129. WORD-DELIMITER))))

(defvar *bibtex-whitespace-chars* '(#\space #\tab #\newline #\line #\page))

(defvar *bibtex-open-close-alist* '((#\( . #\)) (#\[ . #\]) (#\{ . #\})))

(eval-when (compile load eval)
(defun make-keyword (x)
  (cond ((keywordp x)
	 x)
	((symbolp x)
	 (intern (#+Symbolics get-pname #+TI symbol-name x) "KEYWORD"))
	((stringp x)
	 (intern x "KEYWORD"))
	(t
	 (error "~s cannot be converted into a keyword." x))))
)

(defun closing-for-opening (opening)
  (cdr (assoc opening *bibtex-open-close-alist*)))

(defvar *bib-fields* nil)

(defvar *bib-field-completion-alist* nil)

(defvar *bib-entry-types* nil)

(defvar *bib-entry-type-completion-alist* nil)

(eval-when (compile load eval)

(defmacro bib-field-named (name)
  `(get ,name 'bib-field))

(defstruct (bib-field
	     (:conc-name bib-field-)
	     :named-array
	     (:copier nil)
	     (:print "#<BIB-FIELD ~a>" (bib-field-name bib-field)))
  name
  string
  documentation)

(defmacro defbib-field (name &body clauses)
  (let* ((the-name (make-keyword name))
	 (string (or (second (#+Symbolics assq #+TI zlc:assq :string clauses))
		     (string-capitalize-words (#+Symbolics get-pname #+TI symbol-name the-name) t t)))
	 (documentation (second (#+Symbolics assq #+TI zlc:assq :documentation clauses))))
    `(define-bib-field ',the-name ',string ',documentation)))

(defun define-bib-field (name string documentation)
  (let ((bib-field (or (bib-field-named name)
			    (progn
			      (push name *bib-fields*)
			      (push (cons string name) *bib-field-completion-alist*)
			      (setf (bib-field-named name)
				    (make-bib-field))))))
    (setf (bib-field-name bib-field) name)
    (setf (bib-field-string bib-field) string)
    (setf (bib-field-documentation bib-field) documentation)
    name))

(defmacro bib-entry-type-named (name)
  `(get ,name 'bib-entry-type))

(defstruct (bib-entry-type
	     (:conc-name bib-entry-type-)
	     :named-array
	     (:copier nil)
	     (:print "#<BIB-ENTRY-TYPE ~a>" (bib-entry-type-name bib-entry-type)))
  name
  string
  required-fields
  optional-fields
  documentation)

(defmacro defbib-entry-type (name &body clauses)
  (let* ((the-name (make-keyword name))
	 (string (or (second (#+Symbolics assq #+TI zlc:assq :string clauses))
		     (string-capitalize-words (#+Symbolics get-pname #+TI symbol-name the-name) t t)))
	 (required-fields (cdr (#+Symbolics assq #+TI zlc:assq :required-fields clauses)))
	 (optional-fields (cdr (#+Symbolics assq #+TI zlc:assq :optional-fields clauses)))
	 (documentation (second (#+Symbolics assq #+TI zlc:assq :documentation clauses))))
    `(define-bib-entry-type
       ',the-name ',string ',required-fields ',optional-fields ',documentation)))

(defun define-bib-entry-type (name string required-fields optional-fields documentation)
  (let ((bib-entry-type (or (bib-entry-type-named name)
			    (progn
			      (push name *bib-entry-types*)
			      (push (cons string name) *bib-entry-type-completion-alist*)
			      (setf (bib-entry-type-named name)
				    (make-bib-entry-type))))))
    (setf (bib-entry-type-name bib-entry-type) name)
    (setf (bib-entry-type-string bib-entry-type) string)
    (setf (bib-entry-type-required-fields bib-entry-type) 
	  (mapcar #'(lambda (x)
		      (if (listp x)
			  (mapcar #'(lambda (y)
				      (if (#+Symbolics memq #+TI zlc:memq y '(:xor :or))
					  y
					  (bib-field-name (validate-bib-field y))))
				  x)
			  (bib-field-name (validate-bib-field x))))
		  required-fields))
    (setf (bib-entry-type-optional-fields bib-entry-type) 
	  (mapcar #'(lambda (y)
		      (bib-field-name (validate-bib-field y)))
		  optional-fields))
    (setf (bib-entry-type-documentation bib-entry-type) documentation)
    name))

(defun validate-bib-field (x)
  (let ((name (make-keyword x)))
    (or (bib-field-named name)
	(error "~a is not the name of a BibTeX field." name))))
)

(defun bib-entry-type-n-fields (entry-type)
  (+ (loop for required in (bib-entry-type-required-fields entry-type)
	   sum
	     (if (symbolp required)
		 1
		 (length (cdr required))))
     (length (bib-entry-type-optional-fields entry-type))))

(defun bib-entry-type-has-field-name (entry-type field-name)
  (or (#+Symbolics memq #+TI zlc:memq field-name (bib-entry-type-optional-fields entry-type))
      (loop for required in (bib-entry-type-required-fields entry-type)
	    thereis (if (symbolp required)
			(eq required field-name)
			(#+Symbolics memq #+TI zlc:memq field-name (cdr required))))))

;;; Calls FCN on two args: a field object, and one of the following:
;;;   :REQUIRED
;;;   (:OR name name ...)
;;;   (:XOR name name ...)
;;;   :OPTIONAL
(defun map-bib-entry-type-fields (fcn entry-type)
  (loop for required in (bib-entry-type-required-fields entry-type) do
    (cond ((symbolp required)
	   (funcall fcn (bib-field-named required) :required))
	  (t
	   (loop for r in (cdr required) do
	     (funcall fcn (bib-field-named r) required)))))
  (loop for optional in (bib-entry-type-optional-fields entry-type) do
    (funcall fcn (bib-field-named optional) :optional)))

;;; The documentation in the following were derived from three sources: (1) the LaTeX manual
;;; published by Addison-Wesley, fifth printing, September 1986; (2) the BibTeX manual, from
;;; the file BTXDOC.TEX in the BibTeX distribution directory; (3) the documentation of the
;;; standard BibTeX styles, file BTXBST.DOC, Version 0.99b for BibTeX version 0.99c.
;;; Since source (3) is the most recent, it takes precedence over the other two.  Note that
;;; (3) incorporates some recent changes to the standard styles which accomodate more fields
;;; for some entry types.

(defbib-field address
  (:documentation "Usually the address of the publisher or other type of institution.
 For major publishing houses, van Leunen recommends omitting the information entirely.
 For small publishers, on the other hand, you can help the reader
 by giving the complete address.
 For `InProceedings' and `Proceedings' entries, gives the place where the conference
 was held, not the address of the publisher; for those two entry types, include the
 publisher's or organization's address, if necessary, in the `Publisher' or 
 `Organization' field."))

(defbib-field annote
  (:documentation "An annotation.
 It is not used by the standard bibliography styles,
 but may be used by others that produce an annotated bibliography."))

(defbib-field author
  (:documentation "The name(s) of the author(s), in the format described in the LaTeX book."))

(defbib-field booktitle
  (:string "BookTitle")
  (:documentation "Title of a book, part of which is being cited.
 See the LaTeX book for how to type titles.
 For book entries, use the `title' field instead."))

(defbib-field chapter
  (:documentation "A chapter (or section or whatever) number."))

(defbib-field edition
  (:documentation "The edition of a book---for example, \"Second\".
 This should be an ordinal, and should have the first letter capitalized, as shown here;
 the standard styles convert to lower case when necessary."))

(defbib-field editor
  (:documentation "Name(s) of editor(s), typed as indicated in the LaTeX book.
 If there is also an `author' field, then the `editor' field gives
 the editor of the book or collection in which the reference appears."))

(defbib-field howpublished
  (:string "HowPublished")
  (:documentation "How something strange has been published.
 The first word should be capitalized."))

(defbib-field institution
  (:documentation "The sponsoring institution of a technical report."))

(defbib-field journal
  (:documentation "A journal name."))

(defbib-field key
  (:documentation "Used for alphabetizing, cross referencing, and creating a label
 when the `author' information is missing.
 This field should not be confused with the key that appears in the
 \\cite command and at the beginning of the database entry."))

(defbib-field month
  (:documentation "The month in which the work was published or, for an unpublished work,
 in which it was written.
 You should use the standard three-letter abbreviation,
 as described in Appendix B.1.3 of the LaTeX book."))

(defbib-field note
  (:documentation "Any additional information that can help the reader find a reference.
 The first word should be capitalized."))

(defbib-field number
  (:documentation "The number of a journal, magazine, technical report, or of a work in a series.
 An issue of a journal or magazine is usually identified by its volume and number;
 the organization that issues a technical report usually gives it a number;
 and sometimes books are given numbers in a named series."))

(defbib-field organization
  (:documentation "The organization that sponsors a conference or that publishes a manual."))

(defbib-field pages
  (:documentation "One or more page numbers or range of numbers, 
 such as \"42--111\" or \"7,41,73--97\" or \"43+\"
 (the `+' in this last example indicates pages following that don't form a simple range).
 To make it easier to maintain Scribe-compatible databases, 
 the standard styles convert a single dash (as in 7-33) 
 to the double dash used in TeX to denote number ranges (as in 7--33)."))

(defbib-field publisher
  (:documentation "The publisher's name."))

(defbib-field school
  (:documentation "The name of the school where a thesis was written."))

(defbib-field series
  (:documentation "The name of a series or set of books.
 When citing an entire book, the `title' field gives its title and an optional `series'
 field gives the name of a series or multi-volume set in which the book is published."))

(defbib-field title
  (:documentation "The work's title, typed as explained in the LaTeX book."))

(defbib-field type
  (:documentation "The meaning of this field depends on the entry type.
 For `TechReport' entries, the type of a technical report---for example, \"Research Note\".
 For `PhDThesis' and `MastersThesis', gives an alternative thesis type---
 for example, \"Bachelor's Thesis\".
 When a `chapter' field is given with `InBook' or `InCollection' entries,
 gives an alternative interpretation---for example, \"Section\"."))

(defbib-field volume
  (:documentation "The volume of a journal or multivolume book."))

(defbib-field year
  (:documentation "The year of publication or, for an unpublished work, the year it was written.
 Generally it should consist of four numerals, such as \"1984\",
 although the standard styles can handle any `year' whose last four nonpunctuation
 characters are numerals, such as \"(about 1984)\"."))

(defbib-entry-type article
  (:required-fields author title journal year)
  (:optional-fields volume number pages month note)
  (:documentation "An article from a journal or magazine."))

(defbib-entry-type book
  (:required-fields (:xor author editor) title publisher year)
  (:optional-fields volume number series edition address month note)
  (:documentation "A book with an explicit publisher."))

(defbib-entry-type booklet
  (:required-fields title)
  (:optional-fields author howpublished address month year note)
  (:documentation "A work that is printed and bound,
but without a named publisher or sponsoring institution."))

(defbib-entry-type conference
  (:required-fields author title booktitle year)
  (:optional-fields editor volume number series pages organization publisher address month note)
  (:documentation "The same as `InProceedings', included for Scribe compatibility."))

(defbib-entry-type inbook
  (:string "InBook")
  (:required-fields (:xor author editor) title (:or chapter pages) publisher year)
  (:optional-fields volume number series edition type address month note)
  (:documentation "A part of a book, which may be a chapter (or section or whatever if the
 `type' field is given) and/or a range of pages."))

(defbib-entry-type incollection
  (:string "InCollection")
  (:required-fields author title booktitle publisher year)
  (:optional-fields editor volume number series edition type chapter pages address month note)
  (:documentation "A part of a book having its own title."))

(defbib-entry-type inproceedings
  (:string "InProceedings")
  (:required-fields author title booktitle year)
  (:optional-fields editor volume number series pages organization publisher address month note)
  (:documentation "An article in a conference proceedings."))

(defbib-entry-type manual
  (:required-fields title (:or author organization key))
  (:optional-fields edition address month year note)
  (:documentation "Technical documentation."))

(defbib-entry-type mastersthesis
  (:string "MastersThesis")
  (:required-fields author title school year)
  (:optional-fields type address month note)
  (:documentation "A Master's thesis."))

(defbib-entry-type misc
  (:required-fields (:or author key))
  (:optional-fields title howpublished month year note)
  (:documentation "Use this type when nothing else fits."))
  
(defbib-entry-type phdthesis
  (:string "PhDThesis")
  (:required-fields author title school year)
  (:optional-fields type address month note)
  (:documentation "A PhD thesis."))

(defbib-entry-type proceedings
  (:required-fields title year (:or editor organization key))
  (:optional-fields volume number series publisher address month note)
  (:documentation "The proceedings of a conference."))

(defbib-entry-type techreport
  (:string "TechReport")
  (:required-fields author title institution year)
  (:optional-fields type number address month note)
  (:documentation "A report published by a school or other institution,
 usually numbered within a series."))

(defbib-entry-type unpublished
  (:required-fields author title note)
  (:optional-fields month year)
  (:documentation "A document having an author and title, but not formally published."))

(defun write-bib-entry (entry-type-name stream &key
			citation-key field-alist commentsp empty-fields-p
			missing-fields-p extra-fields-p)
  (let* ((entry-type (bib-entry-type-named entry-type-name))
	 ;; Field descriptor: (field-string value-symbol-or-string comment-string-or-nil)
	 (field-descriptors 
	   (make-bib-field-descriptor-list 
	     entry-type field-alist commentsp empty-fields-p missing-fields-p extra-fields-p))
	 (n-field-lines (length field-descriptors)))
    (format stream "@~a(" (bib-entry-type-string entry-type))
    (when citation-key
      (format stream "~a" citation-key))
    (when (> n-field-lines 0)
      (format stream ","))
    (let ((field-string-width
	    (loop for (field-string v c) in field-descriptors
		  maximize (#+Symbolics string-length #+TI length field-string))))
      (loop for descriptor in field-descriptors
	    for field-line from 1 do
	(write-bib-field-line
	  stream descriptor field-string-width (< field-line n-field-lines))))
    (format stream "~%  )")))

(defun make-bib-field-descriptor-list (entry-type field-alist commentsp empty-fields-p
				       missing-fields-p extra-fields-p)
  (let ((field-descriptors '()))
    (map-bib-entry-type-fields
      #'(lambda (field class)
	  (let ((value (cdr (#+Symbolics assq #+TI zlc:assq (bib-field-name field) field-alist))))
	    (when (or missing-fields-p 
		      (and value
			   (or empty-fields-p
			       (symbolp value)
			       (> (#+Symbolics string-length #+TI length value) 0))))
	      (push (list (bib-field-string field)
			  (or value "")
			  (if commentsp
			      (make-bib-field-line-comment field class)
			      nil))
		    field-descriptors))))
      entry-type)
    (when extra-fields-p
      (loop for (name . value) in field-alist do
	(unless (bib-entry-type-has-field-name entry-type name)
	  (when (or empty-fields-p 
		    (symbolp value)
		    (> (#+Symbolics string-length #+TI length value) 0))
	    (push (list (string-capitalize-words name t t)
			(or value "")
			(if commentsp "Ignored by standard styles" nil))
		  field-descriptors)))))
    (nreverse field-descriptors)))

(defun make-bib-field-line-comment (field class)
  (cond ((eq class :optional)
	 "Optional")
	((listp class)
	 (#+Symbolics caseq #+TI case (first class)
	   (:or
	     (let ((others (remove (bib-field-name field) (cdr class))))
	       (format nil "~{and/or ~a~^ ~}"
		       (mapcar #'(lambda (fn)
				   (bib-field-string
				     (bib-field-named fn)))
			       others))))
	   (:xor
	     (let ((others (remove (bib-field-name field) (cdr class))))
	       (format nil "~{or ~a~^ ~}"
		       (mapcar #'(lambda (fn)
				   (bib-field-string
				     (bib-field-named fn)))
			       others))))
	   (otherwise
	     nil)))
	(t
	 nil)))

(defun write-bib-field-line (stream descriptor field-string-width commap)
  (let ((field-string (first descriptor))
	(value (second descriptor))
	(comment (third descriptor)))
    (format stream "~%  ~va = " field-string-width field-string)
    (if (stringp value)
	(format stream "{~a}" value)
	(format stream "~a" value))
    (when commap
      (format stream ","))
    (when comment
      (when (not commap)
	(format stream " "))
      (let* ((current-col (or #+Symbolics (send-if-handles stream :read-cursorpos :character)
			      (+ 6
				 field-string-width
				 (if (stringp value)
				     (+ 2 (#+Symbolics string-length #+TI length value))
				     #+Symbolics (string-length (get-pname value))
				     #+TI        (length (symbol-name value))))))
	     (n-spaces (- (/ *comment-column* (font-space-width)) current-col)))
	(if (< n-spaces 1)
	    (format stream " ~a~a" *comment-begin* comment)
	    (format stream #+Symbolics "~vx~a~a" #+TI "~v@t~a~a"
		    n-spaces *comment-begin* comment))))))

(defun parse-bib-entry (bp)
  (let ((entry-closing nil)
	(entry-type-name nil)
	(citation-key nil)
	(field-alist nil)
	(current-bp (copy-bp bp)))
    (labels ((skip-whitespace ()
	       (setq current-bp (forward-over *bibtex-whitespace-chars* current-bp))
	       (when (= (list-syntax (bp-char current-bp)) list-comment)
		 (setq current-bp (forward-line current-bp))
		 (skip-whitespace)))
	     (parse-optional-comma ()
	       (skip-whitespace)
	       (when (eql (bp-char current-bp) #\,)
		 (setq current-bp (forward-char current-bp))))
	     (parse-character (char)
	       (skip-whitespace)
	       (if (eql char (bp-char current-bp))
		   (setq current-bp (forward-char current-bp 1))
		   (throw 'bib-syntax-error t)))
	     (parse-symbol (&optional case-sensitive)
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-alphabetic)
		   (let* ((new-bp (bibtex-forward-balanced current-bp))
			  (name (string-interval current-bp new-bp)))
		     (prog1 (if case-sensitive
				(intern name "KEYWORD")
				(intern (string-upcase name) "KEYWORD"))
			    (setq current-bp new-bp)))
		   (throw 'bib-syntax-error t)))
	     (parse-opening ()
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-open)
		   (prog1 (bp-char current-bp)
			  (setq current-bp (forward-char current-bp 1)))
		   (throw 'bib-syntax-error t)))
	     (parse-closing ()
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-close)
		   (setq current-bp (forward-char current-bp 1))
		   (throw 'bib-syntax-error t)))
	     (peek-character () 
	       (skip-whitespace)
	       (bp-char current-bp))
	     (parse-balanced () 
	       (skip-whitespace)
	       (if (or (= (list-syntax (bp-char current-bp)) list-open)
		       (= (list-syntax (bp-char current-bp)) list-double-quote))
		   (let* ((new-bp (bibtex-forward-balanced current-bp))
			  (value (string-interval (forward-char current-bp)
						  (forward-char new-bp -1))))
		     (prog1 value
			    (setq current-bp new-bp)))
		   (throw 'bib-syntax-error t))))
      (let ((error-p
	      (catch 'bib-syntax-error
		(parse-character #\@)
		(setq entry-type-name (parse-symbol))
		(setq entry-closing (closing-for-opening (parse-opening)))
		(unless (eq entry-type-name :string)
		  (unless (eql (peek-character) #\,)
		    (setq citation-key (parse-symbol t)))
		  (parse-optional-comma))
		(loop until (eql (peek-character) entry-closing) do
		  (let ((field-name nil)
			(field-value nil))
		    (setq field-name (parse-symbol))
		    (parse-character #\=)
		    (if (let ((next-char (peek-character)))
			  (or (= (list-syntax next-char) list-open)
			      (= (list-syntax next-char) list-double-quote)))
			(setq field-value (parse-balanced))
			(setq field-value (parse-symbol t)))
		    (push (cons field-name field-value) field-alist)
		    (parse-optional-comma)))
		(parse-closing)
		nil)))
	(if error-p
	    (progn
	      (move-bp (point) current-bp)
	      (barf "Syntax error here.")
	      nil)
	    (values entry-type-name citation-key field-alist))))))

;;; Search types include:
;;;    :BEFORE BP - bp of field start immediately before BP
;;;    :AFTER BP  - bp of field start immediately after BP
;;;    :FIELD NAME - bp of field start of field named NAME
;;;    :CITATION - bp of beginning of citation.
;;;    :WHAT-FIELD BP - name of field in which midst BP is in.
;;;    :WHAT-ENTRY-TYPE - name of entry type.
(defun parse-bib-entry-search (start-bp search-type &rest args)
  (let ((entry-closing nil)
	(entry-type-name nil)
	(field-alist nil)
	(previous-field-bp nil)
	(current-bp (copy-bp start-bp)))
    (labels ((skip-whitespace ()
	       (setq current-bp (forward-over *bibtex-whitespace-chars* current-bp))
	       (when (= (list-syntax (bp-char current-bp)) list-comment)
		 (setq current-bp (forward-line current-bp))
		 (skip-whitespace)))
	     (parse-optional-comma ()
	       (skip-whitespace)
	       (when (eql (bp-char current-bp) #\,)
		 (setq current-bp (forward-char current-bp))))
	     (parse-character (char)
	       (skip-whitespace)
	       (if (eql char (bp-char current-bp))
		   (setq current-bp (forward-char current-bp 1))
		   (throw 'bib-search 'error)))
	     (parse-symbol (&optional case-sensitive)
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-alphabetic)
		   (let* ((new-bp (bibtex-forward-balanced current-bp))
			  (name (string-interval current-bp new-bp)))
		     (prog1 (if case-sensitive
				(intern name "KEYWORD")
				(intern (string-upcase name) "KEYWORD"))
			    (setq current-bp new-bp)))
		   (throw 'bib-search 'error)))
	     (skip-symbol ()
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-alphabetic)
		   (setq current-bp (bibtex-forward-balanced current-bp))
		   (throw 'bib-search 'error)))
	     (parse-opening ()
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-open)
		   (prog1 (bp-char current-bp)
			  (setq current-bp (forward-char current-bp 1)))
		   (throw 'bib-search 'error)))
	     (parse-closing ()
	       (skip-whitespace)
	       (if (= (list-syntax (bp-char current-bp)) list-close)
		   (setq current-bp (forward-char current-bp 1))
		   (throw 'bib-search 'error)))
	     (peek-character () 
	       (skip-whitespace)
	       (bp-char current-bp))
	     (skip-balanced () 
	       (skip-whitespace)
	       (if (or (= (list-syntax (bp-char current-bp)) list-open)
		       (= (list-syntax (bp-char current-bp)) list-double-quote))
		   (setq current-bp (bibtex-forward-balanced current-bp))
		   (throw 'bib-search 'error))))
      (let ((found-bp
	      (catch 'bib-search
		(parse-character #\@)
		(setq entry-type-name (parse-symbol))
		(when (eq search-type :what-entry-type)
		  (throw 'bib-search entry-type-name))
		(setq entry-closing (closing-for-opening (parse-opening)))
		(when (eq search-type :citation)
		  (throw 'bib-search current-bp))
		(unless (eq entry-type-name :string)
		  (unless (eql (peek-character) #\,)
		    (skip-symbol))
		  (parse-optional-comma))
		(loop until (eql (peek-character) entry-closing) do
		  (let ((field-name nil)
			(field-value nil))
		    (setq field-name (parse-symbol))
		    (parse-character #\=)
		    (when (and (eq search-type :field)
			       (eq (first args) field-name))
		      (skip-whitespace)
		      (throw 'bib-search current-bp))
		    (let ((next-char (peek-character)))
		      (when (and (eq search-type :before)
				 (not (bp-< current-bp (first args))))
			(throw 'bib-search previous-field-bp))
		      (when (and (eq search-type :after)
				 (bp-< (first args) current-bp))
			(throw 'bib-search current-bp))
		      (setq previous-field-bp current-bp)
		      (if (or (= (list-syntax next-char) list-open)
			      (= (list-syntax next-char) list-double-quote))
			  (skip-balanced)
			  (skip-symbol)))
		    (push (cons field-name field-value) field-alist)
		    (parse-optional-comma)
		    (when (and (eq search-type :what-field)
			       (bp-< (first args) current-bp))
		      (throw 'bib-search field-name))))
		(parse-closing)
		(when (and (eq search-type :before)
			   (not (bp-< current-bp (first args))))
		  (throw 'bib-search previous-field-bp))
		nil)))
	(if (eq found-bp 'error)
	    (progn
	      (move-bp (point) current-bp)
	      (barf "Syntax error here.")
	      nil)
	    found-bp)))))

(defun find-bibtex-field-boundaries (bp)
  (let ((start-bp (bib-entry-start-bp bp))
	(field-start nil)
	(field-end nil))
    (if (null start-bp)
	(values nil nil)
	(progn
	  ;; 1st attempt: assume point is somewhere within field value.
	  ;; (the FORWARD-CHAR takes care of the case where we're right at the start).
	  (setq field-start (parse-bib-entry-search start-bp :before (forward-char bp)))
	  (setq field-end (and field-start (copy-bp (bibtex-forward-balanced field-start))))
	  (when (not (and field-end (not (bp-< field-end bp))))
	    ;; 2nd attempt: assume point is before field value.
	    (setq field-start (parse-bib-entry-search start-bp :after bp))
	    (setq field-end (and field-start (copy-bp (bibtex-forward-balanced field-start)))))
	  (values field-start field-end)))))

;;; The following routines were adapted from Id Mode, (C)1986,1987,1988 Massachusetts
;;; Instititue of Technology.

;;; A bib entry starts with a line containing an @ in the first column.
(defun bib-entry-start-line-p (line)
  (and (> (#+Symbolics string-length #+TI length line) 0)
       (eql (aref line 0) #\@)))

(defun bib-entry-start-line (current-line &optional (n-forward 0))
  (let* ((interior-line
	   ;; First, skip over blank, comment, or form lines.  This gets us either to the
	   ;; beginning of the entry or somewhere within it, but in any event not in the
	   ;; space between the beginning of the extent and the beginning of the entry.
	   (loop with last-line = (bp-line (interval-last-bp *interval*))
		 for line = current-line then (line-next line)
		 until (or (eq line last-line)
			   (not (member (bibtex-line-type line) '(:comment :blank :form))))
		 finally (return line)))
	 ;; Now go back until we find the entry start line for the current entry.
	 (current-start-line
	   (loop with first-line = (bp-line (interval-first-bp *interval*))
		 for line = interior-line then (line-previous line)
		 until (or (eq line first-line)
			   (bib-entry-start-line-p line))
		 finally (return line))))
    ;; Once we know where the entry start line for the current entry is, we just
    ;; iterate N-FORWARD times searching forward or backward for other entry start lines.
    (cond ((and (not (bib-entry-start-line-p current-start-line))
		(>= n-forward 0))
	   nil)
	  ((zerop n-forward)
	   current-start-line)
	  ((minusp n-forward)
	   (loop for start-line = current-start-line
				then (loop for line = (line-previous start-line)
						    then (line-previous line)
					   until (or (null line)
						     (bib-entry-start-line-p line))
					   finally (return line))
		 for i from 1 to (- n-forward)
		 when (null start-line)
		   do (return nil)
		 finally (return start-line)))
	  (t
	   (loop for start-line = current-start-line
				then (loop for line = (line-next start-line)
						    then (line-next line)
					   until (or (null line)
						     (bib-entry-start-line-p line))
					   finally (return line))
		 for i from 1 to n-forward
		 when (null start-line)
		   do (return nil)
		 finally (return start-line))))))


(defun bib-entry-extent-start-line (current-line &optional (n-forward 0))
  (let ((start-line (bib-entry-start-line current-line n-forward)))
    (and start-line
	 (loop with first-line = (bp-line (interval-first-bp *interval*))
	       for line = start-line then (line-previous line)
	       until (or (eq line first-line)
			 (not (member (bibtex-line-type (line-previous line))
				      '(:comment :blank :form))))
	       finally (return line)))))

(defun bib-entry-end-line (current-line &optional (n-forward 0))
  (let ((start-line (bib-entry-start-line current-line n-forward)))
    (if (null start-line)
	(values nil nil)
	(let ((maybe-line (bib-entry-extent-start-line start-line 1)))
	  (if (null maybe-line)
	      (values nil :end-of-buffer)
	      maybe-line)))))

(defun bib-entry-start-bp (bp &optional (n-forward 0) fixup-p)
  (let ((first-bp (interval-first-bp *interval*))
	(start-line (bib-entry-start-line (bp-line bp) n-forward)))
    (cond ((null start-line)
	   (if fixup-p (copy-bp first-bp) nil))
	  (t
	   (if (eq start-line (bp-line first-bp))
	       (copy-bp first-bp)
	       (create-bp start-line 0))))))

(defun bib-entry-extent-start-bp (bp &optional (n-forward 0) fixup-p)
  (let ((first-bp (interval-first-bp *interval*))
	(start-line (bib-entry-extent-start-line (bp-line bp) n-forward)))
    (cond ((null start-line)
	   (if fixup-p (copy-bp first-bp) nil))
	  (t
	   (and start-line
		(if (eq start-line (bp-line first-bp))
		    (copy-bp first-bp)
		    (create-bp start-line 0)))))))

(defun bib-entry-end-bp (bp &optional (n-forward 0) fixup-p)
  (multiple-value-bind (end-line end-of-buffer-flag)
      (bib-entry-end-line (bp-line bp) n-forward)
    (if (null end-line)
	(let ((last-bp (interval-last-bp *interval*))
	      (first-bp (interval-first-bp *interval*)))
	  (if end-of-buffer-flag
	      (copy-bp last-bp)
	      (if fixup-p (copy-bp first-bp) nil)))
	(create-bp end-line 0))))

(defun beyond-last-char-of-bib-entry-p (bp)
  (let* ((end-bp (bib-entry-end-bp bp))
	 (last-char-bp (backward-over *bibtex-whitespace-chars* end-bp)))
    (not (bp-< bp last-char-bp))))


(defun looking-at-empty-bib-field (bp &optional no-whitespace-p)
  (ignore-errors
    (and (= (list-syntax (bp-char bp)) list-open)
	 (let ((next (if no-whitespace-p
			 (forward-char bp)
			 (forward-over *bibtex-whitespace-chars* (forward-char bp)))))
	   (= (list-syntax (bp-char next)) list-close)))))

(defmajor com-bibtex-mode bibtex-mode "BibTeX"
          "Enter a mode for editing BibTeX .BIB files.

Command characters:

  m-n    = Next field           m-p   = Previous field
  c-m-n  = Next empty field     c-m-p = Previous empty field
  m-e    = End of field         m-a   = Beginning of field
  c-m-e  = End of entry         c-m-a = Beginning of entry
  m-h    = Mark field           c-m-h = Mark entry
  c-m-c  = Create new entry     c-m-. = Cleanup entry
  c-m-r  = Reexpand entry       m-'   = Switch between string and abbrev field
  m-Help = Document field       c-m-Help = Document entry

m-X commands:

  m-X Sort BibTeX Entries in Region    - sort a group of entries by citation key
  m-X Cleanup BibTeX Entries in Region - like c-m-. on each entry in region

A typical sequence:  c-m-c to create an empty template for a entry, followed by many
 calls to m-n (or c-m-n) to move to each empty field and fill in a value, followed
 by c-m-. to remove any unfilled optional fields.

 c-m-r restores the empty fields after you do a c-m-.

 m-' is handy when the value for a field is to be an abbreviation (see the LaTeX manual).

 m-Help and c-m-Help tell you what a field or entry type is supposed to be used for."
	  ()
  (set-comtab *mode-comtab* 
	      '(#\m-n com-next-bibtex-field
		#\m-p com-previous-bibtex-field
		#\c-m-n com-next-empty-bibtex-field
		#\c-m-p com-previous-empty-bibtex-field
		#\c-m-a com-beginning-of-bibtex-entry
		#\c-m-e com-end-of-bibtex-entry
		#\c-m-[ com-beginning-of-bibtex-entry
		#\c-m-] com-end-of-bibtex-entry
		#\c-m-h com-mark-bibtex-entry
		#\c-m-c com-create-bibtex-entry
		#\c-m-. com-cleanup-bibtex-entry
		#\c-m-r com-reexpand-bibtex-entry
		#\c-m-Help com-document-bibtex-entry
		#\m-h   com-mark-bibtex-field
		#\m-\'  com-toggle-bibtex-symbol-field
		#\m-a   com-beginning-of-bibtex-field
		#\m-e   com-end-of-bibtex-field
		#\m-Help com-document-bibtex-field)
	      '(("Sort BibTeX Entries in Region" . com-sort-bibtex-entries-in-region)
		("Cleanup BibTeX Entries in Region" . com-cleanup-bibtex-entries-in-region)))
  ;; Tab hacking rubout.
  (setq *space-indent-flag* t)
  (setq *paragraph-delimiter-list* nil)
  (setq *comment-begin* "% ")
  (setq *comment-start* "%")
  (setq *comment-end* "")
  #+TI
  (setq *mode-list-syntax-table* *bibtex-list-syntax-table*)
  #+TI
  (setq *mode-word-syntax-table* *bibtex-word-syntax-table*)
  #+Symbolics
  (set-syntax-table-indirection *mode-list-syntax-table* *bibtex-list-syntax-table*)
  #+Symbolics
  (set-syntax-table-indirection *mode-word-syntax-table* *bibtex-word-syntax-table*))


(set-comtab *standard-comtab* nil (make-command-alist '(com-bibtex-mode)))

(fs:define-canonical-type :bib "BIB")

#+Symbolics
(unless (assq :bib fs:*file-type-mode-alist*)
  (push '(:bib . :bibtex) fs:*file-type-mode-alist*))
#+TI
(unless (zlc:assq :bib fs:*file-type-mode-alist*)
  (push '(:bib . :bibtex) fs:*file-type-mode-alist*))

(defcom com-mark-bibtex-entry "Puts point and mark around current BibTeX entry.
With a positive numeric argument (n), puts point and
mark around n successive entries beginning with the
current one.  With a negative argument (-n), puts point
and mark around n successive entries ending with the
current one." ()
  (let ((start-bp (bib-entry-extent-start-bp (point) (min 0 (+ *numeric-arg* 1))))
	(end-bp (bib-entry-end-bp (point) (max 0 (- *numeric-arg* 1)))))
    (if (and (null start-bp) (null end-bp))
	(barf)
	(progn
	  (setq start-bp (or start-bp (interval-first-bp *interval*)))
	  (setq end-bp (or end-bp (interval-last-bp *interval*)))
	  (setf (window-mark-p *window*) t)
	  (setq *mark-stays* t)
	  (point-pdl-push (point) *window* nil nil)
	  (move-bp (point) start-bp)
	  (move-bp (mark) end-bp))))
  dis-bps)

(defcom com-beginning-of-bibtex-entry "Moves to the beginning of the current BibTeX entry.
It places the point at the atsign (@) which begins the entry.
With a numeric argument (n), it moves back to the
beginning of the nth previous entry." (km)
  (let* ((n-forward (let ((current-start-bp (bib-entry-start-bp (point) 0 t)))
		      (cond ((bp-= current-start-bp (point))
			     (- *numeric-arg*))
			    ((bp-< current-start-bp (point))
			     (if (minusp *numeric-arg*)
				 (- *numeric-arg*)
				 (- 1 *numeric-arg*)))
			    (t
			     (if (minusp *numeric-arg*)
				 (- -1 *numeric-arg*)
				 (- *numeric-arg*))))))
	 (new-bp (if (zerop *numeric-arg*)
		     (point)
		     (bib-entry-start-bp (point) n-forward t))))
    (when (if (minusp *numeric-arg*)
	      (not (bp-< (point) new-bp))
	      (not (bp-< new-bp (point))))
      (barf))
    (point-pdl-push (point) *window*)
    (move-bp (point) new-bp))
  dis-bps)

(defcom com-end-of-bibtex-entry "Moves to the end of the current BibTeX entry.
With a numeric argument (n), it moves to
the end of the nth succeeding entry." (km)
  (let* ((n-forward (if (> *numeric-arg* 0)
			(- *numeric-arg* 1)
			*numeric-arg*))
	 (new-bp (or (bib-entry-end-bp (point) n-forward)
		     (barf))))
    (when (if (minusp *numeric-arg*)
	      (not (bp-< new-bp (point)))
	      (not (bp-< (point) new-bp)))
      (barf))
    (point-pdl-push (point) *window*)
    (move-bp (point) new-bp))
  dis-bps)

#+TI
(eval-when (compile load eval)
(defmacro with-bibtex-interval-output-stream ((stream-var bp) &body body)
  (let ((bp-var (gensym)))
    `(let* ((,bp-var ,bp)
	    (,stream-var (interval-stream ,bp-var ,bp-var)))
       ,@body)))
)

(defcom com-create-bibtex-entry "Creates an empty BibTeX entry.
Prompts for the type of entry, and inserts an empty template for an
entry of that type, at the place where the point currently is.
If point is in the middle of another entry, first moves before it." ()
  (let ((entry-type-name
	  #+Symbolics
	  (typein-line-accept `((dw:menu-choose :alist ,*bib-entry-type-completion-alist*)
				:description "Bibtex entry type")
			      :prompt "Bibtex entry type")
	  #+TI
	  (cdr (completing-read-from-mini-buffer 
		 "BibTeX entry type" *bib-entry-type-completion-alist* nil nil
		 "You are typing in the name of a BibTeX entry type.")))
	(where (if (beyond-last-char-of-bib-entry-p (point))
		   (if (eq (bibtex-line-type (bp-line (point))) :blank)
		       (point)
		       (or (forward-line (point))
			   ;; Case when buffer does not end with blank line.
			   (insert (point) (format nil "~%"))))
		   (bib-entry-extent-start-bp (point)))))
    (let ((start-bp nil))
      (#+Symbolics with-interval-stream #+TI with-bibtex-interval-output-stream (s where)
	;; Don't put extra blank line at top of buffer.
	(unless (bp-= where (interval-first-bp *interval*))
	  (format s "~%"))
	(setq start-bp (send s :read-bp))
	(write-bib-entry entry-type-name s
			 :empty-fields-p t
			 :missing-fields-p t
			 :commentsp t)
	(format s "~%")
	(unless (eq (bibtex-line-type (bp-line (send s :read-bp))) :blank)
	  (format s "~%")))
      (let ((citation-bp (parse-bib-entry-search start-bp :citation)))
	(when citation-bp (move-bp (point) citation-bp)))))
  dis-text)

(defcom com-cleanup-bibtex-entry "Delete all comments and empty fields.
When a entry is expanded all possible fields for that entry are displayed.
You probably don't want to enter information for all of the fields displayed.
After the entry is completed to suit your needs, type c-m-. to delete empty
optional and empty satisfied alternative entries.  If you have forgotten one
of the required fields, you will be warned and the cursor left at the required
field needing information.  Enter the missing information and retype c-m-..
When the entire entry is acceptable, no message will be displayed and the
cursor will be at the end of the entry." ()
  (let ((start (bib-entry-start-bp (point)))
	(end (backward-over *bibtex-whitespace-chars* (bib-entry-end-bp (point)))))
    (multiple-value-bind (entry-type-name citation-key field-alist)
	(parse-bib-entry start)
      (flet ((move-to (&rest args)
	       (let ((bp (#+Symbolics lexpr-funcall #+TI apply #'parse-bib-entry-search start args)))
		 (when bp 
		   (if (and (eq (first args) :field)
			    (= (list-syntax (bp-char bp)) list-open))
		       (move-bp (point) (forward-char bp))
		       (move-bp (point) bp)))))
	     (field-has-value-p (field)
	       (let ((entry (assoc field field-alist)))
		 (and entry 
		      (or (symbolp (cdr entry))
			  (> (#+Symbolics string-length #+TI length (cdr entry)) 0))))))
	(let ((entry-type (bib-entry-type-named entry-type-name)))
	  
	  ;; Do nothing if this is just a @STRING command
	  (unless (eq entry-type-name :string)

	    ;; Make sure we know about this kind of entry.
	    (when (null entry-type)
	      (barf "~a is not a known BibTeX entry type." entry-type-name))
	  
	    ;; Make sure we have a citation key.
	    (when (null citation-key)
	      (move-to :citation)
	      (barf "Missing citation key."))
	  
	    ;; Verify the required fields.
	    (loop for required in (bib-entry-type-required-fields entry-type) do
	      (if (symbolp required)
		  (when (not (field-has-value-p required))
		    (move-to :field required)
		    (barf "Missing value for ~a field." required))
		  (let ((n-supplied (loop for r in (cdr required)
					  count (field-has-value-p r))))
		    (cond ((= n-supplied 0)
			   (move-to :field (second required))
			   (if (eq (first required) :or)
			       (barf "You must specify at least one of ~{~a~^ or ~}."
				     (cdr required))
			       (barf "You must specify one of ~{~a~^ or ~}."
				     (cdr required))))
			  ((and (> n-supplied 1)
				(eq (first required) :xor))
			   (move-to :field (second required))
			   (barf "You may only specify one of ~{~a~^ or ~}."
				 (cdr required)))))))

	  (let ((new-bp (delete-interval start end)))
	    (#+Symbolics with-interval-stream #+TI with-bibtex-interval-output-stream (s new-bp)
	      (write-bib-entry entry-type-name s
			       :citation-key citation-key
			       :field-alist field-alist
			       :commentsp nil
			       :missing-fields-p nil
			       :empty-fields-p nil
			       :extra-fields-p t)
	      (move-bp (point) (send s :read-bp)))))))))
  dis-text)

(defcom com-reexpand-bibtex-entry "Restore any missing fields.
After typing c-m-., all of the empty fields will have been deleted.  Use
this command to reinstate these them.  Typing c-m-R will redisplay all of the
blank fields." ()
  (let ((start (bib-entry-start-bp (point)))
	(end (backward-over *bibtex-whitespace-chars* (bib-entry-end-bp (point)))))
    (multiple-value-bind (entry-type-name citation-key field-alist)
	(parse-bib-entry start)
      (let ((new-bp (delete-interval start end)))
	(#+Symbolics with-interval-stream #+TI with-bibtex-interval-output-stream (s new-bp)
	  (write-bib-entry entry-type-name s
			   :citation-key citation-key
			   :field-alist field-alist
			   :commentsp t
			   :missing-fields-p t
			   :empty-fields-p t
			   :extra-fields-p t)
	  ;; This is a pretty gross way of moving the point to the first empty field.
	  ;; The IGNORE-ERRORS prevents it from barfing (in the ZWEI sense of the word).
	  (ignore-errors
	    (move-bp (point) new-bp)
	    (let ((*numeric-arg* 1))
	      (com-next-empty-bibtex-field)))))))
  dis-text)

(defcom com-next-bibtex-field "Forward to next bibliographic field.
Moves forward to the beginning of the next field's value.  With a positive
argument, moves that many fields forward (but always staying within current
entry).  With a negative argument, go that manay blank fields backward.
See also c-m-N, which moves to blank fields." (km)
  (cond ((zerop *numeric-arg*) dis-bps)
	((minusp *numeric-arg*)
	 (let ((*numeric-arg* (- *numeric-arg*)))
	   (com-previous-bibtex-field)))
	(t
	 (let* ((entry-start (bib-entry-start-bp (point)))
		(current (if (= (list-syntax (bp-char (point))) list-open)
			     (forward-char (point) -1 t)
			     (point))))
	   (loop for i from 1 to *numeric-arg* do
	     (let ((next (parse-bib-entry-search entry-start :after current)))
	       (if (null next)
		   (barf)
		   (setq current next))))
	   (let ((final (if (= (list-syntax (bp-char current)) list-open)
			    (forward-char current)
			    current)))
	     (move-bp (point) final)
	     dis-bps)))))

(defcom com-previous-bibtex-field "Backward to previous bibliographic field.
Moves backward to the beginning of the previous field's value.  With a positive
argument, moves that many fields backward (but always staying within current
entry).  With a negative argument, go that manay blank fields forward.
See also c-m-P, which moves to blank fields." (km)
  (cond ((zerop *numeric-arg*) dis-bps)
	((minusp *numeric-arg*)
	 (let ((*numeric-arg* (- *numeric-arg*)))
	   (com-next-bibtex-field)))
	(t
	 (let* ((entry-start (bib-entry-start-bp (point)))
		(current (if (= (list-syntax (bp-char (forward-char (point) -1))) list-open)
			     (forward-char (point) -1 t)
			     (point))))
	   (loop for i from 1 to *numeric-arg* do
	     (let ((next (parse-bib-entry-search entry-start :before current)))
	       (if (null next)
		   (barf)
		   (setq current next))))
	   (let ((final (if (= (list-syntax (bp-char current)) list-open)
			    (forward-char current)
			    current)))
	     (move-bp (point) final)
	     dis-bps)))))

(defcom com-next-empty-bibtex-field "Forward to next blank bibliographic field.
Like m-N, but skips fields that aren't empty.
With a positive argument, go that many blank fields forward (but always
staying within current entry).  With a negative argument, go that many
blank fields backward." (km)
  (cond ((zerop *numeric-arg*) dis-bps)
	((minusp *numeric-arg*)
	 (let ((*numeric-arg* (- *numeric-arg*)))
	   (com-previous-empty-bibtex-field)))
	(t
	 (let* ((entry-start (bib-entry-start-bp (point)))
		(current (if (= (list-syntax (bp-char (point))) list-open)
			     (forward-char (point) -1 t)
			     (point))))
	   (loop with i = 0 do
	     (let ((next (parse-bib-entry-search entry-start :after current)))
	       (if (null next)
		   (barf)
		   (setq current next))
	       (when (looking-at-empty-bib-field next)
		 (incf i)))
		 until (= i *numeric-arg*))
	   (let ((final (if (= (list-syntax (bp-char current)) list-open)
			    (forward-char current)
			    current)))
	     (move-bp (point) final)
	     dis-bps)))))

(defcom com-previous-empty-bibtex-field "Backward to previous blank bibliographic field.
Like m-P, but skips fields that aren't empty.
With a positive argument, go that many blank fields backward (but always
staying within current entry).  With a negative argument, go that many
blank fields forward." (km)
  (cond ((zerop *numeric-arg*) dis-bps)
	((minusp *numeric-arg*)
	 (let ((*numeric-arg* (- *numeric-arg*)))
	   (com-next-empty-bibtex-field)))
	(t
	 (let* ((entry-start (bib-entry-start-bp (point)))
		(current (if (= (list-syntax (bp-char (forward-char (point) -1))) list-open)
			     (forward-char (point) -1 t)
			     (point))))
	   (loop with i = 0 do
	     (let ((next (parse-bib-entry-search entry-start :before current)))
	       (if (null next)
		   (barf)
		   (setq current next))
	       (when (looking-at-empty-bib-field next)
		 (incf i)))
		 until (= i *numeric-arg*))
	   (let ((final (if (= (list-syntax (bp-char current)) list-open)
			    (forward-char current)
			    current)))
	     (move-bp (point) final)
	     dis-bps)))))

(defcom com-document-bibtex-field "Document the BibTeX field the cursor is near.
With numeric argument, prompts for the name of the field instead." (km)
  (let ((field-name
	  (if *numeric-arg-p*
	      #+Symbolics
	      (typein-line-accept `((dw:menu-choose :alist ,*bib-field-completion-alist*)
				    :description "BibTeX field")
				  :prompt "BibTeX field")
	      #+TI
	      (cdr (completing-read-from-mini-buffer 
		     "BibTeX field" *bib-field-completion-alist* nil nil
		     "You are typing in the name of a BibTeX field."))
	      (let ((start-bp (bib-entry-start-bp (point))))
		(and start-bp
		     (parse-bib-entry-search start-bp :what-field (point)))))))
    (if (null field-name)
	(barf "You don't seem to be near any bibliographic field.")
	(let* ((field (bib-field-named field-name))
	       (string (and field (bib-field-string field)))
	       (documentation (and field (bib-field-documentation field))))
	  (if (null documentation)
	      (barf "No documentation available for the ~a field." field-name)
	      (format t "~&Documentation for the `~a' field:~2& ~a"
		      string documentation)))))
  dis-none)

(defcom com-document-bibtex-entry "Document the type of BibTeX entry the cursor is near.
With numeric argument, prompts for an entry type instead." (km)
  (let ((entry-type-name
	  (if *numeric-arg-p*
	      #+Symbolics
	      (typein-line-accept `((dw:menu-choose :alist ,*bib-entry-type-completion-alist*)
				    :description "BibTeX entry type")
				  :prompt "BibTeX entry type")
	      #+TI
	      (cdr (completing-read-from-mini-buffer 
		     "BibTeX entry type" *bib-entry-type-completion-alist* nil nil
		     "You are typing in the name of a BibTeX entry type."))
	      (let ((start-bp (bib-entry-start-bp (point))))
		(and start-bp
		     (parse-bib-entry-search start-bp :what-entry-type))))))
    (if (null entry-type-name)
	(barf "You don't seem to be near any bibliographic entry.")
	(let* ((entry-type (bib-entry-type-named entry-type-name))
	       (string (and entry-type (bib-entry-type-string entry-type)))
	       (documentation (and entry-type (bib-entry-type-documentation entry-type))))
	  (if (null documentation)
	      (barf "No documentation available for the entries of type ~a." entry-type-name)
	      (format t "~&Documentation for entries of type `~a':~2& ~a"
		      string documentation)))))
  dis-none)

(defcom com-toggle-bibtex-symbol-field "Removes the braces around a field value, so that an abbreviation may be used.
If no braces were there, does the opposite." ()
  (multiple-value-bind (field-start field-end)
      (find-bibtex-field-boundaries (point))
    (if (or (null field-start) (null field-end))
	(barf)
	(let ((moving-field-end (copy-bp field-end :moves)))
	  (if (= (list-syntax (bp-char field-start)) list-open)
	      (progn
		(delete-interval field-start (forward-char field-start 1))
		(when (= (list-syntax (bp-char (forward-char moving-field-end -1)))
			 list-close)
		  (delete-interval (forward-char moving-field-end -1) moving-field-end)))
	      (progn
		(insert field-start "{")
		(insert moving-field-end "}"))))))
  dis-text)

(defcom com-mark-bibtex-field "Puts point and mark around current BibTeX field contents." ()
  (multiple-value-bind (field-start field-end)
      (find-bibtex-field-boundaries (point))
    (if (or (null field-start) (null field-end))
	(barf)
	(progn
	  (when (= (list-syntax (bp-char field-start)) list-open)
	    (setq field-start (forward-char field-start 1)))
	  (when (= (list-syntax (bp-char (forward-char field-end -1))) list-close)
	    (setq field-end (forward-char field-end -1)))
	  (setf (window-mark-p *window*) t)
	  (setq *mark-stays* t)
	  (point-pdl-push (point) *window* nil nil)
	  (move-bp (point) field-start)
	  (move-bp (mark) field-end))))
  dis-bps)

(defcom com-beginning-of-bibtex-field "Move to beginning of current BibTeX field." ()
  (multiple-value-bind (field-start field-end)
      (find-bibtex-field-boundaries (point))
    (if (or (null field-start) (null field-end))
	(barf)
	(progn
	  (when (= (list-syntax (bp-char field-start)) list-open)
	    (setq field-start (forward-char field-start 1)))
	  (move-bp (point) field-start))))
  dis-bps)

(defcom com-end-of-bibtex-field "Move to end of current BibTeX field." ()
  (multiple-value-bind (field-start field-end)
      (find-bibtex-field-boundaries (point))
    (if (or (null field-start) (null field-end))
	(barf)
	(progn
	  (when (= (list-syntax (bp-char (forward-char field-end -1))) list-close)
	    (setq field-end (forward-char field-end -1)))
	  (move-bp (point) field-end))))
  dis-bps)

#+TI
(defmacro bibtex-with-region-bps ((start-bp-var end-bp-var) &body body)
  (let ((m-var (gensym))
	(p-var (gensym)))
    `(let ((,m-var (mark))
	   (,p-var (point)))
       (multiple-value-bind (,start-bp-var ,end-bp-var)
	   (if (bp-< ,m-var ,p-var)
	       (values ,m-var ,p-var)
	       (values ,p-var ,m-var))
	 ,@body))))

#+TI
(defmethod (interval-with-sort-key :sort-key) ()
  sort-key)

(defcom com-sort-bibtex-entries-in-region "Sorts the entries in the region by citation key.
If no region exists, the entire buffer is sorted.
@String commands are sorted according to the abbreviation they define,
but all @String commands will appear before any bibliographic entries (within
the region that is sorted)." ()
  (multiple-value-bind (start end)
      (if (window-mark-p *window*)
	  (#+Symbolics with-region-bps #+TI bibtex-with-region-bps (s e)
	    (values s e))
	  (values (interval-first-bp *interval*) (interval-last-bp *interval*)))
    (flet ((move-to-key-fn (bp)
	     (bib-entry-start-bp bp))
	   (get-key-fn (bp)
	     (let* ((entry-type (parse-bib-entry-search bp :what-entry-type))
		    (citation (forward-over *bibtex-whitespace-chars*
					    (parse-bib-entry-search bp :citation)))
		    (end-citation (bibtex-forward-balanced citation)))
	       (values end-citation
		       (if (eq entry-type :string)
			   (string-append " " (string-interval citation end-citation))
			   (string-interval citation end-citation)))))
	   (move-to-next-fn (bp)
	     (bib-entry-end-bp bp)))
      (bibtex-sort-interval-functions-with-key #'move-to-key-fn #'get-key-fn #'move-to-next-fn
					       #'(lambda (i1 i2)
						   (string-lessp (send i1 :sort-key)
								 (send i2 :sort-key)))
					       start end t)))
  dis-text)

(defcom com-cleanup-bibtex-entries-in-region "Delete all comments and empty fields for every entry in region.
If no region exists, the entire buffer is cleaned up.
This is like doing c-m-. on every entry in the region.
This command can go a long way toward converting a Scribe .BIB file into one
suitable for use with BibTeX, as it will substitute appropriate delimiters." ()
  (multiple-value-bind (start end)
      (if (window-mark-p *window*)
	  (#+Symbolics with-region-bps #+TI bibtex-with-region-bps (s e)
	    (values (copy-bp s) (copy-bp e)))
	  (values (interval-first-bp *interval*) (interval-last-bp *interval*)))
    (move-bp (point) start)
    (loop until (not (bp-< (point) end)) do
      (com-cleanup-bibtex-entry)
      (let ((next (bib-entry-start-bp (point) 1)))
	(if (null next)
	    (move-bp (point) end)
	    (move-bp (point) next)))))
  dis-text)

;;; This is more efficient than FORWARD-SEXP, especially for the TI (on the TI, a call to
;;; FORWARD-SEXP causes a scan from the beginning of the buffer!).  Moves past a delimiter,
;;; a sequence of alphabetics, a double-quoted string, or text balanced with respect to
;;; parens.  Does not consider balancing of parens within strings, nor balancing of double 
;;; quotes within balanced text (this is probably best for parsing bib fields).  It does,
;;; however, exclude parens or double quotes preceded by slashes from matching.
(defun bibtex-forward-balanced (from-bp)
  (let ((state 'begin)
	(stack '()))
    (charmap (from-bp (interval-last-bp *interval*) (interval-last-bp *interval*))
      (let ((char (charmap-char)))
	(#+Symbolics selectq #+TI zlc:selectq state
	  (begin
	   (if (member char *bibtex-whitespace-chars*)
	       nil
	       (select (list-syntax char)
		 (list-alphabetic
		  (setq state 'in-symbol))
		 (list-open
		  (push 'open stack)
		  (setq state 'in-balanced))
		 (list-double-quote
		  (setq state 'in-string))
		 (otherwise
		  (charmap-return (charmap-bp-after))))))
	  (in-symbol
	   (unless (= (list-syntax char) list-alphabetic)
	     (charmap-return (charmap-bp-before))))
	  (in-balanced
	   (select (list-syntax char)
	     (list-slash
	      (setq state 'slash-in-balanced))
	     (list-open
	      (push 'open stack))
	     (list-close
	      (do ((tos (pop stack) (pop stack)))
		  ((or (null stack) (eq tos 'open))))
	      (when (null stack)
		(charmap-return (charmap-bp-after))))))
	  (slash-in-balanced
	   (setq state 'in-balanced))
	  (in-string
	   (select (list-syntax char)
	     (list-slash
	      (setq state 'slash-in-string))
	     (list-double-quote
	      (charmap-return (charmap-bp-after)))))
	  (slash-in-string
	   (setq state 'in-string))
	  (finish
	   (charmap-return (charmap-bp-before))))))))

;;; Fixes to ZWEI routines.

#-Symbolics
(defun bibtex-sort-interval-functions-with-key (&rest args)
  (apply #'sort-interval-functions-with-key args))
#+Symbolics
(DEFUN bibtex-SORT-INTERVAL-FUNCTIONS-WITH-KEY
       (MOVE-TO-KEY-FN GET-KEY-FN MOVE-TO-NEXT-FN LESSP-FN FROM-BP &OPTIONAL TO-BP IN-ORDER-P)
  (GET-INTERVAL FROM-BP TO-BP IN-ORDER-P)
  (LET ((ARRAY (MAKE-ARRAY 20. ':FILL-POINTER 0)))
    (DO ((*INTERVAL* (CREATE-INTERVAL (COPY-BP FROM-BP ':NORMAL) (COPY-BP TO-BP ':MOVES)))
	 (START-BP FROM-BP END-BP)
	 (KEY-START-BP) (KEY-END-BP) (KEY) (END-BP))
	((BP-= START-BP TO-BP))
      (SETQ KEY-START-BP (FUNCALL MOVE-TO-KEY-FN START-BP))
      (MULTIPLE-VALUE (KEY-END-BP KEY)
	(FUNCALL GET-KEY-FN KEY-START-BP))
      (SETQ END-BP (FUNCALL MOVE-TO-NEXT-FN KEY-END-BP))
      ;; Symbolics has ARRAY-PUSH instead of ARRAY-PUSH-EXTEND.
      (ARRAY-PUSH-extend ARRAY (MAKE-INSTANCE 'INTERVAL-WITH-SORT-KEY
				       ':FIRST-BP START-BP
				       ':LAST-BP END-BP
				       ':SORT-KEY KEY)))
    (SORT-INTERVAL-ARRAY ARRAY LESSP-FN FROM-BP TO-BP T)))

#-TI
(defun bibtex-line-type (line) (line-type line))
#+TI
(DEFUN bibtex-line-type (LINE)
  "Return a keyword classifying LINE's contents.
The value is :BLANK, :COMMENT, :ATOM, :NORMAL or :DIAGRAM."
  (IF (GET (LOCF (LINE-PLIST LINE)) ':DIAGRAM) ':DIAGRAM
      (DO ((I (IF (EQ LINE (BP-LINE (INTERVAL-FIRST-BP *INTERVAL*)))
		  (BP-INDEX (INTERVAL-FIRST-BP *INTERVAL*))
		  0)
	      (1+ I))
	   (LIM (IF (EQ LINE (BP-LINE (INTERVAL-LAST-BP *INTERVAL*)))
		    (BP-INDEX (INTERVAL-LAST-BP *INTERVAL*))
		    (LINE-LENGTH LINE))))
	  ((>= I LIM) ':BLANK)
	(LET ((CH (make-char (AREF LINE I))))
	  (OR (zlc:memq CH *BLANKS*)
	      ;; In TI's version of LINE-TYPE, it checks for semicolon instead of for
	      ;; a character with comment syntax.
	      (RETURN (COND ((= (list-syntax ch) list-comment) ':COMMENT)
			    ((= CH #\FF) ':FORM)
			    ((= (LIST-SYNTAX CH) LIST-ALPHABETIC) ':ATOM)
			    (T ':NORMAL))))))))


;;;; Edit History: O:>utilities>bibtex-mode.lisp.newest

;;; [KT] 12/05/88 14:31  Added BIBTEX-FORWARD-BALANCED; fixed Symbolics sorting problem.
