;;; -*- Mode:Common-Lisp; Package:W; Base:10 -*-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;  Adapted from Steve Ward's sidekick clock

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defconstant 2pi 6.2832s0)

(defconstant *clock-default-size* 130)

(defflavor analog-clock-window
	   (
	    center-x
	    center-y
	    radius
	    hourpos
	    minpos
	    secpos
	    hour-outline
	    min-outline
	    sec-outline
	    (second-hand? t)
	    )
	   (w:process-mixin
	    w:graphics-mixin
	    w:minimum-window
	    )
  (:inittable-instance-variables second-hand?)
  (:default-init-plist
    :process '(clock)
    :expose-p t
    :deexposed-typeout-action :permit
    :save-bits t
    :blinker-p nil
    :width *clock-default-size*
    :height *clock-default-size*))

(defmethod (analog-clock-window :after :init) (&rest ignore)
  (send self :initialize))

(defmethod (analog-clock-window :screen-manage-deexposed-visibility) ()
  t)

(defmethod (analog-clock-window :mouse-click) (char x y)
  (ignore x y)
  (case char
    (#\mouse-l-1
     (send self :refresh)
     t)
    (#\mouse-l-2
     (setq second-hand? (not second-hand?))
     (send self :refresh)
     t)
    (#\mouse-m-2
     (send self :set-reverse-video-p (not (send self :reverse-video-p)))
     t)
    (#\mouse-r-1
     (let ((size (min w:width w:height)))
       (process-run-function "Resize clock" self :set-size size size))
     t)
    (t nil)))

(defmethod (analog-clock-window :deexposed-mouse-buttons) (mask x y)
  (send self :mouse-buttons mask x y))

(defmethod (analog-clock-window :who-line-documentation-string) ()
  `(:mouse-l-1 "Refresh clock"
    :mouse-l-2 "Toggle second hand"
    ,@(when (boundp 'w:*dragging-enabled?*) '(:mouse-m-hold "Drag window"))
    :mouse-m-2 "Toggle reverse video"
    :mouse-r-1 "Make square"))

(defmethod (analog-clock-window :deexposed-who-line-documentation-string) ()
  (send self :who-line-documentation-string))

(defmethod (analog-clock-window :verify-new-edges)
	   (new-left new-top new-width new-height)
  (ignore new-left new-top)
  (when (or (< new-width 50) (< new-height 50))
    "Clock size too small"))

(defmethod (analog-clock-window :after :change-of-size-or-margins) (&rest ignore)
  (send self :initialize)
  (send self :refresh))

(defmethod (analog-clock-window :xor-line) (from-x from-y to-x to-y)
  "Xor line without flashing cursor (for second hand)"
  (setq from-x (+ from-x (tv:sheet-inside-left self))
	from-y (+ from-y (tv:sheet-inside-top self))
	to-x (+ to-x (tv:sheet-inside-left self))
	to-y (+ to-y (tv:sheet-inside-top self)))
  (without-interrupts
    (let ((w:mouse-blinker nil))	   ; prevent the mouse blinker from blinking
      (w:prepare-sheet (self)
	(w:%draw-line from-x from-y to-x to-y w:opposite t self)))))

(defun clock (window)
  (send (send window :process) :set-priority -5)
  (send window :refresh)
  (send window :run))

(defmethod (analog-clock-window :initialize) ()
  (setq radius (- (lsh (min w:height w:width) -1) 10)
	center-x (lsh w:width -1)
	center-y (lsh w:height -1))
  (let ((hour-length (round (* radius .5)))
	(min-length (round (* radius .80)))
	(sec-length (round (* radius .75)))
	(hour-width (round (* radius .09)))
	(min-width (round (* radius .07)))
	(sec-back (round (* radius .25))))
    (setq hour-outline (make-hand min-width hour-width hour-length
				  center-x center-y)
	  min-outline (make-hand min-width min-width min-length
				 center-x center-y)
	  sec-outline (make-hand sec-back 0 sec-length
				 center-x center-y))))

(defmethod (analog-clock-window :run) ()
  (loop
    (send self :show-time)
    (process-sleep 58)))

(defmethod (analog-clock-window  :show-time) ()
  (multiple-value-bind (sec min hou)
      (time:get-time)
    (when (not sec) (setq sec 1 min 0 hou 0))
    (setq hou (truncate (rem hou 12.)))
    (and (zerop hou) (setq hou 12.))
    (send self :MinHand min)
    (send self :HourHand (mod (+ (* hou 5) (round min 12)) 60))
    (send self :SecHand sec)
    ; So that window manager will refresh clock
    (setf (w:sheet-output-hold-flag self) 0)
  ))

(defmethod (analog-clock-window :refresh) (&rest ignore)
  (send self :clear-screen)
  (send self :draw-rectangle 0 0 w:width w:height)
  (send self :draw-filled-rectangle
	3 3 (- w:width 6) (- w:height 6) tv:50%-gray-color)
  (send self :draw-filled-circle Center-x Center-y Radius w:white)
  (send self :draw-circle Center-x Center-y Radius 2 w:black)
  (send self :clocklab 11 -2 0 "11")
  (send self :clocklab 0 -4 0 "12")
  (dotimes (i 10)
    (send self :clocklab (1+ i) 0 0 (format nil "~d" (1+ i))))
  (setq HourPos nil)
  (setq MinPos nil)
  (setq SecPos nil)
  )

(defmethod (analog-clock-window :clocklab) (point dx dy text)
  (let ((ang (* 2pi (/ (- 90 (* point 5)) 60.0)))
	(len (* .85 radius)))
    (let ((px (- (truncate (+ Center-x (* (sin ang) len))) 4))
	  (py (- (truncate (+ Center-y (* (cos ang) len))) 3)))
      (send self :string-out-explicit
	    text (+ px dx) (+ py dy)
	    (+ px 100) (+ py 100)
	    fonts:tr8 w:opposite))))

(defstruct clock-hand dxy len dlen ary)

(defun make-hand (dlen dxy len center-x center-y
		  &aux (ary (make-array '(60 6) :type 'art-16b)))
  (loop for point from 0 below 60 do
	(make-hand-position point dlen dxy len ary center-x center-y))
  (make-clock-hand
    :ary ary
    :dxy dxy
    :len len
    :dlen dlen))

(defun make-hand-position (point dlen dxy len ary center-x center-y
			   &aux s c ang px py zx zy wdx wdy)
  (setq ang (* 2pi (/ (rem (- 90 point) 60) 60.0)))
  (setq s (sin ang))
  (setq c (cos ang))
  (setq px (round (+ Center-x (* s len))))		; extremal point.
  (setq py (round (+ Center-y (* c len))))
  (setq zx (round (+ Center-x (- (* s dlen)))))	; back point.
  (setq zy (round (+ Center-y (- (* c dlen)))))
	  (setq wdx (- (round (* c dxy))))
	  (setq wdy (round (* s dxy)))
  (if (zerop dxy)
      (progn
	(setf (aref ary point 0) zx)
	(setf (aref ary point 1) zy)
	(setf (aref ary point 2) px)
	(setf (aref ary point 3) py))

      (progn
	(setq wdx (- (round (* c dxy))))
	(setq wdy (round (* s dxy)))
	(setf (aref ary point 0) (+ zx wdx))
	(setf (aref ary point 1) (+ zy wdy))
	(setf (aref ary point 2) (- zx wdx))
	(setf (aref ary point 3) (- zy wdy))
	(setf (aref ary point 4) px)
	(setf (aref ary point 5) py))))

(defmethod (analog-clock-window :draw-hand) (hand point aluf &optional outline)
  (let ((ary (clock-hand-ary hand)))
    (if (zerop (clock-hand-dxy hand))
	(send self :xor-line
	      (aref ary point 0)
	      (aref ary point 1)
	      (aref ary point 2)
	      (aref ary point 3))
	(progn
	  (send self :draw-filled-triangle
		(aref ary point 0)
		(aref ary point 1)
		(aref ary point 2)
		(aref ary point 3)
		(aref ary point 4)
		(aref ary point 5)
		w:black aluf nil nil nil)
	  (if (null outline)
	      nil
	      (send self :draw-triangle
		    (aref ary point 0)
		    (aref ary point 1)
		    (aref ary point 2)
		    (aref ary point 3)
		    (aref ary point 4)
		    (aref ary point 5)
		    1 w:black outline)
	      )))))

(defmethod (analog-clock-window :HourHand) (n)
  (send self :SecHand nil)
  (unless (equal HourPos n)
    (unless (null HourPos)
      (send self :draw-hand hour-outline HourPos tv:alu-andca)
      (setq HourPos nil)
      (unless (null MinPos)
	(let ((m minpos))
	  (send self :minhand nil)
	  (send self :MinHand m))))
    (if (null n)
	(setq HourPos nil)
	(send self :draw-hand hour-outline (setq HourPos n) w:normal w:erase))))

(defmethod (analog-clock-window :MinHand) (n)
  (send self :SecHand  nil)
  (unless (equal MinPos n)
    (unless (null MinPos)
      (send self :draw-hand min-outline MinPos tv:alu-andca)
      (setq MinPos nil)
      (unless (null HourPos)
	(send self :HourHand nil)))
    (if (null n)
	(setq MinPos nil)
	(send self :draw-hand min-outline (setq MinPos n) w:normal))))

(defmethod (analog-clock-window :SecHand) (n)
  (when second-hand?
    (or (equal SecPos n)				; already set right.
	(progn (or (null SecPos)			; Remove previous.
		   (send self :draw-hand sec-outline SecPos w:opposite))
	       (if (null n)(setq SecPos nil)
		   (send self :draw-hand
			 sec-outline (setq SecPos n) w:opposite))))))





