/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: client.c,v 1.22 1994/02/09 19:41:24 janssen Exp $
*/

#include "m3.h"

extern void SizeArgument(Argument arg, Context context);

static cardinal MethodIndex = 0;

typedef struct _class_node_s {
  Type type;
  boolean leaf;
  cardinal tree_id;
  list parents;		/* list of ClassNode */
  list children;	/* list of ClassNode */
} ClassNode;

static list AllNodes = NULL;

static ClassNode *newClassNode (Type t)
{
  ClassNode *n = (ClassNode *) Malloc(sizeof(ClassNode));

  n->type = t;
  n->leaf = FALSE;
  n->parents = new_list();
  n->children = new_list();

  if (AllNodes == NULL)
    fprintf (stderr, "Null AllNodes\n");
  else
    list_insert(AllNodes, n);

  return (n);
}

static boolean CompareTypes (ClassNode *c, Type t)
{
  return (c->type == t);
}

ClassNode *FindPreviouslyCreatedNode (Type t)
{
  if (AllNodes == NULL)
    return (NULL);
  else
    return ((ClassNode *) list_find (AllNodes, (FindProc) CompareTypes, t));
}

static ClassNode * AddNode (Type st, ClassNode *n)
{
  Type ut;
  ClassNode *nn;

  ut = ultimateType(st);
  if (ut->marked)
    nn = FindPreviouslyCreatedNode(ut);
  else
    {
      nn = newClassNode (ut);
      list_enumerate (class_object(ut)->superclasses, (void (*)(void *, void *)) AddNode, nn);
      ut->marked = TRUE;
    }
  if (n != NULL)
    {
      list_insert (n->parents, nn);
      list_insert (nn->children, n);
    }
  return (nn);
}

static void freeEdges (ClassNode *n)
{
  list_clear (n->parents, FALSE);
  Free(n->parents);
  list_clear (n->children, FALSE);
  Free(n->children);
}

static cardinal TreeID;

static void MarkLeaves2 (ClassNode *n, int *c)
{
  int count = 0;
  if (n->type->marked)
    return;
  else
    *c += 1;
  n->tree_id = TreeID;
  n->type->marked = TRUE;
  list_enumerate (n->children, (EnumProc) MarkLeaves2, &count);
  if (count == 0)
    n->leaf = TRUE;
  else
    n->leaf = FALSE;
}

static void MarkLeaves (ClassNode *r, Type t)
{
  int count;
  TreeID = 0;
  if (list_size(r->parents) == 0	/* root */
      AND !r->type->marked)
    {
      TreeID += 1;
      MarkLeaves2 (r, &count);
    }
}

static void FindFollowers (ClassNode *r, list l)
{
  if (r->leaf)
    list_insert(l, r);
}

list GetFollowers (Type t)
{
  ClassNode *n;
  list l;

  if (t == NULL OR (type_basic_type(t) != object_Type))
    return (NULL);

  UnmarkSupertypes(t);

  if (AllNodes == NULL)
    AllNodes = new_list();
  else
    {
      list_enumerate (AllNodes, (EnumProc) freeEdges, NULL);
      list_clear (AllNodes, TRUE);
    }
  n = AddNode (t, NULL);	/* actually builds graph */
  UnmarkSupertypes (t);
  list_enumerate (AllNodes, (EnumProc) MarkLeaves, t);
  l = new_list();	/* list of follower nodes */
  list_enumerate (n->parents, (EnumProc) FindFollowers, l);

  return (l);
}

static void GenerateFollowerType (ClassNode *f, Context context)
{
  Type follower = ultimateType(f->type);
  char *un;
  char *tn;

  if (context->class == follower)
    return;

  un = UnionTypeName(context->class, follower);
  tn = M3_SHORT_TYPE_NAME(context->class);

  fprintf (context->file, "TYPE Following__%s = %s_x.Surrogate__%s OBJECT\n",
	   un, m3_interface_name(follower->interface), M3_SHORT_TYPE_NAME(follower));
  fprintf (context->file, "    ilu_the_leader : Lead__%s;\n", tn);
  fprintf (context->file, "  OVERRIDES\n");
  fprintf (context->file, "    ILU_Get_Type := ILU_Get_Type_%s;\n", tn);
  fprintf (context->file, "    ILU_Qua_Type := Qua_From_Following__%s\n  END;\n\n", un);

  fprintf (context->file, "PROCEDURE Qua_From_Following__%s (self: Following__%s; ot: Ilu.ObjectType) : Ilu.Object =\n",
	   un, un);
  fprintf (context->file, "  BEGIN\n    RETURN Qua_From_Lead_%s(self.ilu_the_leader, ot)\n", tn);
  fprintf (context->file, "  END Qua_From_Following__%s;\n\n", un);
}

static void generate_method_assignment (Procedure m, Context context)
{
  fprintf (context->file, "    %s := ", M3_SHORT_METHOD_NAME(m));
  if (ultimateType(m->object)->interface != context->interface)
    fprintf (context->file, "%s_x.", m3_interface_name(m->object->interface));
  fprintf (context->file, "Client__%s_%s;\n", M3_SHORT_TYPE_NAME(m->object), M3_SHORT_METHOD_NAME(m));
}

static void GenerateMethodOverrides (Type st, Context context)
{
  Class od;
  Type t;

  if (st == NULL OR (t = ultimateType(st)) == NULL
      OR type_basic_type(t) != object_Type OR (od = class_object(t)) == NULL)
    return;
  t->marked = TRUE;
  if (list_size(od->superclasses) > 0)
    GenerateMethodOverrides ((Type) list_car(od->superclasses), context);
  list_enumerate (od->methods, (EnumProc) generate_method_assignment, context);
}

static void GenerateFollowerSlot (ClassNode *f, Context context)
{
  Type follower = f->type;

  if (context->class == follower)
    return;

  fprintf (context->file, "    ilu_following__%s_%s: Following__%s;\n",
	   m3_interface_name(ultimateType(follower)->interface), M3_SHORT_TYPE_NAME(follower),
	   UnionTypeName(context->class, follower));
}

static void InitializeFollowerSlot (ClassNode *f, Context context)
{
  Type follower = f->type;

  if (context->class == follower)
    return;

  fprintf (context->file, ", ilu_following__%s_%s := NEW(Following__%s)",
	   m3_interface_name(ultimateType(follower)->interface), M3_SHORT_TYPE_NAME(follower),
	   UnionTypeName(context->class, follower));
}

static void InitFollowers (ClassNode *f, Context context)
{
  Type follower = f->type;

  if (context->class == follower)
    return;

  fprintf (context->file, "    x.ilu_following__%s_%s.ilu_the_leader := x;\n",
	   m3_interface_name(ultimateType(follower)->interface), M3_SHORT_TYPE_NAME(follower));
  fprintf (context->file, "    %s_x.Init_Surrogate__%s(x.ilu_following__%s_%s, server);\n",
	   m3_interface_name(ultimateType(follower)->interface), M3_SHORT_TYPE_NAME(follower),
	   m3_interface_name(ultimateType(follower)->interface), M3_SHORT_TYPE_NAME(follower));
}

static void ClimbQuaTree (ClassNode *follower, Context context)
{
  ClassNode *n;
  Type t;

  for (n = follower, t = ultimateType(follower->type);  n != NULL AND (!n->type->marked);  n = list_car(n->parents))
    {
      fprintf (context->file, "    ELSIF ot = %s_x.ObjectType_%s THEN\n",
	       m3_interface_name(ultimateType(n->type)->interface), M3_SHORT_TYPE_NAME(n->type));
      fprintf (context->file, "      RETURN self.ilu_following__%s_%s\n",
	       m3_interface_name(t->interface), M3_SHORT_TYPE_NAME(t));
      n->type->marked = TRUE;
    }
}

static void GenerateObjectSupport (Type type, Context context)
{
  char *tn;
  char *stn;
  list followers;
  Type n;

  tn = M3_TYPE_NAME(type);
  stn = M3_SHORT_TYPE_NAME(type);

  followers = GetFollowers(type);
  context->class = type;

  fprintf (context->file, "REVEAL Surrogate__%s = %s BRANDED OBJECT\n", stn, stn);
  fprintf (context->file, "    server: Ilu.SurrogateServer := NIL;\n");
  fprintf (context->file, "  OVERRIDES\n");
  GenerateMethodOverrides (type, context);
  fprintf (context->file, "    ILU_Get_Server := Get_Surrogate_Server_From_%s;\n", stn);
  fprintf (context->file, "  END;\n\n");

  fprintf (context->file, "TYPE Lead__%s = Surrogate__%s BRANDED OBJECT\n", stn, stn);
  list_enumerate (followers, (EnumProc) GenerateFollowerSlot, context);
  fprintf (context->file, "  OVERRIDES\n    ILU_Qua_Type := Qua_From_Lead_%s\n  END;\n\n", stn);

  list_enumerate (followers, (EnumProc) GenerateFollowerType, context);

  fprintf (context->file, "PROCEDURE Create_Lead__%s (<*UNUSED*> self: IluRuntime.ObjectCreator;\n",
	   stn);
  fprintf (context->file, "    server : Ilu.SurrogateServer) : Ilu.Object RAISES {} =\n");
  fprintf (context->file, "  VAR x := NEW(Lead__%s", stn);
  list_enumerate (followers, (EnumProc) InitializeFollowerSlot, context);
  fprintf (context->file, ");\n");
  fprintf (context->file, "  BEGIN\n");
  list_enumerate (followers, (EnumProc) InitFollowers, context);
  fprintf (context->file, "    Init_Surrogate__%s (x, server);\n", stn);
  fprintf (context->file, "    RETURN x;\n");
  fprintf (context->file, "  END Create_Lead__%s;\n\n", stn);

  fprintf (context->file, "PROCEDURE Init_Surrogate__%s (self: Surrogate__%s; server : Ilu.SurrogateServer)\n",
	   stn, stn);
  fprintf (context->file, "  RAISES {} =\n");
  fprintf (context->file, "  BEGIN\n");
  fprintf (context->file, "    self.ilu_is_surrogate := TRUE;\n    self.server := server;\n");
  fprintf (context->file, "  END Init_Surrogate__%s;\n\n", stn);

  fprintf (context->file, "PROCEDURE Qua_From_Lead_%s (self: Lead__%s; ot : IluRuntime.ObjectType) : Ilu.Object =\n",
	   stn, stn);
  fprintf (context->file, "  BEGIN\n");
  fprintf (context->file, "    IF ot = ObjectType_%s THEN\n      RETURN self\n", stn);
  UnmarkSupertypes (type);
  for (n = (Type) list_car(class_object(type)->superclasses);
       n != NULL;  n = (Type) list_car(class_object(n)->superclasses))
    {
      fprintf (context->file, "    ELSIF ot = %s_x.ObjectType_%s THEN\n",
	       m3_interface_name(ultimateType(n)->interface), M3_SHORT_TYPE_NAME(n));
      fprintf (context->file, "      RETURN self\n");
      n->marked = TRUE;
    }
  type->marked = TRUE;
  list_enumerate (followers, (EnumProc) ClimbQuaTree, context);
  fprintf (context->file, "    ELSE\n      RETURN NIL\n    END;\n");
  fprintf (context->file, "  END Qua_From_Lead_%s;\n\n", stn);

  fprintf (context->file, "PROCEDURE ILU_SBH_To_%s (sbh : TEXT; mostSpecificTypeID: TEXT) : %s\n",
	   stn, stn);
  fprintf (context->file, "  RAISES {IluBasics.Failed, Thread.Alerted} =\n");
  fprintf (context->file, "  VAR x := IluRuntime.M3ObjectFromSbh(sbh, ObjectType_%s, mostSpecificTypeID);\n", stn);
  fprintf (context->file, "  BEGIN\n");
  fprintf (context->file, "    TYPECASE x OF\n      | %s (xx) => RETURN xx;\n", tn);
  fprintf (context->file, "      ELSE RAISE IluBasics.Failed(NEW(IluBasics.Failure,");
  fprintf (context->file, " info := \"sbh \" & sbh & \" leads to non-%s object\"))\n", tn);
  fprintf (context->file, "    END (* typecase *);\n  END ILU_SBH_To_%s;\n\n", stn);

  fprintf (context->file, "PROCEDURE Get_Surrogate_Server_From_%s (self : Surrogate__%s): Ilu.Server =\n",
	   stn, stn);
  fprintf (context->file, "  BEGIN\n    RETURN self.server;\n  END Get_Surrogate_Server_From_%s;\n\n",
	   stn);
}

void DeclareCallerReturnValue (Type type, Context context)
{
  if (type_basic_type(type) != void_Type)
    {
      fprintf (context->file, "  VAR return_value_: %s;\n", M3_TYPE_NAME(type));
    }
}

void ListExceptionName (Exception e, Context context)
{
  fprintf (context->file, ", %s", M3_EXCEPTION_NAME(e));
}

void ListArgument (Argument arg, Context context)
{
  fprintf (context->file, "; %s: %s", m3_argument_name(arg), M3_TYPE_NAME(arg->type));
}

extern boolean IsCacheableMethod(Procedure m);

static void EncodeArgument (Argument arg, Context context)
{
  MarshallValue (arg->type, m3_argument_name(arg), context, 8);
}

static void RemoveFromList (Exception e, list l)
{
  list_remove(l, e);
}

static void AddToList (refany e, list l)
{
  list_insert(l, e);
}

static void list_copy (list to, list from)
{
  list_enumerate (from, (EnumProc) AddToList, to);
}

static void declare_unhandled_exceptions (list elist, Context context)
{
  list l = new_list();
  list_copy (l, context->interface->exceptions);
  list_enumerate (elist, (EnumProc) RemoveFromList, l);
  if (list_size(l) == 0)
    ;
  else
    {
      fprintf (context->file, "  <*FATAL %s", M3_EXCEPTION_NAME((Exception)list_car(l)));
      if (list_size(l) > 1)
	{
	  list c = (list) list_cdr(l);
	  list_enumerate(c, (EnumProc) ListExceptionName, context);
	  Free(c);
	}
      fprintf (context->file, "*>\n");
    }
  list_clear (l, FALSE);
  Free(l);
}

static void generate_method_code (Procedure m, Context context)
{
  enum PrimitiveTypes t = type_basic_type(m->returnType);
  char *stn = M3_SHORT_TYPE_NAME(m->object);

  /* print out formal params */

  fprintf (context->file, "PROCEDURE Client__%s_%s (self_: %s",
	   stn, M3_SHORT_METHOD_NAME(m), stn);
  list_enumerate (m->arguments, (EnumProc) ListArgument, context);
  fprintf (context->file, ")%s%s\n",
	   t != void_Type ? ": " : "",
	   t != void_Type ? M3_TYPE_NAME(m->returnType) : "");
  fprintf (context->file, "  RAISES {IluBasics.Failed, Thread.Alerted");
  if (list_size(m->exceptions) > 0)
    list_enumerate(m->exceptions, (EnumProc) ListExceptionName, context);
  fprintf (context->file, "} =\n");
  declare_unhandled_exceptions (m->exceptions, context);

  /* declare local variables */
/*  DeclareCallerReturnValue(m->returnType, context); */
  fprintf (context->file, "  VAR call_: IluRuntime.Call;\n");
  if (NOT m->asynch)
    fprintf (context->file, "  VAR scode_: INTEGER;\n");

  /* if method is functional, see if it is cached, and if so, return the cached value */

  if (IsCacheableMethod(m))
    {
    }

  /* start code */
  fprintf (context->file, "  BEGIN\n    call_ := IluRuntime.BeginCall (self_);\n");
  fprintf (context->file, "    IF call_ = NIL THEN\n      RAISE IluBasics.Failed(NEW(IluBasics.Failure, info := \"lost connection\"))\n    END;\n");
  fprintf (context->file, "    TRY\n      TRY\n");
  fprintf (context->file, "        IluRuntime.BeginRequest(call_, ObjectType_%s, ADR(%s_x.Methods__%s[%s_x.MethodNames__%s.%s]), 0",
	   stn,
	   m3_interface_name(m->object->interface), stn,
	   m3_interface_name(m->object->interface), stn,
	   M3_SHORT_METHOD_NAME(m));
  list_enumerate (m->arguments, (EnumProc) SizeArgument, context);
  if (NOT context->class->description->structuredDes.object->singleton)
    fprintf (context->file, "\n          + IluRuntime.SizeM3Object(call_, self_, TRUE, ObjectType_%s)",
	     M3_SHORT_TYPE_NAME(context->class));
  fprintf (context->file, ");\n");
  if (!context->class->description->structuredDes.object->singleton)
    fprintf (context->file, "        IluRuntime.OutputM3Object(call_, self_, TRUE, ObjectType_%s);\n",
	     M3_SHORT_TYPE_NAME(context->class));
  list_enumerate (m->arguments, (EnumProc) EncodeArgument, context);
  fprintf (context->file, "        IluRuntime.FinishRequest (call_);\n");
  fprintf (context->file, "      EXCEPT\n");
  fprintf (context->file, "        IluBasics.Failed(e) =>\n");
  fprintf (context->file, "          RAISE IluBasics.Failed(NEW(IluBasics.Failure, info := \"problem marshalling call\", subArg := e));\n");
  fprintf (context->file, "      END (* try *);\n");
  if (BlockingCall(m))
    {
      fprintf (context->file, "      TRY\n");
      fprintf (context->file, "        scode_ := IluRuntime.GetReply(call_);\n");
      fprintf (context->file, "        IF scode_ = 0 THEN\n");
      if (IsBulkyData(m->returnType))
	{
	  fprintf (context->file, "          VAR returnvalue_ : %s;\n", M3_TYPE_NAME(m->returnType));
	  fprintf (context->file, "          BEGIN\n");
	  UnmarshallValue (context, m->returnType, m->def, "returnvalue_", 12, TRUE);
	  fprintf (context->file, "            RETURN returnvalue_;\n");
	  fprintf (context->file, "          END\n");
	}
      else
	{
	  fprintf (context->file, "          RETURN ");
	  UnmarshallValue (context, m->returnType, m->def, NULL, 0, FALSE);
	  fprintf (context->file, "\n");
	}
      fprintf (context->file, "        ELSE\n");
      if (list_size(m->exceptions) > 0)
	fprintf (context->file, "          %s_x.ReSignal_Exception(call_, scode_);\n",
		 m3_interface_name(m->object->interface));
      fprintf (context->file, "            <*ASSERT FALSE*>\n");
      fprintf (context->file, "        END; (* if *)\n");
      fprintf (context->file, "      EXCEPT\n        IluBasics.Failed(e) =>\n");
      fprintf (context->file, "          RAISE IluBasics.Failed(NEW(IluBasics.Failure, info := \"reply problem\", subArg := e))\n");
      fprintf (context->file, "      END (* try *);\n");
    }
  fprintf (context->file, "    FINALLY\n      IluRuntime.FinishCall(call_);\n    END (* try-finally *);\n");
  fprintf (context->file, "  END Client__%s_%s;\n\n", stn, M3_SHORT_METHOD_NAME(m));
}

void generate_class_code (Type type, Context context)
{
  context->class = type;

  GenerateObjectSupport (type, context);
  MethodIndex = 0;
  list_enumerate ((class_object(type))->methods, (EnumProc) generate_method_code, context);
}

static void generate_client_includes (Context context)
{
  fprintf (context->file, "IMPORT Ilu; <*NOWARN*>\n");
  fprintf (context->file, "IMPORT IluBasics, IluRuntime, Thread;\n");
  fprintf (context->file, "IMPORT %s, %s_x; <*NOWARN*>\n",
	   m3_interface_name(context->interface), m3_interface_name(context->interface));
  list_enumerate (context->interface->imports, (EnumProc) ImportInterface, context);
  fprintf (context->file, "\n");
}

void generate_client_code (Interface i, FILE *file)
{
  struct context_s context;

  context.file = file;
  context.interface = i;
  context.class = NULL;
  context.module = ClientModule;

  generate_boilerplate(file, i);
  fprintf (file, "UNSAFE MODULE %s_c EXPORTS %s, %s_x;\n",
	   m3_interface_name(i), m3_interface_name(i), m3_interface_name(i));
  generate_client_includes (&context);

  list_enumerate(i->classes, (EnumProc) generate_class_code, &context);

  fprintf (file, "BEGIN\n  %s_x.BeStarted();\nEND %s_c.\n", m3_interface_name(i), m3_interface_name(i));
}
