(* IluRuntime.i3 *)
(* Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights
   Reserved.

   Unlimited use, reproduction, and distribution of this software is
   permitted.  Any copy of this software must include both the above
   copyright notice of Xerox Corporation and this paragraph.  Any
   distribution of this software must comply with all applicable United
   States export control laws.  This software is made available AS IS,
   and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
   NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY
   FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY
   DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE)
   OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGES. *)
(* Edited by Mike Spreitzer at December 17, 1993 4:52 pm PST *)

INTERFACE IluRuntime;

(* The interface between stubs and the ILU runtime. *)

IMPORT Ctypes, Ilu, IluKernel, Word;
FROM IluBasics IMPORT Failed, Failure;

<*PRAGMA lL, Ll, Main*>

TYPE
  C_String = UNTRACED REF BITS 8 FOR ['\000' .. '\377'];
  Bool = {False, True};
  Integer = Ctypes.long;
  Cardinal = Ctypes.unsigned_long;
  Character = Ctypes.unsigned_short;
  ShortCard = Ctypes.unsigned_short;
  ShortChar = Ctypes.unsigned_char;
  EnumOrd = Ctypes.unsigned_short;

CONST
  NormalLimit: Cardinal = Word.Plus(
                            Word.Plus(2147483647, 2147483647), 1);
  ShortLimit: Cardinal = 65535;

TYPE
  ObjectType = IluKernel.ObjectType;
  Method = IluKernel.Method;
  Method_Rec = IluKernel.Method_Rec;
  Exception = IluKernel.Exception;
  KernObj = IluKernel.Object; (* The ILU kernel's representation of an
                                 object. *)
  M3Obj = Ilu.Object;                (* The M3 rep of an ILU object *)
  Connection = IluKernel.Connection;
  Call = IluKernel.Call;

<*lL, Ll, Main unconstrained*>


<*lL.sup < ocMu*>
PROCEDURE RegisterObjectType (ot: ObjectType; creator: ObjectCreator)
  RAISES {Failed};
  (* All object types that will be used must be registered before any
     objects (of any types) are unmarshalled.  The creator creates a
     surrogate for this type, setting the fields and implementing the
     methods revealed in Ilu.i3, returning the lead object. *)

TYPE
  ObjectCreator = OBJECT
                  METHODS
                    (*Inside(server.ks, result's type)*)
                    apply (server: Ilu.SurrogateServer): M3Obj RAISES {}
                  END;


(* ================ Client side ================ *)

<*Main Invariant holds; Ll otherwise unconstrained*>

PROCEDURE M3ObjectFromSbh (sbh        : Ilu.SBH;
                           static_type: ObjectType;
                           mstid      : TEXT        ): M3Obj
  RAISES {Failed};
  (* Importing an object begins with calling this procedure, which
     returns an object --- maybe a surrogate, maybe a true object.  The
     second argument is a type this object is known to have; the third
     argument identifies the most specific type this individual object
     actually has. *)

<*Ll >= {call's conn's callmu, iomu} afterward iff result non-NIL*>
PROCEDURE BeginCall (on: M3Obj): Call RAISES {Failed};
  (* Client calls this to initiate a call.  Unless this proc raises
     Failed, client must later call FinishCall. *)

<*Ll >= {call's conn's callmu, iomu}*>

PROCEDURE BeginRequest (call      : Call;
                        intro_type: ObjectType;
                        method    : IluKernel.Method;
                        argSize   : Cardinal          ) RAISES {Failed};
  (* Client calls this to introduce the arguments.  intro_type is the
     type that introduced method.  The size includes that of the
     discriminator, if any.  Next, the arguments (including the
     discriminator) are mashalled, using the marshalling routines
     introduced later. *)

PROCEDURE FinishRequest (call: Call) RAISES {Failed};
  (* End bracket for arguments.  The client then waits for a reply. *)

PROCEDURE GetReply (call: Call): Cardinal RAISES {Failed};
  (* For non-ASYNCHRONOUS methods, the client calls this to get a reply
     message with the right serial number and begin decoding it.  The
     result is success or a protocol-level error (reported via Failed).
     If a protocol error is being reported, the client stub next calls
     FinishCall, and then passes the error to client code.  This
     procedure also decodes whether the marshalled results are normal
     results or an exception parameter; result is 0 if success is being
     reported, otherwise the exception index.  The client next calls the
     appropriate unmarshalling routine, then FinishCall, and finally
     returns to the client code. *)

<*lL, Ll, Main unconstrained*>
PROCEDURE ExceptionOfMethod (method: Method; ecode: INTEGER): Exception;

<*Ll    >=    {call's conn's callmu, iomu} before,
 *Ll disjoint {call's conn's callmu, iomu} after*>
PROCEDURE FinishCall (call: Call);
  (* Client calls this to clean up after a call: after
     results/exceptions have been unmarshalled. *)


(* ================ Server side ================ *)

<*lL, Ll, Main unconstrained*>

PROCEDURE SetMethodStub (ot      : ObjectType;
                         idx, mid: INTEGER;
                         stub    : IluKernel.StubProc );
  (* idx is the index into the methods array; mid is the method id.
     Both are given so that a sanity check can be made. *)

<*Main Invariant holds*>

<*Ll          >= {call's conn's callmu, iomu} before;
after: Ll not >= {call's conn's         iomu},
after: Ll     >= {call's conn's callmu} iff protocol not concurrent*>
PROCEDURE RequestRead (call: Call);

<*Ll  not >=  {call's conn's         iomu},
  Ll      >=  {call's conn's callmu} iff protocol not concurrent, before;
  Ll disjoint {call's conn's callmu, iomu} excpn;
  Ll      >=  {call's conn's callmu, iomu} after. *>

PROCEDURE BeginReply (call      : Call;
                      exceptions: BOOLEAN;
                      replySize : CARDINAL ) RAISES {Failed};

PROCEDURE BeginException (call: Call; evalue, parmSize: CARDINAL)
  RAISES {Failed};

<*Ll    >=    {call's conn's callmu, iomu} before,
  Ll disjoint {call's conn's callmu, iomu} after/excpn*>

PROCEDURE FinishReply (call: Call) RAISES {Failed};

PROCEDURE FinishException (call: Call) RAISES {Failed};

<*Ll  not >=  {call's conn's         iomu},
  Ll    >=    {call's conn's callmu} iff protocol not concurrent, before;
  Ll disjoint {call's conn's callmu, iomu} after/excpn*>

PROCEDURE HandleStubFailure1 (call: Call; f: Failure);
  (* A server stub calls this to handle Failed exceptions arising from
     other IluRuntime calls made before RequestRead. *)

PROCEDURE HandleStubAlert1 (call: Call);
  (* A server stub calls this to handle Thread.Alerted exceptions
     arising from other IluRuntime calls made before RequestRead. *)

PROCEDURE HandleServerProcFail (call: Call; f: Failure);
  (* A server stub calls this to handle Failed exceptions arising from
     the service proc. *)

PROCEDURE HandleServerProcAlert (call: Call);
  (* A server stub calls this to handle Thread.Alerted exceptions
     arising from the service proc. *)

PROCEDURE NoReply (call: Call);
  (* A server stub calls this for asynchronous methods. *)

<*Ll disjoint {call's conn's callmu, iomu}*>

PROCEDURE HandleStubFailure2 (call: Call; f: Failure);
  (* A server stub calls this to handle Failed exceptions arising from
     other IluRuntime calls made after RequestRead. *)

PROCEDURE HandleStubAlert2 (call: Call);
  (* A server stub calls this to handle Thread.Alerted exceptions
     arising from other IluRuntime calls made after RequestRead. *)

<*lL, Ll, Main unconstrained*>
PROCEDURE CheckedRuntimeError (what: TEXT);

(* ================ [Un]Marshalling ================ *)

<*Main Invariant holds*>
<*Ll >= {call's connection's callmu, iomu}*>

PROCEDURE EndSequence (call: Call);
PROCEDURE EndUnion (call: Call);
PROCEDURE EndArray (call: Call);
PROCEDURE EndRecord (call: Call);

PROCEDURE OutputShortInteger (call: Call; i: Ilu.ShortInt);
PROCEDURE OutputInteger (call: Call; x: Ilu.Integer);
PROCEDURE OutputLongInteger (call: Call; i: Ilu.LongInt);
PROCEDURE OutputShortCardinal (call: Call; i: Ilu.ShortCard);
PROCEDURE OutputCardinal (call: Call; x: Ilu.Cardinal);
PROCEDURE OutputLongCardinal (call: Call; i: Ilu.LongCard);
PROCEDURE OutputShortReal (call: Call; f: Ilu.ShortReal);
PROCEDURE OutputReal (call: Call; d: Ilu.Real);
PROCEDURE OutputLongReal (call: Call; f: Ilu.LongReal);
PROCEDURE OutputEnum (call: Call; x: EnumOrd);
PROCEDURE OutputCharacter (call: Call; x: Ilu.Character);
PROCEDURE OutputByte (call: Call; b: Ilu.Byte);
PROCEDURE OutputBoolean (call: Call; x: BOOLEAN);
PROCEDURE OutputOptional (call: Call; provided: BOOLEAN);
PROCEDURE OutputSequence (call : Call;
                          len  : Cardinal;
                          limit: Cardinal := NormalLimit );
PROCEDURE OutputUnion (call: Call; discriminator: ShortCard);
PROCEDURE OutputArray (call: Call);
PROCEDURE OutputRecord (call: Call);
PROCEDURE OutputString (call : Call;
                        x    : Ilu.String;
                        limit: Cardinal := NormalLimit );
PROCEDURE OutputShortCharArray (call: Call;
                                READONLY x: ARRAY OF Ilu.PackedShortChar);
PROCEDURE OutputWString (call : Call;
                         x    : Ilu.WString;
                         limit: Cardinal := NormalLimit );
PROCEDURE OutputCharArray (         call: Call;
                           READONLY x   : ARRAY OF Ilu.Character );
PROCEDURE OutputBytes (call : Call;
                       x    : Ilu.Bytes;
                       limit: Cardinal := NormalLimit );
PROCEDURE OutputOpaque (call: Call; READONLY x: ARRAY OF Ilu.PackedByte);
PROCEDURE OutputM3Object (call         : Call;
                          obj          : M3Obj;
                          discriminator: BOOLEAN;
                          static_type  : ObjectType ) RAISES {Failed};

PROCEDURE InputShortInteger (call: Call): Ilu.ShortInt;
PROCEDURE InputInteger (call: Call): Ilu.Integer;
PROCEDURE InputLongInteger (call: Call): Ilu.LongInt;
PROCEDURE InputShortCardinal (call: Call): Ilu.ShortCard;
PROCEDURE InputCardinal (call: Call): Ilu.Cardinal;
PROCEDURE InputLongCardinal (call: Call): Ilu.LongCard;
PROCEDURE InputShortReal (call: Call): Ilu.ShortReal;
PROCEDURE InputReal (call: Call): Ilu.Real;
PROCEDURE InputLongReal (call: Call): Ilu.LongReal;
PROCEDURE InputEnum (call: Call): EnumOrd;
PROCEDURE InputCharacter (call: Call): Ilu.Character;
PROCEDURE InputByte (call: Call): Ilu.Byte;
PROCEDURE InputBoolean (call: Call): BOOLEAN;
PROCEDURE InputOptional (call: Call): BOOLEAN;
PROCEDURE InputSequence (call: Call; limit: Cardinal := NormalLimit):
  Cardinal;
PROCEDURE InputUnion (call: Call): ShortCard;
PROCEDURE InputArray (call: Call);
PROCEDURE InputRecord (call: Call);
PROCEDURE InputString (call: Call; limit: Cardinal := NormalLimit):
  Ilu.String;
PROCEDURE InputShortCharArray (    call: Call;
                               VAR x   : ARRAY OF Ilu.PackedShortChar );
PROCEDURE InputWString (call: Call; limit: Cardinal := NormalLimit):
  Ilu.WString;
PROCEDURE InputCharArray (call: Call; VAR x: ARRAY OF Ilu.Character);
PROCEDURE InputBytes (call: Call; limit: Cardinal := NormalLimit):
  Ilu.Bytes;
PROCEDURE InputOpaque (call: Call; VAR x: ARRAY OF Ilu.PackedByte);

PROCEDURE InputM3Object (call         : Call;
                         discriminator: BOOLEAN;
                         static_type  : ObjectType ): M3Obj
  RAISES {Failed};

PROCEDURE GetSingletonDiscriminator (call: Call): M3Obj RAISES {Failed};

<*Main Invariant holds; Ll otherwise unconstrained*>

PROCEDURE SizeShortInteger (call: Call; i: Ilu.ShortInt): Cardinal;
PROCEDURE SizeInteger (call: Call; x: Ilu.Integer): Cardinal;
PROCEDURE SizeLongInteger (call: Call; i: Ilu.LongInt): Cardinal;
PROCEDURE SizeShortCardinal (call: Call; i: Ilu.ShortCard): Cardinal;
PROCEDURE SizeCardinal (call: Call; x: Ilu.Cardinal): Cardinal;
PROCEDURE SizeLongCardinal (call: Call; i: Ilu.LongCard): Cardinal;
PROCEDURE SizeShortReal (call: Call; d: Ilu.ShortReal): Cardinal;
PROCEDURE SizeReal (call: Call; d: Ilu.Real): Cardinal;
PROCEDURE SizeLongReal (call: Call; d: Ilu.LongReal): Cardinal;
PROCEDURE SizeEnum (call: Call; i: Ilu.ShortCard): Cardinal;
PROCEDURE SizeCharacter (call: Call; i: Ilu.Character): Cardinal;
PROCEDURE SizeByte (call: Call; i: Ilu.Byte): Cardinal;
PROCEDURE SizeBoolean (call: Call; i: BOOLEAN): Cardinal;
PROCEDURE SizeOptional (call: Call; provided: BOOLEAN): Cardinal;
PROCEDURE SizeSequence (call : Call;
                        len  : Cardinal;
                        limit: Ilu.Cardinal := NormalLimit ): Cardinal;
PROCEDURE SizeUnion (call: Call; discriminator: Ilu.ShortCard):
  Cardinal;
PROCEDURE SizeArray (call: Call): Cardinal;
PROCEDURE SizeRecord (call: Call): Cardinal;
PROCEDURE SizeString (call : Call;
                      x    : Ilu.String;
                      limit: Cardinal := NormalLimit ): Cardinal;
PROCEDURE SizeShortCharArray (call: Call;
                              READONLY x: ARRAY OF Ilu.PackedShortChar):
  Cardinal;
PROCEDURE SizeWString (call : Call;
                       x    : Ilu.WString;
                       limit: Cardinal := NormalLimit ): Cardinal;
PROCEDURE SizeCharArray (call: Call; READONLY x: ARRAY OF Ilu.Character):
  Cardinal;
PROCEDURE SizeBytes (call : Call;
                     x    : Ilu.Bytes;
                     limit: Cardinal := NormalLimit ): Cardinal;
PROCEDURE SizeOpaque (call: Call; READONLY x: ARRAY OF Ilu.PackedByte):
  Cardinal;
PROCEDURE SizeM3Object (call         : Call;
                        obj          : M3Obj;
                        discriminator: BOOLEAN;
                        static_type  : ObjectType ): Cardinal RAISES {};
  (* Can have failures, but simply returns 0 if it does, on the
     assumption that the caller, a stub, will later call OutputM3Object,
     which will flag the error. *)

END IluRuntime.
