/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: type.c,v 1.23 1994/04/13 00:25:59 janssen Exp $ */

#define _POSIX_SOURCE

#ifdef MACOS
#pragma segment ilu
#endif

#include "ilu.h"
#include "iluntrnl.h"

#include "object.h"
#include "server.h"
#include "type.h"
#include "vector.h"

#include <string.h>

/*L1, L2, Main unconstrained */

static struct _ilu_Method_s _ilu_InternalMethods[] = {
  { "GetTypes", 0xFF83,
      TRUE, FALSE, NULL, 0, _ilu_HandleGetTypes },
  { "RegisterGCInterest", 0xFF81,
      FALSE, FALSE, NULL, 0, _ilu_HandleGCInterestRegistration },
  { "UnregisterGCInterest", 0xFF82,
      FALSE, FALSE, NULL, 0, _ilu_HandleGCInterestDeregistration },
  { "Ping", 0xFF84,
      FALSE, FALSE, NULL, 0, _ilu_HandlePing }
};

ilu_Method _ilu_GetTypesMethod = &_ilu_InternalMethods[0];
ilu_Method _ilu_RegisterGCInterestMethod = &_ilu_InternalMethods[1];
ilu_Method _ilu_UnregisterGCInterestMethod = &_ilu_InternalMethods[2];
ilu_Method _ilu_PingMethod = &_ilu_InternalMethods[3];

static const nInternal = sizeof(_ilu_InternalMethods)
			/sizeof(struct _ilu_Method_s);

struct _ilu_Class_s _ilu_rootClass_s = {
	"ILU-root",		/* name*/
	"com.xerox.parc",	/* brand */
	"ilu:root",		/* unique_id */
	FALSE,			/* singleton */
	TRUE,			/* collectible */
	NULL,			/* authentication */
	_ilu_InternalMethods,	/* methods */
	sizeof(_ilu_InternalMethods)/sizeof(struct _ilu_Method_s),
				/* method_count */
	0,			/* superclass_count */
	NULL,			/* superclass_ids */
	NULL,			/* superclasses */
	FALSE			/* shown */
	};

ilu_Class _ilu_rootClass = &_ilu_rootClass_s;

/*L1 >= {otmu}*/
/*L2, Main unconstrained*/

static HashTable ClassNameTable = NULL;
/* Object type name -> object type */

static HashTable ClassIDTable = NULL;
/* Object type id -> object type */

static HashTable UnknownTypeIDs = NULL;
/* Object type id -> ID_DAG */

static HashTable UnlinkedClasses = NULL;
/* Mapping from object type id to vector of object types waiting for
   id to be registered. */

static void _ilu_RegisterClass (ilu_Class class)
{
  int new;
  ilu_Class p;
  register ilu_cardinal index;
  ilu_Vector v;

  if (ClassNameTable == NULL)
    ClassNameTable = (ilu_refany) _ilu_hash_MakeNewTable (
	CLASS_HASHTABLESIZE,
	(ilu_hashfnptr) _ilu_hash_HashString,
	(ilu_compfnptr) _ilu_hash_StringCompare);
  if (ClassIDTable == NULL)
    ClassIDTable = (ilu_refany) _ilu_hash_MakeNewTable (
	CLASS_HASHTABLESIZE,
	(ilu_hashfnptr) _ilu_hash_HashString, 
	(ilu_compfnptr) _ilu_hash_StringCompare);
  if (UnlinkedClasses == NULL)
    UnlinkedClasses = (ilu_refany) _ilu_hash_MakeNewTable (
	CLASS_HASHTABLESIZE,
	(ilu_hashfnptr) _ilu_hash_HashString, 
	(ilu_compfnptr) _ilu_hash_StringCompare);

  if (_ilu_hash_FindInTable (ClassNameTable, class_name(class)) == NULL)
    _ilu_hash_AddToTable (ClassNameTable, class_name(class), class);
  new = _ilu_hash_FindInTable(ClassIDTable, class_unique_id(class))==NULL;
  if (new)
    _ilu_hash_AddToTable (ClassIDTable, class_unique_id(class), class);

  /* this handles the case where additional class information is
   * loaded into a running program */
  if (new && UnknownTypeIDs != NULL
      && _ilu_hash_FindInTable (UnknownTypeIDs, class_unique_id(class))
	 != NULL)
    { /* This shouldn't happen! */
      ASSERT(0, buf,
	     (buf, "%s %s %s previously considered unknown.\n",
	      "ilu_RegisterClass:  Config bug!  Registering object type",
	      class_name(class), class_unique_id(class) ));
    }

  /* this links up the superclass of "class" which have already
   * been registered */
  for (index = 0;  index < class_superclass_count(class);  index++) {
      if (class_superclass_id(class,index) != NULL) {
          p = (ilu_Class) _ilu_hash_FindInTable (ClassIDTable,
				class_superclass_id(class, index));
          if (p == NULL) {
              v = (ilu_Vector) _ilu_hash_FindInTable (UnlinkedClasses,
					class_superclass_id(class,index));
              if (v == NULL) {
                  v = _ilu_vector_new(1);
                  _ilu_hash_AddToTable (UnlinkedClasses,
					class_superclass_id(class, index),
					v);
		}
              _ilu_vector_add(v, (ilu_refany) class);
            }
          else
              class_superclass(class,index) = p;
        }
    }

  /*now link those classes which have been waiting for this superclass*/
  v = (ilu_Vector) _ilu_hash_FindInTable (UnlinkedClasses,
					    class_unique_id(class));
  if (v != NULL) {
      ilu_cardinal size = _ilu_vector_size(v);
      ilu_Class *elements = (ilu_Class *) _ilu_vector_elements(v);
      register ilu_cardinal j;

      _ilu_hash_RemoveFromTable (UnlinkedClasses, class_unique_id(class));

      for (index = 0;  index < size;  index++) {
          p = elements[index];
          for (j = 0;  j < class_superclass_count(p);  j++)
            if (strcmp(class_unique_id(class), class_superclass_id(p,j))
		== 0) {
		class_superclass(p,j) = class;
		break;
	      }
        }
      _ilu_vector_destroy(v, NULL);
    }

  return;
}

/*L1_sup < otmu*/

static void PrintClassEntry (ilu_refany entry, ilu_refany junk)
{
  ilu_Class p;

  if (entry != NULL)
    {
      p = (ilu_Class) entry;
      fprintf (stderr, "    %s (%s)   0x%x\n", p->cl_name, p->cl_unique_id, p);
    }
  else
    fprintf (stderr, "<null>\n");    
}

void ilu_RegisterClass (ilu_Class class)
{
  _ilu_AcquireMutex(ilu_otmu);
  _ilu_RegisterClass(class);
  _ilu_ReleaseMutex(ilu_otmu);
  return;
}

ilu_Class ilu_GetGcCallbackClass (void)
{
  _ilu_AcquireMutex(ilu_otmu);
  if (ClassIDTable == NULL ||
      _ilu_hash_FindInTable(ClassIDTable, _ilu_GcCallbackClass->cl_unique_id)
      == NULL)
    _ilu_RegisterClass(_ilu_GcCallbackClass);
  _ilu_ReleaseMutex(ilu_otmu);
  return _ilu_GcCallbackClass;
}

ilu_Class ilu_FindClassFromName (ilu_string classname)
{
  ilu_Class c;
  _ilu_AcquireMutex(ilu_otmu);
  if ((_ilu_DebugLevel & OBJECT_DEBUG) == OBJECT_DEBUG)
    {
      fprintf (stderr, "ilu_FindClassFromName:  classname is %s, class table is 0x%x:\n", classname,
	       ClassNameTable);
      _ilu_hash_TableEnumerate (ClassNameTable, (void (*)(ilu_refany, ilu_refany)) PrintClassEntry, NULL);
    }
  if (strcmp(classname, _ilu_rootClass->cl_name) == 0)
       c = _ilu_rootClass;
  else if (ClassNameTable != NULL)
       c = (ilu_Class) _ilu_hash_FindInTable (ClassNameTable, classname);
  else c = NULL;
  _ilu_ReleaseMutex(ilu_otmu);
  return (c);
}

ilu_Class ilu_FindClassFromID (ilu_string ID)
{
  ilu_Class c;
  _ilu_AcquireMutex(ilu_otmu);
  if ((_ilu_DebugLevel & OBJECT_DEBUG) == OBJECT_DEBUG)
    {
      fprintf (stderr, "ilu_FindClassFromID:  id is %s, class table is 0x%x:\n", ID,
	       ClassIDTable);
      _ilu_hash_TableEnumerate (ClassIDTable, (void (*)(ilu_refany, ilu_refany)) PrintClassEntry, NULL);
    }
  if (strcmp(ID, _ilu_rootClass->cl_unique_id) == 0)
       c = _ilu_rootClass;
  else if (ClassIDTable != NULL)
       c = (ilu_Class) _ilu_hash_FindInTable (ClassIDTable, ID);
  else c = NULL;
  _ilu_ReleaseMutex(ilu_otmu);
  return (c);
}

static ilu_boolean SeekAncestor(ilu_Class a, ilu_Class b)
{
  int j;
  ilu_Class aa;
  if (a==b)
    return (TRUE);
  for (j = 0;  j < class_superclass_count(a);  j++) {
      aa = class_superclass(a, j);
      if (aa != NULL && SeekAncestor(aa, b))
          return (TRUE);
    }
  return (FALSE);
}

ilu_boolean ilu_IsSubObjectType (ilu_Class a, ilu_Class b)
{
  ilu_boolean ans;
  _ilu_AcquireMutex(ilu_otmu);
  ans = (b == _ilu_rootClass) || SeekAncestor(a, b);
  _ilu_ReleaseMutex(ilu_otmu);
  return (ans);
}

typedef struct _iluBuffer_s {
  /*L1 unconstrained*/

  char *buf;
  unsigned long allocated;
  unsigned long used;
} *iluBuffer;
#define BUF_INCREMENT 100

/*L1 unconstrained*/

static void addBytesToBuf (iluBuffer buf, char *string, ilu_cardinal len)
{
  if (buf->allocated - buf->used < len)
    {
      buf->allocated += BUF_INCREMENT;
      buf->buf = (char *) realloc(buf->buf, buf->allocated);
    }
  strncpy (buf->buf + buf->used, string, len);
  buf->used += len;
}

/*L1 >= {otmu}*/

static void UnshowAncestors(ilu_Class c)
{
    int i;
    c->cl_shown = FALSE;
    for (i = 0;  i < class_superclass_count(c);  i++)
        UnshowAncestors(class_superclass(c,i));
    return;
}

static void AddTypeName (ilu_Class type, iluBuffer buf)
{
  int i;
  int noco = 1;
  addBytesToBuf (buf, class_unique_id(type),
		 strlen(class_unique_id(type)));
  if (type->cl_shown != TRUE) {
      type->cl_shown = TRUE;
      if (class_superclass_count(type) == 0)
          return;
      addBytesToBuf (buf, "(", 1);
      for (i = 0;  i < class_superclass_count(type);  i++) {
	  if (noco)
	       noco = 0;
	  else addBytesToBuf (buf, ",", 1);
	  AddTypeName(class_superclass(type,i), buf);
	}
      addBytesToBuf (buf, ")", 1);
    }
  return;
}

/*L1_sup < otmu*/
static ilu_string ClassToString (ilu_Class class)
{
  struct _iluBuffer_s buf;
  buf.allocated = BUF_INCREMENT;
  buf.buf = malloc(BUF_INCREMENT);
  buf.used = 0;
  _ilu_AcquireMutex(ilu_otmu);
  UnshowAncestors(class);
  AddTypeName (class, &buf);
  addBytesToBuf (&buf, "\0", 1);	/* NUL-terminate */
  _ilu_ReleaseMutex(ilu_otmu);
  return (buf.buf);
}

/*L1 >= {otmu}*/

typedef struct _ilu_ID_DAG_s ID_DAG_s, *ID_DAG;

typedef struct _ilu_ID_DAG_Cons {
  /*L1 >= {otmu}*/

  ID_DAG idd;
  struct _ilu_ID_DAG_Cons *next;
} ID_DAG_Cons, *ID_DAG_List;

struct _ilu_ID_DAG_s {
  /*L1 >= {otmu}*/

  char *id;
  ilu_Class it;		/* non-NULL if id registered */
  ID_DAG_List supers;
  ilu_boolean computed;	/* TRUE => mska significant */
  ilu_Class mska;	/* the one most specific known ancestor,
			   or NULL if not well-defined */
  ilu_boolean visited;	/* is this an ancestor of a counted type? */
  ID_DAG anext;		/* threaded list of most spcfc known anc'rs */
  ID_DAG aprev;		/* it's doubly-linked */
};

static ID_DAG_s mska_head;
/* Head of threaded list of most spcfc known anc'rs.
   No one member is an ancestor of another. */

/*L1_sup < otmu*/
ilu_Class _ilu_FindMSKA(ilu_string tid)
{
  ID_DAG n;
  ilu_Class ans;
  _ilu_AcquireMutex(ilu_otmu);
  n = (ID_DAG) _ilu_hash_FindInTable (UnknownTypeIDs, tid);
  if (n!=NULL && n->computed)
    ans = n->mska;
  else ans = NULL;
  _ilu_ReleaseMutex(ilu_otmu);
  return ans;
}

/* spec ::== id [ "(" spec ("," spec)* ")" ]
   At entry, *names is at the start of a spec;
   at exit, *names is at the first char past that spec.
   Result is the ID_DAG having the id in the spec.
   Caller owns storage for names.
 */
static ID_DAG ClassNamesToID_DAG (ilu_string *names)
{
  ID_DAG n;
  char *pl, *pr, *pc, *pe, cr;
  ilu_integer len;

  DEBUG(OBJECT_DEBUG,
	(stderr, "(ClassNamesToID_DAG):  Called with \"%s\"\n", *names));

  if (*names == NULL OR **names == '\0')
    return (NULL);
  len = strlen(*names);
  pl = strchr (*names, '(');
  pr = strchr (*names, ')');
  pc = strchr (*names, ',');
  if (pl==NULL) pl = (*names) + len;
  if (pr==NULL) pr = (*names) + len;
  if (pc==NULL) pc = (*names) + len;
  pe = (pr < pl) ? pr : pl;
  pe = (pc < pe) ? pc : pe;
  cr = *pe;
  *pe = '\0';
  n = (ID_DAG) _ilu_hash_FindInTable (UnknownTypeIDs, *names);
  if (n == NULL) {
      ilu_string id = _ilu_Strdup(*names);
      n = (ID_DAG) malloc(sizeof(ID_DAG_s));
      _ilu_hash_AddToTable (UnknownTypeIDs, id, n);
      n->id = id;
      n->it = (ilu_Class) _ilu_hash_FindInTable (ClassIDTable, id);
      n->supers = NULL;
      n->mska = n->it;
      n->computed = n->mska != NULL;
      n->anext = n->aprev = NULL;
      if (cr == '(') {
          ID_DAG_List *pp = &(n->supers);
          *names = pl + 1;
          while ((**names) != ')') {
              ID_DAG d2 = ClassNamesToID_DAG(names);
              ID_DAG_List l = (ID_DAG_List) malloc(sizeof(ID_DAG_Cons));
              l->idd = d2;
              l->next = NULL;
              *pp = l;
              pp = &(l->next);
              if ((**names) == ',')
                  (*names)++;
            }
          (*names)++;
        }
      else *names = pe;
    }
  else if (pl < pc) {	/* check for consistency */
      ID_DAG_List l = n->supers;
      *names = pl + 1;
      while ((**names) != ')') {
          ID_DAG d2 = ClassNamesToID_DAG(names);
          ASSERT(d2 == l->idd, buf,
		 (buf, "ClassNamesToID_DAG:  %s %s: %s vs. %s.\n",
		  "Disagreement on superclasses of", n->id, d2->id,
		  l->idd->id));
          l = l->next;
          if ((**names) == ',')
              (*names)++;
        }
      (*names)++;
    }
  else *names = pe;
  *pe = cr;
  return (n);
}

static void ClearVisited(ID_DAG t)
{
  ID_DAG_List p;
  t->visited = FALSE;
  for (p = t->supers;  p != NULL;  p = p->next)
    ClearVisited(p->idd);
  return;
}

/* Before and after:
     every node in the mska list is visited;
     for each visited node n:
       1. if n is known then n is an ancestor of a node in the mska list;
       2. every ancestor of n is visited.
 */

/* As above, plus...
   After: all of t's ancestors are visited, and none are in the list.
 */
static void MarkAncestors(ID_DAG t)
{
  ID_DAG_List p;
  for (p = t->supers;  p != NULL;  p = p->next) {
      ID_DAG u = p->idd;
      if (u->anext != NULL) {
          /* must already be visited, and have no proper ancestors in
             the mska list; remove from list. */
          u->aprev->anext = u->anext;
          u->anext->aprev = u->aprev;
        }
      else if (!(u->visited && (u->it != NULL))) {
          MarkAncestors(u);
          u->visited = TRUE;
        }
    }
  return;
}

static void FindMostSpecificType (ID_DAG t)
{
  ID_DAG_List p;
  DEBUG(OBJECT_DEBUG,
	(stderr, "(FindMostSpecificType):  at %s, %s visited.\n",
	 t->id, (t->visited == FALSE) ? "not" : ""));
  if (t->visited == TRUE)
    return;
  if (t->it != NULL) {
      MarkAncestors(t);
      t->anext = mska_head.anext;
      t->aprev = &mska_head;
      t->anext->aprev = t;
      t->aprev->anext = t;
    }
  else {
      for (p = t->supers;  p != NULL;  p = p->next)
        FindMostSpecificType(p->idd);
    }
  t->visited = TRUE;
  return;
}

/*L1_sup < otmu*/
static ilu_Class StringToClass (ilu_string s)
{
  ID_DAG t;
  char *q = s;
  ilu_Class ans;

  if (s == NULL)
    return (NULL);
  _ilu_AcquireMutex(ilu_otmu);
  if (UnknownTypeIDs == NULL)
    UnknownTypeIDs = (ilu_refany) _ilu_hash_MakeNewTable(
			CLASS_HASHTABLESIZE,
			(ilu_hashfnptr) _ilu_hash_HashString,
			(ilu_compfnptr) _ilu_hash_StringCompare);
  t = ClassNamesToID_DAG(&q);
  if (t == NULL)
      return (NULL);
  DEBUG(OBJECT_DEBUG,
	(stderr, "(StringToClass):  Converted names <%s> to DAG.\n", s));
  if (t->computed) {
      DEBUG(OBJECT_DEBUG, (stderr, "(StringToClass):  Old problem.\n"));
    }
  else {
      ClearVisited(t);
      mska_head.anext = mska_head.aprev = &mska_head;
      FindMostSpecificType(t);
      if (  (mska_head.anext != &mska_head)
	    && (mska_head.anext->anext == &mska_head))
          t->mska = mska_head.anext->it;
      else
          t->mska = NULL;
      t->computed = TRUE;
    }
  ans = t->mska;
  _ilu_ReleaseMutex(ilu_otmu);
  if (ans == NULL) {
      fprintf(stderr, "%s <%s> are known!  Unable to resolve type.\n",
	      "(StringToClass):  Multiple supertypes for type", s);
    }
  else {
      DEBUG(OBJECT_DEBUG,
	    (stderr, "(StringToClass):  Found class %s.\n",
	     class_name(ans)));
    }
  return (ans);
}

/*Main Invariant holds; L2 otherwise unconstrained*/

static ilu_ProtocolException ObtainTypes (ilu_Object o, ilu_string *types, ilu_cardinal *typeslen)
{
  ilu_Call call;
  ilu_cardinal estatus = 0;
  ilu_ProtocolException internal;
  ilu_cardinal reqSize;
  ilu_Class pclass = object_class(o);
  ilu_Server s = object_server(o);

  DEBUG(OBJECT_DEBUG, (stderr, "_ilu_FindClassViaRPC:  object 0x%x...\n", (unsigned long) o));

  if (class_singleton(pclass))
    {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "%s %s is singleton, not attempting GetTypes RPC call.\n",
	     "_ilu_FindClassViaRPC:  pclass", class_name(pclass)));
      return (ilu_ProtocolException_NoSuchMethodOnClass);
    }
  if ((call = ilu_BeginCall(s)) == NULL)
    return (ilu_ProtocolException_Unknown);
  _ilu_AcquireMutex(server_lock(s));
  reqSize = ilu_SizeOfObjectID(call, o, TRUE, _ilu_rootClass);
  _ilu_ReleaseMutex(server_lock(s));
  if (NOT ilu_BeginRequest(call, _ilu_rootClass, _ilu_GetTypesMethod, reqSize))
    internal = ilu_ProtocolException_Unknown;
  else
    {
      ilu_EnterServer(s, object_class(o));
      ilu_OutputObjectID(call, o, TRUE, _ilu_rootClass);
      ilu_FinishRequest(call);
      internal = ilu_GetReply(call, &estatus);
      if (internal == ilu_ProtocolException_Success AND estatus == 0)
	/* successful reply */
	ilu_InputString (call, types, typeslen, 0xFFFF, FALSE);
    }
  ilu_FinishCall(call);
  return internal;
}

ilu_Class _ilu_FindClassViaRPC (ilu_Object o)
{
  ilu_string types = NULL;
  ilu_ProtocolException internal;
  ilu_cardinal typeslen, reqSize;
  ilu_Class c = NULL;

  internal = ObtainTypes (o, &types, &typeslen);

  if (types == NULL)
    DEBUG(OBJECT_DEBUG,
	  (stderr, "%s  Status of RPC call is %u.\n",
	   "_ilu_FindClassViaRPC:  no types.", internal));
  else
    DEBUG(OBJECT_DEBUG,
	  (stderr, "_ilu_FindClassViaRPC:  typestring is <%s>...\n",
	   types));

  if (types != NULL)
    {
      c = StringToClass (types);
      FREETOKEN(types);
    }
  DEBUG(OBJECT_DEBUG,
	(stderr, "_ilu_FindClassViaRPC:  class is \"%s\".\n",
	 (c == NULL) ? "*unknown*" : class_name(c)));
  return (c);
}

/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/
void _ilu_HandleGetTypes (ilu_Call call)
{
  ilu_Object disc;
  ilu_string names = NULL;
  
  ilu_InputObjectID (call, &disc, TRUE, _ilu_rootClass);
  if (disc != NULL) {
      _ilu_DeltaHolds(disc, 1);
      ilu_ExitServer(object_server(disc), _ilu_rootClass);
    }

  ilu_RequestRead(call);
  
  if (disc != NULL && object_class(disc) != NULL)
      names = ClassToString(object_class(disc));

  if (names == NULL)
    {
      if (ilu_BeginException(call, 0,
			     ilu_ProtocolException_GarbageArguments))
          ilu_FinishException (call);
    }
  else
    {
      ilu_cardinal len = strlen(names);

      if (ilu_BeginReply(call, FALSE,
			 ilu_SizeOfString(call, names, len, 0xFFFF,
					  FALSE))) {
          ilu_OutputString (call, names, len, 0xFFFF, FALSE);
          ilu_FinishReply (call);
        }
    }

  if (disc != NULL) {
      ilu_Server s = object_server(disc);
      ilu_Class cl = object_class(disc);
      ilu_EnterServer(s, cl);
      _ilu_DeltaHolds(disc, -1);
      ilu_ExitServer(s, cl);
    }
  return;
}
