/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* Last tweaked by Mike Spreitzer April 4, 1994 10:34 am PDT */

#ifndef _ILU_INTERNALS_
#define _ILU_INTERNALS_

#include "hash.h"

#if (defined ( __STDC__ ) || defined ( __cplusplus ))
/* So I can write ansi prototypes and still use Sun cc */
#define	_FN_DECL(type_and_name, args) type_and_name args
#else
#define _FN_DECL(type_and_name, args) type_and_name()
#endif

#define NOT		!
#define AND		&&
#define OR		||

#define TRUE		1
#define FALSE		0

#include <stdlib.h>
#include <errno.h>

#include "iludebug.h"

#define FREETOKEN(s) {if((s)!=NULL)free(s);}

#define PROTOCOL_TYPE(call)	(call->ca_connection->co_protocol->pr_type)

#define OBJECT_HASHTABLESIZE	113		/* number of different object slots */
#define SERVER_HASHTABLESIZE	53		/* number of different server slots */
#define CLASS_HASHTABLESIZE	53		/* number of different classes */

#define INITIAL_GCLIST_SIZE		5
#define INITIAL_GCDOBJECTSET_SIZE	200
#define INITIAL_GCCLIENTS_SIZE		20

#define GC_CALLBACK_CLASS_ID		"ilu.GCCallback"

/* ================ Reply list for putting replies on ================ */

typedef struct reply_s {
  /*L1, L2, Main unconstrained*/
  
  unsigned char * rp_packet;
  ilu_cardinal rp_len;
  ilu_cardinal rp_SN;
  
  /*L2 >= {connection's iomu}*/
  struct reply_s *rp_next;
} Reply;

/* ================ Internal Consistency Checking ================ */

void _ilu_Assert(int t, ilu_string id);
/* Code in the kernel calls this at internal consistency checks.  The first argument should be a C boolean that's true (i.e., and int that's not 0).  The second argument is some string that distinguishes the call point from every other point that calls ilu_Assert; storage is owned by the caller.  This procedure returns iff t.  When we get our error system, this procedure will also return if !t and the caller should try to communicate the brokenness to the application. */

#define ASSERT(flg,buf,spargs) {if (!(flg)) {char buf[1000]; sprintf spargs; _ilu_Assert(0, buf);}}


/* ================ Locking routines ================ */

/*
We use simple mutual exclusion (ie, semaphores with only two states).  When we get an error system, we may have a way of expressing the interruption of an Acquire operation.
*/

/*L1, L2, Main unconstrained*/

ilu_private _ilu_CreateMutex( ilu_string d1, ilu_string d2 );
/* The concatenation of d1 & d2 describes the mutex;
 * storage for them owned by caller. */

/*L1_sup < m before, L1_sup = m after*/
void _ilu_AcquireMutex( ilu_private m );
/* Blocks until acquisition succeeds. */

/*L1 >= {m}*/
void _ilu_HoldMutex(ilu_private m);
/* Checks that the caller holds the given mutex. */

/*L1 >= {m} before, L1 not >= {m} after*/
void _ilu_ReleaseMutex( ilu_private m );
/* Releases held lock. */

void _ilu_DestroyMutex( ilu_private m );
/* Note that this might enable a pending AcquireMutex operation. */

extern ilu_Lock ilu_smu;	/* Global mutex for server table */
extern ilu_Lock ilu_otmu;	/* ..for object type data structures */
extern ilu_Lock ilu_cmu;	/* Global mutex for conn mgmt */
extern ilu_Lock ilu_prmu;	/* Global mutex for transp. reg'y */
extern ilu_Lock ilu_trmu;	/* Global mutex for proto. reg'y */
extern ilu_Lock ilu_gcmu;
extern ilu_Lock ilu_daimu;	/* For default alarm impl */

/* We don't declare here what timu is; each implementation of alarms has the freedom and responsibility to choose an implementation of timu.  ilu_daimu is available for use by one implementation per program. */

ilu_boolean _ilu_CanCondition(void);

ilu_private _ilu_CreateCondition(ilu_string d1, ilu_string d2);

void _ilu_NotifyCondition(ilu_private c);

void _ilu_DestroyCondition(ilu_private c);

/*L1_sup = m*/
void _ilu_WaitCondition(ilu_private c, ilu_private m);


/* ================ Synthesized Locking ================ */

/*L1 >= {conn's server}*/
/*L2 as implied by name*/
/*Main unconstrained*/

void _ilu_AcquireConnIO(ilu_Connection conn);

/*L1 >= {cmu, conn's server}*/
void _ilu_ReleaseConnIO(ilu_Connection conn);
/* Release I/O mutex.
   But first closes the connection if its server is closing. */

/*L1 >= {conn's server};
  conn->co_nOuts==0 => L1 >= {cmu}*/

void _ilu_AcquireConnCall(ilu_Connection conn, ilu_Call call);
void _ilu_ReleaseConnCall(ilu_Connection conn, ilu_Call call);
/* These also shift the connection into and out of the idle list. */

/* ================ FD & Connection Management ================ */

/*L1 >= {cmu}*/
/*L2, Main unconstrained*/

extern ilu_integer ilu_fdbudget;	/* # FDs allowed */
extern ilu_integer ilu_fdsused;		/* # FDs in use */
extern ilu_integer ilu_fdsidle;		/* # FDs idle */

extern ilu_ConnLinks ilu_idleConns;
/* Head of list of idle FD-consuming connections.  A connection is in this list while its callmu isn't held and its nOuts=0.  */

/*L1_sup = cmu*/
/*forall conn: (L2 >= {conn's iomu}) => (L2 >= {conn's callmu})*/
void _ilu_ReduceFdsTo(ilu_integer goal);
/* Close idle connections until ilu_fdsused+ilu_fdsidle <= goal
   or we run out of idle connections. */


/* ================ iluxport.h counterparts ================ */
/* These procedures are like their similarly-named iluxport.h
 * counterparts, except that they require more mutexes to be held.
 */

/*L2, Main unconstrained*/

/*L1 unconstrained*/
ilu_boolean _ilu_BlockingWaitForInputOnConnection(ilu_Connection conn,
						  ilu_boolean *sure,
						  ilu_FineTime *limit);
/* Like the public version, but also sets *sure to FALSE iff we suspect
   the input may have already been read.  Thus, when *sure is TRUE
   and NDataPresent returns 0 we can decide our peer has abandoned
   the connection (this is difficult because ioctl(,FIONREAD,) doesn't
   do us the favor of returning an error in this case). */

/*L1 >= {port's server}*/
void _ilu_ClearPortFromServer(ilu_Port port, ilu_Server s);
/* Unlink this port from its server, s.
   Called when port is closing and last connection is closed. */

/*L1_sup = port's server; L1 >= {cmu}*/
void _ilu_ClosePort(ilu_Port port);
/* If port was the server's default, one of the server's other ports
 * becomes the default. */

/*L1_sup = conn's server; L1 >= {cmu}*/
/*L2 not >= {conn's iomu}*/
void _ilu_CloseConnection (ilu_Connection conn);
/* Usable on either incoming or outgoing connections. */

/*L1 >= {cmu, conn's server}*/
/*L2 >= {conn's iomu}*/
void _ilu_CloseIoingConnection (ilu_Connection conn);
/* Usable on either incoming or outgoing connections. */


/* ================ Disorganized ================ */

/*L1, L2, Main unconstrained*/

void _ilu_AutoSetDebugLevel (void);

ilu_cardinal _ilu_SafeStrlen (ilu_string s);

void _ilu_FreeToken (ilu_refany token);

ilu_string _ilu_Strdup (const ilu_string);

int _ilu_casefree_cmp (const ilu_string, const ilu_string);
     /* returns 0 if s1 == s2, -1 if s1 < s2, 1 if s1 > s2 */

ilu_string _ilu_Strcat3(const ilu_string s1, const ilu_string s2,
			const ilu_string s3);
/* Returns s1+s2+s3, in fresh storage; NULL may be used to input
 * an empty string.  Result will be NULL only if malloc fails. */

ilu_string _ilu_Strcat5(const ilu_string s1, const ilu_string s2,
			const ilu_string s3, const ilu_string s4,
			const ilu_string s5);
/* Like _ilu_Strcat3, only more parts. */

/*L1 >= {trmu}*/
ilu_string _ilu_tcp_CurrentHostInetName (void);

/*L1_sup < prmu, L1_sup < trmu*/
ilu_boolean _ilu_ParseConnectInfo (ilu_string handle,
				   ilu_ProtocolType *protocolType,
				   ilu_string *protocolInfo,
				   ilu_TransportType *transportType,
				   ilu_string *transportInfo);
/* Returns TRUE iff parse successful.  Stores through non-NULL pointer
 * arguments; storage ownership of strings is returned to caller. */

/*L1, L2, Main unconstrained*/

/*L1 >= {cmu, s}*/
void _ilu_DestroyObject (ilu_Server s, ilu_Object obj);
/* Called by routines in gc.c and object.c to do common work.
   Assumes gc cleanup already done. */

/*L1_sup < smu*/
ilu_Server _ilu_FindServer (ilu_string serverid, ilu_boolean add,
			    ilu_string cinfo);
/* Looks for the server with the given id.  If add and not found,
 * adds a new surrogate server of the given name and contact info.
 * Caller owns id and cinfo. */

/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/
void _ilu_HandlePing (ilu_Call call);
/* Handles the built-in Ping method. */

/*L1 >= {cmu, server}*/
void _ilu_ServerRemoveObject (ilu_Server s, ilu_Object obj);


/*Inside(server, result's type)*/
ilu_Object _ilu_FindObjectInServer(ilu_string ih, ilu_Server s);

/*L1 >= {s}*/

ilu_boolean _ilu_Addable(ilu_Server s, ilu_Class t, ilu_Object *h);

ilu_boolean _ilu_AddSingleton(ilu_Server s, ilu_Class t, ilu_Object o);

void _ilu_CacheCall(ilu_Call call, ilu_bytes reply, ilu_integer len);
/* Called from a Protocol's finish_reply and _exception methods to
   cache the reply. Caller relinquishes ownership of reply. */

/*L1_sup = s; L1 >= {cmu}*/
/*L2, Main unconstrained*/

ilu_Connection _ilu_CreateConnection (ilu_Transport bs, ilu_string tinfo,
				      ilu_Protocol pr, ilu_string pinfo,
				      ilu_Port port, ilu_Server s);
/* Used internally to create both incoming and outgoing connections.
 * bs, tinfo, pr, pinfo, and s aren't NULL;
 * port is NULL iff s is a surrogate.
 * String args owned by caller.
 * Result's call and I/O mutexes are not held by any thread.
 * Caller is responsible for staying within FD budget. */

/*L1 >= {conn's server if k=psl; cmu if k=lru}*/

void _ilu_LinkConnection(ilu_ConnLinks *head, ilu_Connection conn,
			 ilu_ConnLinkKind k);

void _ilu_UnlinkConnection(ilu_ConnLinks *head, ilu_Connection conn,
			   ilu_ConnLinkKind k);

/*L1 >= {s}*/
void _ilu_ClearConnectionFromServer (ilu_Connection c, ilu_Server s);

/*L2 >= {conn.iomu}*/
/*L1, Main unconstrained*/

ilu_integer _ilu_NDataPresent(ilu_Connection conn);
/* positive<=>some, 0<=>none, negative<=>error */

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/
ilu_bytes _ilu_ReadPacket(ilu_Connection conn,
			  ilu_cardinal *packetLength,
			  ilu_PacketType *packetType,
			  ilu_cardinal *packetSN);
/* Read a call or reply.  Returns the entire protocol-level
 * packet, including *type and *serialNumber.
 * Caller owns result.
 */

/*L2, Main unconstrained*/

extern ilu_refany _ilu_ioTimeoutAlarm;

extern ilu_refany _ilu_grAlarm;	/* for use in ilu_GetReply */

/*L1_sup < prmu*/
ilu_string _ilu_ContactInfoOfConnection (ilu_Connection conn,
					 ilu_Object obj);
/* Caller owns result, which will not be NULL. */

/*L1_sup < trmu*/

ilu_TransportClass _ilu_GetTransportClassByName (ilu_string name);
/* name, owned by caller, may be a full tinfo or just the name. */

/*Main Invariant holds*/
ilu_Transport _ilu_GetTransportFromClass(ilu_TransportClass tc,
					 ilu_string tinfo);
/* Caller owns tinfo */

ilu_TransportType ilu_FindTransportTypeFromInfo (ilu_string tinfo);
/* Caller owns tinfo */

extern ilu_TransportClass _ilu_tcp_TransportClass(void);
extern ilu_TransportClass _ilu_udp_TransportClass(void);
extern ilu_TransportClass _ilu_xnsspp_TransportClass(void);
extern ilu_TransportClass _ilu_mem_TransportClass(void);
/* TransportClass "instantiators"; see transport.c */

/*L1_sup < prmu*/

ilu_Protocol _ilu_GetProtocolFromInfo (ilu_string pinfo);
/* Caller owns the string arg. */

/*L1 unconstrained*/
ilu_ProtocolType ilu_FindProtocolTypeFromInfo (ilu_string pinfo);
/* Caller owns the string arg. */

extern ilu_Protocol _ilu_sunrpc_Protocol(void);
extern ilu_Protocol _ilu_courier_Protocol(void);
/* protocol "instantiators" (see protocol.c) */

/*Main Invariant holds*/

_FN_DECL(void _ilu_WaitForInputOnFD , (int fd, ilu_boolean *sure,
				       ilu_FineTime *limit));
_FN_DECL(void _ilu_WaitForOutputOnFD, (int fd, ilu_boolean *sure,
				       ilu_FineTime *limit));
/* These two procedures return when either (1) the appropriate kind of I/O can be done on the given file descriptor without blocking, (2) an exceptional condition exists on the FD, or (3) *limit has arrived.  NULL may be passed for limit, in which case *limit is effectively +infinity.  In a multi-threaded program, these procedures block only the calling thread.  *sure is set, but not read.  When *sure is set true, (1) or (2) is known to hold upon return.  When *sure is set false, either (3) holds or (1) or (2) might hold.  In a single-threaded program, these procedures process input on other FDs and the alarm while waiting.  This processing may lead to a nested call on one of these procedures, with the same or a different FD.  When I/O is finally enabled on an FD, all nested calls waiting on the same FD are unblocked; *sure is set TRUE for the innermost call, FALSE for the others.  Thus, after one of these procedures returns, the caller has to call NDataPresent (or something analogous) to tell if there's really input/output waiting.  Because ioctl(,FIONREAD,) will report success & 0 bytes available, instead of a connection-closed error, when the peer closes a TCP connection, we need the sure parameter to be able to reliably detect this situation. */

/*L2 >= {fd's connection's callmu, iomu}*/

_FN_DECL(int _ilu_Read, (int fd, ilu_byte *buf, int nbytes));
_FN_DECL(int _ilu_Write, (int fd, ilu_byte *buf, int nbytes));
/* These are like UNIX read and write, except that they iterate over multiple system calls, blocking only the calling thread, until the entire transfer has been completed or has failed.  In single-threaded runtimes, they assume there'll be no nested attempt to do the same kind of I/O on the same FD. */

/*Inside(obj's server, obj's type)*/

void _ilu_DeltaHolds(ilu_Object obj, ilu_integer dholds);

void _ilu_VIUpdate(ilu_Object obj);
/* Call this after holds or gclist's emptiness changes,
   or lastRemote+timeout passes,
   and you're ready to have the server invariant restored.
   L1 mutexes are exited and re-entered inside this procedure! */


/****************************** from type.c ********************/

/*L1, L2, Main unconstrained */

extern ilu_Class _ilu_rootClass;
/* The one with all the methods every object supports */

extern ilu_Method _ilu_GetTypesMethod;
extern ilu_Method _ilu_RegisterGCInterestMethod;
extern ilu_Method _ilu_UnregisterGCInterestMethod;
extern ilu_Method _ilu_PingMethod;
/* The methods of _ilu_rootClass */

/*L1_sup < otmu*/
ilu_Class _ilu_FindMSKA(ilu_string tid);
/* Returns the one most specific known ancestor of the type identified
   by the given string; returns NULL if that's not well-defined, or we
   don't yet know about the ancestry of the given type. */

/*Main Invariant holds; L2 otherwise unconstrained*/

ilu_Class _ilu_FindClassViaRPC (ilu_Object o);
/* o->ob_class is temporarily set to some known type of o */

/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/
void _ilu_HandleGetTypes (ilu_Call call);

/****************************** Hash Table ops ********************/

/*L1, L2, Main unconstrained*/

ilu_boolean _ilu_hash_PointerCompare (ilu_refany key1, ilu_refany key2);

ilu_cardinal _ilu_hash_HashPointer (ilu_refany key, ilu_cardinal size);

ilu_boolean _ilu_hash_StringCompare (ilu_string key1, ilu_string key2);

ilu_cardinal _ilu_hash_HashString (ilu_string key, ilu_cardinal size);

HashTable _ilu_hash_MakeNewTable (ilu_cardinal size,
        ilu_cardinal (*hashfn) (ilu_refany,ilu_cardinal),
        ilu_boolean (*compfn) (ilu_refany,ilu_refany));

/*L1 >= {some mutex that protects the table}*/

ilu_refany _ilu_hash_FindInTable (HashTable ht, ilu_refany key);

ilu_boolean _ilu_hash_AddToTable (HashTable ht, ilu_refany key,
						ilu_refany obj);
/* Applicable only when FindInTable(ht, key)==NULL */

ilu_refany _ilu_hash_RemoveFromTable (HashTable ht, ilu_refany key);
/* Applicable only when FindInTable(ht, key)!=NULL */

void _ilu_hash_FreeHashTable (HashTable ht, void (*freeKey)(ilu_refany),
					    void (*freeData)(ilu_refany));

ilu_cardinal _ilu_hash_PairsInTable (HashTable ht);

void _ilu_hash_TableEnumerate (HashTable ht,
			       void (*proc) (ilu_refany entry_data,
					     ilu_refany rock),
			       ilu_refany rock);

ilu_refany _ilu_hash_FindViaProc (HashTable ht,
				  ilu_boolean (*proc) (ilu_refany entry_data,
						       ilu_refany rock),
				  ilu_refany rock);

void _ilu_hash_BeginEnumeration (HashTable ht, HashEnumerator *he);
/* Initialize an enumeration state. */

ilu_boolean _ilu_hash_Next(HashEnumerator *he, ilu_refany *key,
					       ilu_refany *data);
/* If there's another pair to enumerate, stores the next pair to be enumerated in *key and *data, and returns TRUE; otherwise returns FALSE.  Unless the protecting mutex is held throughout the enumeration, pairs added or removed during the enumeration may or may not be enumerated, and if any pairs are removed, other pairs may be enumerated more than once.  A pair is in the table when it is enumerated. */

/********************* from pipe.c ********************/

/*L1, L2 not designed*/

ilu_string _ilu_Pipe_ContactInfo (ilu_Pipe p);

/********************* from gc.c ********************/

/*L1, L2, Main unconstrained*/

extern const ilu_Class _ilu_GcCallbackClass;

/*L1 >= {gcmu}*/

_FN_DECL(void _ilu_StartGCingTrueObj, ( ilu_Object obj ));

_FN_DECL(void _ilu_StopGCingTrueObj, ( ilu_Object obj ));

/*L1 >= {gcmu, cmu, obj's server}*/
void _ilu_TouchedObj(ilu_Object obj);
/* Applicable to collectible true objects;
   call this when lastRemote or gclist's emptiness changes. */

extern ilu_refany _ilu_gcoAlarm;
extern ilu_refany _ilu_gccAlarm;

/*Main Invariant holds*/
/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/

_FN_DECL(void _ilu_HandleGCInterestDeregistration, ( ilu_Call call ));
_FN_DECL(void _ilu_HandleGCInterestRegistration, ( ilu_Call call ));
/* Server stubs for two built-in methods. */

/*Main Invariant holds; L2 otherwise unconstrained*/

ilu_boolean _ilu_RegisterGCInterest(ilu_Object obj);
ilu_boolean _ilu_UnregisterGCInterest(ilu_Object obj);
/* Notify the true server of surrogate obj's (non-)existance. */

#endif /* ndef _ILU_INTERNALS_ */
