/* ilu.h */
/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* Last tweaked by Mike Spreitzer April 28, 1994 3:15 pm PDT */

#ifndef _ILU_H_
#define _ILU_H_

#include <stdio.h>
#include "oscalls.h"

#if (defined ( __STDC__ ) || defined ( __cplusplus ))
/* So I can write ansi prototypes and still use Sun cc */
#define	_FN_DECL(type_and_name, args) type_and_name args
#else
#define _FN_DECL(type_and_name, args) type_and_name()
#endif

#define OPTIONAL(x)	x
#define PASS(x)		x
#define RETAIN(x)	x
#define GLOBAL(x)	x

/* ==================== basic types ==================== */

typedef enum ilu_boolean_enum { ilu_TRUE = 1, ilu_FALSE = 0 } ilu_boolean;

typedef long int ilu_integer;
typedef short int ilu_shortinteger;
typedef long int ilu_longinteger[2];

typedef unsigned long int ilu_cardinal;
typedef unsigned short int ilu_shortcardinal;
typedef unsigned long int ilu_longcardinal[2];

typedef double ilu_real;
typedef float ilu_shortreal;
typedef double ilu_longreal;

typedef char * ilu_string;
typedef unsigned short *ilu_wstring;
typedef unsigned char * ilu_opaque;
typedef unsigned char * ilu_bytes;
typedef unsigned char ilu_byte;

typedef unsigned short ilu_character;
typedef unsigned char ilu_shortcharacter;

typedef void *ilu_refany;
typedef void *ilu_private;

typedef void *ilu_HashTable;

#define MAX_CCTINFO_LEN 32

/* ==================== internal structs ==================== */

typedef enum ilu_ProtocolTypes {
  ilu_ProtocolType_None,
  ilu_ProtocolType_SunRPC,		/* "sunrpc_2" */
  ilu_ProtocolType_Courier,		/* "xeroxcourier" */
  ilu_ProtocolType_DCE,			/* "osfdce" */
  ilu_ProtocolType_Micro1		/* "micro1" */
} ilu_ProtocolType;

typedef enum ilu_ProtocolExceptions {
  ilu_ProtocolException_Success = 0,
  ilu_ProtocolException_NoSuchClassAtServer = 1,
  ilu_ProtocolException_ClassVersionMismatch = 2,
  ilu_ProtocolException_NoSuchMethodOnClass = 3,
  ilu_ProtocolException_GarbageArguments = 4,
  ilu_ProtocolException_Unknown = 5,
  ilu_ProtocolException_LostConnection = 6,
  ilu_ProtocolException_RequestRejected = 7,
  ilu_ProtocolException_RequestTimeout = 8
} ilu_ProtocolException;

typedef enum ilu_PacketTypes {
  ilu_PacketType_Request = 0,
  ilu_PacketType_Reply = 1
} ilu_PacketType;

typedef enum ilu_TransportTypes {
  ilu_TransportType_None = 0,		/* bogus type */
  ilu_TransportType_TCP = 1,		/* "tcp" */
  ilu_TransportType_CompressedTCP = 2,	/* "ctcp"
				    -- compressed TCP */
  ilu_TransportType_BufferedTCP = 3,	/* "btcp"
				    -- buffered TCP */
  ilu_TransportType_BuffCompTCP = 4,	/* "bctcp"
				    -- buffered compressed TCP */
  ilu_TransportType_UDP = 5,		/* "udp" */
  ilu_TransportType_XNSSPP = 6,		/* "xns" -- XNS SPP */
  ilu_TransportType_BufferedXNSSPP = 7, /* "bxns"
					-- buffered SPP */
  ilu_TransportType_CompressedXNSSPP = 8, /* "cxns"
					-- compressed SPP */
  ilu_TransportType_BuffCompXNSSPP = 9, /* "bcxns"
				    -- buffered compressed SPP */
  ilu_TransportType_Memory = 10		/* "memory" */
} ilu_TransportType;

enum ilu_PipeDirections { ilu_PipeSink, ilu_PipeSource };

enum ilu_InterpretRequestStatus {
  ilu_InterpretRequest_Failed = 1,	/* couldn't interpret packet */
  ilu_InterpretRequest_Succeeded = 2,	/* successful, go ahead and
					   invoke app's method */
  ilu_InterpretRequest_Handled = 3};	/* successful, invoke built-in
					   method*/

typedef struct ilu_FineTime_s ilu_FineTime;

typedef char * ilu_Exception;
	/* address of exception description */

typedef struct _ilu_Server_s * ilu_Server;
	/* handle held by client on server */

typedef struct _ilu_Client_s * ilu_Client;
	/* handle held by server on client */

typedef struct _ilu_Connection_s * ilu_Connection;
	/* wrapper for bytestream connection */

typedef struct _ilu_Object_s * ilu_Object;
	/* wrapper around server for instances */

typedef struct _ilu_Pipe_s * ilu_Pipe;
	/* pipe for connection to other module */

typedef struct _ilu_TransportClass_s *ilu_TransportClass;
	/* methods for Transport */

typedef struct _ilu_Transport_s *ilu_Transport;
	/* abstraction of a TCP/IP or XNS Transport */

typedef struct _ilu_Protocol_s * ilu_Protocol;
	/* protocol binding */

typedef struct _ilu_PortClass_s * ilu_PortClass;
	/* methods for a Port */

typedef struct _ilu_Port_s * ilu_Port;
	/* handle held by server on bytestream */

typedef struct _ilu_Mooring_s * ilu_Mooring;
	/* abstraction of a listen socket */

typedef struct _ilu_Class_s * ilu_Class;
	/* class description for server */

typedef struct _ilu_Method_s * ilu_Method;
	/* method description for server */

typedef struct _ilu_Call_s * ilu_Call;
	/* call description */

typedef ilu_refany ilu_Lock;
	/* slot for lock implementation's use */

struct ilu_FineTime_s {
  ilu_integer ft_s;	/* seconds since some origin */
  ilu_cardinal ft_t;	/* fraction of a second */
};
/* Represents s + t/N seconds since some origin.  0 <= t < N.
   If ilu_FineTimeRate is 0, N is one greater than the largest
   ilu_cardinal; otherwise, N is ilu_FineTimeRate. */

struct _ilu_Protocol_s {
  /* A protocol is never changed or freed once created, so these fields
   * are readonly.  The locking comments refer to invoc'ns of the methods.
   */

  ilu_ProtocolType	pr_type;
  ilu_boolean		pr_concurrent_requests;
  	/* Does this protocol support concurrent requests on
  	   one connection? */

  /*L2 >= {call's connection's iomu}*/
  /*L1, Main unconstrained*/

  /*L1_sup < prmu*/
  _FN_DECL(ilu_boolean (*pr_begin_request), (ilu_Call call,
					  ilu_Class intro_type,
	/* should be the class that originally defined the method */
					  ilu_Method method,
					  ilu_cardinal arg_size));

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*pr_finish_request), (ilu_Call call));
  /* These bracket the sending of the call parameters,
     including discriminator.  This will be followed
     by some number (0 if !call->ca_connection->co_transport->tr_class->tc_timesout)
     of calls on resend_request, and then exactly one call on
     drop_request. */

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*pr_resend_request), (ilu_Call call));
  /* For use with datagram transports. */

  _FN_DECL(void (*pr_drop_request), (ilu_Call call));
  /* resend_request won't be called again for this call. */

  _FN_DECL(ilu_boolean (*pr_begin_reply),
	   (ilu_Call call, ilu_boolean exceptions_possible,
	    ilu_cardinal reply_size));

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*pr_finish_reply), (ilu_Call call));
  /* These bracket the sending of the results.  This proc calls
     _ilu_CacheCall if transport timesout. */

  _FN_DECL(ilu_boolean (*pr_begin_exception),
	  (ilu_Call call, ilu_cardinal exception_code,
	   ilu_cardinal reply_size));

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*pr_finish_exception), ( ilu_Call call ));
  /* Also calls _ilu_CacheCall if transport timesout. */

  /* some functions concerned with string binding handles */
  /*L2, Main unconstrained*/
  /*L1_sup < prmu (prmu protects class<->prog&ver registry)*/
  _FN_DECL(ilu_string (*pr_form_handle),
			( ilu_Object obj));

  /* lower-level functions used by the language runtime */
  /*Main Invariant holds*/
  /*L2 >= {conn's iomu}*/

  _FN_DECL(ilu_bytes (*pr_read_packet), ( ilu_Transport bs,
	ilu_cardinal *packet_length, ilu_PacketType *packet_type,
	ilu_cardinal *serial_number  ));
  /* *packet_type normally becomes a member of ilu_PacketType;
     *serial_number gets a number identifying the call (eg, XID).
     Result is the whole protocol-level packet, including
     *packet_type and *serial_number.
     Caller owns result. */

  _FN_DECL(ilu_boolean (*pr_interpret_pipe_request),
	   (ilu_Call call, ilu_opaque packet,
	    ilu_cardinal packet_length));
  /* We don't really believe in pipes yet. */

  _FN_DECL(ilu_boolean (*pr_interpret_request),
	   (ilu_Call call, ilu_opaque packet,
	    ilu_cardinal packet_length));
  /* Fills in the intro_type and method fields of call;
     serialNumber, server, connection fields are already set.
     Stores in the call's connection's co_protocol_data slot
     some private data used for unmarshalling arguments.
     */

  /*L1, Main unconstrained*/

  _FN_DECL(void (*pr_request_read), (ilu_Call call));
  /* Called after arguments have been unmarshalled.
     Cleans up after use of call's connection's co_protocol_data slot.
     */

  _FN_DECL(ilu_ProtocolException (*pr_interpret_reply),
	(ilu_Call call, ilu_opaque packet,
	 ilu_cardinal packet_length,
	 ilu_cardinal *exception_code));

  _FN_DECL(void (*pr_finish_call), (ilu_Call call));
  /* Called at the end of process of call.
     Cleans up after use of call's connection's co_protocol_data slot.
     */

  /*Main Invariant holds*/
  _FN_DECL(void (*pr_flush), ( ilu_Call call ));

  /*L2, Main unconstrained*/

  _FN_DECL(ilu_refany (*pr_create_data_block), (void));

  _FN_DECL(void (*pr_free_data_block), ( ilu_refany block ));

  /* [Un]Marshalling routines */
  /*L2 >= {call's connection's iomu} for input & output,
    L2 unconstrained for sizing*/
  /*L1, Main unconstrained*/
  /*(When we cease buffering entire messages: Main Invariant holds)*/
  
  _FN_DECL(ilu_boolean (*pr_output_optional),   (ilu_Call call,
					      ilu_boolean i));
  _FN_DECL(ilu_boolean (*pr_input_optional),    (ilu_Call call,
					      ilu_boolean *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_optional), (ilu_Call call,
					      ilu_boolean i));

  _FN_DECL(ilu_boolean (*pr_output_integer),   (ilu_Call call,
					     ilu_integer i));
  _FN_DECL(ilu_boolean (*pr_input_integer),    (ilu_Call call,
					     ilu_integer *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_integer), (ilu_Call call,
					     ilu_integer i));

  _FN_DECL(ilu_boolean (*pr_output_cardinal),   (ilu_Call call,
					      ilu_cardinal i));
  _FN_DECL(ilu_boolean (*pr_input_cardinal),    (ilu_Call call,
					      ilu_cardinal *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_cardinal), (ilu_Call call,
					      ilu_cardinal i));

  _FN_DECL(ilu_boolean (*pr_output_shortinteger),
			(ilu_Call call, ilu_shortinteger i));
  _FN_DECL(ilu_boolean (*pr_input_shortinteger),
			(ilu_Call call, ilu_shortinteger *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_shortinteger),
			(ilu_Call call, ilu_shortinteger i));

  _FN_DECL(ilu_boolean (*pr_output_shortcardinal),
			(ilu_Call call, ilu_shortcardinal i));
  _FN_DECL(ilu_boolean (*pr_input_shortcardinal),
			(ilu_Call call, ilu_shortcardinal *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_shortcardinal),
			(ilu_Call call, ilu_shortcardinal i));

  _FN_DECL(ilu_boolean (*pr_output_real),
			( ilu_Call call, ilu_real i ));
  _FN_DECL(ilu_boolean (*pr_input_real),
			( ilu_Call call, ilu_real *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_real),
			( ilu_Call call, ilu_real i ));

  _FN_DECL(ilu_boolean (*pr_output_shortreal),
			( ilu_Call call, ilu_shortreal i ));
  _FN_DECL(ilu_boolean (*pr_input_shortreal),
			( ilu_Call call, ilu_shortreal *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_shortreal),
			( ilu_Call call, ilu_shortreal i ));

  _FN_DECL(ilu_boolean (*pr_output_longinteger),
			(ilu_Call call, ilu_longinteger i));
  _FN_DECL(ilu_boolean (*pr_input_longinteger),
			(ilu_Call call, ilu_longinteger *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_longinteger),
			(ilu_Call call, ilu_longinteger i));

  _FN_DECL(ilu_boolean (*pr_output_longcardinal),
			(ilu_Call call, ilu_longcardinal i));
  _FN_DECL(ilu_boolean (*pr_input_longcardinal),
			(ilu_Call call, ilu_longcardinal *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_longcardinal),
			(ilu_Call call, ilu_longcardinal i));

  _FN_DECL(ilu_boolean (*pr_output_longreal),
			( ilu_Call call, ilu_longreal i ));
  _FN_DECL(ilu_boolean (*pr_input_longreal),
			( ilu_Call call, ilu_longreal *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_longreal),
			( ilu_Call call, ilu_longreal i ));

  _FN_DECL(ilu_boolean (*pr_output_byte),
			( ilu_Call call, ilu_byte i ));
  _FN_DECL(ilu_boolean (*pr_input_byte),
			( ilu_Call call, ilu_byte *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_byte),
			( ilu_Call call, ilu_byte i ));

  _FN_DECL(ilu_boolean (*pr_output_boolean),
			( ilu_Call call, ilu_boolean i ));
  _FN_DECL(ilu_boolean (*pr_input_boolean),
			( ilu_Call call, ilu_boolean *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_boolean),
			( ilu_Call call, ilu_boolean i ));

  /* this is a Unicode character */
  _FN_DECL(ilu_boolean (*pr_output_character),
			( ilu_Call call, ilu_character i ));
  _FN_DECL(ilu_boolean (*pr_input_character),
			( ilu_Call call, ilu_character *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_character),
			( ilu_Call call, ilu_character i ));

  /* this is an ISO Latin-1 character using the ISO 8859-1 encoding */
  _FN_DECL(ilu_boolean (*pr_output_shortchar),
			( ilu_Call call, ilu_shortcharacter i ));
  _FN_DECL(ilu_boolean (*pr_input_shortchar),
			( ilu_Call call, ilu_shortcharacter *i ));
  _FN_DECL(ilu_cardinal (*pr_size_of_shortchar),
			( ilu_Call call, ilu_shortcharacter i ));

  _FN_DECL(ilu_boolean (*pr_output_enum_code),
			( ilu_Call call, ilu_shortcardinal i  ));
  _FN_DECL(ilu_boolean (*pr_input_enum_code),
			( ilu_Call call, ilu_shortcardinal *i  ));
  _FN_DECL(ilu_cardinal (*pr_size_of_enum_code),
			( ilu_Call call, ilu_shortcardinal i  ));

  /* a string is a sequence of shortcharacter */
  _FN_DECL(ilu_boolean (*pr_output_string),
	(ilu_Call call, ilu_string s, ilu_cardinal length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_boolean (*pr_input_string),
	(ilu_Call call, ilu_string *s, ilu_cardinal *length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_cardinal (*pr_size_of_string),
	(ilu_Call call, ilu_string s, ilu_cardinal length,
	 ilu_cardinal limit));

  /* a stringvec is a vector of shortcharacter --
   * not null-terminated */
  _FN_DECL(ilu_boolean (*pr_output_stringvec),
	(ilu_Call call, ilu_string s, ilu_cardinal length));
  _FN_DECL(ilu_boolean (*pr_input_stringvec),
	(ilu_Call call, ilu_string *s, ilu_cardinal length));
  _FN_DECL(ilu_cardinal (*pr_size_of_stringvec),
	(ilu_Call call, ilu_string s, ilu_cardinal length));

  /* a wstring is a sequence of character */
  _FN_DECL(ilu_boolean (*pr_output_wstring),
	(ilu_Call call, ilu_wstring s, ilu_cardinal length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_boolean (*pr_input_wstring),
	(ilu_Call call, ilu_wstring *s, ilu_cardinal *length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_cardinal (*pr_size_of_wstring),
	(ilu_Call call, ilu_wstring s, ilu_cardinal length,
	 ilu_cardinal limit));

  /* a wstringvec is a vector of character --
   * not null-terminated */
  _FN_DECL(ilu_boolean (*pr_output_wstringvec),
	(ilu_Call call, ilu_wstring s, ilu_cardinal length));
  _FN_DECL(ilu_boolean (*pr_input_wstringvec),
	(ilu_Call call, ilu_wstring *s, ilu_cardinal length));
  _FN_DECL(ilu_cardinal (*pr_size_of_wstringvec),
	(ilu_Call call, ilu_wstring s, ilu_cardinal length));

  /* opaque is a vector of bytes */
  _FN_DECL(ilu_boolean (*pr_output_opaque),
	(ilu_Call call, ilu_opaque s, ilu_cardinal length));
  _FN_DECL(ilu_boolean (*pr_input_opaque),
	(ilu_Call call, ilu_opaque *s, ilu_cardinal length));
  _FN_DECL(ilu_cardinal (*pr_size_of_opaque),
	(ilu_Call call, ilu_opaque s, ilu_cardinal length));

  /* bytes is a sequence of bytes */
  _FN_DECL(ilu_boolean (*pr_output_bytes),
	(ilu_Call call, ilu_bytes s, ilu_cardinal length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_boolean (*pr_input_bytes),
	(ilu_Call call, ilu_bytes *s, ilu_cardinal *length,
	 ilu_cardinal limit));
  _FN_DECL(ilu_cardinal (*pr_size_of_bytes),
	( ilu_Call call, ilu_bytes s, ilu_cardinal length,
	  ilu_cardinal limit));

  _FN_DECL(ilu_boolean (*pr_output_pipe),
				(ilu_Call call, ilu_Pipe i));
  _FN_DECL(ilu_boolean (*pr_input_pipe),
				(ilu_Call call, ilu_Pipe *i));
  _FN_DECL(ilu_cardinal (*pr_size_of_pipe),
				(ilu_Call call, ilu_Pipe i));

  _FN_DECL(ilu_boolean (*pr_output_sequence),
	   (ilu_Call call, ilu_cardinal length, ilu_cardinal limit));
  _FN_DECL(ilu_boolean (*pr_output_sequence_mark),
	   (ilu_Call call, ilu_cardinal extent));
	/* called every 2^16-1 items */
  _FN_DECL(ilu_boolean (*pr_input_sequence),
			(ilu_Call call, ilu_cardinal *length,
			 ilu_cardinal limit));
  _FN_DECL(ilu_boolean (*pr_input_sequence_mark),
	   (ilu_Call call, ilu_cardinal extent));
	/* called every 2^16-1 items */
  _FN_DECL(ilu_boolean (*pr_end_sequence),
			(ilu_Call call));
  _FN_DECL(ilu_cardinal (*pr_size_of_sequence),
			( ilu_Call call, ilu_cardinal length,
			  ilu_cardinal limit));

  _FN_DECL(ilu_boolean (*pr_output_array),
			( ilu_Call call, ilu_cardinal length ));
  _FN_DECL(ilu_boolean (*pr_input_array),
			( ilu_Call call ));
  _FN_DECL(ilu_boolean (*pr_end_array),
			( ilu_Call call ));
  _FN_DECL(ilu_cardinal (*pr_size_of_array),
			( ilu_Call call, ilu_cardinal length ));

  _FN_DECL(ilu_boolean (*pr_output_record),
				( ilu_Call call ));
  _FN_DECL(ilu_boolean (*pr_input_record),
				( ilu_Call call ));
  _FN_DECL(ilu_boolean (*pr_end_record),
				( ilu_Call call ));
  _FN_DECL(ilu_cardinal (*pr_size_of_record),
				( ilu_Call call ));

  _FN_DECL(ilu_boolean (*pr_output_union),
			( ilu_Call call, ilu_shortcardinal discrim ));
  _FN_DECL(ilu_boolean (*pr_input_union),
			( ilu_Call call, ilu_shortcardinal *typeIndex ));
  _FN_DECL(ilu_boolean (*pr_end_union),
			( ilu_Call call ));
  _FN_DECL(ilu_cardinal (*pr_size_of_union),
			( ilu_Call call, ilu_shortcardinal typeIndex ));

  _FN_DECL(ilu_boolean (*pr_skip_bytes),
			( ilu_Call call, ilu_cardinal numBytes  ));

};

struct _ilu_TransportClass_s {
  /* A transport class is never changed or freed once created, so
   * these fields are readonly.  The locking comments refer to
   * invocationns of the methods. */

  ilu_TransportType	tc_type;
  ilu_cardinal		tc_pFD;	/* # of FDs consumed by a port */
  ilu_cardinal		tc_cFD;	/* # of FDs consumed by a conection */
  ilu_boolean		tc_timesout; /* Retransmissions required? */
  
  /*L1, L2, Main unconstrained*/

  _FN_DECL(ilu_refany (*tc_interpret_info), (ilu_string tinfo));
  /* Interpret transport info, producing private data used in either
     a Mooring or a Transport.  The string arg is owned by the caller.
     The result is owned by the caller (who will put it in a Mooring or
     Transport). */

  _FN_DECL(ilu_string (*tc_form_info), (ilu_refany parameterblock));
  /* Produce a fully-detailed transport info, from the private data of
     either a Mooring or a Transport.  Caller owns arg & result. */

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*tc_connect), (ilu_Transport self));

  /*Main Invariant holds*/
  _FN_DECL(ilu_boolean (*tc_wait_for_input), (ilu_Transport bs,
					   ilu_boolean *sure,
					   ilu_FineTime *limit));
  /* Returns FALSE if error occurred, TRUE if input maybe available.
     Returns TRUE and *sure = TRUE if input or EOF certainly available. */

  /*L1_sup < trmu*/
  _FN_DECL(ilu_Mooring (*tc_create_mooring), (ilu_private mooringInfo));
  
  _FN_DECL(void (*tc_close_mooring), (ilu_Mooring m));

  /*L2 >= {conn's iomu}*/

  _FN_DECL(void (*tc_close), (ilu_Transport bs));

  _FN_DECL(ilu_integer (*tc_n_data_present), (ilu_Transport bs));
  /* positive result <=> some data available;
     negative result <=> transport broken;
     zero result <=> no more data available yet */

  /*Main Invariant holds*/

  _FN_DECL(ilu_boolean (*tc_send_message), (ilu_Transport bs,
	ilu_bytes message, ilu_cardinal messageLength));
  /* Caller retains ownership of message iff bs->tr_class->tc_timesout. */

  _FN_DECL(ilu_boolean (*tc_read_message), (ilu_Transport bs,
	ilu_bytes *message, ilu_cardinal *messageLength ));
  /* Caller owns result. */
  
  _FN_DECL(ilu_boolean (*tc_flush_output), ( ilu_Transport bs));

};

struct _ilu_Transport_s {
  /*L1, L2, Main unconstrained*/

  ilu_TransportClass tr_class;	/* methods and type */
  int tr_fd;			/* every Transport has
				   a UNIX file descriptor */
  ilu_FineTime	tr_to1;		/* Initial timeout */
  ilu_FineTime	tr_toN;		/* Max timeout */
  ilu_FineTime	tr_tto;		/* Total timeout */

  ilu_refany tr_data;		/* For use by class; produced
				   by class->tc_interpret_info */
};

typedef struct {
  ilu_Connection next, prev;	/* links in doubly-linked list */
} ilu_ConnLinks;

struct _ilu_Connection_s {
  /*L1 >= {server}; L2, Main unconstrained*/

  ilu_Call co_mucall;		/* the one holding my call mutex */
  ilu_boolean co_ioing;		/* implements iomu */
  
  /*L1 unconstrained*/
  
  ilu_Protocol co_protocol;
  ilu_refany co_protocol_data;
  ilu_string co_tinfo;		/* never NULL; owned by conn; complete,
				   canonical, and < MAX_CCTINFO_LEN long
				   for incoming connections */
  ilu_string co_pinfo;		/* never NULL; conn owns; not complete */
  ilu_Transport co_transport;	/* actual byte stream */
  ilu_Port co_port;		/* NULL on client side,
				   points to port on server side */
  ilu_private co_auth_info;	/* information about the identity
				   of the caller */
  ilu_Server co_server;		/* never NULL */

  /*L2 >= {iomu}*/

  /* The following three fields are meaningful if the protocol is
     concurrent and the connection is outgoing. */
  ilu_private	*co_replies;		/* queue of unprocessed replies */
  
  /* The following two fields are meaningful if furthermore
     the runtime can do condition variables; NULL otherwise. */
  /*L1 >= {server}; L2 unconstrained*/

  ilu_Call co_reader;		/* the one reading input, queuing it */
  ilu_private co_cc;		/* cond var for changes under server */
  
  ilu_boolean co_closed;
  ilu_cardinal co_next_sn;	/* meaningful for outgoing connections;
				   the next request msg serial number */
  ilu_integer co_nOuts;		/* For concurrent protocols, the
				   number of calls between holding callmu
				   for request and for reply; else 0. */

  /*L1 >= {server} for [ilu_psl], which are for the doubly-linked list of
				the port's or server's connections;
   *L1 >= {cmu} for [ilu_lru],	which are for the global doubly-linked
				LRU list of idle connections. */
  ilu_ConnLinks co_links[2];	/* NULL-terminated, not circular */
  
};

typedef enum {ilu_psl, ilu_lru} ilu_ConnLinkKind;

struct _ilu_Method_s {
  /* read-only; no locks needed */
  ilu_string me_name;
  ilu_cardinal me_id;
  ilu_boolean me_cacheable;	/* functional? */
  ilu_boolean me_asynchronous;	/* need to wait after calling? */
  ilu_Exception *me_exceptionVector; /* list of possible exceptions */
  ilu_cardinal me_exceptionCount;/* number of exceptions in list */
  
  /*Main Invariant holds*/
  /*L2    >=    {conn's callmu, iomu} before,
    L2 disjoint {conn's callmu, iomu} after*/
  void (*me_stubproc) (ilu_Call);
};

struct _ilu_Class_s {
  /*L1, L2, Main unconstrained*/
  
  ilu_string cl_name;		/* ILU name of class */
  ilu_string cl_brand;		/* brand on class */
  ilu_string cl_unique_id;	/* unique id for type graph
				   of class */
  ilu_boolean cl_singleton;	/* TRUE (1) if class is
				   a singleton */
  ilu_boolean cl_collectible;
  ilu_string cl_authentication;	/* non-NULL with name of
				   authentication type, if any */
  ilu_Method cl_methods;
  ilu_cardinal cl_method_count;
  ilu_cardinal cl_scls_count;	/* number of superclasses */
  ilu_string *cl_scls_ids;	/* address of vector of strings */

  /*L1 >= {otmu}*/
  
  ilu_Class *cl_sclses;		/* address o vector o ptrs to supercls */
  ilu_boolean cl_shown;		/* temp bit for use in type.c */
};

struct _ilu_Call_s {
  /*L1, L2, Main unconstrained*/
  
  ilu_cardinal	ca_SN;		/* serial number of request */
  ilu_Server	ca_server;		/* of call */
  ilu_Class	ca_intro_type;	/* Significant in incoming calls */
  ilu_Method	ca_method;		/* ID number of method */
  ilu_Connection ca_connection;	/* connection which points
				   to (server or client) */
  ilu_refany	ca_private;	/* slot for lang-specific
				   implementation's use */
};

struct _ilu_Pipe_s {
  /*L1, L2 not designed yet*/
  ilu_Lock pi_lock;
  ilu_Mooring pi_mooring;		/* non-NULL for local pipes */
  ilu_Call pi_call;		/* wrapper around connection */
  ilu_Protocol pi_protocol;	/* protocol for pipe object */
  ilu_string pi_contact_info;	/* contact info for transport
				   connection */
  ilu_boolean pi_sink_p;		/* pipe end is a sink */
  ilu_boolean pi_valid;		/* pipe can be read/written */
  ilu_string pi_type;		/* name of type of pipe */
};

struct _ilu_Mooring_s {		/* transport-level port */
  /*L1, L2, Main unconstrained*/
  
  int mo_fd;			/* has an FD, like a Transport */
  
  /*For calling L1_sup < trmu; L2, Main unconstrained*/
  _FN_DECL(ilu_Transport (*mo_accept_connection), ( ilu_Mooring self ));
  
  ilu_TransportClass mo_transportClass;
				/* kind of transport returned by
				   accept_connection */
  ilu_private mo_data;		/* For use by transportClass; produced
				   by transportClass->tc_interpret_info */
};

typedef struct {
  char		cc_tinfo[MAX_CCTINFO_LEN]; /* Address of requestor */
  ilu_cardinal	cc_sn;
  ilu_Class	cc_intro_type;
  ilu_Method	cc_meth;
  ilu_bytes	cc_reply;
  ilu_cardinal	cc_reply_len;
} ilu_CachedCall;

struct _ilu_Port_s {		/* where a server listens
				   for connection requests */
  /*L1, L2, Main unconstrained*/
  
  ilu_Server po_server;		/* The server exported via this port. */
  ilu_string po_pinfo;		/* given at create time;
				   owned by port */
  ilu_Protocol po_protocol;	/* protocol being used */
  ilu_string po_tinfo;		/* transport info for this port;
				   owned by the port */
  ilu_Mooring po_mooring;		/* byte stream socket this port
				   listens on */
  
  /*L1 >= {server}*/
  
  ilu_boolean po_closed;
  ilu_Port po_next;		/* ..in chain of server's ports */
  ilu_ConnLinks po_connHead;	/* head of chain of conns from me */
  ilu_CachedCall *po_call_cache;	/* non-NULL when transport timesout */
  int po_call_cache_size;		/* non-0 when transport timesout */
  int po_call_cache_finger;	/* cache is a FIFO; next goes here */
};

typedef struct {
  /* These fields are readonly. */
  /*L2, Main unconstrained for calling*/
	
  /*L1 unconstrained*/
  _FN_DECL(ilu_private (*lt_mcreate), (ilu_string d1, ilu_string d2));
	
  /*L1_sup < m before, L1_sup = m after*/
  _FN_DECL(void (*lt_acquire ), (ilu_private m));
	
  /*L1 >= {m}*/
  _FN_DECL(void (*lt_hold    ), (ilu_private m));
	
  /*L1 >= {m} before, L1 not >= {m} after*/
  _FN_DECL(void (*lt_release ), (ilu_private m));
	
  /*L1 unconstrained*/
  _FN_DECL(void (*lt_mdestroy), (ilu_private m));
	
  /*L1 unconstrained*/
	
  _FN_DECL(ilu_private (*lt_ccreate), (ilu_string d1, ilu_string d2));
  _FN_DECL(void (*lt_notify  ), (ilu_private c));
  _FN_DECL(void (*lt_cdestroy), (ilu_private c));
	
  /*L1_sup = m*/
  _FN_DECL(void (*lt_wait    ), (ilu_private c, ilu_private m));
	
} ilu_LockTech;

#include <iluxport.h>

struct _ilu_Server_s {
  /*L1, L2, Main unconstrained*/
  
  ilu_Lock sr_lock;
  ilu_boolean sr_true;		/* is this a true server? */
  ilu_string sr_id;		/* part of an oid */
  
  /* The next four fields are meaningful for outgoing connections. */
  
  ilu_string sr_tinfo;		/* transport info */
  ilu_TransportClass sr_trcls;	/* from tinfo */
  ilu_string sr_pinfo;		/* protocol info */
  ilu_Protocol sr_protocol;	/* from pfino */
  
  /*L1 >= {self}*/
  
  ilu_boolean sr_closing;		/* ilu_BankServer sets this TRUE */
  ilu_ConnLinks sr_connHead;	/* meaningful for surrogates:
				   list of open connections */
  ilu_Port sr_ports;		/* head of chain of ports usable for
				   contacting this true server. */
  ilu_HashTable sr_objs;		/* maps ih -> ilu_Object,
				   including singletons;
				   goes NULL when closing &&
				   all objects deleted */
  ilu_HashTable sr_singles;	/* maps singleton ilu_Class -> ilu_Object;
				   goes NULL when closing &&
				   all objects deleted */
  ilu_ObjectTable sr_objtab;	/* significant in a true server; if not
				   NULL, the app's chance to instantiate
				   an object not found in objs */
  ilu_Port sr_default_port;	/* default port; could be NULL */
};

struct _ilu_Object_s {
  /*L1, L2, Main unconstrained*/
  
  ilu_string ob_oid;		/* ih@server-id; never NULL */
  ilu_string ob_ih;		/* id wrt server; never NULL */
  ilu_Server ob_server;		/* never NULL */
  unsigned long ob_timeout;	/* after which to GC */
  ilu_Class  ob_class;
    /* Can only be NULL when doing _ilu_FindClassViaRPC.
       No locking requirements because ilu_ObjectOfSBH doesn't insert
       an object in its server's hash table 'till after class is set. */
  ilu_string ob_mstid;		/* unique_id of true object's most
				   specific type (may be NULL) */
  /*L1 >= {server}*/

  ilu_string ob_sbh;		/* really a cache, lazily eval'ed;
				   may be NULL at any time */
  ilu_private ob_lspo;		/* slot in which to store LS pointer */
  ilu_integer ob_holds;		/* Count of uses in root method stubs
				   and gc.c. */
  int ob_intNoted;			/* LSR thinks this about whether
				   kernel is very interested in me */

  /* The following fields are meaningful only for coll. surrogate objs */
  
  int ob_notifying;		/* we're calling true server now */
  int ob_known;			/* when last !notifying, did true
				   server know about this surrogate? */
  
  /* The following fields are meaningful only for collectible true objs */
  
  /*L1 >= {gcmu}*/
  
  ilu_Alarmette_s ob_gco;		/* mxamu==gcmu */
  unsigned long ob_lastRemote;	/* when we last knew the object to be
				   loose in the network; not meaningful
				   while gclist non-empty */
  void * ob_gclist;		/* set of gc.c:counted_client*;
				   NULL may represent the empty set */
};
/*
The string fields are owned by the object.

The kernel is very interested in this object if either:
* gclist not empty (for collectible true objects);
* gclist empty and now < lastRemote+timeout (for coll. true objects);
* notifying && !known (for collectible surrogates);
* holds != 0.

As long as either:
* gclist not empty (for collectible true objects);
* gclist empty and now < lastRemote+timeout (for coll. true objects);
* notifying (for collectible surrogates);
* holds != 0;
* lspo != NULL;
the kernel is "interested" in this object, the object is in the server's hash table, and the object is not freed.

The kernel needs to make a GC callback when
	known != (lspo != NULL || holds != 0).
*/

#endif /* _ILU_H_ */
