/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: dcerpc.c,v 1.2 1994/04/06 22:30:00 printis Exp $ */
/* Last tweaked by Mike Spreitzer January 19, 1994 7:43 am PST */

#ifdef SUNOS
#define _BSD_SOURCE
#include <sys/types.h>
#include <unistd.h>		/* for gethostname(), getgeid(), etc. */
#else /* not SUNOS */
#define _POSIX_SOURCE
#endif


#ifdef MACOS
#pragma segment ilu
#endif

#include <string.h>
#include "ilu.h"
#include "iluntrnl.h"
#include "packet.h"

#include "dcerpc.h"

#include "call.h"
#include "protocol.h"
#include "connect.h"
#include "pipe.h"
#include "transprt.h"
#include "port.h"
#include "object.h"
#include "type.h"
#include "method.h"
#include "mooring.h"

/*************************************************************/
/*		Implement NDR PACKET			     */
/*************************************************************/

/*L1, Main unconstrained*/
/*L2 >= call's conn's iomu*/

#define ODD(x)		(((x)&0x1)!=0)
#define EVEN(x)		(((x)&0x1)==0)  
#define PADDED_SIZE(x)	((((unsigned) (x))+3) & (~0x3))
  
static ilu_boolean _ndr_put_u_short (PACKET p,ilu_shortcardinal l)
{
  if (packet_allocation(p) - packet_size(p) >= 2)
    {
      *p->next++ = (l >> 8) & 0xFF;
      *p->next++ = l & 0xFF;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _ndr_put_long(PACKET p, long l)
{
  return (_ndr_put_u_short (p, (l >> 16) & 0xFFFF) && _ndr_put_u_short (p, l & 0xFFFF));
}

static ilu_boolean _ndr_put_short (PACKET p, short l)
{
  return (_ndr_put_u_short (p, (ilu_shortcardinal) l));
}

static ilu_boolean _ndr_put_u_long (PACKET p, ilu_cardinal l)
{
  return (_ndr_put_u_short (p, (l >> 16) & 0xFFFF) && _ndr_put_u_short (p, l & 0xFFFF));
}

static ilu_boolean _ndr_put_byte (PACKET p, char l)
{
  return (_ndr_put_u_long (p, (unsigned char) l));
}

static ilu_boolean _ndr_put_u_byte (PACKET p, ilu_byte l)
{
  return (_ndr_put_u_long (p, l));
}

static ilu_boolean _ndr_put_float (PACKET p, float l)
{
  float l2 = l;

  return (_ndr_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

static ilu_boolean _ndr_put_double (PACKET p, double l)
{
  double l2 = l;

  return (_ndr_put_u_long (p, ((ilu_cardinal *) &l2)[1]) && _ndr_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

/*L1, L2, Main unconstrained*/
static ilu_cardinal dceRPCMaxStringSize = 65535;

static ilu_boolean _ndr_put_bytes (PACKET p, ilu_byte *b, ilu_cardinal l)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

/*
  DEBUG(DCERPC_DEBUG, (stderr, "_ndr_put_bytes:  allocation 0x%x, size 0x%x (diff %d), needed %u\n",
			packet_allocation(p), packet_size(p),
			packet_allocation(p) - packet_size(p),
			(4 + PADDED_SIZE(l))));
*/
			
  if (l > dceRPCMaxStringSize)
    {
      fprintf (stderr,
	       "%s %u, which exceeds dceRPCMaxStringSize value of %u.\n",
	       "Attempt to pass string of length", l,
	       dceRPCMaxStringSize);
      return (FALSE);
    }

  if (packet_allocation(p) - packet_size(p) >= (4 + PADDED_SIZE(l)))
    {
      register ilu_integer i;

      _ndr_put_u_long (p, l);
      for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	*p1++ = *p2++;
      for (i = PADDED_SIZE(l) - l;  i > 0;  i--)
	*p1++ = 0;
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _ndr_put_opaque (PACKET p, ilu_byte *b, ilu_cardinal l)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) >= PADDED_SIZE(l))
    {
      register ilu_integer i;

      for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	*p1++ = *p2++;
      for (i = PADDED_SIZE(l) - l;  i > 0;  i--)
	*p1++ = 0;
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _ndr_get_u_short (PACKET p, ilu_shortcardinal *l)
{
  if (packet_allocation(p) - packet_size(p) < 2)
    return(FALSE);
  *l = (p->next[0] << 8) + p->next[1];
  p->next += 2;
  return (TRUE);
}

static ilu_boolean _ndr_get_long (PACKET p, long *l)
{
  return (_ndr_get_u_short(p, ((ilu_shortcardinal *) l))
	  && _ndr_get_u_short (p, ((ilu_shortcardinal *) l) + 1));
}

static ilu_boolean _ndr_get_short (PACKET p, short *l)
{
  return (_ndr_get_u_short(p, (ilu_shortcardinal *) l));
}

static ilu_boolean _ndr_get_u_long (PACKET p, ilu_cardinal *l)
{
  ilu_boolean stat;

  stat = _ndr_get_u_short(p, (ilu_shortcardinal *) l)
	 && _ndr_get_u_short (p, ((ilu_shortcardinal *) l) + 1);
  return stat;
}

static ilu_boolean _ndr_get_byte (PACKET p, char *l)
{
  ilu_cardinal l2;
  ilu_boolean stat;

  stat = _ndr_get_u_long(p, &l2);
  if (stat)
    *l = (char) (unsigned char) (l2 & 0xFF);
  return (stat);
}

static ilu_boolean _ndr_get_u_byte (PACKET p, ilu_byte *l)
{
  long l2;
  ilu_boolean stat;

  stat = _ndr_get_long(p, &l2);
  if (stat)
    *l = (ilu_byte) (l2 & 0xFF);
  return (stat);
}

static ilu_boolean _ndr_get_float(PACKET p, float *l)
{
  return (_ndr_get_u_long (p, (ilu_cardinal *) l));
}

static ilu_boolean _ndr_get_double (PACKET p, double *l)
{
  return (_ndr_get_u_long (p, ((ilu_cardinal *) l) + 1)
	  && _ndr_get_u_long(p, ((ilu_cardinal *) l)));
}

static ilu_boolean _ndr_get_bytes(PACKET p, ilu_byte **b, ilu_cardinal *l, ilu_cardinal limit)
{
  ilu_cardinal l2;
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) < 4
      || (!_ndr_get_u_long (p, &l2))
      || packet_allocation(p) - packet_size(p) < PADDED_SIZE(l2))
    return (FALSE);

  *l = 0;
  if (l2 > dceRPCMaxStringSize OR (limit > 0 AND l2 > limit))
    {
      fprintf (stderr,
	       "%s %u, which exceeds dceRPCMaxStringSize value of %u or call limit of %u.\n",
	       "Attempt to read string of length", l2,
	       dceRPCMaxStringSize, limit);
      return (FALSE);
    }

  {
    register ilu_integer i;

    if (*b == NULL)
      *b = (ilu_byte *) malloc(l2 + 1);
    for (p1 = *b, p2 = p->next, c = l2;  c > 0;  c--)
      *p1++ = *p2++;
    for (i = PADDED_SIZE(l2) - l2;  i > 0;  i--)
      p2++;	/* eat padding characters */
    *p1 = '\0';
    p->next = p2;
    *l = l2;
    return (TRUE);
  }
}

static ilu_boolean _ndr_get_opaque (PACKET p, ilu_byte **b, ilu_cardinal len)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;
  register ilu_integer i;

  if (packet_allocation(p) - packet_size(p) < PADDED_SIZE(len))
    return (FALSE);

  if (*b == NULL)
    *b = (ilu_byte *) malloc(len);
  for (p1 = *b, p2 = p->next, c = len;  c > 0;  c--)
    *p1++ = *p2++;
  for (i = PADDED_SIZE(len) - len;  i > 0;  i--)
    p2++;	/* eat padding characters */
  p->next = p2;
    return (TRUE);
}

static void _ndr_destroy (PACKET p)
{
  if (p->buf != NULL && p->count > 0)
    free (p->buf);
  free (p);
}

static PACKET _ndr_CreatePacket (ilu_bytes buf, ilu_cardinal size)
{
  PACKET p = (PACKET) malloc (sizeof(struct ilu_packet_s));
  if (buf != NULL)
    p->buf = buf;
  else if (size > 0)
    p->buf = (ilu_byte *) malloc(size);
  else
    p->buf = NULL;
  p->count = size;
  p->next = p->buf;

  p->put_long = _ndr_put_long;
  p->put_byte = _ndr_put_byte;
  p->put_short = _ndr_put_short;
  p->put_u_long = _ndr_put_u_long;
  p->put_u_byte = _ndr_put_u_byte;
  p->put_u_short = _ndr_put_u_short;
  p->put_float = _ndr_put_float;
  p->put_double = _ndr_put_double;
  p->put_bytes = _ndr_put_bytes;
  p->put_opaque = _ndr_put_opaque;

  p->get_long = _ndr_get_long;
  p->get_byte = _ndr_get_byte;
  p->get_short = _ndr_get_short;
  p->get_u_long = _ndr_get_u_long;
  p->get_u_byte = _ndr_get_u_byte;
  p->get_u_short = _ndr_get_u_short;
  p->get_float = _ndr_get_float;
  p->get_double = _ndr_get_double;
  p->get_bytes = _ndr_get_bytes;
  p->get_opaque = _ndr_get_opaque;

  p->destroy = _ndr_destroy;

  return (p);
}

/**********************************************************************
  End of packet implementation
***********************************************************************/

/*L1, L2, Main unconstrained*/

static ilu_refany _dcerpc_CreateDataBlock (void)
{
  struct dceRPC *new = (struct dceRPC *) malloc(sizeof(struct dceRPC));

  new->dce_in = NULL;
  new->dce_out = NULL;
  return ((ilu_refany) new);
}

static void _dcerpc_FreeDataBlock (struct dceRPC *d)
{
  if (d->dce_in != NULL)
    {
      packet_destroy (d->dce_in);
      d->dce_in = NULL;
    }
  if (d->dce_out != NULL)
    {
      packet_destroy (d->dce_out);
      d->dce_out= NULL;
    }
  free (d);
}

static void formRegistryName (ilu_string buf)
{
  ilu_string registryhome = (ilu_string ) getenv("ILUREGISTRY");
  ilu_string iluhome = (ilu_string ) getenv ("ILUHOME");

#ifdef MACOS
  if (registryhome != NULL)
    sprintf (buf, "%s:%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s:lib:%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s:%s", REGISTRY_LAST_RESORT,
	     PROGRAM_NUMBER_REGISTRY_NAME);
#else
  if (registryhome != NULL)
    sprintf (buf, "%s/%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s/lib/%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s/%s", REGISTRY_LAST_RESORT,
	     PROGRAM_NUMBER_REGISTRY_NAME);
#endif
}

/*L1 >= {prmu}*/

static HashTable ProgramNumberHashTable = NULL;
/* program number -> dcerpcinfo* */

static HashTable RegistryHashTable = NULL;
/* type unique_id -> dcerpcinfo* */

static void _dcerpc_EnsureRegistries(void)
{
  dcerpcinfo *s = NULL;

  if (RegistryHashTable == NULL)
    {
      char buf[1000], line[1000], name[1000], id[1000];
      FILE *f;
      long lineno;

      RegistryHashTable = _ilu_hash_MakeNewTable (137, NULL, NULL);
      ProgramNumberHashTable = _ilu_hash_MakeNewTable (137,
		_ilu_hash_HashPointer, _ilu_hash_PointerCompare);

      formRegistryName (buf);
      if ((f = fopen(buf, "r")) == NULL)
	{
	  fprintf (stderr, "%s program number registry file %s.\n",
		   "_dcerpc_EnsureRegistries:  Can't access", buf);
	  return;
	}
      lineno = 0;
      while (fgets(line, sizeof(line), f) != NULL)
	{
	  lineno++;
     	  s = (dcerpcinfo *) malloc(sizeof(dcerpcinfo));
	  s->sui_class = NULL;
	  if (strncmp(line, ".LastAssignedProgramNumber ", 27) == 0)
	    ; /* skip this record */
	  else {
#ifdef MACOS
            /* AC: all the following code 'fixed' to do a hand-parse due
	       to broken sscanf() */
                /* NB: code now does a return if any line from registry
			is malformed */
		/* NB: We also don't use isspace() to advance over
		 * whitespace, because isspace() is also broken on the
		 * mac...*/
              if ((tmp=strchr(line,':'))==NULL) {
                  fprintf (stderr, "%s %s\"%s\":  %s",
			   "_dcerpc_EnsureRegistries:  Invalid record",
			   "(1) in program number registry file",
			   buf, line);
                  return;
                }
              strncpy(name,line,tmp-line);
              name[tmp-line]='\0';
              tmp++;
              start=tmp;
              ch=*tmp;
              while ((ch!=' ') && (ch!='\0')) {
                  tmp++;
                  ch=*tmp;
                }
              if (ch=='\0') {
                  fprintf (stderr, "%s %s \"%s\":  %s",
			   "_dcerpc_EnsureRegistries:  Invalid record",
			   "(2) in program number registry file",
			   buf, line);
                  return;
                }
              strncpy(id,start,tmp-start);
              id[tmp-start]='\0';
              tmp++;
              /* advance past white-space since strtoul is too stupid
		 to do it for us... */
              while (*tmp==' ')
                        tmp++;
              s->sui_pnumber=_ilu_strtoul(tmp,&tmp,16);
              if (tmp==NULL) {
                  fprintf (stderr, "%s %s \"%s\":  %s",
			   "_dcerpc_EnsureRegistries:  Invalid record",
			   "(3) in program number registry file",
			   buf, line);
                  return;
                }
              while (*tmp==' ')
                  tmp++;
              s->sui_version=_ilu_strtoul(tmp,NULL,10);
#else /* MACOS */
	      if (4 != sscanf (line, "%1000[^:]:%1000s %x %u", name, id,
			       &s->sui_pnumber, &s->sui_version))
                  fprintf (stderr, "%s line %u %s \"%s\":  %s",
			   "_dcerpc_EnsureRegistries:  Invalid record at",
			   lineno, "in program number registry file",
			   buf, line);
	      else
#endif	/* MACOS */
	       {
	         if (_ilu_hash_FindInTable (RegistryHashTable, id)
	   	  OR _ilu_hash_FindInTable (ProgramNumberHashTable,
					    (void *) s->sui_pnumber))
	   	{
		  fprintf (stderr,
			   "%s %s %u %s \"%s\":  <%s> (ignoring)\n",
			   "_dcerpc_EnsureRegistries:  Duplicate type id",
			   "or program number at line", lineno,
			   "in program number registry", buf, line);
		  continue;
		}
	      s->sui_type_id = _ilu_Strdup(id);
	      _ilu_hash_AddToTable (RegistryHashTable, s->sui_type_id, s);
	      _ilu_hash_AddToTable (ProgramNumberHashTable,
				    (void *) s->sui_pnumber, s);
	     }
          }
	}
      fclose (f);
    }
}

static dcerpcinfo *_dcerpc_dceRPCInformationForClass (ilu_Class class)
{
  dcerpcinfo *s = NULL;

  _dcerpc_EnsureRegistries();

  s = (dcerpcinfo *) _ilu_hash_FindInTable (RegistryHashTable,
					    class->cl_unique_id);
  if (s != NULL)
    s->sui_class = class;

  DEBUG(DCERPC_DEBUG,
	(stderr, "%s \"%s:%s\", pnumber is 0x%x, version is %d.\n",
	 "_dcerpc_dceRPCInformationForClass:  Class", class->cl_name,
	 class->cl_unique_id, s == NULL ? 0 : s->sui_pnumber,
	 s == NULL ? 0 : s->sui_version));

  return (s);
}

static dcerpcinfo *_dcerpc_ClassFromProgramNumber (ilu_cardinal pnumber,
						   ilu_cardinal version)
{
  dcerpcinfo *s = NULL;

  _dcerpc_EnsureRegistries();

  s = (dcerpcinfo *) _ilu_hash_FindInTable (ProgramNumberHashTable,
					    (void *) pnumber);

  DEBUG(DCERPC_DEBUG,
	(stderr,
	 "%s 0x%x, version is %u(%u), class is \"%s\" (\"%s:%s\").\n",
	 "_dcerpc_ClassFromProgramNumber:  pnumber is", pnumber, version,
	 (s == NULL) ? 0 : s->sui_version,
	 (s == NULL) ? "unknown" : s->sui_type_id,
	 (s == NULL || s->sui_class == NULL) ? "" : s->sui_class->cl_unique_id,
	 (s == NULL || s->sui_class == NULL) ? "" : s->sui_class->cl_name));

  return (s);
}

/*======================================================================*/
/*======================== Basic I/O code ==============================*/
/*======================================================================*/

#define INPUT_ERROR		1
#define OUTPUT_ERROR		2

/*L1, L2, Main unconstrained (this is only for calling from debugger)*/
static ilu_cardinal ilu_dcerpc_SetMaxStringSize (ilu_cardinal size)
{
  ilu_cardinal old_size = dceRPCMaxStringSize;
  if (size > 0)
    dceRPCMaxStringSize = size;
  return (old_size);  
}

/*L2 >= {call's connection's iomu}*/
/*L1, Main unconstrained*/

/* ==================== integer ==================== */

static ilu_boolean _dcerpc_OutputInteger (ilu_Call call, ilu_integer i)
{
  return (packet_put_long (DCERPC(call_connection(call))->dce_out, i));
}

static ilu_boolean _dcerpc_InputInteger (ilu_Call call, ilu_integer *i)
{
  return (packet_get_long (DCERPC(call_connection(call))->dce_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfInteger (ilu_Call call, ilu_integer i)
{
  return (4);
}

/* ==================== cardinal ==================== */

static ilu_boolean _dcerpc_OutputCardinal (ilu_Call call, ilu_cardinal i)
{
  return (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i));
}

static ilu_boolean _dcerpc_InputCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (packet_get_u_long (DCERPC(call_connection(call))->dce_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfCardinal (ilu_Call call, ilu_cardinal i)
{
  return (4);
}

/* ==================== short integer ==================== */

static ilu_boolean _dcerpc_OutputShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (packet_put_short (DCERPC(call_connection(call))->dce_out, i));
}

static ilu_boolean _dcerpc_InputShortInteger (ilu_Call call, ilu_shortinteger *i)
{
  return (packet_get_short (DCERPC(call_connection(call))->dce_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (4);
}

/* ==================== long integer ==================== */

static ilu_boolean _dcerpc_OutputLongInteger (ilu_Call call, ilu_integer *i)
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i[1]));
  return(stat && (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i[0])));
}

static ilu_boolean _dcerpc_InputLongInteger (ilu_Call call, ilu_longinteger (*i))
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_get_long (DCERPC(call_connection(call))->dce_in, &(*i)[1]));
  return (stat && (packet_get_long (DCERPC(call_connection(call))->dce_in, &(*i)[0])));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfLongInteger (ilu_Call call, ilu_integer *i)
{
  return (8);
}

/* ==================== short cardinal ==================== */

static ilu_boolean _dcerpc_OutputShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (packet_put_u_short (DCERPC(call_connection(call))->dce_out, i));
}

static ilu_boolean _dcerpc_InputShortCardinal (ilu_Call call, ilu_shortcardinal *i)
{
  return (packet_get_u_short (DCERPC(call_connection(call))->dce_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== long cardinal ==================== */

static ilu_boolean _dcerpc_OutputLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  ilu_boolean stat;

  stat = (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i[1]));
  return (stat && (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i[0])));
}

static ilu_boolean _dcerpc_InputLongCardinal (ilu_Call call, ilu_longcardinal (*i))
{
  ilu_boolean stat;

  stat = (packet_get_u_long (DCERPC(call_connection(call))->dce_in, &(*i)[1]));
  return (stat && (packet_get_u_long (DCERPC(call_connection(call))->dce_in, &(*i)[0])));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (8);
}

/* ==================== enumeration ==================== */

static ilu_boolean _dcerpc_OutputEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  ilu_cardinal i2 = i;
  return (packet_put_u_long (DCERPC(call_connection(call))->dce_out, i2));
}

static ilu_boolean _dcerpc_InputEnumeration (ilu_Call call, ilu_shortcardinal *i)
{
  ilu_cardinal i2;
  ilu_cardinal stat;

  stat = packet_get_u_long (DCERPC(call_connection(call))->dce_in, &i2);
  *i = i2;
  return(stat);
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== real ==================== */

static ilu_boolean _dcerpc_OutputReal (ilu_Call call, double d)
{
  return (packet_put_double (DCERPC(call_connection(call))->dce_out, d));
}

static ilu_boolean _dcerpc_InputReal (ilu_Call call, double *d)
{
  return (packet_get_double (DCERPC(call_connection(call))->dce_in, d));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfReal (ilu_Call call, double d)
{
  return (8);
}

/* ==================== short real ==================== */

static ilu_boolean _dcerpc_OutputShortReal (ilu_Call call, float f)
{
  float f2;

  f2 = f;
  return (packet_put_float (DCERPC(call_connection(call))->dce_out, f2));
}

static ilu_boolean _dcerpc_InputShortReal (ilu_Call call, float *f)
{
  return (packet_get_float (DCERPC(call_connection(call))->dce_in, f));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfShortReal (ilu_Call call, float d)
{
  return (4);
}

/* ==================== string ==================== */

static ilu_boolean _dcerpc_OutputString (ilu_Call call, ilu_string s, ilu_cardinal len, ilu_cardinal limit)
{
  ilu_cardinal len2 = len;

  return (packet_put_bytes (DCERPC(call_connection(call))->dce_out, (ilu_bytes) s, len2));
}

static ilu_boolean _dcerpc_InputString (ilu_Call call, ilu_string *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  if (!packet_get_bytes (DCERPC(call_connection(call))->dce_in,
	(ilu_byte **) s, len, limit))
    return (FALSE);
  return (TRUE);
}

  /*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfString (ilu_Call call, ilu_string i, ilu_cardinal l, ilu_cardinal limit)
{
  if (l > dceRPCMaxStringSize)
    {
      fprintf (stderr, "_dcerpc_SizeOfString:  Attempt to pass string of length %u, which exceeds either dceRPCMaxStringSize value of %u, or limit on this sequence type of %u.\n",
	       l, dceRPCMaxStringSize, limit);
      return (0);
    }

  return (4 + PADDED_SIZE(l));
}

/* ==================== bytes ==================== */

static ilu_boolean _dcerpc_OutputBytes (ilu_Call call, ilu_bytes s, ilu_cardinal len, ilu_cardinal limit)
{
  return (packet_put_bytes (DCERPC(call_connection(call))->dce_out, s, len));
}

static ilu_boolean _dcerpc_InputBytes (ilu_Call call, ilu_bytes *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  return (packet_get_bytes (DCERPC(call_connection(call))->dce_in, s, len, limit));
}

  /*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfBytes (ilu_Call call, ilu_bytes i, ilu_cardinal l, ilu_cardinal limit)
{
  if (((l > 0) && (l > limit)) || l > dceRPCMaxStringSize)
    {
      fprintf (stderr, "_dcerpc_SizeOfBytes:  Attempt to pass bytestring of length %u, which exceeds either dceRPCMaxStringSize of %u, or limit on this sequence type of %u.\n",
	       l, dceRPCMaxStringSize, limit);
      return (0);
    }
  return (4 + PADDED_SIZE(l));
}

/* ==================== byte ==================== */

static ilu_boolean _dcerpc_OutputByte (ilu_Call call, ilu_byte b)
{
  return (packet_put_u_byte (DCERPC(call_connection(call))->dce_out, b));
}

static ilu_boolean _dcerpc_InputByte (ilu_Call call, ilu_byte *b)
{
  return (packet_get_u_byte (DCERPC(call_connection(call))->dce_in, b));
}

/*ARGSUSED*/
static ilu_cardinal _dcerpc_SizeOfByte (ilu_Call call, ilu_byte i)
{
  return (4);
}

/* ==================== opaque ==================== */

static ilu_boolean _dcerpc_OutputOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (packet_put_opaque (DCERPC(call_connection(call))->dce_out, o, len));
}

static ilu_boolean _dcerpc_InputOpaque (ilu_Call call, ilu_bytes *o, ilu_cardinal len)
{
  return(packet_get_opaque (DCERPC(call_connection(call))->dce_in, o, len));
}

static ilu_cardinal _dcerpc_SizeOfOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (PADDED_SIZE(len));
}

static ilu_boolean _dcerpc_OutputSequence (ilu_Call c, ilu_cardinal sequenceLength, ilu_cardinal limit)
{
  return (_dcerpc_OutputCardinal (c, sequenceLength));
}

static ilu_boolean _dcerpc_OutputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

static ilu_boolean _dcerpc_InputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

static ilu_boolean _dcerpc_InputSequence (ilu_Call c, ilu_cardinal *sequenceLength, ilu_cardinal limit)
{
  ilu_boolean stat;
  ilu_cardinal len;

  stat = _dcerpc_InputCardinal (c, &len);
  *sequenceLength = len;
  return(stat);
}

static ilu_boolean _dcerpc_EndSequence (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _dcerpc_SizeOfSequence (ilu_Call c, ilu_cardinal length, ilu_cardinal limit)
{
  return (_dcerpc_SizeOfCardinal(c, length));
}

static ilu_boolean _dcerpc_OutputUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_dcerpc_OutputCardinal (c, typeIndex));
}

static ilu_boolean _dcerpc_InputUnion (ilu_Call c, ilu_shortcardinal *typeIndex)
{
  ilu_boolean stat;
  ilu_cardinal i;

  stat = _dcerpc_InputCardinal (c, &i);
  *typeIndex = i;
  return(stat);
}

static ilu_boolean _dcerpc_EndUnion (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _dcerpc_SizeOfUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_dcerpc_SizeOfCardinal(c, typeIndex));
}

static ilu_boolean _dcerpc_OutputArray (ilu_Call c, ilu_cardinal len)
{
  return(TRUE);
}

static ilu_boolean _dcerpc_InputArray (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _dcerpc_EndArray (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _dcerpc_SizeOfArray (ilu_Call c, ilu_cardinal len)
{
  return(0);
}

static ilu_boolean _dcerpc_OutputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _dcerpc_InputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _dcerpc_EndRecord (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _dcerpc_SizeOfRecord (ilu_Call c)
{
  return(0);
}

/*======================================================================
**======================================================================
**====================  Non-I/O code ===================================
**======================================================================
**====================================================================*/

/*L2, Main unconstrained*/
/*L1_sup < prmu*/
static ilu_string _dcerpc_FormProtocolHandle (ilu_Object obj)
{
  char buf[100];
  dcerpcinfo *s = NULL;
  ilu_Class class = object_class(obj);

  _ilu_AcquireMutex(ilu_prmu);
  s = _dcerpc_DcerpcInformationForClass (class);
  if (s == NULL)
    {
      _ilu_ReleaseMutex(ilu_prmu);
      fprintf (stderr, "%s program#/version for class %s.\n",
	       "_dcerpc_FormProtocolHandle:  Can't figure", class->cl_name);
      return (NULL);
    }
  else
    {
      sprintf (buf, "dcerpc_2_0x%x_%d", s->sui_pnumber, s->sui_version);
      _ilu_ReleaseMutex(ilu_prmu);
      return (_ilu_Strdup(buf));
    }
}
     
/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_bytes _dcerpc_ReadPacket (ilu_Transport bs,
			      ilu_cardinal *packetLength,
			      ilu_PacketType *type, ilu_cardinal *sn)
{
  ilu_bytes packet;
  ilu_cardinal serialNumber, packetType, size;

  if (!transport_read_message(bs, &packet, &size)) {
    DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	  (stderr, "%s:  error on transport_read_message\n",
	   "_dcerpc_ReadPacket"));
    return (NULL);
  };

  serialNumber =  (packet[0] << 24) | (packet[1] << 16)
		| (packet[2] <<  8) |  packet[3];
  packetType =    (packet[4] << 24) | (packet[5] << 16)
		| (packet[6] <<  8) |  packet[7];

  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr, "%s %d, SN %d, type %d (%s).\n",
	 "_dcerpc_ReadPacket:  reading packet of length", size,
	 serialNumber, packetType,
	 ((packetType == ilu_PacketType_Request) ? "request"
	  : (packetType == ilu_PacketType_Reply) ? "reply"
	  : "unknown type")));

  *packetLength = size;
  *type = packetType;
  *sn = serialNumber;
  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr, "_dcerpc_ReadPacket:  returning packet SN %d\n",
	 serialNumber));
  return (packet);  
}

static ilu_boolean _dcerpc_InterpretPipeRequest (ilu_Call call, ilu_bytes packet,
					  ilu_cardinal len)
{
  struct dceRPC *c = DCERPC(call_connection(call));
  ilu_cardinal rpcVersion, credSize, verifierSize, programNumber;
  ilu_cardinal programVersion, methodID;
  ilu_cardinal verifierType, credType, serialNumber, packetType;
  ilu_bytes credBuf, verifierBuf;

  c->dce_in = _ndr_CreatePacket (packet, len);

  _dcerpc_InputCardinal (call, &serialNumber);
  _dcerpc_InputCardinal (call, &packetType);
  _dcerpc_InputCardinal (call, &rpcVersion);
  _dcerpc_InputCardinal (call, &programNumber);
  _dcerpc_InputCardinal (call, &programVersion);
  _dcerpc_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr,
	 "%s:  (call SN %d) prognum 0x%x version %u, method ID = %u\n",
	 "_dcerpc_InterpretPipeRequest", call->ca_SN,
	 programNumber, programVersion, methodID));

  _dcerpc_InputCardinal (call, &credType);
  _dcerpc_InputBytes (call, &credBuf, &credSize, 0);
  _dcerpc_InputCardinal (call, &verifierType);
  _dcerpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG,
	(stderr, "_dcerpc_InterpretPipeRequest:  (call SN %d) credBuf is %x, credSize is %d, verifierBuf is %x, verifierSize is %d\n",
	 call->ca_SN, (unsigned long) credBuf, credSize,
	 (unsigned long) verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);
  return (TRUE);
}

static ilu_boolean _dcerpc_InterpretRequest (ilu_Call call, ilu_bytes packet,
				      ilu_cardinal len)
{
  struct dceRPC *c = DCERPC(call_connection(call));
  dcerpcinfo *s;
  ilu_cardinal rpcVersion, serialNumber, packetType;
  ilu_cardinal credSize, credType, verifierSize, verifierType;
  ilu_cardinal programNumber, programVersion, methodID;
  ilu_bytes credBuf, verifierBuf;

  c->dce_in = _ndr_CreatePacket (packet, len);

  _dcerpc_InputCardinal (call, &serialNumber);
  _dcerpc_InputCardinal (call, &packetType);
  _dcerpc_InputCardinal (call, &rpcVersion);
  _dcerpc_InputCardinal (call, &programNumber);
  _dcerpc_InputCardinal (call, &programVersion);
  _dcerpc_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr,
	 "%s  (call SN %d) prognum 0x%x (version %u), method ID = %u\n",
	 "_dcerpc_InterpretRequest:", call->ca_SN, programNumber,
	 programVersion, methodID));
  _ilu_Assert(serialNumber == call->ca_SN,
	      "dcerpc_InterpretRequest: serial number snafu");
  _dcerpc_InputCardinal (call, &credType);
  _dcerpc_InputBytes (call, &credBuf, &credSize, 0);
  _dcerpc_InputCardinal (call, &verifierType);
  _dcerpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG,
	(stderr,
	 "%s %d%s %x, credSize is %d, verfBuf is %x, verfSize is %d\n",
	 "_dcerpc_InterpretRequest:  (call SN", call->ca_SN,
	 ") credBuf is", (unsigned long) credBuf, credSize,
	 (unsigned long) verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);

  _ilu_AcquireMutex(ilu_prmu);
  s = _dcerpc_ClassFromProgramNumber(programNumber, programVersion);
  call_intro_type(call)
    = (s == NULL) ? NULL
		  : (s->sui_class == NULL) ? ilu_FindClassFromID(s->sui_type_id)
				       : s->sui_class;
  _ilu_ReleaseMutex(ilu_prmu);
  if (s == NULL OR call_intro_type(call) == NULL)
    {
      DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	    (stderr, "%s  (call %d) %s 0x%x, version %u\n",
	     "_dcerpc_InterpretRequest:", call->ca_SN,
	     "Can't find ilu_Class with pn",
	     programNumber, programVersion));
      return (FALSE);
    }

  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr,
	 "_dcerpc_InterpretRequest:  (call SN %d) intro_type is %s:%s\n",
	 call->ca_SN, call->ca_intro_type->cl_name,
	 call->ca_intro_type->cl_unique_id));

  call->ca_method = ilu_FindMethodByID (call_intro_type(call), methodID);
  
  if (call->ca_method == NULL)
    {
      DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	    (stderr,
	     "%s  (call %d) %s \"%s\" (pn %x) with methodID = %u.\n",
	     "_dcerpc_InterpretRequest:", call->ca_SN,
	     "Can't find method on class", call_intro_type(call)->cl_name,
	     programNumber, methodID));
      return (FALSE);
    }
  else
    {
      DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	    (stderr, "%s %d is 0x%x (%s).\n",
	     "_dcerpc_InterpretRequest:  record for method", 
	     methodID, (unsigned long) call->ca_method, call->ca_method->me_name));
    }
  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr, "_dcerpc_InterpretRequest:  returning TRUE\n"));
  return (TRUE);
}

/*L1, Main unconstrained*/
/*L2 >= {conn's iomu}*/

static ilu_ProtocolException _dcerpc_InterpretReply (ilu_Call call,
					      ilu_bytes packet,
					      ilu_cardinal len,
					      ilu_cardinal *estatus)
{
  struct dceRPC *c = DCERPC(call->ca_connection);
  ilu_cardinal verifierSize, replyStatus, serialNumber, packetType;
  ilu_bytes verifierBuf;
  ilu_cardinal authenticationType;
  static ilu_ProtocolException replyStatusExceptions[] = {
    ilu_ProtocolException_Success,
    ilu_ProtocolException_NoSuchClassAtServer,
    ilu_ProtocolException_ClassVersionMismatch,
    ilu_ProtocolException_NoSuchMethodOnClass,
    ilu_ProtocolException_GarbageArguments
    };
  static ilu_string acceptErrors[] = {
    "Success",
    "program unavailable",
    "program version mismatch",
    "procedure unavailable",
    "garbage arguments",
  };

  c->dce_in = _ndr_CreatePacket (packet, len);

  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr, "_dcerpc_InterpretReply:  SN %d, %d bytes\n",
	 call->ca_SN, len));

  _dcerpc_InputCardinal (call, &serialNumber);
  _dcerpc_InputCardinal (call, &packetType);
  _dcerpc_InputCardinal (call, &replyStatus);

  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	(stderr, "_dcerpc_InterpretReply:  replyStatus is %d\n",
	 replyStatus));

  if (replyStatus == 0)		/* MSG_ACCEPTED */
    {
      _dcerpc_InputCardinal (call, &authenticationType);
      _dcerpc_InputBytes (call, &verifierBuf, &verifierSize, 0);
      FREETOKEN(verifierBuf);
      _dcerpc_InputCardinal (call, &replyStatus);

      DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	    (stderr,
	     "_dcerpc_InterpretReply:  *real* replyStatus is %d\n",
	     replyStatus));

      if (replyStatus > 0)		/* *really* accepted */
	{
	  DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
		(stderr,
		 "DCE RPC call %d signals protocol error %d (%s).\n",
		 call->ca_SN, replyStatus,
		 (replyStatus < 5) ? acceptErrors[replyStatus]
				   : "unknown error"));
	}
      else if (call->ca_method->me_exceptionCount > 0)
	_dcerpc_InputCardinal (call, estatus);
      else
	*estatus = 0;

      if (replyStatus < 5)
	return (replyStatusExceptions[replyStatus]);
      else
	return(ilu_ProtocolException_Unknown);
    }
  else if (replyStatus == 1)	/* MSG_REJECTED */
    {
      DEBUG((INCOMING_DEBUG | DCERPC_DEBUG),
	    (stderr, "DCE RPC call %d signals \"Message Rejected\".\n",
	     call->ca_SN));
      return(ilu_ProtocolException_RequestRejected);
    }
  return(ilu_ProtocolException_Unknown);
}

/*L1_sup < prmu*/

static ilu_boolean GetDcerpcProgramNumberAndVersion (ilu_Class pclass,
			ilu_cardinal *pnumber, ilu_cardinal *version)
{
  dcerpcinfo *s = NULL;
  _ilu_AcquireMutex(ilu_prmu);
  s = _dcerpc_DcerpcInformationForClass (pclass);
	/* probably consult external DB */
  if (s != NULL) {
      *pnumber = s->sui_pnumber;
      *version = s->sui_version;
    }
  _ilu_ReleaseMutex(ilu_prmu);
  return (s != NULL);
}

/*L1 >= {prmu}*/
#ifdef _BSD_SOURCE
static int nameset = 0;
static char hostname[255];
static int hostnamelen, hostnamelenr, credlen;
static int gid, uid, ngids;
static gid_t gids[100];
#endif /* _BSD_SOURCE */
static int credlenr;

/*L1_sup < prmu*/

static ilu_boolean _dcerpc_BeginRequest (ilu_Call call, ilu_Class pclass,
					 ilu_Method method,
					 ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  ilu_cardinal pnumber=0, version=0;
  struct dceRPC *c = DCERPC(call->ca_connection);

  call->ca_method = method;

#if 0
  if (call_object(call)==NULL) { /* no call object, must be pipe req */
#else
  if (0) {
#endif
      DEBUG(CALL_DEBUG,
	    (stderr,
	     "%s 0x%x (sn %d), argSize %d, class %s (%s), meth %s (%d)\n",
	     "_dcerpc_BeginRequest:  pipe call", (unsigned long) call,
	     call_serial_number(call), argSize, class_name(pclass),
	     class_unique_id(pclass), method_name(method),
	     method_id(method)));
    }
  else {
      DEBUG(CALL_DEBUG,
	    (stderr,
	     "%s 0x%x (sn %d), argSize %d, class %s (%s), meth %s (%d)\n",
	     "_dcerpc_BeginRequest:  call", (unsigned long) call,
	     call_serial_number(call), argSize, class_name(pclass),
	     class_unique_id(pclass), method_name(method),
	     method_id(method)));

      if (!GetDcerpcProgramNumberAndVersion (pclass, &pnumber, &version))
        {
          DEBUG(CALL_DEBUG,
		(stderr, "%s %s of class \"%s\" on call 0x%x.\n",
		 "_dcerpc_BeginRequest:  Can't determine",
		 "program-number/version",
		 pclass->cl_name,  (unsigned long) call));
          return (FALSE);
        }
    }
  DEBUG(CALL_DEBUG,
	(stderr,
	 "%s 0x%x (sn %d), argSize %d, pnum/vers 0x%x/%d, method id %d\n",
	 "_dcerpc_BeginRequest:  call", (unsigned long) call, call_serial_number(call),
	 argSize, pnumber, version, method_id(method)));

#ifdef _BSD_SOURCE
  if (!nameset) {
      _ilu_AcquireMutex(ilu_prmu);
      if (!nameset) {
          nameset = 1;
          _ilu_Assert(gethostname (hostname, sizeof(hostname)) == 0,
                      "gethostname failed");
          hostnamelen = strlen(hostname);
          hostnamelenr = 4 * ((hostnamelen+3)/4);
          gid = getegid();
          uid = geteuid();
          ngids = getgroups(sizeof(gids)/sizeof(gid), gids);
          _ilu_Assert(ngids >= 0, "getgroups() failed");
          credlen =   4/*stamp*/ + 4+hostnamelenr/*machinename*/
                    + 4/*uid*/ + 4/*gid*/ + 4+4*ngids/*gids*/;
          credlenr = 4 * ((credlen+3)/4);
        }
      _ilu_ReleaseMutex(ilu_prmu);
    }
#else
  credlenr = 4 * 2;
#endif
  
  packetSize = argSize
    + (4*10)		/* for fields of header */
    + credlenr		/* for UNIX credentials */;

  c->dce_out = _ndr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _dcerpc_OutputCardinal (call, call->ca_SN);
  _dcerpc_OutputCardinal (call, 0);	/* message type == CALL */
/*  _dcerpc_OutputCardinal (call, 2); */	/* Sun RPC version 2 */
  _dcerpc_OutputCardinal (call, pnumber);
  _dcerpc_OutputCardinal (call, version);
  _dcerpc_OutputCardinal (call, method->me_id);

#ifdef _BSD_SOURCE
  _dcerpc_OutputCardinal (call, 1);	/* UNIX credentials follow */
  _dcerpc_OutputCardinal (call, credlen); /* credentials length */
  _dcerpc_OutputCardinal (call, 1);	/* stamp */
  _dcerpc_OutputString (call, hostname, hostnamelen, 255);
  _dcerpc_OutputCardinal (call, uid);
  _dcerpc_OutputCardinal (call, gid);
  _dcerpc_OutputCardinal (call, ngids);
  {
    int i;
    for (i=0; i<ngids; i++)
      _dcerpc_OutputCardinal (call, gids[i]);
  }
#else
  _dcerpc_OutputCardinal (call, 0);	/* credentials:  AUTH_NULL */
  _dcerpc_OutputCardinal (call, 0);	/* 		 0 bytes */
#endif

  _dcerpc_OutputCardinal (call, 0);	/* verifier:  AUTH_NULL */
  _dcerpc_OutputCardinal (call, 0);	/*            0 bytes */
  
  DEBUG(CALL_DEBUG,
	(stderr, "_dcerpc_BeginRequest:  request %d begun (size %d).\n",
	 call->ca_SN, packetSize));
  return (TRUE);
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static void _dcerpc_FlushBuffer (ilu_Call call)
{
  transport_flush_output(call_transport(call));
}

static void _dcerpc_DropRequest (ilu_Call call)
{
  struct dceRPC *s = DCERPC(call_connection(call));
  if (s->dce_out != NULL)
      packet_destroy(s->dce_out);
  s->dce_out = NULL;
}

static ilu_boolean _dcerpc_SendPacket (ilu_Call call, ilu_boolean isReply)
{
  ilu_Transport bs = call_transport(call);
  struct dceRPC *s = DCERPC(call_connection(call));
  _ilu_Assert(packet_buffer(s->dce_out) != NULL, "_dcerpc_SendPacket");
  transport_send_message (bs, packet_buffer(s->dce_out), packet_size(s->dce_out));
  if (!bs->tr_class->tc_timesout)
    packet_buffer(s->dce_out) = NULL;
  if (!(method_asynchronous(call_method(call))))
    transport_flush_output(bs);
  if (isReply) {
      if (bs->tr_class->tc_timesout) {
          _ilu_CacheCall(call, packet_buffer(s->dce_out),
                         packet_size(s->dce_out));
          packet_buffer(s->dce_out) = NULL;
        }
      _dcerpc_DropRequest(call);
    }
  return (TRUE);
}

static ilu_boolean _dcerpc_FinishRequest (ilu_Call call)
{
  return (_dcerpc_SendPacket(call, FALSE));
}

static ilu_boolean _dcerpc_FinishReply (ilu_Call call)
{
  return (_dcerpc_SendPacket (call, TRUE));
}

static ilu_boolean _dcerpc_FinishException (ilu_Call call)
{
  return (_dcerpc_SendPacket(call, TRUE));
}

/*L1, Main unconstrained*/
/*L2 >= {call's conn's iomu}*/

static ilu_boolean _dcerpc_BeginReply (ilu_Call call,
				       ilu_boolean exceptions,
				       ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct dceRPC *c = DCERPC(call->ca_connection);

  DEBUG(PACKET_DEBUG,
	(stderr, "%s %d, argSize %d, exceptions %s, fd %d.\n",
	 "_dcerpc_BeginReply:  SN", call->ca_SN, argSize,
	 exceptions ? "TRUE" : "FALSE",
	 transport_file_descriptor(call_transport(call))));

  packetSize = argSize
    + (4*6)			/* for the basic header fields */
    + (exceptions ? 4 : 0);	/* possible extra word for excn code */

  c->dce_out = _ndr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _dcerpc_OutputCardinal (call, call->ca_SN);
  _dcerpc_OutputCardinal (call, 1);		/* message type ==REPLY */
  _dcerpc_OutputCardinal (call, 0);		/* message accepted */
  _dcerpc_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _dcerpc_OutputCardinal (call, 0);		/*            0 bytes */
  _dcerpc_OutputCardinal (call, 0);		/* successful execution */
  if (exceptions)
    _dcerpc_OutputCardinal (call, 0);		/* ret code is Success */

  DEBUG(PACKET_DEBUG,
	(stderr, "_dcerpc_BeginReply:  started reply %d (size %d).\n",
	 call->ca_SN, packetSize));
  return (TRUE);
}

static ilu_boolean _dcerpc_BeginException (ilu_Call call,
					   ilu_cardinal evalue,
					   ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct dceRPC *c = DCERPC(call->ca_connection);

  /* if "evalue" == 0, then argSize contains a protocol exception
     detail code. */

  if (evalue == 0)	/* signal protocol error */
    packetSize = (4*6);
  else
    packetSize = (4*7) + argSize;

  c->dce_out = _ndr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _dcerpc_OutputCardinal (call, call->ca_SN);
  _dcerpc_OutputCardinal (call, 1);		/* message type ==REPLY */
  _dcerpc_OutputCardinal (call, 0);		/* message accepted */
  _dcerpc_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _dcerpc_OutputCardinal (call, 0);		/*            0 bytes */
  _dcerpc_OutputCardinal (call, (evalue == 0) ? argSize : 0);
						/* successful execution */
  if (evalue > 0)
    _dcerpc_OutputCardinal (call, evalue);	/* exception value */

  DEBUG(PACKET_DEBUG,
	(stderr, "%s %d:  SN %d, size %d, evalue %d.\n",
	 "_dcerpc_BeginException:  exception started to peer",
	 transport_file_descriptor(call_transport(call)),
	 call_serial_number(call), packetSize, evalue));
  return (TRUE);
}

/*L2, Main unconstrained*/
/*L1 >= {prmu}*/

static ilu_Protocol _dcerpc_NewDcerpc (void)
{
  ilu_Protocol new = (ilu_Protocol)
		     malloc(sizeof(struct _ilu_Protocol_s));

  new->pr_type = ilu_ProtocolType_DCE;
  new->pr_concurrent_requests = ilu_FALSE;

  new->pr_begin_request = _dcerpc_BeginRequest;
  new->pr_finish_request = _dcerpc_FinishRequest;
  new->pr_resend_request = _dcerpc_FinishRequest;
  new->pr_drop_request = _dcerpc_DropRequest;
  new->pr_begin_reply = _dcerpc_BeginReply;
  new->pr_finish_reply = _dcerpc_FinishReply;
  new->pr_begin_exception = _dcerpc_BeginException;
  new->pr_finish_exception = _dcerpc_FinishException;

  new->pr_output_integer = _dcerpc_OutputInteger;
  new->pr_input_integer = _dcerpc_InputInteger;
  new->pr_size_of_integer = _dcerpc_SizeOfInteger;

  new->pr_output_shortinteger = _dcerpc_OutputShortInteger;
  new->pr_input_shortinteger = _dcerpc_InputShortInteger;
  new->pr_size_of_shortinteger = _dcerpc_SizeOfShortInteger;

  new->pr_output_longinteger = _dcerpc_OutputLongInteger;
  new->pr_input_longinteger = _dcerpc_InputLongInteger;
  new->pr_size_of_longinteger = _dcerpc_SizeOfLongInteger;

  new->pr_output_cardinal = _dcerpc_OutputCardinal;
  new->pr_input_cardinal = _dcerpc_InputCardinal;
  new->pr_size_of_cardinal = _dcerpc_SizeOfCardinal;

  new->pr_output_shortcardinal = _dcerpc_OutputShortCardinal;
  new->pr_input_shortcardinal = _dcerpc_InputShortCardinal;
  new->pr_size_of_shortcardinal = _dcerpc_SizeOfShortCardinal;

  new->pr_output_longcardinal = _dcerpc_OutputLongCardinal;
  new->pr_input_longcardinal = _dcerpc_InputLongCardinal;
  new->pr_size_of_longcardinal = _dcerpc_SizeOfLongCardinal;

  new->pr_output_real = _dcerpc_OutputReal;
  new->pr_input_real = _dcerpc_InputReal;
  new->pr_size_of_real = _dcerpc_SizeOfReal;

  new->pr_output_shortreal = _dcerpc_OutputShortReal;
  new->pr_input_shortreal = _dcerpc_InputShortReal;
  new->pr_size_of_shortreal = _dcerpc_SizeOfShortReal;

  new->pr_output_longreal = _dcerpc_OutputReal;
  new->pr_input_longreal = _dcerpc_InputReal;
  new->pr_size_of_longreal = _dcerpc_SizeOfReal;

  new->pr_output_optional =
	(ilu_boolean (*)(ilu_Call,ilu_boolean)) _dcerpc_OutputCardinal;
  new->pr_input_optional =
	(ilu_boolean (*)(ilu_Call,ilu_boolean *)) _dcerpc_InputCardinal;
  new->pr_size_of_optional =
	(ilu_cardinal (*)(ilu_Call,ilu_boolean)) _dcerpc_SizeOfCardinal;

  new->pr_output_enum_code = _dcerpc_OutputEnumeration;
  new->pr_input_enum_code = _dcerpc_InputEnumeration;
  new->pr_size_of_enum_code = _dcerpc_SizeOfEnumeration;

  new->pr_output_byte = _dcerpc_OutputByte;
  new->pr_input_byte = _dcerpc_InputByte;
  new->pr_size_of_byte = _dcerpc_SizeOfByte;

  new->pr_output_character = _dcerpc_OutputShortCardinal;
  new->pr_input_character = _dcerpc_InputShortCardinal;
  new->pr_size_of_character = _dcerpc_SizeOfShortCardinal;

  new->pr_output_boolean =
	(ilu_boolean (*)(ilu_Call,ilu_boolean)) _dcerpc_OutputCardinal;
  new->pr_input_boolean =
	(ilu_boolean (*)(ilu_Call,ilu_boolean *)) _dcerpc_InputCardinal;
  new->pr_size_of_boolean =
	(ilu_cardinal (*)(ilu_Call,ilu_boolean)) _dcerpc_SizeOfCardinal;

  new->pr_output_shortchar = _dcerpc_OutputByte;
  new->pr_input_shortchar = _dcerpc_InputByte;
  new->pr_size_of_shortchar = _dcerpc_SizeOfByte;

  new->pr_output_string = _dcerpc_OutputString;
  new->pr_input_string = _dcerpc_InputString;
  new->pr_size_of_string = _dcerpc_SizeOfString;

  new->pr_output_bytes = _dcerpc_OutputBytes;
  new->pr_input_bytes = _dcerpc_InputBytes;
  new->pr_size_of_bytes = _dcerpc_SizeOfBytes;

  new->pr_output_opaque = _dcerpc_OutputOpaque;
  new->pr_input_opaque = _dcerpc_InputOpaque;
  new->pr_size_of_opaque = _dcerpc_SizeOfOpaque;

  new->pr_output_stringvec =
	(ilu_boolean (*)(ilu_Call,ilu_string,ilu_cardinal))
	_dcerpc_OutputOpaque;
  new->pr_input_stringvec =
	(ilu_boolean (*)(ilu_Call,ilu_string *,ilu_cardinal))
	_dcerpc_InputOpaque;
  new->pr_size_of_stringvec =
	(ilu_cardinal (*)(ilu_Call,ilu_string,ilu_cardinal))
	_dcerpc_SizeOfOpaque;

  new->pr_output_pipe = NULL;
  new->pr_input_pipe = NULL;
  new->pr_size_of_pipe = NULL;

  new->pr_output_sequence = _dcerpc_OutputSequence;
  new->pr_output_sequence_mark = _dcerpc_OutputSequenceMark;
  new->pr_input_sequence = _dcerpc_InputSequence;
  new->pr_input_sequence_mark = _dcerpc_InputSequenceMark;
  new->pr_end_sequence = _dcerpc_EndSequence;
  new->pr_size_of_sequence = _dcerpc_SizeOfSequence;

  new->pr_output_record = _dcerpc_OutputRecord;
  new->pr_input_record = _dcerpc_InputRecord;
  new->pr_end_record = _dcerpc_EndRecord;
  new->pr_size_of_record = _dcerpc_SizeOfRecord;

  new->pr_output_array = _dcerpc_OutputArray;
  new->pr_input_array = _dcerpc_InputArray;
  new->pr_end_array = _dcerpc_EndArray;
  new->pr_size_of_array = _dcerpc_SizeOfArray;

  new->pr_output_union = _dcerpc_OutputUnion;
  new->pr_input_union = _dcerpc_InputUnion;
  new->pr_end_union = _dcerpc_EndUnion;
  new->pr_size_of_union = _dcerpc_SizeOfUnion;

  new->pr_form_handle = _dcerpc_FormProtocolHandle;

  new->pr_read_packet = _dcerpc_ReadPacket;
  new->pr_interpret_pipe_request = _dcerpc_InterpretPipeRequest;
  new->pr_interpret_request = _dcerpc_InterpretRequest;
  new->pr_interpret_reply = _dcerpc_InterpretReply;

  new->pr_create_data_block = _dcerpc_CreateDataBlock;
  new->pr_free_data_block = (void (*)(void *)) _dcerpc_FreeDataBlock;
  new->pr_flush = _dcerpc_FlushBuffer;

  return (new);
}

/*L1_sup < prmu*/

ilu_Protocol _ilu_dcerpc_Protocol(void)
{
  /*L1 >= {prmu}*/
  static ilu_Protocol StandardDCERPC = NULL;
  _ilu_AcquireMutex(ilu_prmu);
  if (StandardDCERPC == NULL)
    StandardDCERPC = _dcerpc_NewDCERPC();
  _ilu_ReleaseMutex(ilu_prmu);
  return (StandardDCERPC);
}
