
static char rcsid[] = "$Header: keys.c,v 1.7 94/07/20 18:34:15 mkant Exp $";
/************************************************************************
 *
 * File: keys.c
 *
 * Mark Kantrowitz <mkant@cs.cmu.edu> or <AI.Repository@cs.cmu.edu>
 * Artificial Intelligence Repository
 * School of Computer Science 
 * Carnegie Mellon University
 * 5000 Forbes Avenue
 * Pittsburgh, PA 15213-3891
 * Fax: 412-681-5739
 *
 * Created:       Mon Jul 18 12:50:11 1994 by Mark Kantrowitz
 * Last Modified: Wed Jul 20 18:33:58 1994 by Mark Kantrowitz
 * RCS info:
 *       Source file: $Source: /usr0/mkant/faq-todo/keys/RCS/keys.c,v $
 *    Current Locker: $Locker:  $
 *             State: $State: Exp $
 *   Accumulated log: $Log:	keys.c,v $
 * Revision 1.7  94/07/20  18:34:15  mkant
 * Fixed problem with getenv returning 0.
 * 
 * Revision 1.6  94/07/20  18:12:38  mkant
 * Added CPP commands for vax bug workaround.
 * 
 * Revision 1.5  94/07/20  17:55:51  mkant
 * Fixed problem where if only one match were found, and it was the last
 * one, it was skipped.
 * 
 * Added defaults in case no environment variables set.
 * 
 * Revision 1.4  94/07/19  22:38:00  mkant
 * Fixed -k and -h
 * 
 * Revision 1.3  94/07/19  21:37:10  mkant
 * Changed copyright formatting.
 * 
 * Revision 1.2  94/07/19  21:02:11  mkant
 * Bug fixes, Mosaic support, more processing, output redirection.
 * Ready for first release.
 * 
 * Revision 1.1  94/07/19  04:59:40  mkant
 * Minor changes.
 * 
 * Revision 1.0  94/07/18  13:33:09  mkant
 * Initial revision
 * 
 * Performs a keyword search of the AI Repository or AI CD-ROM's keyword
 * index and package list files.
 *
 * Copyright (c) 1994 by Mark Kantrowitz. All rights reserved.
 * Use, copying, and distribution for non-commercial and personal use
 * purposes permitted. Publication on CD-ROM is prohibited without the 
 * express prior written approval of the author. 
 *
 * Please see the file 'copying.txt' for details.
 *
 * Note: This program uses Henry Spencer's freely distributable regular
 * expression matching code (regexp.c, regexp.h, regmagic.h).
 *
 ************************************************************************/

/*
 * To Do:
 *    +  More processing is primitive. Any way to use the real 'more'?
 *    +  Hard code a test to see if there are an unreasonable number
 *       of matches?
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include "regexp.h"

#define TRUE 1
#define FALSE 0

#define ERROR(s)        return (perror(s), -1)

/*
 * Newline characters:
 *    UNIX uses a line feed (LF)
 *    Macintosh OS uses a carriage return (CR)
 *    MS-DOS uses both (CR/LF)
 * We've standardized on LF (Unix) in the keyword index and package list
 * files in both the AI Repository and CD ROM. The other #defines are
 * here for completeness.
 */

#define NEWLINE '\n'                    /* newline, may change?           */
#define LFNL '\12'                      /* ASCII line feed ^J             */
#define CRNL '\15'			/* ASCII carriage return ^M       */

/* 
 * We have some hard-coded limits built into this program. 
 *
 *    MAXLINE    = The longest input line this program can handle.
 *                 Since the longest line in ki.txt for AI CD-ROM 1-1
 *                 was 711 characters long, a value of 1024 for MAXLINE
 *                 should be enough.
 *    MAXMATCHES = Maximum number of match tuples (keyword, ki_index, ki_key)
 *                 allowed. 10000 seemed like a reasonable number.
 *    MAXKEYS    = Maximum number of input keywords.
 *    KEYLENGTH  = Longest possible key. 100 is a random (too) big number.
 *
 * Blame C for promoting this kind of programming style -- don't blame me.
 */
#define MAXLINE 1024
#define MAXMATCHES 10000
#define MAXKEYS 60
#define KEYLENGTH 100

/* For more processing */
#define SCREENLINES 23

/* Program default parameters. */
int more = TRUE;              		/* More processing on by default. */
int mosaic = FALSE;                     /* Mosaic output off by default.  */
int screenlines = SCREENLINES;          /* Number of lines per screen.    */
int page_at_preferred = FALSE; 		/* Stop to page only at pref pos  */

/* AI Repository Root Directory */
char *airdir = "/afs/cs.cmu.edu/project/ai-repository/ai/"; 

/* Program version and copyright information. */
char *version = "1.0 (19-JUL-94)";
char *copyright="Copyright (c) 1994 by Mark Kantrowitz. All rights reserved.";

/* Global variables for input options processing. */
extern int getopt();
extern int optind;
extern char *optarg;
char prog_opts[] = "bc:d:hkml:o:r:uv:"; /* getopt() flags                 */
char prog_desc[] = "Does a keyword search of the CMU AI Repository";
char *prog;              		/* Saves program initiation name  */

/* PTF environment variables. These are set by i.csh, i.sh, or i.ksh when
 * used with the CD-ROMs.
 * Example values:
 *    ptf_base  /AI_1_1_A/
 *    ptf_case  lower
 *    ptf_doc   0.doc
 */
char ptf_base[128];			/* Repository/CD base directory   */
char ptf_case[10];			/* Filenames upper or lower case  */
char ptf_doc[32];			/* Name of doc files              */
char ptf_ver[16];			/* Version number for ISO-9660    */
char ptf_ki[128];	                /* Keyword Index file name        */
char ptf_pl[128];	                /* Package List file name         */

/* Match tuple structure and matches array. */
struct match {
  int keynum;
  int ki_index;
  char ki_key[100];
};

struct match matches[MAXMATCHES];
int num_matches = 0;           		/* Number of items in match array */

/* Package List information. */
int pl_entry_counter = 0;               /* Current PL entry number        */

/* Output file and flag. */
char output_file[128];
int output_file_flag = FALSE;
FILE *out_fp;

/************************************
 * MAIN Entry Point to the Program **
 ************************************/

main(argc, argv)
     int argc;
     char **argv;
{
  int i;
  FILE *fopen();
  char *temp;

  /*
   * PTF environment variables. First we give some defaults, in case
   * the environment variables are not present. The environement variables 
   * and these defaults can be overridden by command line switches.
   */

  /* The first default is useful mainly at CMU and other AFS sites. */
  strcpy(ptf_base,"/afs/cs.cmu.edu/project/ai-repository/ai");
  strcpy(ptf_case,"lower");
  strcpy(ptf_doc,"0.doc");

  /* Now copy the environment variable values. */

  temp = getenv("PTF_BASE");
  if ((temp != 0) && (strlen(temp) != 0))
    strcpy(ptf_base,temp);
 
  temp = getenv("PTF_CASE");
  if ((temp != 0) && (strlen(temp) != 0))
    strcpy(ptf_case,temp);
 
  temp = getenv("PTF_DOC");
  if ((temp != 0) && (strlen(temp) != 0))
    strcpy(ptf_doc, temp); 

  /* Can be blank, so always copy environment variable. */
  temp = getenv("PTF_VER");
  if ((temp != 0))
    strcpy(ptf_ver, getenv("PTF_VER")); 

  /* Save program initiation name. */
  prog = argv[0];

  /* Process the arguments to the program. */
  while ((i = getopt(argc, argv, prog_opts)) != EOF) switch (i) {
  case 'b':
    /* Turn off more processing for batch mode */
    more = FALSE;
    break;
  case 'c':
    sprintf(ptf_case, "%s", optarg);
    break;
  case 'd':
    sprintf(ptf_doc, "%s", optarg);
    break;
  case 'h':
    /* Print out help information. */
    help();
    break;
  case 'k':
    /* Print out help information. */
    regexphelp();
    break;
  case 'l':
    /* Set number of lines to the screen. */
    sscanf(optarg, "%d", &screenlines);
    break;
  case 'm':
    mosaic = TRUE; more = FALSE;
    break;
  case 'o':
    output_file_flag = TRUE; more = FALSE;
    sprintf(output_file, "%s", optarg);
    break;
  case 'r':
    sprintf(ptf_base, "%s", optarg);
    break;
  case 'u':
    usage();
    break;
  case 'v':
    sprintf(ptf_ver, "%s", optarg);
    break;
  default:
    /* Print out help information. */
    usage();
    break;
  }
  /* update arg pointers */
  argc -= optind;               
  argv += optind;

  /* Redirect output to a file, leaving error messages on stderr. */
  if (output_file_flag == TRUE) {
    /* 
     * Yes, I know I could use freopen, but this works, and I don't
     * fix what ain't broke.
     */
    if ((out_fp = fopen(output_file, "w+")) == NULL) {
      fprintf(stderr, "%s: can't open output file %s\n", prog, output_file);
      exit(1);
    }
  } else {
    out_fp = stdout;
  }

  /* Mosaic (WWW) output formatting. */
  if (mosaic == TRUE) {
    fprintf(out_fp, "<HEAD>\n");
    fprintf(out_fp, "<TITLE>Results of keyword search of CMU AI Repository</TITLE>\n");
    fprintf(out_fp, "</HEAD>\n");
    fprintf(out_fp, "<BODY>\n");
    fprintf(out_fp, "<H1>CMU AI Repository Keyword Search</H1>\n");

    fprintf(out_fp,"For help with <tt>keys</tt>, see the ");
    fprintf(out_fp,"<A HREF=\"http://www.cs.cmu.edu:8001/Web/Groups/AI/html/");
    fprintf(out_fp,"keys/keys.html\">documentation page</A>.\n");

    fprintf(out_fp,"To return to the keys form, click ");
    fprintf(out_fp,"<A HREF=\"http://www.cs.cmu.edu:8001/Web/Groups/AI/html/");
    fprintf(out_fp,"keys/keysform.html\">here</A>.\n");

    fprintf(out_fp,"For more information on the repository, see the ");
    fprintf(out_fp,"<A HREF=\"http://www.cs.cmu.edu:8001/Web/Groups/AI/html/");
    fprintf(out_fp,"air.html\">CMU AI Repository Home Page</A>.\n");

    fprintf(out_fp, "<H2>Results of keyword search</H2>\n");
    fprintf(out_fp, "<PRE>\n");
  }

  /* Mosaic (WWW) output formatting. */
  if (mosaic == TRUE) {
    /* Was prog instead of "keys", but that was a bad idea. */
    fprintf(out_fp, "%s version %s<BR>\n%s\n\n", "keys", version, copyright);
    fprintf(out_fp, "<B>Keywords:</B> ");
    for (i=0; i<argc; i++) {
      fprintf(out_fp, "<i>%s</i> ", argv[i]);
    }
    fprintf(out_fp, "\n\n");
  } else {
    /* Was prog instead of "keys", but that was a bad idea. */
    fprintf(out_fp, "%s version %s\n%s\n", "keys", version, copyright);
  }

  if ( argc == 0 ) {
    fprintf(stderr, "%s: no keywords specified\n", prog);
    usage();
  }

  /* Set up the keyword index and package list filenames. */
  if (strcmp(ptf_case,"lower") == 0) {
    sprintf(ptf_ki,"%s%s%s", ptf_base, "/a2z/lists/ki.txt", ptf_ver);
    sprintf(ptf_pl,"%s%s%s", ptf_base, "/a2z/lists/pl.txt", ptf_ver);
  } else {
    sprintf(ptf_ki,"%s%s%s", ptf_base, "/A2Z/LISTS/KI.TXT", ptf_ver);
    sprintf(ptf_pl,"%s%s%s", ptf_base, "/A2Z/LISTS/PL.TXT", ptf_ver);
  }

  /* The keyword searching routine. */
  process_keywords(argc,argv);
  if (mosaic == TRUE) {
    fprintf(out_fp, "</PRE>\n");
    fprintf(out_fp, "</BODY>\n");
  }

  if (output_file_flag == TRUE) {
    fclose(out_fp);
  }
}

/* Various usage outputing functions. */
char *helpmsg[] ={
  "",
  "This program searches the keyword index and package list files for",
  "the CMU AI Repository and/or the PTF for AI CD-ROM for all packages",
  "that match the keywords given on the command line. If any packages",
  "match, they are printed.",
  "",
  "A package must match all the keywords to be printed. To perform several",
  "searches at once, separate the keywords with 'or' or '-o'.",
  "",
  "Keywords are compared with the keyword index in a case-insensitive",
  "fashion. Keywords may be regular expressions in the format used by ed",
  "or grep. You may need to quote the regular expressions, as some",
  "characters may be special to the shell.",
  "",
  "Type 'keys -k' for a summary of regular expression syntax, and",
  "'keys -u' for a summary of program usage.",
  ""
  };

help()
{
  int i;
  for (i = 0; i < 17; i++)
    fprintf(stderr, "%s\n", helpmsg[i]);
  exit(1);
}

char *regexphelpmsg[] ={
  "",
  "A synopsis of regular expression syntax is as follows:",
  "   char    \t Matches itself",
  "   .       \t Matches any single character except newline",
  "   ?       \t Preceding item is optional",
  "   *       \t Preceding item may be included 0 or more times",
  "   +       \t Preceding item may be included 1 or more times",
  "   |       \t OR",
  "   ^       \t Beginning of line",
  "   $       \t End of line",
  "   <       \t Beginning of word",
  "   >       \t End of word",
  "   [chars] \t Range of characters, e.g., [a-z], [abde], [0-9]",
  "   \\      \t Use before one of above characters to escape it",
  "   ()      \t Overrides operator precedence, and provides 'memory'",
  "   \\n     \t Matches repeat of text matched earlier in nth ()",
  ""
  };

regexphelp()
{
  int i;
  for (i = 0; i < 17; i++)
    fprintf(stderr, "%s\n", regexphelpmsg[i]);
  exit(1);
}

usage()
{
  fprintf(stderr, "usage: %s [-bhkum] [-r d] [-c c] [-d 0.doc] [-v v] [-o f] [-l l] keywords ...\n", prog);
  fprintf(stderr, "   -b   \tBatch mode, turns off more processing\n");
  fprintf(stderr, "   -m   \tProduces output in HTML format\n");
  fprintf(stderr, "   -h   \tPrints a help message\n");
  fprintf(stderr, "   -u   \tPrints this message\n");
  fprintf(stderr, "   -k   \tPrints regular expressions summary\n");
  fprintf(stderr, "   -l lines \tLines per screen in paged output (default %d)\n", SCREENLINES);
  fprintf(stderr, "   -o file \tSends output to the specified file\n");
  fprintf(stderr, "   -r dir  \tOverrides base directory       ($PTF_BASE)\n");
  fprintf(stderr, "   -c case \tOverrides default case         ($PTF_CASE)\n");
  fprintf(stderr, "   -d doc  \tOverrides default docfile name ($PTF_DOC)\n");
  fprintf(stderr, "   -v ver  \tOverrides default version      ($PTF_VER)\n");
  fprintf(stderr, "   keywords \tKeywords that must match the listing\n");
  exit(1);
}

/* The keyword searching routine. Complicated, but efficient. */
process_keywords(num_keywords,keywords)
     int num_keywords;
     char **keywords;
{
  int i, length, key_index, ki_index, comma_pos;
  char ki_line[MAXLINE];
  FILE *ki_fp,*pl_fp, *fopen();
  char ki_key[KEYLENGTH], ki_key_copy[KEYLENGTH];
  char ki_indices[MAXLINE], ki_indices_copy[MAXLINE];
  char ki_tmp[MAXLINE], *keyword;
  regexp *reg_keywords[MAXKEYS];
  regexp *reg_keyword;
  int prev_start = 0;
  int prev_end = 0;
  int curr_start = 0;
  int curr_end = 0;
  int num_printed = 0;
  int printable = FALSE;
  int consecutive = TRUE; 
  int last_or = FALSE; 
  int curr_index, prev_index = 0;

  if (num_keywords > MAXKEYS) {
    fprintf(stderr, "%s: too many keywords %d > MAXKEYS\n", 
	    prog, num_keywords);
    exit(1);
  }

  /* Open the keyword index and package list for reading. */
  if ((ki_fp = fopen(ptf_ki, "r")) == NULL) {
    fprintf(stderr, "%s: can't open keyword index %s\n", prog, ptf_ki);
    exit(1);
  }

  if ((pl_fp = fopen(ptf_pl, "r")) == NULL) {
    fprintf(stderr, "%s: can't open package list %s\n", prog, ptf_pl);
    exit(1);
  }

  /* Test for two consecutive -o's error. */
  for (key_index = 0; key_index < num_keywords; key_index++) {
    if (is_or(keywords[key_index]) == TRUE) {
      if (last_or == TRUE) {
	/* Two consecutive -o's! Error. */
	fprintf(stderr, 
		"%s: %s followed by %s without intervening keywords\n", 
		prog, keywords[key_index-1], keywords[key_index]);
	exit(1);
      } else {
	last_or = TRUE; 
      }
    } else {
      last_or = FALSE; 
    }
  }

  /* Convert the regular expression to lower case for case-insensitive 
     matching, and then compile it and save the results. */
  for (key_index = 0; key_index < num_keywords; key_index++) {
    keyword = keywords[key_index];
    lcase_string(keyword);
    if (is_or(keyword) == FALSE) {
      reg_keywords[key_index] = regcomp(keyword);
    }
  }

  /* Find matches, and insert them into the match array. */
  while ( (length = readline(ki_fp, ki_line, MAXLINE)) != EOF ) {
    /* Strip off the line's keyword, leaving the indices. This
     * cruft because sscanf's inadequacy prevents us from using it.
     */
    comma_pos = 0;
    for (i = 0; (i < length) && (comma_pos == 0); i++) {
      if (ki_line[i] == ',') {
	comma_pos = i;
	ki_key[i] = '\0';
      } else {
	ki_key[i] = ki_line[i];
      }
    }
    for (i = comma_pos+2; i < length; i++) {
      ki_indices[i-comma_pos-2] = ki_line[i];
    }
    ki_indices[i-comma_pos-2] = '\0';
    /* sscanf(, "%s, %s", &ki_key, &); */

    /* Convert to lower case, for case-insensitive matching. */
    strcpy(ki_key_copy, ki_key);
    lcase_string(ki_key);

    /* For each key, see if it matches. */
    for (key_index = 0; key_index < num_keywords; key_index++) {
      strcpy(ki_indices_copy, ki_indices);
      keyword = keywords[key_index];
      reg_keyword = reg_keywords[key_index];
      if ((is_or(keyword) == FALSE) && (regexec(reg_keyword,ki_key) == TRUE)) {
	/* Parse each index of the ki_indices as a number and add
	 * a tuple to the database, as appropriate.
	 */
	while (sscanf(ki_indices_copy, "%d", &ki_index) == 1) {
	  length = strlen(ki_indices_copy);
	  /* More sscanf lossage avoidance. */
	  comma_pos = 0;
	  for(i=0; (i < length) && (comma_pos == 0); i++) {
	    if ( ki_indices_copy[i] == ',' ) {
	      comma_pos = i;
	    }
	  }
	  if (comma_pos == 0) {
	    ki_indices_copy[0] = '\0';
	  } else {
	    for (i = comma_pos+1; i < length; i++) {
	      ki_indices_copy[i-comma_pos-1] = ki_indices_copy[i];
	    }
	    ki_indices_copy[i-comma_pos-1] = '\0';
	  }
	  /* Add the tuple to the database, inserting in proper position. */
	  add_match(key_index, ki_index, ki_key_copy);
	}
      }
    }
  }

  /*
   * The matches array now consists of a set of ordered tuples of the form
   *    (keynum, ki_index, ki_key)
   * sorted first by ki_index, then by keynum, then by ki_key, where
   * ki_index is an index number from ki.txt, ki_key is the keyword from
   * ki.txt for that index number, and keynum is the index of corresponding
   * keyword input by the user. If, for a given ki_index, the keynums are
   * in consecutive order from one boundary (e.g., start or end keyword,
   * or -or keyword) to the next, print out the package list entry for
   * that index.
   */

  /* For debugging. */
  /*
  for (i=0; i < num_matches; i++) {
    fprintf(out_fp, "(%d, %d, %s)\n", 
	   matches[i].keynum, matches[i].ki_index, matches[i].ki_key); 
  }
  */

  for (i=0; i < num_matches; i++) {
    /*
     * We maintain the following state:
     *   - curr_index is the ki_index of the current tuple
     *   - prev_index is the ki_index of the previous tuple
     *   - consecutive is a flag for whether the current set of keynums
     *     has been consecutive so far.
     *   - printable is a flag which says whether the current ki_index
     *     is printable -- i.e., at least one of the ki_index's keynum
     *     sequences was consecutive
     *   - curr_start and curr_end are the indices of the start and end 
     *     tuples for ki_index's first such consecutive keynum sequence
     *   - prev_start and prev_end are the indices for the previous
     *     ki_index's consecutive keynum sequence
     */
    curr_index = matches[i].ki_index;
    if (i == 0) {
      /* Boundary condition. First tuple. Initialize printable. */
      printable = FALSE;
      if ((matches[i].keynum == 0) 
	  || (is_or(keywords[matches[i].keynum-1]) == TRUE)) {
      /* If this is the first keynum, or just after an -o keynum, */
	consecutive = TRUE;
      } else {
	consecutive = FALSE;
      }
    } else if (curr_index != prev_index) {
      /* Boundary condition. ki_index change. */
      /* If we're consecutive but not yet printable, and the previous
       * tuple was the last keyword or just before a -o, it is
       * printable.
       */
      if ((consecutive == TRUE) && (printable == FALSE) &&
	  ((matches[i-1].keynum == num_keywords-1) ||
	   (is_or(keywords[1+matches[i-1].keynum]) == TRUE))) {
	printable = TRUE;
	curr_end = i-1;
      }

      /* Print printables at ki_index transitions. */
      if (printable == TRUE) {
	/* Print matching keywords and the package list entry. */
	print_pl_keys(prev_start, prev_end, curr_start, curr_end);
	print_pl_entry(pl_fp, prev_index);
	num_printed++;

	/* and save the current starts and ends. */
	prev_start = curr_start; prev_end = curr_end;
      }
      /* Initialize for next ki_index sequence. */
      curr_start = i; curr_end = i;
      printable = FALSE;
      if ((matches[i].keynum == 0) 
	  || (is_or(keywords[matches[i].keynum-1]) == TRUE)) {
      /* If this is the first keynum, or just after an -o keynum, */
	consecutive = TRUE;
      } else {
	consecutive = FALSE;
      }
    } else {
      /* We're within the same ki_index. */
      if ((matches[i].keynum == matches[i-1].keynum) ||
	  (matches[i].keynum == 1+matches[i-1].keynum)) {
	/* Consecutive, so do nothing. */
      } else {
	/* Not consecutive. */
	if ((consecutive == TRUE) &&
	    (is_or(keywords[1+matches[i-1].keynum]) == TRUE)) {
	  /* If we were consecutive, and prev's next was -o,
	   * we're printable if we weren't already.
	   */
	  if (printable == FALSE) {
	    printable = TRUE;
	    curr_end = i-1;
	  }
	}

	/* Initialize for next keynum sequence and verify start. */
	if (printable == FALSE) {
	  curr_start = i; curr_end = i;
	}
	if ((matches[i].keynum == 0) 
	    || (is_or(keywords[matches[i].keynum-1]) == TRUE)) {
	  /* If this is the first keynum, or just after an -o keynum, */
	  consecutive = TRUE;
	} else {
	  consecutive = FALSE;
	}
      }
    }
    prev_index = curr_index;
  }

  /* Boundary condition. Just saw last tuple. */
  if (num_matches != 0) {
    /* There were some tuples. */
    /* If we're consecutive but not yet printable, and the previous
     * tuple was the last keyword or just before a -o, it is
     * printable.
     */
    if ((consecutive == TRUE) && (printable == FALSE) &&
	((matches[i-1].keynum == num_keywords-1) ||
	 (is_or(keywords[1+matches[i-1].keynum]) == TRUE))) {
      printable = TRUE;
      curr_end = i-1;
    } 
    /* Print printables at last-tuple transitions. */
    if (printable == TRUE) {
      /* Print matching keywords and the package list entry. */
      /* Note prev_index == curr_index here.                 */
      print_pl_keys(prev_start, prev_end, curr_start, curr_end);
      print_pl_entry(pl_fp, prev_index);
      num_printed++;
    }
  }

  if ((num_printed == 0) || (num_matches == 0)) {
    /* There were no tuples. */
    fprintf(out_fp, "No matches found.\n");
  }

  /* Always clean up after yourself. It's only polite. */

  /* Don't forget to close the index files! */
  fclose(ki_fp);
  fclose(pl_fp);

  /* Free up space from the regular expressions. */
  for (key_index = 0; key_index < num_keywords; key_index++) {
    keyword = keywords[key_index];
    if (is_or(keyword) == FALSE) {
      free(reg_keywords[key_index]);
    }
  }
}

/* Tells me whether dis keyword dis an 'or'. */
is_or(keyword)
     char *keyword;
{
  if ((strcmp(keyword,"-o") == 0) 
      || (strcmp(keyword,"-O") == 0)
      || (strcmp(keyword,"or") == 0)
      || (strcmp(keyword,"OR") == 0))
    return(TRUE);
  else
    return(FALSE);
}

/*
 * Reads a line from the file FP up to LIMIT characters in length, 
 * and stores it in STRING, returning the length of the line. Note
 * how we do our own end-of-line detection.
 */
readline(fp, string, limit)
     FILE *fp;
     char *string;
     int limit;
{
  int c;
  int i = 0;

  while (--limit > 0 && (c = getc(fp)) != EOF && c != LFNL) /* fixme! */
    string[i++] = c;
  /*  if (c == '\n') string[i++] = c;    */
  string[i] = '\0';
  if ( c == EOF )
    return(EOF);
  else
    return(i);
}

/*
 * Creates a match structure and inserts it into the matches array 
 * in the position dictated by the sort order.
 *
 * Returns the number of matches currently in the matches array.
 */
add_match(key_index, ki_index, ki_key)
     int key_index, ki_index;
     char ki_key[KEYLENGTH];
{
  struct match match;
  struct match temp;
  int i;
  int splice_index = -1;

  /* For debugging. */
  /*
  fprintf(out_fp, "(%d, %d, %s)\n", key_index, ki_index, ki_key); 
  */

  /* Create the match structure. */
  match.keynum = key_index;	/* maybe outside loop? */
  match.ki_index = ki_index;
  strcpy(match.ki_key,ki_key);

  /* Check for an error condition. */
  if ((num_matches + 1) == MAXMATCHES) {
    fprintf(stderr, "%s: num_matches == MAXMATCHES %d\n", prog, num_matches);
    exit(1);
  }

  /* Add the structure to the match array. */
  for (i = 0; (i < num_matches) && (splice_index == -1); i++) {
    /* If match is less than matches[i], */
    if ((match.ki_index < matches[i].ki_index) ||
	((match.ki_index == matches[i].ki_index) &&
	 (match.keynum < matches[i].keynum)) ||
	((match.ki_index == matches[i].ki_index) &&
	 (match.keynum == matches[i].keynum) &&
	 (strcmp(match.ki_key, matches[i].ki_key) < 0))) {
      /* Splice it in. */
      splice_index = i;
    }
  }
  /* Do the actual splicing. */
  if (splice_index == -1) {
    splice_index = num_matches;
  } else {
    for (i = splice_index; i < num_matches; i++) {
      /* Save a copy of current array contents in temp. */
      temp.keynum = matches[i].keynum;
      temp.ki_index = matches[i].ki_index;
      strcpy(temp.ki_key,matches[i].ki_key);
      /* Store our stuff into the array. */
      matches[i].keynum = match.keynum;
      matches[i].ki_index = match.ki_index;
      strcpy(matches[i].ki_key,match.ki_key);
      /* Set up temp as our stuff for next iteration. */
      match.keynum = temp.keynum;
      match.ki_index = temp.ki_index;
      strcpy(match.ki_key,temp.ki_key);
    }
  }
  /* Last iteration, or first, if splice_index was -1 above. */
  matches[num_matches].keynum = match.keynum;
  matches[num_matches].ki_index = match.ki_index;
  strcpy(matches[num_matches].ki_key,match.ki_key);

  return(++num_matches);
}

/* Convert string to lowercase. */
lcase_string(string)
     char *string;
{
  int length, i;

  length = strlen(string);
  for ( i = 0; i < length; i++) {
    if (isupper(string[i])) 
      string[i]=tolower(string[i]);
  }
}

/* Flag for detecting when we haven't printed out any package lists yet. */
int first_print = TRUE;
/* Starts at 2 for copyright notice. */
int num_lines_printed = 2;

/* Decides whether the keys are the same, and hence don't need to be
 * printed, and if they should be printed, prints them.
 */
print_pl_keys(prev_start, prev_end, curr_start, curr_end)
     int prev_start, prev_end, curr_start, curr_end;
{
  int same = TRUE;
  int i, j;

  /* Test if all the ki_keys are the same as the last ones.
   * If so, skip the printing. */
  if ((curr_end - curr_start) == (prev_end - prev_start)) {
    for (i = curr_start, j = prev_start;
	 (i <= curr_end) && (same == TRUE);
	 i++, j++) {
      if (strcmp(matches[i].ki_key, matches[j].ki_key) != 0) {
	/* We aren't the same as the last ones. */
	same = FALSE;
      }
    }
  } else {
    same = FALSE;
  }
  more_wait(TRUE);
  if ((same == FALSE) || (first_print == TRUE)) {
    /* We are not the same or we're the first, so print it. */
    if (first_print == FALSE) {
      fprintf(out_fp, "\n");
      more_wait(FALSE);
    }
    for (i = curr_start; i <= curr_end; i++) {
      fprintf(out_fp, "> %s\n", matches[i].ki_key);
      more_wait(FALSE);
    }
  }
  first_print = FALSE;
}

/* Prints the package list entry. */
print_pl_entry(pl_fp, index)
     int index;
     FILE *pl_fp;
{
  char pl_line[MAXLINE];
  int length, c, pl_index;
  char directory[MAXLINE], disk[1];

  /* To avoid having to maintain a lot of state, we keep our own
   * counter of the package list entry index, and use detection of
   * a character in the left margin to note the next entry. Using
   * ungetc, we can stop and leave the file pointer at the right place.
   */
  while ( (pl_entry_counter < index) && ((c = getc(pl_fp)) != EOF)) {
    ungetc(c, pl_fp);
    if (c == ' ') {
    } else {
      pl_entry_counter++;
    }
    length = readline(pl_fp, pl_line, MAXLINE);
  }
  if (pl_entry_counter == index) {
    sscanf(pl_line, "%s %d %s", directory, &pl_index, disk);
    if (mosaic == TRUE) {
      fprintf(out_fp, "<A HREF=\"%s%s%s\">%s</A>\n", 
	     airdir, directory, ptf_doc, directory);
      more_wait(FALSE);
    } else {
/* Bug in sscanf on vax. */
#ifdef vax
      fprintf(out_fp, "%-68s %4d %s\n", directory, pl_entry_counter, disk);
#else
      fprintf(out_fp, "%-68s %4d %s\n", directory, pl_index, disk);
#endif
      more_wait(FALSE);
    }
    while (((c = getc(pl_fp)) != EOF) && (c == ' ')) {
      length = readline(pl_fp, pl_line, MAXLINE);
      if (mosaic == TRUE) {
	fprintf(out_fp, "%s\n", pl_line+1);
      more_wait(FALSE);
      } else {
	fprintf(out_fp, " %s\n", pl_line);
      more_wait(FALSE);
      }
    }
    ungetc(c, pl_fp);
  }
}

/* CBREAK MODE. Yeah, yeah. */
cbreak_on()
{
  struct sgttyb  IOmode;

  if ( ioctl(0, TIOCGETP, &IOmode) < 0)
    ERROR("Problem getting IO mode...");

  IOmode.sg_flags |= CBREAK;
  if( ioctl(0, TIOCSETP, &IOmode) )
    ERROR("Problem setting IO mode...");
}

cbreak_off()
{
  struct sgttyb  IOmode;

  if ( ioctl(0, TIOCGETP, &IOmode) < 0)
    ERROR("Problem getting IO mode...");

  IOmode.sg_flags &= ~CBREAK;
  if( ioctl(0, TIOCSETP, &IOmode) )
    ERROR("Problem setting IO mode...");
}

/* Rinky dink little more processing routine. So? */
more_wait(preferred)
     int preferred;
{
  int c;

  if (more == TRUE) {
    if (preferred == FALSE) {
      num_lines_printed++;
    }
    if ((page_at_preferred == FALSE) || (preferred == TRUE)) {
      if (num_lines_printed >= screenlines) {
	fprintf(out_fp, "--MORE--");
	cbreak_on();
	c = getchar();
	cbreak_off();
	fprintf(out_fp, "\n");
	if (c == 'q' || c == 'Q') {
	  exit(0);
	} else if (c == LFNL) {
	  /* num_lines_printed -= 4; */
	  num_lines_printed = 0;
	} else {
	  num_lines_printed = 0;
	} 
      }
    }
  }
}

/* EOF */
