/**************************************************************************/
/* communication.c                                              /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>


#include <errno.h>

#define COMMUNICATION_EXTERN


#include "communication.h"
#include "global_communication.h"
#include "datastruct.h"
#include "xgraphics.h"
#include "version.h"
#include "data_code.h"
#include "main.h"

/* for sockets */
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/time.h>


#include <X11/Xlib.h>
#include <X11/Xutil.h>


/*****************************************************************************
 *                                                                           *
 * Code for the main program only                                            *
 *                                                                           *
 *****************************************************************************/

/*
 * ok_exit() and panic_exit() are provided such that other sources don't
 * need to know the code for STOP and PANIC.  Bit silly, really.
 */
void ok_exit(void)
{
  close_down(STOP);
}


void panic_exit(void)
{
  close_down(PANIC);
}

void init_server_communication(char *file)
{
  /*
   * Create a blocking socket.
   */
  setup_communication(file, &control_socket, &PpPSoCkEt);
  connected = 1;
}


void read_hmatrix(char *buffer, int length, unsigned short expected_nr, homo h)
{
  unsigned short nr;

  if (length != sizeof(unsigned short) + sizeof(homo))
  {
	fprintf(stderr, "graphics: received message with illegal size %d\n",
							length);
	panic_exit();
  }

  memcpy(&nr, buffer, sizeof(unsigned short));

  if (nr != expected_nr)
  {
	fprintf(stderr, "graphics: expected block %hu, got block %hu\n",
						expected_nr, nr);
	panic_exit();
  }

  memcpy(h, buffer + sizeof(unsigned short), sizeof(homo));
}



void read_next_message(void)
{
  int length;
  char *buffer, type;
  unsigned short nr;

  nr = 0;
  while (1)
  {
	fd_set fdset;

	/*
	 * Check two sockets: the incoming homogenous matrix socket,
	 * and the socket containing XEvents.
	 */
	FD_ZERO(&fdset);
	FD_SET(ConnectionNumber(display), &fdset);
	if (accept_homos)
		FD_SET(PpPSoCkEt, &fdset);
	if (select(GETDTABLESIZE, &fdset, (fd_set *) NULL, (fd_set *) NULL,
						(struct timeval *) NULL) == -1)
	{
		perror("graphics/read_next_message: select()");
		panic_exit();
	}

	if (FD_ISSET(ConnectionNumber(display), &fdset))
	{
		/*
		 * We got an event.  Immediately jump to the event handling
		 * routine, and jump over the rest of the loop.
		 */
		handle_next_window_event();
		continue;
	}

	if (accept_homos && !FD_ISSET(PpPSoCkEt, &fdset))
	{
		/*
		 * This can never occur.
		 */
		fprintf(stderr, "graphics/read_next_message(): select error\n");
		panic_exit();
	}
	
	length = receive_data(PpPSoCkEt, &buffer, &type);

	/*
	 * Check for the type of message we just received.
	 */
	switch (type)
	{
	case STOP:
	  ok_exit();
	  break;
	case PANIC:
	  panic_exit();
	  break;
	case VERSION:
	  printf("This is %s\n"
		 "written by Patrick van der Smagt\n"
		 "Department of Computer Systems\n"
		 "University of Amsterdam\n"
		 "Amsterdam, The Netherlands\n"
		 "email <smagt@fwi.uva.nl>\n\n", local_version);
	  break;
	  
	case HALT_OK:
	  /*
	   * This should be checked: do we indeed receive
	   * the HALT_OK after having sent the HALT, 
	   * is this one not unexpected, etc.
	   */
	  break;
	case HOMO:
	  /*
	   * Read in homogenous matrices for all
	   * links EXCEPT the first one, which is
	   * the floor.
	   */
	  read_hmatrix(buffer, length, nr+1, homogen[nr+1]);
	  
	  if (++nr == nr_links-1)
	    {
	      draw_next_wire_frame();
	      nr = 0;
	    }
	  break;
	default:
	  panic_exit();
	  break;
	}
	
	if (buffer != NULL)
	  free(buffer);
   }
}



void send_halt(void)
/*
 * Somebody typed an `h' in the window.  We must tell the simulator
 * that it must stop moving.  First send the HALT message to the
 * homogenous matrices passing program, then wait for the accept.
 */
{
  send_data(PpPSoCkEt, HALT, 0, (char *) NULL);
  /*
   * We have to receive a HALT_OK, but in the meantime other
   * data might arrive.  So checking for HALT_OK can only be done above.
   */
}


void close_down(char message)
{
  /*
   * Allow no interruptions whil closing the communication channels.
   */
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT, SIG_IGN);

  exit_graphics();

  if (connected)
  {
	connected = 0;
	send_data(PpPSoCkEt, message, 0, (char *) NULL);
	close(PpPSoCkEt);
  }

  exit(0);
}



/************************************************************************/
/*									*/
/************************************************************************/
void setup_communication(char *socketfile, int *control_socket, int *h_socket)
{
  char type;
  int socknr;
  FILE *sockfile;

  if ((sockfile = fopen(socketfile,"r")) == NULL)
  {
	fprintf(stderr, "graphics: can't open %s\n", socketfile);
	panic_exit();
  }
  fscanf(sockfile, "%d", &socknr);
  fclose(sockfile);

  printf("bemmel: waiting at socket %d\n", socknr);

  /* Setup socket.  The number of the connection is
     socknr (see common.h).  The socket line is
     in control_socket, whereas communication goes over the
     file descriptor data_socket.  The last parameter
     is the maximum number of clients that is allowed
     to connect.
   */
  if (setup_server_socket(socknr, control_socket, 1) == -1)
	panic_exit();

  /* Now for each client accept the connection.  In this case,
     we only have one client.
   */
  accept_client(*control_socket, h_socket);

  /* set up connection with client wait for alive char */
  send_data(*h_socket, START, 0, (char *) NULL);
  do
  {
	receive_data(*h_socket, (char **) NULL, &type);
	if (type == START_ERROR)
	{
		fprintf(stderr, "graphics: homo says it can't start\n");
		panic_exit();
	}
	if (type != START_OK)
		fprintf(stderr, "graphics: received wrong message from homo\n");
  } while (type != START_OK);
}



/************************************************************************/
/*			Socket routines					*/
/************************************************************************/
void send_data(int sock, char code, unsigned short length, char *data)
{
  char *buf;
  unsigned short i;
  int status;

  if ((buf = (char *) malloc(length+3)) == NULL)
  {
	fprintf(stderr, "graphics/send_data(): out of buffer space\n");
	panic_exit();
  }

  buf[0] = code;
  us_encode(length, buf+1);

  if (data != NULL)
	for (i=0; i<length; i++) 
		buf[i+3] = data[i];


  do
	status = send(sock, buf, (int) length+3, 0);
  while (status == -1 && errno == EWOULDBLOCK);

  if (status == -1)
  {
	perror("graphics/send_data()");
	panic_exit();
  }
  else if (status != length+3)
  {
	fprintf(stderr, "graphics/send_data() couldn't send all its bytes\n");
	perror("graphics/send_data()");
	panic_exit();
  }

  free(buf);
}



int receive_data(int sock, char **buffer, char *type)
{
  extern int errno;
  unsigned short message_len;
  char buf[3];
  int bytes_to_receive;

  /* Read the header of the message. */
  bytes_to_receive = 3;
  while (1)
  {
	int status;

	status = recv(sock, buf, bytes_to_receive, 0);

	if (status == bytes_to_receive) break;
	if (status == -1 && errno == EWOULDBLOCK) return -1;
	if (status != -1)
	{
		if (status == 0)
			panic_exit();
		/*
		 * For the moment, we choose not to ignore too little
		 * data, when NO data at all has arrived.  Otherwise,
		 * just continue reading the rest of the data.
		 */
		bytes_to_receive -= status;
	}
	else
	{
		perror("graphics/receive_data()");
		panic_exit();
	}
  }

  *type = buf[0];
  if (*type == PANIC)
  {
	fprintf(stderr, "graphics: got exit request\n");
	panic_exit();
  }

  message_len = us_decode(buf+1);

  /*
   * Read the rest of the message.
   */
  if (buffer != (char **) NULL)
	*buffer = NULL;
  if (message_len > 0)
  {
	int bytes_to_receive;

	if (buffer == NULL)
	{
		fprintf(stderr, "graphics/receive_data(): message too long\n");
		panic_exit();
	}

	*buffer = (char *) malloc(message_len+1);
	if (*buffer == NULL)
	{
		perror("graphics/receive_data()");
		panic_exit();
	}

	bytes_to_receive = message_len;
	while (1)
  	{
		int status;

		status = recv(sock, *buffer + message_len - bytes_to_receive,
						bytes_to_receive, 0);

		if (status == bytes_to_receive) break;
		if (status == -1 && errno == EWOULDBLOCK) continue;
		if (status != -1)
		{
			/*
			 * When NO data at all is available, the other
			 * side may have quit.  Stop work, anyway.
			 */
			if (status == 0) panic_exit();
			bytes_to_receive -= status;
		}
		else
		{
			perror("graphics/receive_data()");
			panic_exit();
		}
	}
  }

  return (int) message_len;
}
