/**************************************************************************/
/* command.c                                                    /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>


#include "config.h"

/*
 * This is only used to pretty-print the commands
 * when a help-request is executed.  To find out
 * how wide the terminal is!
 */
#if SUNOS || SOLARIS
#include <sys/termios.h>
#endif


#define COMMAND_EXTERN

#include "command.h"


/*
 * If a parse error occurred, the rest of the input line
 * is discarded.  This helps if a command on a line is
 * misspelled, and it is followed by a second command on
 * the same line.
 */
static int parse_error = 0;



/* 
 * Reads next word from stdin and
 * assign pointer to it to *p_return_buffer.
 * If the input is an empty line, **p_return_buffer
 * will be '\0'.
 *
 * Returns **p_return_buffer.
 */
char get_next_word(char **p_return_buffer)
{
  static char input_buffer[1000];
  static char *p_input_buffer;

  *p_return_buffer = p_input_buffer;

  if (parse_error || p_input_buffer == NULL || *p_input_buffer == '\0')
  {
	/*
	 * The buffer has been decoded, fill it again.
	 */
	if (gets(input_buffer) == NULL)
		ok_exit();

	/*
	 * End of the buffer is marked by a double '\0'.
	 */
	input_buffer[strlen(input_buffer)+1] = '\0';
	*p_return_buffer = p_input_buffer = input_buffer;
  }

  /*
   * Remove leading whitespace.
   */
  for ( ; *p_input_buffer; (*p_return_buffer)++, p_input_buffer++)
	if (!isspace(*p_input_buffer))
		break;

  /*
   * Select word up to white space.
   */
  for ( ; *p_input_buffer; p_input_buffer++)
	if (isspace(*p_input_buffer))
		break;
  *p_input_buffer = '\0';
  p_input_buffer++;

  return **p_return_buffer;
}


  
/*
 * Look for errors in command.h.
 */
void check_command_integrity(void)
{
  int i, j;

  for (i=0; command[i].code != 0; i++) for (j=i+1; command[j].code != 0; j++)
    if (command[i].code == command[j].code)
      {
	fprintf(stderr,"WARNING: command code %o has multiple occurrences\n",
		command[i].code);
      } else
    if (strcmp(command[i].text, command[j].text) == 0)
    {
      fprintf(stderr,"WARNING: command \"%s\" appears twice, code %o and %o\n",
		command[i].text, command[i].code, command[j].code);
      if (command[i].code == command[j].code)
	fprintf(stderr, "It's not so bad, just useless.\n");
      else
	fprintf(stderr, "and these occurrences have different codes!\n");
    }
}


long int parse_command(char **s)
{
  int i, n_hits;
  long int code;

  if (**s == '\0')
	return 0;

  /*
   * Check if the word is known.
   */
  n_hits = 0;
  for (i=0; command[i].code != 0; i++)
  {
	if (strncmp(*s, command[i].text, strlen(*s)) == 0)
	{
		n_hits++;
		code = command[i].code;
		/*
		 * It may be that a command is a substring
		 * of another command.  If so, it's a unique
		 * hit, and no further search is necessary.
		 */
		if (strlen(*s) == strlen(command[i].text))
		{
			n_hits = 1;
			break;
		}
	}
  }

  if (n_hits == 0)
  {
	printf("unrecognised command \"%s\"\n", *s);
	parse_error = 1;
	return 0;
  }
  if (n_hits > 1)
  {
	printf("ambiguous command \"%s\"; possible completions:\n", *s);
	for (i=0; command[i].code != 0; i++)
	{
		if (strncmp(*s, command[i].text, strlen(*s)) == 0)
			printf("	%s\n", command[i].text);
	}
	parse_error = 1;
	return 0;
  }
  parse_error = 0;
  return code;
}


int mstrcmp(char **a, char **b)
{
  return strcmp(*a, *b);
}



/*
 * Give a list of all commands, taking the width
 * of the screen into account.  The list is sorted
 * alphabetically, btw.
 */
void help(void)
{
  int i, maxlen, column, n_rows, n_columns, n_commands, row;
  char **sorted;
#if SUNOS || SOLARIS
  struct winsize wsize;

  ioctl(0, TIOCGWINSZ, &wsize);
#endif

  n_commands = maxlen = 0;
  for (i=0; command[i].code != 0; i++)
  {
	if (maxlen < strlen(command[i].text))
		maxlen = strlen(command[i].text);
	n_commands++;
  }
#if SUNOS || SOLARIS
  n_columns = wsize.ws_col / (maxlen+2);
#else
  n_columns = 80 / (maxlen+2);
#endif
  n_rows = n_commands / n_columns;
  if (n_commands % n_columns) n_rows++;

  if ((sorted = (char **) calloc(n_commands, sizeof(char *))) == NULL)
  {
	perror("sorted");
	return;
  }
  for (i=0; command[i].code != 0; i++)
	sorted[i] = command[i].text;

  qsort((char *)sorted, n_commands, sizeof(char *),
	(int (*)(const void*,const void*))mstrcmp);

  printf("Recognised commands are:\n");
  column = row = 0;
  for (i=0; i <= n_commands; i++)
  {
	if (row + column*n_rows < n_commands)
		printf("%-*s ", maxlen+1, sorted[row + column*n_rows]);
	if (++column >= n_columns)
	{
		putchar('\n');
		column = 0;
		row++;
	}
  }
  printf("\n");
  printf("Use \"describe <command>\" "
	 "to describe one of the above commands.\n\n");
}



void describe(void)
{
  char *word;
  long code;
  int i;

  if (get_next_word(&word) == '\0')
  {
	printf("Usage:  describe <command>\n");
	return;
  }

  if ((code = parse_command(&word)) == 0)
	return;

  for (i=0; command[i].code != 0; i++)
  {
	if (command[i].code == code)
	{
		puts(command[i].description);
		break;
	}
  }
  putchar('\n');
}
