/**************************************************************************/
/* communication.c                                              /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

/*
 * Contains no robot-specific code.
 */

#define COMMUNICATION_EXTERN

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <errno.h>

#include <signal.h>

#include <sys/types.h>
#include <sys/time.h>

#include "communication.h"
#include "data_code.h"
#include "global_simmel.h"
#include "global_communication.h"




/*
 * The `high-level' communication routines called by the
 * outside world.
 *
 * First: setting up and closing down communication channels.
 */

void open_robot_communication(char *socket_file_name)
{
  setup_client_communication(socket_file_name, getenv("ROBOT_HOST"),
							&robot_socket);
  robot_connected = 1;
  if (fcntl(robot_socket, F_SETFL, O_NDELAY) == -1)
  {
	perror("open_robot_communication(): fcntl()");
	panic_exit();
  }
}


void open_camera_communication(char *socket_file_name)
{
  setup_client_communication(socket_file_name, getenv("CAMERA_HOST"),
							&camera_socket);
  camera_connected = 1;
  if (fcntl(camera_socket, F_SETFL, O_NDELAY) == -1)
  {
	perror("open_camera_communication(): fcntl()");
	panic_exit();
  }
}



void open_bemmel_communication(char *basic_name, int bemmels_connected)
{
  int i;

  if (bemmels_connected == 0)
	return;

  bemmel_socket = (int *) calloc(bemmels_connected, sizeof(int));
  bemmel_connected = (int *) calloc(bemmels_connected, sizeof(int));
  if (bemmel_socket==NULL || bemmel_connected==NULL)
  {
	perror("open_bemmel_communication()");
	panic_exit();
  }

  for (i=0; i<bemmels_connected; i++)
  {
	char socket_file_name[BUF_SIZE];

	sprintf(socket_file_name, "%s%d", basic_name, i+1);
	setup_client_communication(socket_file_name,
			getenv("BEMMEL_HOST"), &(bemmel_socket[i]));
  	bemmel_connected[i] = 1;
  }
}


void accept_halt(void)
{
  send_data(robot_socket, HALT_OK, 0, (char *) NULL);
}




void ok_exit(void)
{
  close_down(STOP, STOP, STOP);
}



void panic_exit(void)
{
  close_down(PANIC, PANIC, PANIC);
}



void close_down(char m_robot, char m_camera, char m_bemmel)
{
  int i;

  /*
   * Allow no interruptions whil closing the communication channels.
   */
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT, SIG_IGN);

  /*
   * Note that send_data itself can invoke close_down().  To prevent
   * endless recursion, we first have to set robot_connected et al
   * to 0.
   */
  if (robot_connected)
  {
	robot_connected = 0;
	send_data(robot_socket, m_robot, 0, (char *) NULL);
	close(robot_socket);
  }

  if (camera_connected)
  {
	camera_connected = 0;
	send_data(camera_socket, m_camera, 0, (char *) NULL);
	close(camera_socket);
  }

  if (bemmel_connected != NULL)
  {
	/*
	 * It might occur that, during connecting, an error occurs.
	 * In this case bemmel_connected might not be allocated yet,
	 * which is why we check for that condition above.
	 */

	for (i=0; i<bemmels_connected; i++)
	{
 		if (bemmel_connected[i])
  		{
			bemmel_connected[i] = 0;
			send_data(bemmel_socket[i], m_bemmel, 0, (char *) NULL);
			close(bemmel_socket[i]);
  		}
	}

	if (bemmels_connected != 0)
	{
		free(bemmel_socket);
		free(bemmel_connected);
	}
  }

  exit(0);
}


/*
 * Next, the routine that checks if there is any new input
 * from the input channels (i.e., the robot and camera
 * sockets).
 *
 * This implementation uses the select() system call.  This
 * is a very neat way to solve the reading from three pipes,
 * were it not that this program must be continuously active,
 * when the robot is kept in motion.  Therefore, this routine
 * alone cannot be used to read incoming data.  We choose for
 * the optimal solution (run-time wise): when the robot does
 * not move, this routine waits for incoming data.  When the
 * robot is in motion, however, the routine receive_next_command()
 * is repeatedly called to check for incoming data.
 */
int select_receive_next_command(char *type, char **buffer, 
				unsigned short *length, int pend)
{
  fd_set fdset;
  int i, socket_type, return_type, r, status;
  struct timeval timeout;

  FD_ZERO(&fdset);
  FD_SET(robot_socket, &fdset);
  FD_SET(camera_socket, &fdset);
  for (i=0; i<bemmels_connected; i++)
	FD_SET(bemmel_socket[i], &fdset);

  if (!pend)
  {
	/*
	 * Wait only the shortest possible time for data.
	 * If you want this process to have a lower execution
	 * priority than the controller, increase the waiting
	 * times.
	 *
	 * getdtablesize() does not exist on some machines.
	 * See config.h.
	 */
	timeout.tv_sec = (time_t) 0;	/* seconds */
  	timeout.tv_usec = (time_t) 0;	/* micro seconds */
	status = select(GETDTABLESIZE, &fdset, (fd_set *) NULL,
		(fd_set *) NULL, &timeout);
	if (status == 0) return EMPTY_SOCKET;
  }
  else
	status = select(GETDTABLESIZE, &fdset, (fd_set *) NULL,
		(fd_set *) NULL, (struct timeval *) NULL);

  if (status == -1)
  {
	perror("receive_next_command(): select()");
	panic_exit();
  }

  return_type = 0;

  /*
   * Check if this was robot data
   */
  if (FD_ISSET(robot_socket, &fdset))
  {
	socket_type = robot_socket;
	return_type = ROBOT;
  }
  /*
   * or camera data
   */
  else if (FD_ISSET(camera_socket, &fdset))
  {
	socket_type = camera_socket;
	return_type = CAMERA;
  }
  /*
   * or bemmel data.
   */
  else for (i=0; i<bemmels_connected; i++)
  {
	if (FD_ISSET(bemmel_socket[i], &fdset))
	{
		socket_type = bemmel_socket[i];
		return_type = BEMMEL;
	}
  }

  /*
   * Check for the impossible.
   */
  if (!return_type)
  {
	/*
	 * This can never occur.
	 */
	fprintf(stderr, "receive_next_command(): select error\n");
	panic_exit();
  }

  /*
   * Note that receive_data() will allocate memory if a message
   * of length exceeding 0 has arrived.  This buffer will be
   * thrown away in decode_camera() or decode_joints().
   */
  if ((r = receive_data(socket_type, buffer, type)) == -1)
  {
	perror("receive_next_command(): receive_data()");
	panic_exit();
  }
  *length = (unsigned short) r;

  return return_type;
}




/*
 * Next, the routine that checks if there is any new input
 * from the input channels (i.e., the robot and camera
 * sockets).  Each channel is checked in turn, i.e., a
 * channel is never checked twice in two consecutive calls.
 *
 * Note that an easier way to implement this is using select().
 */
int receive_next_command(char *type, char **buffer, unsigned short *length)
{
  /*
   * The structure used to store all the necessary data of one channel.
   */
  struct liststruct {
	int socket_channel;
	int type;
	struct liststruct *next;
  };

  static struct liststruct *current_socket, sockets[2];
  int r, received_type;

  /*
   * First initialise the list of sockets we have to listen to.
   * current_socket Points to the socket we want to check next.
   */
  if (current_socket == (struct liststruct *) NULL) 
  {
	sockets[0].socket_channel = robot_socket;
	sockets[0].type = ROBOT;
	sockets[0].next = &(sockets[1]);
	sockets[1].socket_channel = camera_socket;
	sockets[1].type = CAMERA;
	sockets[1].next = &(sockets[0]);
	current_socket = &(sockets[0]);
  }

  /*
   * Note that receive_data() will allocate memory if a message
   * of length exceeding 0 has arrived.  This buffer will be
   * thrown away in decode_camera() or decode_joints().
   */
  if ((r = receive_data(current_socket->socket_channel, buffer, type)) == -1)
  {
	current_socket = current_socket->next;

	if ((r = receive_data(current_socket->socket_channel,buffer,type))==-1)
		return EMPTY_SOCKET;
  }

  received_type = current_socket->type;
  current_socket = current_socket->next;

  *length = (unsigned short) r;

  return received_type;
}


/*
 * bemmel[i] should never send anything back, unless they
 * exited.  Check if anything at all has arrived.
 */
char check_bemmel_channels_for_data(void)
{
  char *buffer, type;
  int i;

  for (i=0; i<bemmels_connected; i++)
  	if (receive_data(bemmel_socket[i], (char **) NULL, &type) != -1)
  	{
		/*
	 	* Wow, we got data!  Something must have happened.
	 	*/
		if (type == HALT)
		{
			/*
			 * The simulator received a message that the
			 * robot should stop moving.  Accept it,
			 * and stop the robot.
			 */
			send_data(bemmel_socket[i], HALT_OK, 0, (char *) NULL);
			return HALT;
		}
		if (type == STOP) ok_exit();
		panic_exit();
  	}
  return type;
}




/*
 * Data sent to the simulator is not coded but sent in raw format.
 * The amount of data sent via these two channels is considerable;
 * therefore, this less robust method is preferred.  Do run both
 * this code and the bemmel simulators on the same architecture!
 */
void send_homo(unsigned short nr, homo h)
{
  char buffer[BUF_SIZE];
  unsigned short length;
  int i;

  length = sizeof(unsigned short) + HOMO_DEG * HOMO_DEG * sizeof(REAL);
  if (length > BUF_SIZE)
  {
	fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
	panic_exit();
  }

  memcpy(buffer, &nr, sizeof(unsigned short));
  memcpy(buffer + sizeof(unsigned short), h, HOMO_DEG*HOMO_DEG * sizeof(REAL));


  for (i=0; i<bemmels_connected; i++)
	send_data(bemmel_socket[i], HOMO, length, buffer);
}




/*
 * Routines for communication via the camera channel.  There are
 * currently three types of return:
 *	1. CAMERA_1: send observed position of end-effector and object;
 *	2. CAMERA_2: send observed position of end-effector and object;
 *	3. HAND_CAMERA: send observed position and area of object.
 * See main.c.
 */
void send_camera(char type, vector view_1, vector view_2)
{
  char buffer[BUF_SIZE];
  unsigned short length;

  /*
   * The first integer sent is a possible error code, in use when the
   * controller is communicating with real image processing software.
   */
  length = encode(buffer, "%d%f%f%f%f%f%f", 0,
	(float) view_1[0], (float) view_1[1], (float) view_1[2],
	(float) view_2[0], (float) view_2[1], (float) view_2[2]);

  if (length > BUF_SIZE)
  {
	fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
	panic_exit();
  }

  send_data(camera_socket, type, length, buffer);
}


void send_end_effector(vector effector)
{
  char buffer[BUF_SIZE];
  unsigned short length;

  length = encode(buffer, "%d%f%f%f%f%f%f", 0,
	(float) effector[0], (float) effector[1], (float) effector[2]);

  if (length > BUF_SIZE)
  {
	fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
	panic_exit();
  }
  send_data(camera_socket, END_EFFECTOR, length, buffer);
}


void send_position_and_derivatives(vector x, vector dx, vector ddx)
     /*
      * Sends the position, velocity, and acceleration of
      * the object relative to the end-effector.  This
      * was introduced for time-to-contact experiments,
      * where the object is stationary and the robot is
      * moving.
      *
      * The three arguments to this function contain the
      * position, velocity, and acceleration
      * in the x ([0]), y ([1]), and z ([2]) direction.
      */
{
  char buffer[BUF_SIZE];
  unsigned short length;

  length = encode(buffer, "%d%f%f%f%f%f%f%f%f%f", 0,
		  (float) x[0], (float) x[1], (float) x[2],
		  (float) dx[0], (float) dx[1], (float) dx[2],
		  (float) ddx[0], (float) ddx[1], (float) ddx[2]);

  if (length > BUF_SIZE)
  {
	fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
	panic_exit();
  }
  send_data(camera_socket, POSITION_AND_D, length, buffer);
}


/*
 * The simulator has to know one way or another where the
 * target is situated.  This is a discrepancy with the
 * real world.  Ideally, the two graphical simulators would
 * used to indicate the position of the target by clicking
 * the mouse in the windows.  Something to do when we have the time.
 */
void decode_camera(char *buffer, int length, vector object_position)
{
  int fields;
  float i1, i2, i3, i4, i5;

  fields = decode(buffer, length, "%f%f%f%f%f", &i1, &i2, &i3, &i4, &i5);
  if (fields != 5)
  {
	fprintf(stderr, "decode_camera() got %d fields, expected 5\n", fields);
	panic_exit();
  }

  object_position[0] = (REAL) i1;
  object_position[1] = (REAL) i2;
  object_position[2] = (REAL) i3;
  object_position[3] = 1.0;
  E = (REAL) i4;
  focus = (REAL) i5;

  free(buffer);

  send_data(camera_socket, OBJECT_RECEIVED, (unsigned short) 0, (char *) NULL);
}

  

/*
 * Routines for communication via the robot channel.
 */
unsigned short decode_joints(char *buffer, int length,
	joint_values new_q, joint_values new_dq, joint_values new_ddq)
{
  float j[NR_LINKS+1], dj[NR_LINKS+1], ddj[NR_LINKS+1];
  char *p_buffer, check_reach;
  unsigned short i, chunk;
  joint_values tmp_q, tmp_dq, tmp_ddq;
  int fields;
  unsigned short error_joint;
  REAL min_joint, max_joint;

  /*
   * The first byte indicates if we should check on
   * illegal joint values.
   */
  check_reach = buffer[0];

  chunk = length / NR_LINKS;
  p_buffer = buffer+1;
  for (i=1; i<=NR_LINKS; i++, p_buffer += chunk)
  {
	if (decode(p_buffer, chunk, "%f%f%f", &j[i], &dj[i], &ddj[i]) != 3)
	{
		fprintf(stderr, "decode_joints failed for joint %d\n", i);
		panic_exit();
	}
	tmp_q[i] = (REAL) j[i];
  }

  free(buffer);

  if (check_reach==CHECK_REACH)
  {
	joint_values test_q, test_dq, test_ddq;

	if (error_joint = check_limits(tmp_q, &min_joint, &max_joint))
  	{
		char error_joint_buf[100];
		unsigned short length;

		length = encode(error_joint_buf, "%d%f%f%f", error_joint,
				tmp_q[error_joint], max_joint, min_joint);
		send_data(robot_socket, NEWPOS_BAD, length, error_joint_buf);
		return error_joint;
  	}

	/*
	 * Check if the end-effector position is legal, too.
	 */
	for (i=1; i<=NR_LINKS; i++)
		tmp_dq[i] = tmp_ddq[i] = 0.0;
	decouple_joints(test_q, test_dq, test_ddq, tmp_q, tmp_dq, tmp_ddq);
	forward_kinematics(test_q);
	if (check_end_effector(joint_homo[NR_LINKS-1][3],
        	joint_homo[NR_LINKS-1][7],
        	joint_homo[NR_LINKS-1][11]))
	{
		char error_joint_buf[100];
		unsigned short length;

		length = encode(error_joint_buf, "%d%f%f%f", 0, 0.0, 0.0);
		send_data(robot_socket, NEWPOS_BAD, length, error_joint_buf);
		return error_joint;
  	}
  }


  for (i=1; i<=NR_LINKS; i++)
  {
	new_q[i] = (REAL) j[i];
	new_dq[i] = (REAL) dj[i];
	new_ddq[i] = (REAL) ddj[i];
  }

  send_data(robot_socket, NEWPOS_OK, (unsigned short) 0, (char *) NULL);
  return (unsigned short) 0;
}


void send_joints(joint_values q, joint_values dq, joint_values ddq)
{
  char *p_buffer, buffer[BUF_SIZE];
  unsigned short length, i, chunk;

  p_buffer = buffer;
  length = 0;
  for (i=1; i<=NR_LINKS; i++)
  {
	chunk = encode(p_buffer, "%f%f%f", (float) q[i], (float) dq[i],
								(float) ddq[i]);
	length += chunk;
	if (length > BUF_SIZE)
	{
		fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
		panic_exit();
	}

	p_buffer += chunk;
  }

  send_data(robot_socket, SENT_JOINTS, length, buffer);
}



/************************************************************************/
/* Lower level communication routines.					*/
/************************************************************************/
void setup_server_communication(char *socketfile, int *control_socket,
						int *data_socket)
{
  char type;
  int socknr;
  FILE *sockfile;

  if ((sockfile = fopen(socketfile,"r")) == NULL)
  {
	fprintf(stderr, "can't open %s\n", socketfile);
	panic_exit();
  }
  fscanf(sockfile, "%d", &socknr);
  fclose(sockfile);

  printf("waiting at socket %d\n", socknr);

  /* Setup socket.  The number of the connection is
     socknr (see common.h).  The socket line is
     in control_socket, whereas communication goes over the
     file descriptor data_socket.  The last parameter
     is the maximum number of clients that is allowed
     to connect.
   */
  if (setup_server_socket(socknr, control_socket, 1) == -1)
	panic_exit();

  /* Now for each client accept the connection.  In this case,
     we only have one client and have to accept only once.
   */
  accept_client(*control_socket, data_socket);

  printf("client accepted\n");

  /* set up connection with client wait for alive char */
  send_data(*data_socket, START, 0, (char *) NULL);
  do
  {
	int status;
	char *buffer;

	type = START_ERROR;
	status = receive_data(*data_socket, &buffer, &type);
	if (status == -1) continue;
	if (status > 0)
	{
		fprintf(stderr, "received illegal message\n");
		panic_exit();
	}

	if (type == START_ERROR)
	{
		fprintf(stderr, "client says it can't start\n");
		panic_exit();
	}
	if (type != START_OK)
		fprintf(stderr, "received wrong message from client\n");
  } while (type != START_OK);

  printf("client accepts start\n");
}



void setup_client_communication(char *socket_file, char *host,
						int *socket_channel)
{
  char *buf, type;
  FILE *socket_fds;
  int socknr;

  if ((socket_fds = fopen(socket_file, "r")) == NULL)
  {
	perror(socket_file);
	panic_exit();
  }
  fscanf(socket_fds, "%d", &socknr);
  fclose(socket_fds);

  printf("socketnr = %d for %s\n", socknr, host);
  if (setup_client_socket(socknr, socket_channel, host, 1) == -1)
	panic_exit();

  do {
	int status;

	status = receive_data(*socket_channel, &buf, &type);
	if (status == -1) continue;
	if (status > 0)
	{
		fprintf(stderr, "received illegal message\n");
		panic_exit();
	}
  } while (type != START);

  printf("received START\n");

  send_data(*socket_channel, START_OK, 0, NULL);
}



void send_data(int socket_channel, char code, unsigned short length, char *data)
{
  char *buf;
  unsigned short i;
  int status, bytes_to_send;

  if ((buf = (char *) malloc(length+3)) == NULL)
  {
	fprintf(stderr, "send_data(): out of buffer space\n");
	panic_exit();
  }

  buf[0] = code;
  us_encode(length, buf+1);

  if (data != NULL)
	for (i=0; i<length; i++) buf[i+3] = data[i];

  bytes_to_send = (int)length+3;
  while (1)
  {
	do
		status = send(socket_channel, buf + (int) length + 3 -
					bytes_to_send, bytes_to_send, 0);
	while (status == -1 && errno == EWOULDBLOCK);

	if (status == -1)
	{
		perror("send_data()");
		panic_exit();
	}
	if (status == bytes_to_send)
		break;

	fprintf(stderr, "couldn't send all bytes (sent %d of %d)\n",
				(int)status, bytes_to_send);
	bytes_to_send -= status;
  }

  free(buf);
}



/*
 * In case the recv() system call cannot be interrupted,
 * the select() statement must be used in a while-loop.
 * A non-blocking read is implemented in this function.
 */
int receive_using_select(int sock, char *buffer, int length)
{
  int received;
  fd_set fdset, fdcopy;
  struct timeval timeout;

  FD_ZERO(&fdset);
  FD_SET(sock, &fdset);

  /*
   * Wait for the minimum possible time: 0 seconds.
   */
  timeout.tv_sec = 0;
  timeout.tv_usec = 0;

  fdcopy = fdset;
  if (select(GETDTABLESIZE, &fdcopy, (fd_set *) NULL, (fd_set *) NULL,
							&timeout) == 0)
	/*
	 * No data available.
	 */
	return -1;

  received = recv(sock, buffer, length, 0);

  if (received == length) return 0;
  if (received == -1 && errno != EWOULDBLOCK)
  {
	perror("receive");
	panic_exit();
  }

  return 0;
}



/*
 * This function receives bytes from the sparc using a blocking recv().
 */
int receive(int sock, char *buf, int length)
{
  int rec;

  if ((rec = recv(sock, buf, length, 0)) != length)
  {
	fprintf(stderr, "Received %d bytes too few\n", length-rec);
	panic_exit();
  }

  return 0;
}


/*
 * Must either call receive() or receive_using_select().
 */
int receive_data(int socket_channel, char **buffer, char *type)
{
  unsigned short message_len, us_decode();
  char buf[3];

  /* Read the header of the message. */
  if (receive_using_select(socket_channel, buf, 3)) return -1;

  *type = buf[0];
  message_len = us_decode(buf+1);

  /*
   * Read the rest of the message.
   */
  if (buffer != (char **) NULL)
	*buffer = NULL;
  if (message_len > 0)
  {
	if (buffer == (char **) NULL)
	{
		fprintf(stderr, "receive_data(): message too long\n");
		panic_exit();
	}

	*buffer = (char *) malloc(message_len+1);
	if (*buffer == NULL)
	{
		fprintf(stderr, "receive_data(): out of buffer space\n");
		panic_exit();
	}

	if (receive_using_select(socket_channel, *buffer, message_len))
	{
		fprintf(stderr, "receive_data() expected more data\n");
		panic_exit();
	}
  }

  return (int) message_len;
}
