/**************************************************************************/
/* data_code.c                                                  /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"


#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>

#include "data_code.h"
#include <stdarg.h>


void panic(char *fmt, ...)
{
  va_list arg_pointer;

  va_start(arg_pointer, fmt);
  vfprintf(stderr, fmt, arg_pointer);
  va_end(arg_pointer);
  exit(1);
}


/****************************************************************
 ** Conversion of unsigned short integers.                     **
 ****************************************************************/
void us_encode(unsigned short val, char *buf)
{
  buf[0] = (val >> CHAR_BIT) & UCHAR_MAX;
  buf[1] = val & UCHAR_MAX;
}



unsigned short int us_decode(char *buf)
{
  return ((( (unsigned) buf[0] ) & UCHAR_MAX) << CHAR_BIT) |
	  (( (unsigned) buf[1] ) & UCHAR_MAX);
}


/****************************************************************
 ** Conversion of unsigned long integers.                      **
 ****************************************************************/
void ul_encode(unsigned long val, char *buf)
{
  buf[0] = (val >> (CHAR_BIT*3)) & UCHAR_MAX;
  buf[1] = (val >> (CHAR_BIT*2)) & UCHAR_MAX;
  buf[2] = (val >> CHAR_BIT) & UCHAR_MAX;
  buf[3] = val & UCHAR_MAX;
}


unsigned long int ul_decode(char *buf)
{
  return ((( (unsigned long) buf[0] ) & UCHAR_MAX) << (CHAR_BIT*3)) |
	 ((( (unsigned long) buf[1] ) & UCHAR_MAX) << (CHAR_BIT*2)) |
	 ((( (unsigned long) buf[2] ) & UCHAR_MAX) << CHAR_BIT) |
	  (( (unsigned long) buf[3] ) & UCHAR_MAX);
}


/****************************************************************
 ** Conversion of signed short integers.                       **
 ** See data_code.h.                                           **
 ****************************************************************/



/****************************************************************
 ** Conversion of signed long integers.                        **
 ** See data_code.h.                                           **
 ****************************************************************/



/****************************************************************
 ** Conversion of floats and doubles.                          **
 ****************************************************************/
void f_encode(double x, char *buf, int precision)
{
  double mantissa, pilot;
  int sign, exp, i, j;

  if (precision < 2)
  {
	fprintf(stderr, "f_encode(): invalid precision field (%d)\n", precision);
	exit(1);
  }

  if (x < 0.0)
  {
	sign = 1 << (CHAR_BIT-1);
	x = -x;
  } else
	sign = 0;

  mantissa = frexp(x, &exp);

  if (exp < DBL_EXP_MIN || exp > DBL_EXP_MAX)
  {
	fprintf(stderr, "f_encode(): Floating point out of range\n");
	exit(1);
  }
  exp -= DBL_EXP_MIN;			/* make exp >= 0 */
  buf[0] = sign | ((exp >> CHAR_BIT) & SCHAR_MAX);
  buf[1] = exp & UCHAR_MAX;

  pilot = 1.0;
  for (j=2; j<=precision+1; j++)
  {
	buf[j] = 0;
	for (i=0; i<CHAR_BIT; i++)
	{
		pilot /= 2.0;
		if (mantissa - pilot > 0.0)
		{
			mantissa -= pilot;
			buf[j] |= 1<<i;
		}
	}
  }
}


double f_decode(char *buf, int precision)
{
  double mantissa, pilot;
  int exp, i, j, sign;

  if (precision < 2)
  {
	fprintf(stderr,"f_decode(): invalid precision field (%d)\n", precision);
	exit(1);
  }

  exp = ((buf[0] & SCHAR_MAX) << CHAR_BIT) +
	(((unsigned) buf[1]) & UCHAR_MAX)  + DBL_EXP_MIN;

  mantissa = 0.0;
  pilot = 1.0;
  for (j=2; j<=precision+1; j++)
	for (i=0; i<CHAR_BIT; i++) {
		pilot /= 2.0;
		if (buf[j] & (1<<i)) mantissa += pilot;
	}

  sign = (buf[0] & (1 << (CHAR_BIT-1)) ? -1 : 1);
  return (double) sign * ldexp(mantissa, exp);
}






/* In var arg C, arguments that are char or short are converted
 * to int and should be accessed as int, arguments that are
 * unsigned char or unsigned short are converted to unsigned int
 * and should be accessed as unsigned int, and arguments that
 * are float are converted to double and should be accessed as
 * double.
 *
 * Even though a good compiler should automatically translate
 * my unsigned shorts to ints, etc., not all of them do.
 * This is a fire-sure method.
 */
#define CHAR	int
#define UCHAR	unsigned int
#define SHORT	int
#define USHORT	unsigned int
#define LONG	long
#define ULONG	unsigned long
#define FLOAT	double
#define DOUBLE	double




unsigned short encode(char *buf, ...)
{
  va_list arg_pointer;
  char *fmt, *start;
  int process = 0,		/* is set after a `%' sign */
      is_short = 1,		/* by default shorts and floats */
      precision,
      ddp = 7,			/* default double precision */
      dfp = 3;			/* default float precision */
  unsigned short length;	/* returned length of buffer */

  va_start(arg_pointer, buf);

  start = buf;
  fmt = va_arg(arg_pointer, char *);

  while (*fmt) {
	switch(*fmt) {
	case '%':
		if (process) {
			*buf++ = *fmt; process = 0;
		} else {
			process = 1;
			is_short = 1;
			precision = 0;
		}
		break;

	case 'c':
		if (process) {
			*buf++ = (char) va_arg(arg_pointer, CHAR);
			process = 0;
		} else
			*buf++ = *fmt;
		break;

	case 's':
		if (process) {
			char *s;
			int len;
			s = va_arg(arg_pointer, char *);
			len = strlen(s);
			if (precision == 0 || precision > len)
				precision = len;
			strncpy(buf, s, precision);
			buf += precision;
			process = 0;
		} else
			*buf++ = *fmt;
		break;

	case '#':
		if (process)
			precision = va_arg(arg_pointer, int);
		else
			*buf++ = *fmt;
		break;

	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
		if (process) {
			precision = *fmt - '0';
			while (isdigit(*(fmt+1)))
				precision = precision*10 + *++fmt - '0';
		} else
			*buf++ = *fmt;
		break;

	case 'l':
		if (process)
			is_short = 0;
		else
			*buf++ = *fmt;
		break;

	case 'h':
		if (process)
			is_short = 1;
		else
			*buf++ = *fmt;

	case 'u': case 'd': case 'i':
		if (process) {
			*buf++ = '\0';
			if (is_short) {
				if (*fmt == 'u')
					us_encode((unsigned short)
						va_arg(arg_pointer, USHORT),
						buf);
				else
					ss_encode((short) va_arg(arg_pointer,
						SHORT), buf);
				buf += 2;
			} else {
				if (*fmt == 'u')
					ul_encode((unsigned long)
						va_arg(arg_pointer, ULONG),
						buf);
				else
					sl_encode((long) va_arg(arg_pointer,
						LONG), buf);
				buf += 4;
			}
			process = 0;
		} else
			*buf++ = *fmt;
		break;

	case 'f':
		if (process) {
			*buf++ = '\0';
			if (precision == 0) precision = is_short ? dfp : ddp;
			f_encode((double) va_arg(arg_pointer, DOUBLE),
							buf, precision);
			buf += precision+2;
			process = 0;
		} else
			*buf++ = *fmt;
		break;

	default:
		*buf++ = *fmt;
 	}
	fmt++;
  }
  *buf++ = '\0';

  va_end(arg_pointer);

  length = (unsigned short) (buf - start);
  return length;
}



#define WS(s)	while (isspace(*s)) s++

int decode(char *buf, ...)
{
  va_list arg_pointer;
  char *fmt, *end;
  int process = 0,		/* is set after a `%' sign */
      is_short = 1,		/* by default shorts */
      precision,
      ddp = 7,			/* default double precision */
      dfp = 3,			/* default float precision */
      dsp = 1,			/* default string precision */
      fields = 0;		/* # assigned fields */
  unsigned short length;	/* check if length of input buffer */

  va_start(arg_pointer, buf);

  length = (unsigned short) va_arg(arg_pointer, USHORT);
  end = buf+length;
  fmt = va_arg(arg_pointer, char *);

  while (*fmt) {
	switch(*fmt) {
	case ' ': case '\t':
		break;

	case '%':
		if (process) {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
			process = 0;
		} else {
			process = 1;
			is_short = 1;
			precision = 0;
		}
		break;

	case '*':
		if (process)
			process = 2;
		else {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
		}
		break;

	case '#':
		if (process)
			precision = va_arg(arg_pointer, int);
		else
			*buf++ = *fmt;

	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
		if (process) {
			precision = *fmt - '0';
			while (isdigit(*(fmt+1)))
				precision = precision*10 + *++fmt - '0';
		} else {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
		}
		break;

	case 'c':
		if (process==1) {
			int i;
			char *s;

			s = va_arg(arg_pointer, char *);
			if (precision == 0) precision = dsp;
			for (i=0; i<precision; i++)
				s[i] = *buf++;
			fields++;
			process = 0;
		} else if (process==2) {
			buf += (precision==0 ? dsp : precision);
			process = 0;
		} else {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
		}
		break;

	case 's':
		WS(buf);
		if (process==1) {
			char *s;
			int i;

			s = va_arg(arg_pointer, char *);

			if (precision == 0) precision = -1;
			for (i=0; i!=precision; i++) {
				if (isspace(*buf) || *buf=='\0') break;
				s[i] = *buf++;
			}
			s[i] = '\0';
			fields++;
			process = 0;
		} else if (process==2) {
			int i;
			if (precision == 0) precision = -1;
			for (i=0; i!=precision; i++) {
				if (isspace(*buf) || *buf=='\0') break;
				*buf++;
			}
			process = 0;
		} else
			if (*buf++ != *fmt) goto exit;
		break;

	case 'l':
		if (process)
			is_short = 0;
		else {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
		}
		break;

	case 'h':
		if (process)
			is_short = 1;
		else {
			WS(buf);
			if (*buf++ != *fmt) goto exit;
		}

	case 'u': case 'd': case 'i':
		WS(buf);
		if (process) {
			if (*buf++) goto exit;
			if (process == 1) if (is_short) {
				if (*fmt == 'u')
					*(va_arg(arg_pointer, unsigned short
						*)) = us_decode(buf);
				else
					*(va_arg(arg_pointer, short *)) =
						ss_decode(buf);
				fields++;
			} else {
				if (*fmt == 'u')
					*(va_arg(arg_pointer, unsigned long *))
						= ul_decode(buf);
				else
					*(va_arg(arg_pointer, long *))
						= sl_decode(buf);
				fields++;
			}
			buf += (is_short ? 2 : 4);
			process = 0;
		} else
			if (*buf++ != *fmt) goto exit;
		break;

	case 'f':
		WS(buf);
		if (process) {
			if (*buf++) goto exit;
			if (precision == 0) precision = is_short ? dfp : ddp;
			if (process==1) {
				if (is_short)
					*(va_arg(arg_pointer, float *)) =
					(float) f_decode(buf, precision);
				else
					*(va_arg(arg_pointer, double *)) =
					f_decode(buf, precision);
				fields++;
			}
			buf += precision+2;
			process = 0;
		} else
			if (*buf++ != *fmt) goto exit;
		break;

	default:
		WS(buf);
		if (*buf++ != *fmt) goto exit;
	}
	fmt++;
	if (buf >= end) goto exit;
  }

  va_end(arg_pointer);

exit:
  return fields;
}
