/*
 * This include file contains definitions of spatial constructs
 *    used in sensory interpretation and map manipulation
 *
 */


#include <sys/types.h>

/*
 * positions and points
 *
 */

typedef enum
{
  PT_INVALID, PT_POS, PT_WALL, PT_CORRIDOR
  } pt_type;

typedef enum
{
  SYSTEM, PERCEPT
  } cat_type;


typedef struct
{
  float x;			/* x,y position of point relative to flakey */
  float y;
  float th;			/* point orientation */
  pt_type type;			/* type of point */
  cat_type cat;			/* category */
} point;

extern float
point_phi(point *p);		/* returns the direction to the point */

extern void
cc_point(point *from, point *to); /* carbon-copy one point to another */

extern point *			/* holds list of points to be updated */
pointlist[];

extern int			/* pointer to last entry */
pointlist_p;

extern void
add_point(point *p);

extern void
rem_point(point *p);

point *
find_cp();			/* returns the first control point in pointlist */


/* wall point, with added fields */

typedef struct
{
  float x;			/* x,y position of point relative to flakey */
  float y;
  float th;			/* point orientation */
  pt_type type;			/* type of point */
  cat_type cat;			/* category */
  boolean new;			/* whether we just found it */
  boolean viewable;		/* whether it's valid */
  float length;			/* length of wall segment */
} wall;


/*
 * rectangular regions
 * usually represented as lower-left point, upper-right point
 */
typedef struct
{
  float x1, y1;
  float x2, y2;
} rect;


/*
 *
 *  Depth buffers
 *
 */

#define DBUF_MAX 128		/* size of buffers */

typedef struct
{
  int x, y, th;			/* where we are */
  int num;			/* number of entries */
  int slice;			/* for stereo slice algorithms */
  unsigned char buf[DBUF_MAX];	/* raw readings */
  float xbuf[DBUF_MAX];		/* array of rw coords */
  float ybuf[DBUF_MAX];		/* array of rw coords */
  float zbuf[DBUF_MAX];		/* array of rw coords */
  int grad[DBUF_MAX];		/* gradient */
  int lap[DBUF_MAX];		/* laplacian */
  int edge[DBUF_MAX];		/* edge threshold */
} dbuf;

extern dbuf left_side, right_side; /* for sonars */

extern void			/* convolve inarr (n) and arr (m) to outarr */
conv(int *arr,int m,unsigned char *inarr,int *outarr,int n);


extern void			/* gap is length to bridge, null is null value */
				/* slope is abs max value of slope to accept */
medval(unsigned char *arr,int n,int gap,float slope,int null);


extern void
find_breaks(unsigned char *arr,int *outarr,int n,int gap,int num);

extern void
find_inflections(int *edge, int *lap, int n, int th);

extern int lap_conv_arr_n;
extern int lap_conv_arr[];


/*
 * surface structures
 *
 */

typedef struct			/* surface segment */
{
  float x1, y1;			/* first coord */
  float x2, y2;			/* second coord */
  float cx, cy;			/* center */
  float d;			/* length */
} surf;

typedef struct			/* surface object */
{
  int t1, t2;			/* left and right break types */
  int tag;			/* for reference */
  int left, right;		/* connections on left and right */
  int used;			/* 0 if free, 1 if used */
  surf surface;			/* associated surface description */
} sobj;


#define SOBJ_MAX 25		/* max number of surface objects */
extern sobj sobj_buf[];		/* surface object buffer */
extern int sobj_freep;		/* free pointer */

void				/* initialize sobj buffer */
init_sobjs();

sobj *
get_sobj();			/* get next sobj from freelist */

void
free_sobj(sobj *s);		/* free it up */

typedef struct			/* sobj list */
{
  int n;			/* count of entries */
  sobj **s;			/* array of pointers */
} sobj_list;

extern sobj_list sobj_r, sobj_l; /* left and right side segments */

typedef enum			/* left, right or front side */
{ LEFT, RIGHT, FRONT }
side;

extern void			
seg_to_point(surf *s, point *p); /* convert from surface segment to point */

extern void
point_to_seg(point *p, float d, surf *s); /* convert from point to surface segment */


/* 
 * Some drawing functions
 *
 */

extern void
draw_seg(sobj *s,int w);	/* draw the segment on display */
				/* w is 0 for normal width, 1 for thick */
extern void
draw_surf(surf *ss,int w);	/* draw the segment on display */

extern void
draw_segs(sobj_list *segl,int w); /* draw a set of segments */
