/*#########################################
 * robot.c
 *
 * Robot definitions for the Erratic simulator.
 *
 *#########################################
 */

#include "structures.h"
#include "constants.h"
#include "parameters.h"

extern double drand48();	/* this function doesn't seem to return a double in SunOS */

/* Erratic robot structure */

simrobot erratic;

void setup_sim_robot()		/* setup of the simulated robot */
{
  erratic.rad = ERRATIC_RAD;	/* radius, in mm */

  erratic.x = 0.0;
  erratic.y = 0.0;
  erratic.th = 0.0;
  erratic.dx = 0.0;
  erratic.dy = 0.0;
  erratic.dth = 0.0;
  erratic.tv = 0.0;
  erratic.rv = 0.0;
  erratic.mtv = ATV_MAX;
  erratic.mrv = ARV_MAX;
  erratic.mta = ATA_MAX;
  erratic.mra = ARA_MAX;
  erratic.atv = ATV_MAX;
  erratic.arv = ARV_MAX;
  erratic.ata = ATA_MAX;
  erratic.ara = ARA_MAX;
  erratic.bumpmov = 0;
  erratic.bumper = 0.0;
  reset_servo_trans();
  reset_servo_rot();
  srand48(1278543L);		/* random function initialization */

  world.x = 0; world.y = 0; world.local = 2500.0;
  cur_seg_list = NULL;

}


void robot_set_origin()		/* reset encoder integrators */
{
  erratic.dx = 0.0;
  erratic.dy = 0.0;
  erratic.dth = 0.0;
}

void print_robot()		/* prints out robot params */
{
  printf("\n Erratic parameters: \n     x: %5f   y: %5f   th: %5f",
	 erratic.x, erratic.y, ang_to_deg(erratic.th));
  printf("\n     tv: %5f   rv: %5f   mtv: %5f   mrv: %5f",
	 erratic.tv, erratic.rv, erratic.mtv, erratic.mrv); 
  printf("\n     bump: %d   dir: %5f",
	 erratic.bumpmov, ang_to_deg(erratic.bumper));
}

/* simulation timing and robot movement */

float inc_time = .10;		/* what part of a second each incr uses */
int last_move = 0;		/* last movement command */
int max_last_move = 50;		/* number of null cycles before halting */

/* randomness parameters */

float encode_jitter = .01;	/* 1 percent randomness in wheel encoder */
float angle_jitter = .02;	/* 2 percent randomness in angle */
float dist_to_ang_jitter = 100.0; /* how long to move before giving differential jitter */
float angle_drift = .003;	/* drift angle */

/* some randomization functions */

float rand_norm = 1.0 / ((float)(1<<15) * (float)(1<<16)); /* using lrand48; drand48 doesn't work */

float randy(float r,float j)	/* randomize r by a fraction j of its value */
{
  j = (float)(j * ABS(r) * ((float)lrand48() * rand_norm - 0.5));
  return(r+j);
}


void move_robot()		/* makes ERRATIC move one time step */
{

  /* Integrators give precise coordinates, real ones are dithered */

  float d, da, r, sn, cs;
  static float ang_dist = 0.0;	/* distance we've moved forward/backwards */
  static float which = 1.0;	/* which way to dither angularly */
  last_move++;
  if (last_move > max_last_move) /* halt if we haven't received commands */
    { erratic.tv = 0.0; erratic.rv = 0.0; }
  d = erratic.tv * inc_time;	/* how far we've moved */

  sn = sin(erratic.dth);	/* update integrated coordinates */
  cs = cos(erratic.dth);
  erratic.dx += cs*d;
  erratic.dy += sn*d;


  sn = sin(erratic.th);	/* update absolute coordinates */
  cs = cos(erratic.th);
  erratic.x += randy(cs*d,encode_jitter); /* real world movement */
  erratic.y += randy(sn*d,encode_jitter); /* is different from commanded movement */

  ang_dist += ABS(d);
  da = 0;
  if (ang_dist > dist_to_ang_jitter)
    { 
      ang_dist = 0.0;
      if (randy(1.0,1.0) > 1.4) which = -1.0 * which; /* switch every now and then */
      da = randy(angle_drift,angle_drift) * which;
    }
  d = erratic.rv * inc_time;	/* how far we've turned */
  erratic.dth = add_ang(erratic.dth,d);	/* update integrated coordinates */
  erratic.th  = add_ang(erratic.th,randy(d,angle_jitter)+da);
}

/* jump the robot a bit... */

void
jump_robot(x,y,th)		/* move to this absolute position */
     float x, y, th;
{
  x -= erratic.x;		/* this is the change in position */
  y -= erratic.y;
  th -= erratic.th;
  erratic.x += x;
  erratic.y += y;
  erratic.th += th;
  erratic.dx += x;
  erratic.dy += y;
  erratic.dth += th;
}


/* check for collisions */

int collided()			/* If we've collided with an obstacle, set vel to 0 */
{
  float x,y;
  seg *s, *r;
  int i,j;

  x = erratic.x;		/* absolute coords */
  y = erratic.y;

  r = NULL;			/* check static segments for intersection */
  if (cur_seg_list != NULL)
    {
      for (i=0; i<cur_seg_list->n; i++)
	{
	  s = (seg *)(cur_seg_list->segs)[i];
	  if (within_dist_to_seg(x,y,s,erratic.rad))
	    { r = s; }
	}
    }

  if (r == NULL)		/* no collision */
    {
      erratic.bumpmov = 0;
      erratic.bumper  = 0.0;
      return(0);
    }
  else				/* we have collided */
    {
      if (erratic.bumpmov == 0) /* first touch */
	{
	  erratic.bumpmov = erratic.tv >= 0.0 ? 1 : -1; /* set direction */
	  erratic.tv = 0.0;
	  erratic.rv = 0.0;
	  reset_servo_trans();
	  reset_servo_rot();	
				/* set angle of collision */
	  erratic.bumper = sub_ang(angle_to_seg(erratic.x,erratic.y,r),
				    erratic.th);
	}
      else if (erratic.bumpmov == (erratic.tv >= 0.0 ? 1 : -1)) /* not backing off */
	{
	  erratic.tv = 0.0;
	  erratic.rv = 0.0;
	  reset_servo_trans();
	  reset_servo_rot();	
	  erratic.bumper = sub_ang(angle_to_seg(erratic.x,erratic.y,r),
				    erratic.th);
	}
      return(1);
    }
}


/* total update function */

void update_sim()		
{
  move_robot();			/* move the robot */
  servo();			/* check motor servoing parameters */
  set_current_segs();		/* update current world segments */
  collided();			/* check collisions */
  update_window();		/* draw the simulation */
}



