/* July 1990  Konolige
 *
 * Functions to display the state of Flakey's sensors in
 *   X windows.
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"
#include "display.h"

static char outstr[100];	/* temp buffer */
rotm flakey_matrix;		/* flakey to RW coords */
rotm flakey_screen_matrix;	/* flakey to screen coords */

/* draw  numeric info */

draw_num(n,d,x,y)		/* n is value, d is format descriptor */
     char *d;
{
  sprintf(outstr,d,n);
  put_text(x, y, outstr, strlen(outstr));
}

/* draw string info */

draw_str(s,x,y,l)		/* s is string, l length */
     char *s;
{
  int i;
  if (l>0)
    { i = 0;
      l = MIN(l,98);
      while ( (i < l) && s[i] )
	{ outstr[i] = s[i]; i++; }
      while ( i < l )
	{ outstr[i] = ' '; i++; }
      outstr[l] = 0;
      put_text(x, y, outstr, l);
    }
  else put_text(x, y, s, strlen(s));
}

message(s)			/* draw into the message window */
     char *s;
{
  draw_str(s,4,CHAR_HEIGHT,25);
  XFlush(display);
}

/* draw a batch of  numbers */

draw_nums(narr,n,xarr,yarr,d)
     int n;
     int xarr[], yarr[], narr[];
     char *d;

{
  int i;
  for (i = 0; i < n; i++)
    { sprintf(outstr,d,narr[i]);
      put_text(xarr[i], yarr[i], outstr, strlen(outstr));
    }
}
  

/* Functions to draw according to screen coords */

draw_sc_vec(x1,y1,x2,y2)	/* draws this vector, using screen coords */
{
  put_vector(x1,y1,x2,y2);
}

draw_sc_rect(x,y,dx,dy)		/* draws a rectangle using screen coords */
{
  put_box(x,y,ABS(dx),ABS(dy));
}


/* Functions to draw according to real-world coords */

draw_vec(float x1,float y1,float x2,float y2) /* draws this vector, using rw coords */
{
  float temp;
  temp = x1;
  x1 = xscale(x1,y1);
  y1 = yscale(temp,y1);
  temp = x2;
  x2 = xscale(x2,y2);
  y2 = yscale(temp,y2);
  put_vector(x1,y1,x2,y2);
}

draw_rect(float x,float y,float dx,float dy) /* draws a rectangle using rw coords */
{
  float temp;
  temp = x;
  x = (x * screen_matrix.a11) + (y * screen_matrix.a12) + screen_matrix.a13;
  y = (temp * screen_matrix.a21) + (y * screen_matrix.a22) + screen_matrix.a23;
  temp = dx;
  dx = dy * screen_matrix.a12;
  dy = temp * screen_matrix.a21;
  put_box(x,y,ABS(dx),ABS(dy));
}

draw_centered_rect(float cx,float cy,float h,float w) /* as in function occupied */
{
  float h2, w2;
  h2 = h * .5;
  w2 = w * .5;
  draw_rect(cx + h2, cy + w2, h, w);
}




/* draw directly on win (bypass pixmap) */

draw_vec_now(x1,y1,x2,y2)		
{
  int temp;
  temp = x1;
  x1 = xscale(x1,y1);
  y1 = yscale(temp,y1);
  temp = x2;
  x2 = xscale(x2,y2);
  y2 = yscale(temp,y2);
  XDrawLine(display, win, gc, x1, y1, x2, y2);
}

draw_rect_now(x,y,dx,dy)		
{
  int x1, y1, w, h;
  x1 = xscale(x,y);
  y1 = yscale(x,y);
  w = dxscale(dx,dy);
  h = dyscale(dx,dy);
  XDrawRectangle(display, win, gc, x1, y1, ABS(w), ABS(h));
}

/* functions to do general rotation and scaling */

c_rotm(rm,th,x,y)		/* sets the members of rm to calculated values */
     rotm *rm;			/* th in rads */
     float x, y, th;
{
  rm->a11 = cos((double)th);
  rm->a12 = sin((double)th);
  rm->a21 = - rm->a12;
  rm->a22 = rm->a11;
  rm->a13 = x;
  rm->a23 = y;
  rm->a31 = 0.0;
  rm->a32 = 0.0;
  rm->a33 = 1.0;
}

mult_rotm(rm1,rm2,rm3)		/* multiplies matrices rm1 and rm2 to produce rm3 */
     rotm *rm1,*rm2,*rm3;
{
  rm3->a11 = (rm1->a11 * rm2->a11) + (rm1->a12 * rm2->a21) + (rm1->a13 * rm2->a31);
  rm3->a21 = (rm1->a21 * rm2->a11) + (rm1->a22 * rm2->a21) + (rm1->a23 * rm2->a31);
  rm3->a31 = (rm1->a31 * rm2->a11) + (rm1->a32 * rm2->a21) + (rm1->a33 * rm2->a31);

  rm3->a12 = (rm1->a11 * rm2->a12) + (rm1->a12 * rm2->a22) + (rm1->a13 * rm2->a32);
  rm3->a22 = (rm1->a21 * rm2->a12) + (rm1->a22 * rm2->a22) + (rm1->a23 * rm2->a32);
  rm3->a32 = (rm1->a31 * rm2->a12) + (rm1->a32 * rm2->a22) + (rm1->a33 * rm2->a32);

  rm3->a13 = (rm1->a11 * rm2->a13) + (rm1->a12 * rm2->a23) + (rm1->a13 * rm2->a33);
  rm3->a23 = (rm1->a21 * rm2->a13) + (rm1->a22 * rm2->a23) + (rm1->a23 * rm2->a33);
  rm3->a33 = (rm1->a31 * rm2->a13) + (rm1->a32 * rm2->a23) + (rm1->a33 * rm2->a33);
}


draw_rm_vec(float x1,float y1,float x2,float y2,rotm *rm)
				/* draw according to full transform matrix rm */
{
  float temp;
  temp = x1;
  x1 = (x1 * rm->a11) + (y1 * rm->a12) + rm->a13;
  y1 = (temp * rm->a21) + (y1 * rm->a22) + rm->a23;
  temp = x2;
  x2 = (x2 * rm->a11) + (y2 * rm->a12) + rm->a13;
  y2 = (temp * rm->a21) +  (y2 * rm->a22) + rm->a23;
  put_vector(x1,y1,x2,y2);
}


draw_rm_rect(float x,float y,int dx,int dy,rotm *rm) 
				/* draws a rectangle using rw coords */
{
  float temp;
  temp = x;
  x = (x * rm->a11) + (y * rm->a12) + rm->a13;
  y = (temp * rm->a21) + (y * rm->a22) + rm->a23;
  put_box(x,y,dx,dy);
}


draw_rotated_vec(x1,y1,x2,y2,rm) /* rm is a rot matrix */
     rotm *rm;
{
  int temp;
  temp = x1;
  x1 = (x1 * rm->a11) + (y1 * rm->a12) + rm->a13;
  y1 = (temp * rm->a12) + rm->a23 - (y1 * rm->a11);
  temp = x2;
  x2 = (x2 * rm->a11) + (y2 * rm->a12) + rm->a13;
  y2 = (temp * rm->a12) + rm->a23 - (y2 * rm->a11);
  draw_vec(x1,y1,x2,y2);
}


draw_rot_vec(x1,y1,x2,y2,th,x,y) /* th,x,y are offset and angle */
     float x, y, th;
{
  rotm rm;
  c_rotm(&rm,th,x,y);		/* set up rotation matrix */
  draw_rotated_vec(x1,y1,x2,y2,rm);
}

/* arc drawing */

void
draw_rm_arc(float x,float y,float w,float h,float a1,float a2,rotm *rm)
     /* all RW coords, x,y center and w,h half-size */
     /* a1, a2 in degrees ccw */
{
  float tx;

  tx = x;
  x = (x * rm->a11) + (y * rm->a12) + rm->a13;
  y = (tx * rm->a21) + (y * rm->a22) + rm->a23;

  tx = w;
  w = dxscale(h,tx);
  h = dyscale(h,tx);
  w = ABS(w);
  h = ABS(h);
  put_arc(x-h,y-w,2*h,2*w,(a1 * 64),(a2 * 64));
}

/* draw Flakey, using x,y,th offset.  x,y are rw coords; th in rads */

draw_flakey()
{
  draw_fl_vec(-160.0 , 290.0 , 160.0 , 290.0);
  draw_fl_vec(290.0 , 160.0 , 290.0 , -160.0);
  draw_fl_vec(-160.0 , -290.0 , 160.0 , -290.0);
  draw_fl_vec(-290.0 , 160.0 , -290.0 , -160.0);
  draw_fl_vec(160.0 , -290.0 , 290.0 , -160.0);
  draw_fl_vec(-290.0 , -160.0 , -160.0 , -290.0);
  draw_fl_vec(160.0 , 290.0 , 290.0 , 160.0);
  draw_fl_vec(-290.0 , 160.0 , -160.0 , 290.0);
  draw_fl_vec(160.0 , -290.0 , 160.0 , 290.0);
}

#define RV_SCALE (-5.0 * RAD_TO_DEG) /* scale the rotation velocity vector */

draw_veloc()			/* draw the velocity components */
{
  draw_fl_vec(0.0, 0.0, flakey.tv, 0.0);
  draw_fl_vec(0.0, 0.0, 0.0, flakey.rv*RV_SCALE);
}

#define CX 1000

draw_control_pt(float c)	/* draws flake's heading control point */
 {
   if (c < 500.0)
     {
       float xc, yc;
       xc = CX * cos(c);
       yc = CX * sin(c);
       draw_fl_box(xc,yc,3,8);
     }
 }


/* sonar point drawing */

draw_sonar_pt(float x,float y,int n) /* x,y sonar reading; n size of box */
{
  draw_rw_box(x,y,n,n);
}



extern int idle_time;

draw_status()			/* puts the status of Flakey into the info window */
{
  char *s;
  static int count = 0;

  /* first column: status, translation and rotation velocity */
  switch(flakey.status)
    {
    case STATUS_NOT_CONNECTED:
      s = "not connected";
      break;
    case STATUS_NO_HIGH_POWER:
      s = "no high power";
      break;
    case STATUS_NOT_SERVOING:
      s = "not servoing";
      break;
    case STATUS_MOVING:
      s = "moving";
      break;
    case STATUS_STOPPED:
      s = "stopped";
      break;
    }
  draw_info_string(s, INFO_COL_1, 1, 13);
  draw_info_num((int)flakey.tv,"%3d ",INFO_COL_1,2);
  draw_info_num((int)(flakey.rv * RAD_TO_DEG),"%3d ",INFO_COL_1,3);
  
  /* second column: packet counts */
  if (++count > 9)
    {
      count = 0;
      draw_info_num((int)flakey.motor_packet_count,"%3d ",INFO_COL_2,1);
      flakey.motor_packet_count = 0;
      draw_info_num((int)flakey.sonar_packet_count,"%3d ",INFO_COL_2,2);
      flakey.sonar_packet_count = 0;
      draw_info_num((int)idle_time / 10,"%3d ",INFO_COL_2,3);
      idle_time = 0;
    }

  draw_control_pt(flakey.control);

}



reset_flakey_matrix()		/* reset all matrices to current flakey coords */
{
  c_rotm(&flakey_matrix,(- flakey.th),flakey.x,flakey.y);
  mult_rotm(&screen_matrix,&flakey_matrix,&flakey_screen_matrix); 
				/* compose with screen coords */
}



/* functions for control window */

#define bs_row(n) ((int)(BS_ROW*n))		  /* n-th row */
#define COL_1 160

static int col_0   = 1;		/* columns */
static int col_1   = COL_1;	
static int col_2   = COL_1 + 2*BS_BAR + 20;
static int col_3   = COL_1 + 4*BS_BAR + 40;

void
draw_b1(int row, float act, float turn, float acc)
{
  row = bs_row(row);		/* pixel value */
  draw_bs_bar(col_1,row,(int)(BS_BAR*act));
  draw_bs_bar(col_2,row,(int)(turn*BS_TR_SCALE));
  draw_bs_bar(col_3,row,(int)(acc*BS_AC_SCALE));
}

void				/* draw one of the behaviors */
draw_behavior_activity(beh_closure *b, int row)
{
  c_ret vals;
  vals = b->vals;
  draw_b1(row,b->activity,vals.turnv,vals.accv);
}

void
draw_behaviors(void)
{
  int i;
  clear_bs_area(col_1,0,b_width-col_1,b_height);
  for (i=0; i<MIN(BS_MAX,num_bcs); i++)
    {
      draw_behavior_activity(behaviors[i],i+1);
    }
  draw_b1(i+1,0.0,exec_turn,exec_acc);
  copy_bs_area(col_1,0,b_width-col_1,b_height);
}


/* 
 * global points
 */

draw_pointlist()
{
  int i;
  point *p;
  for (i=0;i<pointlist_p;i++)
    {
      p = pointlist[i];
      draw_rw_arc(p->x,p->y,60.0,60.0,0.0,360.0);
    }
}
