function gff = gfilterd(gf_b,gf_a)
%GFILTERD	Graphical filter design tool.
% 	[H] = GFILTERD opens the Graphical Filter Design Tool
%	allowing the user to interactively design a digital
%	filter by placing poles and zeros onto a polar plot
%	and viewing the transfer function.  Stability of the
%	designed filter can be tested by generating its impulse
%	response.  Poles and zeros can be placed on the polar
%	plot using rectangular or polar coordinates and with
%	the mouse pointer.  Once placed, poles and zeros may be
%	dragged to new locations.  Poles and zeros may also be
%	deleted using the mouse.
%
%	[H] = GFILTERD(B,A) opens GFILTERD with the initial
%	transfer function equal to the filter coefficients
%	specified by the arguments A and B.
%
%	[H] = GFILTERD(1,a) - Given only transfer function
%	denominator coefficients.
%
%	[H] = GFILTERD(b,1) - Given only transfer function
%	numerator coefficients.
%
%	The optional return argument H is the handle to the
%	GFILTERD figure window.

%       Dennis W. Brown 10-10-93, DWB 2-21-94
%       Copyright (c) 1994 by Dennis W. Brown
%       May be freely distributed.
%       Not for use in commercial products.

if nargin == 0,
    gf_a = 1; gf_b = 1;
elseif nargin ~= 2,
    error('gfilterd: Invalid number of input arguments...');
end;

% figure out if we have vectors
if min(size(gf_a)) ~= 1,
    error('gfilterd: Input arg "a" must be a 1xN or Nx1 vector.');
end;
if min(size(gf_b)) ~= 1,
    error('gfilterd: Input arg "b" must be a 1xN or Nx1 vector.');
end;

if nargin == 2,
	al = length(gf_a);
	bl = length(gf_b);
	if al > bl,
		gf_b = [gf_b zeros(1,al-bl)];
	elseif bl > al,
		gf_a = [gf_a zeros(1,bl-al)];
	end;
	index = find(gf_b ~= 0);
	gain = gf_b(index(1));
	index = find(gf_a ~= 0);
	gain = gain/gf_a(index(1));
	aa = roots(gf_a);
	bb = roots(gf_b);
else
	gain = 1;
	aa = roots(0);
	bb = roots(0);
end;

% -------------------------------------------------------------------------
% read colors and default position
spcolors;
global SPC_WINDOW SPC_TEXT_FORE SPC_TEXT_BACK SPC_AXIS SPC_MARKS SPC_GF_POS
global SPC_GFILT_AXIS

% open window to best size
s = get(0,'ScreenSize');
if exist('SPC_GF_POS'),
    gf_pos = SPC_GF_POS;
elseif s(3) < 800,
    gf_pos = [.1 0 .9 .9];
elseif s(3) >= 800 & s(3) < 1024,
    gf_pos = [.2 0 .8 .8];
elseif s(3) >= 1024,
    gf_pos = [.3 0 .7 .7];
end;
gf = figure('units','normal','position',gf_pos,'color',SPC_WINDOW,...
        'Name','Graphical Filter Design Tool by D.W. Brown',...
        'backingstore','off','NumberTitle','off');

% following line is bug workaround for version 4.1 and below
set(gcf,'BackingStore','off','BackingStore','on');

% make the axis
global SPC_FONTNAME;
polax = axes('Position',[.06 .5  .45 .45],'Color',SPC_AXIS);
polargrf(polax);
set(polax,'UserData','polar');


magax   = axes('Position',[.55 .775 .35  .16],'Color',SPC_AXIS,...
		'UserData','mag');
	set(magax,'Color',SPC_GFILT_AXIS,'FontName',SPC_FONTNAME,...
		'XColor',SPC_TEXT_FORE,'YColor',SPC_TEXT_FORE);
	set(get(magax,'ylabel'),'FontName',SPC_FONTNAME,...
		'Color',SPC_TEXT_FORE);
	title('Magnitude');
	set(get(magax,'title'),'FontName',SPC_FONTNAME,...
		'Color',SPC_TEXT_FORE);
	set(magax,'Box','on');
	grid;

phaax = axes('Position',[.55 .5  .35  .16],'Color',SPC_AXIS,...
		'UserData','phase');
	set(phaax,'Color',SPC_GFILT_AXIS,'FontName',SPC_FONTNAME,...
		'XColor',SPC_TEXT_FORE,'YColor',SPC_TEXT_FORE);
	title('Phase');
	set(get(phaax,'title'),'FontName',SPC_FONTNAME,...
		'Color',SPC_TEXT_FORE);
	xlabel('Frequency (1=fs/2)');
	set(get(phaax,'xlabel'),'FontName',SPC_FONTNAME,...
		'Color',SPC_TEXT_FORE);
	set(phaax,'Box','on');
	grid;

% setup so user can't plot over
set(gf,'NextPlot','new','Pointer','arrow');

% local variables
nbrcols = 4;
b_hite = .05; 	b_int = .015;
b_end = .05;	b_frame = .005;
b_width = (1 - (nbrcols-1)*b_int - 2*b_end)/nbrcols;
b_qtr = b_width/4;
columns = (0:nbrcols) .* (b_width + b_int) + .05;
rows = (0:10) .* (b_hite + b_int) + .05;

if strcmp(computer,'PCWIN'),
	set(gf,'MenuBar','none');
end;

% -------------------------------------------------------------------------

restorecall = [...
	'gf_a = get(findpush(gcf,''Save''),''UserData'');'...
	'gf_b = get(findpush(gcf,''Load''),''UserData'');'...
	'gain = get(findchkb(gcf,''Unwrap Phase''),''UserData'');'...
	'set(finduitx(gcf,''Variables''),''UserData'',gain);'...
	'set(finduitx(gcf,''a''),''UserData'',gf_a);'...
	'set(finduitx(gcf,''b''),''UserData'',gf_b);'...
	'clear gf_a gf_b; gfiltcal(''plotall'');'];

clearcall = [...
	'gf_a = 0;'...
	'gf_b = 0;'...
	'set(finduitx(gcf,''Variables''),''UserData'',1);'...
	'set(finduitx(gcf,''a''),''UserData'',roots(gf_a));'...
	'set(finduitx(gcf,''b''),''UserData'',roots(gf_b));'...
	'clear gf_a gf_b; gfiltcal(''plotall'');'];

killcurves = 'set(findchkb(gcf,''Keep Curves''),''Value'',0);';

m=uimenu('Label','Options');
  uimenu(m,'Label','Restore',...
	'Accelerator','Y',...
	'Callback',[killcurves restorecall]);
  uimenu(m,'Label','Clear',...
	'Accelerator','U',...
	'Callback',[killcurves clearcall]);
  uimenu(m,'Label','Impulse',...
	'Callback','gfiltcal(''impulse'');',...
	'Separator','on',...
	'Accelerator','I');
  uimenu(m,'Label','Print',...
	'Callback','gfiltcal(''print'');', ...
	'Separator','on',...
	'Accelerator','P');
  uimenu(m,'Label','Snapshot',...
	'Callback','gfiltcal(''snapshot'');',...
	'Accelerator','S');
  uimenu(m,'Label','Quit',...
	'Accelerator','Q',...
	'Callback','close(gcf);', ...
	'Separator','on');

% Scale menu
items = str2mat('Linear','Logrithmic');
togmenu(gf,'Scale',items,1,[killcurves ' gfiltcal(''plotall'');']);

% accelerator keys for scale menu
set(findmenu(gf,'Scale','Linear'),'Accelerator','W');
set(findmenu(gf,'Scale','Logrithmic'),'Accelerator','L');

% Angle menu
items = str2mat('Degrees','Radians');
togmenu(gf,'Angles',items,2,[killcurves ' gfiltcal(''plotall'');']);

% accelerator keys for Angles menu
set(findmenu(gf,'Angles','Degrees'),'Accelerator','D');
set(findmenu(gf,'Angles','Radians'),'Accelerator','R');

% Angle menu
items = str2mat('Move Zeros Only','Move Poles Only','Move Poles and Zeros');
togmenu(gf,'Phase',items,1,[' gfiltcal(''plotall'');']);

% accelerator keys for phase menu
set(findmenu(gf,'Phase','Move Zeros Only'),'Accelerator','O');
set(findmenu(gf,'Phase','Move Poles Only'),'Accelerator','X');
set(findmenu(gf,'Phase','Move Poles and Zeros'),'Accelerator','B');

% -------------------------------------------------------------------------
% mag/ang entry
gf_ma_col = 1;
gf_ma_row = 1;


% frame around group
h = uicontrol(gf,...
    'Style','frame','Units','normal',...
    'Position',[columns(gf_ma_col)-b_frame rows(gf_ma_row)-b_frame ...
        b_width+2*b_frame 3*b_hite+2*b_frame+2*b_int]);

	% get the frame's color so we can set our text background the same
	bcolor = get(h,'BackGroundColor');

	% if on a PC, set a special color for edit boxes so that they are
	% visible. otherwise, make same as frame cause that looks better on
	% Sun workstation ***Mathworks!***
	if strcmp(computer,'PCWIN'),
		beditcolor = [1 1 1] * 0.7;
	else,
		beditcolor = bcolor;
	end;

% print "A" label
uicontrol(gf,...
    'Style','text','Units','normal','Horiz','center',...
    'Foreground','black',...
    'Position',[columns(gf_ma_col) rows(gf_ma_row) b_width/8 b_hite],...
    'String','A'...
    );

% draw "A" user edit box
uicontrol(gf,...
    'Style','edit','Units','normal',...
    'BackGroundColor',beditcolor,'ForeGroundColor','black',...
    'Position',[columns(gf_ma_col)+b_width/8 rows(gf_ma_row) ...
		b_width*5/8 b_hite],...
    'String','','Visible','on','Userdata','angle','Value',1,...
    'UserData','A','CallBack',''...
    );

% print "M" label
uicontrol(gf,...
    'Style','text','Units','normal','Horiz','center',...
    'Foreground','black',...
    'Position',[columns(gf_ma_col) rows(gf_ma_row+1) ...
		b_width/8 b_hite],...
    'String','M'...
    );

% draw "M" user edit box
uicontrol(gf,...
    'Style','edit','Units','normal',...
    'BackGroundColor',beditcolor,'ForeGroundColor','black',...
    'Position',...
        [columns(gf_ma_col)+b_width/8 rows(gf_ma_row+1) ...
		b_width*5/8 b_hite],...
    'String','','Visible','on','Userdata','mag','Value',2,...
    'UserData','M','CallBack',''...
    );

% draw "A" pole pushbutton
uicontrol(gf,...
    'Style','pushbutton','Units','normal','Horiz','center',...
    'Position',...
        [columns(gf_ma_col)+b_width*6/8 rows(gf_ma_row) ...
		 b_width/8 2*b_hite+b_int],...
    'String','X','Value',1,...
    'CallBack','gfiltcal(''ma_pole'');'...
    );

% draw mag/ang zero pushbutton
uicontrol(gf,...
    'Style','pushbutton','Units','normal','Horiz','center',...
    'Position',...
        [columns(gf_ma_col)+b_width*7/8 rows(gf_ma_row) ...
		b_width/8 2*b_hite+b_int],...
    'String','0','Userdata','ma',...
    'CallBack','gfiltcal(''ma_zero'');'...
    );

% print "Polar Coord" label
uicontrol(gf,...
    'Style','text','Units','normal','Horiz','center',...
    'Foreground','black',...
    'Position',[columns(gf_ma_col) rows(gf_ma_row+2) b_width b_hite],...
    'String','Polar Coord'...
    );

% -------------------------------------------------------------------------
% real/imag entry
gf_ri_col = 1;
gf_ri_row = 4;

% frame around group
uicontrol(gf,...
    'Style','frame','Units','normal',...
    'Position',[columns(gf_ri_col)-b_frame rows(gf_ri_row)-b_frame ...
        b_width+2*b_frame 3*b_hite+2*b_frame+2*b_int]);

% print "I" label
uicontrol(gf,...
    'Style','text','Units','normal','Horiz','center',...
    'Foreground','black',...
    'Position',[columns(gf_ri_col) rows(gf_ri_row) b_width/8 b_hite],...
    'String','I'...
    );

% draw "I" user edit box
uicontrol(gf,...
	'Style','edit','Units','normal',...
	'BackGroundColor',beditcolor,'ForeGroundColor','black',...
	'UserData','I',...
	'Position',...
		[columns(gf_ri_col)+b_width/8 rows(gf_ri_row) b_width*5/8 b_hite]);

% print "R" label
uicontrol(gf,...
	'Style','text',...
	'Units','normal',...
	'Horiz','center',...
	'Foreground','black',...
	'Position',[columns(gf_ri_col) rows(gf_ri_row+1) b_width/8 b_hite],...
	'String','R');

% draw "R" user edit box
uicontrol(gf,...
	'Style','edit','Units','normal',...
	'ForeGroundColor','black',...
	'BackGroundColor',beditcolor,...
	'UserData','R',...
	'Position',...
        	[columns(gf_ri_col)+b_width/8 rows(gf_ri_row+1) b_width*5/8 b_hite]);

% draw rig/ang pole pushbutton
uicontrol(gf,...
	'Style','pushbutton',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(gf_ri_col)+b_width*6/8 rows(gf_ri_row)...
		b_width/8 2*b_hite+b_int],...
	'String','X',...
	'Value',2,...
	'CallBack','gfiltcal(''ri_pole'');');

% draw mag/ang zero pushbutton
uicontrol(gf,...
	'Style','pushbutton',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[rows(gf_ri_col)+b_width*7/8 rows(gf_ri_row) ...
			b_width/8 2*b_hite+b_int],...
	'String','0','Userdata','ri',...
	'CallBack','gfiltcal(''ri_zero'');');

% print "Rect Coord" label
uicontrol(gf,...
	'Style','text',...
	'Units','normal',...
	'Horiz','center',...
	'Foreground','black',...
	'Position',[columns(gf_ri_col) rows(gf_ri_row+2) b_width b_hite],...
	'String','Rect Coord');

% -------------------------------------------------------------------------
column_save = 4;
row_save = 1;

% frame around group
uicontrol(gf,...
	'Style','frame',...
	'Units','normal',...
	'Position',[columns(column_save)-b_frame rows(row_save)-b_frame ...
		b_width+2*b_frame 4*b_hite+2*b_frame+3*b_int]);

% print property label
uicontrol(gf,...
	'Style','text',...
	'Units','normal',...
	'Horiz','center',...
	'Foreground','black',...
	'UserData',gain,...
	'Position',[columns(column_save) rows(row_save+3) b_width b_hite],...
	'String','Variables');

% draw a user text box
uicontrol(gf,...
	'Style','text',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(column_save) rows(row_save+1) ...
		b_width/4 b_hite],...
	'String','a',...
	'Visible','on',...
	'UserData',aa);

% draw b user text box
uicontrol(gf,...
	'Style','text',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(column_save) rows(row_save+2) b_width/4 b_hite],...
	'String','b',...
	'Visible','on',...
	'UserData',bb);

% draw save pushbutton
uicontrol(gf,...
    'Style','pushbutton','Units','normal','Horiz','center',...
    'Position',[columns(column_save) rows(row_save) b_width/2 b_hite],...
    'String','Save','UserData',aa,...
    'CallBack','gfiltsv');

% draw load pushbutton
uicontrol(gf,...
    'Style','pushbutton','Units','normal','Horiz','center',...
    'Position',[columns(column_save)+b_width/2 rows(row_save) ...
		b_width/2 b_hite],...
    'String','Load','UserData',bb,...
    'CallBack','gfiltld');

% draw a user edit box
uicontrol(gf,...
	'Style','edit',...
	'Units','normal',...
	'BackGroundColor',beditcolor,...
	'ForeGroundColor','black',...
	'Position',[columns(column_save)+b_width/4 rows(row_save+1) ...
		3*b_width/4 b_hite],...
	'String','',...
	'Visible','on',...
	'UserData','a var');

% special color for PC so we can tell diff from a edit box
if strcmp(computer,'PCWIN'),
	special = [1 1 1] * 0.6;
else,
	special = beditcolor;
end;

uicontrol(gf,...
	'Style','edit',...
	'Units','normal',...
	'BackGroundColor',special,...
	'ForeGroundColor','black',...
	'Position',[columns(column_save)+b_width/4 ...
		rows(row_save+2) 3*b_width/4 b_hite],...
	'String','',...
	'Visible','on',...
	'UserData','b var');

% -------------------------------------------------------------------------
property = 2;
row_property = 1;

% frame around group
f = uicontrol(gf,...
	'Style','frame',...
	'Units','normal',...
	'Position',[columns(property)-b_frame rows(row_property)-b_frame ...
		b_width+2*b_frame 6*b_hite+2*b_frame+5*b_int]);

% print property label
uicontrol(gf,...
	'Style','text','Units',...
	'normal','Horiz','center',...
	'Foreground','black',...
	'Position',[columns(property) rows(row_property+5) b_width b_hite],...
	'String','With Mouse');

% handles of radio buttons (need for setting up togradio call)
handles = zeros(3,1);

% draw complex radiobutton
handles(1) = uicontrol(gf,...
	'Style','radiobutton',...
	'Units','normal',...
	'Horiz','center',...
	'BackGround',beditcolor,...
	'Position',[columns(property) rows(row_property+3) b_width b_hite],...
	'String','Complex',...
	'Value',1,...
	'UserData',f,...
	'CallBack','togradio;');

% draw real radiobutton
handles(2) = uicontrol(gf,...
	'Style','radiobutton',...
	'Units','normal',...
	'Horiz','center',...
	'BackGround',beditcolor,...
	'Position',[columns(property) rows(row_property+2) b_width b_hite],...
	'String','Real',...
	'Value',0,...
	'UserData',f,...
	'CallBack','togradio;');

% draw imaginary radiobutton
handles(3) = uicontrol(gf,...
	'Style','radiobutton',...
	'Units','normal',...
	'Horiz','center',...
	'BackGround',beditcolor,...
	'Position',[columns(property) rows(row_property+1) b_width b_hite],...
	'String','Imaginary',...
	'Value',0,...
	'UserData',f,...
	'CallBack','togradio;');

% set up for togradio call
set(f,'UserData',handles);

% draw mouse pole pushbutton
uicontrol(gf,...
	'Style','pushbutton',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(property) rows(row_property+4) b_width/2 b_hite],...
	'String','X','Value',3,...
	'CallBack','gfiltcal(''mouse_pole'');');

% draw mouse pole pushbutton
uicontrol(gf,...
	'Style','pushbutton',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(property)+b_width/2 rows(row_property+4) ...
		b_width/2 b_hite],...
	'String','0','Userdata','mo',...
	'CallBack','gfiltcal(''mouse_zero'');');

% draw delete pushbutton
uicontrol(gf,...
	'Style','pushbutton',...
	'Units','normal',...
	'Horiz','center',...
	'Position',[columns(property) rows(row_property) b_width b_hite],...
	'String','Delete',...
	'CallBack','gfiltcal(''delete'',''start'');');

% -------------------------------------------------------------------------
column_phase = 3;
row_phase = 1;

items = str2mat('Mixed','Minimize','Maximize');
radiogrp(gf,'Phase Condition',items,1,[columns(column_phase) rows(row_phase)],...
	[b_width b_hite b_frame b_int],'CallBack','gfiltcal(''plotall'');');
	
% -------------------------------------------------------------------------
% draw keep transfer function checkbox
uicontrol(gf,...
    'Style','checkbox','Units','normal','Horiz','center',...
    'Position',[columns(4) rows(5) b_width b_hite],...
    'String','Keep Curves',...
    'Value',0,'Callback',...
	['if ~get(findchkb(gcf,''Keep Curves''),''Value''),'...
		'[gf_c,gf_h] = cntlines(gca);'...
		'for gf_k = 1:gf_c, delete(gf_h(gf_k)); end; '...
		'gfiltcal(''plotall''); end; clear gf_k gf_c gf_h; ']);

% -------------------------------------------------------------------------
% draw unwrap phase checkbox
uicontrol(gf,...
    'Style','checkbox','Units','normal','Horiz','center',...
    'Position',[columns(3) rows(5) b_width b_hite],...
    'String','Unwrap Phase','UserData',gain,...
    'Value',0,'CallBack','gfiltcal(''phase'');');

% -------------------------------------------------------------------------
% draw normalize checkbox
uicontrol(gf,...
    'Style','checkbox','Units','normal','Horiz','center',...
    'Position',[columns(3) rows(6) b_width b_hite],...
    'String','Normalize',...
    'Value',1,'CallBack',['set(findchkb(gcf,''Keep Curves''),''Value'',0);'...
		'gfiltcal(''plotall'');']);

% -------------------------------------------------------------------------
% draw non causal checkbox
h = uicontrol(gf,...
    'Style','checkbox','Units','normal','Horiz','center',...
    'Position',[columns(4) rows(6) b_width b_hite],...
    'String','Non-Causal','Enable','on',...
    'Value',0,'CallBack','gfiltcal(''plotall'');');

if al >= bl,
	% causal system
	set(h,'Value',0,'Enable','off');
else
	set(h,'Value',1,'Enable','on');
end;
% -------------------------------------------------------------------------

% snapshot for magnitude axis
uicontrol(gf,'Style','push','Units','normal','Horiz','center',...
	'Position',[.91 .775 .03 .04],...
	'String','S','CallBack',['xxx_h=snapshot(findaxes(gcf,''mag''));'...
		'zoomtool(get(xxx_h,''CurrentAxes'')); clear xxx_h']);

% snapshot for phase axis
uicontrol(gf,'Style','push','Units','normal','Horiz','center',...
	'Position',[.91 .5 .03 .04],...
	'String','S','CallBack',['xxx_h=snapshot(findaxes(gcf,''phase''));'...
		'zoomtool(get(xxx_h,''CurrentAxes'')); clear xxx_h']);

gfiltcal('plotall');

if nargout == 1,
	gff = gf;
end;

