/*
 * $Id: tdat.c,v 1.42 1993/06/09 20:58:40 johans Exp $
 *
 * tdat.c
 *
 */

/* Standard C library include file directives */
#include <sys/types.h>
#include <netinet/in.h>
#ifdef _AIX
#include <net/nh.h>
#endif _AIX
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>

/* Speech library include file directives */
#include <speech.h>

#define READLINE_LENGTH		(1024)

/*
 * Interal data type codes
 */
#define BADTYPE		(0)
#define STRING		(1)
#define INTEGER		(2)
#define SHORT		(3)
#define FLOAT		(4)

/* external variable declaration */
extern int  errno;
extern int  sys_nerr;
extern char *sys_errlist[];


/* local module header declarations */
static char  **ReadData( FILE *fp, AsciiHdr *hdr );
static int   WriteData( FILE *fp, char **data, AsciiHdr *hdr );
static int   WriteFloats( FILE *fp, float **data, int rows, int cols );
static float **ReadFloats( FILE *fp, int rows, int cols );
static int   WriteShorts( FILE *fp, short **data, int rows, int cols );
static short **ReadShorts( FILE *fp, int rows, int cols );
static int   WriteIntegers( FILE *fp, int **data, int rows, int cols );
static int   **ReadIntegers( FILE *fp, int rows, int cols );
static int   StrToTypeCode( char *type );

#ifdef DEBUG
static void PrintHeader( AsciiHdr *hdr )
#endif DEBUG


/*
 * Tdatread (file, data, header)
 *
 * Reads in the dft data into the TDat data structure for
 * use in the Lyre Gspec tool
 *
 * file   (in): name of file (spectrogram)
 * data  (out): pointer to the spectrogram data
 * header (in): Ascii header, used for information retrieval
 *              concerning the dft data
 *
 */

int TdatRead( char *file, char ***data, AsciiHdr **header )
{
	FILE *fp;

	if( (fp = fopen( file, "r" ) ) == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fopen failed";
		}
		return( -1 );
	}

	if( fTdatRead( fp, data, header ) < 0 ) {
		fclose( fp );
		return( -1 );
	}

	fclose( fp );
	return( 1 );
}


int TdatWrite( char *file, char **data, AsciiHdr *header )
{
	FILE *fp;

	if( (fp = fopen( file, "w" ) ) == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fopen failed";
		}
		return( -1 );
	}

	if( fTdatWrite( fp, data, header ) < 0 ) {
		fclose( fp );
		return( -1 );
	}

	fclose( fp );
	return( 1 );
}


int fTdatRead( FILE *fp, char ***data, AsciiHdr **header ) 
{
	AsciiHdr *hdr;
	char **rd;

	hdr = ReadHeader( fp );

	if( hdr == NULL ) {
		*data = NULL;
		*header = NULL;
		return( -1 );
	}

	*header = hdr;

	rd = ReadData( fp, hdr );

	if( rd == NULL ) {
		FreeHeader( hdr );
		*header = NULL;
		*data = NULL;
		return( -1 );
	}

	*header = hdr;
	*data = rd;

#ifdef DEBUG
	PrintHeader( hdr );
#endif

	return( 1 );

}


int fTdatWrite( FILE *fp, char **data, AsciiHdr *header )
{

#ifdef DEBUG
	PrintHeader( header );
#endif 

	if( WriteHeader( fp, header ) < 0 ) {
		return( -1 );
	}
	
	if( WriteData( fp, data, header ) < 0) {
		return( -1 );
	}

	return( 1 );
}


/*
 * ReadData( fp, hdr )
 *
 *  Used by fTdatRead() to read in data part of
 *  Tdat structure.
 * 
 * parameters:
 *   fp - stream to read from
 *   hdr - pointer to header for the data
 *
 * returns:
 *   a pointer to a char * if sucessful, otherwise
 *   ErrorString is set and a NULL is returned.
 *
 */

static char **ReadData( FILE *fp, AsciiHdr *hdr )
{
	char **data;
	int cols;
	int rows;
	char *tmp;


	tmp = FindHeaderVal( hdr, "Rows" );
	
	if( tmp == NULL ) {
		ErrorString = "missing rows in header";
		return( NULL );
	}

	rows = atoi( tmp );

	tmp = FindHeaderVal( hdr, "Columns" );

	if( tmp == NULL ) {
		ErrorString = "missing columns in header";
		return( NULL );
	}

	cols = atoi( tmp );

	tmp = FindHeaderVal( hdr, "Type" );

	if( tmp == NULL ) {
		ErrorString = "missing type in header";
		return( NULL );
	}

	if( strcmp( tmp, "float" ) == 0 ) {

		data = (char **)ReadFloats( fp, rows, cols );

		if( data == NULL ) {
			return( NULL );
		}

	} else if( strcmp( tmp, "short" ) == 0 ) {

	  data = (char **)ReadShorts( fp, rows, cols );

		if( data == NULL ) {
			return( NULL );
		}

	} else if( strcmp( tmp, "integer" ) == 0 ) {

		data = (char **)ReadIntegers( fp, rows, cols );

		if( data == NULL ) {
			return( NULL );
		}

	} else {

		ErrorString = "unknow error in ReadData()";
		return( NULL );

	}

	return( data );
}


/*
 * WriteData( fp, data, hdr ) 
 *
 *  Used By fTdatWrite() to write out data part of
 *  Tdat structure.
 *
 * parameters:
 *  fp - stream to write to
 *  data - data to write out
 *  hdr - pointer to header for the data
 *
 * returns:
 *  1 - if ok
 *  -1 - on error and sets ErrorString
 */

static int WriteData( FILE *fp, char **data, AsciiHdr *hdr )
{
	char *tmp;
	int type;
	int cols;
	int rows;

	tmp = FindHeaderVal( hdr, "Rows" );
	
	if( tmp == NULL ) {
		ErrorString = "rows field missing in header";
		return( -1 );
	}
	rows = atoi( tmp );

	tmp = FindHeaderVal( hdr, "Columns" );
	
	if( tmp == NULL ) {
		ErrorString = "columns field missing in header";
		return( -1 );
	}
	cols = atoi( tmp );

	tmp = FindHeaderVal( hdr, "Type" );

	if( tmp == NULL ) {
		ErrorString = "type field missing in header";
		return( -1 );
	}

	type = StrToTypeCode( tmp );

	switch( type ) {
		case FLOAT:
			WriteFloats( fp, (float **)data, rows, cols );
			break;

		case INTEGER:
			WriteIntegers( fp, (int **)data, rows, cols );
			break;

		case SHORT:
			WriteShorts( fp, (short **)data, rows, cols );
			break;

		case BADTYPE:
			ErrorString = "unknown type";
			return( -1 );
	}

	return( 1 );
}


/*
 * WriteFloats( fp, data, rows, cols )
 *
 * used by WriteData() to write out float tpye data.
 *
 * parameters:
 *   fp - stream to write to
 *   data - data to write out
 *   rows - number of rows in the data
 *   cols - number of columns in the data
 *
 * returns:
 *   1 - if ok
 *   -1 - if write failed
 */

static int WriteFloats( FILE *fp, float **data, int rows, int cols )
{
	unsigned int hl, nl;
	int i,j;
	union {
		float f;
		unsigned int ul;
	} f_ul;


	for( i = 0; i < cols; i++ ) {
		for( j = 0; j < rows; j++ ) {

			f_ul.f = data[i][j];
			hl = f_ul.ul;
			nl = htonl( hl );

			if( fwrite( (char *)&nl, sizeof( nl ), 1, fp ) != 1 ) {
				if( errno < sys_nerr ) {
					ErrorString = sys_errlist[errno];
				} else {
					ErrorString = "write failed";
				}
				return( -1 );
			}

		}
	}

	return( 1 );
}


/*
 * ReadFloats( fp, rows, cols )
 *
 *  Used by ReadData() to read in float type data.
 *
 * parameters:
 *   fp - stream to read from
 *   rows - number of rows in data array
 *   cols - number of cols in data array
 * 
 * return:
 *   a poiner to an array of floats if sucessful, otherwise
 *   ErrorString is set and a NULL is returned.
 *
 */

static float **ReadFloats( FILE *fp, int rows, int cols )
{
	float **tmp;
	unsigned int netint;
	int count;
	int ret;
	int i, j;
	union {
		float f;
		unsigned int ul;
	} f_ul;

	tmp = (float **)Alloc2d( cols, rows, sizeof(float) );

	if( tmp == NULL ) {
		/* ErrorString set by Alloc2d */
		return( NULL );
	}

	for( i = 0; i < cols; i++ ) {

		for( j = 0; j < rows; j++ ) {
		  fread (&netint, sizeof(unsigned int), 1, fp);
		  f_ul.ul = ntohl(netint);
		  tmp[i][j] = f_ul.f;
		}
	}
	return( tmp );
}


/*
 * WriteShorts( fp, data, rows, cols )
 *
 * used by WriteData() to write out short type data.
 *
 * parameters:
 *   fp - stream to write to
 *   data - data to write out
 *   rows - number of rows in the data
 *   cols - number of columns in the data
 *
 * returns:
 *   1 - if ok
 *   -1 - if write failed
 */

static int WriteShorts( FILE *fp, short **data, int rows, int cols )
{
	unsigned short hl, nl;
	int i,j;


	for( i = 0; i < cols; i++ ) {
		for( j = 0; j < rows; j++ ) {

			hl = (unsigned short)data[i][j];
			nl = htons( hl );

			if( fwrite( (char *)&nl, sizeof( nl ), 1, fp ) != 1 ) {
				if( errno < sys_nerr ) {
					ErrorString = sys_errlist[errno];
				} else {
					ErrorString = "write failed";
				}
				return( -1 );
			}

		}
	}

	return( 1 );
}


/*
 * ReadShorts( fp, rows, cols )
 *
 *  Used by ReadData() to read in short type data.
 *
 * parameters:
 *   fp - stream to read from
 *   rows - number of rows in data array
 *   cols - number of cols in data array
 * 
 * return:
 *   a poiner to an array of shorts if sucessful, otherwise
 *   ErrorString is set and a NULL is returned.
 *
 */

static short **ReadShorts( FILE *fp, int rows, int cols )
{
	short **tmp;
	unsigned short netshort;
	int count;
	int ret, samples_read;
	int i, j;

	tmp = (short **)Alloc2d( cols, rows, sizeof(short) );

	if( tmp == NULL ) {
		/* Alloc2d set ErrorString */
		return( NULL );
	}

	for( i = 0; i < cols; i++ ) {
	  for( j = 0; j < rows; j++ ) {
	    fread(&netshort, sizeof(unsigned short), 1, fp);
	    tmp[i][j] = (short) ntohs(netshort);
	  }
	}

	return( tmp );
}


/*
 * WriteIntegers( fp, data, rows, cols )
 *
 * used by WriteData() to write out integer type data.
 *
 * parameters:
 *   fp - stream to write to
 *   data - data to write out
 *   rows - number of rows in the data
 *   cols - number of columns in the data
 *
 * returns:
 *   1 - if ok
 *   -1 - if write failed
 */

static int WriteIntegers( FILE *fp, int **data, int rows, int cols )
{
	unsigned int hl, nl;
	int i,j;


	for( i = 0; i < cols; i++ ) {
		for( j = 0; j < rows; j++ ) {

			hl = (unsigned int)data[i][j];
			nl = htonl( hl );

			if( fwrite( (char *)&nl, sizeof( nl ), 1, fp ) != 1 ) {
				if( errno < sys_nerr ) {
					ErrorString = sys_errlist[errno];
				} else {
					ErrorString = "write failed";
				}
				return( -1 );
			}

		}
	}

	return( 1 );
}


/*
 * ReadIntegers( fp, rows, cols )
 *
 *  Used by ReadData() to read in integer type data.
 *
 * parameters:
 *   fp - stream to read from
 *   rows - number of rows in data array
 *   cols - number of cols in data array
 *
 * returns:
 *   a pointer to an array of integers if sucessful, otherwise
 *   ErrorString is set and a NULL pointer is returned.
 * 
 */

static int **ReadIntegers( FILE *fp, int rows, int cols )
{
	int **tmp;
	unsigned int netint;
	int count;
	int ret;
	int i, j;

	tmp = (int **)Alloc2d( cols, rows, sizeof(int) );

	if( tmp == NULL ) {
		/* ErrorString is set by Alloc2d */
		return( NULL );
	}

	for( i = 0; i < cols; i++ ) {
		for( j = 0; j < rows; j++ ) {
		  fread(&netint, sizeof(unsigned int), 1, fp);
		  tmp[i][j] = (int) ntohl(netint);
		}
	}

	return( tmp );
}


/*
 * StrToTypeCode( type )
 *
 *  Returns the internal type code of type.
 *  Retruns BADTYPE is and unknown type is passed in.
 *
 * parameters:
 *   type - NULL terminated string.
 *
 * returns:
 *   a the type code of type or BADTYPE.
 */

static int StrToTypeCode( char *type )
{

	if( strcmp( type, "float" ) == 0 ) {

		return( FLOAT );

	} else if( strcmp( type, "short" ) == 0 ) {

		return( SHORT );

	} else if( strcmp( type, "integer" ) == 0 ) {

		return( INTEGER );

	} else if( strcmp( type, "string" ) == 0 ) {

		return( STRING );
	}

	return( BADTYPE );
}

#ifdef DEBUG


static void PrintHeader( AsciiHdr *hdr )
{
	struct extradata *ed;
	int i = 0;
	int j;


	while( (hdr+i)->ah_key != NULL ) {


		printf( "%d: key: '%s'\n", i, (hdr+i)->ah_key );
		printf( "%d: vlen: %d\n", i, (hdr+i)->ah_vlen );

		for( j = 0; j < (hdr+i)->ah_vlen; j++ ) {
			printf( "%d: value: '%s'\n", j, 	
						*((hdr+i)->ah_value+j) );
		}

		i++;
	}
}

#endif DEBUG
