/* rasta.c - Hynek Hermansky 6/25/91	                        */
/* modified Nov. 91 to combine filtered and unfiltered spectra  */
/*                                                              */
/* Converted into Real C not f2c by Johan Schalkwyk (March 93)  */
/*                                                              */


/* include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <speech.h>

/* global constant definitions */
#define two_to_the(N) (int)(pow(2.0,(N))+0.5)

/* external variable declaration  */
extern int errno, sys_nerr;
extern char *sys_errlist[];

/* Table of constant values */
static double c_b17 = 0.33;
static double c_b28 = 10.0;


/* private procedure declarations */
int ffthr(float *real, float *power, int ll, int m);

int a2gexp(float *a, float *gexp, int i, int nc, float expon);

int cosine( int m, int nfilt, float *wcos);

int adw( int npoint, int *nfilt, float *cb, float *eql, int *ibegen, float sf);

int bndpss(float *inasp, float *outasp, float num[][5], float denom[][4], 
	   float inbuf[][5], float outbuf[][4], int *numord, int *denord, 
	   int nfilt, int icall);

int setfilt(float num[][5], float denom[][4], int *numord, 
	    int *denord, int nfilt, float pole);

int hwind(float *weight, int npoint);

int rastaplp(float *speech, int nwind, int m, float *a, float *rc, 
	     float *gain, float sf, float rst, float pole);


/*
 * int ComputeRastaPLP(idata, num_idata, Win_offset, Win_size, Step_size,
 *               Sf, Rst, Expon, M, Np, Gain, pole, plpout_p, numframes_p)
 *
 * Calculates the PLP coefficients of an input speech signal using the
 * algorithm of Hynek Hermanski.
 *
 * idata        (in): pointer to input speech data of type short.
 * num_idata    (in): number of data points available in array 
 * Win_offset   (in): adjustment to window relative to 0 start
 * Win_size     (in): analysis window in samples
 * Step_size    (in): step size in samples
 * Sf           (in): Sampling frequency (why double -- figure out)
 * Rst          (in): Rasta factor (0.0 Full PLP)
 * Expon        (in): peak enhancement factor
 * M            (in): order of PLP model
 * Np           (in): Number of parameters to calculate 
 * Gain         (in): gain flag (1 = gain; 0 no)
 * pole         (in): raste filter pole
 * plpout_p    (out): pointer to output plp data of type float
 * numframes_p (out): return number of frames
 *
 * On exit returns (+1) for success and (-1) for failure
 *
 */

int ComputeRastaPLP(short *idata, int num_idata, int Win_offset, int Win_size,
		    int Step_size, float Sf, float Rst, float Expon, 
		    int M, int Np, int Gain, float pole, float ***plpout_p, 
		    int *numframes_p)
{
    static int   ibeg;                  /* Local variables                  */
 
    static float gain;                  /* gain parameter of model          */
    static float a[16];                 /* auto regressive coefficients     */
    static int   ianfr;                
    static int   nwind;                 /* number of windows in signal      */
    static int   istep;   
    static int   ii;                    /* loop counter variable            */
    static float rc[15];                /* reflection coefficients          */
    static float speech[512];           /* array of 512 samples in speech   */
           int   frame;

#ifdef RAP_COMPILE
DEFAULT_RAM = DRAM;
#endif

    nwind = Win_size;                   /* number of points in anal window  */
    istep = Step_size;                  /* number of points in each step    */

    ianfr = (num_idata - nwind) / istep + 1;   /* how many analysis frames  */
    *numframes_p = ianfr;	               /* return value              */

    /*
     * allocate memory for return vector of plp coefficients
     */

    *plpout_p = (float **) Alloc2d(ianfr, (int) Np ,sizeof(float));
    if(*plpout_p == NULL) {                   
          if( errno < sys_nerr ) 
	         ErrorString = sys_errlist[errno];
	  else
	         ErrorString = "ComputePLP:plpout_p - Alloc2d failed";
	return( -1 );
    }


    /* 
     * start of analysis loop                   
     */

    ibeg = Win_offset;
    for (frame = 0; frame < ianfr; frame++)
      {
	/* current beginning of the analysis window */
	for (ii = 1; ii <= nwind; ++ii)
	  {
	    if(ibeg + ii - 2 >= 0)
	      speech[ii - 1] = (float) idata[ibeg + ii - 2];
	    else
	      speech[ii - 1] = 0.0;
	  }

	/* 
	 * here is the analysis  
	 */

	rastaplp(speech, nwind, M, a, rc, &gain, Sf, Rst, pole);
	if(Gain) {
	    for (ii = 0; ii < M + 1; ii++) {
		a[ii] = a[ii]/gain;
	      }
	  }

	a2gexp(a, (*plpout_p)[frame], M, Np, Expon);
	(*plpout_p)[frame][0] = -(*plpout_p)[frame][0];

	ibeg += istep;
    }
    return (1);
}


/*
 * int rastaplp (speech, nwind, m, a, rc, gain, sf, rst, pole)
 *
 * Computes the m-th order (max 15) Rasta PLP model (described by
 * m+1 autoregressive coefficients a and gain or by m reflection
 * coefficients rc) from nwind-points (max 512) of speech signal with
 * sampling frequency sf (max 20 000 Hz). Bandpass filter version.
 *
 * speech (in): pointer to speech signal (of type float)
 * nwind  (in): window (512 points typically) extracted from speech
 * m      (in): order of model. (m+1) autoregressive coefficients
 * a      (in): pointer to auto regressive coefficients
 * gain  (out): pointer to model gain parameter
 * sf     (in): Sampling frequency
 * rst    (in): Rasta filtering flag
 * pole   (in): raste filter pole value
 *
 */

int rastaplp(float *speech, int nwind, int m, float *a, float *rc, 
	     float *gain, float sf, float rst, float pole)
{
    static int  icall = 0;                  /* Initialized data              */

    int  i_1, i_2;                          /* temp calc and loop variables  */
    double d_1;

    /* Local variables */
    static float hwei[512];                 /* hamming window - weigh speech */
    static int   nfft;                      /* number of points in fft       */
    static float wcos[368];                 /* inverse dft coefficients      */
    static int   nspt;                      /* number of points in power spec*/
    static float r[16], s;                  /* array for reflection coeff    */
    static int   jfilt;                     /* loop counter variable         */
    static float rcmct;                    
    static int   nfilt, nfilt_1;            /* number of filters to use      */
    static float cb[900];                   
    static float eql[23];
    static int   ib, ii, kk, mh, ll, ibegen[69], ip, n;
    static float alpmin, audspe[23], audspe2[23], spectr[512];
    static int   npoint;                    /* number of points in window    */
    static int   nsize;                     /* size of window (num points)   */
    static float aib;   
    static int   icb;
    static float aip, alp[17];
    static int   mct, idx, mct2;
    static float inasp[23], outasp[23];     
    static float num[23][5], denom[23][4];  /* numerator and denom coeff    */ 
    static int   numord[23], denord[23];    /* order of polynomials - filter*/ 
    static float inbuf[23][5], outbuf[23][4];
    static Complex fourier[512];            /* Fourier Coefficients         */

    --speech;                               /* parameter adjustment         */
    --a;
    --rc;

    if (icall == 0) {                       /* if called for the first time, */
	hwind(hwei, nwind);                /*     compute window array      */

	nfft = (int ) (log((double) (nwind)) / 0.693148) + 1;
                                      /* get fft size from the window length */

	npoint = (int ) two_to_the((double)nfft) / 2 + 1;
                                      /* get number of spectral points       */

	nsize  = (int ) two_to_the((double)nfft);
	adw(npoint, &nfilt, cb, eql, ibegen, sf);
                                      /* compute spectral weights            */

	cosine(m, nfilt, wcos);
                                      /* compute IDFT coefficient table      */

	setfilt(num, denom, numord, denord, nfilt, pole);
                                      /* set coefficients  bandpass filters  */

	icall = 1;
    }

    for (ii = 1; ii <= nwind; ++ii)         /* window input speech           */
	speech[ii] = hwei[ii-1] * speech[ii];

    /*********************************/
    /* compute speech power spectrum */
    /*********************************/

/*    ffthr(&speech[1], spectr, nwind, nfft);  */
    FloatFFT(&speech[1], fourier, (int) nfft);

    n = (int ) two_to_the((double) m);
    for (ii=0; ii<=n; ii++)
      spectr[ii] = fourier[ii].c_real * fourier[ii].c_real +
	fourier[ii].c_imag * fourier[ii].c_imag;
    
    /*****************************/
    /* compute auditory spectrum */
    /*****************************/

    nfilt_1 = nfilt - 1;
    for (jfilt = 2; jfilt <= nfilt_1; ++jfilt) {
	audspe[jfilt - 1] = 0.0;

	for (kk = ibegen[jfilt - 1]; kk <= ibegen[jfilt + 22]; ++kk) {
	    icb = ibegen[jfilt + 45] - ibegen[jfilt - 1] + kk;
	    audspe[jfilt - 1] += spectr[kk - 1] * cb[icb - 1];
	}
    }

    /********************************/
    /* auditory spectrum processing */
    /********************************/

    for (ii = 2; ii <= nfilt_1; ++ii) 
	audspe[ii-1] = (float) log ((double) audspe[ii - 1]);


    /*****************************************/
    /* bandpass filter log auditory spectrum */
    /*****************************************/

    if (rst != 0.0) {
	bndpss(audspe, audspe2, num, denom, inbuf, 
		outbuf, numord, denord, nfilt, icall);

	icall++;


     /*************************************************/	
     /* mix filtered and unfiltered auditory spectrum */ 
     /* using weighting factor rst                    */
     /*************************************************/

	for (ii = 2; ii <= nfilt_1; ++ii) 
	  audspe[ii-1] = rst * audspe2[ii-1] + (1.0 - rst) * audspe[ii-1];
      }


    /***************************************/
    /* equal-loudness curve and cubic root */
    /* intensity-loudness compression      */
    /***************************************/

    for (ii = 2; ii <= nfilt_1; ++ii) 
	audspe[ii - 1] = c_b17*(audspe[ii-1]+eql[ii]);


    /****************************/
    /*	take inverse logarithm 	*/
    /****************************/

    for (ii = 2; ii <= nfilt_1; ++ii) 
	audspe[ii - 1] = exp((double)audspe[ii - 1]);

    audspe[0] = audspe[1];
    audspe[nfilt - 1] = audspe[nfilt - 2];


    /**************************************/
    /* inverse discrete fourier transform */
    /**************************************/

    nspt = nfilt - 1 << 1;

    for (kk = 1; kk <= m + 1; ++kk) {
	r[kk - 1] = audspe[0];

	for (ll = 2; ll <= nfilt; ++ll) {
	    r[kk - 1] += audspe[ll - 1] * wcos[ll + kk * 23 - 24];
	}
	r[kk - 1] /= nspt;
    }

    /*************************************/
    /* solution for autoregressive model */
    /*************************************/
    a[1]   = 1.0;
    alp[0] = r[0];
    rc[1]  = -r[1] / r[0];
    a[2]   = rc[1];
    alp[1] = r[0] + r[1] * rc[1];

    for (mct = 2; mct <= m; ++mct) {
	s    = 0.0;
	mct2 = mct + 2;
	alpmin = alp[mct - 1];

	for (ip = 1; ip <= mct; ++ip) {
	    idx = mct2 - ip;
	    s += r[idx - 1] * a[ip];
	}
	rcmct = -s / alpmin;
	mh = mct / 2 + 1;

	for (ip = 2; ip <= mh; ++ip) {
	    ib = mct2 - ip;
	    aip = a[ip];
	    aib = a[ib];
	    a[ip] = aip + rcmct * aib;
	    a[ib] = aib + rcmct * aip;
	}
	a[mct + 1] = rcmct;
	alp[mct] = alpmin - alpmin * rcmct * rcmct;
	rc[mct] = rcmct;
    }

    *gain = alp[m]; 
    return 0;
} 


/*
 * int hwind(weight, npoint)
 *
 * Computes the hamming window on speech signal
 *
 * weight (out): pointer to hamming window output
 * npoint  (in): number of points to calculate
 *
 */

int hwind(float *weight, int npoint)
{
    static int  ii;                           /* loop counter variables     */

    --weight;                                 /* Parameter adjustments      */

    for (ii = 1; ii <= npoint; ++ii) 
	weight[ii] = (float) (0.54 - 0.46 * cos(2.0 * M_PI * (ii - 1) / 
						(npoint - 1)));

    return (0);
} 


/* 
 * int setfilt(num, denom, numord, denord, nfilt, pole)
 *
 * num   (out): numerator polynomial for bandpass filter
 * denom (out): denominator polynomial for bandpass filter
 * numord(out): order of numerator polynomials for bandpass filter
 * denord(out): order of denominator polynomials for bandopass filter
 * nfilt  (in): number of filters to use
 * pole   (in): raste filter pole value
 *
 */

int setfilt(float num[][5], float denom[][4], int *numord, 
	    int *denord, int nfilt, float pole)
{
    static int  ii;                         /* Local variables              */
    static int  jj, nfilt_1;

    nfilt_1 = nfilt - 1;
    for (ii=1; ii < nfilt_1; ii++) {
	numord[ii] = 4;
	denord[ii] = 1;

	denom[ii][0] = 1.0;
	denom[ii][1] = pole;

    	for (jj=0; jj <= 4; jj++) {
		num[ii][jj] = -0.1*(float)(jj-2);
	}
    }
} 
		

/*
 * int bndpss(inasp, outasp, num, denom, inbuf, outbuf, numord, denor, nfilt)
 *
 * bandpass filter implementation for rasta plp
 *
 * inasp  (in): 
 * outsap(out):
 * num    (in): numerator coefficients of filter
 * denom  (in): denominator coefficients of filter
 * inbuf  (in):
 * outbuf(out):
 * numord (in): order of numerator band pass filter
 * denord (in): order of denominator polynomial order
 * nfilt  (in): number of points to filter
 * icall  (in): first time call flag
 *
 */

int bndpss(float *inasp, float *outasp, float num[][5], float denom[][4], 
	   float inbuf[][5], float outbuf[][4], int *numord, int *denord, 
	   int nfilt, int icall)
{
    static int  ii;                           /* loop counter variables      */
    static int  jj, nfilt_1;
    static float sum;

    nfilt_1 = nfilt - 1;
    for (ii=1; ii < nfilt_1; ii++) {
	inbuf[ii][0] = inasp[ii];
	sum = 0.0;

	if(icall >=5 ){
	  for (jj=0; jj <= numord[ii]; jj++) {
	    sum=sum+num[ii][jj]*inbuf[ii][jj];
	  }

	  for (jj=1; jj <= denord[ii]; jj++) {
	    sum=sum-denom[ii][jj]*outbuf[ii][jj-1];
	  }

	  for (jj=denord[ii]; jj > 0; jj--) {
	    outbuf[ii][jj] = outbuf[ii][jj-1];
	  }
	}

    	for (jj=numord[ii]; jj > 0; jj--) {
	  inbuf[ii][jj]=inbuf[ii][jj-1];
	}

	outbuf[ii][0] = sum;
	outasp[ii]    = sum;
      }
} 


/*
 * int adw(npoint, nfilt, cd, eql, ibegen, sf)
 *
 * Computes auditory weightins functions
 *
 * npoint (in): number of points in fft spectrum
 * nfilt  (in): number of samples of auditory spectrum
 *              equally spaced on the bark scale
 *              1 st filter at dc and last at the nyquist frequency
 *
 * cb    (out): array of weighting coefficients to simulate
 *              critical band spectral resolution
 *              and equal loudness sensitivity of hearing
 *              on npoint speech power spectrum
 * eql    (in): array of equal loudness values
 * ibegen(out): three-dimensional array which indicates where
 *              to begin and where to end integration
 *              of speech spectrum and where the given
 *              weighting functions starts in array cb
 *              get Nyquist frequency in Bark
 * sf     (in): Sampling frequency in Hertz
 *
 */

int adw( int npoint, int *nfilt, float *cb, float *eql, int *ibegen, float sf)
{
    float  r_1, r_2;                           /* temp calculation variables */
    double d_1;

    static float freq, zdel, rsss;             /* Local variables            */
    static int   i, j;
    static float x, z, f0, z0, f2samp, fh, fl, fnqbar;
    static int   icount, nfilt_1;
    static float fsq;

    --cb;                                      /* Parameter adjustments      */
    ibegen -= 24;

    d_1    = (double) (sf / 1200.0);
    fnqbar = (float) (6.0 * log(d_1 + sqrt(d_1 * d_1 + 1.0))); 

    *nfilt  = (int ) fnqbar + 2;
    nfilt_1 = *nfilt - 1;
                   /* compute number of filters for less than 1 Bark spacing */

    f2samp = (float) ((float) npoint - 1) / (sf / 2.0);
                   /* frequency -> fft spectral sample conversion            */

    zdel = fnqbar / (float) (nfilt_1);        /* compute filter step in Bark */


    icount = 1;                         /* loop over all weighting functions */
    for (j = 2; j <= nfilt_1; ++j) {

	ibegen[j + 69] = icount;

	z0 = zdel * (float) (j - 1);
	                 /* get center frequency of the j-th filter in Bark */

	f0 = (float) (600.0* (exp( ((double)z0)/6.0) - 
			      exp(-((double)z0)/6.0))/2.0);
                           /* get center frequency of the j-th filter in Hz */

	fl = (float) (600.0* (exp( ( ((double)z0) - 2.5)/6.0) - 
			      exp(-( ((double)z0) - 2.5)/6.0))/2.0);
	                      /* get low-cut frequency of j-th filter in Hz */

	r_1 = fl * f2samp;
	ibegen[j + 23] = ((int ) (r_1+0.5)) + 1;
	if (ibegen[j + 23] < 1) 
	    ibegen[j + 23] = 1;

	fh = (float) (600.0* (exp( ( ((double)z0) + 1.3)/6.0) - 
			      exp(-( ((double)z0) + 1.3)/6.0))/2.0);
	                     /* get high-cut frequency of j-th filter in Hz */

	r_1 = fh * f2samp;
	ibegen[j + 46] = ((int ) (r_1 + 0.5)) + 1;
	if (ibegen[j + 46] > npoint) 
	    ibegen[j + 46] = npoint;

	for (i = ibegen[j + 23]; i <= ibegen[j+46]; ++i) {
	    freq = ((float) (i - 1)) / f2samp;
	                      /* get frequency of j-th spectral point in Hz */

	    x = freq / 600.0;
	                    /* get frequency of j-th spectral point in Bark */
	    d_1 = (double) x;
	    z = (float) (6.0* log(d_1 + sqrt(d_1 * d_1 + 1.0)));

	    z -= z0;             /* normalize by center frequency in barks: */

	    if (z <= -0.5) {                           /* compute weighting */
		d_1 = (double) (z + 0.5);
		cb[icount] = (float) pow(c_b28, d_1);
	    } else if (z >= 0.5) {
		d_1 = (-2.5)*( ((double)z) - 0.5);
		cb[icount] = (float) pow(c_b28, d_1);
	    } else {
		cb[icount] = 1.0;
	    }
	    
	    /* calculate the LOG 40 db equal-loudness curve */
	    /* at center frequency                          */

	    fsq = f0 * f0;
	    r_2 = fsq + 1.6e5;
	    rsss = fsq * fsq * (fsq + 1.44e6) / (r_2 * r_2 * (fsq + 9.61e6));

	    eql[j] = log((double)rsss);
	       /* take log and put the equal-loundness curve into eql array */

	    ++icount;
	}
    }

    return (0);
} 


/*
 * int cosine(m, nfilt, wcos)
 *
 * computes the cosine weightings for the Inverse Discrete Fourier Transform
 *
 * m    (in): order of plp auto regressive model
 * nfilt(in): number of points in filter
 * wcos(out): cosine weightings of inverse dft
 *
 */

int cosine( int m, int nfilt, float *wcos)
{
    static int  ii, jj;                        /* Loop counter variables     */
    static int  nfilt_1, mp1;

    wcos -= 24;                               /* Parameter adjustments      */

    nfilt_1 = nfilt - 1;
    mp1     = m + 1;
    for (ii = 1; ii <= mp1; ++ii) 
      {
	for (jj = 2; jj <= nfilt_1; ++jj) 
	  {
	    wcos[jj + ii * 23] = (float) (2.0 * cos(2.0 * M_PI * (ii - 1) * 
					    (jj - 1) / (2.0 * ( nfilt_1))));
	  }
	wcos[nfilt + ii * 23] = (float) (cos(2.0 * M_PI * (ii - 1) * (jj - 1) /
					     (2.0 * (nfilt_1))));
    }

    return (0);
} 


/*
 * int a2gexp(a, gexp, i, nc, expon)
 *
 * a    (in): pointer to autoregressive coefficients
 * gexp(out): 
 * i    (in): number of elements in autoregressive coefficient array
 * nc   (in): number of elemetns in output array
 * expon(in):
 *
 */

int a2gexp(float *a, float *gexp, int i, int nc, float expon)
{
    int    i_2;

    double d_1, d_2;

    static float c[257];                       /* Local variables           */
    static int   j, l, jb, ii, lp;
    static float sum;

    --a;                                       /* Parameter adjustments     */
    --gexp;

    c[0] = (float) log(a[1]);
    c[1] = -a[2] / a[1];
    for (l = 2; l <= nc; ++l) {
	lp = l + 1;
	if (l <= i + 1) {
	    sum = l * a[lp] / a[1];
	} else {
	    sum = 0.0;
	}
	i_2 = l;
	for (j = 2; j <= i_2; ++j) {
	    jb = l - j + 2;
	    if (j <= i + 1) {
		sum += a[j] * c[jb - 1] * (jb - 1) / a[1];
	    }
	}
	c[lp - 1] = -sum / l;
    }
    gexp[1] = c[0];
    for (ii = 2; ii <= nc; ++ii) {
      if (expon != 0.0) 
	gexp[ii] = (float) pow(((double) (ii-1)), ((double) expon)) * 
	  c[ii - 1];
      else 
	gexp[ii] = c[ii - 1];
    }
    
    return (0);
} 


/*
 * int ffthr(real, power, ll, m)
 *
 * Calculates the power spectral density of the fft signal. Subroutine
 * for computing the power spectrum of real signal REAL, of length LL
 * M is exponent of the FFT (max m = 9)
 *
 * real   (in): real signal input for FFT
 * power (out): output power spectral density
 * ll     (in): number of points in signal
 * m      (in): exponent of fft (zero padd signal if needed)
 *
 */

int ffthr(float *real, float *power, int ll, int m)
{
	static int   i, j, k, l, n, k1, n1, n2;
	static float t1, t2, u1, u2, w1, w2, u3, u4;
	static int   id, le, ii;
	static float sigima[512], signal[512];
	static int   le1;

	--real;                            	/* Parameter adjustments    */
	--power;

	k = 0;
	n = two_to_the((double) m);

	for (ii = 1; ii <= ll; ++ii) {
	  sigima[ii - 1] =  0.0;
	  signal[ii - 1] = real[ii];
	}

	for (ii = ll + 1; ii <= n; ++ii) {
	  sigima[ii - 1] = 0.0;
	  signal[ii - 1] = 0.0;
	}

	n2 = n / 2;
#ifdef DEBUG
	printf("n2 = %d, n = %d\n", n2, n);
#endif
	n1 = n - 1;
	j  = 1;
	for (i = 1; i <= n1; ++i) {
	  if (i < j) {
	    t1 = signal[j - 1];
	    t2 = sigima[j - 1];
	    signal[j - 1] = signal[i - 1];
	    sigima[j - 1] = sigima[i - 1];
	    signal[i - 1] = t1;
	    sigima[i - 1] = t2;
	  }

	  k1 = n2;
	  while (k1<j) {
	    j  -= k1;
	    k1 /= 2;
	  }

	  j += k1;
	}

	for (l = 1; l <= m; ++l) {
	  le  = two_to_the((double) l);
	  le1 = le / 2;
	  u1  = 1.0;
	  u2  = 0.0;

	  w1 = (float) cos(M_PI / (double) le1);
	  w2 = (float) -sin(M_PI / (double) le1);

	  for (j = 1; j <= le1; ++j) {
	    for (i = j; le < 0 ? i >= n : i <= n; i += le) {
	      id = i + le1;
	      t1 = signal[id - 1] * u1 - sigima[id - 1] * u2;
	      t2 = sigima[id - 1] * u1 + signal[id - 1] * u2;
	      signal[id - 1] = signal[i - 1] - t1;
	      sigima[id - 1] = sigima[i - 1] - t2;
	      signal[i - 1] += t1;
	      sigima[i - 1] += t2;
	    }
		
	    u3 = u1;
	    if (k != 0) {
	      u4 = w1 * w1 + w2 * w2;
	      u1 = (u1 * w1 + u2 * w2) / u4;
	      u2 = (u2 * w1 - u3 * w2) / u4;
	    } else {
	      u1 = u1 * w1 - u2 * w2;
	      u2 = u2 * w1 + u3 * w2;
	    }
	  }
	}

	if (k !=0 ) {
	  for (i = 1; i <= n; ++i) {
	    sigima[i - 1] /= n;
	    signal[i - 1] /= n;
	  }
	}

	for (ii = 1; ii <= n; ++ii) {
	  power[ii] = signal[ii - 1] * signal[ii - 1] + 
	    sigima[ii - 1] * sigima[ii - 1];
	}
	
	return (0);
}










