/*
 * $Id: nnet.c,v 2.3 1993/05/11 18:52:04 johans Exp $
 *
 *
 * These routines allow OPT-trained (or other) weight files to be
 * used for classification.
 *
 *
 */

/* Standard C library include file directives */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>

/* Speech library include file directives */
#include <netinet/in.h>
#include <speech.h>


#define EMAX		(100.0)
#define EMIN		(-100.0)

/* 
 * extrema functions 
 */
#define n_max(a,b)	(((a)>(b))?(a):(b))
#define n_min(a,b)	(((a)<(b))?(a):(b))

/* 
 * bound b by a and c, so that a <= b <= c obtains 
 */
#define	bound(a,b,c)	(n_max((a),n_min((b),(c))))


/* 
 * Sigmoid nonlinearity in-line function:
 *  sigmoid(s, alpha) == Sigmoid(s/alpha)
 */
#define Sigmoid(x)	(1./(1.+exp(-(bound(EMIN,(x),EMAX)))))

/* external variable declarations */
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/* locale module header declarations */
static int FloatIndexMax (int s, float *a);


/* 
 * Find the index of the maximum value in a float array 
 */

static int FloatIndexMax (int s, float *a)
{
	float *pm;	/* pointer to current maximum */
	float *pa;	/* pointer to the array */


	pa = a;			/* initialize pointer to array */
	pm = pa;		/* initialize pointer to current maximum */

	while( --s ) {			/* for the remaining size */
		if (*++pa > *pm) {	/* if new maximum */
			pm = pa;	/* update pointer to maximum */
		}
	}

	return( pm - a );		/* return index of maximum */
}


/* 
 * Compute the activation values for this layer of the neural network
 */

int NNLayer ( NeuralNet *nn, int layer, float *ivec, float *ovec )
{
	float sum;	/* accumulator for weighted sum */
	float *pa1;	/* pointer to activation layer i */
	float *pw;	/* pointer to weights */
	int szlayer;
	int szlayer1;
	int i1, i2;


	szlayer = nn->nn_sz[layer];
	szlayer1 = nn->nn_sz[layer+1];

	/* 
	 * initialize pointer to weights 
	 */
	pw = nn->nn_w + nn->nn_layeridx[layer]; 


	/*
	 * layer 0 needs the bais put on the end of the ivec.
	 * The loop was duplicated for speed.
	 */
	if( layer == 0 ) {

		/* 
		 * for each node in the layer+1st layer 
		 */
		for(i2 = 0; i2 < szlayer1; i2++) {	

			pa1 = ivec;
			sum = 0.0;

			/* 
			 * for each node in the layer-th layer 
			 */

			for (i1 = 0; i1 < szlayer-1; i1++) {
				sum += *pw++ * *pa1++;
			}
			sum += *pw++ * 1.0;


			/* 
			 * store the value in activation level i+1 
			 */
			*ovec++ = Sigmoid(sum / 1.0);
		}

	} else {

		/* 
		 * for each node in the layer+1st layer 
		 */
		for(i2 = 0; i2 < szlayer1; i2++) {	

			pa1 = ivec;
			sum = 0.0;

			/* 
			 * for each node in the layer-th layer 
			 */
			for (i1 = 0; i1 < szlayer; i1++) {
				sum += *pw++ * *pa1++;
			}

			/* 
			 * store the value in activation level i+1 
			 */
			*ovec++ = Sigmoid(sum / 1.0);
		}

	}

	return( 1 );
}


/* 
 * Neural Network Classification -- full output vector 
 *
 * nn   (in): neural network structure
 * ivec (in): input vector
 * ovec (out): output vector
 *
 */

void NNClassifyVector(NeuralNet *nn, float *ivec, float *ovec)
{
	int i;		/* loop index */


	/* 
	 * process first layer, ivec == nn->act[0] 
	 */
	i = 0;
	NNLayer(nn, i, ivec, nn->nn_act[i + 1]);
	i++;

	/* 
	 * process intermediate layers 
	 * for each intermediate layer
	 */
	for (; i < nn->nn_nlayers - 2; i++) {
		NNLayer(nn, i, nn->nn_act[i], nn->nn_act[i + 1]);
	}

	/* 
	 * process final layer, ovec == nn->nn_act[nn->nn_nlayers - 1] 
	 */
	NNLayer(nn, i, nn->nn_act[i], ovec);
}


/* 
 * Neural Network Classification -- class index output 
 *
 * fvec (in): input (feature) vector
 * nn   (in): neural net work structure
 *
 */

int NNClassify (float *fvec, NeuralNet *nn)
{
	/* 
	 * use final activation layer as output vector 
	 */
	NNClassifyVector(nn, fvec, nn->nn_act[nn->nn_nlayers - 1]);

	/* 
	 * return index of maximum output in final activation layer 
	 */
	return( FloatIndexMax(nn->nn_sz[nn->nn_nlayers-1], nn->nn_act[nn->nn_nlayers-1]) );
}

#ifdef not

/* 
 * Read an ASCII neural net weight file 
 */

int NNReadASCII (FILE *inp, NeuralNet *nn)
{
	int layer;	/* loop index: each layer of the neural net */
	int iit;	/* number of iterations -- discarded */
	int nl;		/* number of layers */
	float alpha;
	int i1,i;
	int i2;
	int c;


	c = fscanf( inp, "%d %f %d", &iit, &alpha, &(nn->nn_nlayers));

	if( c != 3 ) {
		ErrorString = "bad neural net file";
		return( -1 );
	}

	nl = nn->nn_nlayers;

	if (!(nn->nn_sz = (int *) malloc(nl * sizeof(int)))) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}

	/* 
	 * allocate activation and weight arrays 
	 */
	if (!(nn->nn_act = (float **) malloc(nn->nn_nlayers * sizeof(float *)))) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}

	for (layer = 0; layer < nn->nn_nlayers; layer++) {
		nn->nn_act[layer] = (float *)malloc(nn->nn_sz[layer] * 
							sizeof(float *));
		if( nn->nn_act[layer] == NULL ) {
			if( errno < sys_nerr ) {
				ErrorString = sys_errlist[errno];
			} else {
				ErrorString = "malloc failed";
			}
			return( -1 );
		}
	}
    
	nn->nn_w = (float ***) malloc((nn->nn_nlayers - 1) * sizeof(float **));
	if( nn->nn_w == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}

	for (layer = 0; layer < nn->nn_nlayers - 1; layer++) {
		nn->nn_w[layer] = (float **)malloc(nn->nn_sz[layer + 1] * 
							sizeof(float *));

		if( nn->nn_w[layer] == NULL ) {
			if( errno < sys_nerr ) {
				ErrorString = sys_errlist[errno];
			} else {
				ErrorString = "malloc failed";
			}
			return( -1 );
		}

		for (i = 0; i < nn->nn_sz[layer+1]; i++) {
			nn->nn_w[layer][i] = (float *)malloc(nn->nn_sz[layer] * 
								sizeof(float));

			if( nn->nn_w[layer][i] == NULL ) {
				if( errno < sys_nerr ) {
					ErrorString = sys_errlist[errno];
				} else {
					ErrorString = "malloc failed";
				}
				return( -1 );
			}
		}
      }

	/* 
	 * read weight array 
	 */
	for (layer = 0; layer < nl - 1; layer++) {
		for (i1 = 0; i1 < nn->nn_sz[layer + 1]; i1++) {
			for (i2 = 0; i2 < nn->nn_sz[layer]; i2++) {

				c = fscanf(inp, "%f", &(nn->nn_w[layer][i1][i2]));

				if( c != 1 ) {
					ErrorString = "bad neural net file";
					return( -1 );
				}
			}
		}
	}

	return( 1 );
}
#endif


/* 
 * Read a binary neural net weight file
 * produced by opt
 */

int NNRead (FILE *fileptr, NeuralNet *nn)
{
	unsigned int utmp, hutmp;
	int layer;	/* loop index:  each layer of the neural net */
	int i;		/* inner loop index */
	int nconn;
	int j;
	int c;
	union {
		float f;
		unsigned int ul;
	} f_ul;


	c = fread( &utmp, sizeof(unsigned int), 1, fileptr);

	if( c == 0 ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fread failed";
		}
		return( -1 );
	}

	hutmp = ntohl( utmp );
	nn->nn_iter = hutmp;

	c = fread( &utmp, sizeof(unsigned int), 1, fileptr);

	if( c != 1 ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fread failed";
		}
		return( -1 );
	}
	hutmp = ntohl( utmp );
	nn->nn_nlayers = hutmp;

	/* 
	 * allocate and read sz = vector of sizes of each layer 
	 */
	nn->nn_sz = (int *) malloc(nn->nn_nlayers * sizeof(int));
	if( nn->nn_sz == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}

	for( i = 0; i < nn->nn_nlayers; i++ ) {

		c = fread( &utmp, sizeof(unsigned int), 1, fileptr);

		if( c != 1 ) {
			if( errno < sys_nerr ) {
				ErrorString = sys_errlist[errno];
			} else {
				ErrorString = "fread failed";
			}
			free( nn->nn_sz );
			return( -1 );
		}

		hutmp = ntohl( utmp );
		nn->nn_sz[i] = hutmp;

	}

	/*
	 * Allocat space for layer index
	 */
	nn->nn_layeridx = (int *) malloc(nn->nn_nlayers * sizeof(int));
	if( nn->nn_layeridx == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		free( nn->nn_sz );
		return( -1 );
	}

	/*
	 * Compute layer index and number of connections
	 */
	nconn = 0;
        for( i = 0; i < nn->nn_nlayers - 1; i++ ) {
		nn->nn_layeridx[i] = nconn;
                nconn += nn->nn_sz[i] * nn->nn_sz[i + 1];
        }

	/* 
	 * allocate activation and weight arrays 
	 */
	nn->nn_act = (float **)malloc( nconn * sizeof(float *));

	if( nn->nn_act == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		free( nn->nn_sz );
		free( nn->nn_layeridx );
		return( -1 );
	}

	for (layer = 0; layer < nn->nn_nlayers; layer++) {
		nn->nn_act[layer] = (float *) malloc(nn->nn_sz[layer] *
							sizeof(float *));
		if( nn->nn_act[layer] == NULL ) {
			if( errno < sys_nerr ) {
				ErrorString = sys_errlist[errno];
			} else {
				ErrorString = "malloc failed";
			}
			free( nn->nn_sz );
			free( nn->nn_layeridx );
			return( -1 );
		}
	}

	nn->nn_w = (float *)malloc( sizeof(float) * nconn );

	if( nn->nn_w == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		free( nn->nn_sz );
		free( nn->nn_layeridx );
		return( -1 );
	}


	/* 
	 * read weight array 
	 */

	for( i = 0; i < nconn; i++ ) {

		c = fread( &utmp, sizeof(unsigned int), 1, fileptr );

		if( c != 1 ) {
			if( errno < sys_nerr ) {
				ErrorString=sys_errlist[errno];
			} else {
				ErrorString = "fread failed";
			}
			return( -1 );
		}

		hutmp = ntohl( utmp );
		f_ul.ul = hutmp;

		nn->nn_w[i] = f_ul.f;
	}

	return( 1 );
}


/*
 * NNWrite( fp, nn )
 *
 *   Writes a NeuralNet to disk.
 *
 * parameters:
 *	fp - stream to write neural net too
 *	nn - neural net to write
 *
 * return:
 *	a positive number on success, otherwise
 *	a negitive nuber is returned and ErrorString is set
 *
 */

int NNWrite (FILE *fp, NeuralNet *nn )
{
	unsigned int utmp, nutmp;
	int i, nconn;
	int c;
	union {
		float f;
		unsigned int ul;
	} f_ul;



	nconn = 0;
        for( i = 0; i < nn->nn_nlayers - 1; i++ ) {
                nconn += nn->nn_sz[i] * nn->nn_sz[i + 1];
        }

	utmp = nn->nn_iter;
	nutmp = htonl( utmp );
	c = fwrite( &nutmp, sizeof(unsigned int), 1, fp );

	if( c == 0 ) {
		if( errno < sys_nerr ) {
			ErrorString=sys_errlist[errno];
		} else {
			ErrorString = "fwrite failed";
		}
		return( -1 );
	}

	utmp = nn->nn_nlayers;
	nutmp = htonl( utmp );
	c = fwrite( &nutmp, sizeof(unsigned int), 1, fp );

	if( c == 0 ) {
		if( errno < sys_nerr ) {
			ErrorString=sys_errlist[errno];
		} else {
			ErrorString = "fwrite failed";
		}
		return( -1 );
	}

	for( i = 0; i < nn->nn_nlayers; i++ ) {

		utmp = nn->nn_sz[i];
		nutmp = htonl( utmp );
		c = fwrite( &nutmp, sizeof(unsigned int), 1, fp );

		if( c == 0 ) {
			if( errno < sys_nerr ) {
				ErrorString=sys_errlist[errno];
			} else {
				ErrorString = "fwrite failed";
			}
			return( -1 );
		}
	}

	for( i = 0; i < nconn; i++ ) {
		
		f_ul.f = nn->nn_w[i];
		utmp = f_ul.ul;
		nutmp = htonl( utmp );
		c = fwrite( &nutmp, sizeof(unsigned int), 1, fp );

		if( c == 0 ) {
			if( errno < sys_nerr ) {
				ErrorString=sys_errlist[errno];
			} else {
				ErrorString = "fwrite failed";
			}
			return( -1 );
		}
	}

	return( 1 );
}

/*
 * Open, read and close a binary neural net weight file 
 */

int NNLoad (char *filename, NeuralNet *nn)
{
	FILE *fp;

	if ((fp = fopen(filename, "r")) == NULL) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fopen failed";
		}
		return( -1 );
	}

	if( NNRead(fp, nn) < 0 ) {
		fclose( fp );
		return( -1 );

	}

	if( fclose(fp) == EOF ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "fclose failed";
		}
		return( -1 );
	}

	return( 1 );
}


/* 
 * Deallocate memory allocated in a neural net structure by
 * NNRead(), NNReadASCII(), etc.
 */

int NNFree (NeuralNet *nn)
{
	int layer, i;

	for (layer = 0; layer < nn->nn_nlayers; layer++) {
		free(nn->nn_act[layer]);
	}

	free( nn->nn_act );
	free( nn->nn_sz );
	free( nn->nn_w );
	free( nn->nn_layeridx );

	return( 1 );
}
