/* 
 * filt.c
 *
 * $Id: filt.c,v 2.2 1993/05/07 18:12:22 johans Exp $
 * 
 *
 */

/* Standard C library include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

/* Speech library include file directives */
#include <speech.h>
#include <filters.h>


typedef struct  {
	char *ident;	/* character string to identify filter */
	float *coeff;	/* pointer to filter coefficients */
	int numcoeff;	/* number of coefficients */
} FILTER;


static FILTER	filterlist[] = {
	{"0-075", filt0_1200, sizeof(filt0_1200) / sizeof(float)},
	{"0-15625", filt0_2500, sizeof (filt0_2500) / sizeof (float)},
	{"0-04375", filt0_700, sizeof (filt0_700) / sizeof (float)},
	{"0625-125", filt1000_2000, sizeof (filt1000_2000) / sizeof (float)},
	{"0625-21875", filt1000_3500, sizeof (filt1000_3500) / sizeof (float)},
	{"075-5", filt1200_8000, sizeof (filt1200_8000) / sizeof (float)},
	{"125-21875", filt2000_3500, sizeof (filt2000_3500) / sizeof (float)},
	{"0125-05", filt200_800, sizeof (filt200_800) / sizeof (float)},
	{"15625-28125", filt2500_4500, sizeof (filt2500_4500) / sizeof (float)},
	{"15625-5", filt2500_8000, sizeof (filt2500_8000) / sizeof (float)},
	{"015625-21875", filt250_3500, sizeof (filt250_3500) / sizeof (float)},
	{"21875-5", filt3500_8000, sizeof (filt3500_8000) / sizeof (float)},
	{"3125-5", filt5000_8000, sizeof (filt5000_8000) / sizeof (float)},
	{"03125-21875", filt500_3500, sizeof (filt500_3500) / sizeof (float)},
	{"0-25", filt0_4000, sizeof (filt0_4000) / sizeof (float)},
	{"01875-2", filt300_3200, sizeof (filt300_3200) / sizeof (float)},
        {"faa", lp04375, sizeof (lp04375) / sizeof (float)}};

int numfilterlist = sizeof (filterlist) / sizeof (FILTER);

/* external variable declarations */
extern int errno;
extern int sys_nerr;
extern char *sys_errlist[];

/* local module header declarations */
static int FirCPU( short *indata, short **outdata, int numdata, 
		  float *filter, int filtsize, float dcoffset, int decfac );
static int LookupFilterByName( char *filtername );


/*
 * int FiltByName(filtername, adc, numadc, adcfilt)
 *
 */

int FiltByName (char *filtername, short *adc, int numadc, short **adcfilt )
{
	int i;
	

	if( ( i = LookupFilterByName( filtername ) ) < 0 ) {
		ErrorString = "unknown filter";
		return( -1 );
	}

	return( Filt( filterlist[i].coeff, filterlist[i].numcoeff, 
						adc, numadc, adcfilt ));

}

/*
 * FilterCoefficents()
 *
 *  lookups filtername and places the number of
 *  coeffiecents in *numcoeff and the coefficents
 *  in **coeff.
 *
 */

int FiltCoefficents ( char *filtername, float **coeff, int *numcoeff )
{
	int i;

	if( ( i = LookupFilterByName( filtername ) ) < 0 ) {
		ErrorString = "unknow filter";
		return( -1 );
	}

	*numcoeff = filterlist[i].numcoeff;
	*coeff = filterlist[i].coeff;	/* return copy instead?? XXX */

	return( 1 );

}


/*
 * Filt (filter, numfilt, adc, numadc, adcfilt)
 *
 * filter (in): filter coefficients
 * numfilt(in): number of coeff in filter spec
 * adc    (in): input waveform
 * adcfilt(in): address of output waveform
 *
 */

int Filt (float *filter, int numfilt, short *adc, int numadc, short **adcfilt)
{
	short *adctemp;

	/*
	 * Allocate room for the filtered waveform 
	 */
	*adcfilt = (short *) calloc( numadc, sizeof(short) );

	if( *adcfilt == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "calloc failed";
		}
		return( -1 );
	}

	/*
	 * Offset the beginning of the output so that it lines up with
	 * the input.
	 */
	adctemp = *adcfilt + numfilt / 2;

	/*
	 * Filter the waveform.
	 */
	FirCPU( adc, &adctemp, numadc, filter, numfilt, 0.0, 1 );
	return( 1 );

}


/*
 * FirCPU (indata, outdata, numdata, filter, filtsize, dcoffset, decfac)
 *
 * indata (in): array of input data
 * outdata (out): address of output data, space must already exist
 * numdata (in): num of elements in input array
 * filter  (in): filter specification array
 * filtsize(in): number of elements in filter specification
 * dcoffset(in): dc offset to substract off
 * decfac  (in): decimation factor
 *
 */

static int FirCPU( short *indata, short **outdata, int numdata, 
		  float *filter, int filtsize, float dcoffset, int decfac )
{
	short *ptrin, *ptrout;
	int i, j, outcount;
	float temp;
	int temp1;

	/*
	 * Subtract off the dc offset if one existed
	 */
	if( (int)(dcoffset) != 0 ) {

		for( i = 0, ptrin = indata; i < numdata; i++, ptrin++ ) {
			*ptrin = *ptrin + dcoffset;
		}
	}

	/*
	 * Perform the filter operation
	 */
	outcount = (numdata - filtsize + 1) / decfac;
	ptrin = indata;
	ptrout = *outdata;

	for( i = 0; i < outcount; i++ ) {

		temp1 = decfac * i;
		temp = 0.0;

		for( j = 0; j < filtsize; j++ ) {
			temp += ( *(ptrin+temp1+j) * 
				  *(filter+filtsize-j-1) );
		}

		*ptrout++ = temp;
	}

	/*
	 * Return the number of output points upon completion
	 */
	return( outcount );

}


/*
 * LookupFilterByName
 *
 *  Looks up a filter by name and returns the index into
 *  filterlist[] if found or -1 if not found.
 *
 */

static int LookupFilterByName( char *filtername )
{
	int i;

	for( i = 0; i < numfilterlist; i++ ) {

		if( strcmp( filtername, filterlist[i].ident ) == 0 ) {
			return( i );
		}
	}

	return( -1 );

}




