/* global constant definitions */
#define CONTEXTBASE ".dbf"           /* contextual data base file extension */
#define INDEX       ".ndx"           /* index file extensions               */
#define LOLA        ".lola"          /* lola file listing (text file)       */
#define ADC         ".wav"           /* adc file listing                    */
#define IDBASE      ".id"            /* file id data base extension         */

#define SWAPL(x)	((((x)<<24)&0xFF000000) | (((x)<<8)&0x00FF0000) | \
			 (((x)>>8) & 0x0000FF00) | (((x)>>24)&0x000000FF))

#define SWAPW(x)        (((x)<<8) | (0xFF & ((x)>>8)))
					    

#define BUFFER_MARGIN 20             /* size of step increment in buffer    */
#define PHON_WIDTH    20


/* global structure definitions */

/*
 * Contextual Record
 * 
 * Deletion indicated by setting ph_curr = 0;
 *   all other phonemes are index 1..N
 *
 */

struct contextrec {
  char headorder;          /* byte ordering of record              */

  int  from_file_id;       /* id of file where this phoneme occurs */
  int  prevlink;           /* previous link of phoneme chain (-1)  */
  int  nextlink;           /* next link in phoneme chain (-1 end)  */

  short ph_curr;           /* current phoneme (center of context)  */
  int  startloc;           /* current phoneme index in speech file */
  int  endloc;             /* end point in signal of phoneme       */

  short ph_prev;           /* previous phoneme in chain            */
  int  prev_start;         /* offset to previous phoneme in chain  */

  short ph_next;           /* next phoneme in chain                */
  int  next_end;           /* offset to end of next phoneme        */

  int  ph_prevlink;        /* previous phoneme link (contextual)   */
  int  ph_nextlink;        /* next phoneme link (contextual)       */
};

typedef struct contextrec ContextRec; /* Contextual record (prev-curr-next */ 


/*
 * Index file record
 *
 * Deletion of record indicated by setting phoneme = 0
 *
 */

struct phonemerec {
  char headorder;          /* byte ordering of header              */

  char phoneme[PHON_WIDTH];  /* char value of phoneme in data base */
  int  startrec;           /* pointer to first occurrence          */
  int  endrec;             /* pointer to last occurrence           */
};

typedef struct phonemerec PhonemeRec;      /* data base index file record   */


/*
 * FileID record
 *
 * Deletion of record indicated by zero length strings in record
 *
 */

struct fileidrec {
  char phnfilename[200];            /* name of file (phn file)               */
  char wavfilename[200];            /* name of wav file name                 */
  char headorder;                   /* byte ordering of current record       */
  int  start_point;                 /* starting location of phonemes in DB   */
  float ms;                         /* ms of frame in fileid rec             */
};

typedef struct fileidrec FileIDRec;        /* File ID record for wav files   */



/*
 * int Read_Index_File(FileName, PhonemeIndex, PreviousLink)
 *
 * This function attempts to read an Index file. If the file
 * does not exists, it will create and empty file, else it will
 * abort. Index files are assumed always to be less than 32767
 * possible phonemes.
 *
 * FileName      (in): Name of Index file
 * PhonemeIndex (out): pointer to array of record of type phoneme rec
 * PreviousLink (out): pointer to array of previous links in data base
 *
 * On Error returns -1, else 1
 *
 */

int Read_Index_File(char *FileName, PhonemeRec **PhonemeIndex, int **PreviousLink);


/*
 * int Write_Index_File(FileName, PhonemeIndex)
 *
 * This function writes the array of index values out to the index 
 * file.
 *
 * FileName    (in): name of index file (output file name)
 * PhomemeIndex(in): pointer to array of index values
 *
 * On error returns -1, and sets ErrorString, else returns 1
 *
 */

int Write_Index_File(char *FileName, PhonemeRec *PhonemeIndex);


/*
 * int Retrieve_Phoneme_Index(InputPhoneme, PhonemeIndex, PreviousLink)
 *
 * This function determines the index value of the Input Phoneme
 * If this phoneme does not exist, it places it in the index file
 * if it can. It assumes the current file pointer points to
 * the position where the record will appear. If it can not
 * add to the index file it will abort and return (-1)
 *
 * InputPhoneme   (in): string contaning the new or old phoneme
 * PhonemIndex (in/out): pointer to index for retrieval of information
 * PreviousLink(in/out): pointer to vector of previous links of phoneme
 *
 * On error returns -1, else +1
 *
 */

int Retrieve_Phoneme_Index(char *InputPhoneme, PhonemeRec **PhonemeIndex, int **PreviousLink);


/*
 * Open_Contextual_Base(contextfile, FileName)
 *
 * This procedure opens the Contextual data base file, and sets 
 * global pointer positions to the record numbers.
 *
 * contextfile(in): pointer to file stream 
 * FileName   (in): name of contextual database file
 *
 * On Error return -1, else +1
 *
 */

int Open_Contextual_Base(FILE **contextfile, char *FileName);


/*
 * Close_Contextual_Base(contextfile)
 *
 * This procedure closes the contextual data base
 *
 */

void Close_Contextual_Base(FILE **contextfile);


/*
 * int Read_Context_Record(contextfile, Phoneme, recordno)
 *
 * This function reads one phoneme record from the contextual
 * data base.
 *
 * contextfile (in): pointer to the file stream
 * Phoneme    (out): pointer to the contextual record in file
 * recordno    (in): number of record to retrieve
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Read_Context_Record(FILE *contextfile, ContextRec *Phoneme, int recordno);


/*
 * int Write_Context_Record(contextfile, Phoneme, recordno)
 *
 * This function writes one record to the file, at the specified
 * location.
 *
 * contextfile (in): pointer to file stream
 * Phoneme     (in): contextual data base record to write
 * recordno    (in): record to write to file
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Write_Context_Record(FILE *contextfile, ContextRec Phoneme, int recordno);


/*
 * int Update_Context_Record(contextfile, previous_link, current_link)
 *
 * This function reads in a record from the previous link
 * value and updates its next link value.
 *
 * contextfile   (in): pointer to file stream
 * previous_link (in): record number in contextual data base
 * current_link  (in): current record in contextual data base
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Update_Context_Record(FILE *contextfile, int previous_link,
			  int current_link);


/*
 * Print_Context_Record(Phoneme)
 *
 * This procedure prints one phoneme contextual record on the
 * screen.
 *
 * Phoneme (in): Contextual record in the data base
 *
 */

void Print_Context_Record(ContextRec Phoneme);


/*
 * int Open_FileID_Base(idfile, FileName)
 *
 * This procedure opens the FileID data base file, and sets the global
 * file id pointer to the current record number (end of file )
 *
 * idfile   (in): pointer to file stream
 * FileName (in): name of file id data base
 *
 * On error returns -1, else 1
 *
 */

int Open_FileID_Base(FILE **idfile, char *FileName);


/*
 * Close_FileID_Base(idfile)
 *
 * Closes the from file id data base
 *
 */

void Close_FileID_Base(FILE **idfile);


/*
 * int Read_FileID_Record(idfile, FileID, recordno)
 *
 * This function reads one record frm the id file, which describes
 * where the specific sounds can be found on disc.
 *
 * idfile  (in): pointer to file stream
 * FileID (out): one record in data base (pointer)
 * recordno(in): record numebr to retrieve
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Read_FileID_Record(FILE *idfile, FileIDRec *FileID, int recordno);


/*
 * int Write_FileID_Record(idfile, FileID, recordno)
 *
 * This function writes one FileID record to disc
 *
 * idfile   (in): pointer to file stream
 * FileID   (in): one record in the data base
 * recordno (in): record number where to write in file
 *
 * On error returns -1, and sets ErrorString else return +1
 *
 */

int Write_FileID_Record(FILE *idfile, FileIDRec FileID, int recordno);


/* 
 * Print_FileID_Record(idrec)
 *
 * This procedure prints one file id record on screen
 *
 * idrec (in): File ID record in file id database
 *
 */

void Print_FileID_Record(FileIDRec idrec);


/*
 * int Return_FileID(idfile, FileID)
 *
 * This function searches the FileID data base for the required FileID
 * 
 * idfile (in): pointer to file stream
 * FileID (in): record needed for match
 *
 */

int Return_FileID(FILE *idfile, FileIDRec FileID);





