/*
 * merge_lola.c
 *
 * The merge_lola.c program, merges an edited lola file
 * with the data base value. Assumes only the center segment has been
 * edited and does not change any other segment values.
 *
 */

/* standard C library include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* Speech tools include file directives */
#include <speech.h>


/* data base include file directives */
#include <phonebase.h>


/* global constant definitions */
#define    LOGEXT    ".log"              /* log file for mergin process    */
#define    LOLAEXT   ".lola"             /* lola file extension            */


/* external variable declaration */
extern int  errno, sys_nerr;
extern char *sys_errlist[];


/* global variable declarations  */
int        CREATE_DATABASE = 0;     /* create database flag                 */
int        *PreviousLink;           /* array of most recent previous links  */
PhonemeRec *PhonemeIndex;           /* phoneme for index file               */

FILE       *contextfile;            /* contextual phoneme data base         */
FILE       *idfile;                 /* from file id data base               */
FILE       *lolafile;               /* listing of lola files to build base  */
FILE       *wavfile;                /* waveform output file stream          */

char       headorder, machineorder; /* byte ordering of record and machine  */
char       byteswap;                /* must i do byte swapping flag         */


/* global command line parameter variables */
char       contextfilename[200];    /* name of contextual database file     */
char       indexfilename[200];      /* name of index file for contextual    */
char       fileidfilename[200];     /* name of file id data base            */
char       lolafilename[200];       /* name of lola listing of files        */

char       logfilename[200]="";     /* log file name for merging process    */
char       lolafilename[200]="";    /* lola file name for output lola       */

int        post = 0;                /* post segments are not saved          */
int        pre  = 0;                /* pre segments are saved               */

int        destroy = 0;             /* destroy action / replace lola files  */
char       dbname[200]="";          /* name of data base to search          */


/*
 * Usage()
 *
 * print out the usage parameters of the find phone program
 *
 */

void Usage()
{
  fprintf(stderr, "Usage: mergelola [options] {dbname}\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "dbname: name of data base to search\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "\t -f [log filename] Sets the input log file name   [dbname.log]\n");
  fprintf(stderr, "\t -l [lola filename] Sets the lola input file      [dbname.lola]\n");
  /*
  fprintf(stderr, "\t -d DESTROY -- Replace original lola files        [DO NOT]\n");
  */
  fprintf(stderr, "\n");
  fprintf(stderr, "\t -h Help. This help message\n");
  exit(1);
}


/*
 * get_comline (argc, argv)
 *
 * read an interpret the command line parameters
 *
 * argc, argv (in): command line variables
 *
 */

void get_comline (int argc, char **argv)
{
  int c;
  extern int  optind;
  extern char *optarg;

  while ( (c = getopt (argc, argv, "f:l:dh")) != -1) {
    switch (c) 
      {
      case 'f': /* log file name */
	strcpy(logfilename, optarg);
	strcat(logfilename, LOGEXT);
	break;

      case 'l':
	strcpy(lolafilename, optarg);
	strcat(lolafilename, LOLAEXT);
	break;

      case 'd':
	destroy = 1;
	break;

      case 'h':
      default:
	Usage();
      }
  }

  if ((argc - optind) != 1)
    Usage();

  strcpy(dbname, argv[optind]);

  /* create default options */
  if (strlen(logfilename)==0) {
    strcpy(logfilename, dbname);
    strcat(logfilename, LOGEXT);
  }
  if (strlen(lolafilename)==0) {
    strcpy(lolafilename, dbname);
    strcat(lolafilename, LOLAEXT);
  }
  
  /* create data base and index file names */
  strcpy(contextfilename, dbname);
  strcat(contextfilename, CONTEXTBASE);

  strcpy(indexfilename, dbname);
  strcat(indexfilename, INDEX);
  
  strcpy(fileidfilename, dbname);
  strcat(fileidfilename, IDBASE);
}


/*
 * int Delete_Context_Record(contextfile, int recordno)
 *
 * This procedure deletes a specified record no. After deleting the
 * record it updates the links of the chains in the record
 *
 * contextfile (in): pointer to file stream
 * recordno    (in): number of record in contextual data base to delete
 *
 * on exit returns a pointer to the next record in the data base
 * or on error return -1;
 *
 */

int Delete_Context_Record(FILE *contextfile, int recordno)
{
  ContextRec OneContext;
  ContextRec PrevLink, NextLink;          /* updated context records       */
  int        prevlink, nextlink;          /* vertical chain links          */
  int        ph_prevlink, ph_nextlink;    /* horizontal chain links        */
  int        nextrecord;                  /* next record no                */

  if (Read_Context_Record(contextfile, &OneContext, recordno) < 0) {
    fprintf(stderr, "Delete_Context_Record -> 1: %s\n", ErrorString);
    return (-1);
  }
  
  /* reset vertical chain */
#ifdef DEBUG
  printf("Deleting record from vertical chain\n");
#endif
  
  prevlink = OneContext.prevlink;
  if (prevlink >= 0) {
    if (Read_Context_Record(contextfile, &PrevLink, prevlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 2: %s\n", ErrorString);
      return (-1);
    }
    PrevLink.nextlink = OneContext.nextlink;
    if (Write_Context_Record(contextfile, PrevLink, prevlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 3: %s\n", ErrorString);
      return(-1);
    }
  }

  nextlink = OneContext.nextlink;
  if (nextlink >= 0) {
    if (Read_Context_Record(contextfile, &NextLink, nextlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 4: %s\n", ErrorString);
      return (-1);
    }
    NextLink.prevlink = OneContext.prevlink;
    if (Write_Context_Record(contextfile, NextLink, nextlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 5: %s\n", ErrorString);
      return (-1);
    }
  } else {
    PreviousLink[OneContext.ph_curr-1] = OneContext.prevlink;
    PhonemeIndex[OneContext.ph_curr-1].endrec = OneContext.prevlink;
  }

  /* reset horizontal chain */
  nextrecord  = OneContext.ph_nextlink;
  ph_prevlink = OneContext.ph_prevlink;
  if (ph_prevlink >= 0) {
    if (Read_Context_Record(contextfile, &PrevLink, ph_prevlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 6: %s\n", ErrorString);
      return (-1);
    }
    PrevLink.ph_nextlink = OneContext.ph_nextlink;
    PrevLink.ph_next     = OneContext.ph_next;
    PrevLink.next_end    = OneContext.endloc;
    if (Write_Context_Record(contextfile, PrevLink, ph_prevlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 7: %s\n", ErrorString);
      return (-1);
    }
  }

  ph_nextlink = OneContext.ph_nextlink;
  if (ph_nextlink >= 0) {
    if (Read_Context_Record(contextfile, &NextLink, ph_nextlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 8: %s\n", ErrorString);
      return (-1);
    }
    NextLink.ph_prevlink = OneContext.ph_prevlink;
    NextLink.ph_prev     = OneContext.ph_prev;
    NextLink.prev_start  = OneContext.prev_start;
    NextLink.startloc    = OneContext.startloc;
    if (Write_Context_Record(contextfile, NextLink, ph_nextlink) < 0) {
      fprintf(stderr, "Delete_Context_Record -> 9: %s\n", ErrorString);
      return (-1);
    }
  }
  
  OneContext.from_file_id = -1;
  OneContext.prevlink = -1;
  OneContext.nextlink = -1;
  OneContext.ph_nextlink = -1;
  OneContext.ph_prevlink = -1;

  if (Write_Context_Record(contextfile, OneContext, recordno) < 0) {
    fprintf(stderr, "Delete_Context_Record -> 10: %s\n", ErrorString);
    return (-1);
  }

  return (nextrecord);
}


/*
 * int Insert_Context_Record(contextfile, recordno, phoneme,
 *                           length)
 *
 * This procedure inserts a context record in the database, before the
 * record no specified.
 *
 * contextfile (in): pointer to contextual data base file stream
 * recordno    (in): insert record before this record no
 * phoneme     (in): the label of the center phoneme
 * length      (in): length of speech (num units)
 *
 * On exit returns the number of the new record. Always inserts
 * records at the end of the file, else returns -1 on error.
 *
 */

int Insert_Context_Record(FILE *contextfile, 
			  int recordno, char *phoneme, int length)
{
  ContextRec PrevLink, NextLink, Current; /* contextual record links     */
  int        ph_prevlink, ph_nextlink;    /* previous link records       */
  int        newrecord;                   /* name of new record number   */
  char       ph_new;                      /* index of new phoneme        */

  /* get info for requested insertion record */
  if (Read_Context_Record(contextfile, &NextLink, recordno) < 0) {
    fprintf(stderr, "Insert_Context_Record -> 1: %s\n", ErrorString);
    return (-1);
  }
  
  ph_prevlink = NextLink.ph_prevlink;      /* previouslink record no      */
  
  /* new record number */
  fseek (contextfile, 0, SEEK_END);
  newrecord = ftell(contextfile)/(sizeof(ContextRec));
#ifdef DEBUG
  printf("Insert: %d -- %d -- %d\n", ph_prevlink, newrecord, recordno);
#endif

  Current.from_file_id = NextLink.from_file_id;
  Current.ph_nextlink = recordno;
  Current.ph_next     = NextLink.ph_curr;
  Current.next_end    = NextLink.endloc;

  if ((ph_new=Retrieve_Phoneme_Index(phoneme, &PhonemeIndex, &PreviousLink))<0)
    {
      fprintf(stderr, "Insert_Context_Record -> 2: %s\n", ErrorString);
      return (-1);
    }
  
#ifdef DEBUG
  printf("Insert Phoneme: %s  %d -- %d\n", phoneme, ph_new, length);
#endif
  
  if (ph_prevlink >=0) {                  /* information previous context */
    if (Read_Context_Record(contextfile, &PrevLink, ph_prevlink) < 0) {
      fprintf(stderr, "Insert_Context_Record -> 3: %s\n", ErrorString);
      return (-1);
    }
    
    Current.ph_prevlink = ph_prevlink;
    Current.ph_prev     = PrevLink.ph_curr;
    Current.prev_start  = PrevLink.startloc;
    Current.startloc    = PrevLink.endloc;
    Current.endloc      = Current.startloc + length;

    PrevLink.ph_next    = ph_new;
    PrevLink.next_end   = Current.endloc;
    PrevLink.ph_nextlink= newrecord;

    if (Write_Context_Record(contextfile, PrevLink, ph_prevlink) < 0) {
      fprintf(stderr, "Insert_Context_Record -> 4: %s\n", ErrorString);
      return (-1);
    }
  }

  NextLink.ph_prev    = ph_new;
  NextLink.prev_start = Current.startloc;
  NextLink.startloc   = Current.endloc;
  NextLink.ph_prevlink= newrecord;

  if (Write_Context_Record(contextfile, NextLink, recordno) < 0) {
    fprintf(stderr, "Insert_Context_Record -> 5: %s\n", ErrorString);
    return (-1);
  }

  Current.nextlink = -1;
  if ((Current.prevlink = PreviousLink[ph_new-1]) >= 0) {
#ifdef DEBUG
    printf("update: %d -- %d\n", Current.prevlink, ph_new);
#endif
    if (Update_Context_Record(contextfile, Current.prevlink, newrecord) < 0) {
      fprintf(stderr, "Insert_Context_Record -> 5: %s\n", ErrorString);
      return (-1);
    }
  }

  PreviousLink[ph_new-1] = newrecord;
  PhonemeIndex[ph_new-1].endrec = newrecord;

  Current.headorder = machineorder;
  Current.ph_curr   = ph_new;
  fseek(contextfile, 0, SEEK_END);
  
  /* check for record insertion violation */
  if (newrecord != (ftell(contextfile)/sizeof(ContextRec)))
    printf("WARNING RECORD VIOLATION -- DATABASE CORRUPT\n");

  fwrite(&Current, sizeof(ContextRec), 1, contextfile);

  return (newrecord);
}


/*
 * Read_String_List(filestream, strlist, numstr)
 *
 * Read the search string list from log file. Assume memory is 
 * allocated
 *
 * filestream (in): file stream (log file)
 * strlist    (in): search string list
 * numstr     (in): number of strings in listing
 *
 * on error returns -1, else +1
 *
 */

int Read_String_List(FILE *filestream, char **strlist, int numstr)
{
  int i;

  for (i=0; i<numstr; i++) {
    if (fscanf(filestream, "%s", strlist[i])<=0) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not parse log file";
      return (-1);
    }
  }
}


/*
 * Export_Context_Changes(contextfile, recordno)
 *
 * This procedure uses the current contextual phoneme record
 * to edit the prev field in the following contextual phoneme
 * record in the data base
 *
 * contextfile (in): pointer to contextual file stream
 * recordno    (in): current record no
 *
 * On error returns -1, else +1
 *
 */

int Export_Context_Changes(FILE *contextfile, int recordno)
{
  ContextRec OneContext;                   /* current data base entry   */
  ContextRec Ph_NextLink, Ph_PrevLink;     /* next record in data base  */
  int        ph_nextlink, ph_prevlink;     /* record no of nextlink     */
  int        flag;                         /* save to disc flag         */

  if (Read_Context_Record(contextfile, &OneContext, recordno) < 0) {
    fprintf(stderr, "Export_Context_Changes -> 1: %s\n", ErrorString);
    return (-1);
  }
  
  ph_nextlink = OneContext.ph_nextlink;
  if (ph_nextlink >= 0) {                  /* next record exist/ change  */
    if (Read_Context_Record(contextfile, &Ph_NextLink, ph_nextlink) < 0) {
      fprintf(stderr, "Export_Context_Changes -> 2: %s\n", ErrorString);
      return (-1);
    }
    flag = 0;
    if (Ph_NextLink.prev_start != OneContext.startloc) {
      Ph_NextLink.prev_start = OneContext.startloc;
      flag = 1;
    }
    if (Ph_NextLink.ph_prev != OneContext.ph_curr) {
      Ph_NextLink.ph_prev = OneContext.ph_curr;
      flag = 1;
    }
    if (Ph_NextLink.startloc != OneContext.endloc) {
      Ph_NextLink.startloc   = OneContext.endloc;
      flag = 1;
    }
    
    if (flag) {
#ifdef DEBUG
      printf("changing next link borders...\n");
#endif
      if (Write_Context_Record(contextfile, Ph_NextLink, ph_nextlink) < 0) {
	fprintf(stderr, "Export_Context_Changes -> 3: %s\n", ErrorString);
	return (-1);
      }
    }
  }

  ph_prevlink = OneContext.ph_prevlink;
  if (ph_prevlink >= 0) {                  /* previous record exist      */
    if (Read_Context_Record(contextfile, &Ph_PrevLink, ph_prevlink) < 0) {
      fprintf(stderr, "Export_Context_Changes -> 4: %s\n", ErrorString);
      return (-1);
    }
    
    flag = 0;
    if (Ph_PrevLink.next_end != OneContext.endloc) {
      Ph_PrevLink.next_end   = OneContext.endloc;
      flag = 1;
    }
    if (Ph_PrevLink.ph_next != OneContext.ph_curr) {
      Ph_PrevLink.ph_next    = OneContext.ph_curr;
      flag = 1;
    }
    
    if (flag) {
#ifdef DEBUG
      printf("changing previous link borders...\n");
#endif
      if (Write_Context_Record(contextfile, Ph_PrevLink, ph_prevlink) < 0) {
	fprintf(stderr, "Export_Context_Changes -> 5: %s\n", ErrorString);
	return (-1);
      }
    }
  }

  return (1);
}


/*
 * Edit_Context_Record(contextfile, ContextPhon, recordno)
 * 
 * This procedure edits/changes the specified record no in the
 * context file, thus updating the vertical and horizontal links.
 *
 * contextfile (in): pointer to the context file pointer
 * ContextPhon (in): Contextual record to edit
 * recordno    (in): record no to changes
 *
 * On error returns -1 else return +1
 *
 */

int Edit_Context_Record(FILE *contextfile, ContextRec ContextPhon, 
			int recordno)
{
  ContextRec OneContext;                /* current data base entry     */
  ContextRec PrevLink, NextLink;        /* previous and next links     */
  int        prevlink, nextlink;        /* record number of links      */
  char       ph_curr;                   /* new current phoneme index   */
  
  /* read in current phoneme for possible vertical chain edit */
  ph_curr = ContextPhon.ph_curr;
  if (Read_Context_Record(contextfile, &OneContext, recordno) < 0) {
    fprintf(stderr, "Edit_Context_Record -> 1: %s\n", ErrorString);
    return (-1);
  }
  
  if (OneContext.ph_curr != ph_curr) {
#ifdef DEBUG
    printf("changing vertical links\n");
#endif

    prevlink = OneContext.prevlink;
    if (prevlink >= 0) {
      if (Read_Context_Record(contextfile, &PrevLink, prevlink) < 0) {
	fprintf(stderr, "Edit_Context_Record -> 2: %s\n", ErrorString);
	return (-1);
      }
      PrevLink.nextlink = OneContext.nextlink;
      if (Write_Context_Record(contextfile, PrevLink, prevlink) < 0) {
	fprintf(stderr, "Edit_Context_Record -> 3: %s\n", ErrorString);
	return (-1);
      }
    }

    nextlink = OneContext.nextlink;
    if (nextlink >= 0) {
      if (Read_Context_Record(contextfile, &NextLink, nextlink) < 0) {
	fprintf(stderr, "Edit_Context_Record -> 4: %s\n", ErrorString);
	return (-1);
      }
      NextLink.prevlink = OneContext.prevlink;
      if (Write_Context_Record(contextfile, NextLink, nextlink) < 0) {
	fprintf(stderr, "Edit_Context_Record -> 5: %s\n", ErrorString);
	return (-1);
      }
    } else {
      PreviousLink[OneContext.ph_curr-1] = OneContext.prevlink;
      PhonemeIndex[OneContext.ph_curr-1].endrec = OneContext.prevlink;
    }

    if (recordno == PhonemeIndex[OneContext.ph_curr-1].startrec) {
      PhonemeIndex[OneContext.ph_curr-1].startrec = OneContext.nextlink;
#ifdef DEBUG
      printf("start of chain: %s  -- %d\n", 
	     PhonemeIndex[OneContext.ph_curr-1].phoneme,
	     OneContext.nextlink);
#endif
    }

    OneContext.nextlink = -1;
    if ((OneContext.prevlink = PreviousLink[ph_curr-1]) >= 0) {
#ifdef DEBUG
      printf("update: %d  --  %d\n", OneContext.prevlink, ph_curr);
#endif
      if (Update_Context_Record(contextfile,OneContext.prevlink, recordno)<0) {
	fprintf(stderr, "Edit_Context_Record -> 6: %s\n", ErrorString);
	return (-1);
      }
    }

    PreviousLink[ph_curr-1] = recordno;
    PhonemeIndex[ph_curr-1].endrec = recordno;
    if (PhonemeIndex[ph_curr-1].startrec == -1) 
      PhonemeIndex[ph_curr-1].startrec = recordno;
  }

  /* update files in context database record */
  OneContext.ph_curr   = ph_curr;
  OneContext.startloc  = ContextPhon.startloc;
  OneContext.endloc    = ContextPhon.endloc;
  OneContext.ph_prev   = ContextPhon.ph_prev;
  OneContext.prev_start= ContextPhon.prev_start;
  OneContext.ph_next   = ContextPhon.ph_next;
  OneContext.next_end  = ContextPhon.next_end;

#ifdef DEBUG
  printf("new record: %s %s %s\n",
	 PhonemeIndex[OneContext.ph_prev-1].phoneme,
	 PhonemeIndex[OneContext.ph_curr-1].phoneme,
	 PhonemeIndex[OneContext.ph_next-1].phoneme);
#endif
  if (Write_Context_Record(contextfile, OneContext, recordno) < 0) {
    fprintf(stderr, "Edit_Context_Record -> 7: %s\n", ErrorString);
    return (-1);
  }

  return (1);
}


/*
 * Change_Context_Segment(contextfile, recordno, 
 *                        startsegment, endsegment, ms)
 *
 * This procedure edits the data base on a the segment list of a
 * current search and the indicated starting record no.
 *
 * contextfile  (in): pointer to contextual file stream
 * recordno     (in): start record number of editing process
 * startsegment (in): start segment of of context lola search
 * endsegment   (in): end segment of context lola search
 * ms           (in): millisenconds of frame
 *
 * On any major error detected, Change_Context_Segment aborts
 * since it can not recover from these errors. Propagates -1
 * on error to calling procedures.
 *
 * Success is returns as +1
 *
 */

int  Change_Context_Segment(FILE *contextfile, 
			    int recordno, SegmentList *startsegment, 
			    SegmentList *endsegment, float ms)
{
  int  i, j, count, flag;                 /* loop counter variables      */
  int  numsegments;                       /* number of segments in list  */
  int  record;                            /* current record to edit      */
  int  delete, insert;                    /* type of record              */
  char ph_curr, ph_next;                  /* phoneme value in indexfile  */
  int  endloc, next_end;                  /* boundary values             */
  SegmentList *current;                   /* current pointer in segment  */
  ContextRec  OneContext;                 /* one record in contextual DB */
  
  /* number of feasible records to change */ 
  if ((pre) && (post)) 
    count = 3;
  else if ((pre) || (post))
    count = 2;
  else
    count = 1;

  /* count number of segment in change list */
  numsegments = 0;
  current = startsegment;
  do {
    current = current->sl_forward;
    numsegments++;
  } while (current != endsegment);
#ifdef DEBUG
  printf("number of segments ----> : %d\n", numsegments);
#endif

  /* edit data base */
  insert = delete = 0;

  i = 0;
  flag  = 1;
  current = startsegment;
  record  = recordno;

  /* change up to last segment */
  do {
    if (current == endsegment) flag = 0;    
    
    if (Read_Context_Record (contextfile, &OneContext, record) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 1: %s\n", ErrorString);
      return (-1);
    }

#ifdef DEBUG
    printf("%s -- %s\n", PhonemeIndex[OneContext.ph_curr-1].phoneme,
	   current->sl_label);
#endif

    if ((ph_curr = Retrieve_Phoneme_Index(current->sl_label, 
				     &PhonemeIndex, &PreviousLink))<0) {
      fprintf(stderr, "Change_Context_Segment -> 2: %s\n", ErrorString);
      return (-1);
    }
      
    if (ph_curr != OneContext.ph_curr) {  /* center context phoneme       */
#ifdef DEBUG
      printf("numsegments: %d -- count : %d\n", numsegments, count);
#endif
      if (numsegments>count-1) {           /* insert record in data base  */
	if ((record = Insert_Context_Record(contextfile, record, 
	   current->sl_label, (current->sl_end - current->sl_begin))) < 0) {
	  fprintf(stderr, "Change_Context_Segment -> 3: %s\n", ErrorString);
	  return (-1);
	}
	
	if (Read_Context_Record(contextfile, &OneContext, record) < 0) {
	  fprintf(stderr, "Change_Context_Segment -> 4: %s\n", ErrorString);
	  return (-1);
	}
	
#ifdef DEBUG
	printf("inserted ---> %d  %d\n", OneContext.ph_curr, ph_curr);
#endif
	count++;
      } else if (numsegments<count-1) {   /* delete record in data base   */
#ifdef DEBUG
	printf("Deleting record on the fly\n");
#endif
	if ((record = Delete_Context_Record(contextfile, record)) < 0) {
	  fprintf(stderr, "Change_Context_Segment -> 5: %s\n", ErrorString);
	  return (-1);
	}
	
	if (Read_Context_Record(contextfile, &OneContext, record) < 0) {
	  fprintf(stderr, "Change_Context_Segment -> 6: %s\n", ErrorString);
	  return (-1);
	}
	
#ifdef DEBUG
	printf("deleted ----> %d  %d\n", OneContext.ph_curr, ph_curr);
#endif
	OneContext.ph_curr = ph_curr;
	count--;
      }
				       
      OneContext.ph_curr = ph_curr;
    }

    endloc = OneContext.startloc + current->sl_end - current->sl_begin;
    if (endloc != OneContext.endloc)      /* end location of phoneme      */
      OneContext.endloc = endloc;

    if (flag) {                          /* do not do at end of segment   */
      if ((ph_next = Retrieve_Phoneme_Index(current->sl_forward->sl_label, 
				    &PhonemeIndex, &PreviousLink)) < 0) {
	fprintf(stderr, "Change_Context_Segment -> 7: %s\n", ErrorString);
	return (-1);
      }
      
      if (ph_next != OneContext.ph_next)
	OneContext.ph_next = ph_next;

      next_end = endloc + current->sl_forward->sl_end - 
	current->sl_forward->sl_begin;
      if (next_end != OneContext.next_end)
	OneContext.next_end = next_end;

      current = current->sl_forward;
    }

    if (Edit_Context_Record(contextfile, OneContext, record) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 8: %s\n", ErrorString);
      return (-1);
    }

    if (Export_Context_Changes(contextfile, record) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 9: %s\n", ErrorString);
      return (-1);
    }

    record  = OneContext.ph_nextlink;
    i++;
  } while ((current != endsegment) || (flag));

  /* if there are still record to delete */
  while (numsegments<count-1) {
#ifdef DEBUG
    printf("Deleting records at batch\n");
#endif
    if ((record = Delete_Context_Record (contextfile, record)) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 10: %s\n", ErrorString);
      return (-1);
    }

    if (Read_Context_Record(contextfile, &OneContext, record) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 11: %s\n", ErrorString);
      return (-1);
    }

    if (Export_Context_Changes(contextfile, record) < 0) {
      fprintf(stderr, "Change_Context_Segment -> 12: %s\n", ErrorString);
      return (-1);
    }
    
#ifdef DEBUG
    printf("deleted -----> %s\n", PhonemeIndex[OneContext.ph_curr-1].phoneme);
#endif
    count--;
  }
}


/*
 * int Validate_LogFile(logfile, contextlola)
 *
 * This procedure validates the contextual file, with the logfile
 * Any changes in the segment length, and missing record are
 * not validated.
 *
 * logfile     (in): pointer to file stream
 * contextlola (in): pointert to segment listing
 *
 * returns 0 as non-valid and 1, as valid
 *
 */

int Validate_LogFile (FILE *logfile, SegmentList *contextlola)
{
  int         recordno;              /* record number in context base    */
  int         where;                 /* where am im in data base         */
  int         valid;                 /* valid flag                       */
  int         length_1, length_2;    /* length of context segment        */
  SegmentList *centerlola;           /* current pointer in segment list  */
  SegmentList *startsegment;         /* start segment in list            */
  SegmentList *endsegment;           /* end segment in list              */
  ContextRec  OneContext;            /* One Context record in data base  */

  where = ftell (logfile);

  centerlola = contextlola->sl_forward;

  /* loop through log file and validate   */
  valid = 1;
  while (!(feof(logfile))) {
    while (centerlola->sl_forward != NULL) {
      if (fscanf(logfile, "%d\n", &recordno) == EOF) {
	printf("file length mismacth\n");
	valid = 0;
	break;
      }
#ifdef DEBUG
      printf("record --> %d\n", recordno);
#endif

      if (Read_Context_Record(contextfile, &OneContext, recordno) < 0) {
	fprintf(stderr, "Validate_LogFile -> 1: %s\n", ErrorString);
	return (0);
      }

      if ((pre) && (post)) 
	length_1 = OneContext.next_end - OneContext.prev_start;
      else if (pre)
	length_1 = OneContext.endloc - OneContext.prev_start;
      else if (post)
	length_1 = OneContext.next_end - OneContext.startloc;
      else
	length_1 = OneContext.endloc - OneContext.endloc;

#ifdef DEBUG
      printf("%d -- ", recordno);
#endif
      
      startsegment = centerlola->sl_forward;      /* start of lola search   */
      while (strcmp(centerlola->sl_forward->sl_label, "notting") != 0) {
	centerlola = centerlola->sl_forward;
#ifdef DEBUG
	printf("%s ", centerlola->sl_label);
#endif
      }
      endsegment   = centerlola;                 /* end segment lola search */
      length_2 = endsegment->sl_end - startsegment->sl_begin;

#ifdef DEBUG
      printf("%d -- %d\n", length_1, length_2);
#endif
      valid = valid && (length_1 == length_2);

      centerlola = centerlola->sl_forward;
    }
    fscanf(logfile, "%d\n", &recordno);
    if (!(feof(logfile))) {
      valid = 0;
      break;
    }
  }

  fseek(logfile, where, SEEK_SET);
  return (valid);
}

 

/*
 * int Merge_Lola(contextfile, idfile, lolafilename, logfilename)
 *
 * This procedure merges the contextlola segment list with the
 * data base using the logfilename as reference. Assumes only the
 * center segment needs changing.
 *
 * contextfile (in): pointer to contextual data base file stream
 * idfile      (in): pointer to id file data base file stream
 * lolafilename(in): context lola segment list (all changings)
 * logfilename (in): name of search output log
 *
 * On error returns -1, else +1
 *
 */

int Merge_Lola(FILE *contextfile, FILE *idfile, 
	       char *lolafilename, char *logfilename)
{
  int i, j, k, flag;
  
  SegmentList *contextlola;                 /* changing in data base segment */
  SegmentList *startsegment;                /* start of lola search segment  */
  SegmentList *endsegment;                  /* end of lola search segment    */
  SegmentList *centerlola;                  /* current center context phon   */
  ContextRec  OneContext;                   /* context record in context DB  */
  FileIDRec   OneWave;                      /* from file name in fileid DB   */
  float       ms;                           /* segmentation accuracy         */
  char        **prephon;                    /* pre contextual phoneme        */
  char        **centerphon;                 /* center contextual phoneme     */
  char        **postphon;                   /* post contextual phoneme       */
  int         prenum;                       /* number of pre contextual phon */
  int         centernum;                    /* number of center phonemes     */
  int         postnum;                      /* number of post contextual phon*/
  int         recordno;                     /* record number of context      */
  FILE        *logfile;                     /* file stream for search log    */

  /* allocate memory for searching phoneme listings */
  if (!(prephon = (char **) Alloc2d(255, 20, sizeof(char)))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "prehon --> Alloc2d failed";
    fprintf(stderr, "Merge_Lola -> 1: %s\n", ErrorString);
    return (-1);
  }

  if (!(centerphon = (char **) Alloc2d(255, 20, sizeof(char)))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "centerphon --> Alloc2d failed";
    fprintf(stderr, "Merge_Lola -> 2: %s\n", ErrorString);
    return (-1);
  }

  if (!(postphon = (char **) Alloc2d(255, 10, sizeof(char)))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "postphon -> Alloc2d failed";
    fprintf(stderr, "Merge_Lola -> 3: %s\n", ErrorString);
    return (-1);
  }

  /* read in segment lest for merging process */
  if (ReadSegmentList (lolafilename, &contextlola, &ms) < 0) {
    fprintf(stderr, "Merge_Lola -> 3: %s -- %s\n", lolafilename,
	    ErrorString);
    return (-1);
  }

  /* open log file an start extracting information */
  if (!(logfile = fopen(logfilename, "r"))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not open log file";
    fprintf(stderr, "Merge_Lola -> 4: %s\n", ErrorString);
  }

  fscanf(logfile, "%d  %d  %d\n", &prenum, &centernum, &postnum);
  fscanf(logfile, "%d  %d\n", &pre, &post);

  if (Read_String_List(logfile, prephon, prenum)< 0) {
    fprintf(stderr, "Merge_Lola -> 5: %s\n", ErrorString);
    return (-1);
  }
  if (Read_String_List(logfile, centerphon, centernum) < 0) {
    fprintf(stderr, "Merge_Lola -> 6: %s\n", ErrorString);
    return (-1);
  }
  if (Read_String_List(logfile, postphon, postnum) < 0) {
    fprintf(stderr, "Merge_Lola -> 7: %s\n", ErrorString);
    return (-1);
  }
  
  /* start editing data base */
  centerlola = contextlola->sl_forward;

  if (!(Validate_LogFile(logfile, contextlola))) {
    printf("Mismatch between log file and lola file\n");
    fclose(logfile);
    FreeSegmentList(contextlola);
    return(-1);
  }
  
  while (!(feof(logfile))) {
    while (centerlola->sl_forward != NULL) {
      fscanf(logfile, "%d\n", &recordno);
      if (Read_Context_Record(contextfile, &OneContext, recordno) < 0) {
	fprintf(stderr, "Merge_Lola -> 8: %s\n", ErrorString);
	return (-1);
      }
      
#ifdef DEBUG
      printf("%d -- ", recordno);
#endif
      flag = 0;
      
      startsegment = centerlola->sl_forward;      /* start of lola search   */
      while (strcmp(centerlola->sl_forward->sl_label, "notting") != 0) {
	centerlola = centerlola->sl_forward;
#ifdef DEBUG
	printf("%s ", centerlola->sl_label);
#endif
      }
      endsegment   = centerlola;                 /* end segment lola search */

      if (pre)                                   /* start recordno of edit  */
	recordno = OneContext.ph_prevlink;
#ifdef DEBUG
      printf("\n");
#endif
      if (Change_Context_Segment(contextfile,recordno,startsegment,endsegment,ms) < 0) {
	fprintf(stderr, "Merge_Lola -> 9: %s\n", ErrorString);
	return (-1);
      }

      centerlola = centerlola->sl_forward;
    }
  }

  FreeSegmentList(contextlola);
  fclose(logfile);
  
  return (1);
}


/*
 * master magician code
 *
 */

main (int argc, char **argv)
{
  float       ms;
  SegmentList *contextlola;                  /* contextual lola file      */
  
  /* process command line directives */
  get_comline (argc, argv);

  machineorder = LittleIndian();

  /* do data base setup */
  if (Read_Index_File (indexfilename, &PhonemeIndex, &PreviousLink) < 0) {
    fprintf(stderr, "Read_Index_File: %s\n", ErrorString);
    exit(1);
  }

  if (Open_Contextual_Base (&contextfile, contextfilename) < 0) {
    fprintf(stderr, "Open_Contextual_Base: %s\n", ErrorString);
    free ((char *) PhonemeIndex);
    free ((char *) PreviousLink);
    exit(1);
  }

  if (Open_FileID_Base(&idfile, fileidfilename) < 0) {
    fprintf(stderr, "Open_FileID_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    Close_Contextual_Base(&contextfile);
    exit(1);
  }

  /* merge information with lola file */
  CREATE_DATABASE = 1;
  if (Merge_Lola (contextfile, idfile, lolafilename, logfilename) < 0) {
    fprintf(stderr, "Merge_Lola : %s\n", ErrorString);
    return (-1);
  } else {
  
    /* do system cleanup before exiting */
    if (Write_Index_File (indexfilename, PhonemeIndex) < 0) 
      fprintf(stderr, "Write_Index_File: %s\n", ErrorString);
  }
    
  Close_Contextual_Base (&contextfile);
  Close_FileID_Base(&idfile);

  free((char *) PhonemeIndex);
  free((char *) PreviousLink);
}
