/*
 * $Id: makedft.c,v 1.2 1993/10/06 22:46:11 johans Exp $
 *
 * makedft.c
 *
 */

/* Standard C library include file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/* Speech Libary include file directives */
#include <speech.h>


/* Local module header declarations */
static int WriteDFT(FILE *fp, short **dft, int rows, int cols, 
		    float winsize_ms, float inc_ms, float rate, 
		    double start_time );


/* Global constant defintions */
#define DEF_WINDOW_SIZE		(10.0)	/* in ms */
#define DEF_INCREMENT		(3.0)	/* in ms */
#define DEF_COEFFS		(64)

#define DEF_ALPHA		(0.94)

#define TRUE		(1)
#define FALSE		(0)


/* external variable declaration */
extern int errno, sys_nerr;
extern char *sys_errlist[];


/* global variable declarations */
int  numcoeffs=64;                       /* number of dft coefficients   */
double start_time = 0.0;                 /* starting time in ms          */
double end_time = 0.0;                   /* ending time                  */
double winsize_ms = 10.0;                /* window size in milliseconds  */
double inc_ms = 3.0;                     /* window increment in millisec */
float alpha = 0.94;                      /* preemphasis filter factor    */


/*
 * int CreateDFT(InFileName, OutFileName)
 *
 * Given the name of the incoming speech signal, calculates th
 * spectogram for display purposes in LYRE or AUTO_LYRE
 *
 * InFileName (in): name of input speech file
 * OutFileName(in): name of output spectrogram or DFT File
 *
 * On error returns -1, else 1
 *
 */

int CreateDFT(char *InFileName, char *OutFileName)
{
  short **dft;
  short *data;
  int start, end;
  int len;

  FILE *fp;
  int winsize;
  float rate;
  int inc;
  int numframes;
  int c;
  int i;

  alpha = 0.0;
  if( ( len = SpeechRead( InFileName, &data, &rate ) )  < 0 ) 
    return (-1);
  
  /*
   * Find increment and window size
   */

  inc = (int) (((double)(rate/1000.0)) * inc_ms);
  if(((double)inc) != (((double)(rate/1000.0)) * inc_ms)) {
    fprintf( stderr, "makedft: increment not and \n" );
    free ((char *) data);
    return (-1);
  }

  winsize = (int) (((double)(rate/1000.0)) * winsize_ms);
  if(((double)winsize) != 
     (((double)(rate/1000.0))*winsize_ms)){
    fprintf( stderr, "makedft: window size not and \n" );
    free ((char *) data);
    return (-1);
  }

  if( start_time == 0.0 ) {
    start = 0;
  } else {
    start = (int) (((double)(rate/1000.0)) * start_time);
  }
  
  if( end_time == 0.0 ) {
    end = len;
  } else {
    end = (int) (((double)(rate/1000.0)) * end_time);
  }
  
  if( end > len ) {
    end = len;
  }

  len = end - start;

  dft = ComputeDFT( data, end, start, winsize, inc, numcoeffs, alpha, 
		   &numframes);
  if (dft == NULL) {
    free ((char *) data);
    return (-1);
  }
  
  if( strcmp(OutFileName, "-") == 0 ) {
    fp = stdout;
  } else {
    if( ( fp = fopen( OutFileName, "w" ) ) == NULL ) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not create DFT File";
      free((char *) data);
      Free2d((char **) dft);
      return (-1);
    }
  }

  if (WriteDFT( fp, dft, numcoeffs, len / inc, winsize_ms, 
	   inc_ms, rate, start_time ) < 0) {
    free ((char *) data);
    Free2d((char **) dft);
    return (-1);
  }
  
  fclose( fp );
  free((char *) data);
  Free2d((char **) dft);

  return (1);
}


static int WriteDFT(FILE *fp, short **dft, int rows, int cols, 
		    float winsize_ms, float inc_ms, float rate, 
		    double start_time)
{
  AsciiHdr header[20];
  float *hz;
  int tmp;
  float ftmp;
  int i;

  hz = (float *) calloc(rows, sizeof(float));

  if( hz == NULL ) {
    ErrorString =  "WriteDFT: hz --> malloc failed";
    return (-1);
  }

  for( i = 0; i < rows; i++ ) {
    *(hz+i) = (((float)i)/((float)(rows-1))) * 
      (rate/2.0);
  }

  i = 0;
  header[i].ah_key = "Rows";
  header[i].ah_vlen = 1;
  header[i++].ah_value = IntArray2StrArray( &rows, 1 );

  header[i].ah_key = "Columns";
  header[i].ah_vlen = 1;
  header[i++].ah_value = IntArray2StrArray( &cols, 1 );

  header[i].ah_key = "Type";
  header[i].ah_vlen = 1;
  header[i++].ah_value = Str2StrArray( "short" );

  header[i].ah_key = "RowLabels";
  header[i].ah_vlen = rows;
  header[i++].ah_value = FloatArray2StrArray( hz, rows );

  ftmp = rate;
  header[i].ah_key = "SamplesPerSecond";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp =  inc_ms;
  header[i].ah_key = "Increment";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp = winsize_ms;
  header[i].ah_key = "WindowWidth";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp = (float)start_time;
  header[i].ah_key = "StartTime";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  header[i].ah_key = NULL;
  header[i].ah_vlen = 0;
  header[i++].ah_value = NULL;

  if( fTdatWrite( fp, (char **) dft, header ) < 0 ) 
    return (-1);

  return (1);
}
