/*
 * buildbase.c
 *
 * This program accepts as input a text file containing a list
 * of phoneme labeled lola files. From this file it creates a
 * complete cross reference phonem data base, for search and
 * editing purposes.
 *
 * The data base features include
 *  1. Create Database (this program)
 *  2. Add lola files to data bases
 *  2. Search Database
 *  3. Edit (insert and delete records)
 *  4. Pack (restructure database for optimal searching)
 *  
 */

/* Standard C file include file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Speech tools include file directives */
#include <speech.h>

/* data base include file directives */
#include <phonebase.h>


/* external variable declaration */
extern int errno, sys_nerr;
extern char *sys_errlist[];


/* Global variable declaration */
int        CREATE_DATABASE=1;       /* create data base flag                */
int        *PreviousLink;           /* array of most recent previous links  */
PhonemeRec *PhonemeIndex;           /* phoneme for index file               */

FILE       *contextfile;            /* contextual phoneme data base         */
FILE       *idfile;                 /* from file id data base               */
FILE       *lolafile;               /* listing of lola files to build base  */
FILE       *adcfile;                /* listing of adc files to build base   */

char       headorder, machineorder; /* byte ordering of record and machine  */
char       byteswap;                /* must i do byte swapping flag         */


/* global command line parameter variables */
char       contextfilename[200];    /* name of contextual database file     */
char       indexfilename[200];      /* name of index file for contextual    */
char       fileidfilename[200];     /* name of file id data base            */
char       lolalistname[200];       /* name of lola listing of files        */
char       adclistname[200];        /* name of adc listing of files         */


/*
 * int Add_Lola_File(contextfile, idfile, LolaName, WavName)
 *
 * This procedures reads in a lola segment file and adds contents to
 * contextual database file.
 *
 * contextfile(in): pointer to file stream (Context Data Base)
 * idfile     (in): pointer to file stream (File ID data base)
 * LolaName   (in): Name of Lola file 
 * WavName    (in): Name of wave file (directory and file name)
 *
 * On error sets ErrorString and returns -1, else +1
 *
 */

int  Add_Lola_File(FILE *contextfile, FILE *idfile, char *LolaName, 
		   char *WavName)
{
  FILE        *adccheckfile;              /* open adc file for exist     */
  ContextRec  OneContext;                 /* one record in context base  */
  FileIDRec   OneWave;                    /* one record in fileid base   */
  SegmentList *segment;                   /* lola file segment list      */
  SegmentList *current, *next, *previous; /* pointers in segment list    */
  float       ms;                         /* sampling frequency of lola  */
  char        label[100];                 /* lola label (phoneme)        */
  short       currphoneme;                /* index of current phoneme    */
  int         currph;                     /* integer phoneme index       */
  int         FileID;                     /* from file ID                */
  int         temp;                       /* loop counters               */
  int         context_curr;               /* current position in context */ 

  /* read in lola file */
  if (ReadSegmentList(LolaName, &segment, &ms)<0) {
    fprintf(stderr, "Add_Lola_File -> 1: %s -- %s\n", LolaName, ErrorString);
    return (-1);
  }

  if (!(adccheckfile = fopen(WavName, "r"))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not open ADC file";
    fprintf(stderr, "Add_Lola_File -> 2: %s\n", ErrorString);
    FreeSegmentList(segment);
    return (-1);
  }
  fclose(adccheckfile);

  current = segment;

  fseek(contextfile, 0, SEEK_END);
  context_curr = ftell(contextfile)/sizeof(ContextRec);

  do {
    previous = current;
    current  = current->sl_forward;        /* move to previous next link  */
    next     = current->sl_forward;

    OneContext.headorder = machineorder;
    
    strcpy(label, current->sl_label);

    if ((OneContext.ph_curr  = 
	Retrieve_Phoneme_Index(label, &PhonemeIndex, &PreviousLink)) < 0) {
      fprintf(stderr, "Add_Lola_File -> 3: %s\n", ErrorString);
      return (-1);
    }
    
    if (PhonemeIndex[OneContext.ph_curr-1].startrec == -1)
      PhonemeIndex[OneContext.ph_curr-1].startrec = context_curr;

    OneContext.startloc = current->sl_begin;
    OneContext.endloc   = current->sl_end;
    if (previous->sl_backward)            /* fix alignment of lola file */
      OneContext.startloc = previous->sl_end;

    currphoneme = OneContext.ph_curr;
    
    if (previous->sl_backward == NULL) {
      OneContext.ph_prev     = -1;
      OneContext.prev_start  =  0;
      OneContext.ph_prevlink = -1;
      OneContext.prevlink    = -1;
    }
    else {
      strcpy(label, previous->sl_label);     /* previous label in link     */
      if ((OneContext.ph_prev = 
	  Retrieve_Phoneme_Index(label, &PhonemeIndex, &PreviousLink)) < 0) {
	fprintf(stderr, "Add_Lola_File -> 4: %s\n", ErrorString);
	return (-1);
      }
      
      OneContext.prev_start  = previous->sl_begin;
      OneContext.ph_prevlink = context_curr - 1;
      OneContext.prevlink    = -1;
    }
    
    if (next) {
      if (next->sl_forward == NULL) {
	strcpy(label, next->sl_label);
	if ((OneContext.ph_next = 
	     Retrieve_Phoneme_Index(label, &PhonemeIndex, &PreviousLink))< 0) {
	  fprintf(stderr, "Add_Lola_File -> 5: %s\n", ErrorString);
	  return (-1);
	}
	
	OneContext.next_end    = next->sl_end;
	OneContext.ph_nextlink = context_curr + 1;
	OneContext.nextlink    = -1;
      }
      else {
	strcpy(label, next->sl_label);       /* next label in linked list  */
	if ((OneContext.ph_next = 
	     Retrieve_Phoneme_Index(label, &PhonemeIndex, &PreviousLink))<0) {
	  fprintf(stderr, "Add_Lola_File -> 6: %s\n", ErrorString);
	  return (-1);
	}
	
	OneContext.next_end    = next->sl_end;
	OneContext.ph_nextlink = context_curr + 1;
	OneContext.nextlink    = -1;
      }
    } else {
      OneContext.ph_next     = -1;
      OneContext.next_end    =  OneContext.endloc;
      OneContext.ph_nextlink = -1;
      OneContext.nextlink    = -1;
    }

    /* get wave form file id */
    strcpy(OneWave.wavfilename, WavName);
    strcpy(OneWave.phnfilename, LolaName);

    FileID = Return_FileID(idfile, OneWave);
    if (FileID<0) {
      fseek(idfile, 0, SEEK_END);
      OneWave.headorder   = machineorder;
      OneWave.start_point = context_curr;
      OneWave.ms = ms;
      fwrite(&OneWave, sizeof(FileIDRec), 1, idfile);
      FileID = ftell(idfile)/sizeof(FileIDRec);
    }
    OneContext.from_file_id = FileID;

    if ((OneContext.prevlink = PreviousLink[currphoneme-1]) >= 0)  {
      if (Update_Context_Record(contextfile, OneContext.prevlink,context_curr) < 0) {
	fprintf(stderr, "Add_Lola_File -> 7: %s\n", ErrorString);
	return (-1);
      }
    }
    
    PreviousLink[currphoneme-1] = context_curr;
    PhonemeIndex[currphoneme-1].endrec = context_curr;

    fwrite(&OneContext, sizeof(ContextRec), 1, contextfile);

    context_curr++;
  } while (current->sl_forward);

  FreeSegmentList(segment);
  
  return (1);
}


main (int argc, char **argv)
{
  char lolaname[200], wavname[200];
  ContextRec OneContext;
  int        index, count = 0;
  
  if (argc != 3)
    eerror("Usage: database list");

  machineorder = LittleIndian();

  /* phoneme context base ( all lola files stored in this file) */
  strcpy(contextfilename, argv[1]);
  strcat(contextfilename, CONTEXTBASE);

  /* phoneme index file ( index value of phoneme ) */
  strcpy(indexfilename, argv[1]);
  strcat(indexfilename, INDEX);

  /* from file id (file containing listing of lola files )*/
  strcpy(fileidfilename, argv[1]);
  strcat(fileidfilename, IDBASE);

  /* script listing of files */
  strcpy(lolalistname, argv[2]);
  strcat(lolalistname, LOLA);
  strcpy(adclistname, argv[2]);
  strcat(adclistname, ADC);

  /* do data bases setup phase */
  if (Read_Index_File(indexfilename, &PhonemeIndex, &PreviousLink) < 0) {
    fprintf(stderr, "Read_Index_File: %s\n", ErrorString);
    exit(1);
  }
    
  if (Open_Contextual_Base(&contextfile, contextfilename) < 0) {
    fprintf(stderr, "Open_Contextual_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    exit(1);
  }

  if (Open_FileID_Base(&idfile, fileidfilename) < 0) {
    fprintf(stderr, "Open_File_IDBase: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    Close_Contextual_Base(&contextfile);
    exit(1);
  }

  /* add lola files determined from lola script file */
  if (!(lolafile = fopen(lolalistname, "rt"))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not open lola listing file";
    eerror(ErrorString);
  }

  if (!(adcfile = fopen(adclistname, "rt"))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not open waveform listing file";
    eerror(ErrorString);
  }

  while (!(feof(lolafile))) {
    fscanf(lolafile, "%s\n", lolaname);
    fscanf(adcfile, "%s\n", wavname);

    printf("%s\n", lolaname);
    printf("%s\n", wavname);

    if (Add_Lola_File(contextfile, idfile, lolaname, wavname) < 0) {
      fprintf(stderr, "Add_Lola_File: %s\n", ErrorString);
      break;
    }
  }

  fclose(lolafile);
  fclose(adcfile);

  /* close data base */
  if (Write_Index_File(indexfilename, PhonemeIndex) < 0) {
    fprintf(stderr, "Write_Index_File: %s\n", ErrorString);
  }

  Close_Contextual_Base(&contextfile);
  Close_FileID_Base(&idfile);

  free((char *) PhonemeIndex);
  free((char *) PreviousLink);
}



