static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/auto_lyre/RCS/Lolatool.c,v 4.7 1993/05/28 21:27:40 johans Exp johans $";

/*
 * Lolatool.c - Lolatool widget
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 *
 * 25-March-1993 Johan Schalkwyk at Oregon Graduate Institute
 *      made stylistic changes, added lots and lots of comments
 *      and made the overall code more robust.
 *      And changed to ANSI style header declarations
 */

/* Standard C include file directives */
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <c.h>

/* Speech library include file directives */
#include <speech.h>

/* X library include file directives */
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Shell.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw/CommandP.h>
#include <X11/Xaw/DialogP.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Sme.h>
#include <X11/Xaw/SmeBSB.h>

/* Lyre include file directives */
#include <LyreDisp.h>
#include <Scale.h>
#include <LolaP.h>
#include <LolatoolP.h>
#include <Yoke.h>
#include <ToolUtil.h>

/* external variable declarations */
extern char *lyre_version;

/* type definitions */
typedef union fl {
	float f;
	unsigned int   l;
} fl_t;


/* Procedures in this class */
static void LolaRead (Widget w, WidgetList client_data);
static void LolaWrite (Widget w, WidgetList client_data);
static void LolaWriteAligned (Widget w, WidgetList client_data);
static void AlignSegments (SegmentList *seg);

static void Initialize (Widget request_in, Widget new_in, ArgList arglist,
			Cardinal *num_args);
static void Resize (Widget w_in);
static Boolean SetValues (Widget current_in, Widget request_in, Widget new_in,
			  ArgList arglist, Cardinal *num_args);
static void Destroy (Widget w, Widget target_widget, 
		     caddr_t call_data);


static Boolean defTRUE = TRUE;
static Boolean defFALSE = FALSE;

#define offset(field) XtOffset(LolatoolWidget, field)
static XtResource resources[] = { 
   {XtNchild, XtNchild, XtRPointer, sizeof(caddr_t), 
      offset(lolatool.child), XtRPointer, (caddr_t)NULL},
   {XtNvertScaleOn, XtNvertScaleOn, XtRBoolean, sizeof(Boolean), 
      offset(lolatool.vert_scale_on), XtRBoolean, (caddr_t)&defFALSE},
   {XtNhorizScaleOn, XtNhorizScaleOn, XtRBoolean, sizeof(Boolean), 
      offset(lolatool.horiz_scale_on), XtRBoolean, (caddr_t)&defTRUE},
   {XtNscrollWidget, XtNscrollWidget, XtRPointer, sizeof(caddr_t), 
      offset(lolatool.scroll), XtRPointer, (caddr_t)NULL},
   {XtNlyreLabel, XtCParameter, XtRString, sizeof(String), 
      offset(lolatool.lyre_label), XtRString, "Lola"},
};
#undef offset

LolatoolClassRec lolatoolClassRec = {
  {
    (WidgetClass) &formClassRec,	/* superclass		  */	
    "Lolatool",				/* class_name		  */
    sizeof(LolatoolRec),		/* size			  */
    NULL,				/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    XtInheritRealize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    TRUE,				/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    XtInheritExpose,			/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL,				/* extension		  */
  },  /* CoreClass fields initialization */
  { /* composite_class fields */
    /* geometry_manager   */   XtInheritGeometryManager,
    /* change_managed     */   XtInheritChangeManaged,
    /* insert_child       */   XtInheritInsertChild,
    /* delete_child       */   XtInheritDeleteChild,
  },
  { /* constraint_class fields */
    /* subresourses       */   NULL,
    /* subresource_count  */   0,
    /* constraint_size    */   sizeof(LolatoolConstraintsRec),
    /* initialize         */   NULL,
    /* destroy            */   NULL,
    /* set_values         */   NULL
  },
  {
    0,
  }, /* FormClass fields */
  {
    0,                                     /* field not used    */
  },  /* LolatoolClass fields initialization */
};

  /* for public consumption */
WidgetClass lolatoolWidgetClass = (WidgetClass) &lolatoolClassRec;


/****************************************************************
 *
 * Private Procedures
 *
 ****************************************************************/

#define MyXtSetArg(args,i,arg,val)	{XtSetArg (args[i], arg, val); i++;}

/*
 * Initialize (request, new)
 *
 * Create the Lolatool and the menu and graphics widgets.
 *
 *   LolaTool -> ControlWidget (?)
 *            -> hscale
 *            -> child (graphics)
 *            -> command -> menu -> LolaRead()
 *                               -> SetFloat()
 *                               -> LolaWrite()
 *                               -> LolaWriteAligned()
 *                               -> Destroy()
 * 
 */

static void Initialize (Widget request_in, Widget new_in, ArgList arglist,
			Cardinal *num_args)
{
    LolatoolWidget request = (LolatoolWidget) request_in;
    LolatoolWidget new     = (LolatoolWidget) new_in;
    Widget ControlWidget, child, hscale = 0,
           command, menu, entry;
    Arg    args[20];
    int    arg_cnt;
    fl_t   units_psec;
    char   stmp[128];

    /* millisecond units in lola window by default */
    /* (needs resource!) */
    units_psec.f = 1000.0;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNborderWidth, 0);
    ControlWidget = XtCreateManagedWidget ("controls", formWidgetClass,
					   (Widget) new, args, arg_cnt);

    /*
     * Create horizontal scale widget for Lola graph
     *   LolaTool -> hscale
     *
     */

    if (new->lolatool.horiz_scale_on) {
	arg_cnt = 0;
	MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130);
	MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
	MyXtSetArg (args, arg_cnt, XtNright,  XtChainRight);
	MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
	MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
	MyXtSetArg (args, arg_cnt, XtNresizable,  TRUE);
	MyXtSetArg (args, arg_cnt, XtNhorizontal, TRUE);
	MyXtSetArg (args, arg_cnt, XtNborderWidth, 1);
	MyXtSetArg (args, arg_cnt, XtNunitsLabel, "ms");
	MyXtSetArg (args, arg_cnt, XtNshowExtent, FALSE);
	hscale = XtCreateManagedWidget ("scale", scaleWidgetClass, 
					(Widget) new, args, arg_cnt);
	new->lolatool.hscale = hscale;
    }

    /*
     * Create Lola graphics widget
     *   LolaTool -> child (Graphics)
     *
     */

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130);
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainRight);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainBottom);
    if (new->lolatool.horiz_scale_on)
	MyXtSetArg (args, arg_cnt, XtNfromVert, hscale);
    MyXtSetArg (args, arg_cnt, XtNresizable, TRUE);
    MyXtSetArg (args, arg_cnt, XtNhscaleWidget, hscale);
    MyXtSetArg (args, arg_cnt, XtNunitsPsec, units_psec.l);

    child = XtCreateManagedWidget ("lola", lolaWidgetClass, (Widget) new,
				   args, arg_cnt);
    new->lolatool.child = child;

    /*
     * Build the menu and its actions
     *  LolaTool -> command (menu button)
     *           -> command               -> menu
     *
     */

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNborderWidth,   2);
    MyXtSetArg (args, arg_cnt, XtNhorizDistance, 4);
    MyXtSetArg (args, arg_cnt, XtNvertDistance,  4);
    command = XtCreateManagedWidget(new->lolatool.lyre_label, 
				    menuButtonWidgetClass, ControlWidget, 
				    args, arg_cnt);
    menu = XtCreatePopupShell("menu", simpleMenuWidgetClass, command, 
			      NULL, 0);

    /*
     * First Entry in menu: [Set Lola File]
     *   LolaTool -> command -> menu -> LolaRead()
     *
     */

    entry = XtCreateManagedWidget("Set Lola File", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Lola file name:", "", "Display", "Abort", 
	       (caddr_t)LolaRead);

    /*
     * Second Menu Entry: [Set Resolution]
     *  LolaTool -> command -> menu -> SetFloat()
     *
     */

    sprintf (stmp,"%g", ((LolaWidget)child)->lyreDisp.secs_ppix);
    MakeMenuEntry ((Widget)new, menu, (caddr_t)child, "Set Resolution",
		   "Seconds per pixel:", stmp, "Set", "Abort",
		   (caddr_t)SetFloat, XtNsecsPpix);

    /*
     * Third Menu Entry: [Write Lola File]
     *  LolaTool -> command -> menu -> LolaWrite()
     *
     */

    sprintf(stmp,"lolaout.%s",new->lolatool.lyre_label);
    entry = XtCreateManagedWidget("Write Lola File", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Lola file name:", stmp, "Write", "Abort", (caddr_t)LolaWrite);

    /*
     * Fourth Menu Entry: [Write Aligned LolaFile]
     *  LolaTool -> command -> menu -> LolaWriteAligned()
     *
     */

    entry = XtCreateManagedWidget("Write Aligned Lola File", smeBSBObjectClass,
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Lola file name:", stmp, "Write", "Abort",
	        (caddr_t)LolaWriteAligned);

    /*
     * Fifth Menu Entry: [Delete This Display]
     *   LolaTool -> command -> menu -> Destroy()
     *
     */

    entry = XtCreateManagedWidget("Delete This Display", smeBSBObjectClass, 
				  menu, NULL, 0);
    XtAddCallback (entry, XtNcallback, (XtCallbackProc) Destroy, new);
}


/*
 * Resize (w)
 *
 * Resize requested for the lola widget
 *
 */

static void Resize (Widget w_in)
{
    LolatoolWidget w = (LolatoolWidget) w_in;
    CorePart *core;
    int      diff;

    if (XtIsRealized ((Widget) w)) {
      if (w->lolatool.child) {
	core = &(w->lolatool.child->core);
	diff = core->y + core->height + core->border_width << 1 -
	  w->core.height;
	if (diff > 0)
	  core->height -= diff;
      }
    }
  }


/*
 * Boolean SetValues (current, request, new)
 *
 * Set values for the lola display
 *
 */

static Boolean SetValues (Widget current_in, Widget request_in, Widget new_in,
			  ArgList arglist, Cardinal *num_args)
{
    Boolean             redisplay = FALSE;

    return (redisplay);
}


/*
 * LolaRead(w, client_data)
 *
 * Read a lola file into memory and display it
 *
 * w           (in): parent widget from where this routine was called from
 * client_data (in): inter widget communication via client_data
 *
 */

static void LolaRead (Widget w, WidgetList client_data)
{
    LolaWidget      child = (LolaWidget)     client_data[0];
    DialogWidget    dw    = (DialogWidget)   client_data[1];
    LolatoolWidget  tool  = (LolatoolWidget) client_data[2];
    Arg             args[10];
    int             arg_cnt = 0;
    char            *file = XawDialogGetValueString ((Widget) dw);
    SegmentList     *sl;
    fl_t            units_psec;
    float ftmp;

    if( ReadSegmentList( file, &sl, &ftmp ) < 0 ) {
      XtAlert("Could not open file: %s", file);
      return;
    }

    units_psec.f = 1000.0 / ftmp;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNinputSeg, sl);
    MyXtSetArg (args, arg_cnt, XtNunitsPsec, units_psec.l);
    XtSetValues ((Widget) child, args, arg_cnt);
}


/*
 * LolaUnmap(tool)
 *
 * make a lola display disappear; space appears blank, not remapped 
 *
 */

int LolaUnmap(LolatoolWidget tool)
{
  XtUnmapWidget(tool->lolatool.child);
}


/*
 * LolaMap(tool)
 *
 * make an lola display reappear 
 *
 */

int LolaMap(LolatoolWidget tool)
{
  XtMapWidget(tool->lolatool.child);
}


/*
 * LolaPollRead (tool, file)]
 *
 * Read a lola file, through polling the disc for the lates file
 * Used for Auto Lyre
 *
 */

int LolaPollRead (LolatoolWidget tool, char *file)
{
    LolaWidget  child = (LolaWidget) tool->lolatool.child;
    Arg         args[10];
    int         arg_cnt = 0;
    SegmentList *sl = 0;
    fl_t        units_psec;
    float ftmp;

    if(PollForFile(file,1 /* Interval secs */, 1 /* Attempts */) == -1)
      return -1;

    if( ReadSegmentList( file, &sl, &ftmp ) < 0 ) {
	return( -1 );
    }

    units_psec.f = 1000.0 / ftmp;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNinputSeg, sl);
    MyXtSetArg (args, arg_cnt, XtNunitsPsec, units_psec.l);
    XtSetValues ((Widget) child, args, arg_cnt);
    return 0;
}


/*
 * LolaWrite (w, client_data)
 *
 * Write a lola file, based on the labeling done by hand
 * 
 * w           (in): not used, widget where this routine was called from
 * client_data (in): data communication via client_data
 *
 */

static void LolaWrite (Widget w, WidgetList client_data)
{
    LolaWidget     child = (LolaWidget) client_data[0];
    DialogWidget   dw    = (DialogWidget) client_data[1];
    LolatoolWidget tool  = (LolatoolWidget) client_data[2];
    Arg            args[10];
    int            arg_cnt = 0;
    char           *file = XawDialogGetValueString ((Widget) dw);
    SegmentList    *sl = 0;
    fl_t           units_psec;
    float ms;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNseg, &sl);
    MyXtSetArg (args, arg_cnt, XtNunitsPsec, &(units_psec.l) );
    XtGetValues ((Widget) child, args, arg_cnt);

    ms = 1000.0 / units_psec.f;

    if( WriteSegmentList( file, sl, ms ) < 0 ) {
      XtAlert("Could not write lola file: %s", file);
      return;
    }
}


/*
 * LolaWriteAligned(w, clinet_data)
 *
 * Write aligned lola data file
 *
 * w           (in): not used, widget where this routine was called from
 * client_data (in): data structure containing lola segments
 *
 */

static void LolaWriteAligned (Widget w, WidgetList client_data)
{
    LolaWidget      child = (LolaWidget) client_data[0];
    DialogWidget    dw    = (DialogWidget) client_data[1];
    LolatoolWidget  tool  = (LolatoolWidget) client_data[2];
    Arg             args[10];
    int             arg_cnt = 0;
    char            *file = XawDialogGetValueString ((Widget) dw);
    SegmentList     *sl   = 0;
    fl_t            units_psec;
    float ms;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNseg, &sl);
    MyXtSetArg (args, arg_cnt, XtNunitsPsec, &(units_psec.l) );
    XtGetValues ((Widget) child, args, arg_cnt);

    ms = 1000.0 / units_psec.f;

    AlignSegments( sl );
    if( WriteSegmentList( file, sl, ms ) < 0 ) {
      XtAlert("Could not write aligned lola file: %s", file);
      return;
    }

    XClearArea (XtDisplay (child), XtWindow(child), 0, 0, 0, 0, TRUE);
}


/*
 * AlignSegments (seg)
 *
 * Align the marked list of segments before writing the aligned lola file
 *
 * seg (in/out): pointer to the segment list, of type SegmentList
 *
 */

static void AlignSegments( SegmentList *seg )
{
	SegmentList *tsl;

	tsl = seg->sl_forward;

	while( ( tsl != NULL) && ( tsl->sl_forward != NULL ) ) {
	  if(tsl->sl_end < tsl->sl_forward->sl_begin)
		tsl->sl_end = tsl->sl_forward->sl_begin;
  	  tsl = tsl->sl_forward;
	}
}


/*
 * Destroy (w, target_widget, call_data)
 *
 * Destroy the lola widget, deletes the area of lola display
 *
 */

static void Destroy (Widget w, Widget target_widget, caddr_t call_data)
{
    XtDestroyWidget (target_widget);
}

