#| -*-Scheme-*-

Written May 28, 1992 by Max Hailperin <max@nic.gac.edu>
to provide access to the Hyperbot (TM) robotics interface, as well as
a few other features useful in this context.

The Hyperbot controller hardware can be obtained from Bots, 905 South
Springer Rd., Los Altos, CA 94024, USA.  Phone: (415) 949-2126; Fax:
(415) 949-2566; CompuServe: 75500,2027; Internet: 75500.2027@compuserve.com.

Lego(R) kits compatible with the Hyperbot controller can be obtained from
Bots, or directly from Lego Dacta, 555 Taylor Rd., P.O. Box 1600,
Enfield, CT 06083-1600, USA.  Phone: (800) 526-8339; Fax: (203) 763-2466.

Other kits, e.g. fischertechnik(R) and Capsella(R), may also work with
Hyperbot; contact Bots for more information.

This software has been tested with MIT Scheme 7.1.3 on the NeXT.
[Note: it needs a *modified* 7.1.3 that includes the extra primitive
wait-microseconds.  The source-code patch to provide this primitive is
also available from me, as is the pre-compiled executable for the
NeXT.]  It should work with MIT Scheme on other machines as well, but
I haven't tried that.  The call to terminal-set-state in make-hyperbot
is the most likely trouble spot for porting.  The Hyperbot controller
plugs flawlessly into the NeXT serial ports with its included cable;
for other machines an adapter cable may be needed.  (All that is
needed is basic serial I/O, no handshaking is done.  The Hyperbot
controller is a DCE, like a modem, so it uses a straight-through
cable.)  For porting to non-MIT Scheme implementations, likely trouble
spots (other than this extended comment) are the timer and serial I/O
primitives and define-structure.  Define structure is a hairy MIT way of
getting nice data structures, but if you are porting to another Scheme
you could just use a vector in the obvious way.

One final porting note: the argument to make-hyperbot is just tacked
on to the end of /dev/tty to get the filename to open for the serial
port.  The documentation I wrote says it has to be either a or b,
since the two serial devices are /dev/ttya and /dev/ttyb on the NeXT.
If the serial devices are named differently on your machine, you
should change the documentation to match.
----------
Copyright (c) 1992 Gustavus Adolphus College.  All rights reserved.

This software was developed by Gustavus Adolphus College (GAC).
Permission to copy this software, to redistribute it, and to use it
for any purpose is granted, subject to the following restrictions and
understandings.

1. Any copy made of this software must include this copyright notice in full.

2. Users of this software agree to make their best efforts (a) to
return to the GAC Mathematics and Computer Science Department any
improvements or extensions that they make, so that these may be
included in future releases; and (b) to inform GAC of noteworthy uses
of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. GAC makes no express or implied warranty or representation of any
kind with respect to this software, including any warranty that the
operation of this software will be error-free.  ANY IMPLIED WARRANTY
OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE IS HEREBY
DISCLAIMED.  GAC is under no obligation to provide any services, by
way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of Gustavus Adolphus College nor of
any adaptation thereof in any advertising, promotional, or sales
literature without prior written consent from GAC in each case. |#

(declare (usual-integrations))

(define-primitives
  (wait-microseconds 1)
  (gc-space-status 0))

(define (maybe-gc #!optional free-fraction-threshold)
  (let ((free-fraction-threshold (if (default-object? free-fraction-threshold)
                                     .1
                                     free-fraction-threshold)))
    (cond ((not (real? free-fraction-threshold))
           (error:illegal-datum free-fraction-threshold 'maybe-gc))
          ((or (<= free-fraction-threshold 0)
               (> free-fraction-threshold 1))
           (error:datum-out-of-range free-fraction-threshold 'maybe-gc))
          (else
           (let ((status (gc-space-status)))
             (if (< (- (vector-ref status 6) (vector-ref status 5))
                    (* free-fraction-threshold
                       (- (vector-ref status 6) (vector-ref status 4))))
                 (gc-flip)))))))

(define default-hyperbot-maximum-power 14)
(define default-hyperbot-sensitivity 11)
(define default-hyperbot-sensor-mode 0)

(define-structure (hyperbot
                   (conc-name hyperbot/)
                   (print-procedure
                    (unparser/standard-method 'hyperbot
                                              (lambda (state obj)
                                                (unparse-object state
                                                                (hyperbot/name
                                                                 obj)))))
                   (constructor %make-hyperbot
                                (name
                                 input-port output-port
                                 saved-terminal-state)))
  (name #f)
  (input-port #f)
  (output-port #f)
  (maximum-power #f)
  (saved-terminal-state #f)
  )

(define (make-hyperbot name)
  (cond
   ((symbol? name) (make-hyperbot (string-downcase (symbol->string name))))
   ((not (string? name)) (error:illegal-datum name 'make-hyperbot))
   (else
    ;; The following line, which opens then immedeiately closes /dev/console
    ;; is a bug workaround.  It causes /dev/console to become the control
    ;; terminal if the scheme process doesn't already have one.  This in
    ;; turn prevents the Hyperbot's serial device from assuming that role,
    ;; which prevents SIGTTIN from being received if the hyperbot is closed
    ;; and then another open is done on the same device.  (I still don't
    ;; understand why SIGTTIN is sent under those circumstances, but this
    ;; avoids the situation, so I guess I don't care any more.)
    (close-output-port (open-output-file "/dev/console"))
    (let* ((filename (string-append "/dev/tty" name))
           (out (open-output-file filename))
           (in (open-input-file filename))
           (hyperbot (%make-hyperbot name in out
                                     ((access terminal-get-state
                                              (->environment '(runtime
                                                               primitive-io)))
                                      ((output-port/operation out 'channel)
                                       out)))))
      ((access terminal-set-state
               (->environment '(runtime primitive-io)))
       ((output-port/operation out 'channel) out)
       "\015\015\177\025\000\370\003\034\021\023\004\377\032\031\022\017\027\026\000\000\000\000") ;i.e. raw 9600 baud
      (hyperbot/reset hyperbot)
      hyperbot))))

(define (hyperbot/close hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/close)
      (let ((out (hyperbot/output-port hyperbot)))
        ((access terminal-set-state
                 (->environment '(runtime primitive-io)))
         ((output-port/operation out 'channel) out)
         (hyperbot/saved-terminal-state hyperbot))
        (close-output-port out)
        (close-input-port (hyperbot/input-port hyperbot))
        (set-hyperbot/input-port! hyperbot #f)
        (set-hyperbot/output-port! hyperbot #f)
        (set-hyperbot/name! hyperbot 'closed)
        unspecific)))

(define (hyperbot/send hyperbot . codes)
  (let ((out (hyperbot/output-port hyperbot)))
    (let loop ((codes codes))
      (if (null? codes)
          unspecific
          (begin (write-char (integer->char (car codes)) out)
                 (loop (cdr codes)))))))

(define (hyperbot/flush-input hyperbot)
  (if (read-char-no-hang (hyperbot/input-port hyperbot))
      (hyperbot/flush-input hyperbot)))

(define (hyperbot/query-8 hyperbot . codes)
  (hyperbot/flush-input hyperbot) ; for robustness
  (apply hyperbot/send hyperbot codes)
  (char->integer (read-char (hyperbot/input-port hyperbot))))

(define (hyperbot/query-16 hyperbot . codes)
  (let ((in (hyperbot/input-port hyperbot)))
    (hyperbot/flush-input hyperbot) ; for robustness
    (apply hyperbot/send hyperbot codes)
    (let ((first (char->integer (read-char in))))
      (+ (* 256 first)
         (char->integer (read-char in))))))

(define (hyperbot/reset hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/reset)
      (begin
        (set-hyperbot/maximum-power! hyperbot default-hyperbot-maximum-power)
        (hyperbot/flush-input hyperbot)
        (hyperbot/send hyperbot
                       126 126     ; doubled in case box is expecting parameter
                       124 default-hyperbot-sensor-mode
                       128 (hyperbot/maximum-power hyperbot)
                       133 0
                       164 default-hyperbot-sensitivity
                       159 160 161 162 125)
        (wait-microseconds 100000)
        (case (read-char-no-hang (hyperbot/input-port hyperbot))
          ((#\E #\R) unspecific)
          ((#f) (error "No response from hyperbot; maybe not connected?"))
          (else (error "Improper responce from hyperbot; device trouble?")))
        )))

(define (hyperbot/set-sensor-mode hyperbot mode)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/set-sensor-mode)
      (hyperbot/send hyperbot 124 (if mode 1 0))))

(define (hyperbot/set-maximum-power hyperbot limit)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/set-maximum-power))
        ((not (exact-integer? limit))
         (error:illegal-datum limit 'hyperbot/set-maximum-power))
        ((not (< 0 limit 25))
         (error:datum-out-of-range limit 'hyperbot/set-maximum-power))
        (else
         (set-hyperbot/maximum-power! hyperbot limit)
         (hyperbot/send hyperbot 128 limit))))

(define (hyperbot/get-maximum-power hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/get-maximum-power)
      (hyperbot/maximum-power hyperbot)))

(define (hyperbot/calibrate-light hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/calibrate-light)
      (hyperbot/send hyperbot 134)))

(define (hyperbot/calibrate-dark hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/calibrate-dark)
      (hyperbot/send hyperbot 135)))

(define (hyperbot/calibrate-average hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/calibrate-average)
      (hyperbot/send hyperbot 136)))

(define (hyperbot/get-sensitivity hyperbot)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/get-sensitivity)
      (hyperbot/query-8 hyperbot 163)))

(define (hyperbot/set-sensitivity hyperbot sensitivity)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/set-sensitivity))
        ((not (exact-integer? sensitivity))
         (error:illegal-datum sensitivity 'hyperbot/set-sensitivity))
        ((not (<= 0 sensitivity 15))
         (error:datum-out-of-range sensitivity 'hyperbot/set-sensitivity))
        (else
         (hyperbot/send hyperbot 164 sensitivity))))

(define (hyperbot/get-sensor hyperbot sensor)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/get-sensor))
        ((not (exact-integer? sensor))
         (error:illegal-datum sensor 'hyperbot/get-sensor))
        ((not (<= 1 sensor 4))
         (error:datum-out-of-range sensor 'hyperbot/get-sensor))
        (else
         (fix:zero? (fix:and (hyperbot/query-8 hyperbot 137)
                             (fix:lsh 8 sensor))))))

(define (hyperbot/get-counter hyperbot sensor)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/get-counter))
        ((not (exact-integer? sensor))
         (error:illegal-datum sensor 'hyperbot/get-counter))
        ((not (<= 1 sensor 4))
         (error:datum-out-of-range sensor 'hyperbot/get-counter))
        (else
         (hyperbot/query-16 hyperbot (+ 137 sensor)))))

(define (hyperbot/get-period hyperbot sensor)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/get-period))
        ((not (exact-integer? sensor))
         (error:illegal-datum sensor 'hyperbot/get-period))
        ((not (<= 1 sensor 4))
         (error:datum-out-of-range sensor 'hyperbot/get-period))
        (else
         (hyperbot/query-16 hyperbot (+ 141 sensor)))))

(define (hyperbot/reset-counter hyperbot sensor)
  (cond ((not (hyperbot? hyperbot))
         (error:illegal-datum hyperbot 'hyperbot/reset-counter))
        ((not (exact-integer? sensor))
         (error:illegal-datum sensor 'hyperbot/reset-counter))
        ((not (<= 1 sensor 4))
         (error:datum-out-of-range sensor 'hyperbot/reset-counter))
        (else
         (hyperbot/send hyperbot (+ 158 sensor)))))

(define (hyperbot/make-action hyperbot #!optional out1 out2 out3 out4)
  (if (not (hyperbot? hyperbot))
      (error:illegal-datum hyperbot 'hyperbot/make-action)
      (let ((output-port (hyperbot/output-port hyperbot))
            (command-string
             (let ((scale (lambda (output)
                            (cond ((default-object? output) #f)
                                  ((not output) #f)
                                  ((not (real? output))
                                   (error:illegal-datum output
                                                        'hyperbot/make-action))
                                  ((not (<= -1 output 1))
                                   (error:datum-out-of-range
                                    output
                                    'hyperbot/make-action))
                                  (else
                                   (round->exact
                                    (* output
                                       (hyperbot/maximum-power hyperbot)))))))
                   (power? (lambda (x) (and x (not (fix:zero? x)))))
                   (output (lambda (x) (write-char (integer->char x))))
                   (power->drivers (lambda (x)
                                     (cond ((fix:positive? x) 1)
                                           ((fix:negative? x) 2)
                                           (else 0)))))
               (let ((p1 (scale out1))
                     (p2 (scale out2))
                     (p3 (scale out3))
                     (p4 (scale out4)))
                 (with-output-to-string
                   (lambda ()
                     (if (and p1 p2 p3 p4) ;off while changing power
                         (begin (output 133) (output 0))
                         (begin
                           (if p1 (output 148))
                           (if p2 (output 151))
                           (if p3 (output 154))
                           (if p4 (output 157))))
                     (if (power? p1)
                         (begin (output 129) (output (abs p1))))
                     (if (power? p2)
                         (begin (output 130) (output (abs p2))))
                     (if (power? p3)
                         (begin (output 131) (output (abs p3))))
                     (if (power? p4)
                         (begin (output 132) (output (abs p4))))
                     (if (and p1 p2 p3 p4
                              (not (and (fix:zero? p1)
                                        (fix:zero? p2)
                                        (fix:zero? p3)
                                        (fix:zero? p4))))
                         (begin (output 133)
                                (output (+ (fix:lsh (power->drivers p4) 6)
                                           (fix:lsh (power->drivers p3) 4)
                                           (fix:lsh (power->drivers p2) 2)
                                           (power->drivers p1))))
                         (begin
                           (if (power? p1)
                               (output (if (fix:positive? p1) 146 147)))
                           (if (power? p2)
                               (output (if (fix:positive? p2) 149 150)))
                           (if (power? p3)
                               (output (if (fix:positive? p3) 152 153)))
                           (if (power? p4)
                               (output (if (fix:positive? p4) 155 156)))))
                     ))))))
        (lambda ()
          (write-string command-string output-port)))))

(define (wait-until predicate)
  (if (predicate)
      unspecific
      (begin (wait-microseconds 10000)
             (wait-until predicate))))
