;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)

(defparameter *justification-output* *trace-output*)


(defmethod justifier ((dragon classification-specialist) 
		      (type (eql '1)) case engram results) 
  (let (establish-reject-test 
	establish-runtime-threshold 
	suspend-runtime-threshold
	establish-reject-value status context-type-for-dragon  classifier-name
	parents-status translator-function)

    ;; this function determines why the dragon gained the status that it  
    ;; during case and stored in engram.  Status is either Establish 
    ;; Reject or Suspend.  
    
    (setf establish-reject-test (fetch-establish-reject-test (car results))) 
    (setf establish-reject-value  (car (cdr (car (cdr results)))))
    (if (slot-value dragon 'translator)
	(setf translator-function 
	      (slot-value dragon 'display-translator)))
    (setf status (nth 7 (car (cdr results)))) 
    (setf establish-runtime-threshold (nth 3 (car (cdr results))))
    (setf suspend-runtime-threshold (nth 5 (car (cdr results))))
    (setf context-type-for-dragon (slot-value dragon 'context-type)) 
    (setf parents-status (get-parent-status dragon case))
    (setf classifier-name (slot-value dragon 'classifier))
    
    ;; justify by printing the run-time value in comparison to the threshold 
    ;; values and then, if necessary, the dragon's context-type along with 
    ;; the parent(s)  status 
    
    (format *justification-output* 
	    "~%~%  The dragon ~a received the status ~a." 
	    (slot-value dragon 'display-name) status)
    (format *justification-output*
	    "~%   This status was received because the establish-reject test was ~a for which the dragon obtained an answer of ~a" 
	    establish-reject-test establish-reject-value)
    (if translator-function
	(format *justification-output* 
		" after being transformed through ~a" translator-function))
    
    (format *justification-output*
	    "~%  ~a has an establish threshold of ~a" 
	    (slot-value dragon 'display-name)
	    establish-runtime-threshold)
          
    (if (null (equal status 'established))
	(progn (format *justification-output*
		       "~%  ~a has a suspend threshold of ~a"
		       (slot-value dragon 'display-name)
		       suspend-runtime-threshold)
	       
   (if (null (equal parents-status '(case not in existence)))
	(progn 
	  (format 
	   *justification-output* 
	   "~%  ~a has a context type of ~a and its parent(s), ~a, had the status ~a"
	   (slot-value dragon 'display-name) context-type-for-dragon
	   (get-supers (slot-value dragon 'unique-name) 
		       (eval classifier-name)) parents-status)
	  (if (null (slot-empty-p dragon 'parent-join))
	      (format *justification-output*
		      " and joined by an ~a join." 
		      (slot-value dragon 'parent-join))))
        (format
	 *justification-output*
	 "~%  ~a's parent(s) was not invoked in this case, therefore no parent context can be given.  ~a was not invoked as a refinement from its parent(s)" 
	 (slot-value dragon 'display-name)
	 (slot-value dragon 'display-name)))))))


	

(defmethod justifier ((dragon classification-specialist) (type (eql '2)) 
		      case engram results)
  "This function determines how the dragon got invoked during the case in 
question. This will be either one of a few ways, the simplest is that it 
was either invoked from above by a user 'establish-refine' call or by a 
parent refining itself. Another way is by being invoked by an unrelated 
dragon requiring information about this specific dragon"

  (let (invoker invokers-verb invokers-args classifier-name)
    (setf classifier-name (slot-value dragon 'classifier))
    (setf invoker
	  (slot-value
	   (slot-value
	    (slot-value
	     (slot-value engram 'toolbed::invokers-transaction-record)
	     'toolbed::including-engram)
	    'whose-memory?)
	   'unique-name))
    (setf invokers-verb
	  (slot-value
	   (slot-value engram 'toolbed::invokers-transaction-record) 
	   'toolbed::invoke-verb))
    (setf invokers-args
	  (slot-value
	   (slot-value engram 'toolbed::invokers-transaction-record) 
	   'toolbed::invoke-args))

    (format 
     *justification-output*
     "~%~%  The dragon ~a was invoked by the dragon ~a with the verb ~a and arguments ~a." 
     (slot-value dragon 'display-name) invoker invokers-verb invokers-args)

    (if (equal invoker '*PUFF*)
	(format *justification-output* "  The invoke was sent from the user."))
    (if (and (member invoker 
		     (get-supers (slot-value dragon 'unique-name) 
				 (eval classifier-name)))  
	     (or (equal invokers-verb 'refine)
		 (equal invokers-verb 'establish-refine))) 
	(format *justification-output*
		"  The invoke was sent as a refinement from  ~a's parent ~a."
		(slot-value dragon 'display-name) invoker))))



(defmethod justifier ((dragon classification-specialist) 
		      (type (eql '3)) case engram results) 
  (let (establish-reject-test 
	establish-runtime-threshold 
	suspend-runtime-threshold
	establish-reject-value status context-type-for-dragon  classifier-name 
	name
	parents-status translator-function invoker invokers-verb invokers-args)

    ;; this function determines why the dragon gained the status that it  
    ;; during case and stored in engram.  Status is either Establish 
    ;; Reject or Suspend.  
    
    (setf name (slot-value dragon 'display-name))
    (setf establish-reject-test (fetch-establish-reject-test (car results))) 
    (setf establish-reject-value  (car (cdr (car (cdr results)))))
    (if (slot-value dragon 'translator)
	(setf translator-function 
	      (slot-value dragon 'display-translator)))
    (setf status (nth 7 (car (cdr results)))) 
    (setf establish-runtime-threshold (nth 3 (car (cdr results))))
    (setf suspend-runtime-threshold (nth 5 (car (cdr results))))
    (setf context-type-for-dragon (slot-value dragon 'context-type)) 
    (setf parents-status (get-parent-status dragon case))
    (setf classifier-name (slot-value dragon 'classifier))
    (setf invoker
	  (slot-value
	   (slot-value
	    (slot-value
	     (slot-value engram 'toolbed::invokers-transaction-record)
	     'toolbed::including-engram)
	    'whose-memory?)
	   'unique-name))
    (setf invokers-verb
	  (slot-value
	   (slot-value engram 'toolbed::invokers-transaction-record) 
	   'toolbed::invoke-verb))
    (setf invokers-args
	  (slot-value
	   (slot-value engram 'toolbed::invokers-transaction-record) 
	   'toolbed::invoke-args))


    
    ;; justify by printing the run-time value in comparison to the threshold 
    ;; values and then, if necessary, the dragon's context-type along with 
    ;; the parent(s)  status 
    
    (format *justification-output* 
	    "~%~%  The dragon ~a was invoked in an attempt to determine its status in the current case."
	     name)
    (format *justification-output*
	    "~%~% ~a was judged to be relevant to the situation occuring in the present case." 
	    name)
    (format *justification-output* 
		"~%   ~a was invoked by ~a in an attempt to ~a." 
		name invoker invokers-verb)
    (format *justification-output* 
	    "~%   The test applied to ~a to determine its status in this case was ~a to which the result was ~a" 
	    name
	    establish-reject-test establish-reject-value)
    (if translator-function 
	(format *justification-output* 
		"~%   The value for the status was translated via the function ~a." 
		translator-function)) 
       
    (format *justification-output*
	    "~%  ~a has an establish threshold of ~a" 
	   name establish-runtime-threshold)
          
    (if (null (equal status 'established))
	(progn (format *justification-output*
		       "~%  ~a has a suspend threshold of ~a"
		      name suspend-runtime-threshold)
	       
   (if (null (equal parents-status '(case not in existence)))
	(progn 
	  (format 
	   *justification-output* 
	   "~%  ~a has a context type of ~a and its parent(s), ~a, had the status ~a"
	   name context-type-for-dragon
	   (get-supers (slot-value dragon 'unique-name) 
		       (eval classifier-name)) parents-status)
	  (if (null (slot-empty-p dragon 'parent-join))
	      (format *justification-output*
		      " and joined by an ~a join." 
		      (slot-value dragon 'parent-join))))
        (format
	 *justification-output*
	 "~%  ~a's parent(s) was not invoked in this case, therefore no parent context can be given.  ~a was not invoked as a refinement from its parent(s)" 
	 (slot-value dragon 'display-name)
	 (slot-value dragon 'display-name)))))
    (format *justification-output* " giving ~a the status ~a." name status)))




(defmethod justifier ((dragon classification-specialist) (type (eql '4)) 
		      case engram results) 
          
  "This function prints out cs specific values used for justification such
as cs author, cs rationale, etc..."

  (let (user-answer slot-name temp-name classifier-name)
    (setf classifier-name (slot-value dragon 'classifier))
    ;;display list of possible slot-values to access
    (format 
     *justification-output* 
     "~%~% Here is a list of additional ra specific information.  select one:")
    (format *justification-output* "~%   1)  cs rationale")
    (format *justification-output* "~%   2)  author")
    (format *justification-output* "~%   3)  citations")
    (format *justification-output* "~%   4)  expert")
    (format *justification-output* "~%   5)  other knowledge sources")
    (format *justification-output* "~%   6)  disclaimer")
    (format *justification-output* "~%   7)  specific function")
    (format *justification-output* "~%   8)  threhold values")
    (format *justification-output* "~%   9)  translator type")
    (format *justification-output* "~%   10) join types")
    (format *justification-output* "~%   11) sub/super specialist")

    ;; Get the user's choice, set the appriopriate slot name(s) to slot-name 
    (setf user-answer (read *query-io*))
    (case user-answer 
	  (1 (setf slot-name 'rationale))
	  (2 (setf slot-name 'author))
	  (3 (setf slot-name 'citations))
	  (4 (setf slot-name 'expert))
	  (5 (setf slot-name 'other-knowledge-sources))
	  (6 (setf slot-name 'disclaimer))
 	  (7 (setf slot-name 'specific-function))
	  (8 (setf slot-name '(establish-threshold suspend-threshold)))
	  (9 (setf slot-name 'display-translator))
	  (10 (setf slot-name '(parent-join child-join)))
	  (11 (setf slot-name '(superspecialists subspecialists)))
	  (t (format *justification-output* "~%~% not a legal choice")))
    ;; print out the slot-name if there is only 1 slot to print
    (if (null (listp slot-name))
	(progn
	  (if (not (slot-empty-p dragon slot-name))
	      (format
	       *justification-output*
	       "~%~%  The value stored in ~a of ~a is ~a" slot-name 
	       (slot-value dragon 'display-name) (slot-value dragon slot-name))
	    (format
	     *justification-output*
	     "~%~%  The value stored in ~a of ~a is unbound."
	     slot-name (slot-value dragon 'display-name))))
        ;;else there are 2 slots to look at, print them 1 at a time
        (dolist (temp-name slot-name)
		(if (not (slot-empty-p dragon temp-name))
		    (format
		     *justification-output*
		     "~%~%  The value stored in ~a of ~a is ~a"
		     temp-name (slot-value dragon 'display-name)
		     (slot-value dragon temp-name))
		    (format
		     *justification-output*
		     "~%~%  The value stored in ~a of ~a is unbound."
		     temp-name (slot-value dragon 'display-name)))))

    (format *justification-output* "~% would you like another selection?")
    (setf user-answer (read *query-io*))
    (if (or (equal user-answer 'yes)
	    (equal user-answer 'y))
	(justifier dragon 4 case engram results))))



(defmethod explainer ((dragon discrete-pattern-recognition-agent) engram 
		      result-list) 
  "This function determines why the discrete ra, dragon, returned the final 
value that it did."

  (let ((threshold  (slot-value dragon 'success-threshold)) 
	(pattern (slot-value dragon 'display-tests))
     (features (fetch-features-for-display dragon))
	(pattern-name (append '(discrete pattern for)
			      (list (slot-value dragon 'display-name))))
	(run-time-confidence (last (car (last result-list))))
	pattern-to-use
	(list-of-results (reverse (cdr (reverse result-list)))))

    (if (equal run-time-confidence 
	       (list (slot-value dragon 'display-match-confidence)))
          
	;; if the dragon's confidence-value for the case is equal to the
	;; display-match-confidence, then the discrete pattern matched

	(progn 
	  (format 
	   *justification-output* 
	   "~%~%  The runtime values fit the pattern for this discrete ra.
  Therefore, the value is the match confidence for ~a."
	   (slot-value dragon 'display-name))
	  (display-pattern dragon list-of-results threshold))
          
        ;; if the return value is the no-match-confidence, then the pattern did
        ;; not match.  call find-failure-in-pattern to discover why.

        (progn 
	  (format 
	   *justification-output* 
	   "~%   The runtime values did not fit the pattern for the discrete ra ~%~a"
	   (slot-value dragon 'display-name))
	  (format
	   *justification-output* 
	   " therefore, the return value is the no-match-confidence-value")
	  (setf pattern-to-use (fetch-pattern dragon))
	  (find-failure-in-pattern 
	   dragon pattern-to-use 
	   pattern-name list-of-results threshold features)))))




(defmethod explainer ((dragon match-1-recognition-agent) 
		      engram list-of-results) 
  "This function will iterate through each pattern previous to the matched
pattern and determine why that pattern was unable to match. The actual
determination of why the pattern did not match is accomplished by calling the
function find-failure-in-pattern."

  (let (pattern-that-matched pattern-counter pattern-to-use values features)
    (setf features (fetch-features-for-display dragon))
    (if (> (length list-of-results)
	   (slot-value dragon 'number-of-features))
	(setf pattern-that-matched (car (cdr (car (last list-of-results)))))
        (setf pattern-that-matched (slot-value dragon 'number-of-patterns)))

    (if (null (equal 0 pattern-that-matched))
	(progn 

	  ;; If pattern-that-matched is not the first pattern, then 
	  ;; iterate through all unmatched patterns and determine why 
	  ;; they didn't match

	  (format
	   *justification-output* 
	   "~%~%The following are the patterns that didn't match and why they didn't match")

	  (do ((pattern-counter 0 (+ pattern-counter 1)))
	      ((equal pattern-counter pattern-that-matched) nil)
	      (setf pattern-to-use (fetch-pattern dragon pattern-counter))
	      (find-failure-in-pattern
	       dragon pattern-to-use pattern-counter
	       list-of-results 
	       (aref (slot-value dragon 'success-thresholds) pattern-counter) 
	       features)
	      (format *justification-output*
		      "~%  --press return to continue--~%")
	      (read-char)))


        (format *justification-output* "~%~%  The first pattern matched."))
    
    ;;  finish this justification by printing out the pattern that
    ;;  did match and the runtime values to show that the pattern
    ;;  did successfully match.

    (setf values (nbutlast list-of-results))
    (if (< pattern-that-matched (slot-value dragon 'number-of-patterns))
	(display-pattern
	 dragon pattern-that-matched values
	 (aref (slot-value dragon 'success-thresholds) pattern-that-matched))
        (format
	 *justification-output*
	 "~%~% None of the above patterns matched.  The value given by the RA is the NO-MATCH confidence value."))))



(defmethod display-pattern ((dragon match-1-recognition-agent) 
			    pattern-number-that-matched results threshold)

  "This method displays patterns and determines which pattern matched (if
any did).  If no pattern matched, then the no-match action took place.
Otherwise, all patterns in the dragon are checked one at a time to see
if they matched, and if not, what casued the mismatch.  As soon as a
match is found, the pattern checking halts."

  (let (matched-pattern)
        
    ;; Obtain the pattern that matched as a list from the dragon's array 
    ;; of patterns.

    (setf matched-pattern (fetch-pattern dragon pattern-number-that-matched))
 
    ;; display the pattern that matched

    (format *justification-output* 
	    "~%~%  The pattern to match is pattern ~a:~%    ~a" 
	    pattern-number-that-matched matched-pattern)
    (format *justification-output*
	    "~%  with the runtime values of:~%    ~a" results)
    (format *justification-output* 
	    "~%   and ~a as the success threshold (out of ~a features)." 
	    threshold (slot-value dragon 'number-of-features))))



(defmethod display-pattern ((dragon discrete-pattern-recognition-agent) 
			    results threshold)
          
  "This method displays the discrete pattern and determines if it matched 
or not, and if not, why there was a mismatch." 
    
  (let (matched-pattern) 
      
    ;; Obtain the pattern that matched as a list from the dragon's array 
    ;; of patterns.

    (setf matched-pattern (fetch-pattern dragon))
            
    ;; display the pattern that matched

    (format *justification-output* 
	    "~%~%  The pattern for the discrete ra ~a matched:~%    ~a" 
	    (slot-value dragon 'display-name)
	    matched-pattern)
    (format *justification-output*
	    "~%  with the runtime values of:~%    ~a" results)
    (format *justification-output* 
	    "~%   and ~a as the success threshold (out of ~a features)." 
	    threshold (slot-value dragon 'number-of-features))))


(defun fetch-establish-reject-test (list-of-values)
  "This function gets a list equaling (verb dragon [question] returned 
the value run-time-value) where 'returned the value' is a string.  
The macro returns verb dragon and question if question exists."

  (if (equal (car (cdr (cdr list-of-values))) 'returned)
      (append (list (car list-of-values)) (list (car (cdr list-of-values))))
      (append (list (car list-of-values)) (list (car (cdr list-of-values))) 
	      (list (car (cdr (cdr list-of-values)))))))


(defun fetch-features-for-display (dragon) 
          
  "This function fetches the dragon's features and chops off unnecessary
portions of the features for display purposes. A list of all features in 
dragon is returned. These features are now of the form \"string\" or name, 
\"string\" being the question asked, or name being the name of the dragon 
invoked in the feature."

  (let (number list-of-features counter new-feature)

    (setf number (slot-value dragon 'number-of-features))
    (setf list-of-features '(features))
    (do ((counter 0 (+ 1 counter)))
	((equal counter number) nil)
	(setf new-feature (aref (slot-value dragon 'display-features)
				counter))
	(if (atom new-feature)
	    (setf list-of-features 
		  (append list-of-features (list new-feature)))
          
	  ;; if the feature is a judge or invoke, then remove the verb leaving
	  ;; just the name of the dragon being called. if the feature is an
	  ;; ask-user then remove the verb and leave just the question.

	    (if (or (equal (car new-feature) 'judge)
		    (equal (car new-feature) 'ask-user)
		    (equal (car new-feature) 'invoke))
		(setf list-of-features 
		      (append list-of-features (list (car (cdr new-feature)))))
	      
	      ;; else the feature is an ask idb. remove the verb and the name
	      ;; of the idb leaving just the question.

	      (setf list-of-features 
		    (append list-of-features
			    (list (car (cdr (cdr new-feature)))))))))
    (setf list-of-features (cdr list-of-features))))


(defmethod fetch-pattern ((dragon discrete-pattern-recognition-agent))
  "This function retrieves the pattern of a discrete pattern RA for use in 
testing the pattern and run time values."

  (let (pattern-to-use counter)
    (setf pattern-to-use '(pattern))
    (do ((counter 0 (+ 1 counter)))
	((equal counter (slot-value dragon 'number-of-features)) nil)
	(setf pattern-to-use 
	      (append pattern-to-use
		      (list
		       (aref (slot-value dragon 'display-tests) counter)))))
    (cdr pattern-to-use)))


(defmethod fetch-pattern ((dragon match-1-recognition-agent) pattern-number)
  "This function retrieves the pattern of a match 1 RA for use in testing 
the pattern and run time values.  The pattern is found via pattern-number."

  (let (pattern-to-use counter)
    (setf pattern-to-use '(pattern))
    (do ((counter 0 (+ 1 counter)))
	((equal counter (slot-value dragon 'number-of-features)) nil)
	(setf pattern-to-use 
	      (append pattern-to-use
		      (list (aref (slot-value dragon 'display-block) 
				  pattern-number counter)))))
    (cdr pattern-to-use)))
 

(defmethod fetch-results ((dragon classification-specialist) engram) 
  "This function traces through the action records of engram of the dragon
which is a CS, and obtains all stored records on the engram.  Among this
data is the run-time value of the dragon's establish-reject value, the
status of the dragon during the case, and what subsequent actions (if any)
were taken."

  (let (record current-record establish-reject-test run-time-status
	       next-action action-taken list-of-results actions-taken)      

    (setf list-of-results '(results))
              
    ;; iterate through engram obtaining all data stored in the  action-records
    ;; first obtain the establish-reject data stored in the first
    ;; transaction-record

    (setf record (slot-value engram 'first-action-record))
    (setf
     establish-reject-test
     (list  (slot-value record 'toolbed::invoke-verb)
	    (slot-value record 'toolbed::contactee))) 
    (if (slot-value record 'toolbed::invoke-args)
	(setf
	 establish-reject-test
	 (append establish-reject-test
		 (list (slot-value record 'toolbed::invoke-args)))))
    (setf establish-reject-test 
	  (append
	   establish-reject-test
	   '(returned the value)
	   (list (slot-value record 'final-judgement))))
    (if (slot-value dragon 'translator) 
	(setf
	 establish-reject-test
	 (append establish-reject-test 
		 '(and was translated by)
		 (list (slot-value dragon 'display-translator)))))
	
    ;;  get the status of the CS after the establish-reject
    ;; test was returned

    (setf record (slot-value record 'toolbed::next-record))
    (setf run-time-status  (slot-value record 'body))
    
    ;; If more records exist, then the CS had some actions to
    ;; take afterwards (such as refining).  Get all of this
    ;; data and store in actions-taken

    (setf actions-taken '(actions))
    (if (slot-value record 'toolbed::next-record) 
	(do ((current-record
	      (slot-value record 'toolbed::next-record)
	      (slot-value current-record 'toolbed::next-record)))
	    ((null current-record) nil)
	    (if (typep current-record 'transaction-record)
		(progn
		  (setf next-action
			(list
			 (slot-value current-record 'toolbed::invoke-verb))) 
		  (setf next-action
			(append next-action
				(list
				 (slot-value
				  current-record 'toolbed::contactee))))
		  (if (slot-value current-record 'toolbed::invoke-args)
		      (setf 
		       next-action 
		       (append
			next-action
			(list
			 (slot-value current-record 'toolbed::invoke-args)))))
		  (setf next-action
			(append 
			 next-action
			 (list (slot-value current-record 'final-judgement))))
		  (setf actions-taken
			(append actions-taken (list next-action))))

	        (setf action-taken
		      (append
		       actions-taken
		       (list (slot-value current-record 'body))))))
      (setf actions-taken (append actions-taken '(no further actions taken))))
 
    ;; concatenate all of this data together and return

    (setf list-of-results
	  (list establish-reject-test run-time-status (cdr actions-taken)))))





(defmethod fetch-results ((dragon recognition-agent) engram) 
          
  "This function traces through the action records of engram of the dragon 
which is an RA, and obtains all return values from other called dragons.
These values are stored in transaction-records. If a transform is used by the
dragon, the result of the transform is stored in an action-record
immediately following the transaction-record. The last item in the engram
list is an action-record representing the pattern that matched in dragon."
   
  (let (number counter list-of-results record wait temp current difference)
    ;; all of this information is stored in list-of-results and is returned to
    ;; the justify function

    (setf number (slot-value dragon 'number-of-features))
    (setf counter '0)
    (setf list-of-results '(results))
    (setf wait nil)
          
    ;; iterate through engram obtaining each action-record

    (do ((record (slot-value engram 'first-action-record)
		 (slot-value record 'toolbed::next-record)))
	((null record) nil)

	(if (typep record 'transaction-record)
	    (if (null (aref (slot-value dragon 'transforms) counter))     

		;; the action-record represents a call to a dragon without any
		;; transform being applied afterwards

		(progn 
		  (setf list-of-results 
			(append 
			 list-of-results
			 (list 
			  (slot-value record 'final-judgement))))
		  (setf counter (+ counter 1)))
          
	      ;; ignore the transaction-record.  use the action-record which
	      ;; will appear next in the list -- this record contains both the
	      ;; return value and the transform value.

	      (setf wait t))

	  (if wait 
	      ;; the action-record represents the transform applied to the
	      ;; returned value.

	      (progn 
		(setf list-of-results
		      (append list-of-results
			      (list (list (nth 3 (slot-value record 'body))
					  'transformed-to
					  (nth 5 (slot-value record 'body))))))
		(setf counter (+ counter 1))
		(setf wait nil))
          
            ;; the action-record contains the pattern that matched

	    (setf list-of-results 
		  (append list-of-results
			  (list (slot-value record 'body)))))))
    (setf number (slot-value dragon 'number-of-features))
    (setf list-of-results (cdr list-of-results))
    (setf counter 0)
    (setf temp '(results))
          
    ;; determine how many features were evaluated in the case for dragon. 
    ;; also, insert the pattern # of the pattern that matched as the last
    ;; item in list-of-results.

    (dolist (current list-of-results)
	    (if (or (not (listp current))
		    (and (listp current)
			 (not (equal (car current) 'pattern))))
		(progn (setf counter (+ 1 counter))
		       (setf temp (append temp (list current))))))
    (setf difference (- number counter))
          
    ;; if not all features were evaluate, then insert ne into the 
    ;; list-of-results pertaining to the features not evaluated.

    (do ((counter 0 (+ 1 counter)))
	((equal counter difference) nil)
	(setf temp (append temp '(ne))))
    
    ;; return list-of-results

    (if (listp (car (last list-of-results)))
	(setf list-of-results (append (cdr temp)
				      (last list-of-results))))))



(defun find-failure-in-pattern (dragon pattern pattern-name 
				       results threshold features) 
          
  "This function searches through the pattern and results lists to determine
what caused the pattern to fail. For example, if the pattern is:
 y ? ? (le likely)
and the results list is:
 y n n confirmed
then confirmed does not fit (le likely) and so, is the cause for the
pattern not to match. The function takes into account transforms and
success thresholds in its determination."

  (let (number limit-of-bad-matches failures number-of-fails 
	       feature-counter current-value
	       test-value outcome)
    (format *justification-output*
	    "~%~% the pattern ~a failed because:" pattern-name)
    (setf number (slot-value dragon 'number-of-features))
    (setf limit-of-bad-matches (- number threshold))
    (setf failures '(failed patterns))
    (setf number-of-fails 0)
    (format 
     *justification-output* 
     "~%~% feature       test          value (after transform)     match/fail")
          
    ;; the do loop iterates through the lists, comparing the current-value
    ;; in the results list with the test-value in the pattern list.  
    ;; since the pattern may have a success threshold, there may be several
    ;; reasons why the pattern did not match. all of these are detected.
    ;; the loop terminates only when all reasons for the failure are
    ;; found (this is based on the success threshold.)

    (do ((feature-counter 0 (+ 1 feature-counter)))
	((equal feature-counter number) nil)
	(setf current-value (nth feature-counter results))
	(if (listp current-value)
	    (setf current-value (car (cdr (cdr current-value)))))
	(setf test-value (nth feature-counter pattern))
	(if (test-failed test-value current-value dragon 
			 feature-counter)
	    (setf number-of-fails (+ 1 number-of-fails)
		  outcome 'fail)
	  (setf outcome 'match))
	(format *justification-output* "~% ~a~15t~a~29t~a~57t~a" 
		(nth feature-counter features)
		test-value current-value outcome))
    (format 
     *justification-output* 
     "~%~%  The ra had ~a failures and has a success threshold of ~a out of ~a features." 
     number-of-fails threshold number)
    (format *justification-output* "~%")))



(defun get-case-from-memory (dragon &optional (case nil)) 
  "This function takes the name case and finds the appropriate engram in 
dragon's memory list. This engram is returned to the justify function."
  
  (let (engram-list x found engram-to-justify found-engram)
    (setf found nil)
    (if (null case)
	(setf engram-to-justify (slot-value dragon 'current-engram))
        (progn (setf engram-list (slot-value dragon 'memory))
	       (dolist (x engram-list)
		       (if (and (null found)
				(equal (slot-value x 'case)
				       case))
			   (progn (setf found t)
				  (setf found-engram x))))
	       (setf engram-to-justify found-engram)))))




(defun get-parent-status (dragon case)
  ;;  This function determines what the status of the parent 
  ;;  (or parents) of dragon were in this case
  (let (parent singular engram parent-status status-record 
	       this-parent temp classifier-name)

    (setf classifier-name (slot-value dragon 'classifier))
    (setf parent (get-supers (slot-value dragon 'unique-name) 
			     (eval classifier-name)))
    (if (equal (length parent) 1)
	(progn
	  (setf parent (car parent))
	  (setf singular t)))
    (if (null parent)
	;; do if no superspecialists
	(setf parent-status '(non-existent))

        (progn 
	  (if singular
	      ;; do if only 1 superspecialist
	      (progn (setf engram (get-case-from-memory (eval parent) case))
		     (if (null engram) 
			 (setf parent-status '(case not in existence))
		         (progn 
			   (setf 
			    status-record
			    (slot-value
			     (slot-value engram 'first-action-record) 
			     'toolbed::next-record))
			   (setf parent-status
				 (slot-value status-record 'body))
			   (setf parent-status
				 (list (nth 7 parent-status))))))
	      ;; do if more than 1 superspecialist
	      (progn (setf parent-status '(status))
		     (do ((counter 0 (+ 1 counter)))
			 ((equal counter (length parent)) nil)
			 (setf this-parent (eval (nth counter parent)))
			 (setf engram (get-case-from-memory this-parent case))
			 (if (null engram)
			     (setf
			      parent-status
			      (append parent-status
				      (list '(case no longer in existence))))
			     (progn
			       (setf status-record
				     (slot-value
				      (slot-value engram 'first-action-record) 
				      'toolbed::next-record))
			       (setf temp (slot-value status-record 'body))
			       (setf
				parent-status
				(append parent-status (list (nth 7 temp)))))))
		     (setf parent-status (cdr parent-status))))))))



(defmethod justify ((dragon classification-specialist) &optional (case nil))

  "This method is the starting point for justifying a
classification-specialist.  Once the engram to be justified is found,
and pertanent values are obtained from that engram's list of
action-records, a menu is offered for the user to select the type of
justification."

  (let (engram-to-justify list-of-results features)
    (setf engram-to-justify (get-case-from-memory dragon case))
    (setf features nil)
    (if (null case)
	(setf case (get-case dragon)))
    (setf list-of-results (fetch-results dragon engram-to-justify))
    (menu-and-execute dragon case engram-to-justify
		      list-of-results)))



(defmethod justify ((dragon discrete-pattern-recognition-agent) 
		    &optional (case nil)) 
  "This method serves as the starting point for justifying a 
discrete-pattern-ra. The justification is justifying why dragon returned 
the value that it did during case, and why a different value was not 
returned. If no case is given, the case defaults to the dragon's latest case."

  (let (engram-to-justify list-of-results list-of-features)
    ;; find the engram representing the case to be justified. if no case
    ;; was given, then default to latest case and current-engram of dragon

    (setf engram-to-justify (get-case-from-memory dragon case))
    (if (null case)
	(setf case (get-case dragon)))
    (if (null engram-to-justify)
	(format *justification-output* 
		"~%~% The case does not exist (or no longer exists).  No justification possible.")
        (progn 
	  ;; call function fetch-results to find the values returned to
	  ;; dragon by other dragons

	  (setf list-of-results (fetch-results dragon engram-to-justify))
          
	  ;; call fetch-features function to get abbreviated versions
	  ;; of feature names for purpose of displaying
	  
	  (setf list-of-features (fetch-features-for-display dragon))
          
	      
	  (menu-and-execute
	   dragon case engram-to-justify list-of-results)))
    (format *justification-output* "~%~% justification completed.")))



(defmethod justify ((dragon match-1-recognition-agent) &optional (case nil)) 

  "This method serves as the starting point for justifying a match-1-ra. 
The justification is justifying why dragon returned the value that it did
during case, and why a different value was not returned. If no case is 
given, the case defaults to the dragon's current case."

  (let (engram-to-justify list-of-results list-of-features)
    ;; find the engram representing the case to be justified. if no case
    ;; was given, then default to latest case and current-engram of dragon

    (setf engram-to-justify (get-case-from-memory dragon case))
    (if (null case)
	(setf case (get-case dragon)))
    (if (null engram-to-justify)
	(format *justification-output* 
		"~%~% The case does not exist (or no longer exists).  no justification possible.")
        (progn 
	  ;; call function fetch-results to find the values returned to
	  ;; dragon by other dragons

	  (setf list-of-results (fetch-results dragon engram-to-justify))
          
	  ;; call fetch-features function to get abbreviated versions
	  ;; of feature names for purpose of displaying

	  (setf list-of-features (fetch-features-for-display dragon))
          
	  (menu-and-execute
	   dragon case engram-to-justify list-of-results)))
    (format *justification-output* "~%~% justification completed.")))


(defmethod menu-and-execute ((dragon classification-specialist) 
			     case engram results)

  "This method is invoked by justify when the dragon in question is a CS.
This method will display a list of possible justifications for a CS, and 
let the user choose."

  (let (user-answer)
    ;; the following is the menu selection for justify
    ;; there are several \"levels\" of justification possible
          
    ;; level 1 shows why the CS returned the value that it did, in terms 
    ;;  of its establish-refine value and its thresholds
          
    ;; level 2 shows how the CS got context in terms of the problem solving of
    ;;  the case being justified. 
    ;; level 3 is a more verbose answer combining the answers of 1 and 2 
          
    ;; level 4 returns slot-values in the ra which pertain to
    ;;  rationale or documentation


    (format *justification-output* 
	    "~%~%  what level of justification would you like? (choose 1-4)
		      ~%    1)  return how the CS decided on its answer
		      ~%    2)  return how the CS got context in the case
			 ~%    3)  return a combination of 1 and 2
		      ~%    4)  return CS documentation~%")
    (setf user-answer (read *query-io*))
    (if (or (equal user-answer 1)
	    (equal user-answer 2) (equal user-answer 3) (equal user-answer 4))
		(justifier dragon user-answer case engram results)
	   (progn 
	     (format *justification-output* "~%  Not a legal choice.")
	     (menu-and-execute dragon case engram results)))))



(defmethod menu-and-execute ((dragon recognition-agent)
			     case engram results)

  "This method is invoked by justify when the dragon in question is an RA.
This method will display a list of possible justifications for an RA, and 
let the user choose."

  (let (user-answer)
    ;; the following is the menu selection for justify
    ;; there are several \"levels\" of justification possible
          
    ;; level 1 returns the ra's features and the associated runtime values
          
    ;; level 2 returns the above plus for each failed pattern, why
    ;;  it failed, and the pattern that succeeded.
    ;;  if the ra is discrete, it returns the pattern if
    ;;  it matched or the reason why it failed.
          
    ;; level 3 returns slot-values in the ra which pertain to
    ;;  rationale or documentation


    (format *justification-output* 
	    "~%~%  what level of justification would you like? (choose 1-3)
		      ~%    1)  return features and values
		      ~%    2)  return pattern matched and why others did not match.
		      ~%    3)  return ra documentation~%")
    (setf user-answer (read *query-io*))
    (if (or (equal user-answer 1) (equal user-answer 2) (equal user-answer 3))
		(justifier dragon user-answer case engram results)
		(progn
	       (format *justification-output* "~%  Not a legal choice.")
	       (menu-and-execute dragon case engram results)))))




(defmethod justifier ((dragon recognition-agent) (type (eql '1)) 
		      case engram results) 
          
  "This function returns all features and the runtime values for those features
of the ra dragon."

  (let (record counter features)      
    ;; features-list is the array of features stored in the ra

    (setf record (slot-value engram 'first-action-record))
    (format *justification-output* 
	    "~%~%  The value ~a was returned by ~a." 
	    (return-run-time-value results)
	    (slot-value dragon 'display-name))
          
    ;; the following do loop iterates through the array of features and the
    ;; list of results, printing them out as pairs. 
    ;; if the no runtime values where given for some features, the
    ;; phrase \"was not evaluated\" is given as the result.
    (setf features (fetch-features-for-display dragon))
    (format *justification-output* 
	    "~%  The following are the list of features with their corresponding runtime values (ne=feature not evaluated)")

    (do ((counter '0 (+ counter '1)))
	((equal counter (slot-value dragon 'number-of-features)) nil)
	(format *justification-output* 
		"~% ~a received the value of ~a" 
		(nth counter features)
		(car results))
	(setf results (cdr results)))))





(defmethod justifier ((dragon recognition-agent) (type (eql '2)) 
		      case engram results)
  "This method first calls justify-type-1 to give the list of all
features/runtime values."

  (justifier dragon '1 case engram results)
  (format *justification-output* "~%~% --press return to continue--~%")
  (read-char)  
  (explainer dragon engram results))



(defmethod justifier ((dragon recognition-agent) (type (eql '3)) 
		      case engram results) 
          
  "This function prints out ra specific values used for justification such
as ra author, ra rationale, etc..."

  (let (user-answer slot-name)
    (format *justification-output* 
	    "~%~% Here is a list of additional ra specific information.  select one:")
    (format *justification-output* "~%   1)  ra rationale")
    (format *justification-output* "~%   2)  author")
    (format *justification-output* "~%   3)  citations")
    (format *justification-output* "~%   4)  expert")
    (format *justification-output* "~%   5)  other knowledge sources")
    (format *justification-output* "~%   6)  disclaimer")
    (setf user-answer (read *query-io*))
    (case user-answer 
	  (1 (setf slot-name 'rationale))
	  (2 (setf slot-name 'author))
	  (3 (setf slot-name 'citations))
	  (4 (setf slot-name 'expert))
	  (5 (setf slot-name 'other-knowledge-sources))
	  (6 (setf slot-name 'disclaimer))
	  (T (format *justification-output* "~%~% Not a valid choice.")))
    (if (not (slot-empty-p dragon slot-name))
	(format *justification-output*
		"~%~%  The value stored in ~a of the dragon ~a is ~a"
		slot-name (slot-value dragon 'display-name) 
		(slot-value dragon slot-name))
        (format *justification-output*
		"~%~%  The value stored in ~a of the dragon ~a is unbound" 
		slot-name (slot-value dragon 'display-name)))
    (format *justification-output* "~% would you like another selection?")
    (setf user-answer (read *query-io*))
    (if (or (equal user-answer 'yes)
	    (equal user-answer 'y))
	(justifier dragon 3 case engram results))))





(defmacro return-run-time-value (results)
  ;; this macro grabs the run-time value stored in the list of results 
  ;; and returns it
  `(last (car (last ,results))))




(defun test-failed (test-value current-value dragon number) 
  "This function tests a dragon's feature-pattern vs. the run-time value
to determine if that feature had succeeded or failed in the context of 
the given pattern. The test is done in the function apply-test which is 
located in the file apply-test.lsp"

  (if (null (and (listp test-value) (equal current-value 'NE)))
      (not (apply-test test-value current-value dragon number))
    T))

