;;; -*- Mode: LISP; Package: USER; Syntax: Common-lisp -*-

;;;; Closed-World Assumptions
;;;; Version 8
;;;; Last Edited, 6/2/91

;;; Copyright (c) 1986, 1987, 1988, 1989, 1990, 1991 Kenneth D. Forbus, 
;;; Northwestern University, and Johan de Kleer, the Xerox Corporation.  
;;; All rights reserved.

;; A set is specified by the assertion (SET <name>).
;; (<name> HAS-MEMBER <el>) indicates <el> is a member of <name>.
;; (<name> MEMBERS <els>) indicates <els> are the members of <name>.
;; The procedure (CLOSE <name> <ltre>) will close <name>, meaning
;; that the known members are assumed to be the only ones.
;; This assumption is explicitly noted by a closed-world statement.
;; Its syntax is:
;; (<name> CWA <known as members> <known not to be members>)

(setq *cwa-enforcer*
'(rule ((:true (set ?name) :var ?f1))
 ;; Declaring a set activates rules that look for violations
 ;; of closed-world assumptions.  
      (rule ((:true (?name members ?members) :var ?f2)
	     (:true (?name has-member ?new) :var ?f3
		    :test (not (member ?new ?members
				       :test #'equal))))
	    ;; We found something we think is in the set but
	    ;; isn't listed there, so post this constraint.
	    (rassert! (:implies (:and ?f1 ?f3) (:not ?f2))
		      :CWA-EXTRA-MEMBER))
      ;; We must also update the dependence on things we know not
      ;; to be in the set.  This is encapsulated in the CWA
      ;; assertion, so we must trigger on it instead.
      (rule ((:true (?name CWA ?known-members
			   ?known-not) :var ?f2)
	     (:false (?name HAS-MEMBER ?el) :var ?f3
		     :test (not (member ?el ?known-not
					:test #'equal))))
	    (rassert! (:implies (:and ?f1 (:not ?f3)) (:not ?f2))
		      :CWA-EXTRA-KNOWN-NOT-MEMBER))
      (rule ((:true (?name MEMBERS ?els1) :var ?set1)
	     (:true (?name MEMBERS ?els2) :var ?set2
		    :test (not (equal ?els1 ?els2))))
	    (rassert! (:not (:and ?set1 ?set2))
		      :UNIQUE-SET-CONSTRUAL))))

;;;; Making the closed-world assumption

;; Closing sets is done consequently for better control.
;; It is possible to make antecedent versions, but they tend
;; to be inefficient and interact badly with other mechanisms,
;; such as dependency-directed search.

(defun close-set-if-needed (set-name &optional (*LTRE* *LTRE*)
				     &aux construal)
  (multiple-value-bind (construal cwa)
		       (set-members set-name)
  (cond (construal (values construal cwa nil))
	(t (close-set set-name)))))  ;; Must actually do it.

(defun set-members (set-name &optional (*LTRE* *LTRE*)
			     &aux m-s)
  (dolist (mstatement (fetch `(,set-name MEMBERS ?elements)))
	  (if (true? mstatement) (return (setq m-s mstatement))))
  (cond (m-s (values (third m-s) (find-cwa-for-set m-s)))
	(t nil)))

(defun find-cwa-for-set (m-s) ;; (<name> MEMBERS <elements>)
  (dolist (asn (assumptions-of m-s)) 
	  (when (and (equal (car asn) (car m-s))
		     (eq (cadr asn) 'CWA))
		(return-from FIND-CWA-FOR-SET asn))))

(defun close-set (set-name &optional (*LTRE* *LTRE*))
  (multiple-value-bind (known-members known-not)
      (get-set-information set-name)
    (let ((cwa-statement
	   `(,set-name CWA ,known-members ,known-not))
	  (members-statement `(,set-name MEMBERS ,known-members)))
      (assume! cwa-statement ':CWA)
      (unless (true? members-statement) ;; Avoid redundant clauses
       (assert! `(:implies
		  (:and (SET ,set-name)
			,@ (mapcar #'(lambda (el)
				       `(,set-name HAS-MEMBER ,el))
				   known-members)
			,@ (mapcar #'(lambda (el)
				       `(:not (,set-name HAS-MEMBER ,el)))
				   known-not)
			,cwa-statement)
		  ,members-statement)
		':CWA-BASIC-IMPLICATION))
      (values known-members cwa-statement t))))

(defun get-set-information (set-name &optional (*LTRE* *LTRE*))
  (let ((known-in nil) (known-out nil))
    (dolist (possible
	     (fetch `(,set-name HAS-MEMBER ?member)))
      (cond ((true? possible)
	     (push (caddr possible) known-in))
	    ((false? possible)
	     (push (caddr possible) known-out))))
    (values known-in known-out)))

;;;; Contradiction handling and backing up for CWA's

;;; This procedure detects when a CWA lurks within
;;; one of the LTMS' contradictions, and does a THROW 
;;; to a designated tag to signal the assumer that
;;; the set is no longer believed. 

(defun set-cwa-handler (clauses ltms tag &aux asns)
  (dolist (cl clauses)
   (setq asns (assumptions-of-clause cl))
   (dolist (asn asns)
	   (when (equal (datum-lisp-form
			 (tms-node-pdatum asn))
			cwa) ;; got it
		 (retract-assumption asn)
		 (throw cwa tag)))))

;;; When a CWA is no longer needed, it should be retracted
;;; to prevent inappropriate conclusions from it

(defun retract-CWA (set &optional (*LTRE* *LTRE*))
  (dolist (cwa (fetch `(,set CWA ?known ?known-not)))
   (when (true? cwa)
	 (retract! cwa (datum-assumption? (referent cwa))))))

;;;; A simple example

(defmacro With-Closed-Set (set-name &rest body)
  `(multiple-value-bind (members cwa)
       (close-set ,set-name *LTRE*)
     (With-Contradiction-Handler (ltre-ltms *LTRE*)
       #'(lambda (clauses ltms)
	   (set-cwa-handler clauses ltms ':LOST-CWA))
       (let ((answer (catch cwa ,@ body)))
	 (cond ((eq answer ':LOST-CWA) (values nil nil))
	       (t (values t members)))))))

(defun test-cwa (&optional (debugging? nil))
  (In-LTRE (create-ltre "CWA Test" :debugging debugging?))
  (eval *cwa-enforcer*)
  (dolist (data '((set (Parts System))
		  ((Parts System) HAS-MEMBER valve) ;; Assume initial parts
		  ((Parts System) HAS-MEMBER meter)
		  ((Parts System) HAS-MEMBER pump)))
    (assume! data :Initial-Observations))
  (do ((stop? nil))
      (stop?)
  (With-Closed-Set
    '(Parts System) *ltre*
    (do ((form nil)
	 (dropout? nil)
	 (partslist 
	  (remove-if-not #'(lambda (form)
			     (true? form))
			 (fetch `((Parts System) MEMBERS ?els)))
	  (remove-if-not #'(lambda (form)
			     (true? form))
			 (fetch `((Parts System) MEMBERS ?els)))))
	(dropout? (setq stop? t))
      (cond ((cdr partslist)
	     (format t "~%BUG: Conflicting membership statements.")
	     (dolist (pl partslist)
	       (format t "Parts(System) = ~A" pl)))
	    ((null partslist) (setq dropout? t)) 
	    (t (format t "~% Parts are: ~A" (caddr (car partslist)))))
      (cond ((member form '(quit stop end exit)) (setq dropout? t))
	    (t (format t "~%>")
	       (setq form (read))
	       (print (eval form))
	       (run-rules)))))))
