;; -*- Lisp -*-

;;; ATRE database

;; Copyright (c) 1986, 1987, 1988, 1989, 1990, 1991, Kenneth D. Forbus,
;;  Northwestern University, and Johan de Kleer, the Xerox Corporation
;; All rights reserved.

(defun assert! (fact just &optional (*atre* *atre*) &aux datum node)
  (setq datum (referent fact t)
	node (datum-tms-node datum))
  (unless (listp just) (setq just (list just)))
  (debugging-atre "~%    Asserting ~A via ~A." fact just)
  (justify-node (car just) node
		(mapcar #'(lambda (f) (datum-tms-node (referent f t)))
			(cdr just)))
  datum)

(defun assume! (fact reason &optional (*atre* *atre*) &aux datum node)
  (setq datum (referent fact t)
	node (datum-tms-node datum))
  (cond ((not (datum-assumption? datum))
	 (setf (datum-assumption? datum) reason)
	 (debugging-atre "~%    Assuming ~A via ~A." fact reason)
	 (assume-node node))
	((eq reason (datum-assumption? datum)))
	(t (error "Fact ~A assumed because of ~A assumed again because of ~A"
		  (show-datum datum) (datum-assumption? datum) reason)))
  datum)

(defun already-assumed? (fact) (tms-node-assumption? (get-tms-node fact)))

(defun assume-if-needed (fact reason &optional (*atre* *atre*))
  (unless (already-assumed? fact) (assume! fact reason)))

(defmacro rassert! (fact just) `(assert! ,(quotize fact) ,(quotize just)))

(defun contradiction (fact &optional (*atre* *atre*))
  (make-contradiction (datum-tms-node (referent fact t))))

(defmacro rnogood! (informant &rest facts) ;; Takes form of justification
  `(assert! 'False ,(quotize (cons informant facts))))

;;;; Database system

(defun get-class (fact &aux class)
  (cond ((null fact) (error "~% NIL can't be a class."))
	((listp fact) (get-class (car fact)))
	((variable? fact)
	 (cond ((boundp fact) (get-class (symbol-value fact)))
	       (t (error "~%Class unbound: ~A" fact))))
	((symbolp fact)
	 (cond ((setq class (gethash fact (atre-class-table *atre*))) class)
	       (t (setq class
			(make-class :NAME fact :FACTS nil
				    :RULES nil :ATRE *atre*))
		  (setf (gethash fact (atre-class-table *atre*)) class)
		  (push class (atre-classes *atre*))
		  class)))
	(t (error "Bad class type: ~A" fact))))

(defun referent (fact &optional (virtual? nil))
  (if virtual? (insert fact) (referent1 fact)))

(defun referent1 (fact) ;; Could use seperate hash table
  (dolist (candidate (class-facts (get-class fact)))
	  (when (equal (datum-lisp-form candidate) fact)
		(return-from referent1 candidate))))

(defun insert (fact &aux datum)
  (setq datum (referent1 fact))
  (cond (datum (values datum t))
	(t (setq datum (make-datum :COUNTER (incf (atre-datum-counter *atre*))
				   :ATRE *atre*
				   :LISP-FORM fact
				   :CLASS (get-class fact)))
	   (setf (datum-tms-node datum) (tms-create-node (atre-atms *atre*) datum))
	   (push datum (class-facts (datum-class datum)))
	   (try-rules datum)
	   (values datum nil))))

(defun fetch (pattern &optional (*atre* *atre*) &aux bindings unifiers)
  (dolist (candidate (get-candidates pattern) unifiers)
    (setq bindings (unify pattern (datum-lisp-form candidate)))
    (unless (eq bindings 'FAIL)
      (push (sublis bindings pattern) unifiers))))

(defun get-candidates (pattern) (class-facts (get-class pattern)))

;;;; Interface and display of data

(defun true? (fact &optional (*atre* *atre*) &aux r)
  (when (setq r (referent fact nil))
	(true-node? (datum-tms-node r))))

(defun in? (fact env &optional (*atre* *atre*) &aux r)
  (when (setq r (referent fact nil))
	(in-node? (datum-tms-node r) env)))

(defun out? (fact env &optional (*atre* *atre*) &aux r)
  (when (setq r (referent fact nil))
	(out-node? (datum-tms-node r) env)))

(defun consistent-with? (fact env &optional (*atre* *atre*) &aux r)
  (when (setq r (referent fact nil))
	(node-consistent-with? (datum-tms-node r) env)))

(defun why? (fact &optional (*atre* *atre*) (stream *standard-output*)
		  &aux r)
  (when (setq r (referent fact nil))
	(why-node (datum-tms-node r) stream)))

(defun environment-of (facts &optional (*atre* *atre*) &aux node env)
  (setq env (atms-empty-env (atre-atms *atre*)))
  (dolist (fact facts)
	  (setq node (get-tms-node fact *atre*))
	  (unless (tms-node-assumption? node)
		  (error "Non-assumption in ENVIRONMENT-OF: ~A." fact))
	  (setq env (cons-env node env))
	  (when (env-nogood? env)
		(return-from ENVIRONMENT-OF (values nil env))))
  env)

(defun environment-cons (fact env)
  (cons-env (get-tms-node fact) env))    

(defun view-env (env)  (mapcar #'view-node (env-assumptions env)))

(defun justifications (fact &optional (*atre* *atre*) (stream *standard-output*))
  (node-justifications (get-tms-node fact *atre*) stream))
    
;;; More interrogatives

(defun the-e (num &optional (*atre* *atre*)) (e (atre-atms *atre*) num))

(defun get-tms-node (fact &optional (*atre* *atre*))
  (datum-tms-node (referent fact t)))

(defun view-node (node)
  (datum-lisp-form (tms-node-datum node)))

(defun assumptions-of (fact)
  (tms-node-label (datum-tms-node (referent fact t))))

(defun show-datum (datum)
  (format nil "~A" (datum-lisp-form datum)))

(defun show-data (&optional (*atre* *atre*) (stream *standard-output*)
		       &aux counter)
  (setq counter 0)
  (format stream 
	  "~%~D facts total." (atre-datum-counter *atre*))
  (dolist (class (atre-classes *atre*) counter)
    (dolist (datum (class-facts class))
      (incf counter)
      (format stream "~%~A: ~A" (show-datum datum)
	      (assumptions-of (datum-lisp-form datum)))))
  counter)

(defun show-context (env &optional (*atre* *atre*) (stream *standard-output*)
		       &aux counter)
  (setq counter 0)
  (dolist (class (atre-classes *atre*))
    (dolist (datum (class-facts class))
      (when (in-node? (datum-tms-node datum) env)
	(incf counter)
	(format stream "~%~A" (show-datum datum)))))
  (format stream  "~%~D facts total." counter)
  counter)

(defun show-classes (&optional (*atre* *atre*) (stream *standard-output*)
			  &aux counter)
  ;; Handy for finding buggy assertions
  (setq counter 0)
  (dolist (class (atre-classes *atre*) counter)
    (incf counter)
    (format stream "~% ~A: ~D facts, ~D rules"
	    (class-name class) (length (class-facts class))
	    (length (class-rules class)))))

(defun get-datum (num &optional (*atre* *atre*))
  (maphash #'(lambda (key class)
	       (declare (ignore key))
	       (dolist (datum (class-facts class))
		 (when (= (datum-counter datum) num)
		   (return-from GET-DATUM datum))))
	   (atre-class-table *atre*)))

(defun get-just (num &optional (*atre* *atre*))
  (dolist (just (atms-justs (atre-atms *atre*)))
    (when (= (just-index just) num)
      (return-from GET-just just))))
