;; File:  var-typing.lisp
;;
;; Author: Rob Spiger
;; Sponsoring Professor: Oren Etzioni
;; Date:  Jan 1993
;; Docs: In this file.
;;
;; The purpose of this code is to:
;;
;; 1.  Figure out typing information for every variable in the PSG.
;; 2.  To apply the typing information to limit the choice of operators
;;     to achieve goals.  For example, in the strips world you can rule
;;     out the operator push-thru-door to achieve the goal (inroom <key> <room>)
;;     if you know the generator for the variable <key> was (is-key <door <key>) because
;;     you know it isn't an object which can be pushed.
;;
;;  NOTE:  ALL OF THIS TYPING INFORMATION DEALS ONLY WITH STATIC PREDICATES, NO OPEN
;;         WORLD PREDICATES OR ACHIEVABLE PREDICATES ARE USED TO DEAL WITH TYPING INFORMATION.
;;         THERE SHOULD BE NO NON-STATIC PREDICATES IN THE TYPING-INFO-LIST.
;;
;;  NOTE:  This typing information is not used in conjunction with the simplifier.
;;         This typing information is used only to limit the choice of operators when
;;         creating the PSG.  
;;         For information about typing variables for use by the simplifer see
;;         rob-simplify/rob-bindings.lisp
;;
;;
;;  The template for the typing-info-list is:
;;
;; 1.  variable typing information consisting of a list of:
;;
;;      A.  each static predicate in the domain with variables in it.
;;      B.  each other static predicate those variables may occur in.
;;
;; 2.  constant typing information consisting of a list of:
;;
;;      A.  a constant
;;      B.  a list of the allowed static predicates for the constant.
;;
;; If no typing information is present then variable typing information
;;  will not be used.  (no typing information is considered to be present
;;  when the *typing-info-list* is set to '(nil nil))  If any typing information
;;  is present in *typing-info-list* then STATIC assumes the list of typing
;;  information is complete.  (In other words, don't enter it partially.)
;;
;;  

;; Method:
;;
;; For each variable the generator for it is the first static precondition in which it appears
;;  in an operator.  

;;  (A special case is variables in shared literals which are stored as type 
;;  'unknown.  See note below in documentation of the function 
;;  find-typing-information-for-shared-lits.)  
;;

;; Assuming no shared literals take the following operator as an example:
;;
;;(LOCK
;; (params (<door2> <k1> <rm-b>))
;; (preconds
;;     (and (is-door <door2>)
;;	  (is-key <door2> <k1>)
;;	  (holding <k1>)
;;	  (dr-to-rm <door2> <rm-b>)
;;	  (inroom <k1> <rm-b>)
;;	  (next-to robot <door2>)
;;	  (dr-closed <door2>)
;;	  (unlocked <door2>)))
;; (effects 
;;    ((del (unlocked <door2>))
;;     (add (locked <door2>)))))
;;
;; The type for <door2> would be stored as (is-door <door2>).
;; The type for <k1> would be stored as (is-key <door2> <k1>).
;; The type for <rm-b> would be stored as (dr-to-rm <door2> <rm-b>).

;; Now that these types have been stored, when STATIC looks for operators to 
;; achieve (inroom <k1> <rm-b>).  The typing information will be used to rule out
;; operators which would create type conflict.  Take for example the operator
;; PUSH-THRU-DR.  The effect (inroom <b-x> <r-x>) would be matched with 
;; (inroom <k1> <rm-b>).  This would make the static predicates of the operator
;; imply that (pushable <k1>) would result as a precondition once the bindings took place.
;; The generator <k1> is looked up and found in the *typing-info-list*.  The allowable
;; static predicates for the type (is-key <door2> <k1>) do not include (Pushable <k1>) so
;; the operator is rejected because it would result in a typing conflict.
;; 
;; The same type of type checking happens for constants too.  But there is a slight difference.
;; No generators for constants are found by static.  There is just type checking
;; to make sure that what static predicates the constants end up in are legal.
;; None of the domains with a typing-info-list had static predicates in which constants 
;; were allowed to be inside static predicates.  But the ability has been included in STATIC
;; to handle domains in which this is not the case.  The second element of the typing-info-list
;; deals with typing for constants in the domain.  For each constant a list of which static
;; predicates it can legal appear in should paired with the constant.  For example, in the
;; extended-stripsworld.  If (is-object ROBOT) is (pushable ROBOT) were legal static
;; predicates (that is, had a chance at ever being true.) then the second element of the
;; typing-info-list would be ((ROBOT ((is-object ROBOT) (pushable ROBOT)))) instead of 
;; ((ROBOT nil)).
;; 
;; 
;;
;;



;;Because variables in shared literals can have different types depending on which operator
;; they are in they are a special case.  The solution is to only allow them to
;; have the types which are also shared.  But this is not simply the shared literals
;; themselves because it is not necessarily true the that shared literals are all 
;; ANDed into the preconditions of every operator in which they appear.  (For example,
;; in one operator the typing (is-object <V1>) might be ANDed to the rest of the
;; preconditions and in another operator it might just be ORed with the rest of the 
;; preconditions.  It is therefore not safe to conclude the type of <V1> is (is-object <V1>)
;; even though (is-object <V1>) might be a shared literal.  )  
;; A complete solution would compute the true types of each variable inside the shared
;; literals by finding which typing it was in every operator over which the literal is
;; shared.  Then only the type which was present for the variable in every operator would 
;; be kept.  This was not implemented.  Instead, the type of each variable in the
;; shared literals was set to 'unknown.
;; 

(defun find-typing-information-for-shared-lits (shared-lits)
 (let* (
   (vars (find-variables shared-lits)))
   (set-all-vars-to-type vars 'unknown)))


;;The function find-typing-information-for-op-node 
;; finds all the variables in the operator node and stores var
;; typing information for them.  The variable typing which is stored for
;; each variable is the first static predicate which contains the variable
;; in the operator.  If a variable occurs in the operator which has already
;; had typing information defined for it, the orginal typing information is not
;; changed.
;; If no typing for a variable is found its type is set to 'unknown.
;;
;; For the variables in the literal above the operator in the PSG, all the
;; types are set to unknown if not already set.
;;
;; This function could be improved by having it find the types of variables
;; inside forall statements by snatching their generators.


(defun find-typing-information-for-op-node (op preconds)
 (when (AND *use-typing-info-list*
            (not (equal '(nil nil) *typing-info-list*)))
  (let* (
    (parent-lit (literal-name (operator-parent op)))
    (parent-vars (find-variables parent-lit))
    (effects  (operator-effects op))
    (type-preds (find-anded-static-preds-in-exp preconds))
    (type-preds (remove-if-not #'is-static-pred-and-not-meta-func type-preds)))
   (set-all-vars-to-type parent-vars 'unknown)
   (extract-typing-info-from-static-preconds type-preds))))

;;UNLESS already defined each variable in vars has its type set to the type type.

(defun set-all-vars-to-type (vars type)
 (unless (null vars)
  (progn
   (when (null (get (first vars) 'type))
    (putprop (first vars) type 'type))
   (set-all-vars-to-type (rest vars) type))))



(defun extract-typing-info-from-static-preconds (type-preds)
 (unless (null type-preds)
  (progn
   (set-all-vars-to-type (find-variables (first type-preds)) (first type-preds))
   (extract-typing-info-from-static-preconds (rest type-preds)))))


;;makes sure that the variables in the goal don't change type and have a type conflict
;; when being bound by the blist bindings.  This is done by looking at the typing-info
;; for the operator and determinining using the list *typing-info-list* if a type 
;; conflict is occuring.

;;goal can not be negated

(defun typing-info-makes-op-fail (goal typing-info blist)
 (unless (OR (not *use-typing-info-list*)
             (equal '(nil nil) *typing-info-list*))
  (unless (eq blist 'no-match)
   (let* (

     (vars-blist      (remove-if-not #'(lambda (bpair) (AND (is-variable (first bpair))
                                                            (is-variable (second bpair))))
                       blist))
     (blist           (remove-if     #'(lambda (bpair) (AND (is-variable (first bpair))
                                                            (is-variable (second bpair))))
                       blist))
     (goal-vars       (find-variables goal))
     (op-consts-blist (remove-if-not #'(lambda (bpair) (member (first bpair) goal-vars))
                       blist))
     (goal-consts-blist (remove-if     #'(lambda (bpair) (member (first bpair) goal-vars))
                         blist)))
   (OR 
    (typing-info-makes-op-fail-from-goal-consts goal typing-info goal-consts-blist)
    (typing-info-makes-op-fail-from-op-consts goal typing-info op-consts-blist)
    (typing-info-makes-op-fail-from-vars goal typing-info vars-blist))))))


;;This is the case where the bpair has a variable from the operator bound to a constant
;; in the goal.
;;
;;This should fail if the constant in the goal can't be present in the static predicates
;; which the operator variable is present in.

(defun typing-info-makes-op-fail-from-goal-consts (goal typing-info blist)
 (unless (null blist)
  (let* (
    (op-var        (first (first blist)))
    (goal-consts   (second (first blist)))
    (occurances    (find-applicable-typing-predicates op-var typing-info))
    (occurances    (instantiate occurances (list (first blist))))
    (goal-consts-types (get goal-consts 'type)))
   (OR (find-if #'(lambda (occurance) 
                        (not (member occurance goal-consts-types 
                                          :test #'match-nonvar-or-var-to-var-only)))
        occurances)
       (typing-info-makes-op-fail-from-goal-consts goal typing-info (rest blist))))))

    
;;This is the case where the bpair has a variable from the goal bound to a constant
;; in the operator.  If the generator for the variable is false when instantiated with
;; the constant will signal failure.

(defun typing-info-makes-op-fail-from-op-consts (goal typing-info blist)
 (unless (null blist)
  (let* (
    (goal-var       (first (first blist)))
    (op-consts      (second (first blist)))
    (goal-var-typing (get goal-var 'type))
    (const-typing    (get op-consts 'type))  ;;is list of static's consts may appear in
    (occurance       (instantiate goal-var-typing (list (first blist)))))
   (OR (unless (OR (null goal-var-typing) (eq goal-var-typing 'unknown))
        (not (member occurance const-typing :test #'match-nonvar-or-var-to-var-only)))
       (typing-info-makes-op-fail-from-op-consts goal typing-info (rest blist))))))


;;This is the case where the bpair has a variable from the goal bound to a variable
;; from the operator.  What has to be true is for each static predicate that the
;; variable in the operator appears in, the variable bound from the goal must have
;; a generator which is compatible with the new types it will assume by binding to
;; the variable from the operator.


(defun typing-info-makes-op-fail-from-vars (goal typing-info blist)
 (unless (null blist)
  (let* (
    (goal-var (second (first blist)))
    (op-var   (first  (first blist)))
    (goal-var-typing (get goal-var 'type)))
   (if (OR (null goal-var-typing) (eq goal-var-typing 'unknown))
    (typing-info-makes-op-fail-from-vars goal typing-info (rest blist))
    (let* (
      (goal-var-typing (instantiate goal-var-typing (list (reverse (first blist)))))
      (occurances      (find-applicable-typing-predicates op-var typing-info))
      (pred-allowable  (find-if #'(lambda (pair)
                          (match-nonvar-or-var-to-var-only goal-var-typing (first pair)))
                          (first *typing-info-list*)))
      (match-blist     (rob-match (first pred-allowable) goal-var-typing))

      (allowable-occurances (instantiate (second pred-allowable) match-blist)))
     (OR (find-if #'(lambda (occurance) (not (member occurance allowable-occurances 
                   :test #'(lambda (pred1 pred2) (is-subset-without-matching-var pred1 pred2 op-var)))))
                occurances)
         (typing-info-makes-op-fail-from-vars goal typing-info (rest blist))))))))


;;returns true if pred1 is a subset of pred2 with no matching occurring using the variable 
;; var.

(defun is-subset-without-matching-var (pred1 pred2 var)
 (when (match-nonvar-or-var-to-var-only pred1 pred2)
  (let* (
    (blist (rob-match pred1 pred2)))
   (unless (eq blist 'no-match)
    (unless (find-if #'(lambda (bpair) (member var bpair)) blist)
     t)))))

 

    
;;returns those predicates from typing-info which have var in them.
(defun find-applicable-typing-predicates (var typing-info)
 (remove-if-not #'(lambda (pred) (member var (find-variables pred))) typing-info))

  

