;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Main.lisp contains 
;;
;; Contains top level calls for static

;;
;; Author: Julie Roomy
;; Modified by: Rob Spiger
;; Sponsoring Professor: Oren Etzioni
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;-------
;; static
;; if continue is true will perform all of static without prompting.
;; if continue is missing or false will prompt after completing each
;; step.
;; if unsolvable-prob is t then prodigy will assume the domain
;;  contains unsolvable-problems and will create control rules
;;  for top-level node rejection rules.
;;
(defun static (&optional (continue t) (all-psgs t) (unsolvable-probs t))
  (initialize-static)
  (if (null all-psgs)
      (progn
	(format t 
           "~%~%Do you wish to create problem space graphs (PSG's) for all ~%~
           of the achievable preconditions of the current domain (T nil)?")
	(setq all-psgs (read)) (format t "~%")))
  

  (create-the-psgs all-psgs)

  (format t "~%Finding variable types.")
  (find-var-types)
  (format t "~%Done finding variable types.")

  (let ((labeled (label-the-psgs continue)))

    ;; if the psgs were not labeled there is no reason to continue
    (if labeled
	(progn
      (let ((ne-found (find-ne-of-the-psgs continue)))

       (if ne-found

;;have ne here
        (progn
	  ;; check to see if supposed to prompt
	  (if (null continue)
	      (progn
		(format t "~
                  ~%Do you wish to create search control rules based upon ~%~
                  failure and success nodes and necessary effects(T nil)?")
		(setq continue (read)) (format t "~%")))

	  (if continue
	      (progn
        (unless unsolvable-probs
         (progn
          (format t "~%Is this a domain which contains unsolvable")
          (format t "~%problems(T nil)?")
          (setq unsolvable-probs (read)) (format t "~%")))

        (when unsolvable-probs
         (create-rules-for-unsolvable-probs))
        (learn-from-necessary-effects)
		(learn-from-failure)
		(learn-from-success)
		(feed-rules unsolvable-probs)
		)))))))))

(defun initialize-static ()
  (format t "~%~%Initializing static...")
  (let ((domain (read-domain)))  

    ;; in case this is the second time create-the-psgs is being called
    ;; reset *roots* to nothing and unlearn rules.
    (setq *roots* nil) 
    (forget-all-rules)
 
 
    ;; generate domain-dependent axioms 
    ; (make-wild-cards)	don't want to parse preconds more than once.
    (make-non-gratis-list domain)
    (make-invariant-list domain)
    (make-false-list domain)
    (make-non-goals domain)
    (make-bad-ops domain)	
    (make-negation-list domain)
    (make-rarely-unique-list domain)
    (make-typing-info-list domain)
    (initialize-typing-info-constants)
    (make-dont-expand-list domain)

    (initialize-rob-simplify)

    ;; forget any rules that are stored in system 
    (iter:iterate          
     (iter:for op iter:in *achievers*)
     ;; reset rules
     (setf (get (car op) 'rules) nil)
     ;; initialize original-op-name field
     (setf (get (car op) 'op) (car op)))

    ;; initialize *achievers*
    (setq *achievers* (append *operators* *inference-rules*))

    (format t "~%Preprocessing operators...")
    (preprocess)
    (format t "~%Done preprocessing operators.")

    ;; generate achievables
    (setq *achievables*
	  (constify-goals
	   (remove-non-goals
	    (remove-non-gratis (find-achievable-preds *achievers*)))))

    
  
  (format t "~%~%Done Initializing static.")))


;;----------------
;; create-the-psgs, creates a psg for each achievable predicate 
;; in the domain.  It sets the variable *roots* to a list containing
;; all of the created psgs.
(defun create-the-psgs (create-all)

    (format t "~%~%CREATING PSGS~%~%")
    ;; create psgs
    (if create-all
	(iter:iterate (iter:for goal iter:in *achievables*)
		      (setq *roots* (append (create-psg goal) *roots*))
;;		      (gc t)
                   )

      
      ;; prompt user for which roots to expand
      (progn
	(setq *roots* 
	      (append (create-psg (prompt-user-for-root *achievables*))
		      *roots*))
;;	(gc t)
          ))
    (format t "~%~%Finished creating PSGs.  "))


;;---------------------
;; prompt-user-for-root
;; prompt the user for which of the achievables she or he wishes to 
;; expand
(defun prompt-user-for-root (choices)
  (format t "~% possible roots to expand are: ~%")
  (iter:iterate
   (iter:for root iter:in choices)
   (iter:for i from 1 to (length choices))
   (format t "~s ~s ~%" i root))
  (format t "Which root would you like to expand? ~% ~
             Please enter a number: ")
  (let ((users-choice (read)))
    (format t "~%")
    (nth (1- users-choice) choices)))


;;---------------
;; label-the-psgs
(defun label-the-psgs (continue)
  (if (null continue)
      (progn
	(format t "~%Do you wish to label the PSGs? (T, Nil)")
	(setq continue (read)) (format t "~%")))
    
  (if continue
      (progn
	(format t "~%~%LABELING PSGS~%~%")
	(label-PSGs *roots*)
	(format t "~%~%Finished labeling PSGs.  ")))
  continue)

;;-----------------
;; find the necessary-effects
(defun find-ne-of-the-psgs (continue)
 (if (null continue)
  (progn
   (format t "~%~%Do you wish to find the necessary effects")
   (format t "~%of the PSGS? (T or nil) ")))
 (when (or continue (read))
  (format t "~%~%Finding necessary effects of PSGS~%")
  (find-ne-of-roots *roots*)
  (format t "~%Finished finding necessary effects of the PSGS.~%")
  T))


;;-------------------
;; learn-from-failure
(defun learn-from-failure ()
  (format t "~%~%FORMING FAILURE RULES~%~%")
  (setq *frules* nil)
  (setq *brules* nil)
  (iter:iterate
   (iter:for goal iter:in *roots*)
   (form-operator-rejection-rules goal))
  (setq *all-rules* *frules*)
  (setq *frules*
	(iter:iterate
	 (iter:for r iter:in *frules*)
	 (let* ((lhs (sixth r))
		(op-name (third r))
		(lhs-list (get op-name 'rules)))
	   (if (more-general-rule-already-learned? 
		lhs lhs-list)
	       (setf (get op-name 'rules)
		     (remove lhs lhs-list :test #'equal))
	     (iter:collect r)))))
  (format  t "Learned ~s reject rules" (length *frules*)))


;;-------------------
;; learn-from-success
(defun learn-from-success ()
  (format t "~%~%FORMING PREFERENCE RULES~%~%")
  (create-op-success-rules *roots*))
      
;;-------------------------
;;learn-from-necessary-effects
(defun learn-from-necessary-effects ()
  (format t "~%FORMING PREFERENCE RULES FROM NECESSARY EFFECTS.")
  (use-necessary-effects-to-make-sc-rules)

  (format t "~%FORMING GOAL AND OPERATOR REJECTION RULES FROM NECESSARY EFFECTS.")
  (make-rejection-rules-using-ne))





;;--------------------------------------------
;; rule code taken from EBL ebl-top-level.lisp
;;--------------------------------------------

;;-----------------
;; forget-all-rules
(defun forget-all-rules ()
  (g-loop (init r nil)
          (while (setq r (pop *LEARNED-RULES-IN-SYS*)))
          (do (remove-scr r)))
  (dolist (r *LEARNED-RULES*) (setf (symbol-plist r) nil))
  (setq *LEARNED-RULES* nil)
  (setq *NEW-LEARNED-RULES* nil)
  (setq *rule-name-counter* 0)
  t)


(defun remove-scr (rule-nm)
  (let  ((rule-lst-nm (rule-type-to-rule-lst (get rule-nm 'rule-type)))
         rule)
    (setq rule (assoc rule-nm (eval rule-lst-nm)))
    (cond ((null rule)
           (format t "~%Rule ~A doesn't exist in system: not removed" rule-nm))
          (t (set rule-lst-nm
                  (del-eq rule (eval rule-lst-nm)))
             (format t "~%Rule ~a removed from system" rule-nm)))))


; returns NAME of rule-lst
(defun rule-type-to-rule-lst (rule-type)
  (cadr (assoc rule-type '((node-select *SCR-NODE-SELECT-RULES*)
                           (goal-select *SCR-GOAL-SELECT-RULES*)
                           (op-select *SCR-OP-SELECT-RULES*)
                           (bindings-select *SCR-BINDINGS-SELECT-RULES*)
                           (node-reject *SCR-NODE-REJECT-RULES*)
                           (goal-reject *SCR-GOAL-REJECT-RULES*)
                           (op-reject *SCR-OP-REJECT-RULES*)
                           (bindings-reject *SCR-BINDINGS-REJECT-RULES*)
                           (node-pref *SCR-NODE-PREFERENCE-RULES*)
                           (goal-pref *SCR-GOAL-PREFERENCE-RULES*)
                           (op-pref *SCR-OP-PREFERENCE-RULES*)
                           (bindings-pref *SCR-BINDINGS-PREFERENCE-RULES*)))))





;****************************************************************
; Rule subsumption stuff.

;;-----------------------------------
;; more-general-rule-already-learned?  ;;?? move to rules.lisp

; hack. I want the rules without ogs because they are more
; general.  However, in my current rep of the lhses ogs(on-table) is
; equivalent to on-table being on the lhs. Thus, I get the appropraite
; bw rule for rejecting pick-up, only because I process holding before
; on-table.  The fix is to make a richer representation of the lhses
; and encode the bias into the subsumption check. Boring.

; I check that the length of the new-lhs is > than the old one.  If
; they're equal it might be the same rule, and if the new-lhs is
; shorter than I want to keep it.
; hack: will fail on disj lhss.    ;;?? test and change
(defun more-general-rule-already-learned? (new-lhs old-lhses)
  (iter:iterate
   (iter:for l iter:in old-lhses)
   (iter:thereis (and
             (> (length new-lhs) (length l))
             (match l new-lhs)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Utility Routines to generate achievables
;;
;; taken from bruce's static

; Generates goals with unique constants in the goals.
; hack.  Automatically disallows goals such as on(x,x) which isn't
; general.
(defun constify-goals (goals)
  (iter:iterate
   (iter:for g iter:in goals)
   (iter:collect (cons (car g)
		  (iter:iterate
		   (iter:for x iter:in (cdr g))
		   (if (rob-is-var? x)
		       (iter:collect (generate-goal-const))
		     (iter:collect x)))))))


(defun remove-non-gratis (preds)
  (iter:iterate
   (iter:for p iter:in preds)
   (unless (or (member (car p) *non-gratis-list*)
	       (and (eq (car p) '~)
		    (member (car (second p))
			    *non-gratis-list*)))
	   (iter:collect p))))

(defun remove-non-goals (preds)
  (iter:iterate
   (iter:for p iter:in preds)
   (unless (or (non-goal? p)
	       (and (eq (car p) '~)
		    (non-goal? (second p))))
	   (iter:collect p))))

(defun non-goal? (p)
  (let ((pred (car p))
	)
    (find-if 
     #'(lambda (g)
      (when (eq pred (car g))
       (or
 	    (equal nil (cadr g))
 	    (unless
          (find-if
           #'(lambda (bad-pred)
              (not (member bad-pred p)))
           (rest g))))))
     *non-goals*)))
 

; uniquify-vars removed since the operators are assumed
; to have unique vars.
(defun find-achievable-preds (ops)
  (remove-duplicates
   (mapcan #'(lambda (op) (find-adds (car op))) ops)
   :test #'(lambda (x y)
	     (and (eq (car x) (car y))	;same pred
		  (iter:iterate
		   (iter:for v iter:in (cdr x))
		   (iter:for w iter:in (cdr y))
		   (iter:always (or (and
				(rob-is-var? v)
				(rob-is-var? w))
			       (eq v w)))) ;same constant
		  ))))




(defun find-adds (op-name)
  (find-effects (get op-name 'effects) 'add))



; removes the type designation.
(defun find-effects (effects type)
  (iter:iterate (iter:for e iter:in effects)
		(when (eq (car e) type)
		      (iter:collect (if (eq type 'add)
				   (second e)
				 (list '~ (second e)))
			       ))))



(defun rem-equal (e l)
  (remove e l :test #'equal))



