;;****************************************************************
;;****************************************************************
;;
;; by:   Oren Etzioni
;; mod:  Bruce LeSourd
;; mod:  Julie Roomy
;; mod:  Rob Spiger
;;
;; date: Jan 1992
;;
;; IO FUNCTIONS.
;;
;; Top level routines:
;;      (browse)
;;      (print-psgs)
;;      (print-psg <root> <file-name>)
;;      (show-currently-loaded-rules)
;;
;;****************************************************************
;;****************************************************************

;;---------
;; show-lit
;; show all the fields of a literal in a pleasant format
(defun show-lit (lit)
  (format t "~%")
  (format t "** LITERAL ** ~%")
  (format t "NAME:       ~s ~%" (literal-name lit))
  (format t "bad-bindings: ~s ~%" (literal-bad-bindings lit))
  (format t "bindings:   ~s ~%" (literal-bindings lit))

  ;; check for a shared node
  (if (null (Literal-shared lit))
      (format t "parent:     ~s ~%" (just-show-name (literal-parent lit)))
    (format t "parents:     ~s ~%" (just-names 
				    (cons (literal-parent lit)
					  (Literal-shared lit)))))


  (format t "node already traversed: ~s ~%" (Literal-already-traversed lit))
  (format t "goal stack: ~s ~%" (literal-goal-stack lit))
  (format t "necessary-effects ~s ~%" (literal-necessary-effects lit))
  (format t "label:      ~s ~%" (literal-label lit))
  (format t "failure condition: ~s ~%" (literal-failure-condition lit))
  (format t "operators:  ~s ~%" (just-names (Literal-operators lit)))
  (format t "node completely shared ~s ~%" (Literal-completely-shared lit))
  )



;;--------
;; show-op
;; show all the fields of an operator in a pleasant format
(defun show-op (op)
  (format t "~%")
  (format t "** OPERATOR ** ~%")
  (format t "NAME:          ~s ~%" (operator-name op))
  (format t "parent(goal):  ~s ~%" (just-show-name (operator-parent op)))
  (format t "effects:       ~s ~%" (operator-effects op))
  (format t "bindings:      ~s ~%" (operator-bindings op))
  (format t "necessary-efs  ~s ~%" (operator-necessary-effects op))
  (format t "label:         ~s ~%" (operator-label op))
  (format t "failure conditions: ~s ~%" (operator-failure-condition op))
  (format t "precondition(s):    ~s ~%" 
	  (just-show-name (operator-preconditions op))))



;;--------------
;; show-internal
;; show all the fields of an internal in a pleasant format
(defun show-internal (node)
  (format t "~%")
  (format t "** INTERNAL NODE ** ~%")
  (format t "NAME:       ~s ~%" (InternalNode-name node))
  (format t "bindings    ~s ~%" (InternalNode-bindings node))
  (format t "parent:     ~s ~%" 
	  (just-show-name (InternalNode-parent node)))
  (format t "goal stack: ~s ~%" (InternalNode-goal-stack node))
  (format t "necess efs  ~s ~%" (Internalnode-necessary-effects node))
  (format t "ANDed preds ~s ~%" (Internalnode-anded-static-preds node))
  (format t "label:      ~s ~%" (InternalNode-label node))
  (format t "failure condition: ~s ~%" (InternalNode-failure-condition node))
  (format t "operands:   ~s ~%" 
	  (just-names (InternalNode-operands node))))


;;-----------
;; just-names
;; given a list containing structures, return just the names in the list
(defun just-names (some-list)
  (iter:iterate
   (iter:for node iter:in some-list)
   (iter:for index from 1 to (length some-list))
   (iter:collect (just-show-name node))))

	
;;---------------
;; just-show-name
;; don't show all of the information associated with a node
;; because this would cause an infinite loop.  So just return the 
;; name of the node.
(defun just-show-name (node)
  (cond
    ((internalnode-p node)
     (internalnode-name node))
    ((literal-p node)
     (literal-name node))
    ((operator-p node)
     (operator-name node))
    (T 'nil)))


;;----------
;; show-node
;; given any node, determine its type and show the node in a pleasant
;; format
(defun show-node (node)
  (cond
    ((internalnode-p node)
     (show-internal node))
    ((literal-p node)
     (show-lit node))
    ((operator-p node)
     (show-op node))
    ((null node)
     (format t "~%nil ~%"))
    (T (format t "~%Unknown type of node ~%"))))



;;-----------------------------------------------
;; BROWSE
;; Menu-driven, follow the instructions.
;;
;; ideas for futher implementation:
;;   stack representing path from psg-root to current node.

(defun browse (&optional root)
  (format t
	  "Welcome to STATIC browser.~%~%")
  (format t
	  "Enter screen width:  ")
  (setq *screen-width* (read)) (format t "~%")
  (loop
      (format t "~%Current node:  ~% ")
      (if (null root)
	  (setq root (initiate-browse)))
      (show-node root)
      (format t
	      "~%Select Option:  Down Up Show-subtree Quit (DdUuSsQq) ")
      (let ((r (read)))
	(format t "~%")
	(cond ((or (eq r 'D)
		   (eq r 'd))
	       (setq root (traverse-Down root)))
	      ((or (eq r 'U)
		   (eq r 'u))
	       (setq root (traverse-Up root)))
	      ((or (eq r 's)
		   (eq r 'S))
	       (show-subtree root))
	      ((or (eq r 'q)
		   (eq r 'Q))
	       (return))))))



;;----------------
;; initiate-browse
;; if no root was passed in, ask user which root she or he would like
;; to start with.
(defun initiate-browse ()
  (let (r)
    (iter:iterate
     (iter:for root iter:in *roots*)
     (iter:for i from 1 to (length *roots*))
     (format t "~s ~s ~%" i (literal-name root)))
    (format t "Which root would you like to expand? ~%  ~
               Please enter a number: ")	  
    (setq r (read)) (format t "~%")
    (nth (1- r) *roots*)))




;;--------------
;; traverse-down
;; the user has chosen to move down through the tree.
;; check to see what the current node is, then move to it's child
;; if the current node has many children prompt the user for 
;; which child to move to.
;; if the current node has no children, issue an error message and
;; return the current node
(defun traverse-down (current)
  (cond 

   ;; current node is a literal?
   ((literal-p current)
    (let ((operators (literal-operators current)))
      (cond
       ;; if there are no relevant operators, then return
       ((null operators)
	(progn
	  (format t "End of branch.  Unable to go forward.~%")
	  current))
       ;; if there is only one operator, choose it.
       ((eq 1 (length operators))
	(first operators))
       ;; otherwise ask which operator to traverse
       (T      
	(prompt-for-child operators)))))

   ;; current node is an operator?
   ;; there is only one top-level precondition
   ((operator-p current)
    (operator-preconditions current))
   
   ;; current node is an internal node?
   ;; prompt the user which operand to traverse
   ((InternalNode-p current)
    (cond
     ;; if there are no operands, tell the user and return current node
     ((eq 0 (length (InternalNode-operands current)))
      (progn
	(format t "This node has no operands~%")
	'current))
     ;; if there is only one operand return it
     ((eq 1 (length (InternalNode-operands current)))
      (first (InternalNode-operands current)))
     (T (prompt-for-child (InternalNode-operands current)))))

   ;; error
   (T "error")))


;;-----------------
;; prompt-for-child
(defun prompt-for-child (children)
  (let (index)
    (format t "Available children are:~%")
    (iter:iterate
     (iter:for child iter:in children)
     (iter:for i from 1 to (length children))
     (format t "~s ~s ~%" i (just-show-name child)))
    (format t "Which child would you like to expand? ~%  ~
               Please enter a number: ")
    (setq index (read)) (format t "~%")
    (nth (1- index) children)))


;;------------
;; traverse-up
;; user desires to visit the parent of the current node,
;; so find out what kind of node the current node is then
;; return it's parent.
;; Note: shared literals may have more than one parent.
(defun traverse-up (current)
  (cond

    ;; if the current node is a literal
    ((literal-p current)
     ;; check for root (ie, no parent)
     (cond
       ((null (literal-parent current))
	;; if there is no parent reprompt user with all the roots
	(initiate-browse))
       ;; check for shared node
       ((literal-shared current)
	(progn
	  (let 
	      ((index (just-names (cons (literal-parent current)
					(literal-shared current)))))
	    (format t "Which parent would you like to expand?~%~
                 Available parents are ~s.  ~%Please enter the position ~
                 of the parent as a number: " index)
	    (setq index (read)) (format t "~%")
	    (if (eq index 1)
		(literal-parent current)
	      (nth (- index 2) (literal-shared current))))))
       ;; not shared then just return parent
       (T (literal-parent current))))

    ;; if the current node is an operator
    ((operator-p current)
     (operator-parent current))

    ;; if the current node is an internal node
    ((InternalNode-p current)
     (InternalNode-parent current))

    (T "error")))

       

;;-------------
;; show-subtree       
;; user desires to see the subtree with the current node as the root
(defun show-subtree (root)
  (let (limit)
    (format t
	    "Show PSG to what depth from current node? ")
    (setq limit (read)) (format t "~%")
    (psg-depth-first root *screen-width* 0 limit)))


;;----------------
;; psg-depth-first
;; in order to show a fair amount of the tree just show the 
;; type of node and name of node for now
(defun psg-depth-first (root screen-width depth limit)
  (if (< depth limit)
      (progn 
	(tab depth screen-width)

	;; show the root
	(let ((type (return-type root))
	      (name (return-name root)))
	  (format t "~s:  ~s~%" type name)
	  
	  ;; show the root's children - depth first
	  (cond 
       
	   ;; an operator only has one child - its preconditions
	   ((eq type 'OPERATOR)
	    (psg-depth-first
	     (Operator-preconditions root) screen-width (1+ depth) limit))

	   ;; an internal-node has at least one child - its operands
	   ((eq type 'INTERNAL)
	    (iter:iterate
	     (iter:for operand iter:in (InternalNode-operands root))
	     (psg-depth-first
	      operand screen-width (1+ depth) limit)))


	   ;; a literal may have any number (maybe none) 
	   ;;  of children - it operators
	   ((eq type 'LITERAL)
	    (show-literal-children 
	     (Literal-operators root) screen-width (1+ depth) limit))

	   (T (format t "error")))))))



(defun show-literal-children (operators screen-width depth limit)
  (if operators
      (progn
	(psg-depth-first (first operators) screen-width depth limit)
	(show-literal-children (rest operators) screen-width depth limit))))
	   


(defun return-type (node)
  (cond ((operator-p node)
	 'OPERATOR)
	((literal-p node)
	 'LITERAL)
	((InternalNode-p node)
	 'INTERNAL)))


(defun return-name (node)
  (cond ((operator-p node)
	 (Operator-name node))
	((literal-p node)
	 (Literal-name node))
	((InternalNode-p node)
	 (InternalNode-name node))))

;;----
;; tab moves the position of the cursor over two places * depth
;; if this is greater than limit a line of dashes is shown and
;; the indentation begins again at the left hand side.
(defun tab (depth limit)
  (if (> depth (- limit 30))   ;; assumes literal is not longer than 30 chars
      (progn
	(format t ">")
	(tab (- depth (- limit 20)) limit))

    (if (> depth '0 )
	(progn 
	  (format t " ")
	  (tab (- depth 1) limit)))))

	


;****************************************************************
; Functions for interfacing to psgraph.

;; print-psgs
;; To be run after (create-the-psgs)
;; creates a post-script file for each achievable literal in the domain
;; the new file will reside in trees/[domain]/literal.  The directories
;; trees and [domain] must already exist.
(defun print-psgs (&optional (lpr nil) (shrink nil) 
			     (insert nil))
  (iter:iterate
   (iter:for tree iter:in *roots*)
   (let ((fname (make-file-name tree *current-domain*)))
     (print-psg tree fname shrink nil insert lpr)
     (gc t)
   )))


;;----------
;; print-psg
;0 is black, 1 is white.
(defun print-psg (root file-nm &optional (shrink nil) (print nil)
		       (insert nil) (lpr nil) (my-fontsize 10)
		       (my-second-fontsize 8))
  (setq fontsize my-fontsize)
  (setq second-fontsize my-second-fontsize)
  (setq boxkind "stroke") ;fill
  (setq edgegray "0")
  (setq boxgray "0")
  (with-open-file
   (*standard-output* file-nm
		      :direction :output
		      :if-exists :supersede
		      :if-does-not-exist :create)
   (psgraph root #'next-nodes #'print-node-plus-label shrink insert)
   )
  (unless (or insert 
	      (not lpr))
	  (shell (concatenate 'string "lpr " file-nm)))
  )



;;---------------
;; make-file-name
;; create a unique file name based upon the domain, and the top-level
;; literal
(defun make-file-name (tree domain)
  (concatenate 'string "trees/" domain "/"
	       (if (literal-p tree)
		   (stringify (Literal-name tree)))
	       ".ps"))


;;----------
;; stringify
;; change l into a string
;; examples: (a b c)  => "a-b-c"
;;           x => "x"
(defun stringify (l)
  (if (list l)
      (apply #'concatenate
	     (cons 'string
		   (butlast
		    (iter:iterate
		     (iter:for x iter:in l)
		     (iter:collect (princ-to-string x))
		     (iter:collect (princ-to-string '-))))))
    (princ-to-string l)))


;;-----------
;; next-nodes
(defun next-nodes (node)
  (cond 
   ((Literal-p node)
    (Literal-operators node))
   ((InternalNode-p node)
    (InternalNode-operands node))
   ((Operator-p node)
    (list (Operator-preconditions node)))))


;;------------------
;; print-node-static
;; This function's name includes the word static to differentiate it
;; from the function loaded by prodigy.
(defun print-node-static (node)
  (cond
   ((Literal-p node)
    (list (stringify (Literal-name node))))
   ((InternalNode-p node)
    (list (princ-to-string (InternalNode-name node))))
   ((Operator-p node)
    (list (princ-to-string (Operator-name node))))))


;;----------------------
;; print-node-plus-label
(defun print-node-plus-label (node)
  (cond
   ((Literal-p node)
    (list (stringify (Literal-name node)) 
	  (princ-to-string (Literal-label node))))
   ((InternalNode-p node)
    (list (princ-to-string (InternalNode-name node))
	  (princ-to-string (InternalNode-label node))))
   ((Operator-p node)
    (list (princ-to-string (Operator-name node))
	  (princ-to-string (Operator-label node))))))


;;-----------------------------
;; show-currently-loaded-rules
(defun show-currently-loaded-rules ()
  (format t "~%The rules currently loaded in prodigy (perhaps by static) are ~
             ~% ~%")

  (format t "~%~%  *SCR-NODE-SELECT-RULES*     : ")
  (print-rules *SCR-NODE-SELECT-RULES*)
  (format t "~%~%  *SCR-GOAL-SELECT-RULES*     : ")
  (print-rules *SCR-GOAL-SELECT-RULES*)
  (format t "~%~%  *SCR-OP-SELECT-RULES*       : ")
  (print-rules *SCR-OP-SELECT-RULES*)
  (format t "~%~%  *SCR-BINDINGS-SELECT-RULES* : ")
  (print-rules *SCR-BINDINGS-SELECT-RULES*)
  (format t "~%~%  *SCR-NODE-REJECT-RULES*     : ")
  (print-rules *SCR-NODE-REJECT-RULES*)
  (format t "~%~%  *SCR-GOAL-REJECT-RULES*     : ")
  (print-rules *SCR-GOAL-REJECT-RULES*)
  (format t "~%~%  *SCR-OP-REJECT-RULES*       : ")
  (print-rules *SCR-OP-REJECT-RULES*)
  (format t "~%~%  *SCR-BINDINGS-REJECT-RULES* : ")
  (print-rules  *SCR-BINDINGS-REJECT-RULES*)
  (format t "~%~%  *SCR-NODE-PREFERENCE-RULES* : ")
  (print-rules  *SCR-NODE-PREFERENCE-RULES*)
  (format t "~%~%  *SCR-GOAL-PREFERENCE-RULES* : ")
  (print-rules *SCR-GOAL-PREFERENCE-RULES*)
  (format t "~%~%  *SCR-OP-PREFERENCE-RULES*   : ")
  (print-rules *SCR-OP-PREFERENCE-RULES*)
  (format t "~%~%  *SCR-BINDINGS-PREFERENCE-RULES* : ")
  (print-rules *SCR-BINDINGS-PREFERENCE-RULES*))


;;----------
(defun print-rules (list-of-rules)
  (cond
   ((null list-of-rules) 
    (format t "nil"))
   (T
    (iter:iterate
     (iter:for rule iter:in list-of-rules)
     (format t "~%     ~s)~%~s~%~s~%" 
	     (first rule) (second rule) (third rule))))))