;;-------------------------------------------------------------------
;;-  Author        :  Rob Spiger 
;;- Sponsoring Prof:  Oren Etzioni
;;-    Date        :  2/9/93
;;- Partial Docs   :  effects/rob-rules.lisp
;;-------------------------------------------------------------------

(defun use-necessary-effects-to-make-sc-rules ()

 (setq *preference-rules* nil)
 (iter:iterate
 (iter:for goal-to-prefer iter:in *roots*)
  (let (
    (number-of-rules (length *preference-rules*)))

   (make-preference-rules-using-any-op goal-to-prefer) ;;use any op to make rules

   (when (AND (eq number-of-rules (length *preference-rules*))
          *make-preference-rules-using-specific-operator*)

    (make-preference-rules-rules-using-specific-ops goal-to-prefer))))

 (setq *preference-rules* (find-and-number-general-preference-rules))) 


(defun make-preference-rules-using-any-op (goal-to-prefer)
 (let* (
   (goal   (make-all-goal-consts-into-vars
             (literal-name goal-to-prefer)))
   (nes   (literal-necessary-effects goal-to-prefer)))

  (make-preference-rules-if-nes-clobber-other-goals goal nes T "ANY")))


(defun make-preference-rules-using-specific-ops (goal-to-prefer)
 (iter:iterate
 (iter:for op iter:in (literal-operators goal-to-prefer))
   (let* (
      (goal       (make-all-goal-consts-into-vars
                   (literal-name goal-to-prefer)))
      (nes        (operator-necessary-effects op))
      (op-name    (string (operator-name op)))
      (extra-cond (this-op-used-exclusively op)))

     (make-preference-rules-if-nes-clobber-other-goals 
                           goal nes extra-cond op-name))))



(defun make-preference-rules-if-nes-clobber-other-goals
                (goal-to-prefer nes-of-goal-to-prefer extra-cond rule-name-of-op-used)
 (iter:iterate
 (iter:for other-root iter:in *roots*)
  (let* (
    (other-goal (make-all-goal-consts-into-vars 
                 (literal-name other-root)))
    (other-goal-vars (find-variables other-goal))
    (other-goal-blist (make-new-same-kind-blist other-goal-vars))
    (other-goal (instantiate other-goal other-goal-blist)))

   (iter:iterate
   (iter:for ne-of-goal-to-prefer iter:in nes-of-goal-to-prefer)
    (let* (
      (pred-of-ne-of-goal-to-prefer (first ne-of-goal-to-prefer))
      (negation-blists (rob-negate-return-blists-w-axioms 
                          pred-of-ne-of-goal-to-prefer other-goal)))
     (iter:iterate
     (iter:for negation-blist iter:in negation-blists)
      (let* (
        (negation-cond (blist-into-conditional negation-blist)))
       (when negation-cond  ;;when there is a way to negate
        (unless (rob-negate-exactly-w-axioms 
                  (instantiate goal-to-prefer negation-blist)
                  other-goal)
         (let* (
           (unique-cond   (unique-guarantee ne-of-goal-to-prefer))
           (ne-cond-w-gen (find-ne-cond-w-gen ne-of-goal-to-prefer))
           (known-cond    `(AND ,ne-cond-w-gen 
                                ,negation-cond
                                ,extra-cond
                                ,unique-cond))
           (new-rule (rob-make-preference-rule other-goal goal-to-prefer
                                           known-cond rule-name-of-op-used)))
          (when new-rule
           (setq *preference-rules* 
               (cons 
                new-rule
                *preference-rules*)))))))))))))




;;Returns a PDL expression under which all the
;; variables bindings iter:in the input binding list are set to 
;; equal.
;; For example, '((<X> a) (<Z> <A>)) becomes
;;        (AND (is-equal <X> a) (is-equal <Z> <A>))
;; Also, No-match becomes nil.


(defun blist-into-conditional (blist)
 (let* (
   (eqlist (blist-to-eqlist blist)))
  (unless (equal blist 'No-match)
   (if (null blist)
    T
    `(AND ,@eqlist)))))

(defun unique-guarantee (ne)
 (let* (
   (existential-vars1 (find-existential-variables (first ne)))
   (blist-var1-new   (mapcar #'(lambda (var) 
                        `(,var ,(make-new-same-kind-variable var)))
								existential-vars1))
   (vars-that-need-gens (instantiate existential-vars1 blist-var1-new))
   (not-eq-list       (blist-to-noteqlist blist-var1-new))
   (not-eq-cond       (rob-simplify `(OR ,@not-eq-list) 'No-sub))
   (generators           (instantiate (third ne) blist-var1-new))
   (unused-gen        nil)
   (used-gen          nil)
   (lists-of-params-lists            nil))

  (if (not existential-vars1) 
  T
 (when (likely-to-be-unique-generators generators)
  (progn
   (iter:iterate
   (iter:for gen iter:in generators)
    (let* (
      (gen-vars (find-existential-variables gen))
      (params-list    (intersection gen-vars vars-that-need-gens :test #'eq)))
     (if params-list
      (progn
       (setq used-gen (cons gen used-gen))
       (setq lists-of-params-lists (cons params-list
                                         lists-of-params-lists))
       (setq vars-that-need-gens (set-subtraction 
                                   vars-that-need-gens params-list)))
      (setq unused-gen (cons gen unused-gen)))))
       
   (unless vars-that-need-gens     ;;return nil if unconstrain variables
    (let* (
      (f-exp (rob-simplify `(AND ,@unused-gen ,not-eq-cond) 'no-sub)))

     `(~ ,(recurse-exists lists-of-params-lists used-gen f-exp)))))))))


(defun recurse-exists (lists-of-params-lists gens f-exp)
 (if (null lists-of-params-lists)
  f-exp
  `(EXISTS ,(first lists-of-params-lists) ,(first gens) 
    ,(recurse-exists (rest lists-of-params-lists) (rest gens) f-exp))))

;;removes the members of set b from set a
(defun set-subtraction (a b)
 (remove-if #'(lambda (x) (member x b)) a))

(defun likely-to-be-unique-generators (generators)
 (remove-if #'not-unique-generator generators))

(defun not-unique-generator (gen)
 (member (predicate-name gen) *rarely-unique-list*))


(defun find-ne-cond-w-gen (ne)
 `(AND ,@(third ne) 
       ,(second ne)))

;;returns a conditional under which this operator
;; is guaranteed to be used to achieve the goal by
;; ANDing together every other operators failure
;; conditions at this level.
;;
(defun this-op-used-exclusively (node)
 (let* (
   (other-ops-fail (other-ops-failure-conditions node))
   (this-ops-fail  (make-all-goal-consts-into-vars
                    (operator-failure-condition node)))
   (same-fail-as-other-ops (member this-ops-fail
                            other-ops-fail :test #'equal))
   (this-ops-not-fail (negate-sentence this-ops-fail))
   (exp              `(AND ,this-ops-not-fail ,@other-ops-fail)))
  (unless same-fail-as-other-ops
   (rob-simplify exp 'no-sub))))
  

;;returns failure conditions of other operators iter:in a list

(defun other-ops-failure-conditions (node)
 (let* (
   (all-ops   (literal-operators (operator-parent node)))
   (other-ops (remove-if #'(lambda (op) (equal op node)) 
                   all-ops)))
  (make-all-goal-consts-into-vars
   (mapcar #'(lambda (op) 
              (operator-failure-condition op)) other-ops))))

 
(defun rob-make-preference-rule (other-goal goal-to-prefer known-cond name-of-op-used)
 (let* (
   (lhs-cond   `(and
                 (current-node <NODE>)
                 (candidate-goal <NODE> ,other-goal)
                 (candidate-goal <NODE> ,goal-to-prefer)
                 (known <NODE> ,known-cond)))
   (true-list  `(,(negate-exp other-goal) ,(negate-exp goal-to-prefer)))
   (simplified (rob-simplify lhs-cond nil true-list t))
   (lhs-cond   (first  simplified))
   (blist      (second simplified))
   (other-goal      (instantiate other-goal blist))
   (goal-to-prefer      (instantiate goal-to-prefer blist))
   (rhs-effect `(prefer goal ,goal-to-prefer ,other-goal)))
   
  (when (AND lhs-cond 
;;             (not (match-var-to-var-only other-goal goal-to-prefer))
             (not (rob-negate-exactly-w-axioms other-goal goal-to-prefer))
             (first (rob-simplify other-goal)) ;;T if goal not in false-list
             (first (rob-simplify goal-to-prefer)));;T if goal not in false-list
   (let* (
     (goal-name (make-preference-rule-name other-goal goal-to-prefer name-of-op-used)))
    `(,goal-name (lhs   ,lhs-cond)
                 (rhs   ,rhs-effect))))))


;;assumes ne of goal-to-prefer negates other-goal
(defun make-preference-rule-name (other-goal goal-to-prefer name-of-op-used)
 (let* (
   (pred1 (get-pred other-goal))
   (pred2 (get-pred goal-to-prefer)))
  (intern (concatenate 'string
             "-PREFER-"
             (string pred2)
             "-OVER-"
             (string pred1)
             "-USING-OP-"
             name-of-op-used))))
             

;;-----------------------------
;;returns T if both rules have the same rhs.
;;This is a little more complicated than just doing equal because
;; one might have two right hand sides which are the same, only they
;; have different variable names.  

;;This algorithm has NOT been completely tested or thought out
;;completely in theory.  It was just a guess.
(defun two-preference-rules-same-rhs (sc-rule1 sc-rule2)
 (let* (
   (rule1-goal1 (third  (second (third sc-rule1))))
   (rule1-goal2 (fourth (second (third sc-rule1))))
   (rule2-goal1 (third  (second (third sc-rule2))))
   (rule2-goal2 (fourth (second (third sc-rule2))))
   (blist1 (match-var-to-var-only-return-blist rule1-goal1
			 								   rule2-goal1)))
  (unless (eq blist1 'no-match)
   (let* (
     (rule1-goal2 (instantiate rule1-goal2 blist1))
     (blist2 (match-var-to-var-only-return-blist rule1-goal2
												 rule2-goal2))
     (blist1-vars (find-variables blist1))
     (blist2-vars (find-variables blist2)))
    (unless (eq blist2 'no-match)
     (unless (has-duplicate (append blist1-vars blist2-vars))
      T))))))


    
(defun find-and-number-general-preference-rules ()
 (let* (
   (rules
    (mapcar #'number-rule 
     (sub-find-general-preference-rules *preference-rules*))))
  (format t "~%~%Formed ~s goal preference rules." (length rules))
  rules))
 

;;Tacks a Rx where x is a number to the leftmost part of the rule's
;;name.  For example, -PREFER-ON-OVER-HOLDING becomes
;;  R22-PREFER-ON-OVER-HOLDING

(defun number-rule (rule)
 (cons (intern (concatenate 'string
                (string (generate-random-rule-name))
                (string (first rule))))
       (rest rule)))

(defun sub-find-general-preference-rules (rules)
 (unless (null rules)
  (let* (
    (rule         (first rules))
    (like-rules   (remove-if-not 
                   #'(lambda (like-rule?) 
                    (two-preference-rules-same-rhs rule like-rule?))
                   rules))
    (unlike-rules (remove-if
                   #'(lambda (like-rule?) 
                    (two-preference-rules-same-rhs rule like-rule?))
                   rules))
    (like-rules-sizes (mapcar #'rule-complexity like-rules))
    (smallest-rule-size (apply #'min like-rules-sizes)))
   (cons (find-if #'(lambda (rule) 
                     (= smallest-rule-size
                        (rule-complexity rule)))
          like-rules)
         (sub-find-general-preference-rules unlike-rules)))))
                     







;;--------------------------------------------------------
;;prerequisite violation rules
;;--------------------------------------------------------

(defun make-rejection-rules-using-ne ()
 (setq *goal-rejection-rules* nil)
 (setq *operator-rejection-rules* nil)
 (iter:iterate
 (iter:for root iter:in *roots*)
  (let* (
    (number-of-rules  (length *goal-rejection-rules*)))
   (make-goal-rejection-rules-for-single-root root)

   (when (eq
          (length *goal-rejection-rules*)
          number-of-rules)
    (make-operator-rejection-rules-for-single-root root)))))

(defun make-goal-rejection-rules-for-single-root (root)
 (let* (
   (goal-to-reject (make-all-goal-consts-into-vars (literal-name root)))
   (nes            (fetch-ne-of-node root))
   (vars           (append (find-variables goal-to-reject)
                           (find-variables nes))) 
   (vars-blist     (make-new-same-kind-blist vars))
   (goal-to-reject (instantiate goal-to-reject vars-blist))
   (nes            (instantiate nes vars-blist)))
  (iter:iterate
   (iter:for prequisites-clobbered-root iter:in *roots*)
    (let* (
     (clobbered-goal (make-all-goal-consts-into-vars (literal-name prequisites-clobbered-root)))
     (failure-cond   (literal-failure-condition prequisites-clobbered-root))
     (failure-cond   (make-all-goal-consts-into-vars failure-cond))
     (lhs-conds       (condition-under-which-failure-cond-eq-t failure-cond nes 
                                  clobbered-goal goal-to-reject)))
    (when lhs-conds
     (make-goal-rejection-rule clobbered-goal lhs-conds goal-to-reject))))))


;;replaces any static predicates in the expression with false.  
;;simplifies the resulting expression.
(defun replace-static-literals-in-exp-with-false (exp)
 (rob-simplify (sub-replace-static-literals-in-exp-with-false exp) 'no-sub))

(defun sub-replace-static-literals-in-exp-with-false (exp)
 (if (atom exp)
  exp
  (unless (is-static-pred exp)
   (mapcar #'sub-replace-static-literals-in-exp-with-false exp))))


;;returns a conditional under which the failure-conditional will evaluate to
;; true for a given set of necessary effects.
(defun condition-under-which-failure-cond-eq-t (failure-cond nes goal1 goal2)
 (let* (

;;   (failure-cond   (replace-static-literals-in-exp-with-false failure-cond))
;;Not sure if this matters or not.  How could one goals necessary effects change a 
;; static predicate anyway to cause goal interaction?  If it did specify a specific
;; static predicate must be true and that static predicate being true made the other
;; goal impossible to achieve, that would be an impossible goal pair to begin with.

   (failure-cond   (simplify-failure-cond-by-setting-to-false-never-matchable-preds
                            failure-cond nes))
   (true-list      (cons (negate-exp goal1) (list (negate-exp goal2))))
   (failure-cond   (rob-simplify failure-cond 'no-sub true-list t t)))
  (when failure-cond
    (subb-condition-under-which-failure-cond-eq-t failure-cond nes true-list nil))))

(defun subb-condition-under-which-failure-cond-eq-t (failure-cond nes true-list bad-bindings)
 (let* (
   (result (sub-condition-under-which-failure-cond-eq-t failure-cond nes true-list bad-bindings))
   (result-bindings (second (rob-simplify result)))
   (result-bindings-bad (find-if #'(lambda (x) (member x bad-bindings :test #'equal))
                                  result-bindings))
   (bad-bindings (append bad-bindings result-bindings)))
  (when (and result
             (not result-bindings-bad))             
   (append (list result) (subb-condition-under-which-failure-cond-eq-t failure-cond nes
                                                                       true-list bad-bindings)))))

  

;;recursion on nes
;;returns a conditional underwhich the nes will make the failure-cond simplify to T
;;Note:  The bindings will be returned in the conditional.
;;       The conditional is not simplified.
(defun sub-condition-under-which-failure-cond-eq-t (failure-cond nes true-list bad-bindings
                                                    &optional (condition 'T))
 (if (or (null nes) (null condition) (eq failure-cond t))
  (when (eq failure-cond t)
   condition)
  (let* (
    (lits    (subgoalable-literals failure-cond)))
   (sub-sub-condition-under-which-failure-cond-eq-t failure-cond nes
              true-list bad-bindings condition lits))))

(defun sub-sub-condition-under-which-failure-cond-eq-t (failure-cond nes
                                   true-list bad-bindings condition lits )
(if (null lits)
 (sub-condition-under-which-failure-cond-eq-t failure-cond (rest nes) 
                   true-list bad-bindings condition)
 (let* (
   (lit     (first lits))
   (ne      (first nes))
   (ne-pred (first ne))
   (ne-cond (rob-simplify (second ne) 'no-sub true-list t t))
   (match-blist   (match-only-with-goal-or-wild-variables lit ne-pred))
   (match-blist   (make-sure-no-bad-bindings-matched match-blist bad-bindings))
   (negate-blists (negate-return-blists-only-with-goal-or-wild-vars-w-axioms
                  (negate-exp lit) ne-pred))
   (negate-blists (mapcar #'(lambda (blist) 
                             (make-sure-no-bad-bindings-matched blist bad-bindings))
                   negate-blists))
   (negate-blists (remove-if #'(lambda (x) (eq x 'no-match)) negate-blists))
   (blist         (if (eq match-blist 'no-match)  ;;picks one of existing 
                   (if negate-blists              ;;blists arbitrarily.
                    (first negate-blists)
                    'No-match)
                   match-blist))
   (eqlist        (blist-to-eqlist blist))
   (blist-cond    (blist-into-conditional blist)))
  (if (eq blist 'no-match)
   (sub-sub-condition-under-which-failure-cond-eq-t failure-cond nes
             true-list bad-bindings condition (rest lits))
   (let* (
     (ne-pred         (instantiate ne-pred blist))
     (true-list       (cons ne-pred true-list))
     (simplify-result (rob-simplify `(AND ,@eqlist ,failure-cond) nil true-list t t))
     (failure-cond    (first simplify-result))
     (true-list       (instantiate true-list (second simplify-result)))
     (ne-pred-vars    (find-variables ne-pred))
     (ne-cond         (first (rob-simplify ne-cond ne-pred-vars nil t t)))
     (condition       `(AND ,condition ,ne-cond ,blist-cond))
     (nes             (instantiate nes (second simplify-result))))
    (sub-condition-under-which-failure-cond-eq-t 
       failure-cond (rest nes) true-list bad-bindings condition))))))

;;returns 'no-match if the blist contains a binding pair which is in the list of bad-bindings
;; otherwise returns blist
(defun make-sure-no-bad-bindings-matched (blist bad-bindings)
 (if (eq blist 'no-match)
  'no-match
  (if (find-if #'(lambda (bpair) (member bpair bad-bindings :test #'equal)) blist)
    'no-match
    blist)))

(defun make-operator-rejection-rules-for-single-root (root)
 (iter:iterate
 (iter:for op iter:in (literal-operators root))
  (make-operator-rejection-rules-for-single-op op 
         (make-all-goal-consts-into-vars (literal-name root)))))

(defun make-operator-rejection-rules-for-single-op (op goal-to-reject)
 (let* (
   (op-to-reject (operator-name op))
   (goal-to-reject (make-all-goal-consts-into-vars goal-to-reject))
   (nes            (fetch-ne-of-node op))
   (vars           (append (find-variables goal-to-reject)
                           (find-variables nes))) 
   (vars-blist     (make-new-same-kind-blist vars))
   (goal-to-reject (instantiate goal-to-reject vars-blist))
   (nes            (instantiate nes vars-blist)))
  (iter:iterate
   (iter:for prequisites-clobbered-root iter:in *roots*)
   (let* (
     (clobbered-goal (make-all-goal-consts-into-vars (literal-name prequisites-clobbered-root)))
     (failure-cond   (literal-failure-condition prequisites-clobbered-root))
     (failure-cond   (make-all-goal-consts-into-vars failure-cond))
     (lhs-conds      (condition-under-which-failure-cond-eq-t failure-cond nes 
                                  clobbered-goal goal-to-reject)))
    (when lhs-conds
     (make-operator-rejection-rule clobbered-goal lhs-conds goal-to-reject op-to-reject))))))


;;helper-functions

;;Returns the necessary effects of any node
;; sorted with the least complex necessary effects
;; first.

(defun fetch-ne-of-node (node)
 (when node
  (cond
   ((literal-p node)
    (sort-by-complexity-ne 
     (literal-necessary-effects node)))
   ((operator-p node)
    (sort-by-complexity-ne 
     (operator-necessary-effects node)))
   ((internalnode-p node)
    (sort-by-complexity-ne 
     (internalnode-necessary-effects node)))
   (T (format t 
        "~%ERROR in function fetch-ne-of-node")))))

;;Works as function less than for complexity of
;; a necessary effect.
;;Inputs = two necessary effects
;; output = t or nil.
(defun complexity-less-than-for-ne (x y)
 (let* (
   (xcond (second x))
   (ycond (second y))
   (xcomplexity (number-of-atomic-formula-in-exp xcond))
   (ycomplexity (number-of-atomic-formula-in-exp ycond)))
  (if (AND 
       (eq 1 xcomplexity)
       (eq 1 ycomplexity))
   (eq 'not-equal (get-pred xcond))
          ;;considers not-equal conditions slightly less complex
          ;;since they are more likely to simplify down.
   (< xcomplexity ycomplexity))))

(defun sort-by-complexity-ne (nes)
 (stable-sort (copy-list nes) #'complexity-less-than-for-ne))

;;*********************


(defun make-goal-rejection-rule (input-clobbered-goal lhs-conds input-goal-to-reject)
 (unless (null lhs-conds)
 (let* (
   (lhs-cond        (first lhs-conds))
   (true-list (cons (negate-exp input-clobbered-goal) (list (negate-exp input-goal-to-reject))))
   (simplify-result (rob-simplify lhs-cond nil true-list t t))
   (lhs-cond        (first simplify-result))
   (blist           (second simplify-result))
   (clobbered-goal  (instantiate input-clobbered-goal blist))
   (goal-to-reject  (instantiate input-goal-to-reject blist)))
  (unless (or (rob-negate-exactly-w-axioms clobbered-goal goal-to-reject)
              (not (eq 'no-match (rob-match clobbered-goal goal-to-reject))))
  (setq *goal-rejection-rules*
    (cons
     `(,(generate-random-rule-name)
       (lhs 
        (AND (current-node  <NODE>) 
             (candidate-goal <NODE> ,goal-to-reject)
             (is-top-level-goal <NODE> ,clobbered-goal)
             ,@(unless (eq lhs-cond t)
              `((known <NODE> ,lhs-cond)))))
       (rhs 
        (reject goal ,goal-to-reject))) *goal-rejection-rules*)))
   (make-goal-rejection-rule input-clobbered-goal (rest lhs-conds) input-goal-to-reject))))

(defun make-operator-rejection-rule (input-clobbered-goal lhs-conds input-goal-to-reject op-to-reject)
 (unless (null lhs-conds)
 (let* (
   (lhs-cond        (first lhs-conds))
   (true-list (cons (negate-exp input-clobbered-goal) (list (negate-exp input-goal-to-reject))))
   (simplify-result (rob-simplify lhs-cond nil true-list t t))
   (lhs-cond        (first simplify-result))
   (blist           (second simplify-result))
   (clobbered-goal  (instantiate input-clobbered-goal blist))
   (goal-to-reject  (instantiate input-goal-to-reject blist)))
  (unless (or (rob-negate-exactly-w-axioms clobbered-goal goal-to-reject)
              (not (eq 'no-match (rob-match clobbered-goal goal-to-reject))))
 (setq *operator-rejection-rules* 
    (cons
     `(,(generate-random-rule-name)
       (lhs 
        (AND (current-node  <NODE>) 
             (current-goal  <NODE> ,goal-to-reject)
             (is-top-level-goal <NODE> ,clobbered-goal)
             ,@(unless (eq lhs-cond t)
              `((known <NODE> ,lhs-cond)))))
       (rhs 
        (reject operator ,op-to-reject))) *operator-rejection-rules*)))
   (make-operator-rejection-rule input-clobbered-goal (rest lhs-conds) input-goal-to-reject op-to-reject))))

;;******************   

;;NEW PLAN:
;;
;; Take the goal to reject
;;  Find it's necessary effects 
;;   Make a list of every one of its necessary effects, call this list p
;;   Take the other goal.
;;    Take the failure conditional of the other goal.
;;    Find the subgoalable-literals in the failure conditional of the other goal
;;    For each element of the subgoalable literals, call this element e
;;      If e can't be matched to anything in p to make e true AND
;;           if ~e can't be negated by anything negation-matching to it from
;;           p then DESPITE WHAT MATCHINGS ARE CHOOSEN, THEY WILL NEVER MAKE
;;           e SIMPLIFY TO TRUE.  (This excludes using information from the
;;           invariant list.  To access this information, a new match function would
;;           have to be implemented which would find a way to match pred1 to pred2 
;;           such that using the invariant list logic, pred1 made pred2 true or something
;;           like that.) SO e CAN BE SIMPLIFIED IN THE FAILURE CONDITIONAL AS FALSE.

(defun simplify-failure-cond-by-setting-to-false-never-matchable-preds 
     (failure-cond nes)

 (let* (
   (possible-true-list (mapcar #'first nes))
   (lits (subgoalable-literals failure-cond))
   (unmatchable-lits (remove-if #'(lambda (x) 
                                   (matchable x possible-true-list))
                      lits))
   (unnegatable-unmatchable-lits
                     (remove-if #'(lambda (x) 
                                   (negatable (negate-exp x) 
                                              possible-true-list))
                      unmatchable-lits))
;;Since it is impossible using the necessary effects in the possible-true-list
;; to ever prove that anything in unnegatable-unmatchable-lits is true, they
;; can be assume to be false for the purpose of proving that the failure
;; conditional will simplify to T.  So, their negation must be true.

   (known-to-be-true (mapcar #'negate-exp unnegatable-unmatchable-lits)))
  (rob-simplify failure-cond 'No-sub known-to-be-true)))
;;Do not want to use negation axioms here because we only want to
;; substitue in those things as false which we know we won't be able
;; to prove as true.  Using the negation axioms for this simplification
;; would result in possibly using the information that what we are substituting is false
;; to prove something is true which isn't known.
;;
;; Said in another way:
;; The things we know we won't prove to be true is a worst case look at things.
;; Just because we can't prove they aren't true doesn't mean their false.
;; So the information that they are false should not be used via the negation
;;  axioms to prove other things are true or false.
;;
;;We lose a little by inserting all these things as false because it is possible
;; that through lucky bindings with the necessary effects, the negation axioms or
;; the invariant list could be used to prove some of the things we 
;; assume are false are actually true.  But inserting them as false greatly simplifies
;; what to look for as far as binding the necessary effect to what is left in the
;; failure conditional.




;;Returns T if it is possible for something in pred-list tonegate x
;; using the negation axioms.
(defun negatable (x pred-list)
 (find-if #'(lambda (y) 
             (not (eq 'nil (negate-return-blists-only-with-goal-or-wild-vars-w-axioms x y))))
  pred-list))

;;Returns T if it is possible to match x to something in the pred-list
;;This matching also has the constraint that only goal constants may be
;; matched.  Because this is the only type of matching which can be
;; inferred will happen by looking at the goals. 

(defun matchable (x pred-list)
 (find-if #'(lambda (y) (not (eq 'no-match (match-only-with-goal-or-wild-variables x y))))
  pred-list))
   


      
      

      
 


;;-----------------------------------------------
;;-generate search control rules for unsolvable
;;-problems.  These are top-level-node rejection rules.
;;-----------------------------------------------
;;-Basically, if the domain has possible unsolvable problems
;;- create top-level-node-rejection rules when all the
;;- operators are known to fail.

(defun create-rules-for-unsolvable-probs ()
 (setq *top-level-node-rejection-rules* nil)
 (let* (
  (failure-cond nil)
  (goal nil)
  (simplify-result nil)
  (true-list nil)
  (blist nil))
  
 (loop for root in *roots*
  do
   (progn
    (setq goal (literal-name root))
    (setq goal (make-all-goal-consts-into-vars goal))
    (setq true-list (list (negate-exp goal)))
    (setq failure-cond (all-ops-failure-cond root))
    (setq failure-cond (make-all-goal-consts-into-vars failure-cond))
    (setq simplify-result (rob-simplify failure-cond nil true-list t))
    (setq failure-cond (first simplify-result))
    (setq blist (second simplify-result))
    (setq goal (instantiate goal blist))
    (when failure-cond
     (setq *top-level-node-rejection-rules* 
           (cons 
            `(,(generate-random-rule-name)
              (lhs 
               (AND
                (CURRENT-NODE <NODE>)
                (IS-TOP-LEVEL-NODE <NODE>)
                (IS-TOP-LEVEL-GOAL <NODE> ,goal)
                (KNOWN <NODE> ,failure-cond)))
              (rhs
               (REJECT NODE <NODE>)))
            *top-level-node-rejection-rules*)))))))
   

(defun all-ops-failure-cond (literal-node)
 `(AND ,@(mapcar #'(lambda (op) (operator-failure-condition op))
                        (literal-operators literal-node))))
