
;;; DOMAIN FOR KITE-BUILDING
;;;
;;; Rob Spiger,  Julie Roomy


;;;NOTES:
;;; The predicates: (is-stick <stick>), 
;;;                 (stick-properties <stick> size number) ,
;;;                 (stick-properties <stick> type <stick-type>),
;;;                 (is-frame <frame>) and
;;;                 (is-fabric <fabric>) are all
;;;  "semi-static" predicates.  For although prodigy allows
;;;  modification it does not allow creation or destruction of objects,
;;;  while maintaining static generators.  For example, in cut, we wish
;;;  to create two objects from one object.
;;;  Apparently, the only way to do this, is to delete the object from 
;;;  the state after the operator has been applied and add the two new
;;;  objects.  The other solution is to throw away the "other" stick.
;;;
;;; All function names begin with "func-" and all inference rule names
;;;  begin with "infer".
;;;
;;;  REFERENCE: Newman, Lee Scott & Newman, Jay Hartley, 
;;;             "Kite Craft." (Crown Publishers Inc.: New York), 1974



;;; PREDICATES:   (due to the complexity of our domain, we listed many of
;;;  the predicates, all semi-static predicates and all static predicates
;;;  for convenience)

;;;  (is-flyable <kite>)
;;;
;;;
;;;  (stick-properties <stick> surface sanded)
;;;  (stick-properites <stick> notched point)
;;;  (stick-properties <stick> size    number)
;;;  (stick-properties <stick> type    <stick-type>)
;;;  (stick-properties <stick> shape   <stick-shape>)
;;;  (stick-properties <stick> lashed  <point>)
;;;  (stick-properties <stick> notched-and-lashed <point>)    ;;open world
;;;
;;;  (inter-stick-properties glued <stick1> at <point1> <stick2> at <point2>)
;;;  (inter-stick-properties tied  <stick1> at <point1> <stick2> at <point2>)
;;;  (inter-stick-properties tubed <stick1> at <point1> <stick2> at <point2>)
;;;
;;;  (fabric-properties <fabric> type    <fabric-type>)
;;;  (fabric-properties <fabric> surface <fabric-surface>)
;;;  (fabric-properties <fabric> shape   <fabric-shape>)
;;;  (fabric-properties <fabric> edge    grommetted)
;;;
;;;  (frame-properties <frame> size     <frame-size>)
;;;  (frame-properties <frame> shape    <frame-shape>)
;;;  (frame-properties <frame> addition <frame-addition>)
;;;  (frame-properties <frame> bound-to <kite>)
;;;
;;;  (kite-type <kite> diamond)               ;;open world
;;;  (kite-type <kite> rectangle)             ;;open world
;;;  (kite-type <kite> bermuda-three-stick)   ;;open world
;;;  (kite-type <kite> indian-fighter)         ;;open world
;;;  (kite-type <kite> four-circle)           ;;open world
;;;
;;;
;;; SEMI-STATIC PREDICATES:
;;;
;;;   (is-stick  <stick>)
;;;   (is-fabric <fabric>)
;;;   (is-frame  <frame>)
;;;  (stick-properties <stick> size    number)
;;;  (stick-properties <stick> type    <stick-type>)
;;;
;;;
;;; STATIC PREDICATES:
;;;
;;;   (have-string)
;;;   (have-glue)
;;;   (have-grommets)
;;;   (have-saw)
;;;   (have-knife)
;;;   (have-sandpaper)
;;;   (have-thread-and-needle)
;;;   (have-candle-flame)
;;;   (have-grommet-puncher)
;;;   (have-tub-of-water)
;;;   (have-lots-of-time)
;;;   (have-heat-sealer)
;;;   (have-dye)
;;;   (have-paint)
;;;   (have-tape)
;;;   (have-fabric-cutter)
;;;
;;;   (is-kite <kite>)
;;;   (is-bulk <fabric>)     ;;bulk fabrics may no have (is-fabric <fabric>)
;;;                          ;;this is so Prodigy will subgoal to cut bulk fabric to produce (is-fabric <fabric>).
;;;   (is-stick-shape circle)
;;;   (is-stick-shape semicircle)
;;;   (is-stick-shape angled)
;;;   
;;;   (is-stick-type plastic)
;;;   (is-stick-type wooden)
;;;   (is-stick-type bamboo)
;;;   (is-stick-type bamboo-strip)
;;;   (is-stick-type tubing)
;;;
;;;   (is-fabric-type paper)
;;;   (is-fabric-type silk)
;;;   (is-fabric-type cotton-cloth)
;;;   (is-fabric-type polyethylene-film)
;;;
;;;   (is-fabric-surface painted)
;;;   (is-fabric-surface dyed)
;;;
;;;   (is-fabric-shape diamond)
;;;   (is-fabric-shape rectangle)
;;;   (is-fabric-shape circle)
;;;   (is-fabric-shape semicircle)
;;;   (is-fabric-shape stretch-hexagon)
;;;   (is-fabric-shape four-circle)
;;;
;;;   (is-frame-shape diamond)
;;;   (is-frame-shape rectangle)
;;;   (is-frame-shape circle)
;;;   (is-frame-shape semicircle)
;;;   (is-frame-shape stretch-hexagon)
;;;   (is-frame-shape four-circle)
;;;
;;;   (is-frame-addition covering)
;;;   (is-frame-addition tail)
;;;   (is-frame-addition bridle)



(setq *OPERATORS* 
      '(
;;; top level operators

   (bind-kite-to-frame
    (params (<kite> <frame>))
     (preconds 
       (and
         (is-kite <kite>)
         (is-frame <frame>)
         (~ (exists (<other-kite>) (is-kite <other-kite>)
                (frame-properties <frame> bound-to <other-kite>)))))
     (effects (
       (add (frame-properties <frame> bound-to <kite>)))))


   ;in the operators which build frames.  The size of the stick must
   ; generate the stick in order for prodigy to be complete, since
   ; is-stick is only semi-static.  Once a frame is built, all info.
   ; about materials used to build the frame is removed, so that it
   ; can't be used to build a second frame.

   ; Note:  if <stickx> is a variable prodigy perceives goals
   ;  (stick-properties <stick1> size 25)
   ;  (stick-properties <stick2> size 25)
   ;  (stick-properties <stick3> size 25) as being similar and will fail
   ; on the last two.  Therefore these <stickx> must be bound previous
   ; to their call.

   (build-stretch-hexagon-frame                 ; bermuda three-stick kite
    (params (<frame> <stick1> <stick2> <stick3>))
     (preconds 
       (and
	 (func-get-new-stick-name-or-return-true <stick1>)   ;see above note
	 (func-get-new-stick-name-or-return-true <stick2>) 
	 (func-get-new-stick-name-or-return-true <stick3>)

         (stick-properties <stick1> size 25)    ;values recommended by (Newman, 1974)
         (stick-properties <stick2> size 25)    
         (stick-properties <stick3> size 25)
         (is-stick <stick1>)
         (is-straight <stick1>)
         (is-strong-stick <stick1>)
         (is-stick <stick2>)
         (is-straight <stick2>)
         (is-strong-stick <stick2>)
         (is-stick <stick3>)
         (is-straight <stick3>)
         (is-strong-stick <stick3>)
         (stick-properties <stick1> notched-and-lashed 13)      ;;for bridle
         (inter-stick-properties tied-and-glued <stick1> at 13 <stick2> at 13)
         (inter-stick-properties tied-and-glued <stick1> at 13 <stick3> at 13)
         (inter-stick-properties tied-and-glued <stick2> at 13 <stick3> at 13)
         (func-get-new-frame-name <frame>)))
     (effects (
       (del (is-stick <stick1>))
       (del (stick-properties <stick1> <*property1a> <*property-attribute1a>))
       (del (inter-stick-properties <*property1b> <stick1> at <*point1b> <*other-stick1b> at <*other-point1b>))
       (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <stick1> at <*point1c>))
       (del (is-stick <stick2>))
       (del (stick-properties <stick2> <*property2a> <*property-attribute2a>))
       (del (inter-stick-properties <*property2b> <stick2> at <*point2b> <*other-stick2b> at <*other-point2b>))
       (del (inter-stick-properties <*property2c> <*other-stick2c> at <*other-point2c> <stick2> at <*point2c>))
       (del (is-stick <stick3>))
       (del (stick-properties <stick3> <*property3a> <*property-attribute3a>))
       (del (inter-stick-properties <*property3b> <stick3> at <*point3b> <*other-stick3b> at <*other-point3b>))
       (del (inter-stick-properties <*property3c> <*other-stick3c> at <*other-point3c> <stick3> at <*point3c>))
       (add (is-frame <frame>))
       (add (frame-properties <frame> shape stretch-hexagon)))))


   (build-diamond-frame
    (params (<frame> <stick1> <stick2>))
     (preconds
       (and
	 (func-get-new-stick-name-or-return-true <stick1>) 
	 (func-get-new-stick-name-or-return-true <stick2>) 
         (stick-properties <stick1> size 29)    ;values recommended by (Newman, 1974)
         (stick-properties <stick2> size 34)
         (is-stick <stick1>)
         (is-straight <stick1>)
         (is-strong-stick <stick1>)
         (is-stick <stick2>)
         (is-straight <stick2>)
         (is-strong-stick <stick2>)
         (stick-properties <stick1> notched-and-lashed 29)        ;;for guideline
         (stick-properties <stick1> notched-and-lashed 1)
         (stick-properties <stick2> notched-and-lashed 34)
         (stick-properties <stick2> notched-and-lashed 1)
         (have-string)                                            ;;for guideline
         (func-get-new-frame-name <frame>)
         (stick-properties <stick2> notched-and-lashed 26)        ;;for bridle
	 (inter-stick-properties tied-and-glued <stick1> at 15 <stick2> at 8)
))
     (effects (
       (del (is-stick <stick1>))
       (del (stick-properties <stick1> <*property1a> <*property-attribute1a>))
       (del (inter-stick-properties <*property1b> <stick1> at <*point1b> <*other-stick1b> at <*other-point1b>))
       (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <stick1> at <*point1c>))
       (del (is-stick <stick2>))
       (del (stick-properties <stick2> <*property2a> <*property-attribute2a>))
       (del (inter-stick-properties <*property2b> <stick2> at <*point2b> <*other-stick2b> at <*other-point2b>))
       (del (inter-stick-properties <*property2c> <*other-stick2c> at <*other-point2c> <stick2> at <*point2c>))
       (add (is-frame <frame>))
       (add (frame-properties <frame> shape diamond)))))


   (build-rectangle-frame
     (params (<frame> <stick1> <stick2> <stick3> <stick4>))
      (preconds
        (and 
 	  (func-get-new-stick-name-or-return-true <stick1>) 
	  (func-get-new-stick-name-or-return-true <stick2>) 
	  (func-get-new-stick-name-or-return-true <stick3>)
	  (func-get-new-stick-name-or-return-true <stick4>) 
          (stick-properties <stick1> size 25)  ;;values recommmended by (Newman, 1974)
          (stick-properties <stick2> size 25)
          (stick-properties <stick3> size 29)
          (stick-properties <stick4> size 29)
          (is-stick <stick1>)
          (is-straight <stick1>)
	  (is-strong-stick <stick1>)
          (is-stick <stick2>)
          (is-straight <stick2>)
          (is-strong-stick <stick2>)
          (is-stick <stick3>)
          (is-straight <stick3>)
          (is-strong-stick <stick3>)
          (is-stick <stick4>)
          (is-straight <stick4>)
          (is-strong-stick <stick4>)
          (stick-properties <stick1> notched-and-lashed 13)  ;;for guideline
          (stick-properties <stick3> notched-and-lashed 15)  ;;for guideline
          (stick-properties <stick4> notched-and-lashed 15)  ;;for guideline
          (inter-stick-properties tubed <stick1> at  1 <stick3> at  1)
          (inter-stick-properties tubed <stick1> at 25 <stick4> at  1)
          (inter-stick-properties tubed <stick2> at  1 <stick3> at 29)
          (inter-stick-properties tubed <stick2> at 25 <stick4> at 29)
          (func-get-new-frame-name <frame>)))
      (effects (
        (del (is-stick <stick1>))
        (del (stick-properties <stick1> <*property1a> <*property-attribute1a>))
        (del (inter-stick-properties <*property1b> <stick1> at <*point1b> <*other-stick1b> at <*other-point1b>))
        (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <stick1> at <*point1c>))
        (del (is-stick <stick2>))
        (del (stick-properties <stick2> <*property2a> <*property-attribute2a>))
        (del (inter-stick-properties <*property2b> <stick2> at <*point2b> <*other-stick2b> at <*other-point2b>))
        (del (inter-stick-properties <*property2c> <*other-stick2c> at <*other-point2c> <stick2> at <*point2c>))
        (del (is-stick <stick3>))
        (del (stick-properties <stick3> <*property3a> <*property-attribute3a>))
        (del (inter-stick-properties <*property3b> <stick3> at <*point3b> <*other-stick3b> at <*other-point3b>))
        (del (inter-stick-properties <*property3c> <*other-stick3c> at <*other-point3c> <stick3> at <*point3c>))
        (del (is-stick <stick4>))
        (del (stick-properties <stick4> <*property4a> <*property-attribute4a>))
        (del (inter-stick-properties <*property4b> <stick4> at <*point4b> <*other-stick4b> at <*other-point4b>))
        (del (inter-stick-properties <*property4c> <*other-stick4c> at <*other-point4c> <stick4> at <*point4c>))
        (add (is-frame <frame>))
        (add (frame-properties <frame> shape rectangle)))))


   (build-semicircle-frame                       ; indian fighter kite
     (params (<frame> <stick1> <stick2>))
      (preconds
        (and
 	  (func-get-new-stick-name-or-return-true <stick1>) 
	  (func-get-new-stick-name-or-return-true <stick2>) 
          (stick-properties <stick1> size 31)    ;;sizes recommended by (Newman, 1974)
          (stick-properties <stick2> size 26)
          (is-stick <stick1>)
          (is-stick <stick2>)
          (is-strong-stick <stick2>)
          (stick-properties <stick1> shape semicircle)
          (stick-properties <stick1> surface sanded)
	  (is-straight <stick2>)
          (have-string)                                      ;;for guideline
          (stick-properties <stick1> notched-and-lashed 31)  ;;for guideline
          (stick-properties <stick1> notched-and-lashed 1)   ;;for guideline
          (stick-properties <stick2> notched-and-lashed 26)  ;;for guideline
          (stick-properties <stick2> notched-and-lashed 1)   ;;for guideline
          (inter-stick-properties tied-and-glued <stick1> at 16 <stick2> at 21)
          (func-get-new-frame-name <frame>)))
      (effects (
        (del (is-stick <stick1>))
        (del (stick-properties <stick1> <*property1a> <*property-attribute1a>))
        (del (inter-stick-properties <*property1b> <stick1> at <*point1b> <*other-stick1b> at <*other-point1b>))
        (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <stick1> at <*point1c>))
        (del (is-stick <stick2>))
        (del (stick-properties <stick2> <*property2a> <*property-attribute2a>))
        (del (inter-stick-properties <*property2b> <stick2> at <*point2b> <*other-stick2b> at <*other-point2b>))
        (del (inter-stick-properties <*property2c> <*other-stick2c> at <*other-point2c> <stick2> at <*point2c>))
        (add (is-frame <frame>))
        (add (frame-properties <frame> shape semicircle)))))


   (build-four-circle-frame
     (params (<frame> <circle1> <circle2> <circle3> <circle4> <spine> <crosspiece>))
      (preconds
        (and 
 	  (func-get-new-stick-name-or-return-true <circle1>)
 	  (func-get-new-stick-name-or-return-true <circle2>)
 	  (func-get-new-stick-name-or-return-true <circle3>)
 	  (func-get-new-stick-name-or-return-true <circle4>)
	  (func-get-new-stick-name-or-return-true <spine>) 
	  (func-get-new-stick-name-or-return-true <crosspiece>)
          (stick-properties <circle1> size 60)  ;; sizes recommended by (Newman, 1974)
          (stick-properties <circle2> size 60)
          (stick-properties <circle3> size 60)
          (stick-properties <circle4> size 60)
          (stick-properties <spine> size 84)
          (stick-properties <crosspiece> size 43)

          (is-stick <circle1>)
          (is-stick <circle2>)
          (is-stick <circle3>)
          (is-stick <circle4>)
	  (is-stick <spine>)
          (is-stick <crosspiece>)

          (stick-properties <circle1> shape circle)        
          (stick-properties <circle2> shape circle)        
          (stick-properties <circle3> shape circle)        
          (stick-properties <circle4> shape circle)
          (is-straight <spine>)
          (is-straight <crosspiece>)

          (is-strong-stick <spine>)
          (is-strong-stick <crosspiece>)

          (stick-properties      <spine> notched-and-lashed  9)   
          (stick-properties      <spine> notched-and-lashed 43)         ;;for bridle
          (stick-properties <crosspiece> notched-and-lashed  1)
          (stick-properties <crosspiece> notched-and-lashed 43)

	  ; tie and glue at every overlap of 6 sticks
          (inter-stick-properties tied-and-glued  <circle1> at 30      <spine> at 10)
          (inter-stick-properties tied-and-glued  <circle1> at 60      <spine> at 30)
          (inter-stick-properties tied-and-glued  <circle4> at 60      <spine> at 44)
          (inter-stick-properties tied-and-glued  <circle4> at 30      <spine> at 64)
          (inter-stick-properties tied-and-glued  <circle1> at  7    <circle2> at  7)
          (inter-stick-properties tied-and-glued  <circle1> at 54    <circle3> at  7)
          (inter-stick-properties tied-and-glued  <circle2> at 54    <circle4> at 54)
          (inter-stick-properties tied-and-glued  <circle3> at 54    <circle4> at  7)
          (inter-stick-properties tied-and-glued  <circle2> at 30 <crosspiece> at  2)
          (inter-stick-properties tied-and-glued  <circle2> at 60 <crosspiece> at 22)
          (inter-stick-properties tied-and-glued  <circle3> at 60 <crosspiece> at 22)
          (inter-stick-properties tied-and-glued  <circle3> at 30 <crosspiece> at 42)
          (inter-stick-properties tied-and-glued    <spine> at 37 <crosspiece> at 22)

          (func-get-new-frame-name <frame>)))

    (effects (
        (del (is-stick <circle1>))
        (del (stick-properties <circle1> <*property1a> <*property-attribute1a>))
        (del (inter-stick-properties <*property1b> <circle1> at <*point1b> <*other-stick1b> at <*other-point1b>))
        (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <circle1> at <*point1c>))
        (del (is-stick <circle2>))
        (del (stick-properties <circle2> <*property2a> <*property-attribute2a>))
        (del (inter-stick-properties <*property2b> <circle2> at <*point2b> <*other-stick2b> at <*other-point2b>))
        (del (inter-stick-properties <*property2c> <*other-stick2c> at <*other-point2c> <circle2> at <*point2c>))
        (del (is-stick <circle3>))
        (del (stick-properties <circle3> <*property3a> <*property-attribute3a>))
        (del (inter-stick-properties <*property3b> <circle3> at <*point3b> <*other-stick3b> at <*other-point3b>))
        (del (inter-stick-properties <*property3c> <*other-stick3c> at <*other-point3c> <circle3> at <*point3c>))
        (del (is-stick <circle4>))
        (del (stick-properties <circle4> <*property4a> <*property-attribute4a>))
        (del (inter-stick-properties <*property4b> <circle4> at <*point4b> <*other-stick4b> at <*other-point4b>))
        (del (inter-stick-properties <*property4c> <*other-stick4c> at <*other-point4c> <circle4> at <*point4c>))
        (del (is-stick <spine>))
        (del (stick-properties <spine> <*property5a> <*property-attribute5a>))
        (del (inter-stick-properties <*property5b> <spine> at <*point5b> <*other-stick5b> at <*other-point5b>))
        (del (inter-stick-properties <*property5c> <*other-spine5c> at <*other-point5c> <spine> at <*point5c>))
        (del (is-stick <crosspiece>))
        (del (stick-properties <crosspiece> <*property6a> <*property-attribute6a>))
        (del (inter-stick-properties <*property6b> <crosspiece> at <*point6b> <*other-stick6b> at <*other-point6b>))
        (del (inter-stick-properties <*property6c> <*other-stick6c> at <*other-point6c> <crosspiece> at <*point6c>))
        (add (is-frame <frame>))
        (add (frame-properties <frame> shape four-circle)))))


;;; low level operators
   (sand-with-paper
    (params (<stick>))
    (preconds
     (and
      (is-stick <stick>)
      (is-sandable <stick>)
      (have-sandpaper)))
    (effects (
     (add (stick-properties <stick> surface sanded)))))
 
   (bend-with-soaking-in-water
    (params (<stick> <stick-shape>))
    (preconds
     (and
      (is-stick <stick>)
      (is-stick-shape <stick-shape>)
      (have-tub-of-water)
      (have-lots-of-time)
      (is-water-bendable <stick>)))
    (effects (
     (add (stick-properties <stick> shape <stick-shape>)))))

    (bend-with-candle-flame
      (params (<stick> <stick-shape>))
       (preconds
         (and 
           (is-stick <stick>)
           (is-stick-shape <stick-shape>)
           (have-candle-flame)
           (is-candle-flame-bendable <stick>)))
      (effects (
        (add (stick-properties <stick> shape <stick-shape>)))))
     
    (notch-with-saw
      (params (<stick> <point>))
       (preconds
         (and
           (is-stick <stick>)
           (stick-properties <stick> size <stick-size>)
           (func-point-on-stick <stick-size> <point>)
           (is-notchable-at-point <stick> <point>)))
       (effects (
         (add (stick-properties <stick> notched <point>)))))

    (lash-with-string
      (params (<stick> <point>))
       (preconds
         (and
           (have-string)
           (have-glue)
           (is-stick <stick>)
           (stick-properties <stick> size <stick-size>)
           (func-point-on-stick <stick-size> <point>)
	   (is-unattached-stick <stick>)))
       (effects (
         (add (stick-properties <stick> lashed <point>)))))


(strip-bamboo-with-knife
 (params (<stick> <stick-size> <new-strip2> <new-strip3>
				  <new-strip4>))
 ;;one of the new strips ends up with name of <stick>
 (preconds
  (and
   (is-stick <stick>)
   (stick-properties <stick> size <stick-size>)
   (stick-properties <stick> type bamboo)
   (have-knife)
   (is-strippable <stick>)				; checks for inter-stick props.
   (func-get-new-stick-name <new-strip2>)
   (func-get-new-stick-name <new-strip3>)
   (func-get-new-stick-name <new-strip4>)))
 (effects (
		   (del (stick-properties <stick> <*property> <*property-attribute>)) 
		   (add (stick-properties <stick> size <stick-size>))
		   (add (stick-properties <stick> type bamboo-strip))
		   (add (is-stick <new-strip2>))
		   (add (stick-properties <new-strip2> size <stick-size>))
		   (add (stick-properties <new-strip2> type bamboo-strip))
		   (add (is-stick <new-strip3>))
		   (add (stick-properties <new-strip3> size <stick-size>))
		   (add (stick-properties <new-strip3> type bamboo-strip))
		   (add (is-stick <new-strip4>))
		   (add (stick-properties <new-strip4> size <stick-size>))
		   (add (stick-properties <new-strip4> type bamboo-strip)))))


;; Note, the search control rule
;;      Select-best-bindings-for-cut-stick-with-saw will ensure that
;;      the "leftover" sticks will always have the orginal stick name.
;;      And this is utilized by another search control rule, Prefer-
;;      to-cut-original-sticks, so a size has been achieved, the
;;      achieved size is preferably not cut to achieve another stick.
;;      In certain cases (i.e.  stipping bamboo this may be required,
;;      so it is only a preferance rule.  

;;first new stick has same name as <stick>
(cut-stick-with-saw
 (params (<stick> <stick-type> <stick-size> <new-size1> <new-stick2>
				  <new-size2>))  
 (preconds
  (and
   (is-stick <stick>)
   (stick-properties <stick> size <stick-size>)
   (is-stick-type <stick-type>)
   (stick-properties <stick> type <stick-type>)
   (have-saw)
   (is-cuttable <stick>)                ; checks for inter-stick props.
   (func-subtract <stick-size> <new-size1> <new-size2>)
   (func-greater-than <new-size1> 0)
   (func-greater-than <new-size2> 0)
   (func-get-new-stick-name-or-return-true <new-stick2>)))
 (effects (
		   (del (stick-properties <stick> <*property> <*property-attribute>))
		   (add (stick-properties <stick> size <new-size1>))
		   (add (stick-properties <stick> type <stick-type>))
		   (add (is-stick <new-stick2>))
		   (add (stick-properties <new-stick2> size <new-size2>))
		   (add (stick-properties <new-stick2> type <stick-type>)))))


     (glue-two-sticks-with-glue
       (params (<stick1> <point1> <stick2> <point2>))
        (preconds
          (and
            (have-glue)
            (is-stick <stick1>)
            (stick-properties <stick1> size <stick-size1>)
            (func-point-on-stick <stick-size1> <point1>)
            (is-stick <stick2>)
            (not-equal <stick1> <stick2>)
            (stick-properties <stick2> size <stick-size2>)
            (func-point-on-stick <stick-size2> <point2>)))
        (effects (
          (add (inter-stick-properties glued <stick1> at <point1> <stick2> at <point2>)))))

     (tie-two-sticks-with-string
       (params (<stick1> <point1> <stick2> <point2>))
        (preconds
          (and
            (have-string)
            (is-stick <stick1>)
            (stick-properties <stick1> size <stick-size1>)
            (func-point-on-stick <stick-size1> <point1>)
            (is-stick <stick2>)
            (not-equal <stick1> <stick2>)
            (stick-properties <stick2> size <stick-size2>)
            (func-point-on-stick <stick-size2> <point2>)))
        (effects (
          (add (inter-stick-properties tied <stick1> at <point1> <stick2> at <point2>)))))

     (tube-two-sticks-with-tubing
       (params (<tubing-joint> <stick1> <point1> <stick2> <point2>))
        (preconds
          (and
            (have-glue)
            (is-stick <stick1>)
            (is-stick <stick2>)
            (is-tubing-joint <tubing-joint>)
            (stick-properties <stick1> size <stick-size1>)
            (func-point-on-stick <stick-size1> <point1>)
            (is-stick <stick2>)
            (not-equal <stick1> <stick2>)
            (stick-properties <stick2> size <stick-size2>)
            (func-point-on-stick <stick-size2> <point2>)
            (is-tubable-at-point <stick1> <point1>)
            (is-tubable-at-point <stick2> <point2>)))
        (effects (
          (add (inter-stick-properties tubed <stick1> at <point1> <stick2> at <point2>))
          (del (is-stick <tubing-joint>))
          (del (stick-properties <tubing-joint> <*property> <*property-attributes>))
          (del (inter-stick-properties <*property1b> <tubing-joint> at <*point1b> <*other-stick1b> at <*other-point1b>))
          (del (inter-stick-properties <*property1c> <*other-stick1c> at <*other-point1c> <tubing-joint> at <*point1c>)))))

     (cut-fabric-with-fabric-cutter
       (params (<bulk-fabric> <shaped-fabric> <fabric-shape>))
        (preconds
          (and
            (have-fabric-cutter)
	    (is-bulk <bulk-fabric>)
	    (is-fabric-type <fabric-type>)
	    (fabric-properties <bulk-fabric> type <fabric-type>)
	    (func-get-new-fabric-name <shaped-fabric>)
            (is-fabric-shape <fabric-shape>)))
	(effects (
          (add (is-fabric <shaped-fabric>))
	  (add (fabric-properties <shaped-fabric> type <fabric-type>))
	  (add (fabric-properties <shaped-fabric> shape <fabric-shape>)))))

     (dye-fabric-with-dye
       (params (<fabric>))
        (preconds
          (and
	    (have-dye)
            (is-fabric <fabric>)
            (is-dyeable <fabric>)))
        (effects (
          (add (fabric-properties <fabric> surface dyed)))))

      (paint-fabric-with-paint
        (params (<fabric>))
	 (preconds
	   (and
	     (have-paint)
	     (is-fabric <fabric>)
	     (is-paintable <fabric>)))
	 (effects (
	     (add (fabric-properties <fabric> surface painted)))))

      (grommet-fabric-edge-with-grommets
        (params (<fabric>))
         (preconds
           (and
	     (have-grommets)
	     (have-grommet-puncher)
             (is-fabric <fabric>)
             (is-grommettable <fabric>)))
         (effects (
           (add (fabric-properties <fabric> edge grommetted)))))

      (tie-fabric-edge-with-string-to-frame
        (params (<fabric> <frame>))
         (preconds
           (and
             (have-string)
             (is-fabric <fabric>)
             (is-frame <frame>)
             (is-decorated <fabric>)
             (are-tieable <fabric> <frame>)))
	 (effects (
           (add (frame-properties <frame> addition covering))
           (del (is-fabric <fabric>))
           (del (fabric-properties <fabric> <*property> <*property-attribute>)))))

       (sew-fabric-edge-with-thread-and-needle-to-frame
        (params (<fabric> <frame>))
         (preconds
           (and
             (have-thread-and-needle)
             (is-fabric <fabric>)
             (is-frame <frame>)
             (is-decorated <fabric>)
             (are-sewable <fabric> <frame>)))
          (effects (
            (add (frame-properties <frame> addition covering))
            (del (is-fabric <fabric>))
            (del (fabric-properties <fabric> <*property> <*property-attribute>)))))

       (heat-seal-fabric-edge-with-heat-sealer-to-frame
        (params (<fabric> <frame>))
         (preconds
           (and
             (have-heat-sealer)
             (is-fabric <fabric>)
	     (is-frame <frame>)
             (is-decorated <fabric>)
             (are-heat-sealable <fabric> <frame>)))
         (effects (
           (add (frame-properties <frame> addition covering))
           (del (is-fabric <fabric>))
           (del (fabric-properties <fabric> <*property> <*property-attribute>)))))
             
       (tape-fabric-edge-with-tape-to-frame
        (params (<fabric> <frame>))
         (preconds
           (and
             (have-tape)
             (is-fabric <fabric>)
             (is-frame <frame>)
             (is-decorated <fabric>)
             (are-tapeable <fabric> <frame>)))
         (effects (
           (add (frame-properties <frame> addition covering))
           (del (is-fabric <fabric>))
           (del (fabric-properties <fabric> <*property> <*property-attribute>)))))

       (add-tail-to-frame
         (params (<frame>))
          (preconds
            (and
              (have-fabric-cutter)
              (have-string)
              (is-bulk <fabric>)
              (is-frame <frame>)))
          (effects (
            (add (frame-properties <frame> addition tail)))))

       (add-bridle-to-frame
         (params (<frame>))
          (preconds
            (and
              (is-frame <frame>)
              (have-string)))
          (effects (
            (add (frame-properties <frame> addition bridle)))))


))

;;; NOTE: Inference rules have been choosen to facilitate domain expansion.


(setq *INFERENCE-RULES*  '(


;;; general inference rules for sticks:


  ;;; a stick is inferred to be straight if it doesn't have any other shape
  (infer-stick-straight
    (params (<stick> <stick-shape>))
    (preconds 
       (and
         (is-stick <stick>)
         (~(EXISTS (<stick-shape>) (is-stick-shape <stick-shape>)
               (stick-properties <stick> shape <stick-shape>)))))
     (effects (
       (add (is-straight <stick>)))))


  (infer-stick-unnotched
   (params (<stick>))
   (preconds
    (and
     (is-stick <stick>)
     (stick-properties <stick> size <stick-size>)
     (~ (exists (<point>) (func-point-on-stick <stick-size> <point>)
		(stick-properties <stick> notched <point>)))))
   (effects (
     (add (is-unnotched <stick>)))))
    

  (infer-stick-unlashed
    (params (<stick>))
     (preconds
       (and
         (is-stick <stick>)
	 (stick-properties <stick> size <stick-size>)
	 (~ (exists (<point>) (func-point-on-stick <stick-size> <point>)
		    (stick-properties <stick> lashed <point>)))))
     (effects (
       (add (is-unlashed <stick>)))))


(infer-stick-unattached
 (params (<stick>))
 (preconds
  (and
   (is-stick <stick>)
   (~ (EXISTS (<propertya> <point1a> <point2a> <other-sticka>)
              (is-stick <other-sticka>)
			  (inter-stick-properties <propertya> <stick> at <point1a>
									  <other-sticka> at <point2a>)))
   (~ (EXISTS (<propertyb> <point1b> <point2b> <other-stickb>)
              (is-stick <other-stickb>)
			  (inter-stick-properties <propertyb> <other-stick> at <point1b>
									  <stick> at <point2b>)))))
 (effects (
		   (add (is-unattached-stick <stick>)))))


(infer-stick-unattached-at-point
 (params (<stick> <point>))
 (preconds
  (and
   (is-stick <stick>)
   (stick-properties <stick> size <stick-size>)
   (func-point-on-stick <stick-size> <point>)
   (~ (exists (<propertya> <other-sticka> <point2a>)
              (is-stick <other-sticka>)
			  (inter-stick-properties <propertya> <stick> at <point>
									  <other-sticka> at <point2a>)))
   (~ (exists (<propertyb> <other-stickb> <point2b>)
              (is-stick <other-sticka>)
			  (inter-stick-properties <propertyb> <other-stickb> at <point2b>
									  <stick> at <point>)))))
 (effects (
		   (add (is-unattached-stick-at-point <stick> <point>)))))


   (infer-stick-unmodified-at-point
     (params (<stick> <point>))
      (preconds
        (and
          (is-stick <stick>)
          (stick-properties <stick> size <stick-size>)
          (func-point-on-stick <stick-size> <point>)
          (is-unattached-stick-at-point <stick> <point>)
          (~ (stick-properties <stick> notched <point>))
          (~ (stick-properties <stick> lashed <point>))))
      (effects (
        (add (is-unmodified-at-point <stick> <point>)))))


   (infer-two-sticks-tied-and-glued
     (params (<stick1> <point1> <stick2> <point2>))
      (preconds
        (and
          (is-stick <stick1>)
          (is-stick <stick2>)
          (stick-properties <stick1> size <stick-size1>)
          (func-point-on-stick <stick-size1> <point1>)
          (stick-properties <stick2> size <stick-size2>)
          (func-point-on-stick <stick-size2> <point2>)
          (or 
            (inter-stick-properties tied <stick1> at <point1> <stick2> at <point2>)
            (inter-stick-properties tied <stick2> at <point2> <stick1> at <point1>))
          (or
            (inter-stick-properties glued <stick1> at <point1> <stick2> at <point2>)
            (inter-stick-properties glued <stick2> at <point2> <stick1> at <point1>))))
      (effects (
        (add (inter-stick-properties tied-and-glued <stick1> at <point1> <stick2> at <point2>)))))


   (infer-strong-stick
     (params (<stick>))
      (preconds
        (and 
          (is-stick <stick>)
          (or
            (stick-properties <stick> type bamboo)
            (stick-properties <stick> type plastic)
            (stick-properties <stick> type wooden))))
      (effects (
        (add (is-strong-stick <stick>)))))

           
   (infer-stick-notched-and-lashed-at-point
     (params (<stick> <point>))
      (preconds
        (and
          (is-stick <stick>)
          (stick-properties <stick> size <stick-size>)
          (func-point-on-stick <stick-size> <point>)
          (stick-properties <stick> notched <point>)
          (stick-properties <stick> lashed <point>)))
      (effects (
        (add (stick-properties <stick> notched-and-lashed <point>)))))


 
;;; specific inference rules for sticks:


  (infer-stick-sandable
    (params (<stick>))
     (preconds
       (and
         (is-stick <stick>)
	 (is-unattached-stick <stick>)
         (is-unlashed <stick>)))
     (effects (
       (add (is-sandable <stick>)))))


(infer-water-bendable
 (params (<stick>))
 (preconds
  (and
   (is-stick <stick>)
   (is-straight <stick>)
   (or
	(stick-properties <stick> type bamboo-strip)
	(stick-properties <stick> type wooden))
   (is-unlashed <stick>)))				;(glue comes off in water)
 (effects (
		   (add (is-water-bendable <stick>)))))
     
  (infer-candle-flame-bendable
    (params (<stick>))
     (preconds
       (and
         (is-stick <stick>)
         (is-straight <stick>)
         (is-unlashed <stick>)
         (or
            (stick-properties <stick> type plastic)
            (stick-properties <stick> type bamboo-strip)
            (stick-properties <stick> type tubing))))
     (effects (
       (add (is-candle-flame-bendable <stick>)))))


  (infer-stick-notchable-at-point
    (params (<stick> <point>))
     (preconds
       (and
         (is-stick <stick>)
         (is-unattached-stick <stick>)
	 (stick-properties <stick> size <stick-size>)
	 (func-point-on-stick <stick-size> <point>)
	 (~ (stick-properties <stick> lashed <point>))))
     (effects (
       (add (is-notchable-at-point <stick> <point>)))))


  (infer-stick-strippable
   (params (<stick>))
   (preconds
    (and
     (is-stick <stick>)
     (is-unattached-stick <stick>)
     (is-unlashed <stick>)
     (is-unnotched <stick>)))
   (effects (
     (add (is-strippable <stick>)))))


  (infer-stick-cuttable
   (params (<stick>))
   (preconds
    (and
     (is-stick <stick>)
     (is-unattached-stick <stick>)
     (is-unlashed <stick>)
     (is-unnotched <stick>)))        ;All types are cuttable.
   (effects (
      (add (is-cuttable <stick>)))))


(infer-stick-tubable-at-begin-point
 (params (<stick>))
 (preconds
  (and
   (is-stick <stick>)
   (is-unmodified-at-point <stick> 1)));;may tube stick on end
 (effects (
		   (add (is-tubable-at-point <stick> 1)))))


(infer-stick-tubable-at-end-point
 (params (<stick> <point>))
 (preconds
  (and
   (is-stick <stick>)
   (stick-properties <stick> size <point>);;can only tube stick on ends
   (is-unmodified-at-point <stick> <point>)))
 (effects (
		   (add (is-tubable-at-point <stick> <point>)))))


   (infer-stick-is-tubing-joint
     (params (<stick>))
      (preconds
        (and
          (stick-properties <stick> size 1)
          (stick-properties <stick> type tubing)
          (stick-properties <stick> shape angled)))
      (effects (
        (add (is-tubing-joint <stick>)))))


;;; general inference rules for frames and fabrics:


   (infer-frame-uncovered
     (params (<frame>))
      (preconds
        (and
          (is-frame <frame>)
          (~ (frame-properties <frame> addition covering))))
      (effects (
        (add (is-uncovered-frame <frame>)))))


   (infer-fabric-same-shape-as-frame
     (params (<fabric> <frame> <shapes>))
      (preconds
        (and
          (is-fabric <fabric>)
          (is-frame <frame>)
          (is-frame-shape <shapes>)
          (fabric-properties <fabric> shape <shapes>)
          (frame-properties <frame> shape <shapes>)))
      (effects (
        (add (same-shape <fabric> <frame>)))))


   (infer-fabric-decorated
     (params (<fabric>))
      (preconds
        (and
          (is-fabric <fabric>)
          (exists (<surface-decor>) (is-fabric-surface <surface-decor>)
              (fabric-properties <fabric> surface <surface-decor>))))
      (effects (
        (add (is-decorated <fabric>)))))
    

;;; specific inference rules for frames and fabrics:

   (infer-fabric-dyeable
     (params (<fabric>))
      (preconds
        (and
          (is-fabric <fabric>)
          (or
            (fabric-properties <fabric> type silk)
            (fabric-properties <fabric> type cotton-cloth))))
      (effects (
        (add (is-dyeable <fabric>)))))

   (infer-fabric-paintable
     (params (<fabric>))
      (preconds
        (and
          (is-fabric <fabric>) 
          (or
            (fabric-properties <fabric> type paper)
            (fabric-properties <fabric> type polyethylene-film))))
      (effects (
        (add (is-paintable <fabric>)))))

    (infer-fabric-grommettable
      (params (<fabric>))
       (preconds
         (and
           (is-fabric <fabric>)
           (or
             (fabric-properties <fabric> type silk)
             (fabric-properties <fabric> type cotton-cloth)
             (fabric-properties <fabric> type polyethylene-film))))
       (effects (
         (add (is-grommettable <fabric>)))))
 
   (infer-fabric-tieable-to-frame
     (params (<fabric> <frame>))
      (preconds
        (and
          (is-fabric <fabric>)
          (is-frame <frame>)
          (is-uncovered-frame <frame>)
          (same-shape <fabric> <frame>)
          (fabric-properties <fabric> edge grommetted))) 
     (effects (
        (add (are-tieable <fabric> <frame>)))))

   (infer-fabric-sewable-to-frame
     (params (<fabric> <frame>))
      (preconds
        (and
          (is-fabric <fabric>)
          (is-frame <frame>)
          (is-uncovered-frame <frame>)
          (same-shape <fabric> <frame>)
          (or
            (fabric-properties <fabric> type silk)
            (fabric-properties <fabric> type cotton-cloth))))
      (effects (
        (add (are-sewable <fabric> <frame>)))))

   (infer-fabric-heat-sealable-to-frame
     (params (<fabric> <frame>))
      (preconds 
        (and
          (is-fabric <fabric>)
          (is-frame <frame>)
          (is-uncovered-frame <frame>) 
          (same-shape <fabric> <frame>)
          (fabric-properties <fabric> type polyethylene-film)))
      (effects (
        (add (are-heat-sealable <fabric> <frame>)))))

   (infer-fabric-tapeable-to-frame
     (params (<fabric> <frame>))
      (preconds
        (and
          (is-fabric <fabric>)
          (is-frame <frame>)
          (is-uncovered-frame <frame>)
          (same-shape <fabric> <frame>)
          (or 
            (fabric-properties <fabric> type polyethylene-film)
            (fabric-properties <fabric> type paper))))
      (effects (
        (add (are-tapeable <fabric> <frame>)))))



;;; inference rules for kites:

   (infer-kite-is-flyable
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> addition covering)
          (frame-properties <frame> addition tail)
          (frame-properties <frame> addition bridle)))
      (effects (
        (add (is-flyable <kite>)))))

   (infer-kite-type-to-be-diamond
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> shape diamond)))
      (effects (
        (add (kite-type <kite> diamond)))))

   (infer-kite-type-to-be-bermuda-three-stick
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> shape stretch-hexagon)))
      (effects (
        (add (kite-type <kite> bermuda-three-stick)))))

   (infer-kite-type-to-be-rectangle
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> shape rectangle)))
      (effects (
        (add (kite-type <kite> rectangle)))))

   (infer-kite-type-to-be-indian-fighter
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> shape semicircle)))
      (effects (
        (add (kite-type <kite> indian-fighter)))))

   (infer-kite-type-to-be-four-circle
     (params (<kite>))
      (preconds
        (and
          (is-kite <kite>)
          (is-frame <frame>)
          (frame-properties <frame> bound-to <kite>)
          (frame-properties <frame> shape four-circle)))
      (effects (
        (add (kite-type <kite> four-circle)))))


))
