;;; Allegro CL interface to C code implementing UNIX socket client and server
;;; Copyright (c) 1993  Leonard Dickens
;;; Intelligent Data Management, NASA Goddard Space Flight Center
;;;
;;; This software is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
;;; to anyone for the consequences of using it or for whether it serves any 
;;; particular purpose or works at all.
;;;
;;; You may copy, modify and generally use this software freely, as long as
;;; this copyright notice is left intact in every copy or derivative work.
;;;

(defpackage :socket
  (:use common-lisp ff stream excl)
  (:export open-client
	   server-end-of-file
#|
	   server-get-connection
|#
	   server-get-port
	   server-initialize
	   server-poll
	   server-shutdown
	   server-startup
	   ))

(in-package :socket)

;;;;;;;;;;;;;;;;;;;;;;; Client (output) interface ;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;(load "../obj/open_socket.o")

;;;
;;; Open socket function takes two strings (port and host),
;;;  and returns a file descriptor for the socket.
;;;
(ff:defforeign 'open_socket 
  :entry-point (ff:convert-to-lang "open_socket")
  :arguments '(simple-string simple-string)
  :return-type :integer)


(defun open-client (&key port (host "localhost"))
  (declare (type simple-string port)
	   (type simple-string host))
  (let ((socket-descriptor (open_socket host port)))
    (if (< socket-descriptor 0)
	(warn "failed to open output socket")
      ;; See chapter 18 in the user's guide for more on this:
      (make-instance 'excl::bidirectional-terminal-stream
       :fn-in  socket-descriptor
       :fn-out socket-descriptor))))







;;;;;;;;;;;;;;;;;;;;;;; Server (input) interface ;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;
;;; C object code that implements a generic server interface.
;;;
#|
(load "../obj/socket_server.o")
(load "../obj/lisp-interface.o")
|#

#|
(ff:def-c-type (connection :in-foreign-space) :struct
	       (connect_fd     :signed-long)
	       (max_clients    :signed-long)
	       (n_clients      :signed-long)
	       (port           *:char)
	       (input_procedure           *:unsigned-long)  ; actually these
	       (new_connection_procedure  *:unsigned-long)  ; 3 fields are
	       (end_of_file_procedure     *:unsigned-long)) ; proc pointers
|#

(ff:defforeign 'c-socket-server-initialize
	       :entry-point (ff:convert-to-lang "lisp_socket_server_initialize")
	       :arguments nil
	       :return-type :integer)

(ff:defforeign 'c-socket-server-startup
	       :entry-point (ff:convert-to-lang "socket_server_startup")
	       :arguments '(string integer fixnum integer integer)
	       :return-type :integer)

#|
(ff:defforeign 'server-get-connection
   :entry-point (ff:convert-to-lang "socket_server_get_connection")
   :arguments nil
   :return-type (:pointer connection))))
|#

(ff:defforeign 'server-get-port
   :entry-point (ff:convert-to-lang "lisp_socket_server_get_port")
   :arguments nil
   :return-type :integer)

(ff:defforeign 'server-poll
   :entry-point (ff:convert-to-lang "lisp_socket_server_poll")
   :arguments '(integer)
   :return-type :integer)

(ff:defforeign 'server-end-of-file
   :entry-point (ff:convert-to-lang "socket_server_end_of_file")
   :arguments '(integer)
   :return-type :void)

(ff:defforeign 'server-shutdown
   :entry-point (ff:convert-to-lang "socket_server_shutdown")
   :arguments nil
   :return-type :integer)



(ff:defun-c-callable new-connection-handler ((file-descriptor :signed-long))
		     (format t "New FD: ~s~%" file-descriptor)
  (setf (aref *input-stream-lookup* file-descriptor)
	(make-instance 'excl::bidirectional-terminal-stream
		       :fn-in file-descriptor
		       :fn-out file-descriptor))
  0)


(ff:defun-c-callable end-of-file-handler ((file-descriptor :signed-long))
		     (format t "EOF on FD: ~s~%" file-descriptor)
  (close (aref *input-stream-lookup* file-descriptor))
  (setf  (aref *input-stream-lookup* file-descriptor) *standard-output*)
  0)


(ff:defun-c-callable all-input-handler ((file-descriptor :signed-long))
		     (format t "Input on FD: ~s~%" file-descriptor)
  (setq *input-file-descriptor* file-descriptor)
  (let* ((s (aref *input-stream-lookup* file-descriptor))
	 (ch nil)
	 (port (server-get-port))
	 (handler (gethash port *function-hash*)))
    (funcall handler s)
    (setq ch (read-char-no-hang s nil :eof))
    (cond ((eq ch :eof) (server-end-of-file file-descriptor))
	  ((eq ch nil))
	  (t (unread-char ch s))))
  0)



;;;
;;; Global variables to allow us to see values in handle-input that
;;;  get initialized in socket-server-loop.
;;;
(defvar *input-file-descriptor* nil)
(defvar *input-handlers* nil)
(defvar *input-stream-lookup* nil)

(defvar *function-hash* nil)


;;; 
;;; Definitions for the size external functions defined here:
;;;

(defun server-initialize ()
  (setq *function-hash* (make-hash-table :test #'equal :size 7))
	;; Note the assumed limit of 64 fds following:
  (setq *input-handlers* nil)
  (setq *input-stream-lookup*
	(make-array 64 :initial-element *standard-output*))
  (c-socket-server-initialize))


#|
;;; This didn't work in allegro 4.1 due to bugs in the foreign call
;;; handling.  Try it again when 4.2 comes out.
(defun server-startup (&key port read-stream-input-proc)
  (declare (type function read-stream-input-proc))
  (let* ((n-c-h (register-function 'new-connection-handler))
	 (eof-h (register-function 'end-of-file-handler))
	 (inp-h (register-function 'all-input-handler))
	 (portn))
    (cond ((stringp port) 
	   (setq portn (parse-integer port)))
	  ((integerp port)
	   (setq portn port)
	   (setq port (write-to-string port)))
	  (t nil))
    (setf (gethash portn *function-hash*) read-stream-input-proc)
    (format t "Startup with ~s ~s ~s ~s~%" port inp-h n-c-h eof-h)
    (c-socket-server-startup port inp-h 0 n-c-h eof-h)))
|#

(defun server-startup (&key port read-stream-input-proc)
  (declare (type function read-stream-input-proc))
  (let* (foo bar n-c-h eof-h inp-h portn)
    (multiple-value-setq (foo n-c-h bar)
			 (register-function 'new-connection-handler))
    (multiple-value-setq (foo eof-h bar)
			 (register-function 'end-of-file-handler))
    (multiple-value-setq (foo inp-h bar)
			 (register-function 'all-input-handler))
    (cond ((stringp port) 
	   (setq portn (parse-integer port)))
	  ((integerp port)
	   (setq portn port)
	   (setq port (write-to-string port)))
	  (t nil))
    (setf (gethash portn *function-hash*) read-stream-input-proc)
    (format t "Startup with ~s ~s ~s ~s~%" port inp-h n-c-h eof-h)
    (c-socket-server-startup port inp-h 0 n-c-h eof-h)))


#|
;;; This didn't work in Lucid: the closure was probably being garbage collected
;;; somehow or something like that.  If I had time to really debug this
;;; for a while, this is a much more elegant way to get callback functions.

;;; I haven't tried it yet in allegro
(defun server-startup (&key port read-stream-input-proc)
  (declare (type simple-string port)
	   (type function read-stream-input-proc))
  (let* ((n-c-h (register-function "_new_connection_handler"))
	 (eof-h (register-function "_end_of_file_handler"))
	 (inp-h-symbol (gensym "input_handler"))
	 inp-h inp-h-string inp-h-function
	 )
    (if read-stream-input-proc
	(progn
	  (setq inp-h-function
		(eval `(def-foreign-callable
			 (,inp-h-symbol  (:return-type :signed-long))
			 ((file-descriptor :signed-long))
			 ;;
			 (setq *input-file-descriptor* file-descriptor)
			 (let ((s (aref *input-stream-lookup* file-descriptor))
			       (ch nil))
			   (funcall ,read-stream-input-proc s)
			   (setq ch (read-char-no-hang s nil :eof))
			   (cond ((eq ch :eof)
				  (c-socket-server-end-of-file 
				    file-descriptor))
				 ((eq ch nil))
				 (t (unread-char ch s))))
			 0)))
	  (setq inp-h-string (concatenate 'string "_" 
					  (symbol-name inp-h-symbol)))
	  (setq inp-h (register-function inp-h-string))
	  (push inp-h-function *input-handlers*)
	  )
      (setq inp-h 0))
    (setf (gethash *function-hash* port) read-stream-input-proc)
    (c-socket-server-startup port inp-h 0 n-c-h eof-h)))
|#

