/******************************************************************************/
/*                                                                            */
/*   theorize.c                                                               */
/*   Main program                                                             */
/*                                                                            */
/*   written by: Eric R. Melz                                                 */
/*               UCLA Department of Psychology                                */
/*               Franz Hall, UCLA                                             */
/*               Los Angeles, CA, 90024                                       */
/*               USA                                                          */
/*                                                                            */
/*               Phone: (310) 825-8712                                        */
/*               email: emelz@cognet.ucla.edu                                 */
/*                                                                            */
/*   Portions of the code for quickprop and cascade-correlation have been     */
/*   adapted from the C versions of these algorithms written by               */
/*   Terry Reiger and R. Scott Crowder (RSC), respectively.                   */
/*                                                                            */
/******************************************************************************/


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "theorize.h"

struct mappingstruct mappings;
struct symbolstruct  symbols;
struct mapnodestruct mapnodes;
struct netstruct     *networks[MAXNETWORKS];  /* Array of network pointers     */
struct cmdstruct     *cmds[MAXCMDS];      /* Structure containing commands     */
struct epochstruct   epoch;               /* Structure containing epoch cmds   */
struct globalstruct  globals;             /* Structure containing global parms */
struct tsstruct      ts;                  /* Structure containing training set */
struct simstruct     sim;                 /* Structure containing the          */
                                          /*   simulation state                */
unsigned int         x;                   /* Used to init RNG                  */
FILE                 *weightfp;           /* File Pointer if weights are       */
                                          /*   loaded                          */
int                  weights_are_loaded_flag; /* 1=>Use loaded weights rather  */
                                          /* than generated weights            */

/******************************************************************************/
/*                                                                            */
/*   Miscellaneous utility functions                                          */
/*                                                                            */
/******************************************************************************/


/* Return true if c is alphanumeric or a period */
int ispalnum(c)
int c;
{
  return isalnum(c)||(c=='-')||(c=='.');
}

int blank_or_comment(line)
char *line;
{
  char *lineptr;

  for(lineptr=line;*lineptr!='\0';lineptr++) 
    if (*lineptr == '#')
      return 1;
    else
      if (!isspace(*lineptr)) 
	return 0;
  return 1;
}


char *pop_word(line)
char *line;
{
  char *lineptr;
  char *wordptr;

  /* skip blanks and garbage */
  for(lineptr=line;!ispalnum(*lineptr);lineptr++); 
  /* skip over word */
  for(;ispalnum(*lineptr)&&*lineptr!='\0';lineptr++); 
  return lineptr;
} 



char *pop_field(line)
char *line;
{
  char *lineptr;

  for(lineptr=line;*lineptr!=':';lineptr++);
  for(;!ispalnum(*lineptr);lineptr++);
  return lineptr;
}


/* Return 0 if there's no next word, 1 otherwise */
int get_next_word(line,word)
char *line;
char *word;
{
  char *lineptr,
       *wordptr;
  
  wordptr=word;
  if (blank_or_comment(line)) {
    *wordptr = '\0';  
    return 0;
  }
  /* skip blanks and garbage */
  for(lineptr=line;!ispalnum(*lineptr);lineptr++);
  for(;ispalnum(*lineptr)&&*lineptr!='\0';lineptr++,wordptr++) {
    *wordptr = *lineptr;
  }
  *wordptr = '\0';  
  return 1;
}

char *get_string_param(line)
char *line;
{
  char param[MAXLINE];
  char word[MAXLINE];

  strcpy(param,line);
  get_next_word(pop_field(param),word);
  return word;
}



/******************************************************************************/
/*                                                                            */
/*   read_net_file()                                                          */
/*                                                                            */
/*   Read in the network file in three steps                                  */
/*   1) Read the network definitions                                          */
/*   2) Read the commands for the execution loop                              */
/*   3) Read the global parametmers (cycles to stop, etc)                     */
/*                                                                            */
/******************************************************************************/

/* The following are routines which are used in loops to over the gap between
   hidden and output units and weights.  This structural change was made in
   order to implement the cascade correlation algorithm easier                */

void skip_node_forward(i,net)
int *i;
struct netstruct *net;
{
  if(*i==net->inputnodes+net->hiddennodes-1)
    *i=MAXNODES-net->outputnodes;
  else
    (*i)++;
}

void skip_node_backward(i,net)
int *i;
struct netstruct *net;
{
  if(*i==MAXNODES-net->outputnodes)
    *i=net->inputnodes+net->hiddennodes-1;
  else
    (*i)--;
}

/* The index of the the first node after the input nodes */
int after_innodes(net)
struct netstruct *net;
{
  if(net->hiddennodes==0)
    return MAXNODES-net->outputnodes;
  else
    return net->inputnodes;
}


void skip_weight_forward(i,net)
int *i;
struct netstruct *net;
{
  if (*i==net->nonoutputweights-1)
    *i=MAXWEIGHTS-net->outputweights;
  else
    (*i)++;
}

void skip_layer_forward(i,net)
int *i;
struct netstruct *net;
{
  if((net->unitvector[*i+1]==0)&&(*i<MAXLAYERS-1))
    *i=MAXLAYERS-1;
  else
    (*i)++;
}

int read_net_file(name) 
char *name;
{
  FILE *fp;
  int t;

  printf("\nReading network file %s...\n",name);

  fp = fopen(name,"r");
  if (get_globals(&fp))
    if (t=get_net_defs(&fp)) {
      if (t==2) {
	if (get_epochs(&fp))
	  if (get_cmds(&fp)) {
	    fclose(fp);
	    return 1;
	  }
      }
      else if (get_cmds(&fp)) {
	fclose(fp);
	return 1;
      }
    }
  return 0;
}


/* Get random weight for a new link */
float get_weight(net)
struct netstruct *net;
{
  char line[MAXLINE],
       line2[MAXLINE];
  float w,
        min,
        max;
  char *charptr;

  w=0.0;
  /* Get the weights from a file or generate a random weight */
  if (weights_are_loaded_flag) {
    fgets(line,MAXLINE,weightfp);
    while(!strncmp(line,"#",1))
      fgets(line,MAXLINE,weightfp);
    if (net->weights_from_file_flag) {
/**/printf("old read weight %s\n",line);
/*      strncpy(line2,line,strlen(line)-1); */
     for(charptr=line;(*charptr)&&ispalnum(*charptr);charptr++);
     *charptr='\0';
/**/printf("reading weight %s!\n",line);
      w=(float)strtod(line,&charptr);
/**/printf("float weight = %6.4f\n",w);
    }
  }
  if (net->local_weights_flag) {
    min=net->weight_min;
    max=net->weight_max;
  }
  else {
    min=globals.weight_min;
    max=globals.weight_max;
  }
  if (!(weights_are_loaded_flag && net->weights_from_file_flag)
      || net->local_weights_flag)
  w=w+((float)(rand()%1000) /1000.0) *
    ((max>min) ?
     (max-min) :
     (min-max)) + min;
  return w;
}


/* Initialize a new node */
void init_node(newnode,net)
struct node *newnode;
struct netstruct *net;
{
  int i;

  newnode->bias_weight= get_weight(net);
/**/printf("set bias weight to %6.4f\n",newnode->bias_weight);
  newnode->first_weight_to=0;
  newnode->last_weight_to=0;
  newnode->clamped=0;
  newnode->activation=0.0;
  newnode->delta=0.0;
  newnode->bed=0.0;
  newnode->prevbed=0.0;
  newnode->dbias=0.0;
  newnode->desired_output=0.0;
  newnode->error=0.0;
  newnode->freezetag=0;
  newnode->bedcache=0.0;
  newnode->dbiascache=0.0;
  newnode->cand_sum_value=0.0;
  newnode->freezetag=0;

  for(i=0;i<MAXESTIMATES;i++)
    newnode->bias_est[i]=0.0;
}


/* Intialize a new network structure and the data structures used to 
     process it                                                             */
void init_newnet(newnet)
struct netstruct *newnet;
{
  int i,j,k;

  strcpy(newnet->ID,"----");
  newnet->totalnodes = 0;
  newnet->inputnodes = 0;
  newnet->outputnodes = 0;
  newnet->hiddennodes = 0;
  newnet->nonoutputweights = 0;
  newnet->outputweights = 0;
  newnet->eta = 0.0;
  newnet->alpha = 0.0;
  newnet->gamma = 0.0;
  newnet->weight_min=0.0;
  newnet->weight_max=0.0;
  newnet->weights_from_file_flag=1;
  newnet->local_weights_flag=0;
  newnet->accumweightchange = 0.0;
  newnet->sigmoid_prime_offset = 0.0;
  newnet->out_sigmoid_prime_offset = 0.0;
  newnet->error_margin = 0.1;
  newnet->epoch_learn_flag=0;
  newnet->lflag=1;
  newnet->hyperr_flag=0;
  newnet->linear_flag=0;
  newnet->linear_thresh_flag=0;
  newnet->quickprop_flag=0;
  newnet->estimate_number=0;
  newnet->estimate_counter=0;
  newnet->decay= -0.0;
  newnet->mu=0.0;
  newnet->mode_switch_threshold=0.0;
  newnet->cutoff_error_flag=0;
  newnet->cutoff_margin=0.0;
  newnet->nepochs=0;
  newnet->no_bias_flag=0;
  newnet->domain_switch_node=0;
  newnet->avg_mapped_weights_flag=0;
  newnet->avg_mapped_weds_flag=0;
  newnet->avg_pen_mapped_weds_flag=0;
  newnet->distributed_layer=0;

  /* cascade correlation parameters */
  newnet->cascor_flag=FALSE;
  newnet->cascor_training_mode=OUTMODE;
  newnet->just_won_flag=TRUE;
  newnet->cycle_at_settling=0;
  newnet->first_out_epoch=TRUE;
  newnet->output_patience=8;
  newnet->output_timeout=0;
  newnet->timeout_epoch=0;
  newnet->quit_epoch=0;
  newnet->first_in_epoch=0;
  newnet->first_in_train_epoch=0;
  newnet->input_patience=8;
  newnet->input_timeout=0;
  newnet->num_candidates=MAXCANDIDATES;
  newnet->errorbits=0;
  newnet->true_error=0.0;
  newnet->last_error=0.0;
  newnet->best_candidate_score=0.0;
  newnet->last_score=0.0;
  newnet->output_change_threshold=0.01;
  newnet->input_change_threshold=0.03;                                            
  newnet->sum_error=0.0;
  newnet->avg_error=0.0;
  newnet->sum_sq_error=0.0;
  newnet->weight_multiplier= -1.0;

  /* Annealing parameters */
  newnet->anneal_flag=0;
  newnet->anneal_rate=0.0;
  newnet->anneal_type=UNIFORM;
  newnet->anneal_variance=0.0;
  newnet->variance_scale=1.0;
  newnet->grad_coeff=1.0;
  newnet->anneal_coeff=1.0;
  newnet->boltz_cons=1.0;

  /* Training set noise parameters */
  newnet->ts_noise_variance=0.0;

  /* Array initialization */

  for(i=0;i<MAXNODES;i++) 
    newnet->nodes[i] = NULL;

  for(i=0;i<MAXLAYERS;i++) {
    newnet->connectivity_vector[i].to_layer = 0;
    newnet->connectivity_vector[i].from_layer = 0;
    newnet->unitvector[i] = 0;
  }

  for(i=0;i<MAXWEIGHTS;i++) {
    for(j=0;j<MAXESTIMATES;j++)
      newnet->weightest[i][j]=0.0;
    newnet->weights[i]=0.0;
    newnet->wed[i]=0.0;
    newnet->dweights[i]=0.0;
    newnet->nodesfrom[i]=0;
    newnet->prevwed[i]=0.0;
    newnet->wedcache[i]=0.0;
    newnet->dweightscache[i]=0.0;
    newnet->freezetag[i]=0;
  }
  
  for(i=0;i<MAXCANDIDATES;i++)
    for(j=0;j<MAXOUTNODES;j++) {
      newnet->CandCor[i][j]=0.0;
      newnet->CandPrevCor[i][j]=0.0;
    }

  for(i=0;i<MAXOUTNODES;i++) 
    for(j=0;j<MAXPATTERNS;j++) 
      newnet->errorcache[i][j]=0.0;

  for(i=0;i<MAXLAYERS;i++)
    for(j=0;j<MAXPATTERNS;j++)
      for(k=0;k<MAXINNODES;k++) {
	newnet->layerpat_buff1[i][j][k]=0.0;
	newnet->layerpat_buff2[i][j][k]=0.0;
      }
}


void init_globals()
{
  globals.maxcycles = 0;     
  globals.cycle_criteria = 0;
  globals.no_bias_flag = 0;
  globals.no_output_bias_flag = 0;
  globals.print_tn_change_flag = 0;
  globals.print_dn_change_flag = 0;
  globals.print_weight_flag = 0;
  globals.print_vc_tn_settled_flag = 0;
  globals.print_ve_cor_tn_settled_flag = 0;
  globals.print_ve_lin_tn_settled_flag = 0;
  globals.print_tss_flag = 0;
  globals.scale_error_flag = 0;
  globals.abs_error_flag = 0;
  globals.invert_error_flag=0;
  globals.randomize_ts_flag = 0;
  globals.error_scale = 0.0;
  globals.error_offset = 0.0;
  globals.cause_offset = 0.0;
  globals.error_threshold = 0.0;
  globals.prob_decision_flag = 0;
  globals.weight_min = -1.0;
  globals.weight_max = 1.0;
  globals.mid_il_plateau_range=0.0;
  globals.inner_loop_increment=0.05;
  globals.inner_loop_min=0.0;
  globals.inner_loop_max=1.01;
}


/* Insert a weight to an arbitray output node, and
   adjust all indices of previous output weights                         */

/* Please finish me. */
void insert_out_weight(net,fromnodeindex,node)
struct netstruct *net;
int fromnodeindex;
struct node *node;
{
  int i;

/* insert the weight to the end of this node's group of weights. */

  /* Check for overflow */
  if(net->outputweights+net->nonoutputweights>=MAXWEIGHTS-2) {
    printf("**Error: Exceed the maximum number of weights (out) \n");
  }
  else {
    node->first_weight_to--;
    net->outputweights++;
    /* Shift previous weights back */
    for (i=MAXWEIGHTS-net->outputweights;i<node->last_weight_to;i++) {
      net->nodesfrom[i]=net->nodesfrom[i+1];
      net->weights[i]=net->weights[i+1];
    } 
    /* Adjust indices of all other outnodes */
    for(i=MAXNODES-net->outputnodes;net->nodes[i]!=node;i++) {
      net->nodes[i]->first_weight_to--;
      net->nodes[i]->last_weight_to--;
    }
    /* create the new weight */
    net->nodesfrom[node->last_weight_to]=fromnodeindex;
    net->weights[node->last_weight_to]=get_weight(net);
  }
}

/* This procedure is identical as the one above, except the weight is 
   chosen depending on the correlation between the candidate and output unit */
void insert_cand_to_out_weight(net,fromnodeindex,node,weight)
struct netstruct *net;
int fromnodeindex;
struct node *node;
float weight;
{
  int i;

/* insert the weight to the end of this node's group of weights. */

  /* Check for overflow */
  if(net->outputweights+net->nonoutputweights>=MAXWEIGHTS-2) {
    printf("**Error: Exceed the maximum number of weights (out) \n");
  }
  else {
    node->first_weight_to--;
    net->outputweights++;
    /* Shift previous weights back */
    for (i=MAXWEIGHTS-net->outputweights;i<node->last_weight_to;i++) {
      net->nodesfrom[i]=net->nodesfrom[i+1];
      net->weights[i]=net->weights[i+1];
    } 
    /* Adjust indices of all other outnodes */
    for(i=MAXNODES-net->outputnodes;net->nodes[i]!=node;i++) {
      net->nodes[i]->first_weight_to--;
      net->nodes[i]->last_weight_to--;
    }
    /* create the new weight */
    net->nodesfrom[node->last_weight_to]=fromnodeindex;
    net->weights[node->last_weight_to]=weight;
  }
}


/* Insert a weight to an output node to the end of the weightlist, and
   adjust all indices of previous output weights                         */

int insert_weight_to_end(net,fromnodeindex,node)
struct netstruct *net;
int fromnodeindex;
struct node *node;
{
  int i;

  /* Check for overflow */
  if(net->outputweights+net->nonoutputweights>=MAXWEIGHTS-2) {
    printf("**Error: Exceed the maximum number of weights (out) \n");
    return 0;
  }
  else {
    node->first_weight_to--;
    net->outputweights++;
    /* Shift previous weights back */
    for (i=MAXWEIGHTS-net->outputweights;i<MAXWEIGHTS-1;i++) {
      net->nodesfrom[i]=net->nodesfrom[i+1];
      net->weights[i]=net->weights[i+1];
    }
    /* Adjust indices of all other outnodes */
    for (i=MAXNODES-1;net->nodes[i]!=NULL;i--) {
      net->nodes[i]->first_weight_to--;
      net->nodes[i]->last_weight_to--;
    }
    /* create the new weight */
    net->nodesfrom[MAXWEIGHTS-1]=fromnodeindex;
    net->weights[MAXWEIGHTS-1]=get_weight(net);

    return 1;
  }
}

int insert_node_to_end(net,node)
struct netstruct *net;
struct node *node;
{
  int firstoutnode,i;

  /* Shift all previous nodes back */
  for(i=MAXNODES-1;net->nodes[i]!=0;i--);
  firstoutnode=i;
  for(i=firstoutnode;i<MAXNODES-1;i++)
    net->nodes[i]=net->nodes[i+1];
  /* Link the node into the network, at the end of the node list */
  net->nodes[MAXNODES-1]=node;
  return 1;
}


/* This creates a new network.  The highest node layer is relegated to
   MAXLAYERS-1 position in the connectivity and unitvectors, and the output
   nodes and their associated weights are placed at the end of their
   lists.  This facilitates the insertion of candidate nodes and output
   weights for Cascade Correlation.
*/

int create_net(newnet)
struct netstruct *newnet;
{
  int i,
      sum,
      current_weight_index,                   
      current_node_index,
      con,
      layer,                        
      tolayernode,
      fromlayernode,
      fromnodeindex;
  struct node *newnode;


  /* Total the number of units */
  for(i=0,sum=0;newnet->unitvector[i]!=0;i++)
    sum+=newnet->unitvector[i];
  /* MAXLAYERS is where the output layer is */
  sum+=newnet->unitvector[MAXLAYERS-1];

  newnet->totalnodes = sum;
  newnet->inputnodes = newnet->unitvector[0];
  newnet->outputnodes = newnet->unitvector[MAXLAYERS-1];
  newnet->hiddennodes=newnet->totalnodes-(newnet->outputnodes+newnet->inputnodes);
  /* Make sure there's enough spaces to fill the nodes */
  if (newnet->totalnodes>=MAXNODES-2) {
    printf("**Error: exceeded node memory\n");
    return 0;
  }
     
  current_weight_index=0;
  current_node_index=0;

  /* Loop for each layer */
  for(layer=0;layer<MAXLAYERS;skip_layer_forward(&layer,newnet)) {
    /* Loop for each node in the layer */
    for(tolayernode=0;tolayernode<newnet->unitvector[layer];tolayernode++) {
      /* Create a new node */ 
      if((newnode=(struct node *)malloc(sizeof(struct node)))==NULL) {
	printf("\n** Error: Out of memory\n");
	return 0;
      }
      init_node(newnode,newnet);

      /* If we are creating weights for nodes in the output layer, insert them
         at the end of the node list, initialize indices accordingly.        */

      if (layer==MAXLAYERS-1) {
	newnode->first_weight_to=MAXWEIGHTS;
	newnode->last_weight_to=MAXWEIGHTS-1;
      }
      else {
	newnode->first_weight_to = current_weight_index;
	/* Assume at least one link, otherwise last_weight_to will be negative */
	newnode->last_weight_to = current_weight_index-1;
	/* Loop through the connectivity */
      }

      for(con=0;newnet->connectivity_vector[con].from_layer!=0;con++) {
	/* Check if the connectivity is to the current layer */
	if((newnet->connectivity_vector[con].to_layer-1)==layer) {
	  /* Set up connections between the layers specified by the
              connectivity item */
	  for(fromlayernode=0;
	      fromlayernode<
           newnet->unitvector[newnet->connectivity_vector[con].from_layer-1];
	      fromlayernode++) {

	    /* If this is an output layer which we are creating weights for,
	       insert the weights at the end of the list.  Otherwise, grow
	       list from left to right                                     */

	    fromnodeindex=get_node_index(newnet,
					 newnet->connectivity_vector[con].from_layer,
					 fromlayernode);
	    if (layer==MAXLAYERS-1) {
	      if(!insert_weight_to_end(newnet,fromnodeindex,newnode))
		return 0;
	    }
	    else {
	      newnode->last_weight_to++;
	      newnet->nonoutputweights++;
	      /* Store index of the node on the source end of the link */
	      newnet->nodesfrom[current_weight_index]=fromnodeindex;
	      newnet->weights[current_weight_index++]=get_weight(newnet);
	      /* Note: this is a redundant error check but oh well */
	      if (newnet->weights[current_weight_index+1]!=0.0) {
		printf("**Error: exceed maximum number of weights \n");
		return 0;
	      }
	    }
	  }
	} /* fromlayer loop */
      } /* connectivity loop */
      /* link the new node into the network structure.  Insert the node to
         the end of the list if it is an ouput node. */
      if (layer==MAXLAYERS-1) {
	if(!insert_node_to_end(newnet,newnode))
	  return 0;
      }
      else {
	newnet->nodes[current_node_index++]=newnode;
	if (newnet->nodes[current_node_index+1]!=NULL) {
	  printf("**Error: exceed maximum nodes\n");
	  return 0;
	}
      }
    } /* for layernode */
  } /* layer */
  return 1;
} 


/* Retrieve the unitvector index of the lastlayer before the layers are blank.
   When the output layer hasn't been moved yet, it is the index of the unmoved
   output layer.  After the output layer has been moved, it is the index of
   the last hidden layer. */

int lastlayer(net)
struct netstruct *net;
{ 
  int i;

  for(i=0;net->unitvector[i]!=0;i++);
  return i-1;
}

/* Move the output layer (unitvector and connectivity), to MAXLAYERS-1 */
void move_out_layer(net)
struct netstruct *net;
{
  int l,i;
  
  l=lastlayer(net);
  net->unitvector[MAXLAYERS-1]=net->unitvector[l];
  net->unitvector[l]=0;

  for(i=0;net->connectivity_vector[i].from_layer!=0;i++) {
    if(net->connectivity_vector[i].to_layer==l+1)
      net->connectivity_vector[i].to_layer=MAXLAYERS;
  }
}

    
int get_net_defs(fp)
FILE **fp;
{
  char             linestring[MAXLINE];     /* Temporary line                */
  char             word[MAXLINE],           /* Temporary word                */
                   *line;                   /* Pointer to the current pos    */
                                            /*   in the line                 */
  struct netstruct *newnet;                 /* Pointer to a new network      */
  int              i,                       
                   sum,                     /* Total number of units         */
                   curr_net;                /* The current network           */
  float           temp;
  

  /* Read an indefinite number of network definitions */
  curr_net=0;
  while (1) {
    /* Create new network structure and initialize temporary variables */
    printf("Creating network :");
    if((newnet=(struct netstruct *) malloc(sizeof(struct netstruct))) == NULL) {
      printf("\n** Error: Out of memory\n");
      return 0;
    }

    init_newnet(newnet);
    line=linestring;
    fgets(line,MAXLINE,*fp);

    /* Read lines until the next Network definition is hit, or the COMMANDS 
          section is encountered */
    while (strncmp(line,"NETWORK",7) &&
	   strncmp(line,"COMMANDS",8) &&
	   strncmp(line,"EPOCHS",6)) {
      /* Set variables in here depending on the field read */
      if (!blank_or_comment(line)) {
	if (!strncmp(line,"ID",2)) {
	  strcpy(newnet->ID,get_string_param(line));  
	  printf("%s\n",newnet->ID);
	}
	else if (!strncmp(line,"CONNECTIVITY",12)) {
	  /* Iterate through the numbers in the paramter list in grous of 2 */
	  i=0;
	  line=pop_field(line);
	  while (get_next_word(line,word)) {
	    newnet->connectivity_vector[i].from_layer = atoi(word);
	    line=pop_word(line);
	    get_next_word(line,word);
	    newnet->connectivity_vector[i].to_layer = atoi(word);
            line=pop_word(line);
	    i++;
	  }
	  newnet->connectivity_vector[i].from_layer = 0;
	}
	else if (!strncmp(line,"UNITS",5)) {
	  /* Iterate through the numbers in the paramter list  */
	  i=0;
	  line=pop_field(line);
	  while (get_next_word(line,word)) {
	    newnet->unitvector[i] = atoi(word);
	    line=pop_word(line);
	    i++;
	  }
	  newnet->unitvector[i] = 0;
	}
	else if (!strncmp(line,"ETA",3)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->eta=atof(word);
	}
	else if (!strncmp(line,"ALPHA",5)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->alpha=atof(word);
	}
	else if (!strncmp(line,"GAMMA",5)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->gamma=atof(word);
	}
	else if (!strncmp(line,"WEIGHT-MAX",10)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->weight_max=atof(word);
	  newnet->local_weights_flag=1;
	}
	else if (!strncmp(line,"WEIGHT-MIN",10)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->weight_min=atof(word);
	  newnet->local_weights_flag=1;
	}
	else if (!strncmp(line,"FREEZE-WT",9)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->freezetag[atoi(word)]=1;
	}
	else if (!strncmp(line,"DOMAIN-SWITCH-NODE",18)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->domain_switch_node=atoi(word);
	}
	else if (!strncmp(line,"DISTRIBUTED-LAYER",17)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->distributed_layer=atoi(word);
	}
	else if (!strncmp(line,"AVG-MAPPED-WEIGHTS",18)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->avg_mapped_weights_flag=1;
	  else
	    newnet->avg_mapped_weights_flag=0;
	}
	else if (!strncmp(line,"AVG-MAPPED-WEDS",15)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->avg_mapped_weds_flag=1;
	  else
	    newnet->avg_mapped_weds_flag=0;
	}
	else if (!strncmp(line,"AVG-PEN-MAPPED-WEDS",19)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->avg_pen_mapped_weds_flag=1;
	  else
	    newnet->avg_pen_mapped_weds_flag=0;
	}
	else if (!strncmp(line,"WEIGHT-FROM-FILE",16)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->weights_from_file_flag=1;
	  else
	    newnet->weights_from_file_flag=0;
	}
       	else if (!strncmp(line,"SIG'-OFFSET",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->sigmoid_prime_offset=atof(word);
	}
       	else if (!strncmp(line,"OUT-SIG'-OFFSET",15)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->out_sigmoid_prime_offset=atof(word);
	}
       	else if (!strncmp(line,"ERROR-MARGIN",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->error_margin=atof(word);
	}
       	else if (!strncmp(line,"NEPOCHS",7)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->nepochs=atoi(word);
	}
	else if (!strncmp(line,"EPOCH-LEARN-FLAG",16)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->epoch_learn_flag=1;
	  else
	    newnet->epoch_learn_flag=0;
	}
	else if (!strncmp(line,"NO-BIAS",7)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->no_bias_flag=1;
	  else
	    newnet->no_bias_flag=0;
	}
	else if (!strncmp(line,"HYPERR-FLAG",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->hyperr_flag=1;
	  else
	    newnet->hyperr_flag=0;
	}
	else if (!strncmp(line,"LINEAR-FLAG",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->linear_flag=1;
	  else
	    newnet->linear_flag=0;
	}
	else if (!strncmp(line,"LINEAR-THRESH-FLAG",18)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->linear_thresh_flag=1;
	  else
	    newnet->linear_thresh_flag=0;
	}
	else if (!strncmp(line,"SYMMETRIC-SIG-FLAG",18)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->symmetric_sig_flag=1;
	  else
	    newnet->symmetric_sig_flag=0;
	}
	else if (!strncmp(line,"QUICKPROP-FLAG",14)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->quickprop_flag=1;
	  else
	    newnet->quickprop_flag=0;
	}
	else if (!strncmp(line,"CUTOFF-ERROR-FLAG",17)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->cutoff_error_flag=1;
	  else
	    newnet->cutoff_error_flag=0;
	}
       	else if (!strncmp(line,"CUTOFF-MARGIN",13)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->cutoff_margin=atof(word);
	}
       	else if (!strncmp(line,"ESTIMATE-NUMBER",15)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->estimate_number=atoi(word);
	}
       	else if (!strncmp(line,"CURVATURE-THRESH",16)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->curvature_threshold=atof(word);
	}
       	else if (!strncmp(line,"MU",2)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->mu=atof(word);
	}
       	else if (!strncmp(line,"INPUT-ETA",9)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->input_eta=atof(word);
	}
       	else if (!strncmp(line,"DECAY",5)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->decay=atof(word);
	}
       	else if (!strncmp(line,"MODE-SWITCH-THRESH",18)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->mode_switch_threshold=atof(word);
	}
	/* cascade correlation parameters */
       	else if (!strncmp(line,"OUTPUT-CHANGE-THRESH",20)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->output_change_threshold=atof(word);
	}
       	else if (!strncmp(line,"INPUT-CHANGE-THRESH",19)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->input_change_threshold=atof(word);
	}
	else if (!strncmp(line,"CASCOR-FLAG",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->cascor_flag=1;
	  else
	    newnet->cascor_flag=0;
	}
       	else if (!strncmp(line,"OUTPUT-PATIENCE",15)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->output_patience=atoi(word);
	}
       	else if (!strncmp(line,"OUTPUT-TIMEOUT",14)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->output_timeout=atoi(word);
	}
       	else if (!strncmp(line,"INPUT-PATIENCE",14)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->input_patience=atoi(word);
	}
       	else if (!strncmp(line,"INPUT-TIMEOUT",13)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->input_timeout=atoi(word);
	}
       	else if (!strncmp(line,"WEIGHT-MULTIPLIER",17)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->weight_multiplier=atof(word);
	}
       	else if (!strncmp(line,"CANDIDATES",10)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->num_candidates=atoi(word);
	}
	else if (!strncmp(line,"ANNEAL-FLAG",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"Y",1))
	    newnet->anneal_flag=1;
	  else
	    newnet->anneal_flag=0;
	}
	else if (!strncmp(line,"ANNEAL-RATE",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->anneal_rate=atof(word);
	}
	else if (!strncmp(line,"ANNEAL-TYPE",11)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"CAUCHY",6))
	    newnet->anneal_type=CAUCHY;
	  else if (!strncmp(line,"UNIFORM",7))
	    newnet->anneal_type=UNIFORM;
	}
	else if (!strncmp(line,"ANNEAL-VARIANCE",15)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->anneal_variance=atof(word);
	}
	else if (!strncmp(line,"GRAD-COEFF",10)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->grad_coeff=atof(word);
	}
	else if (!strncmp(line,"ANNEAL-COEFF",12)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->anneal_coeff=atof(word);
	}
	else if (!strncmp(line,"BOLTZ-CONS",10)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->boltz_cons=atof(word);
	}
	else if (!strncmp(line,"TS-NOISE-VARIANCE",17)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  newnet->ts_noise_variance=atof(word);
	}
	else if (!strncmp(line,"TS-NOISE-TYPE",13)) {
	  line=pop_field(line);
	  get_next_word(line,word);
	  if (!strncmp(line,"CAUCHY",6))
	    newnet->ts_noise_type=CAUCHY;
	  else if (!strncmp(line,"UNIFORM",7))
	    newnet->ts_noise_type=UNIFORM;
	}
	else {
	  /* Line is neither a blank, comment, or valid field */
	  printf("**Error: Invalid Network definition\n");
	  return 0;
	}
      }
      line=linestring;
      fgets(line,MAXLINE,*fp);
    }
    /* process varables to create networks (see readnet) */
    if (newnet->unitvector[0] == 0) {
      printf("**Error: No units specified\n");
      return 0;
    }
    else if (newnet->connectivity_vector[0].from_layer == 0) {
      printf("**Error: No connectivity specified\n");
      return 0;
    }
    else {
      /* Move the output layer to MAXLAYERS-1, for cascade-correlation 
	 flexibility */
      move_out_layer(newnet);

      /* Create the network from the definition and link it into the net list */
      if (!create_net(newnet))
	return 0;
      networks[curr_net++] = newnet;
      if (curr_net>=MAXNETWORKS) {
	printf("**Error: too many networks\n");
	return 0;
      }
    }
    if (!strncmp(line,"COMMANDS",8)) {
      /* Terminate the list of networks with a NULL and return */
      networks[curr_net] = NULL;
      return 1;
    }
    else if (!strncmp(line,"EPOCHS",6)) {
      /* Terminate the list of networks with a NULL and return */
      networks[curr_net] = NULL;
      return 2;
    }
  }
}


int get_epochs(fp)
FILE **fp;
{
  char line[MAXLINE];
  char *ret;
  
  printf("\nReading epochs....\n\n");
  epoch.cmdline=0;
  ret=fgets(line,MAXLINE,*fp);
  while ((ret!=NULL) && (strncmp(line,"COMMANDS",8))) {
    if (!blank_or_comment(line)) {
      strncpy(epoch.cmd[epoch.cmdline],line,MAXLINE);
      epoch.cmdline++;
     }
    ret=fgets(line,MAXLINE,*fp);
  }
  if (ret==NULL) {
    printf("Error: must specify EPOCHS then COMMANDS\n");
    return 0;
  }
  else 
    return 1;
}

void init_cmds()
{
  int i;

  for (i=0;i<MAXCMDBLOCKS;i++) 
    cmds[i]=NULL;
}

int get_cmds(fp)
FILE **fp;
{
  char line[MAXLINE];
  char *ret;
  int current_cmdblock;
  
  init_cmds();
  current_cmdblock=0;

  while(1) {
    if((cmds[current_cmdblock]=(struct cmdstruct *) malloc(sizeof(struct cmdstruct))) == NULL)
      printf("error while reading commands: out of memory\n");
    printf("\nReading commands....\n\n");
    cmds[current_cmdblock]->cmdline=0;
    ret=fgets(line,MAXLINE,*fp);
    while ((ret!=NULL) && (strncmp(line,"COMMANDS",8))) {
      if (!blank_or_comment(line)) {
	if (!strncmp(line,"ID",2)) {
	  strcpy(cmds[current_cmdblock]->ID,get_string_param(line));  
	}
	else {
	  strncpy(cmds[current_cmdblock]->cmd[cmds[current_cmdblock]->cmdline],line,MAXLINE);
	  cmds[current_cmdblock]->cmdline++;
	}
      }
      ret=fgets(line,MAXLINE,*fp);
    }
    if (ret==NULL) 
      return 1;
    else 
      current_cmdblock++;
  }
}

int get_globals(fp)
FILE **fp;
{
  char linestring[MAXLINE],
       word[MAXLINE];
  char *line,
       *ret;

  init_globals();
  /* Skip over comments and blank spaces at beginning of file */
  line=linestring;
  fgets(line,MAXLINE,*fp);
  while (strncmp(line,"GLOBALS",7)) {
    /* Check for non blank or comment lines and return error otherwise */
    if (!blank_or_comment(line)) {
      printf("** Error: Non blank or comment line encountered\n");
      return 0;
    }
    fgets(line,MAXLINE,*fp);
  }


  printf("Reading Global Paramters...\n\n\n");
  line=linestring;
  ret=fgets(line,MAXLINE,*fp);
  while (strncmp(line,"NETWORK",7)) {
    if (!blank_or_comment(line)) {
      if (!strncmp(line,"MAXCYCLES",9)) {
	line=pop_field(line);
        get_next_word(line,word);
	globals.maxcycles=atoi(word);
      }
      else if (!strncmp(line,"CYCLE-CRITERIA",14)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.cycle_criteria=atoi(word);
      }
      else if (!strncmp(line,"PRINT-WEIGHTS",13)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_weight_flag=1;
	else
	  globals.print_weight_flag=0;
      }
      else if (!strncmp(line,"PRINT-VE-LIN-TN-SETTLED",23)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_ve_lin_tn_settled_flag=1;
	else
	  globals.print_ve_lin_tn_settled_flag=0;
      }
      else if (!strncmp(line,"PRINT-VE-COR-TN-SETTLED",23)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_ve_cor_tn_settled_flag=1;
	else
	  globals.print_ve_cor_tn_settled_flag=0;
      }
      else if (!strncmp(line,"PRINT-VC-TN-SETTLED",19)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_vc_tn_settled_flag=1;
	else
	  globals.print_vc_tn_settled_flag=0;
      }
      else if (!strncmp(line,"PRINT-DN-CHANGE",15)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_dn_change_flag=1;
	else
	  globals.print_dn_change_flag=0;
      }
      else if (!strncmp(line,"PRINT-TN-CHANGE",15)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_tn_change_flag=1;
	else
	  globals.print_tn_change_flag=0;
      }
      else if (!strncmp(line,"PRINT-TSS",9)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.print_tss_flag=1;
	else
	  globals.print_tss_flag=0;
      }
      else if (!strncmp(line,"NO-BIAS",7)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.no_bias_flag=1;
	else
	  globals.no_bias_flag=0;
      }
      else if (!strncmp(line,"NO-OUTPUT-BIAS",14)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.no_output_bias_flag=1;
	else
	  globals.no_output_bias_flag=0;
      }
      else if (!strncmp(line,"SCALE-ERROR-FLAG",16)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.scale_error_flag=1;
	else
	  globals.scale_error_flag=0;
      }
      else if (!strncmp(line,"ABS-ERROR-FLAG",14)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.abs_error_flag=1;
	else
	  globals.abs_error_flag=0;
      }
      else if (!strncmp(line,"INVERT-ERROR-FLAG",17)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.invert_error_flag=1;
	else
	  globals.invert_error_flag=0;
      }
      else if (!strncmp(line,"PROB-DECISION-FLAG",18)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.prob_decision_flag=1;
	else
	  globals.prob_decision_flag=0;
      }
      else if (!strncmp(line,"MID-IL-PLATEAU-FLAG",19)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.mid_il_plateau_flag=1;
	else
	  globals.mid_il_plateau_flag=0;
      }
      else if (!strncmp(line,"RANDOMIZE-TS-FLAG",17)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.randomize_ts_flag=1;
	else
	  globals.randomize_ts_flag=0;
      }
      else if (!strncmp(line,"MID-IL-PLATEAU-RANGE",20)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.mid_il_plateau_range=atof(word);
      }
      else if (!strncmp(line,"RAND-IL-PLATEAU-FLAG",20)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  globals.rand_il_plateau_flag=1;
	else
	  globals.rand_il_plateau_flag=0;
      }
      else if (!strncmp(line,"WEIGHT-MAX",10)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.weight_max=atof(word);
      }
      else if (!strncmp(line,"WEIGHT-MIN",10)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.weight_min=atof(word);
      }
      else if (!strncmp(line,"ERROR-SCALE",11)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.error_scale=atof(word);
      }
      else if (!strncmp(line,"ERROR-OFFSET",12)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.error_offset=atof(word);
      }
      else if (!strncmp(line,"CAUSE-OFFSET",12)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.cause_offset=atof(word);
      }
      else if (!strncmp(line,"ERROR-THRESHOLD",15)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.error_threshold=atof(word);
      }
      else if (!strncmp(line,"INNER-LOOP-INCREMENT",20)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.inner_loop_increment=atof(word);
      }
      else if (!strncmp(line,"INNER-LOOP-MIN",14)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.inner_loop_min=atof(word);
      }
      else if (!strncmp(line,"INNER-LOOP-MAX",14)) {
	line=pop_field(line);
	get_next_word(line,word);
	globals.inner_loop_max=atof(word);
      }
      else if (!strncmp(line,"DN-CC-TRIGGER-FLAG",18)) {
	line=pop_field(line);
	get_next_word(line,word);
	if (!strncmp(line,"Y",1))
	  sim.dn_cc_trigger_flag=1;
	else
	  sim.dn_cc_trigger_flag=0;
      }
      else {
	printf("\n**Error: Unknown global parameter encounterd :\n  %s\n",line);
	return 0;
      }
    }
    line=linestring;
    ret=fgets(line,MAXLINE,*fp);
  }
  
  return 1;
}




/******************************************************************************/
/*                                                                            */
/*   read_mapping_file(), read_symbol_table()                                 */
/*                                                                            */
/*   Read in the mappings and symbol tables                                   */
/*                                                                            */
/******************************************************************************/

void read_mappings_file(name)
char *name;
{
  FILE *fp;
  char *ret,
       *line;
  char linestring[MAXLINE],
       word[MAXLINE];
    
  fp=fopen(name,"r");
  printf("\nReading mappings....\n\n");
  mappings.lastrow=0;
  line=linestring;
  ret=fgets(line,MAXLINE,fp);
  while (ret!=NULL) {
    if (!blank_or_comment(line)) {
      get_next_word(line,word);
      strcpy(mappings.source[mappings.lastrow],word);
      line=pop_word(line);
      get_next_word(line,word);
      strcpy(mappings.target[mappings.lastrow],word);
      mappings.lastrow++;
    }
    line=linestring;
    ret=fgets(line,MAXLINE,fp);
  }
  fclose(fp);
}

void read_symbol_table(name)
char *name;
{
  FILE *fp;
  char *ret,
       *line;
  char linestring[MAXLINE],
       word[MAXLINE];
    
  fp=fopen(name,"r");
  printf("\nReading symbol table....\n\n");
  symbols.lastrow=0;
  line=linestring;
  ret=fgets(line,MAXLINE,fp);
  while (ret!=NULL) {
    if (!blank_or_comment(line)) {
      get_next_word(line,word);
      symbols.index[symbols.lastrow]=atoi(word);
      line=pop_word(line);
      get_next_word(line,word);
      strcpy(symbols.symbol[symbols.lastrow],word);
      symbols.lastrow++;
    }
    line=linestring;
    ret=fgets(line,MAXLINE,fp);
  }
  fclose(fp);
}


/* Lookup the index of a symbol */
int get_symbol_index(symbol)
char *symbol;
{
  int i;

  for(i=0;i<symbols.lastrow;i++)
    if(!strcmp(symbol,symbols.symbol[i]))
      return i;
  return -1;
}

/* Create table of node correspondences */
void compute_node_correspondences()
{
  int i;

  mapnodes.lastrow=mappings.lastrow;
  for(i=0;i<mapnodes.lastrow;i++) {
    mapnodes.sourcenode[i]=get_symbol_index(mappings.source[i]);
    mapnodes.targetnode[i]=get_symbol_index(mappings.target[i]);
  }
}   
    

/******************************************************************************/
/*                                                                            */
/*   read_ts_file()                                                           */
/*                                                                            */
/*   Read in the training sets from the training set file                     */
/*                                                                            */
/******************************************************************************/

int read_ts_file(name)
char *name;
{
  FILE *fp;
  char linestring[MAXLINE],
       word[MAXLINE];
  char *line,
       *ret;
  int  layer,
       i,
       j;

  fp=fopen(name,"r");

  ts.in=0;
  ts.out=0;
  ts.neg_cases=0;
  for (i=0;i<MAXLAYERS;i++)
    ts.layer[i]=0;

  line=linestring;
  ret=fgets(line,MAXLINE,fp);
  while ((blank_or_comment(line) || !strncmp(line,"IN",2) || 
	  !strncmp(line,"OUT",3) || !strncmp(line,"NEG-CASES",9) ||
	  !strncmp(line,"L",1))
	 && (ret != NULL)) {
    if(!blank_or_comment(line)) {
      if (!strncmp(line,"IN",2)) {
	line=pop_field(line);
	get_next_word(line,word);
	ts.in=atoi(word);
      }
      else if (!strncmp(line,"L",1)) {
        get_next_word(line,word);
	layer=atoi(word+1);
        line=pop_field(line);
	get_next_word(line,word);
        ts.layer[layer]=atoi(word);
      }
      else if (!strncmp(line,"OUT",3)) {
	line=pop_field(line);
	get_next_word(line,word);
	ts.out=atoi(word);
      }
      else if (!strncmp(line,"NEG-CASES",9)) {
	line=pop_field(line);
	get_next_word(line,word);
	ts.neg_cases=atoi(word);
      }
    }
    line=linestring;
    ret=fgets(line,MAXLINE,fp);
  }
  if (ts.in==0 || ts.out==0) {
    printf("**Error: Must specify both the number of inputs and outputs ");
    printf("at the start of the file\n");
    return 0;
  }
   
  
  /* Read in the training sets */
  
  ts.tsnum=0;
  while (ret!=NULL) {
    if(!blank_or_comment(line)) {
      for(i=0;i<ts.in;i++) {
	get_next_word(line,word);
	ts.inpat[ts.tsnum][i] = atof(word);
	ts.inpat_buff[ts.tsnum][i] = atof(word);
	line=pop_word(line); 
      } 
      for(layer=0;layer<MAXLAYERS;layer++) {
	if (ts.layer[layer]!=0)
	  for(i=0;i<ts.layer[layer];i++) {
	    get_next_word(line,word);
	    ts.layerpat[layer][ts.tsnum][i] = atof(word);
	    line=pop_word(line); 
	  }
      }
      for(i=0;i<ts.out;i++) {
	get_next_word(line,word);
	ts.outpat[ts.tsnum][i] = atof(word);
	ts.outpat_buff[ts.tsnum][i] = atof(word);
	line=pop_word(line); 
      }
    ts.tsnum++;
    }
    line=linestring;
    ret=fgets(line,MAXLINE,fp);
  }

/*****
    for (layer=0;layer<MAXLAYERS;layer++) {   
     if (ts.layer[layer]!=0) {          
      printf("L%d (%d):\n ",layer,ts.layer[layer]);
      for(i=0;i<ts.tsnum;i++) {
	printf("P%d: ",i);
	for (j=0;j<ts.layer[layer];j++)
	  printf("%4.2f ",ts.layerpat[layer][i][j]);
	printf("\n");
      }
    }
  }
       
*****/  

  return 1;
}



/******************************************************************************/
/*                                                                            */
/*   Network Manipulation Routines and Utilities                              */
/*                                                                            */
/******************************************************************************/

struct netstruct *get_net(netID)
char *netID;
{
  int i;

  i=0;
  while(networks[i] != NULL) {
    if (!strcmp(networks[i]->ID,netID))
      return networks[i];
    i++;
  }
  return NULL;
}

void print_node_label(net,layer, node)
struct netstruct *net;
int layer, node;
{
  /* Compute what the output layer would be indexed as if it weren't moved
     to MAXLAYERS-1 */
  if (layer==MAXLAYERS)
    layer=lastlayer(net)+2;
  printf("L%d-N%d ",layer,(node+1));
}


/* Retrieve the index of the weight from a given node with input from a given
   node index */
int get_weight_index(net, tonodeindex, fromnodeindex)
struct netstruct *net;
int tonodeindex;
int fromnodeindex;
{
  int i;

  for(i=net->nodes[tonodeindex]->first_weight_to;i<=net->nodes[tonodeindex]->last_weight_to;i++) 
    if (net->nodesfrom[i]==fromnodeindex)
      return i;
  return -1;
}


/* Retrieve the index of the node given the layer and the index of the node
   in the layer */
int get_node_index(net, layer, node)
struct netstruct *net;
int layer, node;
{
  int i,j,sum;

  /* The output layer is a special case, since these nodes are at the
     end of the nodes list */
  if (layer==MAXLAYERS)
    sum=MAXNODES-net->outputnodes;
  else {
    sum=0;
    for(i=0;i<(layer-1);i++)
      sum+=net->unitvector[i];
  }
/*printf("ni=%d\n",sum+node);*/
  return sum+node;


}

/* Retrieve a pointer to a node give a layer and the index of the node in
   the layer */
struct node *get_node(net,layer,node)
struct netstruct *net;
int layer, node;
{
  return net->nodes[get_node_index(net,layer,node)];
}


/* print out comprehensive summary of the state of the network */

void print_net_vitals(net_ID)
char *net_ID;
{
  struct netstruct *net;
  int i,j;

  net=get_net(net_ID);
  printf("\nPrinting state for network %s:\n\n",net->ID);

  for(i=0;net->unitvector[i]!=0;i++) {
    printf("            ");
    for(j=0;j<net->unitvector[i];j++) {
      print_node_label(net,i+1,j);
      printf("      ");
    }
    printf("\n");
    printf("Activation: ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->activation);
    printf("\n");
    printf("Output    : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->output);
    printf("\n");
    printf("Desired   : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->desired_output);
    printf("\n");
    printf("Error     : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->error);
    printf("\n");
    printf("Delta     : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->delta);
    printf("\n");
    printf("Bias      : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->bias_weight);
    printf("\n");
    printf("dbias     : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->dbias);
    printf("\n");
    printf("bed       : ");
    for(j=0;j<net->unitvector[i];j++) 
      printf("%9.6f   ",net->nodes[get_node_index(net,i+1,j)]->bed);
    printf("\n\n");
  }
}
    
    

void print_net_weights(net)
struct netstruct *net;
{
  int i,j,k,weightindex;
  struct node *node;

  /* Print a table of weights, with the rows being the units in the from 
        layer, and the columns being the units in the to layer.  The last
        row is the bias */
  
  /* weightindex is the current weight in the network's weight vector */
  printf("Weight table for %s:\n\n",net->ID);
  for(i=0;net->connectivity_vector[i].from_layer != 0;i++) {
    /* Print out column header */
    printf("          ");
    for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) {
      print_node_label(net,net->connectivity_vector[i].to_layer,k);
      printf("    ");
    }
    printf("\n");
    for(j=0;j<net->unitvector[net->connectivity_vector[i].from_layer-1];j++) {
      print_node_label(net,net->connectivity_vector[i].from_layer,j);
      printf(": ");
      /* Print the weights */
      for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) {
	for(weightindex= 
	    (node=
        get_node(net,net->connectivity_vector[i].to_layer,k))->first_weight_to;
	    net->nodesfrom[weightindex]!=
	       get_node_index(net,net->connectivity_vector[i].from_layer,j);
	    weightindex++);
/**//*printf("%+ 12.10f ",net->weights[weightindex]); */
	printf("%f ",net->weights[weightindex]);
      } /* for k - to layer (column) loop */
      printf("\n");
    } /* for j - fromlayer (row) loop */
    printf("Bias  : ");
    for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) {
/**//* printf("%+ 12.10f ",
        (get_node(net,net->connectivity_vector[i].to_layer,k))->bias_weight);*/
      printf("%f ",
        (get_node(net,net->connectivity_vector[i].to_layer,k))->bias_weight);
      }
    printf("\n\n");
  } /* for i - table loop */
}


void print_net_weight_errors(net)
struct netstruct *net;
{
  int i,j,k,weightindex;
  struct node *node;

  /* Print a table of weights derivatives  with the rows being the 
        units in the from layer, and the columns being the units in 
        the to layer.  The last row is the bias */
  
  /* weightindex is the current weight in the network's weight vector */
  printf("Weight error table for %s:\n\n",net->ID);
  for(i=0;net->connectivity_vector[i].from_layer != 0;i++) {
    /* Print out column header */
    printf("          ");
    for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) {
      print_node_label(net,net->connectivity_vector[i].to_layer,k);
      printf("    ");
    }
    printf("\n");
    for(j=0;j<net->unitvector[net->connectivity_vector[i].from_layer-1];j++) {
      print_node_label(net,net->connectivity_vector[i].from_layer,j);
      printf(": ");
      /* Print the weights */
      for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) {
	for(weightindex= 
	    (node=
        get_node(net,net->connectivity_vector[i].to_layer,k))->first_weight_to;
	    net->nodesfrom[weightindex]!=
	       get_node_index(net,net->connectivity_vector[i].from_layer,j);
	    weightindex++);
	printf("%+ 9.4f ",net->wed[weightindex]); 
      } /* for k - to layer (column) loop */
      printf("\n");
    } /* for j - fromlayer (row) loop */
    printf("Bias  : ");
    for(k=0;k<net->unitvector[net->connectivity_vector[i].to_layer-1];k++) 
      printf("%+ 9.4f ",
        (get_node(net,net->connectivity_vector[i].to_layer,k))->bed);
    printf("\n\n");
  } /* for i - table loop */
}



void print_all_weights()
{
  int i;

  for (i=0;networks[i] !=NULL;i++) {
    print_net_weights(networks[i]);
  }
}


/* Generate cauchy distributed noise for the input or output layer */

float ts_noise(net)
struct netstruct *net;
{
  float r;

  r=(float)(rand()%1000)/1000.0;
  if (net->ts_noise_type==CAUCHY) {
    r=(r*PI)-(PI/2);
    return net->ts_noise_variance*tan(r);
  }
  else if (net->ts_noise_type==UNIFORM) {
    r=r-.5;
    return net->ts_noise_variance*r;
  }
}

/* Clamp the target output values on the output layer, and add noise */

void clamp_noisy_desired_outputs(netID)
char *netID;
{
  int i,j;
  struct netstruct *net;

  net=get_net(netID);

  for(j=0,i=MAXNODES-net->outputnodes;i<MAXNODES;i++,j++)
    net->nodes[i]->desired_output=ts.outpat[sim.current_pattern][j]+ts_noise(net);
}

/* Clamp the target output values on the output layer */

void clamp_desired_outputs(netID)
char *netID;
{
  int i,j;
  struct netstruct *net;

  net=get_net(netID);

  for(j=0,i=MAXNODES-net->outputnodes;i<MAXNODES;i++,j++)
    net->nodes[i]->desired_output=ts.outpat[sim.current_pattern][j];
}


/* Clamp the inpt pattern on the input layer */

void clamp_inputs(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);

  for(i=0;i<net->inputnodes;i++)
    net->nodes[i]->output=ts.inpat[sim.current_pattern][i];
}

void clamp_layers(netID)
char *netID;
{
  int i,layer,node;
  struct netstruct *net;

  net=get_net(netID);

  for(layer=0;layer<MAXLAYERS;layer++)
    if(ts.layer[layer]!=0) {
      for(node=0;node<ts.layer[layer];node++) {
	i=get_node_index(net,layer,node);
	net->nodes[i]->clamped=1;
	net->nodes[i]->desired_output=ts.layerpat[layer][sim.current_pattern][node];
      }
    }
}

void clamp_distributed_representations(netID,l)
char *netID;
int l;
{
  int i,node;
  struct netstruct *net;

  net=get_net(netID);

  for(node=0;node<net->unitvector[l];node++) {
    i=get_node_index(net,l,node);
    net->nodes[i]->clamped=1;
    net->nodes[i]->desired_output=net->layerpat_buff2[l][sim.current_pattern][node];
  }
}

/* Store a layer of activations in a temporary buffer */

void store_layer(netID,l)
char *netID;
{
  int i,layer,node;
  struct netstruct *net;
  
  net= get_net(netID);
  for(layer=0;net->unitvector[layer]!=0;layer++)
  for(node=0;node<net->unitvector[l];node++) {
    i=get_node_index(net,l,node);
    net->layerpat_buff1[l][sim.current_pattern][node]=net->nodes[i]->output;
  }

/**/printf("stored pat:\n");
/**/for(i=0;i<net->unitvector[l];i++)
    printf("%+4.2f ",net->layerpat_buff1[l][sim.current_pattern][i]);

}

/* Return 1 if the node index n is a "source node", defined by the mapping
 * table  */
int source(n)
int n;
{
  int i;

  for(i=0;i<mapnodes.lastrow;i++) {
    if (n==mapnodes.sourcenode[i])
      return 1;
  }
  return 0;
}

/* Return the index of a node's mapped node */
int get_mapped_node(n)
int n;
{
  int i;

  for(i=0;i<mapnodes.lastrow;i++) {
    if (n==mapnodes.sourcenode[i])
      return mapnodes.targetnode[i];
    else if (n==mapnodes.targetnode[i])
      return mapnodes.sourcenode[i];
  }
  return -1;
}


/* Return the index of an active input node for the current training pattern */
int active_input_node()
{
  int i;

  for (i=0;i<ts.in;i++)
    if (ts.inpat[sim.current_pattern][i]!=0.0)
      return i;
  return -1;
}


/* Retrieve the index of a training pattern which has a given input node
 * active */
int find_pattern_with_active_input_node(n)
int n;
{
  int p;

  for (p=0;p<MAXPATTERNS;p++) {
    if (ts.inpat[p][n]!=0.0)
      return p;
  }
  return -1;
}


/* "Map" distributed representations stored in the d.r. buffer by averaging
 * mapped distributed representations, and setting the domain switch to the
 * extreme value (1 for source, 0 for target */
void map_distributed_representations(netID,l)
char *netID;
int l;
{
  struct netstruct *net;
  int i,node,
      input_node,
      mapped_node,
      mapped_pattern;
 
  net= get_net(netID);

  /* Find the input node the current input nodes' mapped to then
     find a pattern which contains that input node */
  input_node=active_input_node();
  mapped_node=get_mapped_node(input_node);
  mapped_pattern=find_pattern_with_active_input_node(mapped_node);
/**/printf("pat=%d\n",sim.current_pattern);
/**/printf("before act:\n");
/**/for(i=0;i<net->unitvector[l];i++)
    printf("%+4.2f ",net->layerpat_buff1[l][sim.current_pattern][i]);
/**/printf("mapped pat=%d\n",mapped_pattern);
/**/for(i=0;i<net->unitvector[l];i++)
    printf("%+4.2f ",net->layerpat_buff1[l][mapped_pattern][i]);


  for (node=0;node<net->unitvector[l];node++) {
    if (node==net->domain_switch_node) {
      if (source(input_node))
	net->layerpat_buff2[l][sim.current_pattern][node]=1.0;
      else
	net->layerpat_buff2[l][sim.current_pattern][node]=0.0;
    }
    else {
      /* Average the current pattern's activation with the mapped pattern's 
	 activation */
      net->layerpat_buff2[l][sim.current_pattern][node]=
	(net->layerpat_buff1[l][sim.current_pattern][node]+
	 net->layerpat_buff1[l][mapped_pattern][node])/2.0;
    }
  }

/**/printf("After pattern:\n");
/**/for(i=0;i<net->unitvector[l];i++)
    printf("%+4.2f ",net->layerpat_buff2[l][sim.current_pattern][i]);

}


/* Linear threshold function */

float linear_threshold(x)
float x;
{
  float r;

  if (x>0.0)
    return 1.0;
  else if (x<0.0)
    return 0.0;
  else {      /* flip a coin to decide if the activaitons are equal */
    r=(rand()%10)/10.0;
    if (r>=.5)
      return 1.0;
    else 
      return 0.0;
  }
}


/* Symmetric sigmoid function, range = <-.5,+.5>  */

float symmetric_sigmoid(x)
float x;
{
  if (x < -15.0)
    return(-0.5);
  else if (x > 15.0)
    return(0.5);
  else
    return (1.0 /(1.0 + exp(-x)) - 0.5);
}


/* Logistic (Sigmoid)  function for output */

float logistic(x)
float x;
{
  return 1/(1+exp(-1*x));
}

/* Return hyperbolic arctangent error function or simple linear 
 * Error function
 */
float error_function(net,node)
struct netstruct *net;
struct node *node;
{
  float diff;

  diff=node->desired_output-node->output;
  if (net->hyperr_flag)
    if (diff<DIFFMIN)
      return HYPMIN;
    else if (diff>DIFFMAX)
      return HYPMAX;
    else
      return log((1.0+diff)/(1.0-diff));
  else if (net->cutoff_error_flag && !node->clamped) {
    if (node->desired_output > 0.5) {
      if (diff < net->cutoff_margin)
	return 0.0;
      else
	return diff;
    }
    else {
      if (diff > -net->cutoff_margin)
	return 0.0;
      else
	return diff;
    }
  }
  else
    return diff;
}

float output_function(net,x)
struct netstruct *net;
float x;
{
  if (net->linear_thresh_flag)
    return linear_threshold(x);
  else if (net->linear_flag)
    return x;
  else if (net->symmetric_sig_flag)
    return symmetric_sigmoid(x);
  else
    return logistic(x);
}


/*  Feedforward the activation from bottom (input layer) 
    Updated errors on output                                                  */

void sa_feedforward(netID)
char *netID;
{
  struct netstruct *net;
  int i,j;

  net=get_net(netID);


  /* Spread activation forward */
  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
    /* Update activations and output of the node only if it isn't clamped */
    net->nodes[i]->activation= ((globals.no_bias_flag || net->no_bias_flag) ? 0.0 : 
				  ((globals.no_output_bias_flag && 
				    i>=(MAXNODES-net->outputnodes)) ? 0.0 :
				   net->nodes[i]->bias_weight));
    for (j=net->nodes[i]->first_weight_to;
	 j<=net->nodes[i]->last_weight_to;
	 j++) {
      if (net->nodes[net->nodesfrom[j]]->clamped)
	net->nodes[i]->activation +=
	  net->weights[j]*net->nodes[net->nodesfrom[j]]->desired_output;
      else
	net->nodes[i]->activation +=
	  net->weights[j]*net->nodes[net->nodesfrom[j]]->output;
      net->nodes[i]->output=output_function(net,net->nodes[i]->activation);
    }
  }

  /* Update error on clamped nodes */
  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) 
    if (net->nodes[i]->clamped)
      net->nodes[i]->error=error_function(net,net->nodes[i]);
  
  /* Update output error */
  for (i=(MAXNODES-net->outputnodes);i<MAXNODES;skip_node_forward(&i,net)) 
    net->nodes[i]->error=error_function(net,net->nodes[i]);
}

	
/* Squared error of a node */

float square_error(node)
struct node *node;
{
  return ((node->error)*(node->error)) + globals.error_offset;
}

/* Scaled error */

float scaled_error(node)
struct node *node;
{
  float x;
 
  x=globals.error_scale*square_error(node);
  if (globals.error_threshold==0.0)
    return x;
  else
    return (x<globals.error_threshold?x:globals.error_threshold);
}

float abs_error(node)
struct node *node;
{
  return fabs(node->error) + globals.error_offset;
}

float invert_error(node)
struct node *node;
{
  return 1.0-fabs(node->error);
}
     
/* Return error depending on error function flags */
float error_fn(node)
{
  if (globals.scale_error_flag)
    return scaled_error(node);
  if (globals.abs_error_flag)
    return abs_error(node);
  if (globals.invert_error_flag)
    return invert_error(node);
  else
    return square_error(node);
}

/* Sum of the squared errors on the output */
					   
float sum_square_error(netID)
char *netID;
{
  int i;
  float sum;
  struct netstruct *net;

  net=get_net(netID);
  sum=0.0;
  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++)
    sum+=square_error(net->nodes[i]);

  return sum;
}


/* Print the output of a network */

void print_output(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++)
    printf("%4.2f ",net->nodes[i]->output);
  printf("\n");
}

/* Print the output of a network */

void print_desired_outputs(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++)
    printf("%4.2f ",net->nodes[i]->desired_output);
  printf("\n");
}

/* Print the output of a network */

void print_unsquared_error(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++)
    printf("%4.2f ",net->nodes[i]->error);
  printf("\n");
}


/* Find the state of the input layer that minimizes the error of the output
   layer */

void inner_loop(netID)
char *netID;
{
  float i,
         t,
         least_error,
         first_least_cause,
         last_least_cause,
         r,
         r2;
  struct netstruct *net;

  /* Initialize least_error to some high number so the least error is 
     initialized  to the first error */
  least_error=1000.0;
  first_least_cause=globals.inner_loop_min;
  last_least_cause=globals.inner_loop_min;
  net=get_net(netID);
  for(i=globals.inner_loop_min;i<=globals.inner_loop_max;i+=globals.inner_loop_increment) {
    net->nodes[0]->output=i;
    sa_feedforward(netID);
    if ((t=sum_square_error(netID))<=least_error) {
      if ((t=sum_square_error(netID))<least_error) {
	least_error=t;
	first_least_cause=i;
	last_least_cause=i;
      }
      else
	last_least_cause=i;
    }
  }

  r=0.0;
  r2=0.0;
/**/if(last_least_cause-first_least_cause>0.0) {
  r=((float)(rand()%(int)((last_least_cause-first_least_cause)*100.0))/100.0);
  if (globals.mid_il_plateau_range>0)
    r2=((float)(rand()%(int)(globals.mid_il_plateau_range*100.0))/100.0)-
      (globals.mid_il_plateau_range/2);
/**//*printf(" range=%6.4f  rnd=%6.4f ",last_least_cause-first_least_cause,r);*/
}
	  
  if (globals.mid_il_plateau_flag)
    net->nodes[0]->output=(last_least_cause+first_least_cause)/2 +r2;
  else if (globals.rand_il_plateau_flag)
    net->nodes[0]->output=first_least_cause+r;
  else
    net->nodes[0]->output=first_least_cause;
  sa_feedforward(netID);
}


/* Find the state of the input layer that minimizes the error of the output
   layer for each output item independently, then average the minimizing causes */

void avg_inner_loop(netID)
char *netID;
{
  float i,
        t,  /* This is the analytically derived best cause for linear output units only */
        least_cause[MAXOUTNODES],
        sum_cause;
  struct netstruct *net;
  int j,k;

  net=get_net(netID);

  /* Initialize least_error to some high number so the least error is 
     initialized  to the first error */
 
  for (k=0,j=MAXNODES-net->outputnodes;j<MAXNODES;j++,k++)
    least_cause[k]=(net->nodes[j]->desired_output-net->nodes[j]->bias_weight)/net->weights[net->nodes[j]->first_weight_to];
  for (j=0;j<net->outputnodes;j++)
    sum_cause+=least_cause[j];
  /* Average the three independent causes */
  net->nodes[0]->output=sum_cause/(float)net->outputnodes;
  sa_feedforward(netID);
}


float symmetric_sig_prime(net,output)
struct netstruct *net;
float output;
{
  return (0.25-output*output)+net->sigmoid_prime_offset;
}

float output_symmetric_sig_prime(net,output)
struct netstruct *net;
float output;
{
  return (0.25-output*output)+net->out_sigmoid_prime_offset;
}


float sigmoid_prime(net,output)
struct netstruct *net;
float output;
{
  return (output*(1.0-output))+net->sigmoid_prime_offset;
}

/* Previous routine: delete this 

float out_sigmoid_prime(net,output)
struct netstruct *net;
float output;
{
  return (output*(1.0-output))+net->sigmoid_prime_offset;
}

*/

float output_sigmoid_prime(net,output,error)
struct netstruct *net;
float output;
{
  float t;

  t=(output*(1.0-output));
  if (error>0.0 && t<net->out_sigmoid_prime_offset)
    return net->out_sigmoid_prime_offset;
  else
    return t;
}


float net_out_derivative(net,output)
struct netstruct *net;
float output;
{
  if (net->linear_flag)
    return 1.0;
  else if (net->symmetric_sig_flag)
    return symmetric_sig_prime(net,output);
  else
    return sigmoid_prime(net,output);
}

/* Special derivative function for output units */
float net_output_out_derivative(net,output,error)
struct netstruct *net;
float output;
{
  if (net->linear_flag)
    return 1.0;
  else if (net->symmetric_sig_flag)
    return output_symmetric_sig_prime(net,output);
  else
    return output_sigmoid_prime(net,output,error);
}


/* Back-Propogate deltas from the output layer to the lowest hidden layer */

void calculate_deltas(netID)
char *netID;
{
  int i,j;
  struct netstruct *net;

  net=get_net(netID);
  /* Initialize the error on the hidden nodes */
  for(i=after_innodes(net);i<MAXNODES-net->outputnodes;skip_node_forward(&i,net))
    if (!net->nodes[i]->clamped)
      net->nodes[i]->error=0.0;
  /* Calculate the delta: f'*error */
  for(i=MAXNODES-1;i>=net->inputnodes;skip_node_backward(&i,net)) {
    if (i>=MAXNODES-net->outputnodes) 
      net->nodes[i]->delta=net->nodes[i]->error*net_output_out_derivative(net,net->nodes[i]->output,net->nodes[i]->error);
    else
      net->nodes[i]->delta=net->nodes[i]->error*net_out_derivative(net,net->nodes[i]->output);
    /* sum up them errors */
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
      if (!net->nodes[net->nodesfrom[j]]->clamped)
	net->nodes[net->nodesfrom[j]]->error += 
	  net->weights[j]*net->nodes[i]->delta;
    }
  }
}

/* Compute how much to adjust each weight based on the deltas and outputs */

void calculate_weight_changes(netID)
char *netID;
{
  struct netstruct *net;
  int i,j;

  net=get_net(netID);

  /* Reset weight error derivatives depending on the learning-grain flag
     and whether it is the beginning of a cycle */
  if ((net->epoch_learn_flag && sim.current_pattern==0) || 
      (!net->epoch_learn_flag)) {
    for(i=MAXNODES-1;i>=net->inputnodes;skip_node_backward(&i,net)) {
      net->nodes[i]->prevbed=net->nodes[i]->bed;
/**//*      if (net->quickprop_flag) { */
	net->nodes[i]->bed=net->decay*net->nodes[i]->bias_weight;
/*      }
      else
	net->nodes[i]->bed=0.0; */
      for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
	net->prevwed[j]=net->wed[j];
/**//*	if (net->quickprop_flag) */
	net->wed[j]=net->decay*net->weights[j];
/**//*	else
	  net->wed[j]=0.0; */
      }
    }
  }

  for(i=MAXNODES-1;i>=net->inputnodes;skip_node_backward(&i,net)) {
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
      if (net->nodes[net->nodesfrom[j]]->clamped)
	net->wed[j] +=
	  net->nodes[i]->delta*net->nodes[net->nodesfrom[j]]->desired_output;
      else
	net->wed[j]+=net->nodes[i]->delta*net->nodes[net->nodesfrom[j]]->output;
    }
    /* Bias change is a special case: like a link from node w/output = 1 */
    net->nodes[i]->bed+=((globals.no_bias_flag || net->no_bias_flag) ? 0.0 : 
			 ((globals.no_output_bias_flag && 
			   i>=(MAXNODES-net->outputnodes)) ? 0.0 :
			  net->nodes[i]->delta));
  }
}

void reset_errors(net)
struct netstruct *net;
{

  if (sim.dn_cc_trigger_flag)
    sim.dn_true_error=0.0;
  net->true_error=0.0;
  net->errorbits=0.0;
  net->sum_sq_error=0.0;
  net->sum_error=0.0;
  net->avg_error=0.0;

}

void reset_error_derivatives(net) 
struct netstruct *net;
{
  int i,j,k;

  /* Note: this should add in a decay term if using quickprop */
  for(j=MAXNODES-net->outputnodes;j<MAXNODES;j++) {
    net->nodes[j]->prevbed=net->nodes[j]->bed;
/**//*printf(" $$setting prevbed=%6.4f ",net->nodes[j]->prevbed);*/
    if (net->quickprop_flag) 
/**/{
      net->nodes[j]->bed=net->decay*net->nodes[j]->bias_weight;
/**//*if(!strncmp(net->ID,"TN",2)) printf(" !os decay! ");*/
      }
    else
      net->nodes[j]->bed=0.0;
    for(k=net->nodes[j]->first_weight_to;
	k<=net->nodes[j]->last_weight_to;k++) {
      net->prevwed[k]=net->wed[k];
/**//*    if (net->quickprop_flag) */
      net->wed[k]=net->decay*net->weights[k];
/*    else
      net->wed[k]=0.0; */
    }
  }
}


/* Only calculate the weight changes for the bias terms, not for the weights */

void calculate_bias_changes(netID)
char *netID;
{
  struct netstruct *net;
  int i,j;

  net=get_net(netID);

  /* Reset weight error derivatives depending on the learning-grain flag
     and whether it is the beginning of a cycle */
  if ((net->epoch_learn_flag && sim.current_pattern==0) || 
      (!net->epoch_learn_flag))
    reset_error_derivatives(net);
  for(i=MAXNODES-1;i>=net->inputnodes;skip_node_backward(&i,net)) {
    /* Bias change is a special case: like a link from node w/output = 1 */
    net->nodes[i]->bed+=((globals.no_bias_flag || net->no_bias_flag) ? 0.0 : 
			 ((globals.no_output_bias_flag && 
			   i>=(MAXNODES-net->outputnodes)) ? 0.0 :
			  net->nodes[i]->delta));
  }
/**//*printf("cbc bed = %10.8f\n",net->nodes[MAXNODES-1]->bed);*/
}

update_variance_scale(net)
struct netstruct *net;
{
  if (net->anneal_type==UNIFORM) {
    net->variance_scale=net->variance_scale-net->anneal_rate;
    if (net->variance_scale<0.0)
      net->variance_scale=0.0;
  }
  else if (net->anneal_type==CAUCHY) {
    net->variance_scale+=1.0;
  }
}

/* Compute a random amount to vary the weight by based on annealing parameters */
float anneal_epsilon(net,current_weight_or_node,object_type)
struct netstruct *net;
int current_weight_or_node;
int object_type;
{
  float r,
         deltaw,
         ealpha,
         ebeta,
         c,
         x;


  if (net->anneal_flag) {
    if (net->anneal_type==UNIFORM) {
      if (net->variance_scale*net->anneal_variance>0.0) {
	r=((float)(rand()%
		    (int)((net->anneal_variance*net->variance_scale)*100.0))/100.0);
	return r-(net->anneal_variance*net->variance_scale)/2;
      }
      else
	return 0.0;
    }
    else if (net->anneal_type==CAUCHY) {
      /* Generate a potential weight change based on cauchy distribution       */
      r=(PI-.01)*((float)(rand()%100)/100.0)-(PI-.01)/2;
      deltaw=(net->anneal_variance/net->variance_scale)*tan(r);
      /* Filter the potential weight change based on Boltzmann distribution    */
      /* Probability of accepting a weight change based on change in the error */
      /* function: P(c)=exp(-c/t), c=E(alpha)-E(beta)                          */
      if (object_type==WEIGHT) {
/**//*printf("\nwed[%d]=%10.8f ",current_weight_or_node,net->wed[current_weight_or_node]);*/
	c=net->wed[current_weight_or_node]*deltaw;
  }
      else {
/**//*printf("\nwed[%d]=%10.8f ",current_weight_or_node,net->nodes[current_weight_or_node]->bed);*/
	c=net->nodes[current_weight_or_node]->bed*deltaw;
  }
 /**//*if (!strncmp(net->ID,"TN",2))
	printf("r=%6.4f deltaw[%d]=%6.4f, c=%6.4f, t=%6.4f, p=%6.4f",*
	     r,current_weight_or_node,deltaw,c,
	     net->anneal_variance/net->variance_scale,
	     exp(c/(net->anneal_variance*net->boltz_cons/net->variance_scale))); */
      r=(float)(rand()%100)/100.0;
      if (exp(c/(net->anneal_variance*net->boltz_cons/net->variance_scale))>r)
	return deltaw;
      else
	return 0.0;
    }
  }
  else  /* not net->anneal_flag */
    return 0.0;
}


/* This is the standard backprop weight updating procedure */

void change_vanilla_bp_weights(net)
struct netstruct *net;
{
  int i,j;
  float t1,t2,r;

  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
      if (!net->freezetag[j]) {
	r=anneal_epsilon(net,j,WEIGHT);
	/* Set delta weight, and adjust the weight itself */
	if ((net->gamma!=0.0) && (net->wed[j]<0.0)) {
	   /* Special learning rate for nonreinforced learning */
	    net->dweights[j]=net->wed[j]*net->gamma + net->alpha*net->dweights[j];
        }
	else
	  net->dweights[j]=net->wed[j]*net->eta + net->alpha*net->dweights[j];
	net->weights[j]+=net->grad_coeff*net->dweights[j]+net->anneal_coeff*r;
      }
    }
    if (!net->nodes[i]->freezetag) {
      t1=net->nodes[i]->bed*net->eta + net->nodes[i]->dbias*net->alpha;
      t2=net->nodes[i]->dbias;
      net->nodes[i]->dbias=t1;
      r=anneal_epsilon(net,i,BIAS);
      net->nodes[i]->bias_weight+=net->grad_coeff*t1+net->anneal_coeff*r;
      net->accumweightchange+=fabs(t1+t2);
    }
  }
  update_variance_scale(net);
}


void change_quickprop_weights(net)
struct netstruct *net;
{
  int i,j;
  float next_step, shrink_factor,r;

  shrink_factor = net->mu/(1.0+net->mu);
  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
      next_step=0.0;
      if (net->dweights[j]>net->mode_switch_threshold) {
	/* Last step was significantly positive */
	/* Add in eta if postive slope */
	if (net->wed[j]>0.0)
	  next_step += net->eta*net->wed[j];
	if (net->wed[j] > shrink_factor * net->prevwed[j])
	  next_step += (net->mu*net->dweights[j]);
	else   /* Use quadratic estimate */
	  next_step += ((net->wed[j]/(net->prevwed[j]-net->wed[j])) 
			* net->dweights[j]);
      }
      else if (net->dweights[j]< -net->mode_switch_threshold) {
	/* Last step was significantly negative */
	/* Add in eta if negative slope */
	if (net->wed[j]<0.0)
	  next_step += net->eta*net->wed[j];
	/* If slope < (or close to) prev slope, take max step size */
	if (net->wed[j] < (shrink_factor*net->prevwed[j]))
	  next_step += (net->mu*net->dweights[j]);
	else
	  next_step += ((net->wed[j]/(net->prevwed[j]-net->wed[j])) 
			* net->dweights[j]);
      }
      else /* normal gradient descent */ {
	sim.did_gradient++;
	next_step += net->eta*net->wed[j] + net->alpha*net->dweights[j];
      }

      r=anneal_epsilon(net,j,WEIGHT);
      /* Set delta weight, and adjust the weight itself */
      if (!net->freezetag[j]) {
	net->dweights[j]=next_step;
	net->weights[j]+=net->grad_coeff*next_step+net->anneal_coeff*r;
      }
    }
    /* Do the same thing for the bias as we did for the weights */
    next_step=0.0;
    if (net->nodes[i]->dbias>net->mode_switch_threshold) {
      /* Last step was significantly positive */
      /* Add in eta if postive slope */
      if (net->nodes[i]->bed>0.0) {
	next_step += net->eta*net->nodes[i]->bed;
      }
	if (net->nodes[i]->bed > shrink_factor * net->nodes[i]->prevbed) {
	  next_step += (net->mu*net->nodes[i]->dbias);
	}
	else   /* Use quadratic estimate */ {
	  next_step += ((net->nodes[i]->bed/
			 (net->nodes[i]->prevbed-net->nodes[i]->bed))
			* net->nodes[i]->dbias);
	}
      }
      else if (net->nodes[i]->dbias< -net->mode_switch_threshold) {
	/* Last step was significantly negative */
	/* Add in eta if negative slope */
	if (net->nodes[i]->bed<0.0) {
	  next_step += net->eta*net->nodes[i]->bed;
	}
	/* If slope < (or close to) prev slope, take max step size */
	if (net->nodes[i]->bed < (shrink_factor*net->nodes[i]->prevbed)) {
	  next_step += (net->mu*net->nodes[i]->dbias);
	}
	else {
	  next_step += ((net->nodes[i]->bed/(net->nodes[i]->prevbed-net->nodes[i]->bed)) 
			* net->nodes[i]->dbias);
	}
      }
      else /* normal gradient descent */ {
	sim.did_gradient++;
	next_step += net->eta*net->nodes[i]->bed + net->alpha*net->nodes[i]->dbias;
      }
      r=anneal_epsilon(net,i,BIAS);
      net->nodes[i]->dbias=next_step;
      net->nodes[i]->bias_weight+=net->grad_coeff*next_step+net->anneal_coeff*r;
  }
  /* Update schedule for annealing */
  update_variance_scale(net);
}

/* Generate an estimate of the optimal weight for a given weight and pattern
   based on the quickprop method                                              */
float weight_estimate(net,weightindex)
struct netstruct *net;
int weightindex;
{
  float wed1,wed2,curv;

  /* estimate the weight-error curvature */
  wed1=net->wed[weightindex];
  net->weights[weightindex]+=QPEPSILON;
  sa_feedforward(net);
  calculate_deltas(net);
  calculate_weight_changes(net);
  wed2=net->wed[weightindex];
  curv=(wed2-wed1)/QPEPSILON;
  net->weights[weightindex]-=QPEPSILON;
/**/printf("\ncurv[%d]=%8.6f\n",weightindex,curv);

  if (curv>-(net->curvature_threshold)) {
    if (curv>net->curvature_threshold) {
      /* Generate the quadratic estimate and estimate a jump in the opposite
	 direction */
/**/printf("neg quad[%d]=%8.6f\n",weightindex,(wed2/curv));
      return (wed2/curv);
    }
    else {
      /* Slope is too close to 0: just do a gradient descent */
/**//*printf("grad descent[%d]=%8.6f\n",weightindex,net->weights[weightindex]+wed1*net->eta);*/
      return net->weights[weightindex]+wed1*net->eta;
    }
  }
  else {
    /* Generate a reguluar quadratic estimate */
/**//*printf("qud[%d]=%8.6f\n",weightindex,-(wed2/curv));*/
    return -(wed2/curv);
  }
}
  
  
/* Generate an estimate of the optimal bias weight for a given weight and
   pattern based on the quickprop method                                     */
float bias_estimate(net,nodeindex)
struct netstruct *net;
int nodeindex;
{
  float bed1,bed2,curv;

  /* estimate the weight-error curvature */
  bed1=net->nodes[nodeindex]->bed;
  net->nodes[nodeindex]->bias_weight+=QPEPSILON;
  sa_feedforward(net);
  calculate_deltas(net);
  calculate_weight_changes(net);
  bed2=net->nodes[nodeindex]->bed;
  curv=(bed2-bed1)/QPEPSILON;
  net->nodes[nodeindex]->bias_weight-=QPEPSILON;
/**//*printf("bias curv[%d]=%8.6f\n",nodeindex,curv);*/

  if (curv>-(net->curvature_threshold)) {
    if (curv>net->curvature_threshold) {
      /* Generate the quadratic estimate and estimate a jump in the opposite
	 direction */
/**//*printf("bias neg quad[%d]=%8.6f\n",nodeindex,(bed2/curv));*/
      return (bed2/curv);
    }
    else {
      /* Slope is too close to 0: just do a gradient descent */
/**//*printf("bias grad descent[%d]=%8.6f\n",nodeindex,net->nodes[nodeindex]->bias_weight+bed1*net->eta);*/
      return net->nodes[nodeindex]->bias_weight+bed1*net->eta;
    }
  }
  else {
    /* Generate a reguluar quadratic estimate */
/**//*printf("bias quad[%d]=%8.6f\n",nodeindex,-(bed2/curv));*/
    return -(bed2/curv);
  }
}


/* Average the estimates for a given weight */
float avg_weight_estimate(net,weightindex)
struct netstruct *net;
int weightindex;
{
  int i;
  float sum;

  sum=0.0;
  for(i=0;i<net->estimate_number;i++)
    sum+=net->weightest[weightindex][i];
/**//*printf("avg weight estimate [%d]=%8.6f\n",weightindex,sum/net->estimate_number);*/
  return sum/net->estimate_number;
}

/* Average the estimates for a given bias weight */
float avg_bias_estimate(net,nodeindex)
struct netstruct *net;
int nodeindex;
{ 
  int i;
  float sum;

  sum=0.0;
  for(i=0;i<MAXESTIMATES;i++)
    sum+=net->nodes[nodeindex]->bias_est[i];
/**//*printf("avg bias estimate [%d]=%8.6f\n",nodeindex,sum/net->estimate_number);*/
 return sum/net->estimate_number;
}
		       

/* Pattern-by-pattern quickprop procedure: Generate estimates for the optimal
   weights for each trial, and after a fixed number of trials average the 
   estimates                                                                  */
void change_quickprop_pattern_weights(net)
struct netstruct *net;
{
  int i,j;

  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
      net->weightest[j][net->estimate_counter]=weight_estimate(net,j);
    }
    net->nodes[i]->bias_est[net->estimate_counter]=bias_estimate(net,i);
  }
  net->estimate_counter++;

  /* Average the estimate and update the weights if we've hit the fixed # of
     esimate */
  if(net->estimate_counter==net->estimate_number) {
    net->estimate_counter=0;
    for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
      for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) {
	net->weights[j]=avg_weight_estimate(net,j);
      }
      net->nodes[i]->bias_weight=avg_bias_estimate(net,i);
    }
  }
}


/* average corresponding weights */

void average_mapped_weights(net)
struct netstruct *net;
{
  int i,n,node,w,
      mapped_node,
      mapped_weight;

  for (node=0;node<net->unitvector[net->distributed_layer];node++) {
    n=get_node_index(net,net->distributed_layer,node);
    net->nodes[n]->bias_weight=0.0;
    for (w=net->nodes[n]->first_weight_to;w<=net->nodes[n]->last_weight_to;w++) {
      if (node==net->domain_switch_node) {
	if (source(net->nodesfrom[w]))
	  net->weights[w]=10.0;
	else
	  net->weights[w]= -10.0;
      }
      else {
        /* Average the current weight with it's mapped weight */
	mapped_node=get_mapped_node(net->nodesfrom[w]);
	mapped_weight=get_weight_index(net,n,mapped_node);
	net->weights[w]=(net->weights[w]+net->weights[mapped_weight])/2.0;
	net->weights[mapped_weight]=net->weights[w];
      }
    }
  }
}


/* average corresponding weight error derivatives */

void average_mapped_weds(net)
struct netstruct *net;
{
  int i,n,node,w,
      mapped_node,
      mapped_weight;

  for (node=0;node<net->unitvector[net->distributed_layer];node++) {
    n=get_node_index(net,net->distributed_layer,node);
    for (w=net->nodes[n]->first_weight_to;w<=net->nodes[n]->last_weight_to;w++) {
      if (node==net->domain_switch_node) {
	if (source(net->nodesfrom[w]))
	  net->wed[w]= .001;
	else
	  net->wed[w]= -.001;
      }
      else {
        /* Average the current weight with it's mapped weight */
	mapped_node=get_mapped_node(net->nodesfrom[w]);
	mapped_weight=get_weight_index(net,n,mapped_node);
	net->wed[w]=(net->wed[w]+net->wed[mapped_weight])/2.0;
	net->wed[mapped_weight]=net->wed[w];
      }
    }
  }
}

/* average corresponding weight error derivatives */

void average_penultimate_mapped_weds(net)
struct netstruct *net;
{
  int i,n,mn,node,w,
      mapped_node,
      mapped_weight,
      mwo;  /* mapped weight offset */
      
  for (node=0;node<net->unitvector[MAXLAYERS-1];node++) {
    n=get_node_index(net,MAXLAYERS,node);
    mapped_node=get_mapped_node(node);
    mn=get_node_index(net,MAXLAYERS,mapped_node);
    /* Average the bias derivatives of the mapped nodes */
/*    net->nodes[n]->bed=(net->nodes[n]->bed+net->nodes[mn]->bed)/2.0;
    net->nodes[mn]->bed=net->nodes[n]->bed;   */
    for (mwo=0,w=net->nodes[n]->first_weight_to;w<=net->nodes[n]->last_weight_to;mwo++,w++) {
      /* Average the current weight with it's mapped weight */
      mapped_weight=net->nodes[mn]->first_weight_to+mwo;
      net->wed[w]=(net->wed[w]+net->wed[mapped_weight])/2.0;
      net->wed[mapped_weight]=net->wed[w];
    }
  }
}


/* Change weights based on the computed weight change, and the previous 
   weight change                                                              */

void change_weights(netID)
char *netID;
{
  struct netstruct *net;
  int i;

  net=get_net(netID);

  /* skip if lflag is set to 0 */
  if (net->lflag) {
    /* If learning by epochs (cycles), don't adjust the weights unless on the last 
       pattern */  
    if ((net->epoch_learn_flag && sim.current_pattern==ts.tsnum-1)
	|| (!net->epoch_learn_flag)) {
      if (net->avg_mapped_weds_flag)
	average_mapped_weds(net);
      if (net->avg_pen_mapped_weds_flag)
	average_penultimate_mapped_weds(net);
      if (net->quickprop_flag) {
	change_quickprop_weights(net);
      }
      else
	change_vanilla_bp_weights(net);
      if (net->avg_mapped_weights_flag)
	average_mapped_weights(net);
    }
  }
}


/* Used for negative tn learning with revision on output bias only */
void change_out_bias(netID)
char *netID;
{
  struct netstruct *net;
  int i;

  net=get_net(netID);

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) {
    /* Feedforward from input output bias terms only */
    net->nodes[i]->output=logistic(net->nodes[i]->bias_weight);
    /* Calculate error and delta */
    net->nodes[i]->error=net->nodes[i]->desired_output-net->nodes[i]->output;
    net->nodes[i]->delta=net->nodes[i]->error*net->nodes[i]->output*
                         (1.0-net->nodes[i]->output);
    /* Compute the bias error derivative, the bias change, and the new bias
         weight */
    net->nodes[i]->bed=net->nodes[i]->delta;
    net->nodes[i]->dbias=net->nodes[i]->bed*net->eta + 
                         net->nodes[i]->dbias*net->alpha;
    net->nodes[i]->bias_weight+=net->nodes[i]->dbias;
  }
}


/* Used for negative tn learning with revision on bias but not weights */
void change_bias_freeze_wts(netID)
char *netID;
{
  struct netstruct *net;
  int i;
  float t1,t2;

  net=get_net(netID);
  
  for(i=after_innodes(net);i<MAXNODES;skip_node_forward(&i,net)) {
    t1=net->nodes[i]->bed*net->eta + net->nodes[i]->dbias*net->alpha;
    t2=net->nodes[i]->dbias;
    net->nodes[i]->dbias=t1;
    net->nodes[i]->bias_weight+=t1;
    net->accumweightchange+=fabs(t1+t2);
  }
}


/******************************************************************************/
/*                                                                            */
/*   Cascade Correlation Routines                                             */
/*                                                                            */
/******************************************************************************/

int node_passed_threshold(net,node)
struct node *node;
struct netstruct *net;
{
  if(fabs(node->output-node->desired_output) < net->error_margin)
    return 1;
  else
    return 0;
}

/* Update the error measures for the settling criteria, and store the
 * errors (actually the deltas) in the error cache
 */
void update_errors(net)
struct netstruct *net;
{
  int i;

  if (sim.dn_cc_trigger_flag) {
    sim.dn_true_error+=sum_square_error("YN")+sum_square_error("NN");
/**/printf(" dn err: %6.4f ",sim.dn_true_error);
  }
  net->true_error+=sum_square_error(net->ID);
  for (i=MAXNODES-net->outputnodes;i<MAXNODES;i++) {
    if(!node_passed_threshold(net,net->nodes[i]))
      net->errorbits++;
    /* NOTE: the stored error is the NEGATIVE delta */
    net->errorcache[i-(MAXNODES-net->outputnodes)][sim.current_pattern]=
      -net->nodes[i]->delta;
    net->sum_sq_error+=net->nodes[i]->delta*net->nodes[i]->delta;
    net->sum_error+=net->nodes[i]->delta;
  }
}


/* Freeze all the weights in the network */
void freeze_all_weights_and_nodes(net)
struct netstruct *net;
{
  int i;

  for(i=0;i<MAXWEIGHTS;i++) {
    net->freezetag[i]=1;
  }
  for(i=0;i<MAXNODES;i++) {
    if(net->nodes[i]!=NULL) {
      net->nodes[i]->freezetag=1;
    }
  }
}


/* Freeze all but the weights going to output nodes */
freeze_cand_weights(net)
struct netstruct *net;
{
  int i,j;

  freeze_all_weights_and_nodes(net);
  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) { 
    if(net->nodes[i]!=NULL) {
      net->nodes[i]->freezetag=0;
      for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++) 
	net->freezetag[j]=0;
    }
  }
}

/* Update the slopes of the output weights.  This is a horrible kluge that I
   am utterly ashamed of.  Since EPOCH-FLAG is N for cascade correlation, to
   facilitate the candidate phase, we must accumulate slopes by this method:
   Sum up the errocache multiplied the input values */
void update_output_slopes(net)
struct netstruct *net;
{
  int i,j;
  
  if ((sim.current_pattern==0 && net->epoch_learn_flag) || !net->epoch_learn_flag) 
    reset_error_derivatives(net);

  /* Since the errorcache stores the negative delta, we must reverse it again */
  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) {
    net->nodes[i]->bed+=-net->errorcache[i-(MAXNODES-net->outputnodes)][sim.current_pattern];
    for(j=net->nodes[i]->first_weight_to;j<=net->nodes[i]->last_weight_to;j++)
      net->wed[j]+=net->nodes[net->nodesfrom[j]]->output*
	-net->errorcache[i-(MAXNODES-net->outputnodes)][sim.current_pattern];
  }
/**/printf(" osbed[999]=%10.8f\n",net->nodes[MAXNODES-1]->bed);
}

/* Train the output weights.  If we exhaust max_epochs, stop with value
 * TIMEOUT.  If there are zero error bits, stop with value WIN.  Else,
 * keep going until the true error has changed by a significant amount,
 * and then until it does not change significantly for Patience epochs.
 * Then return STAGNANT.  If Patience is zero, we do not stop until victory
 * or until max_epochs is used up.  -RSC
 */
int train_outputs(net)
struct netstruct *net;
{
  float tempeta;

  if (sim.current_pattern==0)
    reset_errors(net);
  sa_feedforward(net->ID);
  calculate_deltas(net->ID);
  update_errors(net);
  update_output_slopes(net);
  change_weights(net);

  /* Only do this on the last pattern of the current epoch/cycle */
  if (sim.current_pattern==ts.tsnum-1) {
/**/printf(" TN err: %6.4f  DN err: %6.4f ",net->true_error,sim.dn_true_error);
    /* Divide learning rate by the # of cases: this is done in original qp, but
       it is commented out here for consistency with normal learning in epoch mode */
    tempeta=net->eta;
    /* net->eta=net->eta/(ts.tsnum); */
    net->eta=tempeta;
    if (net->errorbits == 0)
      return WIN;
    else if (net->output_patience ==0)
      return CONTINUE;
    else if (net->first_out_epoch) {
      net->first_out_epoch=FALSE;
      sim.dn_last_error=sim.dn_true_error;
      net->last_error=net->true_error;
      net->timeout_epoch=sim.current_cycle+net->output_timeout;
      net->quit_epoch=sim.current_cycle+net->output_patience;
      return CONTINUE;
    }
    /* Change is too big, so keep pushing the quit epoch */
    else if (((sim.dn_cc_trigger_flag) && 
	      ((fabs(sim.dn_true_error-sim.dn_last_error) > 
		(sim.dn_last_error*net->output_change_threshold)))) ||
	     ((!sim.dn_cc_trigger_flag) && 
	      ((fabs(net->true_error-net->last_error) > 
		(net->last_error*net->output_change_threshold))))) {
/**/printf(" !push quit ");
      sim.dn_last_error=sim.dn_true_error;
      net->last_error=net->true_error;
      net->true_error=0.0;
      net->quit_epoch=sim.current_cycle + net->output_patience;
      return CONTINUE;
    }
    else if (sim.current_cycle>=net->quit_epoch) {
      return STAGNANT;
    }
    else if (sim.current_cycle>=net->timeout_epoch) {
      return TIMEOUT;
    }
    else {
      /* Change is under the threshold, so just continue w/o pushing quit epoch */
      sim.dn_last_error=sim.dn_true_error;
      net->last_error=net->true_error;
      return CONTINUE;
    }
  }
  else
    return CONTINUE;
}

void insert_candidate_weight(net,node,nodefrom)
struct netstruct *net;
struct node *node;
int nodefrom;
{
  if(net->outputweights+net->nonoutputweights>MAXWEIGHTS-2) {
    printf("**Error: Out of weight memory while creating candidate\n");
    return;
  }
  else {
   net->weights[net->nonoutputweights]=get_weight();
    net->nodesfrom[net->nonoutputweights]=nodefrom;
    net->freezetag[net->nonoutputweights]=0;
    node->last_weight_to++;
    net->nonoutputweights++;
  }
}

insert_candidate_node(net,node)
struct netstruct *net;
struct node *node;
{
  if (net->totalnodes>=MAXNODES+2) {
    printf("**Error: out of node memory while creating candidate node\n");
    return;
  }
  net->nodes[net->totalnodes-net->outputnodes]=node;
  net->totalnodes++;
  net->hiddennodes++;
}

/* Create a battery of candidate nodes, and link them into the network
 * structure
 */
void create_candidates(net)
struct netstruct *net;
{
  int i,j,current_layer,current_connectivity;
  struct node *newnode;
  
  /* All weights are to be frozen except those from the input nodes to the
     candidate nodes */
  freeze_all_weights_and_nodes(net);

  /* Remember where in the list we started creating these suckers */
  net->first_candidate=net->totalnodes-net->outputnodes;
  net->first_candidate_weight=net->nonoutputweights;
  for(i=0;i<net->num_candidates;i++) {
    if ((newnode=(struct node *)malloc(sizeof(struct node)))==NULL) {
      printf("**Error: Out of memory while creating candidate node\n");
      return;
    }
    init_node(newnode,net);
    newnode->first_weight_to=net->nonoutputweights;
    newnode->last_weight_to=net->nonoutputweights-1;
    newnode->bias_weight=get_weight();;
  
    /* Link the new node to the rest of the network */
    for(j=0;j<net->first_candidate;j++)
      insert_candidate_weight(net,newnode,j);
    insert_candidate_node(net,newnode);
  }
  /* Note that all previous layers are connected to this layer of candidates 
     with the unitvector and connectivity structures */
  for(i=0;net->unitvector[i]!=0;i++);
  current_layer=i;
  net->unitvector[current_layer]=net->num_candidates;
  for(j=0,current_connectivity=i;j<current_layer;j++,current_connectivity++) {
    net->connectivity_vector[current_connectivity].from_layer=j+1;
    net->connectivity_vector[current_connectivity].to_layer=current_layer+1;
  }
}

/* Copy the best candidate to the position of the first candidate, then 
 * wipe out the rest of the candidates.
 */
void select_candidate(net)
struct netstruct *net;
{
  int i,j,o,current_layer;
  float cpc;

  net->nodes[net->first_candidate]->bias_weight=
    net->nodes[net->best_candidate]->bias_weight;
  for(i=0;i<net->first_candidate;i++) {
    net->weights[i+net->first_candidate_weight]=
      net->weights[i+net->nodes[net->best_candidate]->first_weight_to];
    net->wed[i+net->first_candidate_weight]=
      net->wed[i+net->nodes[net->best_candidate]->first_weight_to];
    net->prevwed[i+net->first_candidate_weight]=
      net->prevwed[i+net->nodes[net->best_candidate]->first_weight_to];
  }

  for(i=net->first_candidate_weight+net->first_candidate;i<net->nonoutputweights;i++) {
    net->nodesfrom[i]=0;
    net->weights[i]=0.0;
    net->dweights[i]=0.0;
    net->wed[i]=0.0;
    net->prevwed[i]=0.0;
    net->freezetag[i]=1;
  }

  for(i=net->first_candidate+1;i<net->totalnodes-net->outputnodes;i++) {
    free(net->nodes[i]);
    net->nodes[i]=NULL;
  }

  net->nonoutputweights=net->first_candidate_weight+net->first_candidate;
  net->totalnodes=(net->first_candidate+1)+net->outputnodes;
  net->hiddennodes=net->totalnodes-(net->inputnodes+net->outputnodes);

  /* Update the unitvector structure, so there's only one unit in the layer*/
  for(i=0;net->unitvector[i]!=0;i++);
  current_layer=i;
  net->unitvector[current_layer-1]=1;

  /* Update the connectivity structure, so there's links from the new hidden
     layer to the output layer */
  for(i=0;net->connectivity_vector[i].from_layer!=0;i++);
  net->connectivity_vector[i].from_layer=current_layer;
  net->connectivity_vector[i].to_layer=MAXLAYERS;

  /* Add links from the candidate node to the output nodes */
  for(o=MAXNODES-net->outputnodes;o<MAXNODES;o++) {
    cpc=net->CandPrevCor[net->best_candidate][o-(MAXNODES-net->outputnodes)];
/**/printf(" !!cpc=%8.6f, cc=%8.6f\n",cpc,net->CandCor[net->best_candidate][o-(MAXNODES-net->outputnodes)]);
    insert_cand_to_out_weight(net,net->first_candidate,net->nodes[o],net->weight_multiplier*cpc);
  }
}


/* Note: Ideally, after each adjustment of the candidate weights, we would  */
/* run two epochs.  The first would just determine the correlations         */
/* between the candidate unit outputs and the residual error.  Then, in a   */
/* second pass, we would adjust each candidate's input weights so as to     */
/* maximize the absolute value of the correlation.  We need to know the     */
/* direction to tune the input weights.                                     */
/*                                                                          */
/* Since this ideal method floats the number of epochs required for        */
/* training candidates, we cheat slightly and use the correlation values    */
/* computed BEFORE the most recent weight update.  This combines the two    */
/* epochs, saving us almost a factor of two.  To bootstrap the process, we  */
/* begin with a single epoch that computes only the correlation.            */
/*                                                                          */
/* Since we look only at the sign of the correlation after the first ideal  */
/* epoch and since that sign should change very infrequently, this probably */
/* is OK.  But keep a lookout for pathological situations in which this     */
/* might cause oscillation.                                                 */
/* -RSC                                                                     */

/* After the correlations have been computed, we do a second pass over
 * the training set and adjust the input weights of all candidate units.
 */

void compute_correlation_slopes(net)
struct netstruct *net;
{
  int i,j,o,u,fw;
  float sum,
        value,
        actprime,
        error,
        direction,
        change;

  sum=0.0;
  value=0.0;
  actprime=0.0;
  direction=0.0;
  for(u=net->first_candidate;u<net->first_candidate+net->num_candidates;u++) {
    change=0.0;
    value=net->nodes[u]->output;
    actprime=net_out_derivative(net,net->nodes[u]->output);
    for(o=0;o<net->outputnodes;o++) {
      error=net->errorcache[o][sim.current_pattern];
      if (net->CandPrevCor[u][o]<0.0)
	direction= -1.0;
      else
	direction=1.0;
      change -= direction*actprime*((error-net->avg_error)/net->sum_sq_error);
      net->CandCor[u][o]+=error*value;
    }
    if(sim.current_pattern==0)
      net->nodes[u]->bed= -change;
    else
      net->nodes[u]->bed-=change;
    fw=net->nodes[u]->first_weight_to;
    for(i=fw;i<fw+net->first_candidate;i++) {
      if(sim.current_pattern==0)
	net->wed[i]= -change*net->nodes[net->nodesfrom[i]]->output;
      else
	net->wed[i]-=change*net->nodes[net->nodesfrom[i]]->output;
    }
  }
}

/* NORMALIZE each accumulated correlation value, and stuff the normalized
 * form into the CandPrevCor data structure.  Then zero CandCor to
 * prepare for the next round.  Note the unit with the best total
 * correlation score.
 * -RSC
 * Also, store the previous weight and bias derivatives
 */
void adjust_correlations(net)
struct netstruct *net;
{
  int j,u,k,o;
  float cor,
        score,
        offset;

  net->best_candidate=net->first_candidate;
  net->best_candidate_score=0.0;
  for(u=net->first_candidate;u<net->first_candidate+net->num_candidates;u++) {
    score=0.0;
    cor=0.0;
    offset=net->avg_error*net->nodes[u]->cand_sum_value;
    for(o=0;o<net->outputnodes;o++){
      cor=(net->CandCor[u][o]-offset)/net->sum_sq_error;
      net->CandPrevCor[u][o]=cor;
      net->CandCor[u][o]=0.0;
      score+=fabs(cor);
    }
/**/printf("\ncandscor[%d]=%6.4f  ",u,score);
    if (score>net->best_candidate_score) {
      net->best_candidate_score=score;
      net->best_candidate=u;
    }
  }      
/**/printf("\nbest candidate=%d\n",net->best_candidate);  
  for(j=net->first_candidate;j<net->first_candidate+net->num_candidates;j++) {
    net->nodes[j]->prevbed=net->nodes[j]->bed;
    net->nodes[j]->bed=0.0;
    for(k=net->nodes[j]->first_weight_to;
	k<=net->nodes[j]->last_weight_to;k++) {
      net->prevwed[k]=net->wed[k];
      net->wed[k]=0.0;
    }
  }
}


/* For the current training pattern, compute the value of each candidate
 * unit and begin to compute the correlation between that unit's value and
 * the error at each output.  We have already done a forward-prop and
 * computed the error values for active units.
 * -RSC
 */
void correlations_epoch(net)
struct netstruct *net;
{
  int i,j;

  sa_feedforward(net);
  if(sim.current_pattern==0) {
    net->avg_error=net->sum_error/(ts.tsnum*net->outputnodes);
  }
  for(i=net->first_candidate;i<net->first_candidate+net->num_candidates;i++) {
    if (sim.current_pattern == 0) 
      net->nodes[i]->cand_sum_value=net->nodes[i]->output;
    else
      net->nodes[i]->cand_sum_value+=net->nodes[i]->output;
    for(j=0;j<net->outputnodes;j++) {
      net->CandCor[i][j]+=
	net->nodes[i]->output*net->errorcache[j][sim.current_pattern];
    }
  }
  if (sim.current_pattern==ts.tsnum-1) {
    net->first_in_epoch=FALSE;
    net->first_in_train_epoch=TRUE;
    adjust_correlations(net);
  }
}

/* After the initial correlation computation phase, this is the main 
 * candidate training controller 
 */
void train_candidates(net)
struct netstruct *net;
{
  int i;
  float tempeta;

  sa_feedforward(net);
  compute_correlation_slopes(net);
  if(sim.current_pattern==ts.tsnum-1) {
    tempeta=net->eta;
    net->eta=net->input_eta/(ts.tsnum*(net->first_candidate+1));
    change_weights(net);
    net->eta=tempeta;
    adjust_correlations(net);
  }
}

int train_inputs(net)
struct netstruct *net;
{
  if (net->first_in_epoch) {
    correlations_epoch(net);
    return CONTINUE;
  }
  else {
    train_candidates(net);
    if (sim.current_pattern==ts.tsnum-1) {
      if(net->input_patience==0)
	return CONTINUE;
      else if(net->first_in_train_epoch) {
	net->first_in_train_epoch=FALSE;
	net->last_score=net->best_candidate_score;
	net->timeout_epoch=sim.current_cycle+net->input_timeout;
	net->quit_epoch=sim.current_cycle+net->input_patience;
	return CONTINUE;
      }
      else if(fabs(net->best_candidate_score-net->last_score) >
	      (net->last_score*net->input_change_threshold)) {
	net->last_score=net->best_candidate_score;
	net->quit_epoch=sim.current_cycle+net->input_patience;
	return CONTINUE;
      }
      else if (sim.current_cycle >=net->quit_epoch)
	return STAGNANT;
      else if (sim.current_cycle>=net->timeout_epoch)
	return TIMEOUT;
      else
	return CONTINUE;
    }
    else
      return CONTINUE;
  }
}

/* Swap the cached weight-error-derivatives, and their dweights
 * for switching between input and ouput training phases.  May have
 * to save prevwed, also.
 */
void swap_dweights_and_weds(net)
struct netstruct *net;
{
  int i;
  float temp;

  for(i=0;i<MAXWEIGHTS;i++) {
    temp=net->wed[i];
    net->wed[i]=net->wedcache[i];
    net->wedcache[i]=temp;
    temp=net->dweights[i];
    net->dweights[i]=net->dweightscache[i];
    net->dweightscache[i]=temp;
  }
  for(i=0;i<MAXNODES;i++) {
    if (net->nodes[i]!=NULL) {
      temp=net->nodes[i]->bed;
      net->nodes[i]->bed=net->nodes[i]->bedcache;
      net->nodes[i]->bedcache=temp;
      temp=net->nodes[i]->dbias;
      net->nodes[i]->dbias=net->nodes[i]->dbiascache;
      net->nodes[i]->dbiascache=temp;
    }
  }
}

/* The main cascade-correlation update routine: controls switching between  
 *  candidate and output unit training 
 */
void cascor_update(net)
struct netstruct *net;
{
  switch(net->cascor_training_mode) {
  case OUTMODE:
    switch (train_outputs(net)) {
    case WIN:
      if (net->just_won_flag) {
	printf("\n** Net %s met criteria (output) at cycle %d\n **",net->ID,
	       sim.current_cycle);
	net->cycle_at_settling=sim.current_cycle;
      }
      break;
    case TIMEOUT:
      printf("\n** Net %s timed out (output) at cycle %d\n **",net->ID,
	     sim.current_cycle);
      net->cascor_training_mode=INMODE;
      create_candidates(net);
      
      net->first_in_epoch=TRUE;
      net->first_in_train_epoch=FALSE;
      
      swap_dweights_and_weds(net);
      break;
    case STAGNANT:
      printf("\n** Net %s stagnated output training at cycle %d\n",net->ID,
	     sim.current_cycle);
      net->cascor_training_mode=INMODE;
      create_candidates(net);
      net->first_in_epoch=TRUE;
      net->first_in_train_epoch=FALSE;
      swap_dweights_and_weds(net);
      break;
    case CONTINUE:
      break;
    default:
      printf("** Error in program: Bad return from TRAIN-OUTPUTS\n");
    }
    break;
  case INMODE:
    switch(train_inputs(net)) {
    case TIMEOUT:
      printf("\n** Net %s timed out (input) at cycle %d\n **",net->ID,
	     sim.current_cycle);
      select_candidate(net);
      net->cascor_training_mode=OUTMODE;
      net->first_out_epoch=TRUE;
      swap_dweights_and_weds(net);
      freeze_cand_weights(net);
      break;
    case STAGNANT:
      printf("\n** Net %s stagnated input training at cycle %d.",net->ID,
	     sim.current_cycle);
      printf(" Selecting candidate %d\n",net->best_candidate-net->first_candidate+1);
      select_candidate(net);
      net->cascor_training_mode=OUTMODE;
      net->first_out_epoch=TRUE;
      swap_dweights_and_weds(net);
      freeze_cand_weights(net);
      break;
    case CONTINUE:
      break;
    default:
      printf("** Error in program: Bad return from TRAIN-INPUTS\n");
    }
  }
}


/******************************************************************************/
/*                                                                            */
/*  Program Guts:  routines called by run_cycle                               */
/*                                                                            */
/******************************************************************************/

/* Print the clamped outputs and the input training pattern, used to show noise */

void print_clamped(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);

  for(i=0;i<ts.in;i++) 
    printf("%4.2f ",ts.inpat[sim.current_pattern][i]);
  printf(" | ");

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++)
    printf("%4.2f ",net->nodes[i]->desired_output);
  printf("  ");
}


void print_cycle_number()
{
  printf("C%d: ",sim.current_cycle);
}

void print_training_pattern()
{
  int i;

  for(i=0;i<ts.in;i++) 
    printf("%4.2f ",ts.inpat[sim.current_pattern][i]);
  printf(" | ");
    for(i=0;i<ts.out;i++)
      printf("%4.2f ",ts.outpat[sim.current_pattern][i]);
  printf("  ");
 }


/* Print the value of the first input node */

void print_inner_loop_value(netID)
char *netID;
{
  struct netstruct *net;

  net=get_net(netID);
  printf("Cause: %4.2f ",net->nodes[0]->output);
}

/* Print the value of the last output node */

void print_effect(netID)
char *netID;
{
  struct netstruct *net;
  int i;

  net=get_net(netID);
  sa_feedforward(netID);
  printf("Effect: ");

  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) 
    printf("%+4.2f ",net->nodes[i]->output);
}

/* Print the activation vector for a specified layer */

void print_layer(netID,layer)
char *netID;
int layer;
{
  struct netstruct *net;
  int i,node;

  net=get_net(netID);
  sa_feedforward(netID);
  printf("layer %d: ",layer);

  printf("(%d) : ", net->unitvector[layer-1]);
  for(node=0;node<net->unitvector[layer-1];node++) {
    i=get_node_index(net,layer,node);
    printf("%+4.2f ",net->nodes[i]->output);
  }
}

/* Print the desired activation vector for a specified layer */

void print_desired_layer(netID,layer)
char *netID;
int layer;
{
  struct netstruct *net;
  int i,node;

  net=get_net(netID);
  sa_feedforward(netID);
  printf("layer %d: ",layer);

  printf("(%d) : ", net->unitvector[layer-1]);
  for(node=0;node<net->unitvector[layer-1];node++) {
    i=get_node_index(net,layer,node);
    printf("%+4.2f ",net->nodes[i]->desired_output);
  }
}

void print_output_errors(netID)
char *netID;
{
  int i;
  struct netstruct *net;

  net=get_net(netID);
  printf("Errors: ");
  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) 
    printf("%4.2f ",error_fn(net->nodes[i]));
}


/* Copy a layer in one network to a layer in the other */

void copy(netID1,layer1,netID2,layer2)
char *netID1; 
char *layer1; 
char *netID2; 
char *layer2;
{
  int i,j;
  struct netstruct *net1,
                   *net2;


  net1=get_net(netID1);
  net2=get_net(netID2);

  /* Really ought to do some error checking here to be sure that the layer sizes
     are compatible */

  if(!strncmp(layer1,"O",1)) 
    if(!strncmp(layer2,"O",1))
      for(i=MAXNODES-net1->outputnodes,
	  j=MAXNODES-net2->outputnodes;
	  i<MAXNODES;i++,j++)
	net2->nodes[j]->output=net1->nodes[i]->output;
    else
      for(i=MAXNODES-net1->outputnodes,j=0;i<MAXNODES;i++,j++)
	net2->nodes[j]->output=net1->nodes[i]->output;
  else if (!strncmp(layer1,"I",1)) 
    if(!strncmp(layer2,"O",1))
      for(i=0,j=MAXNODES-net2->outputnodes;i<net1->inputnodes;i++,j++)
	net2->nodes[j]->output=net1->nodes[i]->output;
    else
      for(i=0,j=0;i<net1->inputnodes;i++,j++)
	net2->nodes[j]->output=net1->nodes[i]->output;
  /* Copy errors */
  else if (!strncmp(layer1,"E",1)) 
    if(!strncmp(layer2,"O",1))
      for(i=MAXNODES-net1->outputnodes,
	  j=MAXNODES-net2->outputnodes;
	  i<MAXNODES;i++,j++)
	net2->nodes[j]->output=error_fn(net1->nodes[i]);
    else
      for(i=MAXNODES-net1->outputnodes,j=0;i<MAXNODES;i++,j++)
	net2->nodes[j]->output=error_fn(net1->nodes[i]);
  sa_feedforward(net2->ID);
}



/* Copy the bigest output value of one layer to the first node of another */

void copy_biggest(netID1,layer1,netID2,layer2)
char *netID1; 
char *layer1; 
char *netID2; 
char *layer2;
{
  int i;
  struct netstruct *net1,
                   *net2;
  float bigerr;

  net1=get_net(netID1);
  net2=get_net(netID2);

  bigerr=0.0;

  /* Copy biggest error */
  if (!strncmp(layer1,"E",1) && !strncmp(layer2,"I",1)) {
    for(i=MAXNODES-net1->outputnodes;i<MAXNODES;i++)
      if ((error_fn(net1->nodes[i]) >bigerr) &&
	  ((i==MAXNODES-net1->outputnodes) || (i==MAXNODES-1)))
	bigerr=error_fn(net1->nodes[i]);
    net2->nodes[0]->output=bigerr;
  }
}



/* Copy the first input node of the first network to the last input node of
   the second */

void cause_copy(netID1,netID2)
char *netID1;
char *netID2;
{
  struct netstruct *net1,
                   *net2;

  net1=get_net(netID1);
  net2=get_net(netID2);

   net2->nodes[(net2->inputnodes)-1]->output=net1->nodes[0]->output + globals.cause_offset; 
}
    

/* The first network should be the "yes" network, and the second the "no" 
   network. The decision is Y if the yes network has greater output than 
   the no net,  N otherwise                                                */
	  
void generate_decision(netID1, netID2)
char *netID1, *netID2;
{
  struct netstruct *YN,
                   *NN;
  float YO,
         NO,
         r;

  YN=get_net(netID1);
  NN=get_net(netID2);

  sa_feedforward(netID1);
  sa_feedforward(netID2);
  /* get the values of the last output nodes */
  YO=YN->nodes[MAXNODES-1]->output;
  NO=NN->nodes[MAXNODES-1]->output;
/**/printf("Y=%4.2f N=%4.2f ",YO,NO);
  if (globals.prob_decision_flag) {
    r=(rand()%100)/100.0;
    if (r<(YO/(YO+NO)))
      sim.decision=1;
    else
      sim.decision=0;
  }
  else {
    if (YO==NO) {
      /* flip a coin to decide if the activaitons are equal */
      r=(rand()%10)/10.0;
      if (r>=.5)
	sim.decision=1;
      else 
	sim.decision=0;
    }
    else {
      if (YO>NO)
	sim.decision=1;
      else
	sim.decision=0;
    }
  }
  if (!(sim.decision==expected())) {
    sim.cycles_correct=0;
    sim.total_errors++;
  }
}

void print_pss(netID)
char *netID;
{
  printf("pss:%7.4f ",sum_square_error(netID));
}

void test_thresh_with_margin()
{
  struct netstruct *TN;
  int i,
      correctflag;

  TN=get_net("TN");

  /* default: assume that all output nodes will be within the margin of error */
  correctflag=1;
  for(i=MAXNODES-TN->outputnodes;i<MAXNODES;i++) {
    if(TN->nodes[i]->desired_output>OUT_MIDVAL) {
      if((TN->nodes[i]->desired_output-TN->nodes[i]->output)>TN->error_margin)
	correctflag=0;
    }
    else if ((TN->nodes[i]->output-TN->nodes[i]->desired_output)>TN->error_margin)
      correctflag=0;
  }
  if (correctflag)
    printf("** Passed");
  else {
    sim.cycles_correct=0;
    sim.total_errors++;
  }
}

void test_net_thresh_with_margin(netID)
char *netID;
{
  struct netstruct *net;
  int i,
      correctflag;

  net=get_net(netID);

  /* default: assume that all output nodes will be within the margin of error */
  correctflag=1;
  for(i=MAXNODES-net->outputnodes;i<MAXNODES;i++) {
    if(net->nodes[i]->desired_output>OUT_MIDVAL) {
      if((net->nodes[i]->desired_output-net->nodes[i]->output)>net->error_margin)
	correctflag=0;
    }
    else if ((net->nodes[i]->output-net->nodes[i]->desired_output)>net->error_margin)
      correctflag=0;
  }
  if (correctflag)
    printf("** Passed");
  else {
    sim.cycles_correct=0;
    sim.total_errors++;
  }
}
    
void print_thresh_errors()
{
  struct netstruct *TN;
  int i;

  TN=get_net("TN");

  printf(" ::  ");
  for(i=MAXNODES-TN->outputnodes;i<MAXNODES;i++) 
    printf("%4.2f ",OUT_MIDVAL-fabs(TN->nodes[i]->output-TN->nodes[i]->desired_output));
  printf(" ");
}

/* depending on where the appropriate feedback is stored, in the inpat or outpat,
    find it: it's the one with the length one.  Then compare it to the decision
*/
int expected()
{
  if (ts.in==1)
    if (ts.inpat[sim.current_pattern][0]>0.0)
      return 1;
    else
      return 0;
  else
    if (ts.outpat[sim.current_pattern][0]>0.0)
      return 1;
    else
      return 0;
}


void print_decision()
{

  printf("Decision: ");
  if (sim.decision)
    printf("Y ");
  else
    printf("N ");

  if (sim.decision==expected())
    printf("** Correct Guess ");
}


/* clamp the outputs of the decision net */

void clamp_dn(netID1, netID2)
char *netID1, *netID2;
{     
  struct netstruct *YN,
                   *NN;

  YN=get_net(netID1);
  if (strncmp(netID2,"X",1)) 
    NN=get_net(netID2);

  if (expected()) {
    YN->nodes[(MAXNODES)-1]->desired_output=1.0;
    if (strncmp(netID2,"X",1)) 
      NN->nodes[(MAXNODES)-1]->desired_output=0.0;
  }
  else {
    YN->nodes[(MAXNODES)-1]->desired_output=0.0;
    if (strncmp(netID2,"X",1)) 
      NN->nodes[(MAXNODES)-1]->desired_output=1.0;
  }
}


/* revise the weights of the decision net */

void revise_dn(netID1, netID2)
char *netID1, *netID2;
{
  /* Clamp the outputs of the YN and NN depending on the exepected value */
  clamp_dn(netID1,netID2);

  sa_feedforward(netID1);
  calculate_deltas(netID1);
  calculate_weight_changes(netID1);
  change_weights(netID1);

  if (strncmp(netID2,"X",1)) {
    sa_feedforward(netID2);
    calculate_deltas(netID2);
    calculate_weight_changes(netID2);
    change_weights(netID2);
  }
  
}


/* Call cascade-correlation learning procedure if the network is in    
   cascor mode.  Otherwise, do regular backprop                       */
void revise_tn_with_bias(netID)
char *netID;
{
  struct netstruct *net;
  
/**//*printf(" *revise tn with bias* ");*/
  net=get_net(netID);
  if (net->cascor_flag) {
/**//*printf("# cascor # ");*/
    cascor_update(net);
  }
  else {
/**//*printf(" @no cascor@ ");*/
    sa_feedforward(netID);
    calculate_deltas(netID);
    calculate_weight_changes(netID);
    change_weights(netID);
  }
}


/* Don't set the cause to 0 and revise the bias but not the weights */

void revise_tn_freeze_wt(netID)
char *netID;
{
  struct netstruct *net;
  
/**//*printf(" *revise tn freeze wt* ");*/
  net=get_net(netID);
  sa_feedforward(netID);
  calculate_deltas(netID);
  calculate_bias_changes(netID);
  if (( net->epoch_learn_flag && sim.current_pattern==ts.tsnum-1)
      || (!net->epoch_learn_flag))
    change_bias_freeze_wts(netID);
}


/* set the TN cause to 0 and revise the weights */

void revise_tn_only_bias(netID)
char *netID;
{
  struct netstruct *net;
  int i;

  net=get_net(netID);

  /* Set the input pattern to 0 */
  for(i=0;i<net->inputnodes;i++)
    net->nodes[i]->output=0.0;
  /* Call normal revision procedure */
  revise_tn_with_bias(netID);
}


/* Only revise the bias terms to the output nodes */
void revise_tn_out_bias_only(netID)
char *netID;
{
  change_out_bias(netID);
}


void case_feedback(linenum)
int *linenum;
{
  char arg1[MAXLINE];
  char *line;
  void execute_cmd();

  (*linenum)++;
  while(strncmp(cmds[sim.current_cmdblock]->cmd[*linenum],"ENDCASE",7)) {
    line=cmds[sim.current_cmdblock]->cmd[*linenum];
    if (!strncmp(line,"Y",1) && expected()) {
      line=pop_field(line);
      execute_cmd(line,linenum);
    }
    else if (!strncmp(line,"N",1) && !expected()) {
      line=pop_field(line);
      execute_cmd(line,linenum);
    }  
    (*linenum)++;
  }
}



void print_weights()
{
   print_all_weights();
}




/******************************************************************************/
/*                                                                            */
/*   run_theory()                                                             */
/*                                                                            */
/*   Main execution loop: execute the simulation cycles                       */
/*                                                                            */
/******************************************************************************/

void init_sim()
{
  int i,j;

  sim.current_cmdblock=0;
  sim.total_errors=0;
  sim.cycles_correct=0;
  sim.current_cycle=0;
  sim.current_pattern=0;
  sim.decision=0;
  sim.first=0;
  sim.tnsettledcycles=0;
  sim.dn_total_change=0.0;
  sim.tn_total_change=0.0;
  sim.did_gradient=0.0;
  sim.dn_true_error=0.0;
  sim.dn_last_error=0.0;

  /* Reset bias terms if they're not being used */
  for (i=0;networks[i]!=0;i++) 
    for(j=0;j<MAXNODES;skip_node_forward(&j,networks[i]) )
      if (((j>=(MAXNODES-networks[i]->outputnodes)) && globals.no_output_bias_flag) || 
	  (globals.no_bias_flag || networks[i]->no_bias_flag)) {
	networks[i]->nodes[j]->bias_weight=0;
	networks[i]->nodes[j]->bed=0;
	networks[i]->nodes[j]->dbias=0;
      }
}


/* Initialize all the weight change accumulators at the beginning of each 
   cycle */
void init_accum_weights()
{
  int i;

  for(i=0;networks[i]!=0;i++)
    networks[i]->accumweightchange=0;
}


/* Execute a command line */

void execute_cmd(line,linenum)
char *line;
int *linenum;
{
  char arg1[MAXLINE],
       arg2[MAXLINE],
       arg3[MAXLINE],
       arg4[MAXLINE];


  if (!strncmp(line,"PRINT-TRAINING-PATTERN",22)) {
    print_training_pattern();
  }
  else if (!strncmp(line,"PRINT-CYCLE-NUMBER",18)) {
    print_cycle_number();
  }
  else if (!strncmp(line,"CLAMP-DESIRED",13)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    clamp_desired_outputs(arg1);
  }
  else if (!strncmp(line,"CLAMP-LAYERS",12)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    clamp_layers(arg1);
  }
  else if (!strncmp(line,"PRINT-CLAMPED",13)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_clamped(arg1);
  }
  else if (!strncmp(line,"CLAMP-NOISY-DESIRED",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    clamp_noisy_desired_outputs(arg1);
  }
  else if (!strncmp(line,"CLAMP-INPUTS",12)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    clamp_inputs(arg1);
  }
  else if (!strncmp(line,"CLAMP-DISTRIBUTED-REPS",22)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    clamp_distributed_representations(arg1,atoi(arg2));
  }
  else if (!strncmp(line,"MAP-DISTRIBUTED-REPS",20)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    map_distributed_representations(arg1,atoi(arg2));
  }
  else if (!strncmp(line,"STORE-LAYER",11)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    store_layer(arg1,atoi(arg2));
  }
  else if (!strncmp(line,"INNER-LOOP",10)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    inner_loop(arg1);
  }
  else if (!strncmp(line,"AVG-INNER-LOOP",14)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    avg_inner_loop(arg1);
  }
  else if (!strncmp(line,"PRINT-INNER-LOOP-VALUE",22)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_inner_loop_value(arg1);
  }
  else if (!strncmp(line,"PRINT-RETURN",12)) {
    printf("\n");
  }
  else if (!strncmp(line,"PRINT-EFFECT-VALUE",18)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_effect(arg1);
  }
  else if (!strncmp(line,"FEEDFORWARD",11)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    sa_feedforward(arg1);
  }
  else if (!strncmp(line,"PRINT-LAYER",11)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    print_layer(arg1,atoi(arg2));
  }
  else if (!strncmp(line,"PRINT-DESIRED-LAYER",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    print_desired_layer(arg1,atoi(arg2));
  }
  else if (!strncmp(line,"PRINT-OUTPUT-ERRORS",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_output_errors(arg1);
  }
  else if (!strncmp(line,"COPY",4)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    line=pop_word(line);
    get_next_word(line,arg3);
    line=pop_word(line);
    get_next_word(line,arg4);
    copy(arg1,arg2,arg3,arg4);
  }
  else if (!strncmp(line,"BIGGEST-COPY",12)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    line=pop_word(line);
    get_next_word(line,arg3);
    line=pop_word(line);
    get_next_word(line,arg4);
    copy_biggest(arg1,arg2,arg3,arg4);
  }
  else if (!strncmp(line,"CAUSE-COPY",10)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    line=pop_word(line);
    cause_copy(arg1,arg2); 
  }
  else if (!strncmp(line,"GENERATE-DECISION",17)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    generate_decision(arg1,arg2);
  }
  else if (!strncmp(line,"TEST-THRESH-WITH-MARGIN",23)) {
    test_thresh_with_margin();
  }
  else if (!strncmp(line,"PRINT-PSS",9)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_pss(arg1);
  }
  else if (!strncmp(line,"TEST-NET-THRESH-WITH-MARGIN",27)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    test_net_thresh_with_margin(arg1);
  }
  else if (!strncmp(line,"PRINT-THRESH-ERRORS",19)) {
    print_thresh_errors();
  }
  else if (!strncmp(line,"PRINT-OUTPUT-ERRORS",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_output_errors(arg1);
  }
  else if (!strncmp(line,"PRINT-DECISION",14)) {
    print_decision();
  }
  else if (!strncmp(line,"REVISE-DN",9)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    line=pop_word(line);
    get_next_word(line,arg2);
    revise_dn(arg1,arg2);
  }
  else if (!strncmp(line,"CASE FEEDBACK",13)) {
    case_feedback(linenum);
  }
  else if (!strncmp(line,"REVISE-TN-WITH-BIAS",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    revise_tn_with_bias(arg1);
  }
  else if (!strncmp(line,"REVISE-TN-ONLY-BIAS",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    revise_tn_only_bias(arg1);
  }
  else if (!strncmp(line,"REVISE-TN-FREEZE-WT",19)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    revise_tn_freeze_wt(arg1);
  }
  else if (!strncmp(line,"REVISE-OUT-BIAS-ONLY",20)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    revise_tn_out_bias_only(arg1);
  }
  else if (!strncmp(line,"PRINT-VITALS",12)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_net_vitals(arg1);
  }
  else if (!strncmp(line,"PRINT-WEIGHTS",13)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_net_weights(get_net(arg1));
  }
  else if (!strncmp(line,"PRINT-NET-DELTAS",16)) {
    line=pop_field(line);
    get_next_word(line,arg1);
    print_net_weight_errors(arg1);
  }
  else {
    printf("**Error: Unknown command encountered: %s\n",line);
  }
}


/* Check to see if the VC TN has settled yet: Effects partitioned? */

void check_vc_tn_settled(hineg,tnsettled)
float *hineg; 
int *tnsettled;
{
  float temp;
  struct netstruct *net;

  net=get_net("TN");
  clamp_inputs("TN");
  sa_feedforward("TN");
  temp=net->nodes[MAXNODES-1]->output;
  if((temp>*hineg) && (sim.current_pattern<ts.neg_cases)) 
    *hineg=temp;
  if((sim.current_pattern>=ts.neg_cases) && (temp<*hineg))
    *tnsettled=0;
}



/* Check to see if the VE correlated TN has settled yet: Errors partitioned? */

void check_ve_cor_tn_settled(loneg,hipos,tnsettled)
float *loneg,
      *hipos;
int *tnsettled;
{
  float temp;
  struct netstruct *net;

  net=get_net("TN");
  clamp_desired_outputs("TN");
  inner_loop("TN");
  sa_feedforward("TN");
  temp=error_fn(net->nodes[MAXNODES-1])+
    error_fn(net->nodes[MAXNODES-3]);
  if((*loneg>temp) &&
       (ts.inpat[sim.current_pattern][0]==0.0))
       *loneg=temp;
  if((*hipos<temp) &&
       (ts.inpat[sim.current_pattern][0]==1.0))
       *hipos=temp;
  if((ts.inpat[sim.current_pattern][0]==1.0) && (temp>*loneg))
    *tnsettled=0;
  if((ts.inpat[sim.current_pattern][0]==0.0) && (temp<*hipos))
    *tnsettled=0;
}


/* Check to see if the VE correlated TN has settled yet: Causes partitioned? */

void check_ve_lin_tn_settled(hineg,lopos,tnsettled)
float *hineg,
      *lopos;
int *tnsettled;
{
  float temp;
  struct netstruct *net;

  clamp_desired_outputs("TN");
  inner_loop("TN");
  net=get_net("TN");
  temp=net->nodes[0]->output;

  if((*hineg<temp) &&
       (ts.inpat[sim.current_pattern][0]==0.0))
       *hineg=temp;
  if((*lopos>temp) &&
       (ts.inpat[sim.current_pattern][0]==1.0))
       *lopos=temp;
  if((ts.inpat[sim.current_pattern][0]==1.0) && (temp<*hineg))
     *tnsettled=0;
  if((ts.inpat[sim.current_pattern][0]==0.0) && (temp>*lopos))
     *tnsettled=0;
}

/* This function chooses a new random number seed based on the
   system time.  It was implemented in order to avoid apparent
   regularities in the random number sequence which cause the
   trail-randomization procedure to always a choose a particular
   trial at the beginning of an epoch.  This routine was copied
   from pg. 510 of "C: The Complete Reference" (Osborne/McGraw Hill) */
void rerandomize()
{
  int i, stime;
  int r;
  long ltime;

  /* get the current calendar time */
  ltime = time(NULL);
  stime= ((unsigned) ltime/2)%326;
  r=rand()%100;
  srand(stime*r);
}

/* scramble the training patterns each epoch */
void randomize_ts()
{
  int i,j,r;
  int num_list[MAXPATTERNS];

  rerandomize();
  for(i=0;i<ts.tsnum;i++)
    num_list[i]=i;
  for(i=0;i<ts.tsnum;i++) {
    r=rand();
    r=r%(ts.tsnum-i);
    for (j=0;j<ts.in;j++)
      ts.inpat[i][j]=ts.inpat_buff[num_list[r]][j];
    for (j=0;j<ts.out;j++)
      ts.outpat[i][j]=ts.outpat_buff[num_list[r]][j];
    for (j=r;j<ts.tsnum;j++)
      num_list[j]=num_list[j+1];
  }
}


void run_cycle()
{
  int i,
      j;
  char line[MAXLINE];
  int tnsettled;
  float hineg,
        loneg,
        hipos,
        lopos,
        tss;

  /* Init the tn settling info */
  tnsettled=1;
  hineg=0.0;
  loneg=10.0;
  hipos=0.0;
  lopos=10.0;
  tss=0.0;

  /* Initialize the weight change accumulators */
  init_accum_weights();

  /* Set/update the correct cycle counter: will be reset when an error is 
     encountered */
  sim.cycles_correct++;
  printf("Cycle : %d\n",sim.current_cycle);

  if (globals.randomize_ts_flag && (get_net("TN"))->cascor_training_mode!=INMODE)
    randomize_ts();

  /* Iterate through the training set, for each training pattern */
  for(sim.current_pattern=0;sim.current_pattern<ts.tsnum;sim.current_pattern++) {


    /* Update the TN settling info */
    if (globals.print_vc_tn_settled_flag)
      check_vc_tn_settled(&hineg,&tnsettled);
    else if (globals.print_ve_cor_tn_settled_flag)
      check_ve_cor_tn_settled(&loneg,&hipos,&tnsettled);
    else if (globals.print_ve_lin_tn_settled_flag)
      check_ve_lin_tn_settled(&hineg,&lopos,&tnsettled);

    /* Iterate through the command list */
    for(j=0;j<cmds[sim.current_cmdblock]->cmdline;j++) {
      strcpy(line,cmds[sim.current_cmdblock]->cmd[j]); 
      execute_cmd(line,&j);
    }
    /* Update sum-squared error */
    if (globals.print_tss_flag)
      tss+=sum_square_error("TN");
  }
  if (globals.print_tn_change_flag) 
    printf("TN weight change: %8.6f\n",(get_net("TN"))->accumweightchange);
  if (globals.print_dn_change_flag) 
    printf("DN weight change: %8.6f\n",(get_net("YN"))->accumweightchange+
	                               (get_net("NN"))->accumweightchange);
  if (globals.print_vc_tn_settled_flag || 
      globals.print_ve_cor_tn_settled_flag ||
      globals.print_ve_lin_tn_settled_flag)
    printf("settled : %d\n",tnsettled);
  if(tnsettled && sim.first==0) {
    sim.tnsettledcycles=sim.current_cycle;
    sim.first=1;
  }
  /* Reset tnsettledcycles if TN "unsettles" */
  if(!tnsettled && sim.first) {
    sim.tnsettledcycles=0;
    sim.first=0;
  }
  if (globals.print_tss_flag)
    printf("tss : %7.4f\n",tss);
  printf("\n");
}
    
/* Check if some networks should run additional cycles: freeze the other
   networks and run the networks which require cycles                        */
void run_subcycles()
{
  int i,j,maxcycles,
      templflag[MAXNETWORKS];

  maxcycles=0;
  i=0;
  while(networks[i] != NULL) {
    templflag[i]=networks[i]->lflag;
    if (networks[i]->nepochs>maxcycles)
      maxcycles=networks[i]->nepochs;
    i++;
  }
  for(i=0;i<maxcycles;i++) {
    j=0;
    while(networks[j] != NULL) {
      if (networks[j]->nepochs>i)
	networks[j]->lflag=1;
      else 
	networks[j]->lflag=0;
      j++;
    }
    run_cycle();
  }
  i=0;
  while(networks[i] != NULL) {
    networks[i]->lflag=templflag[i];
    i++;
  }
}


int get_cmdblock_index(cmdID)
char *cmdID;
{
  int i;

  i=0;
  while(cmds[i] != NULL) {
    if (!strcmp(cmds[i]->ID,cmdID))
      return i;
    i++;
  }
  return -1;
}


void run_epoch()
{
  int i;
  int t;
  char word[MAXLINE];

  /* If there's only one command block, skip the epochs section */
  if (cmds[1]==NULL) {
    sim.current_cmdblock=0;
    run_cycle();
    if (globals.print_weight_flag)
      print_weights();
    run_subcycles();
    sim.current_cycle++;
  }
  else {
    for (i=0;i<epoch.cmdline;i++) {
      get_next_word(epoch.cmd[i],word);
      t=get_cmdblock_index(word);
      if (t<0) {
	printf("*** Error: Unknown command block: %s!\n",word);
	return;
      }
      else
	sim.current_cmdblock=t;
      run_cycle();
      if (globals.print_weight_flag)
	print_weights();
      run_subcycles();
    }
    sim.current_cycle++;
  }
}

void run_theory()
{
  init_sim();                           /* Initialize the simulation state    */
    
  printf("\nInitial weights: \n");
  if (globals.print_weight_flag)
    print_weights();
  printf("\n\n");
  while(sim.cycles_correct<globals.cycle_criteria && 
	sim.current_cycle<globals.maxcycles) 
    run_epoch();
  printf("\nTotal Errors: %d\n",sim.total_errors);
  printf("Total Cycles: %d\n",sim.current_cycle);
  if (globals.print_vc_tn_settled_flag || 
      globals.print_ve_cor_tn_settled_flag ||
      globals.print_ve_lin_tn_settled_flag)
    printf("tn settled cycles: %d\n",sim.tnsettledcycles);
  if (get_net("TN")->quickprop_flag)
    printf("Gradient Descent times: %d\n",sim.did_gradient);
}


/* Save the weights in the network */

void save_weights(filename)
char *filename;
{
  FILE *fp;
  int net,layer,layernode,node,weight;

  fp=fopen(filename,"w");
  for(net=0;networks[net]!=0;net++) {
    fprintf(fp,"#net: %s\n",networks[net]->ID);    
    for(layer=0;layer<MAXLAYERS;skip_layer_forward(&layer,networks[net])) {
      fprintf(fp,"#layer: %d\n",layer);
      for(layernode=0;layernode<networks[net]->unitvector[layer];layernode++) {
	node=get_node_index(networks[net],layer+1,layernode);
	fprintf(fp,"#node=%d\n",node);
	fprintf(fp,"%10.8f\n",networks[net]->nodes[node]->bias_weight);
	for(weight=networks[net]->nodes[node]->first_weight_to;
	    weight<=networks[net]->nodes[node]->last_weight_to;weight++) {
	  fprintf(fp,"%10.8f\n",networks[net]->weights[weight]);
         }
      }
    }
  }
  fclose(fp);
}


main (argc, argv)
int argc;
char *argv[];
{

  if (argc < 3) {
    printf("** Error: ");
    printf("You must specify a network file and a training set file\n");
  }
  else {
    /* Initialize random number generator */
    
    printf("Enter a number between 0 and 32676.  Be creative >");
    scanf("%d",&x);
    srand(x);

    weights_are_loaded_flag=0;
    if((argc>3)&&(*argv[3]!='-')) {
      weights_are_loaded_flag=1;
      weightfp=fopen(argv[3],"r");
    }
    if(argc>5) {                         /* Read in mappings and symbol tables */
      read_mappings_file(argv[5]);
      read_symbol_table(argv[6]);
      compute_node_correspondences();
    }
    if (read_net_file(argv[1]))         /* Read in net and ts definitions */
      if (read_ts_file(argv[2])) {
	run_theory();                   /* Execute the simulation         */
	if((argc>4) && (*argv[4]!='-'))
	  save_weights(argv[4]);
      }
    if (weights_are_loaded_flag)
      fclose(weightfp);
  }
}

