;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  scrollers.lisp
;;
;;
;;  1989, Apple Computer, Inc
;;
;;  code for views with scroll bars.
;;

;;;;;;;;;;;;;
;;
;; Modification History
;;
;; 01/06/92 bill fix bug in reposition-scroll-bars.
;;               scroll-bar-limits now positions, at full scroll, the lower
;;               right-hand corner of the field-size at the lower
;;               right-hand corner of the display. It used to place the
;;               lower right-hand corner of the field-size at the
;;               upper left-hand corner of the display.
;;               Scroll bars are now disabled when the entire
;;               field-size is visible and the view is scrolled to 0.
;; 07/29/91 bill set-view-scroll-position does update-thumbs (thanks to Christopher Owens)
;;

(in-package :ccl)

(defclass box-dialog-item (simple-view) ())

(defmethod point-in-click-region-p ((self box-dialog-item) point)
  (declare (ignore point))
   nil)

(defmethod view-draw-contents ((self box-dialog-item))
  (let* ((pos (view-position self))
        (end (add-points pos (view-size self))))
    (rlet ((r :rect
              :topleft pos
              :bottomright end))
      (#_FrameRect r))))

(defclass scroller-mixin ()
  ((v-scroller :accessor v-scroller)
   (h-scroller :accessor h-scroller)
   (field-size :initarg :field-size :initform nil :reader field-size)
   (scroller-outline :accessor scroller-outline) 
   (scroll-bar-correction :accessor scroll-bar-correction)))

(defclass scroller (scroller-mixin view) ())

(defmethod initialize-instance ((self scroller-mixin) &rest initargs &key
                                view-container (v-scrollp t) (h-scrollp t)
                                (draw-scroller-outline t)
                                track-thumb-p
                                (scroll-bar-class 'scroll-bar-dialog-item)
                                h-scroll-class v-scroll-class)
  (declare (dynamic-extent initargs))
  (setf (v-scroller self) nil)          ; fix start-up transient.
  (setf (h-scroller self) nil)
  (apply #'call-next-method self :view-container nil initargs)   ; delay the set-view-container
  (let* ((v-scroll (if v-scrollp
                     (make-instance (or v-scroll-class scroll-bar-class)
                                    :scrollee self
                                    :direction :vertical
                                    :track-thumb-p track-thumb-p)))
         (h-scroll (if h-scrollp
                     (make-instance (or h-scroll-class scroll-bar-class)
                                    :scrollee self
                                    :direction :horizontal
                                    :track-thumb-p track-thumb-p)))
         (outline (if draw-scroller-outline
                    (make-instance 'box-dialog-item))))
    (setf (scroll-bar-correction self) (make-point (if v-scroll 17 2)
                                             (if h-scroll 17 2)))
    (setf (v-scroller self) v-scroll)
    (setf (h-scroller self) h-scroll)
    (setf (scroller-outline self) outline)
    (if (and (view-position self) (view-size self))
      (update-scroll-bars self :length t :position t))
    (when view-container
      (set-view-container self view-container))))

;; This is how a view communicates it's scroll bar limits to a scroller.
;; Returns two points, the limits for the horizontal & vertical scroll bars
;; This is the coordinate system passed to set-view-scroll-position

(defmethod scroll-bar-limits ((view scroller-mixin))
  (let ((field-size (field-size view))
        (size (view-size view))
        (h-scroller (h-scroller view))
        (v-scroller (v-scroller view)))
    (if field-size
      (values (make-point 0 (max (if h-scroller (scroll-bar-setting h-scroller) 0)
                                 (- (point-h field-size) (point-h size))))
              (make-point 0 (max (if v-scroller (scroll-bar-setting v-scroller) 0)
                                 (- (point-v field-size) (point-v size)))))
      (let ((size (view-size view)))
        (normal-scroll-bar-limits view (add-points size size))))))

(defmethod normal-scroll-bar-limits ((view scroller-mixin) max-h &optional max-v)
  (let ((size (view-size view))
        (max (make-point max-h max-v)))
    (values (make-point 0 (max 0 (- (point-h max) (point-h size))))
            (make-point 0 (max 0 (- (point-v max) (point-v size)))))))

;; And here's how a view communicates it's page size
;; Returns a point.
(defmethod scroll-bar-page-size ((view scroller-mixin))
  (view-size view))

(defmethod set-view-container ((self scroller-mixin) new-container)
  (let ((need-to-update? (not (and (view-position self) (view-size self)))))
    (call-next-method)
    (when (v-scroller self)  (set-view-container (v-scroller self) new-container))
    (when (h-scroller self)  (set-view-container (h-scroller self) new-container))
    (when (scroller-outline self)  
      (set-view-container (scroller-outline self) new-container))
    (when need-to-update?
      (update-scroll-bars self :length t :position t))
    new-container))

(defmethod set-view-position ((self scroller-mixin) h &optional v)
  (declare (ignore h v))
  (without-interrupts
   (prog1
     (call-next-method)
     (update-scroll-bars self :position t))))

(defmethod set-view-size ((self scroller-mixin) h &optional v)
  (declare (ignore h v))  
  (without-interrupts
   (prog1
     (call-next-method)
     (update-scroll-bars self :length t :position t))))

(defmethod update-scroll-bars ((self scroller-mixin) &key length position)
  (let* ((pos (view-position self))
         (size (view-size self))
         (h-scroller (h-scroller self))
         (v-scroller (v-scroller self))
         (outline (scroller-outline self)))
    (when (and pos size)                ; auto-sizing may not have happenned yet 
      (without-interrupts
       (reposition-scroll-bars self h-scroller v-scroller :length length :position position)
       (when length
         (update-scroll-bar-limits self h-scroller v-scroller))
       (when outline
         (setq pos (subtract-points pos #@(1 1))
               size (add-points size (scroll-bar-correction self)))
         (set-view-position outline pos)
         (set-view-size outline size))))))

(defmethod update-scroll-bar-limits ((self scroller-mixin) &optional
                                     (h-scroller (h-scroller self))
                                     (v-scroller (v-scroller self)))
  (multiple-value-bind (h-limits v-limits) (scroll-bar-limits self)
    (let ((page-size (scroll-bar-page-size self)))
      (when  h-scroller
        (set-scroll-bar-min h-scroller (point-h h-limits))
        (set-scroll-bar-max h-scroller (point-v h-limits))
        (setf (scroll-bar-page-size h-scroller) (point-h page-size)))
      (when v-scroller
        (set-scroll-bar-min v-scroller (point-h v-limits))
        (set-scroll-bar-max v-scroller (point-v v-limits))
        (setf (scroll-bar-page-size v-scroller) (point-v page-size))))))

;; Call this whenever the thumb position changes.
(defmethod update-thumbs ((self scroller-mixin))
  (let ((pos (view-scroll-position self))
        (h-scroller (h-scroller self))
        (v-scroller (v-scroller self))
        (update-limits? nil))
    (when (and h-scroller
               (not (eql (scroll-bar-setting h-scroller) (point-h pos))))
      (when (eql (scroll-bar-min h-scroller)
                 (set-scroll-bar-setting h-scroller (point-h pos)))
        (setq update-limits? t)))
    (when (and v-scroller
               (not (eql (scroll-bar-setting v-scroller) (point-v pos))))
      (when (eql (scroll-bar-min v-scroller)
                 (set-scroll-bar-setting v-scroller (point-v pos)))
        (setq update-limits? t)))
    (when update-limits? (update-scroll-bar-limits self))))

; Seperate from update-scroll-bars so that users can specialize it.
(defmethod reposition-scroll-bars ((self scroller-mixin) h-scroller v-scroller &key length position)
  (let* ((pos (view-position self))
         (size (view-size self))
         (width (point-h size))
         (height (point-v size)))
    (when (and pos size)                ; auto-sizing may not have happenned yet 
      (without-interrupts
       (when h-scroller
         (when position
           (set-view-position h-scroller (add-points pos (make-point -1 height))))
         (when length
           (set-scroll-bar-length h-scroller (+ 2 width))))
       (when v-scroller
         (when position
           (set-view-position v-scroller (add-points pos (make-point width -1))))
         (when length
           (set-scroll-bar-length v-scroller (+ 2 height))))))))
 
(defmethod scroll-bar-changed ((view scroller-mixin) scroll-bar)
  (let* ((new-value (scroll-bar-setting scroll-bar))
         (horizontal-p (eq (scroll-bar-direction scroll-bar) :horizontal))
         (old-pos (view-scroll-position view)))
    (set-view-scroll-position 
     view
     (if horizontal-p
       (make-point new-value (point-v old-pos))
       (make-point (point-h old-pos) new-value)))
    (when (eql new-value (scroll-bar-min scroll-bar))
      (update-scroll-bar-limits view)))
  (window-update-event-handler (view-window view)))

(defmethod set-view-scroll-position ((view scroller-mixin) h &optional v scroll-visibly)
  (declare (ignore h v scroll-visibly))
  (prog1
    (call-next-method)
    (update-thumbs view)))

(provide 'scrollers)   

