;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

;;;=======================================
;;;  type-in-window

(defclass type-in-window (window)
  ((message :initarg :message :initform "" :accessor type-in-window-message)
   (initial-string :initarg :initial-string :initform "" :accessor type-in-window-initial-string)))

(defmethod view-key-event-handler ((window type-in-window) char)
  (if (eq (char-code char) 179)
    (let ((fred (view-named :fred window)))
      (multiple-value-bind (sexp sexp-p) (ed-current-sexp fred)
        (when (and sexp-p (atom sexp))
          (let ((r-struct (get-r-struct sexp)))
            (when (r-p r-struct)
              (user-edit-relation r-struct))))))
    (call-next-method window char)))
    


;;;_______________________________________
;;;  move-fred-and-buttons

(defmethod move-fred-and-buttons ((window type-in-window) offset)
  (without-interrupts
   (with-focused-view window
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
       (#_eraserect view-rect)
       (#_beginupdate (wptr window))
       (let* ((fred (view-named :fred window))
              (ok (view-named :ok window))
              (cancel (view-named :cancel window)))
         (set-view-size fred (add-points (view-size fred) offset))
         (set-view-position ok (add-points (view-position ok) offset))
         (set-view-position cancel (add-points (view-position cancel) offset)))
       (#_endupdate (wptr window))
       (#_invalrect view-rect)))))

;;;_______________________________________
;;;  set-view-size

(defmethod set-view-size ((window type-in-window) h &optional v)
  (without-interrupts
   (let ((old-size (view-size window)))
     (apply #'call-next-method window h v)
     (move-fred-and-buttons window (subtract-points (view-size window) old-size)))))

;;;_______________________________________
;;;  window-zoom-event-handler

(defmethod window-zoom-event-handler ((window type-in-window) message)
  (declare (ignore message))
  (without-interrupts
   (let ((old-size (view-size window)))
     (call-next-method)
     (move-fred-and-buttons window (subtract-points (view-size window) old-size)))))



;;;=======================================
;;;  modal-type-in-window

(defclass modal-type-in-window (type-in-window)
  ((ok-text :initarg :ok-text :initform " OK " :accessor modal-type-in-window-ok-text)
   (cancel-text :initarg :cancel-text :initform " Cancel " :accessor modal-type-in-window-cancel-text)))

;;;_______________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window modal-type-in-window) &rest initargs)
  (setf (getf initargs :window-show) nil
        (getf initargs :window-type) :document-with-zoom
        (getf initargs :close-box-p) nil)
  (apply #'call-next-method window initargs)
  (set-view-scroll-position window 0 0)
  (let* ((ws (view-size window))
         (wh (point-h ws))
         (wv (point-v ws))
         (mv (add-message window (type-in-window-message window))))
    (add-subviews
     window
     (make-dialog-item 'ccl::scrolling-fred-dialog-item (make-point 10 mv)
                       (make-point (- wh 35) (- wv mv 52)) (type-in-window-initial-string window) nil
                       :allow-tabs t :allow-returns t :view-font '("monaco" 9 :plain) :view-nick-name :fred)
     (make-dialog-item 'button-dialog-item (make-point (- wh 175) (- wv 27)) #@(70 20) (modal-type-in-window-ok-text window)
                       #'(lambda (item) (return-from-modal-dialog (dialog-item-text (find-named-sibling item :fred))))
                       :view-nick-name :ok)
     (make-dialog-item 'button-dialog-item (make-point (- wh 95) (- wv 27)) #@(70 20) (modal-type-in-window-cancel-text window)
                       #'(lambda (item) item (return-from-modal-dialog :cancel))
                       :view-nick-name :cancel)))
  (collapse-selection (view-named :fred window) t)
  (fred-update (view-named :fred window)))


;;;_______________________________________
;;;  get-text-from-user

(defun get-text-from-user (message &key (size #@(540 200)) 
                                   (position :centered)
                                   (initial-string "")
                                   (title "Enter Information")
                                   (ok-text " OK ")
                                   (cancel-text " Cancel "))
  (modal-dialog (make-instance 'modal-type-in-window
                  :message message
                  :view-size size
                  :view-position position
                  :initial-string initial-string
                  :window-title title
                  :ok-text ok-text
                  :cancel-text cancel-text) t))


;;;=======================================
;;;  eval-window

(defclass eval-window (type-in-window)
  ((original :initarg :original :initform nil :accessor eval-window-original)))

;;;_______________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window eval-window) &rest initargs)
  (setf (getf initargs :window-show) nil
        (getf initargs :window-type) :document-with-zoom
        (getf initargs :close-box-p) nil)
  (apply #'call-next-method window initargs)
  (set-view-scroll-position window 0 0)
  (let* ((ws (view-size window))
         (wh (point-h ws))
         (wv (point-v ws))
         (mv (add-message window (type-in-window-message window))))
    (add-subviews
     window
     (make-dialog-item 'ccl::scrolling-fred-dialog-item (make-point 4 mv)
                       (make-point (- wh 23) (- wv mv 52)) (type-in-window-initial-string window) nil
                       :allow-tabs t :allow-returns t :view-font '("monaco" 9 :plain) :view-nick-name :fred)
     (make-dialog-item 'button-dialog-item (make-point (- wh 180) (- wv 27)) #@(70 20) " Define "
                       #'(lambda (item)
                           (multiple-value-bind (value error) 
                                                (catch-error-quietly 
                                                  (eval (read-from-string (dialog-item-text (find-named-sibling item :fred)))))
                             (if error
                               (notify-error "~%~a" error)
                               (let ((window (view-container item)))
                                 (setq *rules-changed* t)
                                 (when (r-p (eval-window-original window))
                                   (update-windows (get-r-struct value) (eval-window-original window)))
                                 (window-close (view-container item))))))
                       :default-button t
                       :view-nick-name :ok)
     (make-dialog-item 'button-dialog-item (make-point (- wh 95) (- wv 27)) #@(70 20) " Cancel "
                       #'(lambda (item) (window-close (view-container item)))
                       :view-nick-name :cancel)))
  (collapse-selection (view-named :fred window) t)
  (fred-update (view-named :fred window)))

;;;_______________________________________
;;; text-edit

(defun text-edit (&key (message "")
                       (initial-string nil)
                       (title "Text Edit")
                       (header nil)
                       (extra-lines nil)
                       (size #@(520 300))
                       (position :centered)
                       (original nil))
  (let ((string initial-string))
    (when extra-lines (setq size (add-points size (make-point 0 (* 12 extra-lines)))))
    (when (and header (not string)) (setf string (string-concat header string)))
    (window-select (make-instance 'eval-window
                     :message message
                     :initial-string string
                     :window-title title
                     :view-size size
                     :view-position position
                     :original original))))

;;;_______________________________________
;;; text-edit-relation

(defun text-edit-relation (r-struct &key
                                    (size #@(520 300))
                                    (header "")
                                    (extra-lines nil)
                                    (position :centered)
                                    (initial-string nil))
  (if (member r-struct *special-r-structs*)
    (special-r-struct-message r-struct)
    (let* ((error nil)
           (message (format nil "Edit ~A" (name-vars-string (r-name r-struct) (r-vars r-struct))))
           (string (or initial-string 
                       (with-output-to-string (temp)
                         (case (r-kind r-struct)
                           (:extensional (dump-pred r-struct temp))
                           (:builtin (dump-builtin r-struct temp))
                           (:intensional (dump-rule r-struct temp))
                           (:arithmetic-op (dump-arithmetic-op r-struct temp))
                           (otherwise
                            (setf error t)
                            (message-dialog (format nil "There is currently no way of editing ~A relations."  (r-kind r-struct)) :position :centered)
                            ))))))
      (unless error
        (when extra-lines (setq size (add-points size (make-point 0 (* 12 extra-lines)))))
        (when header (setf string (string-concat header string)))
        (text-edit :message message
                   :initial-string string
                   :header nil
                   :extra-lines nil
                   :size size
                   :position position
                   :original r-struct)))))

;;;_______________________________________
;;; text-edit-type

(defun text-edit-type (&optional (type nil))
  (text-edit :message (if type (format nil "Edit ~S"  type) "Create Type")
             :initial-string (with-output-to-string (temp)
                               (if (or (member type *all-types*)
                                       (assoc type *domain*))
                                 (dump-type type temp)
                                 (format temp "(def-type ~S~%          instances)~%" (or type 'TYPE-NAME))))
             :title "Text Edit Type"))

;;;_______________________________________
;;; text-edit-cliche

(defun text-edit-cliche (&optional (cliche nil))
  (text-edit :message (if cliche (format nil "Edit ~S" (cliche-name cliche)) "Create Clich")
             :initial-string (with-output-to-string (temp)
                               (if (cliche-p cliche)
                                 (dump-cliche cliche temp)
                                 (format temp "(def-cliche ~s~%  :pred-restrictions  ()~%  :var-restrictions   ()~%  :cache?  NAMED )~%" (or cliche 'CLICHE-NAME))))
             :title "Text Edit Clich"))

;;;_______________________________________
;;; text-edit-rule

(defun text-edit-rule ()
  (catch-cancel
    (let ((r (select-a-rule "Select a rule to edit" "Text Edit Rule")))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-text-edit-rule *user-monitor*)))
      (text-edit-relation r))))

;;;_______________________________________
;;; text-edit-fact

(defun text-edit-fact ()
  (catch-cancel
    (let ((r (select-a-fact "Select a fact to edit" "Text Edit Fact")))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-text-edit-fact *user-monitor*)))
      (text-edit-relation r))))

;;;_______________________________________
;;; text-edit-template

(defun text-edit-template (&optional (template nil))
  (text-edit :message (if template (format nil "Edit ~S Example Template"  (example-template-name template))  "Create Example Template")
             :initial-string (with-output-to-string (temp)
                               (if (example-template-p template)
                                 (dump-example-template template temp)
                                 (format temp "(def-example-template FACT_NAME (?ARGUMENTS)~%  (relation_name ?ARGUMENT _ _)~%  )")))
             :title "Text Edit Example Template"))