
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

(defparameter *h-bar* 81)
(defparameter *v-bar* 20)
(defparameter *new-state* :cliche)


;;;=========================================================
;;; EDIT WINDOW

;;;_______________________________________
;;; edit-window

(defclass edit-window (window)
  ((nodes :initarg :nodes :initform nil :accessor edit-window-nodes)
   (analyzed :initarg :analyzed :initform nil :accessor edit-window-analyzed)))

;;;_______________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window edit-window) &rest initargs)
  (setf (getf initargs :close-box-p) nil
        (getf initargs :window-type) :document-with-zoom
        (getf initargs :window-show) nil
        (getf initargs :view-position) :centered
        (getf initargs :color-p) t
        (getf initargs :view-font) *default-font*
        (getf initargs :view-scroll-position) #@(0 0))
  (apply #'call-next-method window initargs)
  (add-subviews
   window
   (make-dialog-item 'static-text-dialog-item #@(10 2) #@(1000 16) "" nil :view-nick-name :original-coverage)
   (make-instance
     'graph-scroller
     :view-position (make-point *h-bar* *v-bar*)
     :view-size (subtract-points (view-size window) (make-point *h-bar* *v-bar*))
     :view-font *default-font*
     :view-nick-name :graph-scroller))
  (let ((view (graph-view window)))
    (set-font-dependent-attributes view)
    (reset-scroll-bars (graph-scroller window)))
  )

;;;_______________________________________
;;;   set-view-size

(defmethod set-view-size ((window edit-window) h &optional (v nil))
  (let ((scroller (graph-scroller window)))
    (if (null v)
      (setf v (point-v h)
            h (point-h h)))
    (reset-view-size scroller (- h *h-bar*) (- v *v-bar*))
    (call-next-method)))

;;;_______________________________________
;;;   resize-window

(defmethod resize-window ((window edit-window) &optional (h nil) (v nil))
  (general-resize-window window h v *h-bar* *v-bar*))

;;;_______________________________________
;;;   grow-window-if-needed

(defmethod grow-window-if-needed ((window edit-window))
  (general-grow-window-if-needed window *h-bar* *v-bar*))

;;;_______________________________________
;;;   view-draw-contents

(defmethod view-draw-contents ((window edit-window))
  (with-focused-view window
    (call-next-method)
    (let ((v (- *v-bar* 1)) (h (- *h-bar* 1)))
      (with-focused-view window
        (#_pensize 1 1)
        (#_penpat *black-pattern*)
        (with-fore-color *black-color*
          (#_moveto -1 v)
          (#_lineto 3000 v)
          (#_moveto h v)
          (#_lineto h 3000)
          )))))

;;;_______________________________________
;;;   window-zoom-event-handler

(defmethod window-zoom-event-handler ((window edit-window) message)
  (declare (ignore message))
  (call-next-method)
  (reset-view-size (graph-scroller window) (subtract-points (view-size window) (make-point *h-bar* *v-bar*)))
  )

;;;_______________________________________
;;;   window-hardcopy

(defmethod window-hardcopy ((window edit-window) &optional (show-dialog? t))
  (declare (ignore show-dialog?))
  (hardcopy-graph-in-window window))

;;;_______________________________________
;;;   copy

(defmethod copy ((window edit-window))
  (copy-graph-to window))

;;;_______________________________________
;;;   copy

(defmethod copy ((window edit-window))
  (copy-graph-to window))

;;;_______________________________________
;;;   clear

(defmethod clear ((window edit-window))
  (editor-DELETE window))

;;;_______________________________________
;;;   cut

(defmethod cut ((window edit-window))
  (copy-graph-to window)
  (editor-DELETE window))

;;;_______________________________________
;;;   paste

(defmethod paste ((window edit-window))
  (multiple-value-bind (prolog conjunction-p) (read-prolog-from-string (get-scrap :TEXT) nil nil t)
    (when prolog
      (let* ((graph (window-graph window))
             (u-base (graph-u-base graph)))
        (if conjunction-p
          (let ((new-nodes (connect-clause graph u-base prolog nil *new-state* :never nil)))
            (mapcar #'(lambda (n) (setf (node-aux n) *new-state*)) new-nodes)
            (setf (node-antecedents u-base) (nconc (node-antecedents u-base) (list new-nodes)))
            (select-node new-nodes t)
            (window-fixup window))
          (let ((new-node (connect-literal graph u-base prolog nil *new-state* :never nil)))
            (setf (node-aux new-node) *new-state*)
            (setf (node-antecedents u-base) (nconc (node-antecedents u-base) (list (list new-node))))
            (select-node new-node t)
            (window-fixup window)))))))

;;;_______________________________________
;;;   window-close

(defmethod window-close ((window edit-window))
  (dispose-graph-view (graph-view window))
  (call-next-method))

;;;_______________________________________
;;; window-fixup

(defmethod window-fixup ((window edit-window) &optional (position-graph nil))
  (let ((view (graph-view window)))
    (without-interrupts
     (display-tree-cells view (root view))
     (size-all-cells view)
     (position-cells view)
     (grow-window-if-needed window)
     (if position-graph
       (position-graph view :centered t)
       (re-position-graph view :centered t))
     (auto-position-window window)
     ;;(reset-scroll-bars (graph-scroller window))
     (configure-controls window)
     (if (and  (edit-window-analyzed window) (null *retain-analysis-while-editing*))
       (undo-analyze window)
       (invalidate-view window t)))))
 

;;;=========================================================
;;;  PRIMATIVE EDITOR FUNCTION

;;;_______________________________________
;;;  only-one

(defun only-one (&rest lists)
  (let ((node nil))
    (dolist (list lists)
      (cond ((null list) nil)
            ((null (rest list)) (setf node (if node t (first list))))
            (t (setf node t))))
    (when (node-p node) node)))

;;;_______________________________________
;;; hide-antecedents

(defun hide-antecedents (node-or-collection graph)
  (cond ((node-p node-or-collection)
         (case (node-kind node-or-collection)
           (:intensional (disconnect-and-free-tree (node-antecedents node-or-collection) graph)
                         (setf (node-antecedents node-or-collection) nil
                               (node-recursive? node-or-collection) t))
           (:not (hide-antecedents (node-antecedents node-or-collection) graph))
           (otherwise nil)))
        ((consp node-or-collection)
         (hide-antecedents (first node-or-collection) graph)
         (hide-antecedents (rest node-or-collection) graph))))

;;;_______________________________________
;;; order-nodes-based-on-screen-positions

(defun order-nodes-based-on-cell-positions (node-or-collection view &optional (recursive nil))
  (if (node-p node-or-collection)
    (when (or (node-second-order? node-or-collection) recursive)
      (setf (node-antecedents node-or-collection)
            (order-nodes-based-on-cell-positions (node-antecedents node-or-collection) view recursive)))
    (let ((ordered node-or-collection)
          (conjunctions-with-some-node-selected 0)
          (conjunctions-with-every-node-selected 0))
      (dolist (conjunction node-or-collection)
        (dolist (node conjunction)
          (order-nodes-based-on-cell-positions node view recursive))
        (when (some #'node-selected? conjunction)
          (incf conjunctions-with-some-node-selected)
          (when (every #'node-selected? conjunction)
            (incf conjunctions-with-every-node-selected))))

      (when (and (= conjunctions-with-some-node-selected 1)
                 (= conjunctions-with-every-node-selected 1))
        (setf ordered (stable-sort node-or-collection
                                   #'(lambda (disjunct1 disjunct2) (cell-preceeds (node-cell view (first disjunct1))
                                                                                  (node-cell view (first disjunct2)))))))
      (when (and (= conjunctions-with-some-node-selected 1)
                 (= conjunctions-with-every-node-selected 0))
        (do* ((conjunctions node-or-collection (rest conjunctions))
              (conjunction (first conjunctions) (car conjunctions)))
             ((null conjunctions))
          (when (rest conjunction)
            (rplaca conjunctions
                    (stable-sort conjunction
                                 #'(lambda (node1 node2) (cell-preceeds (node-cell view node1) (node-cell view node2))))))))
      ordered)))

;;;_______________________________________
;;; cell-preceeds

(defun cell-preceeds (cell1 cell2)
  (< (cell-top cell1) (cell-top cell2)))

;;;_______________________________________
;;;  collect-vars-and-types

(defun collect-vars-and-types (node)
  (let ((vars nil)
        (types nil))
    (labels
      ((node-vars-and-types (node)
         (when (node-r-struct node)
           (do ((vs (node-vars node) (rest vs))
                (ts (r-type (node-r-struct node)) (rest ts)))
               ((null vs))
             (unless (member (first vs) vars)
               (push (first vs) vars)
               (push (first ts) types))))
         (disjunction-vars-and-types (node-antecedents node)))
      (conjunction-vars-and-types (conjunction)
        (dolist (node conjunction)
          (node-vars-and-types node)))
      (disjunction-vars-and-types (disjunction)
        (dolist (node disjunction)
          (conjunction-vars-and-types node))))
      (node-vars-and-types node))
    (values vars types)))

;;;_______________________________________
;;; swap-r-aux

(defun swap-r-aux (node)
  (let ((original (node-r-struct node)))
    (when (r-p original)
      (setf (node-aux node) original
            (node-r-struct node) (copy-r original)))))

;;;=========================================================
;;; RULE EDITOR

;;;_______________________________________
;;; rule-edit-window

(defclass rule-edit-window (edit-window)
  ((original-r-struct :initarg :original-r-struct :initform nil :accessor original-r-struct)
   (modified-r-struct :initarg :modified-r-struct :initform nil :accessor modified-r-struct)))

;;;_______________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window rule-edit-window) &rest initargs)
  (setf (getf initargs :view-size) #@(540 350))
  (apply #'call-next-method window initargs)
  (let ((x 5) (y 30)
        (button-size #@(70 18))
        (button-step 22)
        (button-font '("Chicago" 12 :plain))
        (space-step 30))
    (add-subviews 
     window
     (make-dialog-item
      'button-dialog-item (make-point x y) button-size " Edit Def "
      #'(lambda (item) (editor-EDIT-DEF (view-container item)))
      :view-font button-font :view-nick-name :edit-def)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Edit Call "
      #'(lambda (item) (editor-EDIT-CALL (view-container item)))
      :view-font button-font :view-nick-name :edit-call)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Create "
      #'(lambda (item) (editor-CREATE (view-container item)))
      :view-font button-font :view-nick-name :create)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Copy "
      #'(lambda (item) (editor-COPY (view-container item)))
      :view-font button-font :view-nick-name :copy)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Negate "
      #'(lambda (item) (editor-NEGATE (view-container item)))
      :view-font button-font :view-nick-name :negate)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Attach "
      #'(lambda (item) (editor-ATTACH (view-container item)))
      :view-font button-font :view-nick-name :attach)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Replace "
      #'(lambda (item) (editor-REPLACE (view-container item)))
      :view-font button-font :view-nick-name :replace)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Delete "
      #'(lambda (item) (editor-DELETE (view-container item)))
      :view-font button-font :view-nick-name :delete)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Show "
      #'(lambda (item) (editor-SHOW (view-container item)))
      :view-font button-font :view-nick-name :show)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Hide "
      #'(lambda (item) (editor-HIDE (view-container item)))
      :view-font button-font :view-nick-name :hide)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Analyze "
      #'(lambda (item) (editor-ANALYZE (view-container item)))
      :view-font button-font :view-nick-name :analyze)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y space-step)) button-size " Cancel "
      #'(lambda (item) (editor-CANCEL (view-container item)))
      :view-font button-font :view-nick-name :cancel)
     (make-dialog-item
      'button-dialog-item (make-point x (incf y button-step)) button-size " Define "
      #'(lambda (item) (editor-DEFINE (view-container item)))
      :view-font button-font :view-nick-name :define)
     )))

;;;_______________________________________
;;;  configure-controls

(defmethod configure-controls ((window rule-edit-window))
  (let* ((view (graph-view window))
         (graph (graph view))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph))
         (a-nodes (selected-nodes a-base t))
         (u-nodes (selected-nodes u-base t))
         (intensional-selected nil)
         (expanded-intensional-selected nil)
         
         (node (only-one a-nodes u-nodes))
         (a-conjunction (antecedent-conjunction-of a-base a-nodes))
         (u-conjunction (antecedent-conjunction-of u-base u-nodes)))
    
    (dolist (node (graph-used-nodes graph))
      (when (node-selected? node)
        (when (node-intensional? node)
          (setf intensional-selected t)
          (when (and (node-antecedents node) (not (eq node a-base)))
              (setf expanded-intensional-selected t)))))
    
    (if (and node (not (node-not? node)))
      (dialog-item-enable (view-named :edit-def window))
      (dialog-item-disable (view-named :edit-def window)))
    
    (if (and node (antecedent-of a-base node) (not (node-not? node)))
      (dialog-item-enable (view-named :edit-call window))
      (dialog-item-disable (view-named :edit-call window)))
    
    (if (or a-nodes u-nodes)
      (dialog-item-enable (view-named :copy window))
      (dialog-item-disable (view-named :copy window)))
    
    (if (or a-conjunction
            u-conjunction
            (and node (or (antecedent-of a-base node)
                          (antecedent-of u-base node))))
      (dialog-item-enable (view-named :negate window))
      (dialog-item-disable (view-named :negate window)))
    
    (if (and (or a-conjunction (eq (only-one a-nodes) a-base))
             u-nodes)
      (dialog-item-enable (view-named :attach window))
      (dialog-item-disable (view-named :attach window)))
    
    (if (and node
             (node-antecedents node)
             (or (antecedent-of a-base node)
                 (antecedent-of u-base node)))
      (dialog-item-enable (view-named :replace window))
      (dialog-item-disable (view-named :replace window)))
    
    (if (and (or a-nodes u-nodes)
             (every #'(lambda (n) (antecedent-of a-base n)) a-nodes)
             (every #'(lambda (n) (antecedent-of u-base n)) u-nodes))
      (dialog-item-enable (view-named :delete window))
      (dialog-item-disable (view-named :delete window)))
    
    (if intensional-selected
      (dialog-item-enable (view-named :show window))
      (dialog-item-disable (view-named :show window)))
    
    (if expanded-intensional-selected
      (dialog-item-enable (view-named :hide window))
      (dialog-item-disable (view-named :hide window)))

    (if (some #'(lambda (n) (some #'coverage-p (node-coverage n))) (edit-window-nodes window))
      (dialog-item-enable (view-named :analyze window))
      (dialog-item-disable (view-named :analyze window)))

    ))

;;;_______________________________________
;;; reorder-nodes

(defmethod reorder-nodes ((window rule-edit-window))
  (let* ((view (graph-view window))
         (graph (window-graph window))
         (a-base (graph-a-base graph)))
    (setf (node-antecedents a-base) (order-nodes-based-on-cell-positions (node-antecedents a-base) view))
    (position-cells view)
    (invalidate-view view t)))

;;;_______________________________________
;;;   view-key-event-handler

(defmethod view-key-event-handler ((window rule-edit-window) char)
  (let ((view (graph-view window)))
    (case (char-code char)
      ((67 99) (editor-COPY window))                       ;;; C c
      ((69 101) (editor-EDIT-DEF window))                  ;;; E e
      ((72 104) (editor-HIDE window))                      ;;; H h
      ((73 105) (inspect-last-node-selected view))         ;;; I i
      ((82 114) (undo-analyze window))                     ;;; R r
      ((83 115) (editor-SHOW window))                      ;;; S s
      ((84 116) (show-tuples-last-node-selected view))     ;;; T t
      (others nil))))

;;;_______________________________________
;;;  editor-EDIT-DEF

(defmethod editor-EDIT-DEF ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (node (first (selected-nodes (graph-root graph) t))))
    (catch-cancel
      (if (eq node a-base)
        (let ((r-struct (modified-r-struct window)))
          (when (user-monitor-p *user-monitor*)
            (incf (user-monitor-rule-editor-edit-definition *user-monitor*)))
          (setf r-struct (user-modify-relation r-struct)
                (modified-r-struct window) r-struct
                (node-vars a-base) (r-vars r-struct)
                (node-r-struct a-base) r-struct
                (cell-text (node-cell (graph-view window) a-base)) (node-string a-base)))
        (user-edit-relation (node-r-struct node)))
      (window-fixup window))))

;;;_______________________________________
;;;  editor-EDIT-CALL

(defmethod editor-EDIT-CALL ((window rule-edit-window))
  (user-create-nodes window (first (or (selected-nodes (graph-a-base (window-graph window)) t)
                                       (selected-nodes (graph-u-base (window-graph window)) t)))))


;;;_______________________________________
;;;  editor-CREATE

(defmethod editor-CREATE ((window rule-edit-window))
  (user-create-nodes window nil))

;;;_______________________________________
;;;  editor-COPY

(defmethod editor-COPY ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (u-base (graph-u-base graph)))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-copy *user-monitor*)))
    (dolist (node (selected-nodes graph t))
      (if (node-antecedents u-base)
        (rplaca (node-antecedents u-base) (nconc (first (node-antecedents u-base)) (list (duplicate-node node u-base graph t))))
        (setf (node-antecedents u-base) (list (list (duplicate-node node u-base graph t)))))
      (deselect-node node t))
    (apply-function-to-tree u-base #'(lambda (node) (unless (eq (node-kind node) :undefined)
                                                      (setf (node-state node) *new-state*
                                                            (node-aux node) *new-state*))))
    (window-fixup window)))

;;;_______________________________________
;;;  editor-NEGATE

(defmethod editor-NEGATE ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph))
         (nodes (selected-nodes graph t))
         (conjunction (conjunction-containing-node (first nodes))))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-negate *user-monitor*)))
    (cond ((or (antecedent-conjunction-of a-base nodes)
               (antecedent-conjunction-of u-base nodes))
           (negate-conjunction conjunction graph t))
          ((only-one nodes) (negate-node (first nodes) graph t)))
    (let ((consequent (node-consequent (first nodes))))
      (setf (node-state consequent) *new-state*
            (node-aux consequent) *new-state*))
    (window-fixup window)))

;;;_______________________________________
;;;  editor-ATTACH

(defmethod editor-ATTACH ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (a-nodes (selected-nodes a-base t))
         (u-nodes (selected-nodes (graph-u-base graph) t)))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-attach *user-monitor*)))
    (cond ((node-selected? a-base) (disjoin a-base u-nodes graph))
          ((antecedent-conjunction-of a-base a-nodes) (conjoin a-nodes u-nodes graph)))
    (window-fixup window)))

;;;_______________________________________
;;;  editor-REPLACE

(defmethod editor-REPLACE ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph))
         (nodes (selected-nodes graph t))
         (node (first nodes)))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-replace *user-monitor*)))
    (when (and (only-one nodes) (node-antecedents node)
               (or (antecedent-of a-base node)
                   (antecedent-of u-base node)))
      (replace-node node graph)
      (window-fixup window))))

;;;_______________________________________
;;;  editor-DELETE

(defmethod editor-DELETE ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (u-base (graph-u-base graph)))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-delete *user-monitor*)))
    (dolist (node (selected-nodes graph t))
      (when (or (antecedent-of a-base node)
                (antecedent-of u-base node))
        (disconnect-and-free-tree node graph)))
    (window-fixup window)))

;;;_______________________________________
;;;  editor-SHOW

(defmethod editor-SHOW ((window rule-edit-window))
  (let ((graph (window-graph window))
        (fix? nil))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-show *user-monitor*)))
    (dolist (node (selected-nodes graph))
      (when (and (node-intensional? node)
                 (node-r-struct node)
                 (null (node-antecedents node)))
        (setf (node-antecedents node) (connect-clauses graph node (editor-all-antecedents (node-r-struct node) (node-vars node)) (node-state node) :never 0)
              (node-recursive? node) nil
              fix? t)
        (dolist (a (node-antecedents node))
          (dolist (n a)
            (setf (node-aux n) (node-aux node)))))
      (dolist (a (node-antecedents node))
        (dolist (n a)
          (setf (node-selected? n) t))))
    (cond (fix? (window-fixup window))
          (t (configure-controls window)
             (invalidate-view (graph-view window))))))

;;;_______________________________________
;;;  editor-HIDE

(defmethod editor-HIDE ((window rule-edit-window))
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-rule-editor-hide *user-monitor*)))
  (let ((graph (window-graph window)))
    (dolist (node (selected-nodes (graph-u-base graph)))
      (hide-antecedents node graph))
    (dolist (node (selected-nodes (node-antecedents (graph-a-base graph))))
      (hide-antecedents node graph)))
  (window-fixup window))

;;;_______________________________________
;;;  editor-ANALYZE

(defmethod editor-ANALYZE ((window rule-edit-window))
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-rule-editor-analyze *user-monitor*)))
  (let* ((graph (window-graph window))
         (base (graph-a-base graph))
         (view (graph-view window))
         (*batch* t)
         coverage)
    (with-focused-view view
      (set-view-font window (view-font view))
      (undo-analyze window)
      (deselect-node (graph-root graph) t)
      (dolist (node (edit-window-nodes window))
        (when (coverage-p (setf coverage (first (node-coverage node))))
          (multiple-value-bind (input-pos input-neg input-vars input-type) (fix-constants-in-tuples-for-editor base node coverage)
            (insert-node-tuples (list base) input-pos input-neg input-vars input-type node view t t))))
      (hilight-and-display-tree-coverage (graph-base graph) view window)
      (let ((original-pos 0)
            (original-neg 0)
            (extended-pos 0)
            (extended-neg 0)
            (original-pos-covered 0)
            (original-neg-covered 0))
        (dolist (coverage (node-coverage base))
          (incf original-pos (length (coverage-input-pos coverage)))
          (incf original-neg (length (coverage-input-neg coverage)))
          (incf extended-pos (length (coverage-output-pos coverage)))
          (incf extended-neg (length (coverage-output-neg coverage)))
          (incf original-pos-covered (count-originals-extended (coverage-input-pos coverage) (coverage-output-pos coverage)))
          (incf original-neg-covered (count-originals-extended (coverage-input-neg coverage) (coverage-output-neg coverage))))
        (set-dialog-item-text (view-named :original-coverage window)
                                (format nil "              Covered  ~A/~A+  ~A/~A-     Extended  ~A+  ~A-"
                                        original-pos-covered original-pos
                                        original-neg-covered original-neg
                                        extended-pos extended-neg)))
        (setf (edit-window-analyzed window) t)
        (invalidate-view window t))))

;;;_______________________________________
;;;  undo-ANALYZE

(defun undo-analyze (window)
  (let* ((view (graph-view window))
         (root (root view)))
    (apply-function-to-tree root #'(lambda (node) (setf (node-state node) (node-aux node))))
    (clear-external-text view)
    (invalidate-view view t)))

;;;_______________________________________
;;;  editor-CANCEL
    
(defmethod editor-CANCEL ((window edit-window))
  (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-rule-editor-cancel *user-monitor*)))
  (window-close window))

;;;_______________________________________
;;;  editor-DEFINE

(defmethod editor-DEFINE ((window rule-edit-window))
  (let* ((graph (window-graph window))
         (a-base (graph-a-base graph))
         (antecedents (node-antecedents a-base))
         (original (original-r-struct window))
         (original-name (r-name (original-r-struct window)))
         (modified (modified-r-struct window))
         (name (r-name modified))
         (vars (r-vars modified))
         new-r-struct)
    (when (or (equal original-name name)
              (null (setf original (get-r-struct name)))
              (eq (r-kind original) :undefined)
              (y-or-n-dialog (format nil "There is already a relation named ~A.~%Do you want to change its definition?" name) :position :centered :cancel-text nil))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-rule-editor-define *user-monitor*)))
      (deselect-node a-base t)
      (select-node antecedents)
      (setf (r-clauses modified) (convert-antecedents-to-clauses antecedents name vars)
            new-r-struct (def-from-r-struct modified)
            *rules-changed* t)
      (window-close window)
      (update-windows new-r-struct original))))

(defun convert-antecedents-to-clauses (antecedents rule-name vars)
  (correct-singleton-variables-in-head
   (mapcar #'(lambda (conjunction)
               (let ((prolog (convert-tree-to-prolog conjunction)))
                 (if (and (eql (first prolog) 'and)
                          (not (node-and? (first conjunction))))
                   (rplaca prolog (cons rule-name vars))
                   (list (cons rule-name vars) prolog))))
           antecedents)))

(defun correct-singleton-variables-in-head (clauses)
  (dolist (clause clauses)
    (let ((head (first clause))
          (body (rest clause)))
      (rplacd head (mapcar #'(lambda (var)
                               (if (pcvar-p var)
                                 (if (used-in body var)
                                   var
                                   (let ((id (pcvar-id var)))
                                     (if (eq #\_ (char (format nil "~S" id) 0))
                                       var
                                       (intern (format nil "?_~a" id)))))
                                 var))
                           (rest head)))))
  clauses)

;;;_______________________________________
;;; updates-windows

(defun update-windows (new-r-struct old-r-struct)
  (when (and (r-p new-r-struct)
             (r-p old-r-struct))
    (let ((analyze-window-class (find-class 'analyze-window)))
      (with-cursor *watch-cursor*
        (setf (r-nodes new-r-struct) (r-nodes old-r-struct))
        (dolist (node (r-nodes old-r-struct))
          (setf (node-r-struct node) new-r-struct
                (node-kind node) (r-kind new-r-struct))
          (when (eq (node-state node) :undefined)
            (setf (node-state node) nil
                  (node-aux node) nil))
          (let ((graph (node-graph node)))
            (when (graph-p graph)
              (let ((views (graph-views graph)))
                (when (node-antecedents node)
                  (disconnect-and-free-tree (node-antecedents node) graph)
                  (setf (node-antecedents node) (connect-clauses graph node (editor-all-antecedents new-r-struct (node-vars node)) (node-state node) :every-use 0)))
                (dolist (view views)
                  (let ((window (graph-window view)))
                    (display-tree-cells view)
                    (size-all-cells view)
                    (position-cells view)
                    (grow-window-if-needed window)
                    (re-position-graph view :centered t)
                    (auto-position-window window)
                    (when (eq (class-of window) analyze-window-class)
                      (deanalyze-path-containing-node node view)))
                  (invalidate-view view t))))))
        (update-relations))
      
      (when *reanalyze-automatically*
        (let ((windows nil))
          (dolist (node (r-nodes old-r-struct))
            (when (graph-p (node-graph node))
              (dolist (view (graph-views (node-graph node)))
                (pushnew (graph-window view) windows))))
          (dolist (window windows)
            (when (eq (class-of window) analyze-window-class)
              (eval-enqueue `(re-analyze-coverage ',window)))))))))


;;;=========================================================
;;; LITERAL AND RELATION DIALOGS

;;;_______________________________________
;;;  user-modify-relation

(defun user-modify-relation (r-struct)
  (let* ((h 350) (v 200) (x 10) (x1 60) (y 10) (w (- h x1 x))
         (questions (r-questions r-struct))
         (editable-text-h 12) (user-font '("Monaco" 9 :plain))
         (window (make-instance 
                   'window
                   :window-type :document
                   :close-box-p nil
                   :window-title (format nil  "Edit ~S" (r-name r-struct))
                   :window-show nil
                   :view-position '(:top 50)
                   :view-size (make-point h v))))
    (add-subviews 
     window
     (make-dialog-item
      'static-text-dialog-item (make-point x y) nil "Name" nil)
     (make-dialog-item
      'editable-text-dialog-item (make-point x1 y) (make-point w editable-text-h)
      (if (r-name r-struct) (format nil "~(~S~)" (r-name r-struct)) "new_relation") nil :allow-returns nil
      :view-font user-font :view-nick-name :name)
     (make-dialog-item
      'static-text-dialog-item (make-point x (incf y 20)) nil "Vars" nil)
     (make-dialog-item
      'editable-text-dialog-item (make-point x1 (incf y 1)) (make-point w editable-text-h)
      (if (r-vars r-struct) (format nil "~S" (r-vars r-struct)) "()") nil :allow-returns nil
      :view-font user-font :view-nick-name :vars)
     (make-dialog-item
      'static-text-dialog-item (make-point x (incf y 20)) nil "Type" nil)
     (make-dialog-item
      'editable-text-dialog-item (make-point x1 (incf y 1)) (make-point w editable-text-h)
      (if (r-type r-struct) (format nil "~S" (r-type r-struct)) "()") nil :allow-returns nil
      :view-font user-font :view-nick-name :type)
     (make-dialog-item
      'static-text-dialog-item (make-point x (incf y 20)) nil "Mode" nil)
     (make-dialog-item
      'editable-text-dialog-item (make-point x1 (incf y 1)) (make-point w editable-text-h)
      (if (r-mode r-struct) (format nil "~S" (r-mode r-struct)) "()") nil :allow-returns nil
      :view-font user-font :view-nick-name :mode)
     (make-dialog-item
      'static-text-dialog-item (make-point x (incf y 20)) nil "Det" nil)
     (make-dialog-item
      'editable-text-dialog-item (make-point x1 (incf y 1)) (make-point w editable-text-h)
      (if (r-determinacy r-struct) (format nil "~S" (r-determinacy r-struct)) "()") nil :allow-returns nil
      :view-font user-font :view-nick-name :determinacy)
     (make-dialog-item
      'check-box-dialog-item (make-point x (incf y 22)) nil "induction" nil
      :check-box-checked-p (r-induction r-struct) :view-nick-name :induction)
     (make-dialog-item
      'check-box-dialog-item (make-point x (incf y 16)) nil "commutative" nil
      :check-box-checked-p (r-commutative r-struct) :view-nick-name :commutative)
     (make-dialog-item
      'check-box-dialog-item (make-point x (incf y 16)) nil "constraint" nil
      :check-box-checked-p (r-constraint r-struct) :view-nick-name :constraint)
     (make-dialog-item
      'button-dialog-item (make-point x (- v 30)) #@(90 20) " English "
      #'(lambda (item) (setf questions (edit-translation (value-from-dialog-item-text (find-named-sibling item :name))
                                                         (value-from-dialog-item-text (find-named-sibling item :vars))
                                                         :intensional
                                                         questions))))
     (make-dialog-item
      'button-dialog-item (make-point (- h 140) (- v 30)) #@(60 20) " Change "
      #'(lambda (item)
          (setf (r-name r-struct) (value-from-dialog-item-text (find-named-sibling item :name))
                (r-vars r-struct) (value-from-dialog-item-text (find-named-sibling item :vars))
                (r-type r-struct) (value-from-dialog-item-text (find-named-sibling item :type))
                (r-mode r-struct) (value-from-dialog-item-text (find-named-sibling item :mode))
                (r-determinacy r-struct) (value-from-dialog-item-text (find-named-sibling item :determinacy))
                (r-kind r-struct) :intensional
                (r-questions r-struct) questions
                (r-induction r-struct) (check-box-checked-p (find-named-sibling item :induction))
                (r-commutative r-struct) (check-box-checked-p (find-named-sibling item :commutative))
                (r-constraint r-struct) (if (check-box-checked-p (find-named-sibling item :constraint)) :unique-vars)
                )
          (return-from-modal-dialog r-struct))
      :default-button t)
     (make-dialog-item
      'button-dialog-item  (make-point (- h 70) (- v 30)) #@(60 20) " Cancel "
      #'(lambda (item) item (return-from-modal-dialog :cancel))
      :default-button nil))
    (let ((literal (view-named :name window)))
      (collapse-selection literal t)
      (fred-update literal))
    (modal-dialog window t)))

;;;_____________________________________
;;;  create-nodes-window

(defclass create-nodes-window (window) ())

;;;_____________________________________
;;;  move-selectors-and-buttons

(defmethod move-selectors-and-buttons ((window create-nodes-window))
  (without-interrupts
   (with-focused-view window
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
       (#_eraserect view-rect)
       (#_beginupdate (wptr window))
       (let* ((length (point-v (view-size window)))
              (width (point-h (view-size window)))
              (selector-length (round (- length 80) 3))
              (selector-width (- width 8))
              (selector-cell-width (- selector-width 15))
              (relations (view-named :relations window))
              (variables-caption (view-named :variables-caption window))
              (variables (view-named :variables window))
              (literal-caption (view-named :literal-caption window))
              (literal (view-named :literal window))
              (create (view-named :create window))
              (erase (view-named :erase window))
              (done (view-named :done window)) )
         (set-view-size relations selector-width selector-length)
         (set-cell-size relations selector-cell-width 14)
         
         (set-view-position variables-caption (add-points (view-position relations) (make-point 0 (+ selector-length 1))))
         (set-view-position variables (add-points (view-position variables-caption) (make-point 0 16)))
         (set-view-size variables selector-width selector-length)
         (set-cell-size variables selector-cell-width 14)
         
         (set-view-position literal-caption (add-points (view-position variables) (make-point 0 (+ selector-length 1))))
         (set-view-position literal (add-points (view-position literal-caption) (make-point 0 17)))
         (set-view-size literal (- selector-width 15) (- selector-length 21))
         
         (set-view-position erase (- width 220) (- length 26))
         (set-view-position done  (- width 155) (- length 26))
         (set-view-position create (- width 85) (- length 26))

         (#_endupdate (wptr window))
         (#_invalrect view-rect))))))

;;;_____________________________________
;;;  set-view-size

(defmethod set-view-size ((window create-nodes-window) h &optional v)
  (without-interrupts
   (call-next-method window h v)
   (move-selectors-and-buttons window)))

;;;_____________________________________
;;;  window-zoom-event-handler

(defmethod window-zoom-event-handler ((window create-nodes-window) message)
  (without-interrupts
   (call-next-method window message)
   (move-selectors-and-buttons window)))

;;;_______________________________________
;;;  view-key-event-handler

(defmethod view-key-event-handler ((window create-nodes-window) char)
  (if (and (shift-key-p) (control-key-p))
    (let* ((relations (view-named :relations window))
           (upper-case (char-upcase (code-char (+ (char-code char) 64)))))
      
      (if (and (string<= #\A upper-case) (string<= upper-case #\Z))
        (scroll-to-cell relations 0 (or (position-if #'(lambda (table-element) 
                                                         (string<= upper-case (elt (symbol-name (r-name table-element)) 0)))
                                                     (table-sequence relations))
                                        (- (point-v (table-dimensions relations)) 1)))
        (call-next-method window char)))
    (call-next-method window char)))

;;;_______________________________________
;;;  create-node-insert

(defun create-node-insert (string literal-item)
  (clear literal-item)
  (collapse-selection literal-item t)
  (let* ((position (selection-range literal-item))
         (mark (fred-buffer literal-item))
         (size (buffer-size mark))
         (prev (if (or (<= position 1) (>= position size))
                 #\space
                 (buffer-char mark (buffer-position mark (- position 1)))))
         (next (if (or (<= position 0) (>= position size))
                 #\space
                 (buffer-char mark (buffer-position mark position)))))
    (cond
     ((and (or (eql prev #\() (eql prev #\space))
           (or (eql next #\)) (eql next #\space)))
      (buffer-insert mark (format nil "~A" string)))
     ((or (eql prev #\() (eql prev #\space))
      (buffer-insert mark (format nil "~A " string)))
     ((or (eql next #\)) (eql next #\space))
      (buffer-insert mark (format nil " ~A" string)))
     (t
      (buffer-insert mark (format nil " ~A " string)))))
  (fred-update literal-item))

;;;_______________________________________
;;;  user-create-nodes

(defun user-create-nodes (window current-node &optional (swap-r-aux nil))
  (let* ((h 400) (v 270) (w 392)
         (input-font '("Monaco" 9 :plain))
         (graph (window-graph window))
         (u-base (graph-u-base graph))
         (dialog (make-instance 
                   'create-nodes-window
                   :window-title (if current-node "Modify Literal" "Create Literal")
                   :window-show nil
                   :view-size (make-point h v)
                   :view-position '(:top 50)
                   )))
    (add-subviews 
     dialog
     (make-dialog-item
      'static-text-dialog-item (make-point 4 0) (make-point w 15) "Relations" nil)
     (make-dialog-item
      'sequence-dialog-item (make-point 4 16) (make-point w 56) "Relations"
      #'(lambda (item)
          (let ((selected-cell (car (selected-cells item))))
            (when selected-cell
              (let* ((literal (find-named-sibling item :literal))
                     (defined-vars (table-sequence (find-named-sibling item :variables)))
                     (r-struct (cell-contents item selected-cell))
                     (vars (r-vars r-struct))
                     (all-vars-defined (and vars (every #'(lambda (v) (member v defined-vars)) vars)))
                     (relation-string (if (r-type r-struct)
                                        (if all-vars-defined
                                          (format nil "(~(~S~)~{ ~S~})" (r-name r-struct) vars)
                                          (format nil "(~(~S~) )" (r-name r-struct)))
                                        (format nil "~S" (r-name r-struct)))))
                (create-node-insert relation-string literal)
                (cell-deselect item selected-cell)
                (collapse-selection literal t)
                (unless (or (null (r-type r-struct)) all-vars-defined) 
                  (move-mark (fred-buffer literal) -1))
                (fred-update literal)))))
      :cell-size (make-point (- w 15) 14)
      :table-hscrollp nil
      :table-vscrollp t
      :table-sequence *r-structs*
      :table-print-function #'pretty-print-r-struct-name
      :selection-type :single
      :view-font input-font
      :view-nick-name :relations)
     
     (make-dialog-item
      'static-text-dialog-item (make-point 4 73) (make-point w 15) "Variables and Constants" nil :view-nick-name :variables-caption)
     
     (make-dialog-item
      'sequence-dialog-item (make-point 4 89) (make-point w 42) "Variables"
      #'(lambda (item)
          (let ((selected-cell (first (selected-cells item)))
                (literal (find-named-sibling item :literal)))
            (when selected-cell
              (create-node-insert (format nil "~S" (cell-contents item selected-cell)) literal)
              (cell-deselect item selected-cell))))
      :cell-size (make-point (- w 15) 14)
      :table-hscrollp nil
      :table-vscrollp t
      :table-sequence (sort (collect-vars-and-types (graph-root graph)) #'universal<)
      :table-print-function #'(lambda (var stream) (format stream "~S" var))
      :selection-type :single
      :view-font input-font
      :view-nick-name :variables)
     
     (make-dialog-item
      'static-text-dialog-item (make-point 4 132) (make-point w 15) "Literal    - for example -   (age ?PERSON 21)" nil :view-nick-name :literal-caption)
     
     (make-dialog-item
      'ccl::scrolling-fred-dialog-item (make-point 6 149) (make-point (- h 27) (- v 197)) "" nil
      :allow-returns t :allow-tabs t :view-font input-font :view-nick-name :literal)
     
     (make-dialog-item
      'button-dialog-item (make-point (- h 215) (- v 26)) #@(60 20) " Erase "
      #'(lambda (item) (set-dialog-item-text (find-named-sibling item :literal) ""))
      :default-button nil
      :dialog-item-enabled-p t
      :view-nick-name :erase)
     
     (make-dialog-item
      'button-dialog-item (make-point (- h 145) (- v 26)) #@(60 20) (if current-node " Cancel " " Done ")
      #'(lambda (item) item (return-from-modal-dialog nil))
      :default-button nil
      :dialog-item-enabled-p t
      :view-nick-name :done)
     
     (make-dialog-item
      'button-dialog-item (make-point (- h 80) (- v 26)) #@(60 20) (if current-node " Change " " Create ")
      #'(lambda (item)
          (multiple-value-bind (new-literals conjunction-p)
                               (read-prolog-from-string (dialog-item-text (find-named-sibling item :literal)) nil current-node)
            (when new-literals
              (when (user-monitor-p *user-monitor*)
                (incf (user-monitor-rule-editor-create-literal *user-monitor*)))
              (cond (current-node
                     (let* ((consequent (node-consequent current-node))
                            (new-node (connect-literal graph consequent new-literals nil *new-state* :never nil))
                            (nodes-to-free (nodes-in-tree current-node)))
                       (setf (node-aux new-node) *new-state*
                             (node-antecedents consequent) (nsubst new-node current-node (node-antecedents consequent))
                             (node-selected? new-node) t)
                       (when swap-r-aux (swap-r-aux new-node))
                       (dolist (node nodes-to-free)
                         (reset-node node))
                       (setf (graph-free-nodes graph) (nconc (graph-free-nodes graph) nodes-to-free))
                       (window-fixup window)
                       (return-from-modal-dialog nil)))
                    (conjunction-p
                     (let ((new-nodes (connect-clause graph u-base new-literals nil *new-state* :never nil)))
                       (mapcar #'(lambda (n) (setf (node-aux n) *new-state*)) new-nodes)
                       (when swap-r-aux (mapcar #'swap-r-aux new-nodes))
                       (setf (node-antecedents u-base) (nconc (node-antecedents u-base) (list new-nodes)))
                       (mapcar #'(lambda (n) (setf (node-selected? n) t)) new-nodes)
                       (set-table-sequence (find-named-sibling item :variables) (sort (collect-vars-and-types (graph-root graph)) #'universal<))
                       (window-fixup window)))
                    (t
                     (let ((new-node (connect-literal graph u-base new-literals nil *new-state* :never nil)))
                       (setf (node-aux new-node) *new-state*)
                       (when swap-r-aux (swap-r-aux new-node))
                       (setf (node-antecedents u-base) (nconc (node-antecedents u-base) (list (list new-node)))
                             (node-selected? new-node) t)
                       (set-table-sequence (find-named-sibling item :variables) (sort (collect-vars-and-types (graph-root graph)) #'universal<))
                       (window-fixup window)))))))
      :default-button t
      :dialog-item-enabled-p t
      :view-nick-name :create)
     )
    
    (let ((literal-item (view-named :literal dialog)))
      (set-dialog-item-text literal-item (if current-node (format nil "~S" (convert-tree-to-prolog current-node)) ""))
      (collapse-selection literal-item t)
      (fred-update literal-item))
    (move-selectors-and-buttons dialog)
    (modal-dialog dialog t)))

;;;_______________________________________
;;;  read-prolog-from-string

(defun read-prolog-from-string (string &optional (must-be-conjunction nil)
                                       (must-be-literal nil)
                                       (no-warnings nil))
  "Takes a string as input, returns a list of the prolog literals contained in the string"
  (multiple-value-bind (value error) (catch-error-quietly (read-from-string string))
    (let ((conjunction-p nil))
      (unless error
        (cond ((consp value)
               (cond ((equal value '(fail)))
                     ((consp (first value))   ;; value is a conjunction
                      (setf conjunction-p t)
                      (unless (every #'(lambda (l) (or (and (consp l) (not (consp (first l)))) (eq l '!))) value)
                        (setf error "Not a valid conjunction of literals.")))
                     (t                       ;; value is a literal
                      (unless (rest value)
                        (setf error "Not a valid literal.")))))
              (t
               (case value 
                 ((! and or not fail) nil)
                 (otherwise (setf error "Not a valid literal."))))))
      (cond (error
             (unless no-warnings
               (notify-error "~%~a~%  ~a ill-formed literal" error string))
             nil)
            ((and must-be-conjunction (not conjunction-p))
             (unless no-warnings
               (notify-error "~%~a~%  ~a is not a conjunction." "Input is supposed to be a conjunction of literals." string))
             nil)
            ((and must-be-literal conjunction-p)
             (unless no-warnings
               (notify-error "~%~a~%  ~a is a conjunction." "Input is supposed to be a literal." string))
             nil)
            (t
             (values value conjunction-p))))))

;;;_______________________________________
;;;  user-create-literals

(defun user-create-literals (current-literal head)
  (let* ((h 400) (v 270) (w 392)
         (input-font '("Monaco" 9 :plain))
         (dialog (make-instance 
                   'create-nodes-window
                   :window-title (format nil "~A for ~A" (if current-literal "Modify Literal(s)" "Create Literal(s)") head)
                   :window-show nil
                   :view-size (make-point h v)
                   :view-position '(:top 50)
                   )))
    (add-subviews 
     dialog
     (make-dialog-item
      'static-text-dialog-item (make-point 4 0) (make-point w 15) "Relations" nil)
     (make-dialog-item
      'sequence-dialog-item (make-point 4 16) (make-point w 56) "Relations"
      #'(lambda (item)
          (let ((selected-cell (car (selected-cells item))))
            (when selected-cell
              (let* ((literal (find-named-sibling item :literal))
                     (defined-vars (table-sequence (find-named-sibling item :variables)))
                     (r-struct (cell-contents item selected-cell))
                     (vars (r-vars r-struct))
                     (all-vars-defined (and vars (every #'(lambda (v) (member v defined-vars)) vars)))
                     (relation-string (if (r-type r-struct)
                                        (if all-vars-defined
                                          (format nil "(~(~S~)~{ ~S~})" (r-name r-struct) vars)
                                          (format nil "(~(~S~) )" (r-name r-struct)))
                                        (format nil "~S" (r-name r-struct)))))
                (create-node-insert relation-string literal)
                (cell-deselect item selected-cell)
                (collapse-selection literal t)
                (unless (or (null (r-type r-struct)) all-vars-defined) 
                  (move-mark (fred-buffer literal) -1))
                (fred-update literal)))))
      :cell-size (make-point (- w 15) 14)
      :table-hscrollp nil
      :table-vscrollp t
      :table-sequence *r-structs*
      :table-print-function #'pretty-print-r-struct-name
      :selection-type :single
      :view-font input-font
      :view-nick-name :relations)
     
     (make-dialog-item
      'static-text-dialog-item (make-point 4 73) (make-point w 15) "Variables" nil :view-nick-name :variables-caption)
     
     (make-dialog-item
      'sequence-dialog-item (make-point 4 89) (make-point w 42) "Variables"
      #'(lambda (item)
          (let ((selected-cell (first (selected-cells item)))
                (literal (find-named-sibling item :literal)))
            (when selected-cell
              (create-node-insert (format nil "~S" (cell-contents item selected-cell)) literal)
              (cell-deselect item selected-cell))))
      :cell-size (make-point (- w 15) 14)
      :table-hscrollp nil
      :table-vscrollp t
      :table-sequence (sort (collect-literal-vars-and-types (cons head current-literal)) #'universal<)
      :table-print-function #'(lambda (var stream) (format stream "~S" var))
      :selection-type :single
      :view-font input-font
      :view-nick-name :variables)
     
     (make-dialog-item
      'static-text-dialog-item (make-point 4 132) (make-point w 15) "Literal(s) - for example - ((age ?PERSON ?a)(> ?a 17))" nil :view-nick-name :literal-caption)
     
     (make-dialog-item
      'ccl::scrolling-fred-dialog-item (make-point 6 149) (make-point (- h 27) (- v 197)) "" nil
      :allow-returns t :allow-tabs t :view-font input-font :view-nick-name :literal)
     
     (make-dialog-item
      'button-dialog-item (make-point (- h 215) (- v 26)) #@(60 20) " Erase "
      #'(lambda (item) (set-dialog-item-text (find-named-sibling item :literal) ""))
      :default-button nil
      :dialog-item-enabled-p t
      :view-nick-name :erase)

     (make-dialog-item
      'button-dialog-item (make-point (- h 155) (- v 26)) #@(60 20) " Cancel "
      #'(lambda (item) item (return-from-modal-dialog nil))
      :default-button nil
      :dialog-item-enabled-p t
      :view-nick-name :done)

     (make-dialog-item
      'button-dialog-item (make-point (- h 85) (- v 26)) #@(60 20) " Done "
      #'(lambda (item)
          (multiple-value-bind (value conjunction-p) (read-prolog-from-string (dialog-item-text (find-named-sibling item :literal)) nil nil)
            (when value
              (return-from-modal-dialog (values value conjunction-p)))))
      :default-button t
      :dialog-item-enabled-p t
      :view-nick-name :create)
     )
    
    (let ((literal-item (view-named :literal dialog)))
      (set-dialog-item-text literal-item (if current-literal (format nil "~S" current-literal) ""))
      (collapse-selection literal-item t)
      (fred-update literal-item))
    (move-selectors-and-buttons dialog)
    (modal-dialog dialog t)))


;;;_______________________________________
;;;  collect-literal-vars-and-types

(defun collect-literal-vars-and-types (literals &optional (vars nil) (types nil))
  (cond ((null literals) (values vars types))
        ((consp (first literals))
         (multiple-value-setq (vars types) (collect-literal-vars-and-types (first literals) vars types))
         (collect-literal-vars-and-types (rest literals) vars types))
        (t (let* ((r-struct (get-r-struct (first literals)))
                  (args (reverse (rest literals)))
                  (type (when r-struct (reverse (r-type r-struct)))))
             (compute-new-vars-and-types args type nil vars types)))))


;;;=========================================================
;;; EDIT QUESTIONS & TRANSLATIONS

(defun find-translation (value-wanted values-known translations)
  (case value-wanted
    (:FACT (assoc :FACT translations))
    (:QUESTION (assoc :QUESTION translations))
    (otherwise (find-if #'(lambda (question)
                            (and (eql (first question) value-wanted)
                                 (equal (first (second question)) values-known)))
                        translations))))

(defun record-new-translation (value-wanted values-known english number-of-values translations)
  (let ((old-translation (find-translation value-wanted values-known translations))
        (new-translation
         (case value-wanted
           (:FACT (list :FACT english))
           (:QUESTION (list :QUESTION english))
           (otherwise (list value-wanted (list values-known english number-of-values))))))
    (if old-translation
      (substitute new-translation old-translation translations)
      (case value-wanted
        (:FACT (push new-translation translations))
        (:QUESTION (if (eq (caar translations) :FACT)
                     (rplacd translations (cons new-translation (rest translations)))
                     (push new-translation translations)))
        (otherwise (nconc translations (list new-translation)))))))

(defun decode-translation (translation)
  (let* ((value-wanted (first translation))
         (second-translation (second translation))
         values-known english number-of-values)
    (case value-wanted
      ((:FACT :QUESTION) (setf values-known nil
                               english second-translation
                               number-of-values nil))
      (otherwise (setf values-known (first second-translation)
                       english (second second-translation)
                       number-of-values (third second-translation))))
    (values value-wanted values-known english number-of-values)))


(defun print-translation (translation stream)
  (multiple-value-bind (value-wanted values-known english number-of-values)
                       (decode-translation translation)
    values-known
    (format stream "~12S ~A~{ ~S~}"
            (case value-wanted
              (:FACT :STATEMENT)
              (otherwise value-wanted))
            (case number-of-values
              (:multi-valued "")
              (:single-valued " ")
              (otherwise " ")) 
            english)))


;;;=========================================================
;;; EDIT QUESTIONS & TRANSLATIONS

;;;_______________________________________
;;; edit-translation

(defun edit-translation (name vars kind translations)
  (let* ((h 450) (v 270) (y 5) (x 10)
         (user-font '("monaco" 9 :plain))
         (dialog-font '("chicago" 12 :plain))
         (window (make-instance 
                   'window
                   :window-type :document
                   :close-box-p nil
                   :window-title (format nil "~@(~A~) Translation" name) 
                   :window-show nil
                   :view-font dialog-font
                   :view-size (make-point h v)
                   :view-position '(:top 50))))
    (add-subviews
     window
     (make-dialog-item
      'static-text-dialog-item (make-point x y)  nil (name-vars-string name vars) nil)
     
     (make-dialog-item
      'static-text-dialog-item (make-point x (incf y 20)) nil "Value             English" nil)

     (make-dialog-item
      'sequence-dialog-item (make-point 2 (incf y 20)) (make-point (- h 4) (- v y 103)) ""
      #'(lambda (item)
          (let ((cell (first (selected-cells item))))
            (when cell
              (multiple-value-bind (value-wanted values-known english number-of-values)
                                   (decode-translation (cell-contents item cell))
                values-known
                (update-translation-editor (view-container item) value-wanted number-of-values)
                (set-dialog-item-text (find-named-sibling item :english) (format nil "~{~S~^ ~}" english))))))
      :table-sequence (case kind
                        ((:intensional :builtin) (list (list :FACT translations)))
                        (otherwise (copy-list translations)))
      :view-nick-name :translations
      :view-font user-font
      :sequence-order :vertical
      :table-vscrollp t
      :table-hscrollp nil
      :table-print-function #'print-translation)

     (make-dialog-item
      'radio-button-dialog-item (make-point 5 (setf y (- v 93))) nil "Statement"
      #'(lambda (item) (update-translation-editor (view-container item) :FACT nil))
      :radio-button-cluster 0 :view-nick-name :FACT)

     (make-dialog-item
      'radio-button-dialog-item (make-point 95 y) nil "Question"
      #'(lambda (item) (update-translation-editor (view-container item) :QUESTION nil))
      :radio-button-cluster 0 :dialog-item-enabled-p (eq kind :extensional)
      :view-nick-name :QUESTION)

     (make-dialog-item
      'radio-button-dialog-item (make-point 175 y) nil "Value"
      #'(lambda (item) (update-translation-editor (view-container item) nil :single-valued))
      :radio-button-cluster 0 :dialog-item-enabled-p (eq kind :extensional)
      :view-nick-name :VALUE)
     
     (make-dialog-item
      'editable-text-dialog-item (make-point 240 (incf y 2)) (make-point 80 12) "" nil
      :allow-returns nil :view-font user-font :view-nick-name :value-wanted)

     (make-dialog-item
      'radio-button-dialog-item (make-point 330 (decf y 9)) nil "single-valued" nil
      :radio-button-cluster 1 :view-nick-name :single-valued)

     (make-dialog-item
      'radio-button-dialog-item (make-point 330 (incf y 15)) nil "multi-valued" nil
      :radio-button-cluster 1 :view-nick-name :multi-valued)
     
     (make-dialog-item
      'ccl::scrolling-fred-dialog-item (make-point 5 (incf y 18)) (make-point (- h 10) 14) "" nil
      :allow-returns nil :view-font user-font :v-scrollp nil :view-nick-name :english)

     (make-dialog-item
      'button-dialog-item (make-point (- h 325) (- v 27)) #@(70 20) " Record "
      #'(lambda (item)
          (let* ((window (view-container item))
                 (translations (view-named :translations window))
                 (selected-cell (first (selected-cells translations)))
                 (english (read-from-string (format nil "(~A)" (dialog-item-text (view-named :english window)))))
                 (values-known (mapcan #'(lambda (word) (if (variable-p word) (list word))) english))
                 (number-of-values (when (pushed-radio-button window 1) (view-nick-name (pushed-radio-button window 1))))
                 (value-wanted
                  (case (view-nick-name (pushed-radio-button window 0))
                    (:FACT 
                     (cond ((some #'(lambda (v) (and (variable-p v) (not (member v vars)))) english)
                            (message-dialog "The English translation uses varibles not contained in the relation." :position :centered))
                           ((some #'(lambda (v) (not (member v english))) vars)
                            (message-dialog "The English translation does not use all variables contained in relation." :position :centered)))
                     :FACT)
                    (:QUESTION
                     (cond ((some #'(lambda (v) (and (variable-p v) (not (member v vars)))) english)
                            (message-dialog "The question uses varibles not contained in the relation." :position :centered))
                           ((some #'(lambda (v) (not (member v english))) vars)
                            (message-dialog "The question does not use all variables contained in relation." :position :centered)))
                     :QUESTION)
                    (:VALUE
                     (let ((value-wanted (value-from-dialog-item-text (view-named :value-wanted window))))
                       (cond ((not (member value-wanted vars))
                              (message-dialog "The question is asking about a variable that is not contained in the fact." :position :centered))
                             ((member value-wanted english)
                              (message-dialog "The question uses the value it is asking about." :position :centered))
                             ((some #'(lambda (v) (and (variable-p v) (not (member v vars)))) english)
                              (message-dialog "The question uses varibles not contained in the relation." :position :centered)))
                       value-wanted)))))
            (when selected-cell
              (cell-deselect translations selected-cell))
            (set-table-sequence translations (record-new-translation value-wanted values-known english number-of-values
                                                                     (table-sequence translations)))))
      :default-button t
      :view-nick-name :record)

     (make-dialog-item
      'button-dialog-item (make-point (- h 240) (- v 27)) #@(70 20) " Delete "
      #'(lambda (item)
          (let* ((window (view-container item))
                 (translations (view-named :translations window))
                 (selected-cell (first (selected-cells translations))))
            (when selected-cell
              (cell-deselect translations selected-cell)
              (set-table-sequence translations (delete (cell-contents translations selected-cell)
                                                       (table-sequence translations)))))))
     (make-dialog-item
      'button-dialog-item (make-point (- h 160) (- v 27)) #@(70 20) " Done "
      (case kind
        ((:intensional :builtin)
         #'(lambda (item)
             (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-edit-translation-define *user-monitor*)))
             (return-from-modal-dialog (second (first (table-sequence (find-named-sibling item :translations)))))))
        (otherwise
         #'(lambda (item)
             (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-edit-translation-cancel *user-monitor*)))
             (return-from-modal-dialog (table-sequence (find-named-sibling item :translations)))))))

     (make-dialog-item
      'button-dialog-item (make-point (- h 80) (- v 27)) #@(70 20) " Cancel "
      #'(lambda (item) item
         (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-edit-translation-cancel *user-monitor*)))
         (return-from-modal-dialog :cancel))))
    (update-translation-editor window :FACT nil)
    (modal-dialog window t)))


(defun update-translation-editor (window value-wanted number-of-values)
  (let ((value-wanted-item (view-named :value-wanted window))
        (multi-valued-item (view-named :multi-valued window))
        (single-valued-item (view-named :single-valued window)))
    (case value-wanted
      ((:QUESTION :FACT)
       (set-dialog-item-text value-wanted-item "")
       (dialog-item-disable value-wanted-item)
       (radio-button-unpush multi-valued-item)
       (radio-button-unpush single-valued-item)
       (dialog-item-disable multi-valued-item)
       (dialog-item-disable single-valued-item)
       (radio-button-push (view-named value-wanted window)))
      (otherwise
       (dialog-item-enable value-wanted-item)
       (set-dialog-item-text value-wanted-item (if value-wanted (format nil "~S" value-wanted) ""))
       (dialog-item-enable multi-valued-item)
       (dialog-item-enable single-valued-item)
       (radio-button-push (view-named :VALUE window))
       (case number-of-values
         (:multi-valued (radio-button-push multi-valued-item))
         (:single-valued (radio-button-push single-valued-item)))))))


;;;=========================================================
;;; HIGH LEVEL FUNCTIONS
                                      
;;;_______________________________________
;;;  user-create-relation

(defun user-create-relation (&optional (kind nil) (r-struct nil))
  (case kind
    (:intensional
     (when (user-monitor-p *user-monitor*)
       (incf (user-monitor-new-intensional-relation *user-monitor*)))
     (let ((rule (or r-struct (make-r :name 'new_rule
                                      :vars '(?arg)
                                      :arity 1
                                      :type '(:anything)
                                      :mode '(:?) 
                                      :clauses (if *edit-graphically* nil '(((new_rule ?arg) (clause_body ?arg))))
                                      :kind :intensional))))
       (if *edit-graphically* (user-edit-rule rule t) (text-edit-relation rule))))
    (:extensional
     (when (user-monitor-p *user-monitor*)
       (incf (user-monitor-new-extensional-relation *user-monitor*)))
     (user-edit-relation (or r-struct (make-r :name 'new_fact
                                              :vars '(?arg)
                                              :arity 1
                                              :type '(:anything)
                                              :mode '(:?)
                                              :pos '((pos_example))
                                              :neg '((neg_example))
                                              :kind :extensional))))
    (:builtin
     (when (user-monitor-p *user-monitor*)
       (incf (user-monitor-new-builtin-relation *user-monitor*)))
     (user-edit-relation (or r-struct (make-r :name 'new_builtin
                                              :function-def '#'(lambda (?arg) (lisp_function ?arg))
                                              :arity 1
                                              :type '(:anything)
                                              :mode '(:+)
                                              :sort-fn-def '#'<
                                              :kind :builtin))))
    (otherwise
     (if (r-p r-struct)
       (let ((new-kind (specify-relation-kind (r-name r-struct))))
         (setf (r-kind r-struct) new-kind)
         (user-create-relation new-kind r-struct))
       (user-create-relation (specify-relation-kind 'new_relation))))))

;;;_______________________________________
;;;  user-edit-relation

(defun user-edit-relation (r-struct &optional (nodes nil))
  (if (r-p r-struct)
    (if (member r-struct *special-r-structs*)
      (special-r-struct-message r-struct)
      (case (r-kind r-struct)
        (:intensional (if *edit-graphically* (user-edit-rule r-struct nil nodes) (text-edit-relation r-struct)))
        (:extensional (if *edit-graphically*
                        (display-examples (r-name r-struct) :pred r-struct :use-template-when-possible nil)
                        (text-edit-relation r-struct)))
        (:builtin (text-edit-relation r-struct))
        (:arithmetic-op (text-edit-relation r-struct))
        (:undefined (user-create-relation :undefined r-struct))
        (otherwise (message-dialog (format nil "~S is defined as a ~(~A~) relation for which there is currently no editor."
                                           (r-name r-struct) (r-kind r-struct)) :position :centered))))
    (user-create-relation)))

(defun special-r-struct-message (r-struct)
  (message-dialog (format nil "The relation ~(~s~) is defined internally and and may not be altered or removed." (r-name r-struct)) :position :centered))

;;;_______________________________________
;;;  user-delete-relation

(defun user-delete-relation (r-struct)
  (cond ((member r-struct *special-r-structs*)
         (special-r-struct-message r-struct))
        ((or *expert-mode*
             (not (y-or-n-dialog (format nil "Do you really want to delete~%  ~(~S~)?" (r-name r-struct))
                                 :yes-text " No " :no-text " Yes " :cancel-text nil :position :centered)))
         (delete-r-struct r-struct)
         (update-relations))))

;;;_______________________________________
;;;  user-display-relation

(defun user-display-relation (r-struct)
  (if *edit-graphically*
    (display-r-structs (list r-struct))
    (case (r-kind r-struct)
      (:intensional (show-rule-def (r-name r-struct)))
      (:extensional (describe-fact (r-name r-struct)))
      (:builtin (format t "~%Builtin: ~A   variable types: ~(~S~)   functional definition:  ~S"
                        (name-vars-string (r-name r-struct) (r-vars r-struct)) (r-type r-struct) (r-function-def r-struct)) )
      (:arithmetic-op (format t "~%Arithmetic-Op: ~A   variable types: ~(~S~)   functional definition:  ~S"
                              (name-vars-string (r-name r-struct) (r-vars r-struct)) (r-type r-struct) (r-function-def r-struct)))
      (:= (format t "~%Equal: ~A   variable types: ~(~S~)" (name-vars-string (r-name r-struct) (r-vars r-struct)) (r-type r-struct)))
      (:is (format t "~%Is: ~A   variable types: ~(~S~)" (name-vars-string (r-name r-struct) (r-vars r-struct)) (r-type r-struct)))
      (otherwise nil) )))

;;;_______________________________________
;;;  user-edit-rule

(defun user-edit-rule (r-struct &optional (create nil) (nodes nil))
  (let ((*batch* t))
    (without-interrupts
     (set-cursor *watch-cursor*)
     (catch-cancel
       (let* ((name (r-name r-struct))
              (vars (r-vars r-struct))
              (graph (create-graph))
              (root (graph-root graph))
              a-base u-base antecedents window view title)
         (if create
           (setf a-base (get-node graph :consequent root :vars vars :r-struct r-struct :kind :intensional)
                 antecedents nil
                 title (format nil "Create ~(~A~)" name))
           (setf a-base (connect-literal graph root (cons name vars) nil nil :never 0)
                 antecedents (connect-clauses graph a-base (editor-all-antecedents r-struct vars) nil :never 0)
                 title (format nil "Edit ~(~A~)" name)))
         (setf window (make-instance 'rule-edit-window :original-r-struct r-struct :modified-r-struct (copy-r r-struct) :window-title title :nodes nodes)
               view (graph-view window)
               (expand view) :every-use
               (root view) root
               (graph view) graph
               (graph-views graph) (push view (graph-views graph))
               u-base (get-node graph :consequent root :kind :intensional)
               (node-antecedents root) (list (list a-base) (list u-base))
               (node-antecedents a-base) antecedents
               (node-recursive? a-base) nil)
         (get-cell view :text "u-base" :node u-base :hidden? t)
         (get-cell view :text (name-vars-string name vars) :node a-base)
         (window-fixup window t)
         (window-select window)
         (when create 
           (select-node a-base)
           (editor-EDIT-DEF window)))))))

;;;_______________________________________
;;; fix-constants-in-tuples-for-editor

(defun fix-constants-in-tuples-for-editor (base node coverage)
  (let* ((n-vars (node-vars node))
         (b-vars (node-vars base))
         (input-pos (coverage-input-pos coverage))
         (input-neg (coverage-input-neg coverage))
         (input-vars (substitute-vars (coverage-input-vars coverage) (unify n-vars b-vars)))
         (input-type (coverage-input-type coverage))
         extend-vars extend-vals extend-type)
    (do* ((bs b-vars (rest bs))
          (b (first bs) (first bs))
          (ns n-vars (rest ns))
          (n (first ns) (first ns))
          (ts (when (node-r-struct node) (r-type (node-r-struct node))) (rest ts)))
         ((null bs))
      (unless (or (variable-p n) (equal b n))
        (push b extend-vars)
        (push n extend-vals)
        (push (or (first ts) :anything) extend-type)))
    (when extend-vars
      (setq input-vars (append input-vars (nreverse extend-vars))
            input-type (append input-type (nreverse extend-type))
            extend-vals (nreverse extend-vals)
            input-pos (mapcar #'(lambda (tuple) (append tuple extend-vals)) input-pos)
            input-neg (mapcar #'(lambda (tuple) (append tuple extend-vals)) input-neg)))
    (values input-pos input-neg input-vars input-type)))


(defun hilight-and-display-tree-coverage (root view window)
  (dolist (conjunction (node-antecedents root))
    (dolist (node conjunction)
      (hilight-and-display-tree-coverage node view window)
      (hilight-node-coverage node (some #'coverage-output-pos (node-coverage node)) (some #'coverage-output-neg (node-coverage node))))
    (when view
      (let* ((last-node (first (last conjunction)))
             (cell (node-cell view last-node))
             (number-pos 0)
             (number-neg 0))
        (dolist (coverage (node-coverage last-node))
          (incf number-pos (length (coverage-output-pos coverage)))
          (incf number-neg (length (coverage-output-neg coverage))))
        (when cell (update-external-text cell (format nil "~A+ ~A-" number-pos number-neg)))))))

;;;_______________________________________
;;; edit-last-node-selected

(defun edit-last-node-selected (view)
  (let ((node (last-node-selected view)))
    (when (node-p node)
      (let ((r-struct (node-r-struct node)))
        (if (r-p r-struct)
          (user-edit-relation r-struct (when (and *edit-graphically* *maintain-examples-when-editing*)
                                         (collect-nodes-with-same-r-struct (node-root node) r-struct)))
          (message-dialog (format nil "~A has no associated r-struct." node)))))))

;;;_______________________________________
;;; collect-nodes-with-same-r-struct

(defun collect-nodes-with-same-r-struct (root r-struct)
  (let ((nodes nil))
    (apply-function-to-tree root #'(lambda (node) (when (eq (node-r-struct node) r-struct) (setf nodes (push node nodes)))))
    nodes))




