
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

(defun insert-new-fact (p tuple) (assert-fact (cons p tuple)))
(defun destroy-fact (p tuple) (retract-fact (cons p tuple)))

;;judge the accuracy of the definition of illegal
(defun judge(n &aux p a i)
  (cond ((= n 0) 0)
	(t (setq p (generate-example))
	   (setq a (prove-function?  (get-r-function 'learned-concept) nil nil p))
	   (setq i (illegal-position p))
	   (+ (cond ((or (and i a)(and (not i)(not a))) 1)(t 0)) 
	      (judge (- n 1))))))

;;______________________________________________________________________________
;; judge-examples (examples)
;;    judge the accuracy of the definition of illegal on a given set of examples

(defun judge-examples (examples &optional (rule 'illegal)
				&aux actual-class
				theory-class
				(correct 0)
				(neg-classified-pos 0)
				(pos-classified-neg 0)
                                (rule-fun (r-prolog-function (get-rule rule))))

  (dolist (example examples correct)
	  (setq actual-class (illegal-position example))
	  (setq theory-class (prove-function?  rule-fun nil nil example))
	  (cond ((and actual-class theory-class) (incf correct))
		((and (not actual-class) (not theory-class)) (incf correct))
		((and actual-class (not theory-class)) (incf pos-classified-neg))
		((and (not actual-class) theory-class) (incf neg-classified-pos))))
  (/ correct (length examples)))


;;______________________________________________________________________________
;; generate-example-set (n pp)
;;     generates n examples, pp percent of which are positive.

(defun generate-example-set (n pp &aux (examples nil))
  (dotimes (i  n examples)
    (if (< (random 100) pp)              
        (push (return-positive-example) examples)
        (push (return-negative-example) examples))))


;;;_____________________________________
;;; RETURN-POSITIVE-EXAMPLE

(defun return-positive-example ()
  (do ((example (generate-example)))
      ((illegal-position example) example)
    (setq example (generate-example))))

;;;_____________________________________
;;; RETURN-NEGATIVE-EXAMPLE

(defun return-negative-example ()
  (do ((example (generate-example))) 
      ((not (illegal-position example)) example)
    (setq example (generate-example))))

;;;_____________________________________
;;; GENERATE-EXAMPLE

(defun generate-example ()
  (let ((result nil))
    (dotimes (i 6 result) (push (+ 1 (random 8)) result))))


;;;______________________________________________________________________________
;;; LISP DOMAIN CONCEPT DESCRIPTION                             (Domain Specific)
;;;______________________________________________________________________________
;;; illegal-position (L)
;;;     returns T if position is illegal.
;;;     L is a vector of peices (white king, white rook, and black king)

(defun adjacent (i j k l)
  (or ( and (eq i k) (near j l)) 
       (and (eq j l) (near i k))
       (and (near i k) (near j l))))

(defun near (i j)
  (or (eq i (+ 1 j))
      (eq i (- j 1))))

(defun between (a b c d e f)
  (or (and (= a c e)
           (or (< d b f)
               (> d b f)))
      (and (= b d f)
           (or (< c a e)
               (> c a e)))))

(defun same-line (i j k l)
  (or (eq i k) 
      (eq j l)))

(defun illegal-position (L)
   (cond ((and (eq (first L) (third L))          ;; white king and white rook at same position
               (eq (second L) (fourth L))) t)
         ((and (eq (first L) (fifth L))          ;; white king and black king at same position
               (eq (second L) (sixth L))) t)
         ((and (eq (third L) (fifth L))          ;; white rool and black king at same position
               (eq (fourth L) (sixth L))) t)
         ((adjacent (first L) (second L)         ;; white king capture black king
                    (fifth L) (sixth L)) t) 
         ((and (same-line (third L) (fourth L)   ;; white rook captures black king
                          (fifth L) (sixth L))
               (not (between (first L) (second L) 
                             (third L) (fourth L) 
                             (fifth L) (sixth L))) t))))


;generates definition of illegal with N examples
;;if type is true, uses :column row, else :column

(defun illegal-generator (n                      ; number of examples to generate
                          &optional 
			  (pp  50)                   ; percent positive
                             (noise 0)           ; percent error rate
                             (type? nil)         ; if true uses :column :row, else :column
                             (print-training-set? nil)
                          &aux
                             (p-noise noise)     ; rate of mis-classification of positive examples
                             (n-noise noise)     ; rate of mis-classification of negative examples
                             (in nil)            ; list of positive examples (containing noise)
                             (out nil))          ; list of negative examples (containing noise)
  (dotimes (i  n)
    (if (< (random 100) pp)                    ; Choose class of example (pos or neg)
      (if (< (random 100) p-noise)             ; Introduce noise into positive examples
        (push (return-negative-example) in)
        (push (return-positive-example) in))
      (if (< (random 100) n-noise)             ; Introduce noise into negative examples
        (push (return-positive-example) out)
        (push (return-negative-example) out))))

  (if print-training-set?
    (progn
      (format t "~%~%Training Set")
      (format t "~%Pos: ~a" in)
      (format t "~%Neg: ~a" out)))

  (if type? 
    (eval `(def-pred illegal 
             :induction nil 
             :type (:row :column :row :column :row :column)
             :pos ,in 
             :neg ,out))
    (eval `(def-pred illegal 
             :induction nil
             :pos ,in 
             :neg ,out))))


;;judge accuracy of original theory (of illegal-board)

(defun judge-original(n &aux p a i)
  (cond ((= n 0) 0)
	(t (setq p (generate-example))
	   (setq a (prove-function?  (get-r-function 'illegal-board) nil nil p))
	   (setq i (illegal-position p))
	   (+ (cond ((or (and i a)(and (not i)(not a))) 1)(t 0)) 
	      (judge-original (- n 1))))))

  
 
;;learns illegal and tests on 100 examples
(defun run-illegal-test(n)
  (illegal-generator n  50 0 t)
  (make-prolog-rule 'learned-concept 6 (focl 'illegal))
  (judge 100))


(defun test-mutatator-v (&optional (mutations '(0 1 2 4 6 8 10 12 14 16 20 24)) &aux
				(foil-acc nil) foil-tuples domain-acc
				saved-examples
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq saved-examples (generate-example-set 1000 36))
  (load-source-test-file "typed-641")
  (test-illegal)
  (setq foil-tuples *variablizations-checked*)
  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (test-illegal-ebl)
	  (setq focl-tuples *variablizations-checked*)
	  (format t "~%~a~a~a~a~4F~a~4F"
		  m #\tab 0 #\tab foil-tuples #\tab domain-acc)
	  (format t "~%~a~a~a~a~4F~a~4f"
		  m #\tab 1 #\tab focl-tuples #\tab domain-acc)))

(defun test-mutatator (&optional (examples 80) (mutations '(0 1 2 4 6 8 10 12 14 16 20 24)) &aux
				foil-acc foil-tuples domain-acc
				saved-examples c
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq  *conditions-for-extensional-threshold-conjunctions*
	 #'(lambda(&rest x) x nil))
  (setq saved-examples (generate-example-set 1000 50))
  (load-source-test-file "untyped-between-equal-near")
  (illegal-generator examples 50 0 nil)
  
  (load-source-test-file "untyped-chess-domain")
  (format t ";Mutations DT?  tuples learned-accuracy original-accuracy~%")
  (make-prolog-rule 'learned-concept 6 (print(focl 'illegal :save-examples nil :gain-function :ratio
					     :intensional-induction nil)))



  (setq foil-acc (judge-examples saved-examples 'learned-concept))
  (setq foil-tuples *variablizations-checked*)

  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (print-modified-theory)
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (make-prolog-rule 'learned-concept 6 (print(test-illegal-ebl :leaves :ratio)))
	  (setq focl-acc (judge-examples saved-examples 'learned-concept))
	  (setq focl-tuples *variablizations-checked*)
	  (format t "~%^~a~a~a~a~4F~a~4f~a~4f~%"
		  m #\tab 0 #\tab foil-tuples #\tab foil-acc #\tab domain-acc)
	  (format t "^~a~a~a~a~4F~a~4f~a~4f~%"
		  m #\tab 1 #\tab focl-tuples #\tab focl-acc #\tab domain-acc)
	  (make-prolog-rule 'learned-concept 6 (setq c (test-illegal-ebl :frontier :ratio)))
	  (setq focl-acc (judge-examples saved-examples 'learned-concept))
	  (setq focl-tuples *variablizations-checked*)
	  (format t "^~a~a~a~a~4F~a~4f~a~4f~%"
		  m #\tab 2 #\tab focl-tuples #\tab focl-acc #\tab domain-acc)
	  (print c)))




(defun test-mutatator-lt (&optional (print nil) (examples 80) (mutations '(0 1 2 4 6 8 10 12 14 16 20 24 30 36)) &aux
				foil-acc foil-tuples domain-acc
				saved-examples front-acc
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq *all-additions* '(less equal near))
  (setq *all-mutatables* 
    '(illegal-board between same-loc king-attack-king  rook-attack-king  king-not-between))
  (setq  *conditions-for-extensional-threshold-conjunctions*
	 #'(lambda(&rest x) x nil))
  (setq saved-examples (generate-example-set 1000 50))
  (load-source-test-file "untyped-less-equal-near")
  (illegal-generator examples 50 0 nil)
  
  (load-source-test-file "untyped-chess-domain-less")
  (format t ";N Mutations induction-accuracy dt-accuracy lv-acc f-acc~%")
  (make-prolog-rule 'learned-concept 6 (if print (print(focl 'illegal :save-examples nil :gain-function :ratio
					     :intensional-induction nil))
					 (focl 'illegal :save-examples nil :gain-function :ratio
					     :intensional-induction nil)))



  (setq foil-acc (judge-examples saved-examples 'learned-concept))
  (setq foil-tuples *variablizations-checked*)

  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (when print (print-modified-theory))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (make-prolog-rule 'learned-concept 6 (if print (print(test-illegal-ebl :leaves :ratio nil))
						 (test-illegal-ebl :leaves :ratio nil)))
	  (setq focl-acc (judge-examples saved-examples 'learned-concept))
	  (setq focl-tuples *variablizations-checked*)
	  (make-prolog-rule 'learned-concept 6 (if print (print (test-illegal-ebl :frontier :ratio nil))
						  (test-illegal-ebl :frontier :ratio nil)))
	  (setq front-acc (judge-examples saved-examples 'learned-concept))
	  (format t "~%^~a~a~a~a~4F~a~4f~a~4f~a~4f"
		  examples #\tab m #\tab foil-acc #\tab domain-acc  #\tab focl-acc #\tab front-acc)

))

(defun test-mutatator-vb (&optional (print nil) (examples 80) (mutations '(0 1 2 4 6 8 10 12 14 16 20 24 30 36)) &aux
				foil-acc foil-tuples domain-acc
				saved-examples front-acc
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq *all-additions* '(less equal near))
  (setq *all-mutatables* 
    '(illegal-board between same-loc king-attack-king  rook-attack-king  king-not-between))
  (setq  *conditions-for-extensional-threshold-conjunctions*
	 #'(lambda(&rest x) x nil))
  (setq saved-examples (generate-example-set 1000 50))
  (load-source-test-file "untyped-less-equal-near")
  (illegal-generator examples 50 0 nil)
  
  (load-source-test-file "untyped-chess-domain-less")
  (format t ";N Mutations induction-accuracy dt-accuracy lv-acc f-acc~%")
  (make-prolog-rule 'learned-concept 6 (if print (print(focl 'illegal :save-examples nil :gain-function :ratio
					     :intensional-induction nil))
					 (focl 'illegal :save-examples nil :gain-function :ratio
					     :intensional-induction nil)))



  (setq foil-acc *variablizations-checked*)

  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (when print (print-modified-theory))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (make-prolog-rule 'learned-concept 6 (if print (print(test-illegal-ebl :leaves :ratio nil))
						 (test-illegal-ebl :leaves :ratio nil)))
	  (setq focl-acc *variablizations-checked*)
	  (setq focl-tuples *variablizations-checked*)
	  (make-prolog-rule 'learned-concept 6 (if print (print (test-illegal-ebl :frontier :ratio nil))
						  (test-illegal-ebl :frontier :ratio nil)))
	  (setq front-acc *variablizations-checked*)
	  (format t "~%^~a~a~a~a~4F~a~4f~a~4f~a~4f"
		  examples #\tab m #\tab foil-acc #\tab domain-acc  #\tab focl-acc #\tab front-acc)

))



(defun test-illegal-knight (&optional (examples 100) (trials 20) 
				      &aux
				      foil-acc foil-tuples domain-acc
				      saved-examples front-acc front-tuples
				      focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq  *conditions-for-extensional-threshold-conjunctions*
	 #'(lambda(&rest x) x nil))
  (setq saved-examples (generate-example-set 1000 50))
  (load-source-test-file "typed-illegal")
  (load-source-test-file "chess-knight-and-rook")
  (dotimes (j trials)
	   (illegal-generator examples 50 0 t)
	   (time(make-prolog-rule 'learned-concept 6 (focl 'illegal :save-examples nil
					:intensional-induction nil
					:gain-function :ratio
					:refinement :LEAVES
					:max-new-variables 0 
					:intensional-induction nil
					:operationalize-intensional t)))

	   (setq foil-tuples *variablizations-checked*)
	   (setq foil-acc (judge-examples saved-examples 'learned-concept))
	   (time(make-prolog-rule 'learned-concept 6   
				  (focl 'illegal :save-examples nil
					:intensional-induction nil
					:gain-function :ratio
					:refinement :LEAVES
					:goal-concept-name 'illegal-board
					:max-new-variables 0 
					:intensional-induction nil
					:operationalize-intensional t)))

	   (setq focl-acc (judge-examples saved-examples 'learned-concept))
	   (setq focl-tuples *variablizations-checked*)
	   (time(make-prolog-rule 'learned-concept 6   
				  (focl 'illegal :save-examples nil
					:intensional-induction nil
					:gain-function :ratio
					:refinement :frontier
					:goal-concept-name 'illegal-board
					:max-new-variables 0 
					:intensional-induction nil
					:operationalize-intensional t)))
	   (setq front-acc (judge-examples saved-examples 'learned-concept))
	   (setq front-tuples *variablizations-checked*)
	   (format t "~%^~4F~a~4f~a~4F~a~4f~a~4f~a~4f"
		   foil-acc  #\tab focl-acc #\tab front-acc #\tab
		   foil-tuples  #\tab focl-tuples #\tab front-tuples))

  )





(defun test-students-leaves (&optional (trials 16) 
				       (examples '(10 15 20 25 30 40 50 60 75 100)) 
			     &aux
			     foil-acc foil-tuples domain-acc
			     saved-examples front-acc
			     focl-acc focl-tuples (m 0))
  (reset-preds)
  (load-source-test-file "worse-loan")
  (load-source-test-file "random-students")
  (format t ";N induction-accuracy  lv-acc f-acc~%")
  (dolist (e examples)
    (dotimes (ignore trials)
      (split-pred 'no_payment_due-fact 'npd e 'learning-set 200)
      (make-prolog-rule 'learned-concept 1 (focl 'npd :gain-function :ratio
						 :simplify-operationalizations t
						 :max-new-variables 2  
						 :intensional-induction nil))



      (setq foil-acc (judge-pos-and-neg (r-pos (get-r-struct 'learning-set))
					(r-neg (get-r-struct 'learning-set))
					'learned-concept))
      (make-prolog-rule 'learned-concept 1  (focl 'npd :gain-function :ratio
						  :goal-concept-name 'not_in_default
						  :simplify-operationalizations t
						  :max-new-variables 2  
						  :intensional-induction nil))
      (setq focl-acc (judge-pos-and-neg (r-pos (get-r-struct 'learning-set))
					(r-neg (get-r-struct 'learning-set))
					'learned-concept))
      (make-prolog-rule 'learned-concept 1 (focl 'npd :gain-function :ratio
						 :goal-concept-name 'not_in_default
						 :simplify-operationalizations t
						 :refinement :frontier
						 :max-new-variables 2  
						 :intensional-induction nil))
      (setq front-acc (judge-pos-and-neg (r-pos (get-r-struct 'learning-set))
					 (r-neg (get-r-struct 'learning-set))
					 'learned-concept))
      (format t "~%^~a~a~4F~a~4f~a~4f"
	      e #\tab foil-acc   #\tab focl-acc #\tab front-acc)

      )))




(defun print-modified-theory ()
  (dolist (rule *all-mutatables*)
    (pprint (get-clauses rule))))

(defun mutate-theory (&optional 
		      &aux choice)
  
  (setq choice (random 4))		; 1..4
  (format t "~&;make-n-random-mutations: selecting choice ~a~%" choice)
  (cond ((= choice 1) (delete-random-term))
	((= choice 2) (delete-random-clause ))
	((= choice 3) (add-random-term ))
	((= choice 0) (add-random-clause ))))

(defun find-clauses-that-are-true-of-example (r e)
    (remove-if-not #'(lambda(c)
		       (prove-function? (clause-prolog-function c)
					e nil))
		   (r-clauses (get-rule r))))


(defun jack-knife(concept &rest keys &key (print nil) &allow-other-keys)
  (declare (ignore the-ignored-key))
  (let* ((s (get-r-struct concept))
	 (p (r-pos s))
	 (n (r-neg s))
	 (r nil)
	 (arity (r-arity s))
	 (total 0)
	 (correct 0)
	 (rule-fun nil))
    (dolist (pos p)
      (incf total)
      (destroy-fact concept pos)
      (setq r (apply #'focl concept keys))
      (when r (make-prolog-rule 'learned-concept arity r)
	    (setq rule-fun (r-prolog-function (get-rule 'learned-concept)))
	    (cond ((prove-function?  rule-fun nil nil pos)
		   (when print (format t "~%correct")
			 (format t "+~a ~a" pos (find-clauses-that-are-true-of-example 'learned-concept pos)))
		   (incf correct))
		  (print (format t "~%wrong")
			 (format t "+~a ()" pos)))
	    (when print (print r)))
      (insert-new-fact concept pos)
      )
    (dolist (neg n)
      (incf total)
      (setf (r-neg s)(delete neg (r-neg s)))
      (setq r (apply #'focl concept keys))
      (when r (make-prolog-rule 'learned-concept arity r)
	    (setq rule-fun (r-prolog-function (get-rule 'learned-concept)))
	    (cond  ((not (prove-function?  rule-fun nil nil neg))
		    (when print (format t "~%correct")
			  (format t "-~a ()" neg))
		    (incf correct))
		   (print (format t "~%wrong")
			  (format t "-~a ~a" neg (find-clauses-that-are-true-of-example 'learned-concept neg))))
            
	    (when print (print r)))
      (push  neg  (r-neg s)))
    (/ CORRECT TOTAL)))



(defun jack-knife-amm(concept &rest keys &key (voters 7)(print nil) &allow-other-keys)
  (declare (ignore the-ignored-key))
  (let* ((s (get-r-struct concept))
	 (p (r-pos s))
	 (n (r-neg s))
	 (r nil)
	 (arity (r-arity s))
	 (total 0)
	 (correct 0)
	 (rule-fun nil))
    (dolist (pos p)
      (incf total)
      (destroy-fact concept pos)
      (setq r (apply #'amm voters concept keys))
      (when r (make-voting-rule 'learned-concept arity r)
	    (setq rule-fun (r-prolog-function (get-rule 'learned-concept)))
	    (cond ((prove-function?  rule-fun nil nil pos)
		   (when print (format t "~%correct")
			 (format t "+~a ~a" pos (find-clauses-that-are-true-of-example 'learned-concept pos)))
		   (incf correct))
		  (print (format t "~%wrong")
			 (format t "+~a ()" pos)))
	    (when print (print r)))
      (insert-new-fact concept pos)
      )
    (dolist (neg n)
      (incf total)
      (setf (r-neg s)(delete neg (r-neg s)))
      (setq r (apply #'focl concept keys))
      (when r (make-prolog-rule 'learned-concept arity r)
	    (setq rule-fun (r-prolog-function (get-rule 'learned-concept)))
	    (cond  ((not (prove-function?  rule-fun nil nil neg))
		    (when print (format t "~%correct")
			  (format t "-~a ()" neg))
		    (incf correct))
		   (print (format t "~%wrong")
			  (format t "-~a ~a" neg (find-clauses-that-are-true-of-example 'learned-concept neg))))
            
	    (when print (print r)))
      (push  neg  (r-neg s)))
    (/ CORRECT TOTAL)))


(defun test-op(&aux saved-examples domain-acc mutations)
  (reset-preds)
  (load-source-test-file  "untyped-between-equal-near")
  (dotimes (j 2)
    (format t "~%;;================ ~a ================" (+ 50 (* 25 j))) 
    (dotimes (k 12)
      (setq saved-examples (generate-example-set 1000 50))
      (load-source-test-file  "untyped-chess-domain")
      (setq mutations 0)
      (dolist (i '(1 1 2 2 3 4 5))
	(setq domain-acc (judge-examples saved-examples 'illegal-board))
	(illegal-generator (+ 50 (* 25 j)) 50 0 nil)
	(make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :frontier))
	(format t "~%info  frontier~a~a~a~a~a~4F~a~4f~a~4f"
		#\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
		#\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)

	(make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :frontier :ratio))
	(format t "~%ratio frontier~a~a~a~a~a~4F~a~4f~a~4f"
		#\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
		#\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)
	(make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :leaves))
	(format t "~%info  leaves  ~a~a~a~a~a~4F~a~4f~a~4f"
		#\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
		#\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)
	(dotimes (j i)
	  (incf mutations)
	  (mutate-theory))))))



(defun test-mutatator-v2 (mutator &optional(mutations '(0 1 2 3 4 6 8 10 12))
					   (foil-acc nil) foil-tuples domain-acc
					   saved-examples
					   focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq saved-examples (generate-example-set 1000 36))
  (load-source-test-file "typed-641")
  (dolist (mm mutations)
    (do ()
	((= m mm))
      (incf m)
      (funcall mutator))
    (setq domain-acc (judge-examples saved-examples 'illegal-board))
    (test-illegal-ebl)
    (setq focl-tuples *variablizations-checked*)
    (format t "~%~a~a~a~a~4F~a~4f"
	    m #\tab 1 #\tab focl-tuples #\tab domain-acc)))


;;;______________________________________________________________________________
;;; RANDOMLY-EXTRACT-SUBSEQUENCE (destructive)
;;;
;;;  Given a sequence of length seq-length this function will randomly extract a
;;;  subsequence of length sub-length.  The function returns the residue of the
;;;  original sequence, and the a randomly extracted subsequence. 
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   8/06/91  Created

(defun randomly-extract-subsequence (sequence sub-length &optional (seq-length (length sequence)))
  (cond ((> 0 sub-length) (error "Subsequence length must be non negative."))
	((> sub-length seq-length) (error "Subsequence length must not exceed sequence length."))
	(t (let ((number-needed sub-length)
		 (number-remaining seq-length)
		 (subsequence nil)
		 (last nil)
		 (prev nil))
	     (do* ((cons-cell sequence (cdr cons-cell)))
		 ((null cons-cell) nil)
	       (if (< (random number-remaining) number-needed)
		   (progn
		     (if prev
			 (rplacd prev (cdr cons-cell))
		       (setf sequence (cdr sequence)))
		     (if subsequence
			 (rplacd last cons-cell)
		       (setf subsequence cons-cell))
		     (setf last cons-cell)
		     (decf number-needed 1))
		 (setf prev cons-cell))
	       (decf number-remaining 1))
	     (if last
		 (rplacd last nil))
	     (values sequence subsequence)))))

(defun split-pred (original-name learning-name size 
		   &optional (testing-name nil)
			     (testing-size nil))
  (let* ((s (get-r-struct original-name))
	 (pos (r-pos s))
	 (neg (r-neg s))
	 (new (copy-r s))
	 new-e rest new-p new-n
	 (all (nconc (copy-list pos) (copy-list neg))))
    (multiple-value-setq (rest new-e)
      (randomly-extract-subsequence all size))
    (setf (r-name new) learning-name)
    (set-r-struct learning-name new)
    (setf new-p (remove-if-not #'(lambda (e) (member e pos)) new-e)
	  new-n (remove-if-not #'(lambda (e) (member e neg)) new-e))
    (setf (r-pos new) new-p
	  (r-neg new) new-n)
    (when testing-name
      (setq new (copy-r s))
      (multiple-value-setq (rest new-e)
	(randomly-extract-subsequence rest testing-size))
      (setf (r-name new) testing-name)
      (set-r-struct testing-name new)
      (setf new-p (remove-if-not #'(lambda (e) (member e pos)) new-e)
	    new-n (remove-if-not #'(lambda (e) (member e neg)) new-e))
      (setf (r-pos new) new-p
	    (r-neg new) new-n))))

(defun judge-pos-and-neg (pos neg &optional (rule 'illegal)
			  &aux (correct 0)
			       (total 0)
			       (rule-fun (r-prolog-function (get-rule rule))))

  (dolist (example pos)
    (incf total)
    (when (prove-function?  rule-fun nil nil example)
      (incf correct)))
  (dolist (example neg)
    (incf total)
    (unless (prove-function?  rule-fun nil nil example)
      (incf correct)))
  (/ correct total))