;;; -*- Mode:Common-Lisp; Package:user; Base:10 -*-

; Implement a discrimination tree using code from Norvig's book Section
; 14.8.

(in-package :user)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(export '(clear-cache insert-cache query-cache))

;; An nlist is implemented as a (count . elements) pair:
(defun make-empty-nlist () 
  "Create a new, empty nlist."
  (cons 0 nil))

(defun nlist-n (x) "The number of elements in an nlist." (car x))
(defun nlist-list (x) "The elements in an nlist." (cdr x))

(defun nlist-push (item nlist)
  "Add a new element to an nlist."
  (incf (car nlist))
  (push item (cdr nlist))
  nlist)

(defstruct (dtree (:type vector))
  (first nil) (rest nil) (atoms nil) (var (make-empty-nlist)))

(defun lookup (key alist)
  (cdr (assoc key alist)))

(defun lookup-atom (atom dtree)
  "Return (or create) the nlist for this atom in dtree."
  (or (lookup atom (dtree-atoms dtree))
      (let ((new (make-empty-nlist)))
        (push (cons atom new) (dtree-atoms dtree))
        new)))

(defun dtree-index (key value dtree)
  "Index value under all atoms of key in dtree."
  (cond
    ((consp key)               ; index on both first and rest
     (dtree-index (first key) value
                  (or (dtree-first dtree)
                      (setf (dtree-first dtree) (make-dtree))))
     (dtree-index (rest key) value
                  (or (dtree-rest dtree)
                      (setf (dtree-rest dtree) (make-dtree)))))
    ((null key))               ; don't index on nil
    ((var? key)          ; index a variable
     (nlist-push value (dtree-var dtree)))
    (t ;; Make sure there is an nlist for this atom, and add to it
     (nlist-push value (lookup-atom key dtree)))))

(defun dtree-fetch (pat dtree var-list-in var-n-in best-list best-n)
  "Return two values: a list-of-lists of possible matches to pat,
  and the number of elements in the list-of-lists."
  (if (or (null dtree) (null pat) (var? pat))
      (values best-list best-n)
      (let* ((var-nlist (dtree-var dtree))
             (var-n (+ var-n-in (nlist-n var-nlist)))
             (var-list (if (null (nlist-list var-nlist))
                           var-list-in
                           (cons (nlist-list var-nlist)
                                 var-list-in))))
        (cond
          ((>= var-n best-n) (values best-list best-n))
          ((atom pat) (dtree-atom-fetch pat dtree var-list var-n
                                        best-list best-n))
          (t (multiple-value-bind (list1 n1)
                 (dtree-fetch (first pat) (dtree-first dtree)
                              var-list var-n best-list best-n)
               (dtree-fetch (rest pat) (dtree-rest dtree)
                            var-list var-n list1 n1)))))))

(defun dtree-atom-fetch (atom dtree var-list var-n best-list best-n)
  "Return the answers indexed at this atom (along with the vars),
  or return the previous best answer, if it is better."
  (let ((atom-nlist (lookup atom (dtree-atoms dtree))))
    (cond
      ((or (null atom-nlist) (null (nlist-list atom-nlist)))
       (values var-list var-n))
      ((and atom-nlist (< (incf var-n (nlist-n atom-nlist)) best-n))
       (values (cons (nlist-list atom-nlist) var-list) var-n))
      (t (values best-list best-n)))))

(defvar *cache-predicates* nil)

(defun get-cache (predicate)
  "Fetch (or make) the dtree cache for this predicate."
  (cond ((get predicate 'cache))
	(t (push predicate *cache-predicates*)
	   (setf (get predicate 'cache) (make-dtree)))))
  
(defun clear-cache ()
  "Remove all the caches for all the predicates."
  (dolist (predicate *cache-predicates*)
    (setf (get predicate 'cache) nil))
  (setf *cache-predicates* nil))

(defun insert-cache (node depth)
  (dtree-index (aaa::node-datum node) (cons depth node)
	       (get-cache (predicate (aaa::node-datum node)))))

; Return two values: node and exact-match?.  The node returned is an
; exact match if possible; else it is a generalization of node-datum if
; possible; else return nil.  The node returned has depth >= the given
; depth.

(defun query-cache (node-datum depth &aux poss-matches)
  (multiple-value-setq (poss-matches)
    (dtree-fetch node-datum (get-cache (predicate node-datum))
		 nil 0 nil most-positive-fixnum))
  (dolist (bucket poss-matches)
    (dolist (b bucket)
      (when (and (>= (car b) depth)
		 (equal (aaa::node-datum (cdr b)) node-datum))
	(return-from query-cache (values (cdr b) t)))))
  (dolist (bucket poss-matches)
    (dolist (b bucket)
      (when (and (>= (car b) depth)
		 (instance-of? node-datum (aaa::node-datum (cdr b))))
	(return-from query-cache (values (cdr b) nil))))))
