;;; -*- Mode:Common-Lisp; Package:aaa; Base:10 -*-

;;;; Copyright (c) 1992 by Hwee Tou Ng. This program may be freely
;;;; copied, used, or modified provided that this copyright notice
;;;; is included in each copy of this code and parts thereof.

(in-package :aaa)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(use-package 'user)

(defvar *trace-cc* nil)

; Return t iff env is inconsistent.

(defun inconsistent? (env &aux assumptions)
  (when (or (and (eq user::*domain* 'user::plan-recognition)
		 (check-pr env))
	    (and (eq user::*domain* 'user::qsim-diagnosis)
		 (check-qval env))
	    (and (eq user::*domain* 'user::circ-diagnosis)
		 (check-circ-val env)))
    (return-from inconsistent? t))
  (when (some #'(lambda (nogood)
		  (subsume? nogood (environment-assumptions env)))
	      user::*assumption-nogoods*)
    (return-from inconsistent? t))
  (when user::*nogoods*
    (setf assumptions (environment-assumptions env))
    (dolist (a *input-atoms*)
      (pushnew a assumptions :test #'equal))
    (some #'(lambda (nogood)
	      (subsume? nogood assumptions))
	  user::*nogoods*)))

(defun check-pr (env)
  (or (check-uniq-arg env)
      (sort-inconsistent? env)
      (check-precedence (environment-assumptions env))))

; Enforce the constraints:
; (plan-step ?x ?z) and (plan-step ?y ?z) -> (= ?x ?y)
; (uniq-pred ?x ?y) and (uniq-pred ?x ?z) -> (= ?y ?z)

(defun check-uniq-arg (env &aux plan-steps uniq-preds
		       distinct-plan-steps distinct-uniq-preds b b2)
  (dolist (a (environment-assumptions env))
    (when (get (predicate a) 'user::plan-step)
      (push a plan-steps))
    (when (get (predicate a) 'user::uniq-pred)
      (push a uniq-preds)))
  (dolist (a *input-atoms*)
    (when (get (predicate a) 'user::plan-step)
      (push a plan-steps))
    (when (get (predicate a) 'user::uniq-pred)
      (push a uniq-preds)))
  (dolist (plan-step plan-steps)
    (unless (some #'(lambda (dplan-step)
		      (and (eq (first plan-step) (first dplan-step))
			   (eq (third plan-step) (third dplan-step))))
		  distinct-plan-steps)
      (push plan-step distinct-plan-steps)))
  (dolist (uniq-pred uniq-preds)
    (unless (some #'(lambda (duniq-pred)
		      (and (eq (first uniq-pred) (first duniq-pred))
			   (eq (second uniq-pred) (second duniq-pred))))
		  distinct-uniq-preds)
      (push uniq-pred distinct-uniq-preds)))
  (setf b *empty-bindings*)
  (dolist (dplan-step distinct-plan-steps)
    (dolist (plan-step plan-steps)
      (when (and (not (equal plan-step dplan-step))
		 (eq (first plan-step) (first dplan-step))
		 (eq (third plan-step) (third dplan-step)))
	(setf b2 (unify plan-step dplan-step b))
	(when (null b2)
	  (return-from check-uniq-arg t))
	(setf b (join-bindings b b2)))))
  (dolist (duniq-pred distinct-uniq-preds)
    (dolist (uniq-pred uniq-preds)
      (when (and (not (equal uniq-pred duniq-pred))
		 (eq (first uniq-pred) (first duniq-pred))
		 (eq (second uniq-pred) (second duniq-pred)))
	(setf b2 (unify uniq-pred duniq-pred b))
	(when (null b2)
	  (return-from check-uniq-arg t))
	(setf b (join-bindings b b2)))))
  (unless (equal b *empty-bindings*)
    (setf (environment-assumptions env)
	  (delete-duplicates (substitute-bindings (environment-assumptions env) b)
			     :test #'equal :from-end t))
    (setf (environment-rules env)
	  (substitute-bindings (environment-rules env) b))
    (setf (environment-subst env)
	  (join-bindings (environment-subst env) b)))
  nil)

; Check for inconsistency of sort assumptions.
; Return t iff the sort assertions are inconsistent.
; Also, add the appropriate rules
; (inst ?x subsort) -> (inst ?x supersort)
; to env.
; In addition, check role-predicates to make sure that
; instance constraining rules are not violated, and
; add the appropriate instance-constraining rules
; (inst ?x ?a) and (role-predicate ?x ?y) -> (inst ?y ?b)
; to env

(defun sort-inconsistent? (env &aux inst-atoms role-pred-atoms sorts most-subsort
			   ante conseq id brule rule-pred conseqs sort-y)
  (dolist (a (environment-assumptions env))
    (when (inst-atom? a)
      (pushnew a inst-atoms :test #'equal)))
  (dolist (a *input-atoms*)
    (when (inst-atom? a)
      (pushnew a inst-atoms :test #'equal)))
  (dolist (sort-term (delete-duplicates (mapcar #'second inst-atoms)))
    (setf sorts
	  (mapcan #'(lambda (atom) (if (and (eq (second atom) sort-term)
					    (get (third atom) 'user::sort-symbol?))
				       (list (third atom))))
		  inst-atoms))
    (if (not (compatible-sort sorts))
	(return-from sort-inconsistent? t))
    ; add the appropriate rules to env
    (when (rest sorts)
      (setf most-subsort
	    (dolist (s1 sorts)
	      (if (every #'(lambda (s2) (or (eq s1 s2) (subsort? s1 s2)))
			 sorts)
		  (return s1))))
      (setf ante (list 'user::inst sort-term most-subsort))
      (dolist (s2 sorts)
	(unless (eq most-subsort s2)
	  (setf conseq (list 'user::inst sort-term s2))
	  (setf (environment-assumptions env)
		(delete conseq (environment-assumptions env) :test #'equal))
	  (setf rule-pred (gensym))
	  (setf id (list rule-pred sort-term))
	  (setf brule (make-brule :id id
				  :conseq conseq
				  :antes (list ante)))
	  (setf (get rule-pred 'user::brule) brule)
	  (push id (environment-rules env))))))
  (setf conseqs
	(mapcar #'(lambda (r &aux brule)
		    (setf brule (get (predicate r) 'user::brule))
		    (substitute-bindings (user::brule-conseq brule)
					 (unify (user::brule-id brule) r)))
		(environment-rules env)))
  (setf inst-atoms nil)
  (dolist (a (environment-assumptions env))
    (cond ((inst-atom? a)
	   (push a inst-atoms))
	  ((get (predicate a) 'user::role-pred)
	   (push a role-pred-atoms))))
  (dolist (a *input-atoms*)
    (cond ((inst-atom? a)
	   (push a inst-atoms))
	  ((get (predicate a) 'user::role-pred)
	   (push a role-pred-atoms))))
  (dolist (a conseqs)
    (cond ((inst-atom? a)
	   (push a inst-atoms))
	  ((get (predicate a) 'user::role-pred)
	   (push a role-pred-atoms))))
  (dolist (a role-pred-atoms)
    (dolist (a2 inst-atoms)
      (when (and (eq (second a2) (second a))
		 (setf sort-y
		       (second (assoc (third a2)
				      (get (first a) 'user::inst-constr)))))
	(dolist (a3 inst-atoms)
	  (when (eq (second a3) (third a))
	    (when (incompatible-sort (third a3) sort-y)
	      (return-from sort-inconsistent? t))
	    (when (and (eq (third a3) sort-y)
		       (member a3 (environment-assumptions env) :test #'equal))
	      (setf (environment-assumptions env)
		    (delete a3 (environment-assumptions env) :test #'equal))
	      (setf rule-pred (gensym))
	      (setf id (cons rule-pred (rest a)))
	      (setf brule (make-brule :id id :conseq a3 :antes (list a2 a)))
	      (setf (get rule-pred 'user::brule) brule)
	      (push id (environment-rules env)))))))))

; Enforce the constraints:
; (plan-step ?x ?x) -> (falsity)
; (plan-step ?x ?y) and (precede ?x ?y) -> (falsity)
; (inst ?x plan) and (step-1 ?x ?1) and (step-2 ?x ?2) -> (precede ?1 ?2)
; (plan-step ?x ?y) and (precede ?x ?z) -> (precede ?y ?z)
; Return t if some precedence relation is violated. Else return nil.

(defun check-precedence (assumptions &aux (first-precede? t) precedes
			 insts plan-steps plan-objs plan-type plan-step
			 actions actions-list right-siblings)
  (dolist (i *input-atoms*)
    (when (eq (predicate i) 'user::precede)
      (when first-precede?
	(push (third i) precedes)
	(setf first-precede? nil))
      (push (second i) precedes)))
  (dolist (a assumptions)
    (cond ((inst-atom? a)
	   (push a insts))
	  ((get (predicate a) 'user::plan-step)
	   (push a plan-steps))))
  
  (dolist (p plan-steps)
    (when (or (equal (second p) (third p))
	      (incorrect-temporal-order (third p) (second p) precedes))
      (return-from check-precedence t)))

  (dolist (p plan-steps)
    (pushnew (second p) plan-objs))
  (dolist (o plan-objs)
    (setf actions nil)
    (setf plan-type (third (find o insts :test #'eq :key #'second)))
    (dolist (subplan (rest (assoc plan-type user::*plan-steps*)))
      (when (setf plan-step (find-if #'(lambda (ps) (and (eq (first ps) subplan)
							 (eq (second ps) o)))
				     plan-steps))
	(push (third plan-step) actions)))
    (setf actions (nreverse actions))
    (do ((rem actions (rest rem)))
	((null (rest rem)))
      (when (incorrect-temporal-order (first rem) (second rem) precedes)
	(return-from check-precedence t)))      
    (push o actions)
    (push actions actions-list))
  (dolist (actions actions-list)
    (when (setf right-siblings
		(rest (some #'(lambda (as)
				(member (first actions) (rest as) :test #'eq))
			    actions-list)))
      (dolist (a (rest actions))
	(when (incorrect-temporal-order a (first right-siblings) precedes)
	  (return-from check-precedence t)))))
  )

(defun incorrect-temporal-order (a1 a2 precedes &aux p1 p2)
  (and (setf p1 (position a1 precedes))
       (setf p2 (position a2 precedes))
       (> p1 p2)))

; (The following two procedures can be combined into one general
; procedure that checks for equality of value atoms)

; Domain-dependent procedure to check consistency of qvals,
; and perform forced unification of qvals if necessary.
; This procedure encodes the rule:
; (qval ?v ?m1 ?d1 ?t) and (qval ?v ?m2 ?d2 ?t) 
; ->  (= ?m1 ?m2) and (= ?d1 ?d2)
; Return t if inconsistent, else return nil (and have the side effect 
; of possibly changing the given env by unifying qvals).

(defun check-qval (env &aux qvals distinct-qvals b b2)
  ; check <>
  (when (or (procedural-eval-inconsistent? (environment-assumptions env))
	    (behavior-mode-inconsistent? (environment-assumptions env)))
    (return-from check-qval t))

  ; check corresponding magnitude
  (when (check-corr-mag (environment-assumptions env))
    (return-from check-qval t))
  
  (setf qvals
	(mapcan #'(lambda (r &aux brule conseq)
		    (setf brule (get (predicate r) 'user::brule))
		    (setf conseq (user::brule-conseq brule))
		    (when (qval? conseq)
		      (list (substitute-bindings
			      conseq (unify (user::brule-id brule) r)))))
		(environment-rules env)))
  (dolist (a (environment-assumptions env))
    (when (qval? a)
      (pushnew a qvals :test #'equal)))
  (dolist (a *input-atoms*)
    (when (qval? a)
      (pushnew a qvals :test #'equal)))
  (dolist (qval qvals)
    (unless (some #'(lambda (dqval) (and (eq (second qval) (second dqval))
					 (eq (fifth qval) (fifth dqval))))
		  distinct-qvals)
      (push qval distinct-qvals)))
  (setf b *empty-bindings*)
  (dolist (dqval distinct-qvals)
    (dolist (qval qvals)
      (when (and (not (equal qval dqval))
		 (eq (second qval) (second dqval))
		 (eq (fifth qval) (fifth dqval)))
	(setf b2 (unify qval dqval b))
	(when (null b2)
	  (return-from check-qval t))
	(setf b (join-bindings b b2)))))
  (unless (equal b *empty-bindings*)
    (setf (environment-assumptions env)
	  (delete-duplicates (substitute-bindings (environment-assumptions env) b)
			     :test #'equal :from-end t))
    (setf (environment-rules env)
	  (substitute-bindings (environment-rules env) b))
    (setf (environment-subst env)
	  (join-bindings (environment-subst env) b)))
  nil)

(defun qval? (atom)
  (eq (predicate atom) 'user::qval))

; Domain-dependent procedure to check consistency of circuit values,
; as expressed in the atoms with predicates in1, in2, and out.
; Perform forced unification of these atoms if necessary.
; This procedure encodes the rules:
; (in1 ?x ?u ?t) and (in1 ?x ?v ?t) -> (= ?u ?v)
; (in2 ?x ?u ?t) and (in2 ?x ?v ?t) -> (= ?u ?v)
; (out ?x ?u ?t) and (out ?x ?v ?t) -> (= ?u ?v)
; Return t if inconsistent, else return nil (and have the side effect 
; of possibly changing the given env by unifying qvals).

(defun check-circ-val (env &aux circ-vals distinct-circ-vals b b2)
  (when (behavior-mode-inconsistent? (environment-assumptions env))
    (return-from check-circ-val t))
  (setf circ-vals
	(mapcan #'(lambda (r &aux brule conseq)
		    (setf brule (get (predicate r) 'user::brule))
		    (setf conseq (user::brule-conseq brule))
		    (when (circ-val? conseq)
		      (list (substitute-bindings
			      conseq (unify (user::brule-id brule) r)))))
		(environment-rules env)))
  (dolist (a (environment-assumptions env))
    (when (circ-val? a)
      (pushnew a circ-vals :test #'equal)))
  (dolist (a *input-atoms*)
    (when (circ-val? a)
      (pushnew a circ-vals :test #'equal)))
  (dolist (circ-val circ-vals)
    (unless (some #'(lambda (dcirc-val) (and (eq (first circ-val) (first dcirc-val))
					     (eq (second circ-val) (second dcirc-val))
					     (eq (fourth circ-val) (fourth dcirc-val))))
		  distinct-circ-vals)
      (push circ-val distinct-circ-vals)))
  (setf b *empty-bindings*)
  (dolist (dcirc-val distinct-circ-vals)
    (dolist (circ-val circ-vals)
      (when (and (not (equal circ-val dcirc-val))
		 (eq (first circ-val) (first dcirc-val))
		 (eq (second circ-val) (second dcirc-val))
		 (eq (fourth circ-val) (fourth dcirc-val)))
	(setf b2 (unify circ-val dcirc-val b))
	(when (null b2)
	  (return-from check-circ-val t))
	(setf b (join-bindings b b2)))))
  (unless (equal b *empty-bindings*)
    (setf (environment-assumptions env)
	  (delete-duplicates (substitute-bindings (environment-assumptions env) b)
			     :test #'equal :from-end t))
    (setf (environment-rules env)
	  (substitute-bindings (environment-rules env) b))
    (setf (environment-subst env)
	  (join-bindings (environment-subst env) b)))
  nil)

(defun circ-val? (atom &aux predicate)
  (setf predicate (predicate atom))
  (or (eq predicate 'user::in1)
      (eq predicate 'user::in2)
      (eq predicate 'user::out)))

(defun behavior-mode-inconsistent? (assumptions &aux mode)
  (dolist (c user::*components*)
    (setf mode nil)
    (dolist (a assumptions)
      (when (and (behavior-mode-assumption? a)
		 (eq (second a) c))
	(if (null mode)
	    (setf mode (predicate a))
	    (when (not (eq (predicate a) mode))
	      (return-from behavior-mode-inconsistent? t)))))))


(defun procedural-eval-inconsistent? (assumptions)
  (some #'(lambda (a) (and (procedural-form? a)
			   (not (apply (first a) (rest a)))))
	assumptions))

(defvar user::*corr-mags*)

(defun check-corr-mag (assumptions)
  (when user::*corr-mags*
    (dolist (a assumptions)
      (when (and (eq (predicate a) 'user::corr-mag.m0+)
		 (not (var? (fourth a)))
		 (not (var? (fifth a)))
		 (some #'(lambda (cm)
			   (and (eq (second a) (second cm))
				(eq (third a) (third cm))
				(or (and (eq (fourth a) (fourth cm))
					 (not (eq (fifth a) (fifth cm))))
				    (and (eq (fifth a) (fifth cm))
					 (not (eq (fourth a) (fourth cm)))))))
		       user::*corr-mags*))
	(return-from check-corr-mag t)))))
