;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;;; Copyright (c) 1992 by Hwee Tou Ng. This program may be freely
;;;; copied, used, or modified provided that this copyright notice 
;;;; is included in each copy of this code and parts thereof.

(in-package :user)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(export '(*empty-bindings* var? skconst?
	  init-var-count new-var-count
	  init-skconst-count new-skconst-count
	  init-const-count new-const-count 
	  empty-bindings? failed-bindings?
	  unify join-bindings
	  substitute-bindings uniquify
	  vars/skconsts-in-bindings terms-in-bindings skolemize
	  collect-vars collect-skconsts
	  procedural-form? evaluable-procedural-form?
	  contain-var? ground? contain-skconst?
	  instance-of? instantiated? subsume?
	  row-print-list trace-print trace-row-print
	  print-vector compute-run-time print-run-time
	  predicate same-predicate?
	  lex> number-of-1-bits n-choose-2 average
	  first-n first-n-destr
	  set-< set-<= minimal-set minimal-set-push
	  cartesian-product
	  alphabetic-variant-subset?
	  alphabetic-variant-set=?
	  renaming-subst? alphabetic-variant?
	  defhierarchy undefhierarchy sort? subsort?
	  compatible-sort compatible-sort-pair
	  incompatible-sort primitive-sort?
	  inst-atom? inst-sort-atom? parent-sort))


;  The code for unification was originally written by Ray Mooney.
;  Modified by htng to deal with skolem constants.
;
;------------------------------------------------------------------------------
;                               UNIFICATION 
;------------------------------------------------------------------------------

;;; This section defines the unification pattern matcher. A variable must
;;; begin with a leading "?". A skolem constant must begin with "SK"
;;; followed by a number.
;;; The unify function takes two patterns and a binding list and unifies the
;;; patterns in the context of the current bindings and returns an updated
;;; binding list.  A binding list is of the form:
;;; (T (<var> <value>) (<var> <value>) ... )
;;; The leading T is used to distinguish the empty binding list: (T) from
;;; failure to unify for which NIL is returned.
;;; See AI Programming 2nd ed. for details (slightly different).

; Print out a note whenever variant-subsets are detected.
(defparameter *trace-variant-subset* nil)

(defconstant *empty-bindings* '(t))
(defparameter *occur-check* t)  ; Performs an occur check if T
(defvar *var-count* 0) ; number used for newly created vars
(defvar *skconst-count* 0) ; number used for newly created skolem constants
(defvar *const-count* 0) ; number used for newly created constants

(defun var? (pat)
  (and (symbolp pat)
       (char= (char (string pat) 0) #\?)))

(defun skconst? (pat &aux string length)
  (when (symbolp pat)
    (setf string (string pat)
	  length (length string))
    (and (> length 2)
	 (char= (char string 0) #\S)
	 (char= (char string 1) #\K)
	 (do ((i 2 (+ i 1)))
	     ((= i length) t)
	   (unless (digit-char-p (char string i))
	     (return nil))))))

(defun init-var-count ()
  (setf *var-count* 0))

(defun init-skconst-count ()
  (setf *skconst-count* 0))

(defun init-const-count ()
  (setf *const-count* 0))

(defun new-var-count ()
  (incf *var-count*))

(defun new-skconst-count ()
  (incf *skconst-count*))

(defun new-const-count ()
  (incf *const-count*))

(defun make-new-var (form)
  (declare (ignore form))
  (intern (concatenate 'string "?" (prin1-to-string (new-var-count))) 'user))

(defun make-new-skconst (form)
  (declare (ignore form))
  (intern (concatenate 'string "SK" (prin1-to-string (new-skconst-count))) 'user))

(defun make-new-const (form)
  (declare (ignore form))
  (intern (concatenate 'string "C" (prin1-to-string (new-const-count))) 'user))

(defun empty-bindings? (bindings)
  (equal bindings *empty-bindings*))

(defun failed-bindings? (bindings)
  (null bindings))

; Note that (unify '?a '?b) returns (t (?a ?b)).
; That is, if f1 and f2 are variants, then
; (unify f1 f2) returns the mgu U such that
; (f1)U = f2.

; Note: The default is that skolem constants are unifiable.
;       sk=consts is a list of skolem constants to be treated
;       as constants (as for example in forward chaining).

; Return a most general binding list which unifies a & b

(defun unify (a b &optional (bindings *empty-bindings*)
	      &key sk=consts &aux bindings2)
  (cond ((eq a b) bindings)
	((var-or-unifiable-skconst? a sk=consts)
	 (var-unify a b bindings sk=consts))
	((var-or-unifiable-skconst? b sk=consts)
	 (var-unify b a bindings sk=consts))
	((or (atom a) (atom b)) nil)
	((setf bindings2
	       (unify (first a) (first b) bindings :sk=consts sk=consts))
	 (unify (rest a) (rest b) bindings2 :sk=consts sk=consts))))

; Unify a variable or unifiable skolem constant with a wff b.
; If must bind var and *occur-check* flag is set,
; then check for occur-check violation.

(defun var-unify (var b bindings sk=consts)
  (if (eq var b)
      bindings
      (let ((binding (get-binding var bindings)))
	(cond (binding
	       (unify (second binding) b bindings :sk=consts sk=consts))
	      ((and (var-or-unifiable-skconst? b sk=consts)
		    (bound-to-p b var bindings))
	       bindings)  ; prevent cycle
	      ((or (null *occur-check*) (free-in-p var b bindings))
	       (add-binding var b bindings))))))

(defun var-or-unifiable-skconst? (a sk=consts)
  (or (var? a)
      (and (skconst? a) (not (member a sk=consts :test #'eq)))))

(defun get-binding (var bindings)
  ;;; Get the variable binding for var
  (assoc var (rest bindings)))

(defun add-binding (var val bindings)
  ;;; Add the binding of var to val to the existing set of bindings
  (append `(t (,var ,val)) (rest bindings)))

(defun join-bindings (bindings1 bindings2)
  (append bindings1 (rest bindings2)))

(defun bound-to-p (var1 var2 bindings)
  ;;; Check if var1 is eventually bound to var2 in the bindings
  (cond ((eq var1 var2) t)
	((let ((val (second (get-binding var1 bindings))))
	   (and val
		(or (var? val) (skconst? val))
		(bound-to-p val var2 bindings))))))

(defun free-in-p (var b bindings)
  ;;; Return T if var does not occur in wff b
  (cond ((eq var b) nil)
	((or (var? b) (skconst? b))
	 (free-in-p var (second (get-binding b bindings)) bindings))
	((atom b) t)
	((and (free-in-p var (first b) bindings)
	      (free-in-p var (rest b) bindings)))))

; Note: Always return a new copy of form substituted from bindings.

(defun substitute-bindings (form bindings)
  (cond ((null form) nil)
	((or (var? form) (skconst? form))
	 (let ((binding (get-binding form bindings)))
	   (if binding
	       (substitute-bindings (second binding) bindings)
	       form)))
	((atom form) form)
	(t (cons (substitute-bindings (first form) bindings)
		 (substitute-bindings (rest form) bindings)))))

(defun uniquify (form &key (new-var-fn #'make-new-var)
		           (new-skconst-fn #'make-new-skconst))
  ;;; Make all the variables and skolem constants in form "unique"
  (let ((new-names (rename-list form nil new-var-fn new-skconst-fn)))
    (if (null new-names)
	form
	(rename form new-names))))

(defun rename-list (form new-names new-var-fn new-skconst-fn)
  (cond ((var? form)
	 (if (assoc form new-names)
	     new-names
	     (cons (list form (funcall new-var-fn form)) new-names)))
	((skconst? form)
	 (if (assoc form new-names)
	     new-names
	     (cons (list form (funcall new-skconst-fn form)) new-names)))
	((consp form)
	 (rename-list (rest form)
		      (rename-list (first form) new-names new-var-fn new-skconst-fn)
		      new-var-fn new-skconst-fn))
	(t new-names)))

(defun rename (form new-names)
  (cond ((or (var? form) (skconst? form))
	 (let ((entry (assoc form new-names)))
	   (if entry (second entry) form)))
	((atom form) form)
	(t (cons (rename (first form) new-names)
		 (rename (rest form) new-names)))))

(defun vars/skconsts-in-bindings (b)
  (mapcar #'first (rest b)))

(defun terms-in-bindings (b)
  (mapcar #'second (rest b)))

; Replace variables with newly created skolem constants.

(defun skolemize (expr)
  (sublis (mapcar #'(lambda (var) (cons var (make-new-skconst var)))
		  (collect-vars expr))
	  expr))

; Return a bindings such that all variables
; are bound to their corresponding terms (i.e. no "transitive chain").

(defun canonical-bindings (bindings)
  (cons 't
	(mapcar #'(lambda (b)
		    (list (first b)
			  (substitute-bindings (second b) bindings)))
		(rest bindings))))

; Return a list of all the variables occurring in the expression expr.

(defun collect-vars (expr)
  (labels ((cvs (expr)
		(cond ((var? expr) (list expr))
		      ((atom expr) nil)
		      (t (mapcan #'cvs expr)))))
    (delete-duplicates (cvs expr) :test #'eq)))

; Return a list of all the skolem constants occurring in the expression expr.

(defun collect-skconsts (expr)
  (labels ((csks (expr)
		 (cond ((skconst? expr) (list expr))
		       ((atom expr) nil)
		       (t (mapcan #'csks expr)))))
    (remove-duplicates (csks expr) :test #'eq)))

(defvar *procedurally-defined-predicates* '(<> incompatible-sort))

(defun procedural-form? (form)
  (and (consp form)
       (member (first form) *procedurally-defined-predicates*)))
  
(defun evaluable-procedural-form? (form)
  (and (procedural-form? form)
       (not (contain-var? form))
       (not (contain-skconst? form))))

(defun contain-var? (form)
  (cond ((var? form) t)
        ((atom form) nil)
        ((or (contain-var? (first form))
             (contain-var? (rest form))))))

(defun ground? (form)
  (not (contain-var? form)))

(defun contain-skconst? (form)
  (cond ((skconst? form) t)
        ((atom form) nil)
        ((or (contain-skconst? (first form))
             (contain-skconst? (rest form))))))

(defun instance-of? (n1 n2)
  (unify n2 (uniquify n1 :new-var-fn #'make-new-const)))

; Return t iff bindings instantiate a.

(defun instantiated? (a bindings)
  (not (equal (substitute-bindings a bindings) a)))

; Return t iff the set of assumptions a1s subsumes the set of assumptions a2s.
; Use the subsumption algorithm in Chang & Lee, Pg 95.

(defun subsume? (a1s a2s)
  (do ((w (uniquify a2s :new-var-fn #'make-new-const))
       (uk (list a1s))
       (uk+1 nil)
       b)
      ((member '() uk) t)
    (dolist (a1s uk)
      (dolist (a2 w)
	(dolist (a1 a1s)
	  (if (setf b (unify a1 a2))
	      (push (substitute-bindings (remove a1 a1s :test #'equal :count 1)
					 b)
		    uk+1)))))
    (cond ((null uk+1) (return-from subsume? nil))
	  (t (setf uk uk+1) (setf uk+1 nil)))))


;-------------------------------------
;        Utility functions
;-------------------------------------

; Print the elements in the list l row by row.

(defun row-print-list (l &optional (double-space nil) (header-msg nil)
		       (print-length nil) (stream t) (num-elts-per-row 1))
  (when header-msg
    (format stream header-msg))
  (when print-length
    (format stream "~%length = ~D~%" (length l)))
  (cond ((null l) (format stream "~%nil~%"))
	(t
	 (format stream "~%")
	 (do ((rem l))
	     ((null rem))
	   (dotimes (i num-elts-per-row)
	     (format stream "~s " (first rem))
	     (setf rem (rest rem))
	     (if (null rem) (return)))
	   (format stream "~:[~%~;~%~%~]" double-space))))
  (values))

(defmacro trace-print (test-var &rest format-form)
  `(if ,test-var (format t ,@format-form)))

(defmacro trace-row-print (test-var l header)
  `(if ,test-var (row-print-list ,l t ,header t)))

; print the contents of a vector

(defun print-vector (v stream)
  (cond ((null v) (format stream "nil"))
	(t (format stream "[ ")
	   (dotimes (i (first (array-dimensions v)))
	     (format stream "~A " (aref v i)))
	   (format stream "]"))))

(defun compute-run-time (start-time)
  (/ (/ (- (get-internal-run-time) start-time)
	internal-time-units-per-second)
     60))

(defun print-run-time (start-time)
  (format t "~%Run Time = ~,2F min~%"
	  (/ (/ (- (get-internal-run-time) start-time)
		internal-time-units-per-second)
	     60)))

(defun <> (x y) (not (equal x y)))

(defun predicate (datum)
  (first datum))

(defun same-predicate? (p1 p2)
  (eq (predicate p1) (predicate p2)))

; lexicographic > relation :
; Given two lists of numbers l1 and l2, return t
; iff l1 lexicographically precedes l2.

(defun lex> (l1 l2)
  (cond ((null l1) nil)
	((null l2) t)
	((> (first l1) (first l2)) t)
	((= (first l1) (first l2)) (lex> (rest l1) (rest l2)))
	(t nil)))

; Return the number of one bits present in the given bit vector.

(defun number-of-1-bits (bit-vector &aux (sum 0))
  (dotimes (i (first (array-dimensions bit-vector)) sum)
    (incf sum (bit bit-vector i))))

(defun n-choose-2 (n)
  (if (or (= n 0) (= n 1))
      0
      (/ (* n (1- n)) 2)))

; Assume that list l is not nil.

(defun average (l)
  (/ (apply '+ l) (length l)))

; Return the first n elements of a list. Assume n >= 0.
; If the list has less than n elements, return the list.
; This function is non-destructive and list is unchanged.

(defun first-n (list n)
  (if (or (null list) (= n 0))
      nil
      (cons (first list) (first-n (rest list) (1- n)))))

; Return the first n elements of a list. Assume n >= 0.
; If the list has less than n elements, return the list.
; This is the destructive version and list is modified.

(defun first-n-destr (list number)
  (labels ((1st-n (l n)
		  (cond ((null (cdr l))
			 list)
			((= n 1)
			 (setf (cdr l) nil)
			 list)
			(t (1st-n (rest l) (1- n))))))
    (if (or (null list) (= number 0))
	nil
	(1st-n list number))))

; Return t iff s is a set with no duplicate elements.

(defun set-p (s &key (test #'eql))
  (or (null s)
      (and (set-p (rest s))
	   (every #'(lambda (e) (not (funcall test (first s) e)))
		  (rest s)))))

(defun set-< (s1 s2)
  (and (< (length s1) (length s2))
       (subsetp s1 s2 :test #'equal)))

(defun set-<= (s1 s2)
  (and (<= (length s1) (length s2))
       (subsetp s1 s2 :test #'equal)))

(defun minimal-set (s &key (keyfn #'identity) (fn-<= #'set-<=)
		      &aux m-s)
  (dolist (elt s (nreverse m-s))
    (setf m-s
	  (minimal-set-push elt m-s :keyfn keyfn :fn-<= fn-<=))))

(defun minimal-set-push (elt m-s &key (keyfn #'identity) (fn-<= #'set-<=))
  (unless (some #'(lambda (m-elt) (funcall fn-<=
					   (funcall keyfn m-elt)
					   (funcall keyfn elt)))
		m-s)
    (setf m-s (delete-if #'(lambda (m-elt) (funcall fn-<=
						    (funcall keyfn elt)
						    (funcall keyfn m-elt)))
			 m-s))
    (push elt m-s))
  m-s)

; Given a list of lists of coordinate elements, return the cartesian product.
; eg. (cartesian-product '((a1 a2) (b1 b2 b3) (c1))) -> ((a1 b1 c1) ...)

(defun cartesian-product (l)
  (if (null l)
      '(())
      (let ((cp (cartesian-product (rest l))))
	(mapcan #'(lambda (a)
		    (mapcar #'(lambda (b) (cons a b)) cp))
		(first l)))))

; Return the appropriate bindings iff an alphabetic variant of set1 is a
; subset of set2; else return nil.
; NOTE: ASSUME that the atoms in set1 and set2 contain only variables.

(defun alphabetic-variant-subset? (set1 set2 &aux result)
  (setf result
	(and (<= (length set1) (length set2))
	     (avs? set1 set2 *empty-bindings* nil)))
  (when (and *trace-variant-subset*
	     result
	     (not (equal result *empty-bindings*))
	     (< (length set1) (length set2)))
    (format t "~%variant-subset detected:~%")
    (format t "smaller set:~%~A~%" set1)
    (format t "larger set:~%~A~%" set2)
    (format t "bindings:~%~A~%" result))
  result)

(defun alphabetic-variant-set=? (set1 set2)
  (and (= (length set1) (length set2))
       (avs? set1 set2 *empty-bindings* nil)))

(defun avs? (s1 s2 b s1-in)
  (if (null s1)
      b
      (some #'(lambda (e2 &aux b2 new-s1-in)
		(setf b2 (unify (first s1) e2 b))
		(and b2
		     (renaming-subst? b2 (setf new-s1-in (cons (first s1) s1-in)))
		     (avs? (rest s1)
			   (remove e2 s2 :test #'eq :count 1)
			   b2 new-s1-in)))
	    s2)))

; NOTE: A renaming substitution only substitutes
;       vars for vars.

(defun renaming-subst? (b s &aux b2 vars terms)
  (setf b2 (canonical-bindings b))
  (setf vars (vars/skconsts-in-bindings b2))
  (setf terms (terms-in-bindings b2))
  (and (every #'var? terms)
       (set-p terms :test #'eq)
       (null (nintersection (nset-difference (collect-vars s) vars)
			    terms))))

(defun alphabetic-variant? (term1 term2)
  (alphabetic-variant-subset? (list term1) (list term2)))


;--------------------------------
;     Sort hierarchy
;--------------------------------

(defvar *parent-children-sort-alist*)
(defvar *child-parent-sort-alist*)

; global hash table used for maintaining the sort hierarchy
(defvar *sort-table* (make-hash-table :test #'equal))

; assume that sort-hierarchy is a tree

(defun defhierarchy (sort-hierarchy)
  (setf *parent-children-sort-alist* sort-hierarchy)
  (setf *child-parent-sort-alist*
	(build-child-parent-sort-alist sort-hierarchy))
  (dolist (predicates *parent-children-sort-alist*)
    (dolist (p predicates)
      (setf (get p 'sort-symbol?) t)))
  (build-sort-table sort-hierarchy))

(defun undefhierarchy ()
  (setf *parent-children-sort-alist* nil)
  (setf *child-parent-sort-alist* nil)
  (dolist (predicates *parent-children-sort-alist*)
    (dolist (p predicates)
      (setf (get p 'sort-symbol?) nil)))
  (clrhash *sort-table*))

(defun build-child-parent-sort-alist (sort-hierarchy &aux alist)
  (dolist (parent-children sort-hierarchy)
    (dolist (child (rest parent-children))
      (push (list child (first parent-children)) alist)))
  (reverse alist))

; assume that the order of sort given in sort hierarchy
; is such that any subsort appears after its supersort

(defun build-sort-table (sort-hierarchy &aux ancestor-descendants-alist)
  (dolist (parent-children-pair (reverse sort-hierarchy))
    (push (cons (first parent-children-pair)
		(apply 'append
		       (mapcar #'(lambda (child)
				   (or (assoc child ancestor-descendants-alist)
				       (list child)))
			       (rest parent-children-pair))))
	  ancestor-descendants-alist))
  (mapc #'(lambda (ancestor-descendants-pair)
	    (mapc #'(lambda (descendant)
		      (declare-subsort descendant
				       (first ancestor-descendants-pair)))
		  (rest ancestor-descendants-pair)))
	ancestor-descendants-alist))


(defun declare-subsort (subsort supersort)
  (setf (gethash (list subsort supersort) *sort-table*) t))

(defun sort? (s)
  (or (eq s 'any)
      (assoc s *child-parent-sort-alist*)))

(defun subsort? (s1 s2)
  (gethash (list s1 s2) *sort-table*))

; return t iff all the sorts are compatible,
; where two sorts s1 and s2 are compatible iff
; s1 <= s2 or s2 <= s1

(defun compatible-sort (sorts)
  (or (null (rest sorts))
      (and (every #'(lambda (s2)
		      (or (eq (first sorts) s2)
			  (subsort? (first sorts) s2)
			  (subsort? s2 (first sorts))))
		  (rest sorts))
	   (compatible-sort (rest sorts)))))

(defun compatible-sort-pair (s1 s2)
  (or (eq s1 s2)
      (subsort? s1 s2)
      (subsort? s2 s1)))

(defun incompatible-sort (x y)
  (not (compatible-sort (list x y))))

(defun primitive-sort? (s)
  (null (assoc s *parent-children-sort-alist*)))

(defun inst-atom? (l)
  (and (consp l)
       (eq (first l) 'inst)))

(defun inst-sort-atom? (l)
  (and (inst-atom? l)
       (sort? (third l))))

(defun parent-sort (s)
  (second (assoc s *child-parent-sort-alist*)))
