;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         struct.l
; Description:  Contains the structures for the frobs.
; Author:       Eric G. Muehle
; Created:      27-Jun-86
; Package:      FROBS
; RCS $Header: struct.l,v 2.5 88/10/08 20:46:47 neff Exp $
;
; (c) Copyright 1986, 1987, University of Utah, all rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)

;;; The generic structure of a class frob
(defstruct (class-frob  (:print-function class-frob-print))
  name 
  vect 
  vect-size
  children 
  parent 
  class-children 
  open
  set-open
  generic
  set-generic
  private
  set-private
  mv
  set-mv
  methods
  in-mod-methods
  private-methods
  show-methods
  gensym
  prop)

;;; Prints out a CLASS-FROB in a curly bracket format
(defun class-frob-print (str stream depth)
  (declare (ignore depth))
  (format stream "{CLASS ~S}" (class-frob-name str)))

;;; A generic frob
(defstruct (frob (:print-function frob-print))
  parent
  name
  vect)

;;; Prints out a FROB in curly bracket format.  This allows generic
;;; print-self methods to change the printing characteristics of
;;; frobs. 
(defun frob-print (str stream depth)
  (cond ((get-generic-fn* 'print-self str)
         (print-self str stream depth))
	((listp (frob-name str))
	 (format stream "{~S ~S}" (car (frob-name str))(cadr (frob-name str))))
	(t (format stream "{~S}" (frob-name str)))))

;;; The read macro function for the { bracket
(defun bracket (stream char)
  (declare (ignore char))
  (let ((list (read-delimited-list #\}  stream t)))
    (cond ((and (eq (car list) 'class) (cdr list))
	   `(get-class-frob ',(cadr list)))
      ((numberp (second list))
       `(get-frob ',list))
      (t `(get-frob ',(car list))))))


;;; Install the function into the readtable
(set-macro-character #\{ #'bracket)

;;; Redefine the } bracket as returning \space
(defun end (stream char)
  (declare (ignore char)(ignore stream))
  #\space)

;;; Install the function into the readtable
(set-macro-character #\} #'end)

;;; returns a class frob by symbol name
(defun get-class-frob (name)
  (gethash name *class-hash*))

;;;  returns an instance frob by symbol name
(defun get-frob (name)
  (gethash name *hash*))

;;; Returns the name of a class frob
(defun class-name (class)
  (class-frob-name class))

;;; Returns the name of an instance frob
(defun instance-name (frob)
  (frob-name frob))

;;; stores a frob onto the frob hash
(defun put-frob (name frob)
  (cond (frob (setf (gethash name *hash*) frob))
	(t (remhash name *hash*))))

;;; stores a class frob onto the frob hash  
(defun put-class-frob (name frob)
  (cond (frob (setf (gethash name *class-hash*) frob))
	(t (remhash name *class-hash*))))

;; End of file.
