;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         p-frobs.l
; Description:  contains the p-frob routines
; Author:       Eric Muehle
; Created:      27-May-87
; Package:      FROBS
; RCS $Header: p-frobs.l,v 2.2 88/03/30 14:32:05 jed Exp $
;
; (c) Copyright 1987, University of Utah, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)


;;; From p-frobs.l

(export '(check-out-file check-in-file use-pfrobs unuse-pfrobs
	  *user-name* *pfrob-loaddir* file-status save-frobs
	  page-value set-page-value *default-type*))

(def-class pfrob nil
  :slots (class status page))

(def-method ({class pfrob} make)(class type page-value)
  (let ((inst (get-pfrob class)))
    (unless inst 
      (setf inst (new-instance* $self))
      (setf (class inst) class)
      (setf (page inst) page-value))
    (setf (status inst) type)))

;;; Returns T if the class has been checked in
(def-method ({class pfrob} checked-in?)()
  (eq :checked-in (status $self)))

;;; Removes a pfrob out of the system
(def-method ({class pfrob} remove-pfrob)()
  (setf (status $self) :checked-in))


(def-sys 'pfrob)
(def-sys-meth '(class set-class page set-page status set-status
		       make checked-in? remove-pfrob))

;;; Will return the pfrob instance of a given class
(defun get-pfrob (class-name)
  (dolist (p (instances (get-class-frob 'pfrob)))
    (when (eq class-name (class p))
      (return-from get-pfrob p)))
  nil)

;;; Returns the file status of a class, returns :locked, :read-only, 
;;; or :not-pfrob
(defun file-status (class)
  (let ((pfrob (get-pfrob (class-frob-name class))))
    (if pfrob 
      (status pfrob)
      :not-pfrob)))

;;; Returns the page value of a class.  If the class has not been
;;; checked out then NIL is returned.
(defun page-value (class)
  (let ((pfrob (get-pfrob (class-frob-name class))))
    (if pfrob
      (page pfrob)
      nil)))

;;; Sets the page value of a class.  If the class has not been
;;; checked out then nothing is done
(defun set-page-value (class value)
  (let ((pfrob (get-pfrob (class-frob-name class))))
    (if pfrob
      (setf (page pfrob) value)
      nil)))

;;; We know we have a pfrob if the frob vect is null
;;; Redefine the old frob vect to the new:
(defun use-pfrobs (&key pager threshold)
  (unless *pfrobs*
    (setf *paging* pager)
    (when (numberp threshold)
      (setf *threshold* threshold))
    (setf (symbol-function 'old-frob-vect)
	  (symbol-function 'frob-vect))
    (setf (symbol-function 'frob-vect)
	  (symbol-function 'new-frob-vect))
    (setf *pfrobs* t)))
  
;;; Dont want ot use the pfrobs anymore
(defun unuse-pfrobs ()
  (when *pfrobs*
    (setf *paging* nil)
    (setf (symbol-function 'frob-vect)
	  (symbol-function 'old-frob-vect))
    (setf *pfrobs* nil)))

;;; New version of frob-vect
(defun new-frob-vect (frob)
  (cond ((null (old-frob-vect frob)) 
	 (check-out-file (frob-parent frob))
	 (old-frob-vect (get-frob (frob-name frob))))
	(t (old-frob-vect frob))))

;;; Printing function for instance frobs
(defun struct-frob-print (stream frob)
  (format stream "~S ~S~%"
    (frob-name frob)(frob-vect frob)))

;;; Writes a class of frobs to the appropriate file
(defun check-in-file (class-frob &key (save t))
  (unless (class-frob? class-frob)
    (error "~S is not a class frob.~%" class-frob))
  (when (member (class-frob-name class-frob) *sys-class*)
    (error "Cannot save a system class~%"))
  (unless *pfrobs* (use-pfrobs))
  (let ((path (merge-pathnames 
	       *pfrob-loaddir* 
	       (string-downcase (symbol-name (class-frob-name class-frob)))))
	(lock (merge-pathnames 
	       *pfrob-loaddir* 
	       (string-upcase (symbol-name (class-frob-name class-frob)))))
	(pfrob (get-pfrob (class-frob-name class-frob)))
	(len  (- (length (class-frob-vect class-frob))
		 (length (class-frob-generic class-frob))))
	(num-frobs (length (class-frob-children class-frob))))
    (decf *locations* (* len num-frobs))
    ;; if this file has been checked out once before
    (when pfrob
      ;; the file is read-only
      (when (eq :read-only (status pfrob))
	(remove-pfrob pfrob)
	(dolist (instance (instances class-frob))
	  ;; remove the vector frob from the frob
	  (setf (frob-vect instance) nil))
	(return-from check-in-file))
      ;; the file is already checked in
      (when (checked-in? pfrob)
	(return-from check-in-file)))
    ;; We need to make a pfrob for this class
    (make {class pfrob} (class-frob-name class-frob) :checked-in 0)
    ;; we either have a new file or it was a locked file
    (with-open-file (output lock :direction :output :if-exists :supersede)
      (format output ":unlocked ~S~%" *user-name*))
    ;; if the file should not be saved then lets leave
    (unless save (return-from check-in-file))
    (with-open-file (output path :direction :output :if-exists :supersede)
      ;; print out the current gensym number
      (format output "~d~%" (class-frob-gensym class-frob))
      ;; print out the number of instances next
      (format output "~d~%" num-frobs)
      ;; put all of the instance frobs into the file
      (dolist (instance (instances class-frob))
	(struct-frob-print output instance)
	;; remove the vector frob from the frob
	(setf (frob-vect instance) nil)))))


;;; Given a class pfrob that has no class frob; read-pfrob will read in the
;;; appropriate file of pfrobs and relink all of the existing empty pfrobs
;;; to the pfrobs that were read in.
(defun check-out-file (class-frob &key (type *default-type*) (page-value 0) override)
  (unless (class-frob? class-frob)
    (error "~S is not a class frob.~%" class-frob))
  (unless *pfrobs* (use-pfrobs))
  (let ((path (merge-pathnames 
	       *pfrob-loaddir* 
	       (string-downcase (symbol-name (class-frob-name class-frob)))))
	(lock (merge-pathnames 
	       *pfrob-loaddir* 
	       (string-upcase (symbol-name (class-frob-name class-frob)))))
	(pfrob (get-pfrob (class-frob-name class-frob)))
	(len  (- (length (class-frob-vect class-frob))
		 (length (class-frob-generic class-frob))))
	owner status probe frob name vect gensym num-frobs)
    ;; If the file is already checked out we may need to stop
    (when (and pfrob (not override)(not (checked-in? pfrob)))
      (format t "File for class ~S has already been checked out.~%" class-frob)
      (return-from check-out-file))
    (setf probe (probe-file lock))
    (unless probe
      (error "No file for class ~S.~%" class-frob))
    ;; need to get the protection from the lock file
    (with-open-file (input lock :direction :input)
      (setf status (read input))
      (setf owner (read input)))
    (when (and (eq status :locked)
	       (eq type :locked))
      (format t "File for ~S is locked by ~S.~%Checking out read-only copy.~%"
	      class-frob owner)
      (setf type :read-only))
    ;; The file is unlocked and we are checking out a locked copy.  
    ;; Need to update the lock file.
    (when (eq type :locked)
      (with-open-file (output lock :direction :output :if-exists :supersede)
        (format output ":locked ~S~%" *user-name*)))
    (make {class pfrob} (class-frob-name class-frob) type page-value)
    (with-open-file (input path :direction :input)
      ;; read in the gensym number
      (setf gensym (read input))
      ;; we may need to update the gensym number
      (when (> gensym (class-frob-gensym class-frob))
	(setf (class-frob-gensym class-frob) gensym))
      ;; need to update the number of storage locations for the pager
      (setf num-frobs (read input))
      (incf *locations* (* len num-frobs))
      ;; read in the instance frobs
      (dotimes (k num-frobs)
	(setf name (read input))
	(setf vect (read input))
	;; install the vect into the frob
	;; if there is no pfrob then we should make one
	(setf frob (gethash name *hash*))
	;; frob does not exist
	(cond ((not frob)
	       (setf frob (make-frob :name name :parent class-frob))
	       (setf (frob-vect frob) vect)
	       (push frob *frobs*)
	       (push frob (class-frob-children class-frob))
	       (put-frob name frob))
	      ;; generic frob
	      ((listp name)
	       (setf (frob-vect frob) vect))
	      ;; named frob
	      (t
	       (setf frob (new-instance* class-frob :name name :default nil))
	       (setf (frob-vect frob) vect)))))))


;;; Saves the frobs
(defun save-frobs (&key ask check-in)
  (let (pfrob)
    (dolist (class (all-class-frobs))
      (when (setf pfrob (get-pfrob (class-frob-name class)))
	(when (or (not ask)
		  (and ask (yes-or-no-p "Do you want class ~S saved?~%" class)))
	  (when (and check-in (checked-in? pfrob))
	    (check-out-file class))
	  (check-in-file class))))))

;(in-package 'frobs)
;;; The paging function
(defun pager (class)
  (when (and (> *locations* *threshold*)
	     (not (eq class {class pfrob})))
    (let* ((pfrob   (get-pfrob (class-frob-name class)))
	   (worst 
	    (get-worst-class (remove pfrob (instances {class pfrob})))))
      (when worst
	(check-in-file (get-class-frob (class worst)))))))
;(in-package 'user)

;;; Returns the worst class of the pfrobs
(defun get-worst-class (pfrobs)
  (when pfrobs
    (let (min-val)
      (setf min-val (apply #'min (mapcar #'page pfrobs)))
      (car (member min-val pfrobs :key #'page)))))

;; End of file.
