;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         define.l
; Description:  Contains frob definitions macros.
; Author:       Eric G. Muehle
; Created:      18-Dec-86
; Package:      FROBS
; RCS $Header: /u/misc/pass/lisp/tools/frobs/RCS/define.l,v 2.10 1993/05/03 17:51:57 eeide Exp $
;
; (c) Copyright 1986, 1987, University of Utah, all rights reserved.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)

;;; Returns the set-slot method for a slot
(defun get-set-slot (slot)
  (cond ((get slot 'frob-set-method))
	(t (setf (get slot 'frob-set-method)
		 (multiple-value-bind (a b c fn d)
		   (get-setf-method `(,slot ignore))
		   (declare (ignore a b c d))
		   (car fn))))))

;;; Creates the name for a slot setting method
(defun make-set-slot (slot)
  (intern (concatenate 'string "SET-" (symbol-name slot)) (symbol-package slot)))

;;; Macro for make slot that adds slot tuples to a type list if they are
;;; already not present.
(defmacro check-slot-type (access slot slot-type)
  `(dolist (s (,access p))
     (unless (member (second s) ,slot :key #'second)
       (pushnew s ,slot-type :test #'equal))))
  
;;; Macro that creates the tuple format of the form ((class slot) type index)
(defmacro make-tuple-list (get set type get-result set-result index)
  `(let (stemp gtemp (tindex ,index))
     (dolist (s ,get)
       (push (list s ',type ,index) gtemp)
       (incf ,index))
     (dolist (s ,set)
       (push (list s ',type tindex) stemp)
       (incf tindex))
     (setf ,get-result gtemp)
     (setf ,set-result stemp)))

;;; Does checking for mixed methods.  A mixed method is one that is a slot
;;; method as well as a user defined method.
(defmacro check-for-mixed (slot-list methods add-slot add-method)
  `(let (temp)
     (dolist (s ,slot-list)
      (when (setf temp (member (second (car s)) ,methods :key #'second))
	(push s ,add-slot)
	(setf ,slot-list (delete s ,slot-list :test #'equal))
	(dolist (y temp)
	  (when (eq (second (car s)) (second y))
	    (setf ,add-method (adjoin y ,add-method :test #'equal))))))))

;;; Does checking for context methods.  A context method is one
;;; that has the same name but is inherited from more than one
;;; class.  Each class provides a different version (context) 
;;; for the method.
(defmacro remove-and-add (check fn remove add)
  `(progn
    (dolist (s ,check)
      (when 
	(member (funcall ,fn s)
		(cdr (member (funcall ,fn s) ,check :key ,fn))
		:key ,fn)
	(push s ,add)))
    (dolist (s ,add)
      (setf ,remove (remove s ,remove :test #'equal)))))

;;; Inherits all of the slots and methods.  If there is a problem
;;; with this function then it is a "feature".  Returns the following 
;;; information: get-slots, set-slots, mixed-get, mixed-set,
;;; context-get, context-set, meth-context, and inherited-methods.
(defun make-slots (open generic private mv frob name parents)
  (let (set-open set-generic set-private set-mv mixed-get mixed-set 
	context-get context-set inherit-methods (index 0) slots set-slots)
    (let (context-methods both tmixed)
    ;; name all of the slots in the form (class slot-name)
    (setf set-open (mapcar #'(lambda (y) (list name (make-set-slot y))) open))
    (setf open (mapcar #'(lambda (y) (list name y)) open))
    (setf set-generic (mapcar #'(lambda (y) (list name (make-set-slot y))) generic))
    (setf generic (mapcar #'(lambda (y) (list name y)) generic))
    (setf set-private (mapcar #'(lambda (y) (list name (make-set-slot y))) private))
    (setf private (mapcar #'(lambda (y) (list name y)) private))
    (setf set-mv (mapcar #'(lambda (y) (list name (make-set-slot y))) mv))
    (setf mv (mapcar #'(lambda (y) (list name y)) mv))    
    (setf slots (append open private mv generic))
    (setf set-slots (append set-open set-private set-mv set-generic))
    (setf both (append slots set-slots))
    (dolist (p parents)
      (check-slot-type class-frob-open slots open)
      (check-slot-type class-frob-generic slots generic)
      (check-slot-type class-frob-mv slots mv)
      (check-slot-type class-frob-private slots private)
      (check-slot-type class-frob-set-open set-slots set-open)
      (check-slot-type class-frob-set-generic set-slots set-generic)
      (check-slot-type class-frob-set-mv set-slots set-mv)
      (check-slot-type class-frob-set-private set-slots set-private)
      (check-slot-type class-frob-methods both inherit-methods))
    ;; add the open.private,mv,generic slots to the frob here
    (setf (class-frob-open frob) open)
    (setf (class-frob-set-open frob) set-open)
    (setf (class-frob-generic frob) generic)
    (setf (class-frob-set-generic frob) set-generic)    
    (setf (class-frob-private frob) private)
    (setf (class-frob-set-private frob) set-private)
    (setf (class-frob-mv frob) mv)
    (setf (class-frob-set-mv frob) set-mv)
    (setf (class-frob-methods frob) inherit-methods)
    (setf (class-frob-in-mod-methods frob)
	  (apply #'append 
		 (mapcar #'class-frob-in-mod-methods parents)))
    (setf (class-frob-show-methods frob)
	  (apply #'append 
		 (mapcar #'class-frob-show-methods parents)))
    (setf (class-frob-private-methods frob)
	  (apply #'append 
		 (mapcar #'class-frob-private-methods parents)))
    (make-slots-aux open generic private mv frob name parents
		    set-open set-generic set-private set-mv mixed-get
		    mixed-set context-get context-set inherit-methods
		    index slots set-slots context-methods both tmixed))))

(defun make-slots-aux (open generic private mv frob name parents
			    set-open set-generic set-private set-mv mixed-get
			    mixed-set context-get context-set
			    inherit-methods index slots set-slots
			    context-methods both tmixed)
    ;; make the tuple lists
    (make-tuple-list open set-open open open set-open index)
    (make-tuple-list mv set-mv mv mv set-mv index)
    (make-tuple-list private set-private private private set-private index)
    (make-tuple-list generic set-generic generic generic set-generic index)
    ;;; need to create the mixed slots
    (check-for-mixed open inherit-methods mixed-get tmixed)
    (check-for-mixed private inherit-methods mixed-get tmixed)
    (check-for-mixed generic inherit-methods mixed-get tmixed)
    (check-for-mixed mv inherit-methods mixed-get tmixed)
    (make-slots-aux-aux open generic private mv frob name parents
			    set-open set-generic set-private set-mv mixed-get
			    mixed-set context-get context-set
			    inherit-methods index slots set-slots
			    context-methods both tmixed))

(defun make-slots-aux-aux (open generic private mv frob name parents
			    set-open set-generic set-private set-mv mixed-get
			    mixed-set context-get context-set
			    inherit-methods index slots set-slots
			    context-methods both tmixed)
    (dolist (s tmixed)
      (setf inherit-methods (remove s inherit-methods :test #'equal))
      (push (list s 'meth 'meth) mixed-get))
    (setf tmixed nil)
    (check-for-mixed set-open inherit-methods mixed-set tmixed)
    (check-for-mixed set-private inherit-methods mixed-set tmixed)
    (check-for-mixed set-generic inherit-methods mixed-set tmixed)
    (check-for-mixed set-mv inherit-methods mixed-set tmixed)
    (dolist (s tmixed)
      (setf inherit-methods (remove s inherit-methods :test #'equal))
      (push (list s 'meth 'meth) mixed-set))
    ;; now that we have the mixed methods lets get the context methods
    (remove-and-add inherit-methods #'second inherit-methods context-methods)
    ;; ok now we have the following: mixed-get,mixed-set,
    ;; context-methods, and inherit-methods
    ;; lets get the context-get and the context-set slot methods
    (setf slots (append open private mv generic))
    (setf set-slots (append set-open set-private set-mv set-generic))
    (remove-and-add slots #'(lambda (y) (second (car y))) slots context-get)
    (remove-and-add set-slots #'(lambda (y) (second (car y))) set-slots context-set)
    (values slots set-slots context-get context-set context-methods 
	  inherit-methods mixed-get mixed-set))

;;; Returns the error for undefined methods
(defun undefined-method-error (name class)
  (error "Undefined method ~S for class ~S" name class))


;;; Returns the correct AKO of a frob
(defmacro get-type (frob)
  `(cond ((frob-p ,frob)(class-frob-name (frob-parent ,frob)))
	 ((class-frob-p ,frob) (class-frob-name ,frob))
	 (t (error "~S is not a frob" ,frob))))


;;; Finds the name of a private slot given an index.  This is done
;;; for the undefined-method-error function.
(defun find-private-slot-name (loc frob set?)
  (let ((prop (class-frob-prop (get-class-frob (get-type frob)))))
    (do ((index (cdr prop) (cddr index))
	 (name prop (cddr name)))
	((null index))
	(cond ((symbolp (cdar index)))
	      ((and set? (= (cdar index) loc))
	       (setf set? nil))
	      ((= (cdar index) loc)
	       (return-from find-private-slot-name (car name)))))))

;;; returns the vector of a frob
(defmacro get-vect (frob)
  `(cond ((frob-p ,frob) (frob-vect ,frob))
         ((class-frob-p ,frob) (class-frob-vect ,frob))
	 (t (error "~S is not a frob" ,frob))))


;;;; The system get-slot methods

(defun get-open (frob index)
  (svref (get-vect frob) index))

(defun get-mv (frob index)
  (case *mv*
    (test 'frob-mv-slot)
    (otherwise (svref (get-vect frob) index))))

(defun get-private (frob index)
  (cond ((eq *module* (get-type frob))
	 (svref (get-vect frob) index))
	(t (undefined-method-error (find-private-slot-name index frob nil)
				   (get-type frob)))))

(defun get-generic (frob index)
  (svref (get-vect (get-class-frob (get-type frob))) index))

;;;; The system set-slot methods

(defun set-open (frob index value)
  (setf (svref (get-vect frob) index) value))

(defun set-mv (frob index value)
  ;; the variable *mv* controls the behavior of a MV slot
  (case *mv*
    (normal
     ;; we only keep unique values in a MV slot 
     (pushnew value (svref (get-vect frob) index) :test #'equal))
    (erase
     (setf (svref (get-vect frob) index) nil))
    (set
     (setf (svref (get-vect frob) index) value))))

(defun set-private (frob index value)
  (cond ((eq *module* (get-type frob))
	 (setf (svref (get-vect frob) index) value))
	(t (undefined-method-error (find-private-slot-name index frob t)
				   (get-type frob)))))

(defun set-generic (frob index value)
  (cond ((class-frob-p frob)
	 (setf (svref (get-vect frob) index) value))
	(t (warn "Illegal to set GENERIC slot from an Instance.~%"))))

;;; Installs a generic function and (its index if a slot method or the symbol
;;; METHOD if it is a user defined method) in the class frob.
;;; Currently methods are kept in a property list in the class frob.
(defun install-method (name class fn index main-methods params)
  (when (symbolp fn)
    (setf fn (symbol-function fn)))
  (unless (eq 'unoptimized (gethash name *main-methods*))
    (setf (gethash name *main-methods*) main-methods))
  (when params (setf (gethash name *params*) params))
  ;; put it in the class frob
  (setf (getf (class-frob-prop (get-class-frob class)) name)(cons fn index)))

;;; Returns the generic function out of the class frob.
(defmacro get-generic-fn (name frob)
  `(getf (cond ((frob-p ,frob)
		(class-frob-prop (frob-parent ,frob)))
	       ((class-frob-p ,frob)
		(class-frob-prop ,frob))
	       (t (error "~S is not a frob" ,frob)))
	 ',name))

;;; Returns the generic function out of the class frob.
(defun get-generic-fn* (name frob)
  (getf (cond ((frob-p frob)
		(class-frob-prop (frob-parent frob)))
	       ((class-frob-p frob)
		(class-frob-prop frob))
	       (t (error "~S is not a frob" frob)))
	name))

;;; Determines if 2 parameter lists are equal
(defun params-ok (p1 p2)
  (cond ((and (null p1)(null p2)))
	((null p1) nil)
	((null p2) nil)
	((member (car p1) lambda-list-keywords) nil)
	((member (car p2) lambda-list-keywords) nil)
	(t (params-ok (cdr p1)(cdr p2)))))

;;; Fixes up a parameter list, removing any &descriptors, for splicing
;;; in the funcalls for optimized method calls.
(defun fix-params (params)
  (cond ((null params) nil)
	((member (car params) lambda-list-keywords)
	 (fix-params (cdr params)))
	(t (cons (car params) (fix-params (cdr params))))))


;;; Finds a version of this method elsewhere in this tree
(defun get-prev-method (meth-name)
  (dolist (p *new-parent*)
    (when (get-generic-fn* meth-name (get-class-frob p))
      (return t))))

;;; Creates a main method if needed and installs the correct slot method
;;; or inherited method on the hash table.  This is the only place that 
;;; method code is created.  A method comes in one of two versions: optimized
;;; and unoptimized.  
(defun make-one-method (name class aux-name index new-param 
			     &key (inherit t))
  (let ((old-param   (gethash name *params*)))
    ;; case 1 existing method but parameter list are the same
    (cond ((or inherit
	       (and (params-ok new-param old-param)
		    (or
		     ;; if it's here then the main method exists in 
		     ;; the current file being compiled
		     (member name *comp-meths-list*)
		     ;; the function may have been defined elsewhere
		     ;; in this inheritance lattice
		     (get-prev-method name))))
	   `(install-method ',name ',class ',aux-name ',index 'optimized nil))
	  ;; case 2 no previous existing method - create optimized
	  ;; dispatch fn with load time check for existing unoptimized 
	  ;; dispatch fns.
	  ((and (params-ok new-param new-param) ; no & keywords here
		(or (not old-param)
		    (params-ok new-param old-param)))
	   (when *comp-mode*
	     (push name *comp-meths-list*))
	   `(progn
	     (defun ,(intern (format nil "generic ~S" name)) ,new-param
	       (let ((fn (get-generic-fn ,name $self)))
		 (cond ((integerp (cdr fn))
			,(case (length new-param)
			   (1 `(funcall (car fn) $self (cdr fn)))
			   (2 `(funcall (car fn) $self (cdr fn) 
					,(second new-param)))
			   (otherwise nil)))
		       (fn (funcall (car fn) ,@(fix-params new-param)))
		       (t (undefined-method-error ',name (frob-type $self))))))
	     (unless (eq 'unoptimized (gethash ',name *main-methods*))
	       (setf (symbol-function ',name)
		     (symbol-function 
		      ',(intern (format nil "generic ~S" name)))))
	     (install-method ',name ',class ',aux-name ',index 'optimized 
			      ',new-param)))
	  ;; case 3 existing previous method but parameter lists are 
	  ;; not the same or a method with lambda-list-keywords in it.
	  ;; Need to create an unoptimized dispatch fn.
	  (t
	   (setf (gethash name *main-methods*) 'unoptimized)
	   `(progn
	     (defun ,name ($self &rest args)
	       (let ((fn (get-generic-fn ,name $self)))
		 (cond ((integerp (cdr fn))
			(if args
			    (funcall (car fn) $self (cdr fn) (car args))
			    (funcall (car fn) $self (cdr fn))))   
		       (fn
			(apply (car fn) $self args))
		       (t (undefined-method-error ',name (frob-type $self))))))
	     (install-method ',name ',class ',aux-name ',index 'unoptimized 
			     '($self &rest args)))))))

;;; returns the method name from a 3 tuple
(defun method-name (x) (second (slot-tuple x)))

;;; returns the method name from a 2 tuple
(defun method-name2 (x) (second x))

;;; returns the method context from a 3 tuple
(defun method-context (x) (car (slot-tuple x)))

;;; returns the method context from a 2 tuple
(defun method-context2 (x) (car x))

;;; other access functions
(defun slot-tuple (x) (first x))
(defun slot-type (x) (second x))
(defun slot-index (x) (third x))

;;; Returns the correct slot method for the gettable methods.
(defun  get-sys-slot-method (tuple)
  (case (slot-type tuple)
    (open     'get-open)
    (mv       'get-mv)
    (private  'get-private)
    (generic  'get-generic)))

;;; Returns the correct slot method for the settable methods.
(defun  set-sys-slot-method (tuple)
  (case (slot-type tuple)
    (open    'set-open)
    (mv      'set-mv)
    (private 'set-private)
    (generic 'set-generic)))

;;; creates the getting methods from the slots list
(defun make-get (slots class)
  (mapcar 
   #'(lambda (x) 
       (make-one-method (method-name x)
			class
			(get-sys-slot-method x)
			(slot-index x)
			'($self)
			:inherit (not (eq (method-context x) class))))
   slots))


;;; creates the setting methods from the slots list
(defun make-set (slots class)
  (mapcar 
   #'(lambda (x) 
       (make-one-method (method-name x)
			class
			(set-sys-slot-method x)
			(slot-index x)
			'($self val)
			:inherit (not (eq (method-context x) class))))
   slots))


;;; fixes up the context slots list so they are in the form:
;;; ((list of all the same slot names)(....)(....))
(defun fix-slots (slots result)
  (cond ((null slots) result)
        (t
	 (let ((temp (create-and-extract (method-name (car slots)) slots nil)))
	   (dolist (s temp)
	     (setf slots (delete s slots :test #'equal)))
	   (fix-slots slots (cons temp result))))))

;;; does the slot list extraction for fix slots
(defun create-and-extract (slot slots result)
  (cond ((null slots) result)
        ((eq (method-name (car slots)) slot) 
	 (create-and-extract slot (cdr slots) (cons (car slots) result)))
	(t (create-and-extract slot (cdr slots) result))))

;;; Finds the default method given a slot-list and a list of parents
(defun get-default (slot-list parents fn)
  (let (temp)
    (cond ((null parents) (error "problem in get-default"))
	  ((setf 
	    temp 
	    (car (member (class-frob-name (car parents)) slot-list :key fn)))
	   temp)
	  (t (get-default
	      slot-list 
	      (append (class-frob-parent (car parents))(cdr parents))
	      fn)))))

;;; creates the aux method name from a tuple
(defun aux-method-name (tuple class)
  (intern (format nil "~S$$~S" class (method-name tuple)) (symbol-package class)))

;;; creates the aux method name for user methods
(defun user-aux-method-name (class tuple)
  (intern (format nil "~S$$~S" class (method-name2 tuple)) (symbol-package class)))

;;; creates the context get methods
(defun make-context-get (slots class parents)
  (setf slots (fix-slots slots nil))
  (mapcar
   #'(lambda (slot-list)
       (let ((default (get-default slot-list parents #'method-context)))
	 `(progn
	    (defun ,(aux-method-name (car slot-list) class) (frob)
	      (cond
	       ,@(mapcar 
		  #'(lambda (x)
		      `((eq *context* ',(method-context x))
			(,(get-sys-slot-method x) frob ,(slot-index x))))
		  slot-list)
	       (t (,(get-sys-slot-method default) frob ,(slot-index default)))))
	    ,(make-one-method (method-name (car slot-list))
			      class
			      (aux-method-name (car slot-list) class)
			      'context-slot
			      nil))))
   slots))

;;; creates the context set methods
(defun make-context-set (slots class parents)
  (setf slots (fix-slots slots nil))
  (mapcar
   #'(lambda (slot-list)
       (let ((default (get-default slot-list parents #'method-context)))
	 `(progn
	    (defun ,(aux-method-name (car slot-list) class) (frob value)
	      (cond
	       ,@(mapcar 
		  #'(lambda (x)
		      `((eq *context* ',(method-context x))
			(,(set-sys-slot-method x) frob ,(slot-index x) value)))
		  slot-list)
	       (t (,(set-sys-slot-method default) frob ,(slot-index default) value))))
	    ,(make-one-method (method-name (car slot-list))
			      class
			      (aux-method-name (car slot-list) class)
			      'context-slot
			      nil))))
   slots))

;;; inherits the user defined methods.  The meth-list is of the form:
;;; ((class meth-name)(class meth-name) ...)
(defun make-methods (meth-list class)
  (mapcar 
   #'(lambda (x) 
       (let ((aux-fn (user-aux-method-name class x))
	     (par-fn (user-aux-method-name (method-context2 x) x)))
	 `(progn
	    (setf (symbol-function ',aux-fn) (symbol-function ',par-fn))
	    (install-method ',(method-name2 x) ',class ',aux-fn 'method
			    'optimized nil))))
   meth-list))

;;; fixes up the context slots list so they are in the form:
;;; ((list of all the same slot names)(....)(....))
(defun fix-method-slots (slots result)
  (cond ((null slots) result)
        (t
	 (let ((temp (create-and-extract-method 
		      (method-name2 (car slots)) slots nil)))
	   (dolist (s temp)
	     (setf slots (delete s slots :test #'equal)))
	   (fix-method-slots slots (cons temp result))))))

;;; does the slot list extraction for fix slots
(defun create-and-extract-method (slot slots result)
  (cond ((null slots) result)
        ((eq (second (car slots)) slot) 
	 (create-and-extract-method slot (cdr slots) (cons (car slots) result)))
	(t (create-and-extract-method slot (cdr slots) result))))

;;; Creates the code for one context method
(defun build-context-method (x)
  (let ((par-fn (user-aux-method-name (method-context2 x) x)))
    `((eq *context* ',(car x))
      (apply #',par-fn frob args))))


;;; Inherits the user defined context methods.  The meth-list is of the form:
;;; ((class meth-name)(class meth-name) ...)
(defun make-context-methods (meth-list class parents)
  (setf meth-list (fix-method-slots meth-list nil))
  (mapcar
   #'(lambda (slot-list)
       (let ((default (get-default slot-list parents #'method-context2)))
	 `(progn
	    (defun ,(user-aux-method-name class (car slot-list))
	      (frob &rest args)
	      (cond
	       ,@(mapcar 
		  #'(lambda (x) (build-context-method x))
		  slot-list)
	       (t ,(second (build-context-method default)))))
	    ,(make-one-method (method-name2 (car slot-list))
			      class
			      (user-aux-method-name class (car slot-list))
			      'context-method
			      nil))))
   meth-list))

;;; Creates the code for one mixed get method
(defun build-mixed-get (x)
  (let ((par-fn (user-aux-method-name (method-context x) x)))
    (cond ((eq 'meth (slot-type x))
	   `((eq *context* ',(method-context x))
	     (apply #',par-fn frob args)))
	  (t
	   `((eq *context* ',(method-context x))
	     (,(get-sys-slot-method x) frob ,(slot-index x)))))))

;;; Creates the mixed get methods
(defun make-mixed-get (mixed-list class parents)
  (setf mixed-list (fix-slots mixed-list nil))
  (mapcar
   #'(lambda (slot-list)
       (let ((default (get-default slot-list parents #'method-context)))
	 `(progn
	    (defun ,(aux-method-name (car slot-list) class) (frob &rest args)
	      (cond
	       ,@(mapcar 
		  #'(lambda (x) (build-mixed-get x))
		  slot-list)
	       (t ,(second (build-mixed-get default)))))
	    ,(make-one-method (method-name (car slot-list))
			      class
			      (aux-method-name (car slot-list) class)
			      'mixed-slot
			      nil))))
   mixed-list))


;;; Creates the code for one mixed set method
(defun build-mixed-set (x)
  (let ((par-fn (user-aux-method-name (method-context x) x)))
    (cond ((eq 'meth (slot-type x))
	   `((eq *context* ',(method-context x))
	     (apply #',par-fn frob args)))
	  (t
	   `((eq *context* ',(method-context x))
	     (,(set-sys-slot-method x) frob ,(slot-index x) (car args)))))))

;;; Creates the mixed set methods
(defun make-mixed-set (mixed-list class parents)
  (setf mixed-list (fix-slots mixed-list nil))
  (mapcar
   #'(lambda (slot-list)
       (let ((default (get-default slot-list parents #'method-context)))
	 `(progn
	    (defun ,(aux-method-name (car slot-list) class) (frob &rest args)
	      (cond
	       ,@(mapcar 
		  #'(lambda (x) (build-mixed-set x))
		  slot-list)
	       (t ,(second (build-mixed-set default)))))
	    ,(make-one-method (method-name (car slot-list))
			      class
			      (aux-method-name (car slot-list) class)
			      'mixed-slot
			      nil))))
   mixed-list))


;;; creates the class frob vector and installs it in the frob
(defun install-vect (frob)
;  (setf (class-frob-vect-size frob)
;	(+ (max  (length (class-frob-open frob))
;		 (length (class-frob-set-open frob)))
;	   (max  (length (class-frob-mv frob))
;		 (length (class-frob-set-mv frob)))
;	   (max  (length (class-frob-private frob))
;		 (length (class-frob-set-private frob)))))
  (setf (class-frob-vect frob)
	(make-array
	 (+ (class-frob-vect-size frob)
	    (max (length (class-frob-generic frob))
		 (length (class-frob-set-generic frob))))
	 :initial-element *undefined*)))


;;; Creates the defsetf forms for all of the slot methods
(defun make-defsetf (&rest slots)
  (let (result name)
    (dolist (l slots)
      (dolist (s l)
	(setf name (make-set-slot s))
	(push `(setf (get ',s 'frob-set-slot) ',name) result)
	(push `(defsetf ,s ,name) result)))
    result))

;;; Returns the class frob name from a list like (get-class-frob (quote name))
;;; and get the frob off of the *class-hash* table
(defun get-class-name (f-list)
  (let ((class (get-class-frob (second (second f-list)))))
    (unless class 
      (error "Class ~S does not exist.~%" (second (second f-list))))
    class))

;;; Turn the parent frobs into a list and evaluate them.  They will
;;; either be symbols or lists of the form (get-class-frob (quote name))
(defun fix-parents (parent)
  (cond ((null parent) nil)
	((symbolp parent)
	 (let ((class (list (get-class-frob parent))))
	   (unless (car class)
	      (error "Class ~S does not exist.~%" parent))
	   class))
	((eq 'get-class-frob (car parent))
	 (list (get-class-name parent)))
	(t
	 (mapcar #'(lambda (x)
		     (if (listp x)
		       (get-class-name x)
		       (get-class-frob x)))
		 parent))))


;;; Strips out the slot names from the list of slot init pairs
(defun slot-names-only (slot)
  (let (result)
    (dolist (s slot)
      (if (listp s)
	(push (car s) result)
	(push s result)))
    result))


;;; Returns a list of only those slots that have init values
(defun get-evaled-init-pairs (&rest slots)
  (let (result)
    (dolist (l slots)
      (dolist (s l)
	(when (listp s)
	  (push s result))))
    result))

;;; returns the vect-size for instance frobs
(defun get-vect-size (frob)
  (+ (length (class-frob-open frob))
     (length (class-frob-mv frob))
     (length (class-frob-private frob))))

;;; Creates a class frob
(defmacro def-class (name parent &key slots private generic init 
			  mv (close-class t))
  (let (frob vect-size eval-init get-slots set-slots
	     context-get context-set context-methods 
	     inherit-methods mixed-get mixed-set)
    ;; get the eval-init list
    (setf eval-init (get-evaled-init-pairs slots private generic mv))
    (setf slots (slot-names-only slots))
    (setf private (slot-names-only private))
    (setf generic (slot-names-only generic))
    (setf mv (slot-names-only mv))
    ;; get the parents list evaluated
    (setf parent (fix-parents parent))
    ;; lets make the frob
    (setf frob (make-class-frob :name name :parent parent :gensym 0))
    ;; remove any children frobs from the system
    ;; put the frob on the hash table.  I hate doing this!
    ;; need to update the parents on the compile hash table for
    ;; retroactive inheritance
    ;; lets inherit those methods
    (multiple-value-setq 
      (get-slots set-slots context-get context-set context-methods 
	     inherit-methods mixed-get mixed-set)
     (make-slots slots generic private mv frob name parent))
      ;; lets make those methods
    (setf vect-size (get-vect-size frob))
    `(eval-when (load compile eval)
       ;; I know using set is gross but it suppresses the compiler
       ;; from printing out *new-frob* every time a new class is 
       ;; compiled
       (set '*new-parent* ',(mapcar #'class-frob-name parent))
       (set '*new-frob*
	     (make-class-frob :name ',name 
		:gensym 0
		:open ',(class-frob-open frob)
		:set-open ',(class-frob-set-open frob)
		:generic ',(class-frob-generic frob)
		:set-generic ',(class-frob-set-generic frob)
		:private ',(class-frob-private frob)
		:set-private ',(class-frob-set-private frob)
		:mv ',(class-frob-mv frob)
		:set-mv ',(class-frob-set-mv frob)
		:vect-size ,vect-size
		:methods ',(class-frob-methods frob)
		:in-mod-methods ',(class-frob-in-mod-methods frob)
		:private-methods ',(class-frob-private-methods frob)
		:show-methods ',(class-frob-show-methods frob)))
       (before-methods ',name *new-frob* *new-parent*)
       ,@(make-context-get context-get name parent)
       ,@(make-context-set context-set name parent)
       ,@(make-context-methods context-methods name parent)
       ,@(make-mixed-get mixed-get name parent)
       ,@(make-mixed-set mixed-set name parent)
       ,@(make-get get-slots name)
       ,@(make-set set-slots name)
       ,@(make-methods inherit-methods name)
       ,@(make-defsetf slots generic mv private)
       (after-methods ',name *new-parent* ',init ',eval-init)
       (when ',close-class 
	 (close-class-aux ',name nil))
       ; account for PCLS init code brain damage
#+PCLS ',*new-frob*
#-PCLS *new-frob*)))

;;; Do some misc things before the methods are defined in the def-class
;;; expansion.
(defun before-methods (name class-frob parent)
  (setf parent (mapcar #'get-class-frob parent))
  ;; reset the module key
  (remhash name *key*)
  ;; remove any children frobs from the system
  (when (get-class-frob name)
    (remove-class-children (get-class-frob name)))
  ;; put the frob on the real hash table
  (put-class-frob name class-frob)
  ;; add the parents to the frob
  (setf (class-frob-parent class-frob) parent)
  ;; create the parent link
  (when parent
    (update-parents parent name)))

;;; Do some misc things after the methods are defined in the def-class
;;; expansion.
(defun after-methods (name parent init eval-init)
  (setf parent (mapcar #'get-class-frob parent))
  (install-vect (get-class-frob name))
  ;; need to look for daemons
  (when (gethash name *daemons*)
    (remhash name *daemons*))
  (when (and (member 'daemon 
		     (class-frob-generic (get-class-frob name)) :key #'car)
	     (not (eq 'daemon name)))
    (setf (gethash name *daemons*) t))
  (init-vect (get-class-frob name) parent init eval-init)
  (pushnew name *class-frobs*)
  (frob-message "~&Defining Class ~S~%" name)
  (get-class-frob name))
  

(defmacro init-from-parent (fn parent frob)
  `(let ((name        (class-frob-name ,frob))
	 (parent-name (class-frob-name ,parent))
	 val)
     (dolist (s (,fn ,parent))
       ;; set the context to the true context owner of the slot
       (setf *context* (car s))
       ;; when we get out the old value we should be in the module
       (setf *module*  parent-name)
       (setf val (funcall (second s) ,parent))
       ;; we need to be in the current def-class module
       (setf *module*  name)
       (funcall (make-set-slot (second s)) ,frob val))
     (setf *context* nil)))

;;; Returns T if the given slot is a MV slot and the default slot.
(defun mv-default (slot parent)
  (cond ((null parent) nil)
	((member slot (class-frob-mv parent) :key #'second))
	(t (mv-default slot (car (class-frob-parent parent))))))

;;; Runs the init time daemons
(defun run-init-time-daemons (parents class-frob)
  (declare (ignore parents)(ignore class-frob)))

;;; Initializes the slot vector using inheritance and intial values from
;;; the init list.
(defun init-vect (frob parent init eval-init)
  ;; there are 2 cases for initialization:
  ;; 1: there are no parents, then we need to init the vector and
  ;;    install the init values
  ;; 2: if there are parents then we need to inherit any default values
  ;;    and then install the init values
  (let ((vect   (class-frob-vect frob))
	(offset (length (class-frob-open frob))))
    (cond (parent ; case 2
	   ;; init all of the mv locations 
 	   (with-mv-slot set
             ;; first lets make the new mv slots to NIL
             (dolist (slot (class-frob-mv frob))
	       (funcall (make-set-slot (second slot)) frob nil))
	     ;; reverse the parents so the first parent is done last.
	     (dolist (p (reverse parent))
	       (init-from-parent class-frob-open p frob)
	       (init-from-parent class-frob-mv p frob)
	       (init-from-parent class-frob-generic p frob)
	       (init-from-parent class-frob-private p frob)))
	   (run-init-time-daemons parent frob))
	  (t ; case 1
	   ;; init all of the mv locations to NIL
	   (dotimes (k (length (class-frob-mv frob)))
	     (setf (svref vect (+ k offset)) nil))))
    ;; now install the init values
    (setf *module* (class-frob-name frob))
    (dolist (l init)
      (cond ((car l)
	     (setf *context* (car l)))
	    (t (setf *context* (class-frob-name frob))))
      (dolist (tuple (cdr l))
	(if (mv-slot? frob (car tuple))
	  (assert-vals frob (car tuple) (eval (second tuple)))
	  (assert-val frob (car tuple) (eval (second tuple))))))
    (setf *context* nil)
    (dolist (tuple eval-init)
	(if (mv-slot? frob (car tuple))
	  (assert-vals frob (car tuple) (eval (second tuple)))
	  (assert-val frob (car tuple) (eval (second tuple)))))))


;;; Creates an instance frob.
(defun new-instance* (parent &key name init (default t))
  (unless name (setf name (create-gen-name parent)))
  (let ((frob    (make-frob :name name :parent parent))
	(len 	 (class-frob-vect-size parent))
	(old-con *context*)
	(old-mod *module*))
    (setf (frob-vect frob)
	  (make-array len :initial-element *undefined*))
    ;; Need to take care of some pfrob stuff
;    (when *pfrobs*
;      (incf *locations* len)
;      ;; we may need to page out some frobs.
;      (when *paging*
;	(pager parent)))
    ;; We may need to remove it off of some old lists
    (let ((old-frob (gethash name *hash*)))
      (when old-frob
	;; need to decrement.  Assume no change in vect size.
;       (when *pfrobs* (decf *locations* len))
;;      (setf *frobs* 
;;	      (delete name *frobs* :count 1 :key #'frob-name :test #'equal))
	(remhash name (class-frob-children (frob-parent old-frob)))
	))
    ;; take care of putting the frob on the right lists
;;  (push frob *frobs*)
    (unless (class-frob-children parent)
      (setf (class-frob-children parent) (make-hash-table :size 64
							  :rehash-size 2.0
							  :rehash-threshold 0.8
							  :test #'equal)))
    (setf (gethash name (class-frob-children parent)) frob)
    (put-frob name frob)
    ;; lets put our self into the parent module so that we can store 
    ;; private values
    (setf *module* (class-frob-name parent))
    ;; init the child vector
    (cond (default
	    (dotimes (k (length (frob-vect frob)))
	      (setf (svref (frob-vect frob) k)
		    (svref (class-frob-vect parent) k))))
	  ;; if default is nil then we need to give all of the mv slots nil.
	  (t
	   (with-mv-slot erase
	     (dolist (sl (class-frob-set-mv parent))
	       (setf *context* (car sl))
	       (funcall (second sl) frob nil)))))
    ;; now install the init values
    (dolist (l init)
      (cond ((car l)
	     (setf *context* (car l)))
	    (t (setf *context* (class-frob-name parent))))
      (dolist (tuple (cdr l))
	(if (mv-slot? frob (car tuple))
	  (assert-vals frob (car tuple) (eval (second tuple)))
	  (assert-val frob (car tuple) (eval (second tuple))))))
    (setf *module* old-mod)
    (setf *context* old-con)
    frob))

;;; Creates an instance frob.
(defmacro new-instance (parent &key name init (default t))
  `(eval-when (compile load eval)
     (new-instance* ,parent :name ',name :init ',init :default ',default)))

;;; Creates a generic name
(defun create-gen-name (class)
  (do ((name (list (class-frob-name class) (class-frob-gensym class))
	     (list (class-frob-name class) (class-frob-gensym class))))
      ((not (get-frob name)) name)
    (incf (class-frob-gensym class))))

;;; kills off all of the children of a class
(defun remove-children (class)
  (when (> (class-frob-gensym class) 0)
    (frob-message "~&Removing up to ~S generic instances of ~S~%" 
		   (class-frob-gensym class) class))
  (when (class-frob-children class)
    (maphash #'(lambda (key child)
		 (when (symbolp (frob-name child))
		   (frob-message "~&Removing ~S~%" child))
;;		 (setf *frobs* 
;;		       (delete (frob-name child)
;;			       *frobs*
;;			       :count 1 :key #'frob-name :test #'equal))
;		 (when *pfrobs*
;		   (decf *locations* (length (frob-vect child))))
		 ;; remove the frob from the *hash* hash table
		 (put-frob (frob-name child) nil))
	     (class-frob-children class)))
  (setf (class-frob-children class) nil))

;;; kills off all of the class children of a class
(defun remove-class-children (class)
  (remove-children class)
  (dolist (child (class-frob-class-children class))
    (when (get-class-frob child)
      (remove-children (get-class-frob child))
      (frob-message "~&Removing ~S~%" (get-class-frob child))
      (setf *class-frobs* (delete child *class-frobs* :count 1))
      (put-class-frob child nil))))
  
;;; updates the parent links for the class frob
(defun update-parents (parents name)
  (dolist (p parents)
    (pushnew name (class-frob-class-children p))))

;; End of file.
