;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; File:         ask-assert.l
; Description:  The ask and assert functions for frobs.
; Author:       Eric G. Muehle
; Created:      10-Jul-86
; Package:      FROBS
; RCS $Header: /u/misc/pass/lisp/tools/frobs/RCS/ask-assert.l,v 2.6 1993/05/03 17:49:51 eeide Exp $
;
; (c) Copyright 1986, 1987, University of Utah, all rights reserved.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)

;;; internal fn that returns T if the slot is an MV slot
(defun mv-slot? (frob slot)
  (eq 'frob-mv-slot (with-mv-slot test (funcall slot frob))))

;;; Returns T if the given method is a slot defined over the class.
(defun slot? (class slot)
  (let ((stuff (cdr (get-generic-fn* slot class))))
    (or (integerp stuff)
	(eq 'mixed-slot stuff)
	(eq 'context-slot stuff))))

;;; Predicate returns T if a slot is bound.  NIL is the unbound value for
;;; MV slots and *undefined* is the value for other slots.  If the slot is
;;; not defined over this class of frob then NIL is returned.
(defun bound? (frob slot)
  (let ((parent (get-class-frob (get-type frob)))
	val)
    ;; CHANGED - RRK - 5/5/88 - It doesn't make sense to try to call a
    ;; method associated with a slot if there is no slot defined.  So,
    ;; the code just checks to see if there is no slot and then
    ;; returns nil.  This whole function could be reworked, but for
    ;; now, I'll just leave it as it is.
    ;;
    ;; OLD CODE FOLLOWS
    ;; if is not a slot then return the slot value
    ;; (unless (slot? parent slot)
    ;;    (let ((method-value (funcall slot frob)))
    ;;      (return-from bound? (if method-value
    ;;   		       (if (eq method-value *undefined*) nil t)))))
    ;;
    ;; NEW CODE 
    (if (not (slot? parent slot)) (return-from bound? nil))
    ;; End of patch
    (setf val (funcall slot frob))
    ;; could be an mv slot
    (cond ((null val)
	   (if (mv-slot? frob slot) 
	     nil
	     t))
	  ;; SV slot that is undefined.
	  ((eq val *undefined*)
	   nil)
	  ;; Any other value and the slot must be bound.
	  (t t))))

;;; Frame style ask with inheritance.  Multiple inheritance is done depth
;;; first.  If no bound slot is found in the inheritance tree, *undefined*
;;; is returned.
(defun ask (frob slot &key (use-context nil))
  (let (val)
    (when (gethash (get-type frob) *daemons*)
      (check-daemons :before-ask frob slot (get-class-frob (get-type frob)) nil))
    (setf val
	  (cond ((bound? frob slot)
		 (funcall slot frob))
		(use-context 
		 (inherit-with-context 
		  (list (get-class-frob (get-type frob))) slot (list *context* slot)))
		(t (inherit (list (get-class-frob (get-type frob))) slot))))
    (when (gethash (get-type frob) *daemons*)
      (check-daemons :after-ask frob slot (get-class-frob (get-type frob)) val))
    (if (eq val *undefined*)
      ;; need to get the real undefined value
      (funcall slot frob)
      val)))

;;; Inheritance function for ASK.  Uses depth first search.
(defun inherit (parents slot)
  (cond ((null parents) *undefined*)
	((bound? (car parents) slot)
	 (funcall slot (car parents)))
	(t (inherit (append (class-frob-parent (car parents)) (cdr parents)) slot))))

;;; Context based inheritance for ask
(defun inherit-with-context (parents slot context)
  (cond ((null parents) *undefined*)
	((eq (car context) (class-frob-name (car parents)))
	 (cond ((bound? (car parents) slot)
		(funcall slot (car parents)))
	       (t *undefined*)))
	((or (member context (class-frob-open (car parents)) :test #'equal)
	     (member context (class-frob-mv (car parents)) :test #'equal)
	     (member context (class-frob-generic (car parents)) :test #'equal)
	     (member context (class-frob-private (car parents)) :test #'equal))
	 (cond ((bound? (car parents) slot)
		(funcall slot (car parents)))
	       (t (inherit-with-context (class-frob-parent (car parents)) slot context))))
	(t (inherit-with-context (cdr parents) slot context))))

;;; Frame style assert.
(defun assert-val (frob slot val)
  (let ((set-slot (get-set-slot slot))
	(old-val  (funcall slot frob))
	(class    (get-type frob))
	set-fn)
    (when (gethash class *daemons*)
      (check-daemons :before-assert frob slot (get-class-frob class) val))
    (setf set-fn (get-generic-fn* set-slot frob))
    (cond ((integerp (cdr set-fn))
	   (funcall (car set-fn) frob (cdr set-fn) val))
	  (set-fn (funcall (car set-fn) frob val))
	  (t (undefined-method-error set-slot (get-type frob))))
    (unless (eq old-val (funcall slot frob))
      (when (gethash class *daemons*)
	(check-daemons :after-assert frob slot (get-class-frob class) val))
      (check-rules frob slot class val)
      (run-rules))
    val))

;;; assert multiple values
(defun assert-vals (frob slot vals)
  (let ((set-slot (get-set-slot slot))
	(class    (get-type frob))
	(parent   (get-class-frob (get-type frob)))
	set-fn old-val daemon)
    (setf daemon (gethash class *daemons*))
    (setf set-fn (get-generic-fn* set-slot frob))
    (unless set-fn
      (undefined-method-error set-slot class))
    (dolist (val vals)
      (when daemon
	(check-daemons :before-assert frob slot parent val))
      (setf old-val (funcall slot frob))
      (cond ((integerp (cdr set-fn))
	     (funcall (car set-fn) frob (cdr set-fn) val))
	    (set-fn (funcall (car set-fn) frob val)))
      (unless (eq old-val (funcall slot frob))
	(when daemon
	  (check-daemons :after-assert frob slot parent val))
	(check-rules frob slot class val)
	(run-rules)))
      vals))

;;; Removes a value from a slot
(defun remove-val (frob slot val)
  (let* ((set-slot (get-set-slot slot))
	 (old-val  (funcall slot frob))
	 (class    (get-type frob))
	 (set-fn   (get-generic-fn* set-slot frob)))
    (unless set-fn
      (undefined-method-error set-slot class))
    (when (gethash class *daemons*)
      (check-daemons :before-remove frob slot (get-class-frob class) val))
    (cond ((mv-slot? frob slot)
	   (when (member val old-val :test #'equal)
	     (with-mv-slot set 
               (funcall set-slot frob (delete val old-val :test #'equal)))
	     (when (gethash class *daemons*)
	       (check-daemons :after-remove frob slot (get-class-frob class) val))))
	  ((equal val old-val)
	   (funcall set-slot frob *undefined*)
	   (when (gethash class *daemons*)
	     (check-daemons :after-remove frob slot (get-class-frob class) val))))
    (funcall slot frob)))

;;; Stubs for now
(defun check-daemons (time frob slot class value)
  (declare (ignore time)(ignore frob)(ignore slot)(ignore class)(ignore value))
  nil)

(defun run-rules ())

(defun check-rules (frob slot class val)
  (declare (ignore frob)(ignore slot)(ignore class)(ignore val))
  nil)

;;; Erases the value of a slot.  For single valued slots erase and remove-val
;;; have the same effect when the slot value to remove-val is the same as val
(defun erase (frob slot)
  (let ((set-fn  (get-set-slot slot)))
    (with-mv-slot erase (funcall set-fn frob *undefined*))
    (funcall slot frob)))

;;; Aux macro for show
(defmacro show-aux (slot-type class frob)
  `(dolist (s (,slot-type ,class))
     (setf *context* (car s))
     (format t "~S --> ~S~%" s (funcall (second s) ,frob))))

;;; Frame style show.  Will only show those slots that should be shown.
(defun show (frob)
  (format t "~S has the following values:~%" frob)
  (let ((old-context *context*)
	(class (get-class-frob (get-type frob))))
    (show-aux class-frob-open class frob)
    (show-aux class-frob-mv class frob)
    (show-aux class-frob-generic class frob)
    (when (eq *module* (class-frob-name class))
      (show-aux class-frob-private class frob))
    (dolist (s (class-frob-show-methods class)) 
      (cond ((not (member s (class-frob-private-methods class) :test #'equal))
	     (setf *context* (car s))
	     (format t "~S --> ~S~%" s (funcall (second s) frob)))
	    ((eq *module* (class-frob-name class))
	     (setf *context* (car s))
	     (format t "~S --> ~S~%" s (funcall (second s) frob)))))
    (setf *context* old-context))
  nil)

;;; returns a list of all of the frobs in the system
(defun all-frobs ()
  (nconc (all-class-frobs)(all-instance-frobs)))

;;; returns a list of all of the class frobs in the system
(defun all-class-frobs ()
  (mapcar #'get-class-frob *class-frobs*))

;;; returns a list of all of the instance frobs in the system
(defun all-instance-frobs ()
  (let ((flist nil))
    (maphash #'(lambda (k v) (setf flist (cons v flist))) *hash*)
    flist))

;;(defun all-instance-frobs ()
;;  *frobs*)

;;; Cleans out the system.  All frobs and methods are undefined.
(defun reset-system ()
  (clrhash *key*)
  (clrhash *hash*)
  (setf *locations* 0)
  (remove-all-but-sys-meths)
  (remove-all-but-sys-class)
  (clrhash *daemons*)
;;  (setf *frobs* nil)
  (setf *class-frobs* *sys-class*)
  (setf *context* nil)
  (setf *module* nil))

;;; Does not remove the system frobs
(defun remove-all-but-sys-class ()
  (maphash #'(lambda (key entry)	
	       (declare (ignore entry))
	       (cond ((member key *sys-class*)
		      (let ((frob (get-class-frob key)))
			(setf (class-frob-children frob) nil)
			(setf (class-frob-gensym frob) 0)
			(setf (class-frob-class-children frob) nil)))
		     (t (remhash key *class-hash*))))
	   *class-hash*))

;;; Does not remove the system methods from *params* or *main-methods*
(defun remove-all-but-sys-meths ()
  (maphash #'(lambda (key entry)	
	       (declare (ignore entry))
	       (unless (member key *sys-meths*)
		 (remhash key *params*)))
	   *params*)
  (maphash #'(lambda (key entry)	
	       (declare (ignore entry))
	       (unless (member key *sys-meths*)
		 (remhash key *main-methods*)))
	   *main-methods*))

;;; Declares a class frob to be a sys-frob
(defun def-sys (name)
  (push name *sys-class*))

;;; Declares a method to be a system method
(defun def-sys-meth (name)
  (if (symbolp name)
    (push name *sys-meths*)
    (setf *sys-meths* (append name *sys-meths*))))

;;; Returns all of the instance children of a class
(defun instances (class)
  (unless (class-frob-p class)
    (error "~S is not a class frob" class))
  (let ((instance-list nil))
    (when (class-frob-children class)
      (maphash #'(lambda (key value)
		   (setf instance-list (cons value instance-list)))
	       (class-frob-children class)))
    instance-list))

;;; Returns all of the children descendants of a class frob
(defun all-instances (class)
  (unless (class-frob-p class)
    (error "~S is not a class frob" class))
  (mapcan #'instances (cons class (all-class-instances class))))

;;; Flattens a list
(defun flatten (x)
  (cond ((null x) nil)
        ((atom x) (list x))
	(t (append (flatten (car x))(flatten (cdr x))))))

;;; Returns all of the instance children of a class
(defun class-instances (class &optional (lookup #'get-class-frob))
  (unless (class-frob-p class)
    (error "~S is not a class frob" class))
  (mapcar lookup (class-frob-class-children class)))

;;; Returns the child classes of a frob class
(defun all-class-instances (class &optional (lookup #'get-class-frob))
  (unless (class-frob-p class)
    (error "~S is not a class frob" class))
  (append (class-instances class lookup)
	  (flatten (mapcar #'all-class-instances 
		    (mapcar lookup (class-frob-class-children class))))))

;;; Returns the parent of an instance frob or the parents of a class frob
(defun class-parent (frob)
  (cond ((class-frob-p frob)(class-frob-parent frob))
	((frob-p frob) (list (frob-parent frob)))))
	 
;;; Returns an inheritance list of all of the classes a frob inherits from
(defun inherits-from (frob)
  (when (frob-p frob)
    (setf frob (frob-parent frob)))
  (cons (frob-type frob)
	(apply #'append (mapcar #'inherits-from (class-parent frob)))))

;;; Returns T if the frob inherits from type
(defun frob-type? (frob type)
  (when (member (frob-type type) (inherits-from frob))
    t))
    

;;; kills a class frob
(defun kill-class-frob (frob)
  (dolist (parent (class-frob-parent frob))
    (setf (class-frob-class-children parent)
	  (delete (class-frob-name frob)
		  (class-frob-class-children parent)
		  :count 1)))
  (remove-children frob)
  (remove-class-children frob)
  (setf *class-frobs* (delete (class-frob-name frob) *class-frobs* :count 1))
  (put-class-frob (class-frob-name frob) nil))
  
;;; Kills all of the instances in the system.
(defun kill-instance-frobs ()
  (clrhash *hash*)
;;  (setf *frobs* nil)
  (dolist (class *class-frobs*)
    (setf class (get-class-frob class))
    (setf (class-frob-children class) nil)
    (setf (class-frob-gensym class) 0)))

;;; kills all of the instances of a class
(defun kill-instances (class)
  (remove-children class)
  (setf (class-frob-gensym class) 0)
  (setf (class-frob-children class) nil))

;;; kills an instance frob
(defun kill-frob (frob)
  (let ((name (frob-name frob)))
    (remhash name (class-frob-children (frob-parent frob)))
;;  (setf *frobs* 
;;	  (delete name *frobs* :key #'frob-name :test #'equal :count 1))
;    (when *pfrobs*
;      (decf *locations* (length (frob-vect frob))))
    ;; remove the frob
    (put-frob name nil)))

;; End of file.
